//========================================================================================
//  SPC.java
//    en:SCSI protocol controller
//    ja:SCSIプロトコルコントローラ
//  Copyright (C) 2003-2025 Makoto Kamada
//
//  This file is part of the XEiJ (X68000 Emulator in Java).
//  You can use, modify and redistribute the XEiJ if the conditions are met.
//  Read the XEiJ License for more details.
//  https://stdkmd.net/xeij/
//========================================================================================

//----------------------------------------------------------------------------------------
//  内蔵SCSIと拡張SCSIの兼用
//  MB89352
//----------------------------------------------------------------------------------------

package xeij;

import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;

public class SPC {

  public static final boolean SPC_DEBUG_ON = false;
  public static final boolean SPC_DEBUG_COMMAND = SPC_DEBUG_ON && true;
  public static final boolean SPC_DEBUG_PHASE = SPC_DEBUG_ON && true;
  public static final boolean SPC_DEBUG_PORT = SPC_DEBUG_ON && false;

  public static final boolean SPC_REPORT_UNIMPLEMENTED_COMMAND = true;  //未実装コマンドを表示する
  public static final boolean SPC_REPORT_UNIMPLEMENTED_XOR = false;  //XOR系の未実装コマンドも表示する
  public static final boolean SPC_CHECK_BLOCK_ZERO = false;  //ブロック0に書き込もうとしたら内容を表示する

  public static final boolean SPC_REMOVABLE_HDD = true;  //true=リムーバブル

  public static final int SPC_MAX_BYTES_PER_BLOCK = 2048;  //セクタ長の上限

  //データインフェーズで残りが8バイトになったらステータスフェーズに切り替えてINTSのSRをセットする
  //ACKがFalseになるのを待たずにステータスフェーズに移行するハードディスクを接続したときS_DATAINIが最後の数バイトを受け取り損ねるバグを露見させる
  public static final boolean SPC_EXPOSE_DATAINI_BUG = false;

  //ベースアドレス
  public static final int SPC_BASE_EX = 0x00ea0000;  //拡張SCSIのSPCのベースアドレス
  public static final int SPC_BASE_IN = 0x00e96020;  //内蔵SCSIのSPCのベースアドレス

  //ROM起動ハンドル
  public static final int SPC_HANDLE_EX = 0x00ea0020;  //拡張SCSIのROM起動ハンドル
  public static final int SPC_HANDLE_IN = 0x00fc0000;  //内蔵SCSIのROM起動ハンドル

  //フェーズ
  public static final int SPC_PHASE_MASK        = 0b00000111;
  public static final int SPC_DATA_OUT_PHASE    = 0b00000000;  //Data Out Phase。データアウトフェーズ
  public static final int SPC_DATA_IN_PHASE     = 0b00000001;  //Data In Phase。データインフェーズ
  public static final int SPC_COMMAND_PHASE     = 0b00000010;  //Command Phase。コマンドフェーズ
  public static final int SPC_STATUS_PHASE      = 0b00000011;  //Status Phase。ステータスフェーズ
  public static final int SPC_MESSAGE_OUT_PHASE = 0b00000110;  //Message Out Phase。メッセージアウトフェーズ
  public static final int SPC_MESSAGE_IN_PHASE  = 0b00000111;  //Message In Phase。メッセージインフェーズ

  //レジスタ
  public static final int SPC_BDID       = 0x01;        //Bus Device ID。自分のID
  public static final int SPC_BDID_I7    = 0b10000000;  //ID 7
  public static final int SPC_BDID_I6    = 0b01000000;  //ID 6
  public static final int SPC_BDID_I5    = 0b00100000;  //ID 5
  public static final int SPC_BDID_I4    = 0b00010000;  //ID 4
  public static final int SPC_BDID_I3    = 0b00001000;  //ID 3
  public static final int SPC_BDID_I2    = 0b00000100;  //ID 2
  public static final int SPC_BDID_I1    = 0b00000010;  //ID 1
  public static final int SPC_BDID_I0    = 0b00000001;  //ID 0
  public static final int SPC_SCTL       = 0x03;        //SPC Control
  public static final int SPC_SCTL_RD    = 0b10000000;  //Reset & Disable。1=ハードウェアリセット
  public static final int SPC_SCTL_CR    = 0b01000000;  //Control Reset。1=転送回路リセット
  public static final int SPC_SCTL_DM    = 0b00100000;  //Diag Mode。1=自己診断モード
  public static final int SPC_SCTL_AE    = 0b00010000;  //Arbitration Enable。0=アービトレーションフェーズなし(SASI),1=あり(SCSI)
  public static final int SPC_SCTL_PE    = 0b00001000;  //Parity Enable。1=入力データのパリティをチェックする。出力データは常にパリティが付く
  public static final int SPC_SCTL_SE    = 0b00000100;  //Select Enable。0=常にイニシエータ,1=セレクションフェーズが来たらターゲットになる
  public static final int SPC_SCTL_RE    = 0b00000010;  //Reselect Enable。1=リセレクションフェーズに応答する
  public static final int SPC_SCTL_IE    = 0b00000001;  //Interrupt Enable。0=割り込み禁止,1=許可。禁止されていてもINTSは変化する
  public static final int SPC_SCMD       = 0x05;        //SPC Command
  public static final int SPC_SCMD_CC    = 0b11100000;  //Command Code
  public static final int SPC_SCMD_CC_BR = 0b00000000;  //Bus Release。ターゲットのときバスフリーフェーズへ移行
  public static final int SPC_SCMD_CC_SL = 0b00100000;  //Select。セレクション/リセレクションを開始
  public static final int SPC_SCMD_CC_RA = 0b01000000;  //Reset ATN。ATNをクリア
  public static final int SPC_SCMD_CC_SA = 0b01100000;  //Set ATN。ATNをセット
  public static final int SPC_SCMD_CC_TR = 0b10000000;  //Transfer。転送開始
  public static final int SPC_SCMD_CC_TP = 0b10100000;  //Transfer Pause。転送中断
  public static final int SPC_SCMD_CC_RR = 0b11000000;  //Reset ACK/REQ。CPU転送のときACK/REQをクリア
  public static final int SPC_SCMD_CC_SR = 0b11100000;  //Set ACK/REQ。CPU転送のときACK/REQをセット
  public static final int SPC_SCMD_RO    = 0b00010000;  //RST Out。1=SCSIバスリセット
  public static final int SPC_SCMD_IT    = 0b00001000;  //Intercept Transfer。1=CPU転送時FIFOバッファの内容を保持
  public static final int SPC_SCMD_PT    = 0b00000100;  //Program Transfer。0=DMA転送(DREQあり),1=CPU転送(DREQなし)
  public static final int SPC_SCMD_TM    = 0b00000001;  //Termination Mode。イニシエータのとき0=カウント0で終了する,1=カウント0で終了しない(Padding)、ターゲットのとき0=パリティエラーがあってもカウント0まで転送する,1=パリティエラーがあると直ちに終了する
  public static final int SPC_INTS       = 0x09;        //Interrupt Sense
  public static final int SPC_INTS_SL    = 0b10000000;  //Selected。1=他のデバイスのセレクションフェーズで選択されてターゲットになった
  public static final int SPC_INTS_RS    = 0b01000000;  //Reselected。1=他のデバイスのリセレクションフェーズで選択されてイニシエータになった
  public static final int SPC_INTS_DC    = 0b00100000;  //Disconnected。1=バスフリーフェーズになった。バスを使うとき0に戻すこと
  public static final int SPC_INTS_CC    = 0b00010000;  //Command Complete。1=SelectやTransferなどのコマンドが終了した。ターゲットのときパリティエラーで停止した
  public static final int SPC_INTS_SR    = 0b00001000;  //Service Required。1=PCTLレジスタとバスのフェーズが一致しないんだけどどうにかして。入力中、送信の終わった相手がフェーズをさっさと切り替えた。受信するデータがまだFIFOに残っているのに
  public static final int SPC_INTS_TO    = 0b00000100;  //Time Out。1=セレクション/リセレクションに応答がない。セレクションタイムアウトのときSELが1のままなのでTEMP=$00で復旧させる必要がある
  public static final int SPC_INTS_HE    = 0b00000010;  //SPC Hard Error。1=なんかエラーが出た。SERRを見てくれ
  public static final int SPC_INTS_RC    = 0b00000001;  //Reset Condition。1=SCSIバスがリセットされた(RST信号が1になった)
  public static final int SPC_PSNS       = 0x0b;        //(Read) Phase Sense
  public static final int SPC_PSNS_REQ   = 0b10000000;  //REQ
  public static final int SPC_PSNS_ACK   = 0b01000000;  //ACK
  public static final int SPC_PSNS_ATN   = 0b00100000;  //ATN
  public static final int SPC_PSNS_SEL   = 0b00010000;  //SEL。1=セレクションフェーズ
  public static final int SPC_PSNS_BSY   = 0b00001000;  //BSY
  public static final int SPC_PSNS_MSG   = 0b00000100;  //MSG
  public static final int SPC_PSNS_CD    = 0b00000010;  //C/D
  public static final int SPC_PSNS_IO    = 0b00000001;  //I/O
  public static final int SPC_SDGC       = 0x0b;        //(Write) SPC Diag Control
  public static final int SPC_SDGC_REQ   = 0b10000000;  //Diag REQ
  public static final int SPC_SDGC_ACK   = 0b01000000;  //Diag ACK
  public static final int SPC_SDGC_XFER  = 0b00100000;  //Xfer Enable。CPU転送のとき1=データ転送割り込み許可
  public static final int SPC_SDGC_BSY   = 0b00001000;  //Diag BSY
  public static final int SPC_SDGC_MSG   = 0b00000100;  //Diag MSG
  public static final int SPC_SDGC_CD    = 0b00000010;  //Diag C/D
  public static final int SPC_SDGC_IO    = 0b00000001;  //Diag I/O
  public static final int SPC_SSTS       = 0x0d;        //SPC Status
  public static final int SPC_SSTS_INIT  = 0b10000000;  //Connected INIT。1=イニシエータ
  public static final int SPC_SSTS_TARG  = 0b01000000;  //Connected TARG。1=ターゲット
  public static final int SPC_SSTS_BUSY  = 0b00100000;  //SPC Busy。1=コマンド実行中
  public static final int SPC_SSTS_TRIP  = 0b00010000;  //Transfer in Progress。1=DMA転送中
  public static final int SPC_SSTS_SRIN  = 0b00001000;  //SCSI Reset In。RST信号の状態
  public static final int SPC_SSTS_TC0   = 0b00000100;  //TC=0。1=転送カウンタが0
  public static final int SPC_SSTS_DF    = 0b00000010;  //DREG status Full。1=8バイトのFIFOが一杯
  public static final int SPC_SSTS_DE    = 0b00000001;  //DREG status Empty。1=8バイトのFIFOが空
  public static final int SPC_SERR       = 0x0f;        //SPC Error Status
  public static final int SPC_SERR_DI    = 0b10000000;  //Data Error SCSI。1=入力データにパリティエラーがある
  public static final int SPC_SERR_DO    = 0b01000000;  //Data Error SPC。1=出力データにパリティエラーがある
  public static final int SPC_SERR_XO    = 0b00100000;  //Xfer Out。1=Xfer EnableのときData Request中
  public static final int SPC_SERR_PE    = 0b00001000;  //TC Parity Error。1=転送カウンタにパリティエラーがある
  public static final int SPC_SERR_ST    = 0b00000010;  //Short Transfer Period。1=REQ/ACKが速すぎてSPCが追従できない
  public static final int SPC_PCTL       = 0x11;        //Phase Control
  public static final int SPC_PCTL_IE    = 0b10000000;  //Busfree INT Enable。1=バスフリーフェーズを検出したらDisconnected割り込みを要求する
  public static final int SPC_PCTL_TP    = 0b00000111;  //Transfer Phase。転送フェーズ
  public static final int SPC_PCTL_MSG   = 0b00000100;  //MSG
  public static final int SPC_PCTL_CD    = 0b00000010;  //C/D
  public static final int SPC_PCTL_IO    = 0b00000001;  //I/O。0=Out,1=In
  public static final int SPC_PCTL_SR    = 0b00000001;  //Selectコマンドのセレクション/リセレクション選択
  public static final int SPC_PCTL_SR_R  = 0b00000001;  //Selectコマンドでリセレクションを開始
  public static final int SPC_PCTL_SR_S  = 0b00000000;  //Selectコマンドでセレクションを開始
  public static final int SPC_MBC        = 0x13;        //Modified Byte Counter
  public static final int SPC_DREG       = 0x15;        //Data Register
  public static final int SPC_TEMP       = 0x17;        //Temporary Register
  public static final int SPC_TCH        = 0x19;        //Transfer Counter High
  public static final int SPC_TCM        = 0x1b;        //Transfer Counter Mid
  public static final int SPC_TCL        = 0x1d;        //Transfer Counter Low

  //ステータスバイト
  public static final int SPC_GOOD                       = 0x00;
  public static final int SPC_CHECK_CONDITION            = 0x02;
  public static final int SPC_CONDITION_MET              = 0x04;
  public static final int SPC_BUSY                       = 0x08;
  public static final int SPC_INTERMEDIATE               = 0x10;
  public static final int SPC_INTERMEDIATE_CONDITION_MET = 0x14;
  public static final int SPC_RESERVATION_CONFLICT       = 0x18;
  public static final int SPC_COMMAND_TERMINATED         = 0x22;
  public static final int SPC_QUEUE_FULL                 = 0x28;

  //メッセージコード
  public static final int SPC_COMMAND_COMPLETE                  = 0x00;
  public static final int SPC_EXTENDED_MESSAGE                  = 0x01;  //拡張メッセージ長nとnバイトの拡張メッセージが続く
  public static final int SPC_SAVE_DATA_POINTER                 = 0x02;
  public static final int SPC_RESTORE_POINTERS                  = 0x03;
  public static final int SPC_DISCONNECT                        = 0x04;
  public static final int SPC_INITIATOR_DETECTED_ERROR          = 0x05;
  public static final int SPC_ABORT                             = 0x06;
  public static final int SPC_MESSAGE_REJECT                    = 0x07;
  public static final int SPC_NO_OPERATION                      = 0x08;
  public static final int SPC_MESSAGE_PARITY_ERROR              = 0x09;
  public static final int SPC_LINKED_COMMAND_COMPLETE           = 0x0a;
  public static final int SPC_LINKED_COMMAND_COMPLETE_WITH_FLAG = 0x0b;
  public static final int SPC_BUS_DEVICE_RESET                  = 0x0c;
  public static final int SPC_ABORT_TAG                         = 0x0d;
  public static final int SPC_CLEAR_QUEUE                       = 0x0e;
  public static final int SPC_INITIATE_RECOVERY                 = 0x0f;
  public static final int SPC_RELEASE_RECOVERY                  = 0x10;
  public static final int SPC_TERMINATE_IO_PROCESS              = 0x11;
  public static final int SPC_SIMPLE_QUEUE_TAG                  = 0x20;  //0x20-0x7fは2バイト
  public static final int SPC_HEAD_OF_QUEUE_TAG                 = 0x21;
  public static final int SPC_ORDERED_QUEUE_TAG                 = 0x22;
  public static final int SPC_IGNORE_WIDE_RESIDUE               = 0x23;
  public static final int SPC_IDENTIFY                          = 0x80;

  //センスエラーコード[0]
  public static final int SPC_INVALID_COMMAND        = 0x20;
  public static final int SPC_INVALID_SECTOR_ADDRESS = 0x21;
  public static final int SPC_EXTENDED_SENSE         = 0x70;  //拡張センスデータ(8バイト以上)。[2]=センスキー,[7]=[8]以降の長さ
  //センスキー[2]
  public static final int SPC_NO_SENSE        = 0x00;
  public static final int SPC_RECOVERED_ERROR = 0x01;
  public static final int SPC_NOT_READY       = 0x02;
  public static final int SPC_MEDIUM_ERROR    = 0x03;
  public static final int SPC_HARDWARE_ERROR  = 0x04;
  public static final int SPC_ILLEGAL_REQUEST = 0x05;
  public static final int SPC_UNIT_ATTENTION  = 0x06;
  public static final int SPC_DATA_PROTECT    = 0x07;
  public static final int SPC_BLANK_CHECK     = 0x08;
  public static final int SPC_COPY_ABORTED    = 0x0a;
  public static final int SPC_ABORTED_COMMAND = 0x0b;
  public static final int SPC_EQUAL           = 0x0c;
  public static final int SPC_VOLUME_OVERFLOW = 0x0d;
  public static final int SPC_MISCOMPARE      = 0x0e;

  //デバイスタイプ
  //  スタンダードInquiryデータの1バイト目の下位5bit
  public static final int SPC_DIRECT_ACCESS_DEVICE     = 0x00;  //ダイレクトアクセスデバイス(HDD)
  public static final int SPC_SEQUENCIAL_ACCESS_DEVICE = 0x01;  //シーケンシャルアクセスデバイス(磁気テープ)
  public static final int SPC_PRINTER_DEVICE           = 0x02;  //プリンタデバイス
  public static final int SPC_PROCESSOR_DEVICE         = 0x03;  //プロセッサデバイス
  public static final int SPC_WRITE_ONCE_DEVICE        = 0x04;  //ライトワンスデバイス(追記型光ディスク)
  public static final int SPC_CDROM_DEVICE             = 0x05;  //CD-ROMデバイス
  public static final int SPC_SCANNER_DEVICE           = 0x06;  //スキャナデバイス
  public static final int SPC_OPTICAL_MEMORY_DEVICE    = 0x07;  //光メモリデバイス(消去可能光ディスク)
  public static final int SPC_MEDIUM_CHANGER_DEVICE    = 0x08;  //メディアチェンジャデバイス(磁気テープライブラリ)
  public static final int SPC_COMMUNICATION_DEVICE     = 0x09;  //コミュニケーションデバイス

  //CDBの長さ
  //  0xc1  Load/Unload SHARP MO  (6)
  //  0xc2  Assign Drive(SASI)    (6)
  //  0xd8  Read CDDA SONY        (12)
  public static final int[] SPC_CDB_LENGTH = {
    // g0    g1   group2  group3  group4  group5  group6  group7
    //0x 1x 2x 3x 4x  5x  6x  7x  8x  9x  ax  bx  cx  dx  ex  fx
    6, 6, 10, 10, 10, 10,  1,  1, 16, 16, 12, 12,  6, 12,  1,  1 };

  //コマンド名
  //  デバイスタイプ
  //    A  全デバイス共通
  //    D  ダイレクトアクセスデバイス
  //    T  シーケンシャルアクセスデバイス
  //    P  プリンタデバイス
  //    G  プロセッサデバイス
  //    W  ライトワンスデバイス
  //    C  CD-ROMデバイス
  //    S  スキャナデバイス
  //    O  光メモリデバイス
  //    L  メディアチェンジャデバイス
  //    N  コミュニケーションデバイス
  //  サポート
  //    m  必須
  //    e  拡張仕様
  //    o  オプション
  //    z  タイプ依存
  //    v  ベンダ固有
  //    r  予約
  //  参考  SCSI-2詳細解説;菅谷誠一著;CQ出版社
  public static final String[] SPC_COMMAND_NAME = (
    //グループ0  コマンド長6バイト                           | SCSI-1|   SCSI-2  |
    //                                                       |ADTPGWC|ADTPGWCSOLN|
    "Test Unit Ready," +                                //$00|ooooooo|mmmmmmmmmmm|
    "Rezero Unit/Rewind," +                             //$01| omvvoo| omvrooroor|DWCOL:Rezero Unit/T:Rewind
    "," +                                               //$02| vvvvvv| vvvvvvrrrr|
    "Request Sense," +                                  //$03|mmmmmmm|mmmmmmmmmmm|
    "Format Unit," +                                    //$04| mrovrr| mrorrrrorr|DO:Format Unit/P:Format
    "Read Block Limits," +                              //$05| vevvvv| vmvvvvrrrr|
    "Format Block(SASI)," +                             //$06| vvvvvv| vvvvvvrrrr|
    "Reassign Blocks/Initialize Element Status," +      //$07| ovvvor| ovvrorroor|DWO:Reassign Blocks/L:Initialize Element Status
    "Read(6)/Receive/Get Message(6)," +                 //$08| mmvooo| mmvooororo|DWCO:Read(6)/T:Read/G:Receive/N:Get Message(6)
    "," +                                               //$09| vvvvvv| vvvvvvrrrr|
    "Write(6)/Print/Send/Send Message(6)," +            //$0A| mmmmor| ommmorrorm|DWO:Write(6)/T:Write/P:Print/G:Send/N:Send Message(6)
    "Seek(6)/Track Select/Slew and Print," +            //$0B| ooovoo| ororoororr|DWCO:Seek(6)/T:(Track Select)/P:Slew and Print
    "," +                                               //$0C| vvvvvv| vvvvvvrrvr|
    "," +                                               //$0D| vvvvvv| vvvvvvrrrr|
    "," +                                               //$0E| vvvvvv| vvvvvvrrrr|
    "Read Reverse," +                                   //$0F| vovvvv| vovvvvrrrr|
    "Write Filemarks/Synchronize Buffer," +             //$10| vmovvv| vmovvvrrrr|T:Write Filemarks/P:Synchronize Buffer
    "Space," +                                          //$11| vovvvv| vmvvvvrrrr|
    "Inquiry," +                                        //$12|eeeeeee|mmmmmmmmmmm|
    "Verify," +                                         //$13| vovvvv| vovvvvrrrr|
    "Recover Buffered Data," +                          //$14| voovvv| voovvvrrrr|
    "Mode Select(6)," +                                 //$15|zooovoo|zomoroooooo|
    "Reserve Unit," +                                   //$16| ooovoo| mmmrmmmmor|DWCOL:Reserve/TPS:Reserve Unit
    "Release Unit," +                                   //$17| ooovoo| mmmrmmmmor|DWCOL:Release/TPS:Release Unit
    "Copy," +                                           //$18|ooooooo|ooooooooorr|
    "Erase," +                                          //$19| vovrvv| vmvvvvrrrr|
    "Mode Sense(6)," +                                  //$1A|zoooroo|zomoroooooo|
    "Start-Stop Unit/Load Unload/Stop Print/Scan," +    //$1B| oooroo| oooroooorr|DWCO:Start-Stop Unit/T:Load Unload/T:Stop Print/S:Scan
    "Receive Diagnostic Results," +                     //$1C|ooooooo|ooooooooooo|
    "Send Diagnostic," +                                //$1D|ooooooo|mmmmmmmmmmm|
    "Prevent-Allow Medium Removal," +                   //$1E| oorroo| oorrooroor|
    "," +                                               //$1F| rrrrrr| rrrrrrrrrr|
    //グループ1  コマンド長10バイト                          | SCSI-1|   SCSI-2  |
    //                                                       |ADTPGWC|ADTPGWCSOLN|
    "," +                                               //$20| vrrrvv| vrrrvvrvrr|
    "," +                                               //$21| vrrrvv| vrrrvvrvrr|
    "," +                                               //$22| vrrrvv| vrrrvvrvrr|
    "," +                                               //$23| vrrrvv| vrrrvvrvrr|
    "Set Window," +                                     //$24| vrrrvv| vrrrvvmrrr|
    "Read Capacity/Get Window," +                       //$25| errree| mrrrmmomrr|DWO:Read Capacity/C:Read CD-ROM Capacity/S:Get Window
    "," +                                               //$26| vrrrvv| vrrrvvrrrr|
    "," +                                               //$27| vrrrvv| vrrrvvrrrr|
    "Read(10)/Get Message(10)," +                       //$28| errrmm| mrrrmmmmro|DWCO:Read(10)/S:Read/N:Get Message(10)
    "Read Generation," +                                //$29| vrrrvv| vrrrvvrorr|
    "Write(10)/Send/Send Message(10)," +                //$2A| errrmr| orrrmromro|DWO:Write(10)/S:Send/N:Send Message(10)
    "Seek(10)/Locate/Position to Element," +            //$2B| oorroo| oorrooroor|DWCO:Seek(10)/T:Locate/L:Position to Element
    "Erase(10)," +                                      //$2C| vrrrrv| vrrrvrrorr|
    "Read Updated Block," +                             //$2D| vrrrrv| vrrrvrrorr|
    "Write and Verify(10)," +                           //$2E| orrror| orrrorrorr|
    "Verify(10)," +                                     //$2F| orrroo| orrroororr|
    "Search Data High(10)," +                           //$30| orrroo| orrroororr|
    "Search Data Equal(10)/Object Position," +          //$31| orrroo| orrroooorr|DWCO:Search Data Equal(10)/S:Object Position
    "Search Data Low(10)," +                            //$32| orrroo| orrroororr|
    "Set Limits(10)," +                                 //$33| orrroo| orrroororr|
    "Pre-Fetch/Read Position/Get Data Buffer Status," + //$34| rrrrrr| oorroooorr|DWCO:Pre-Fetch/T:Read Position/S:Get Data Buffer Status
    "Synchronize Cache," +                              //$35| rrrrrr| orrroororr|
    "Lock-Unlock Cache," +                              //$36| rrrrrr| orrroororr|
    "Read Defect Data(10)," +                           //$37| rrrrrr| orrrrrrorr|
    "Medium Scan," +                                    //$38| rrrrrr| rrrrorrorr|
    "Compare," +                                        //$39|oooo-oo|ooooooooorr|
    "Copy and Verify," +                                //$3A|oooo-oo|ooooooooorr|
    "Write Buffer," +                                   //$3B|rrrrrrr|ooooooooooo|
    "Read Buffer," +                                    //$3C|rrrrrrr|ooooooooooo|
    "Update Block," +                                   //$3D| rrrrrr| rrrrrrrorr|
    "Read Long," +                                      //$3E| rrrrrr| orrroororr|
    "Write Long," +                                     //$3F| rrrrrr| orrrorrorr|
    //グループ2  コマンド長10バイト                          | SCSI-1|   SCSI-2  |
    //                                                       |ADTPGWC|ADTPGWCSOLN|
    "Change Definition," +                              //$40|rrrrrrr|ooooooooooo|
    "Write Same," +                                     //$41| rrrrrr| orrrrrrrrr|
    "Read Sub-Channel," +                               //$42| rrrrrr| rrrrrorrrr|
    "Read TOC," +                                       //$43| rrrrrr| rrrrrorrrr|
    "Read Header," +                                    //$44| rrrrrr| rrrrrorrrr|
    "Play Audio(10)," +                                 //$45| rrrrrr| rrrrrorrrr|
    "," +                                               //$46| rrrrrr| rrrrrrrrrr|
    "Play Audio MSF," +                                 //$47| rrrrrr| rrrrrorrrr|
    "Play Audio Track Index," +                         //$48| rrrrrr| rrrrrorrrr|
    "Play Audio Track Relative(10)," +                  //$49| rrrrrr| rrrrrorrrr|
    "," +                                               //$4A| rrrrrr| rrrrrrrrrr|
    "Pause Resume," +                                   //$4B| rrrrrr| rrrrrorrrr|
    "Log Select," +                                     //$4C|rrrrrrr|ooooooooooo|
    "Log Sense," +                                      //$4D|rrrrrrr|ooooooooooo|
    "," +                                               //$4E| rrrrrr| rrrrrrrrrr|
    "," +                                               //$4F| rrrrrr| rrrrrrrrrr|
    "XDWRITE," +  //$50  XOR  10byte
    "XPWRITE," +  //$51  XOR  10byte
    "XDREAD," +  //$52  XOR  10byte
    "," +                                               //$53| rrrrrr| rrrrrrrrrr|
    "," +                                               //$54| rrrrrr| rrrrrrrrrr|
    "Mode Select(10)," +                                //$55|rrrrrrr|zoooroooooo|
    "," +                                               //$56| rrrrrr| rrrrrrrrrr|
    "," +                                               //$57| rrrrrr| rrrrrrrrrr|
    "," +                                               //$58| rrrrrr| rrrrrrrrrr|
    "," +                                               //$59| rrrrrr| rrrrrrrrrr|
    "Mode Sense(10)," +                                 //$5A|rrrrrrr|zoooroooooo|
    "," +                                               //$5B| rrrrrr| rrrrrrrrrr|
    "," +                                               //$5C| rrrrrr| rrrrrrrrrr|
    "," +                                               //$5D| rrrrrr| rrrrrrrrrr|
    "," +                                               //$5E| rrrrrr| rrrrrrrrrr|
    "," +                                               //$5F| rrrrrr| rrrrrrrrrr|
    //グループ3  予約
    "," +  //$60
    "," +  //$61
    "," +  //$62
    "," +  //$63
    "," +  //$64
    "," +  //$65
    "," +  //$66
    "," +  //$67
    "," +  //$68
    "," +  //$69
    "," +  //$6A
    "," +  //$6B
    "," +  //$6C
    "," +  //$6D
    "," +  //$6E
    "," +  //$6F
    "," +  //$70
    "," +  //$71
    "," +  //$72
    "," +  //$73
    "," +  //$74
    "," +  //$75
    "," +  //$76
    "," +  //$77
    "," +  //$78
    "," +  //$79
    "," +  //$7A
    "," +  //$7B
    "," +  //$7C
    "," +  //$7D
    "," +  //$7E
    "," +  //$7F
    //グループ4  コマンド長16バイト
    "XDWRITE EXTENDED," +  //$80  XOR  16byte
    "REBUILD," +  //$81  XOR  16byte
    "REGENERATE," +  //$82  XOR  16byte
    "," +  //$83
    "," +  //$84
    "," +  //$85
    "," +  //$86
    "," +  //$87
    "," +  //$88
    "," +  //$89
    "," +  //$8A
    "," +  //$8B
    "," +  //$8C
    "," +  //$8D
    "," +  //$8E
    "," +  //$8F
    "," +  //$90
    "," +  //$91
    "," +  //$92
    "," +  //$93
    "," +  //$94
    "," +  //$95
    "," +  //$96
    "," +  //$97
    "," +  //$98
    "," +  //$99
    "," +  //$9A
    "," +  //$9B
    "," +  //$9C
    "," +  //$9D
    "," +  //$9E
    "," +  //$9F
    //グループ5  コマンド長12バイト                          | SCSI-1|   SCSI-2  |
    //                                                       |ADTPGWC|ADTPGWCSOLN|
    "," +                                               //$A0| rrrrrr| rrrrrrrrrr|
    "," +                                               //$A1| rrrrrr| rrrrrrrrrr|
    "," +                                               //$A2| rrrrrr| rrrrrrrrrr|
    "," +                                               //$A3| rrrrrr| rrrrrrrrrr|
    "," +                                               //$A4| rrrrrr| rrrrrrrrrr|
    "Play Audio(12)/Move Medium," +                     //$A5| rrrrrr| rrrrrorrmr|C:Play Audio(12)/L:Move Medium
    "Exchange Medium," +                                //$A6| rrrrrr| rrrrrrrror|
    "," +                                               //$A7| rrrrrr| rrrrrrrrrr|
    "Read(12)/Get Message(12)," +                       //$A8| rrrrrr| rrrroororo|WCO:Read(12)/N:Get Message(12)
    "Play Audio Track Relative(12)," +                  //$A9| rrrrrr| rrrrrorrrr|
    "Write(12)/Send Message(12)," +                     //$AA| rrrrrr| rrrrorroro|WO:Write(12)/N:Send Message(12)
    "," +                                               //$AB| rrrrrr| rrrrrrrrrr|
    "Erase(12)," +                                      //$AC| rrrrrr| rrrrrrrorr|
    "," +                                               //$AD| rrrrrr| rrrrrrrrrr|
    "Write and Verify(12)," +                           //$AE| rrrrrr| rrrrorrorr|
    "Verify(12)," +                                     //$AF| rrrrrr| rrrroororr|
    "Search Data High(12)," +                           //$B0| rrrrrr| rrrroororr|
    "Search Data Equal(12)," +                          //$B1| rrrrrr| rrrroororr|
    "Search Data Low(12)," +                            //$B2| rrrrrr| rrrroororr|
    "Set Limits(12)," +                                 //$B3| rrrrrr| rrrroororr|
    "," +                                               //$B4| rrrrrr| rrrrrrrrrr|
    "Request Volume Element Address," +                 //$B5| rrrrrr| rrrrrrrror|
    "Send Volume Tag," +                                //$B6| rrrrrr| rrrrrrrror|
    "Read Defect Data(12)," +                           //$B7| rrrrrr| rrrrrrrorr|
    "Read Element Status," +                            //$B8| rrrrrr| rrrrrrrror|
    "," +                                               //$B9| rrrrrr| rrrrrrrrrr|
    "," +                                               //$BA| rrrrrr| rrrrrrrrrr|
    "," +                                               //$BB| rrrrrr| rrrrrrrrrr|
    "," +                                               //$BC| rrrrrr| rrrrrrrrrr|
    "," +                                               //$BD| rrrrrr| rrrrrrrrrr|
    "," +                                               //$BE| rrrrrr| rrrrrrrrrr|
    "," +                                               //$BF| rrrrrr| rrrrrrrrrr|
    //グループ6  ベンダ固有
    "," +                                               //$C0
    "Load/Unload SHARP MO," +                           //$C1  コマンド長6バイト
    "Assign Drive(SASI)," +                             //$C2  コマンド長6バイト
    "," +                                               //$C3
    "," +                                               //$C4
    "," +                                               //$C5
    "," +                                               //$C6
    "," +                                               //$C7
    "," +                                               //$C8
    "," +                                               //$C9
    "," +                                               //$CA
    "," +                                               //$CB
    "," +                                               //$CC
    "," +                                               //$CD
    "," +                                               //$CE
    "," +                                               //$CF
    "," +                                               //$D0
    "," +                                               //$D1
    "," +                                               //$D2
    "," +                                               //$D3
    "," +                                               //$D4
    "," +                                               //$D5
    "," +                                               //$D6
    "," +                                               //$D7
    "Read CDDA SONY," +                                 //$D8  コマンド長12バイト
    "," +                                               //$D9
    "," +                                               //$DA
    "," +                                               //$DB
    "," +                                               //$DC
    "," +                                               //$DD
    "," +                                               //$DE
    "," +                                               //$DF
    //グループ7  ベンダ固有
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    "," +
    ",").split (",", 256);

  //レジスタ名
  public static final String[] SPC_REGISTER_NAME = (
    //0    00    00    0000    00    00    00    11    11   11    11    11   11   11   11
    //1    23    45    6789    ab    cd    ef    01    23   45    67    89   ab   cd   ef
    ",BDID,,SCTL,,SCMD,,,,INTS,,PSNS,,SSTS,,SERR,,PCTL,,MBC,,DREG,,TEMP,,TCH,,TCM,,TCL,,").split (",", 32);

  //インタフェイスの有無
  public static boolean spcSCSIEXRequest;  //次回のリセットでspcSCSIEXOnに設定される値
  public static boolean spcSCSIEXOn;  //true=拡張SCSIポートあり
  public static boolean spcSCSIINOn;  //true=内蔵SCSIポートあり

  //インタフェイス
  public static SPCChip spcSCSIEXChip;  //拡張SCSIポート
  public static SPCChip spcSCSIINChip;  //内蔵SCSIポート

  //ユニット
  public static SCUnit[] spcUnitArray;  //SCSIユニットの配列

  //メニュー
  public static JMenu spcMenu;
  public static JCheckBoxMenuItem spcSCSIINMenuItem;
  public static JCheckBoxMenuItem spcSCSIEXMenuItem;

  //ファイルフィルタ
  public static javax.swing.filechooser.FileFilter spcFileFilter;  //java.io.FileFilterと紛らわしい

  //開くダイアログ
  public static OpenDialog spcOpenDialog;  //開くダイアログ。null=作る前
  public static int spcOpenUnit;  //開くユニットの番号
  public static ArrayList<File[]> spcOpenHistory;  //作る前に追加されたヒストリ

  //フォーマットダイアログ
  public static JDialog spcFormatDialog;  //ダイアログ
  public static JFileChooser2 spcFormatFileChooser;  //ファイルチューザー
  public static SpinnerNumberModel spcFormatModel;  //容量のスピナーモデル
  public static int spcFormatPartitionMegaBytes;  //容量(MB)。1MB倍して32KB加えたサイズで作成する
  public static int spcFormatBytesPerRecord;  //1レコードあたりのバイト数(2の累乗)
  public static JCheckBox spcFormatPartitioningCheckBox;  //領域確保チェックボックス
  public static boolean spcFormatPartitioningOn;  //true=領域確保する
  //protected static boolean spcFormatCopySystemFiles;  //true=システム転送
  public static JCheckBox spcFormatCopyHumanSysCheckBox;  //HUMAN.SYSチェックボックス
  public static JCheckBox spcFormatCopyCommandXCheckBox;  //COMMAND.Xチェックボックス
  public static boolean spcFormatCopyHumanSysOn;  //true=HUMAN.SYSを書き込む
  public static boolean spcFormatCopyCommandXOn;  //true=COMMAND.Xを書き込む
  public static javax.swing.filechooser.FileFilter spcFormatFilter;  //SCSI HDイメージファイルフィルタ

  //spcInit ()
  //  初期化
  public static void spcInit () {

    //インタフェイスの有無
    spcSCSIEXRequest = Settings.sgsGetOnOff ("scsiex");
    spcSCSIEXOn = false;
    spcSCSIINOn = false;

    //インタフェイス
    spcSCSIEXChip = new SPCChip (true);  //拡張SCSI
    spcSCSIINChip = new SPCChip (false);  //内蔵SCSI

    //ファイルフィルタ
    //  SASI/SCSIハードディスクのイメージファイルかどうかを調べる
    //  ファイルチューザーとドロップターゲットで使う
    spcFileFilter = new javax.swing.filechooser.FileFilter () {  //java.io.FileFilterと紛らわしい
      @Override public boolean accept (File file) {
        if (file.isDirectory ()) {
          return true;
        }
        String path = file.getPath ();
        if (spcIsInserted (path)) {  //既に挿入されている
          return false;
        }
        long longLength = file.length ();
        for (HDMedia media : HDMedia.HDM_ARRAY) {
          if (media.humDiskEndByte == longLength) {  //ファイルサイズが一致
            return true;
          }
        }
        return spcIsHds (file, true) || spcIsIso (file);  //拡張子がHDSで装置初期化されているか、拡張子がISO
      }
      @Override public String getDescription () {
        return (Multilingual.mlnJapanese ?
                "SASI/SCSI ハードディスクまたは CD-ROM のイメージファイル (*.HDF,*.HDS,*.ISO)" :
                "SASI/SCSI hard disk or CD-ROM image files (*.HDF,*.HDS,*.ISO)");
      }
    };

    //開くダイアログ
    spcOpenDialog = null;
    spcOpenUnit = 0;
    spcOpenHistory = new ArrayList<File[]> ();
    for (int i = JFileChooser2.MAXIMUM_HISTORY_COUNT - 1; 0 <= i; i--) {
      spcAddHistory (JFileChooser2.pathsToFiles (Settings.sgsGetString ("schistory" + i)));
    }

    //ユニット
    //  ダイアログが書き込み禁止でもパラメータは:Rを付けなければ書き込み禁止にしない
    spcUnitArray = new SCUnit[16];
    for (int u = 0; u < 16; u++) {
      SCUnit unit = spcUnitArray[u] = new SCUnit (u, null);
      if (HostCDROM.HCD_ENABLED &&
          HostCDROM.hcdConnected &&
          HostCDROM.hcdSCSIId == u) {
        unit.abuConnected = true;
        unit.abuDisconnectable = false;
        Multilingual.mlnToolTipText (unit.abuConnectCheckBox, "en", null, "ja", null);
        unit.abuConnectCheckBox.setSelected (true);
        unit.abuConnectCheckBox.setEnabled (false);
        unit.abuModeButton.setDisabledIcon (LnF.LNF_CD_ICON);
        unit.abuModeButton.setEnabled (false);
        unit.abuEjectButton.setEnabled (false);
        Multilingual.mlnToolTipText (unit.abuOpenButton, "en", null, "ja", null);
        unit.abuOpenButton.setEnabled (false);
        unit.abuOpenTextField.setText (HostCDROM.hcdRootPath +
                                       (Multilingual.mlnJapanese ? " (ホスト CD-ROM)" : " (Host CD-ROM)"));
        unit.abuOpenTextField.setEditable (false);
        unit.abuOpenTextField.setEnabled (true);
        unit.abuProtectCheckBox.setEnabled (false);
        continue;
      }
      String path = Settings.sgsGetString ("sc" + u);
      String hdN = Settings.sgsGetString ("hd" + u);  //SASIまたはSCSI
      if (!hdN.equals ("")) {  //hdNが指定されている
        String hdNWithoutR = hdN.endsWith (":R") || hdN.endsWith (":r") ? hdN.substring (0, hdN.length () - 2) : hdN;  //":R"を取り除く
        File file = new File (hdNWithoutR);
        if (spcIsHds (file, true) ||
            spcIsIso (file)) {  //hdNはSCSI HD/CDらしい
          path = hdN;
          Settings.sgsPutString ("hd" + u, "");  //消しておく
        }
      }
      boolean userWriteProtect = false;
      if (path.toUpperCase ().endsWith (":R")) {  //書き込み禁止モードで開く
        path = path.substring (0, path.length () - 2);
        userWriteProtect = true;
      }
      boolean hostWriteProtect = !new File (path).canWrite ();
      if (path.length () != 0) {
        unit.connect (true);  //接続されていなければ接続する
        if (unit.insert (path,
                         userWriteProtect || hostWriteProtect)) {  //挿入できた
          spcAddHistory (new File (path).getAbsoluteFile ());
        }
      }
    }

    //フォーマットダイアログ
    spcFormatDialog = null;
    spcFormatFileChooser = null;
    spcFormatPartitionMegaBytes = 100;
    spcFormatBytesPerRecord = 512;
    spcFormatPartitioningCheckBox = null;
    spcFormatPartitioningOn = true;
    //spcFormatCopySystemFiles = true;
    spcFormatCopyHumanSysCheckBox = null;
    spcFormatCopyCommandXCheckBox = null;
    spcFormatCopyHumanSysOn = true;
    spcFormatCopyCommandXOn = true;
    spcFormatFilter = new javax.swing.filechooser.FileFilter () {  //java.io.FileFilterと紛らわしい
      @Override public boolean accept (File file) {
        if (file.isDirectory ()) {
          return true;
        }
        String path = file.getPath ();
        if (spcIsInserted (path)) {  //既に挿入されている
          return false;
        }
        return spcIsHds (file, false);  //拡張子がHDSならばファイルが存在しなくてもよい
      }
      @Override public String getDescription () {
        return (Multilingual.mlnJapanese ?
                "SCSI ハードディスクのイメージファイル (*.HDS)" :
                "SCSI hard disk image files (*.HDS)");
      }
    };

  }  //spcInit()

  //spcReset ()
  public static void spcReset () {
    //拡張SCSI
    spcSCSIEXOn = spcSCSIEXRequest;
    XEiJ.busSuper (spcSCSIEXOn ? MemoryMappedDevice.MMD_EXS : MemoryMappedDevice.MMD_NUL, 0x00ea0000, 0x00ea2000);  //必要なときだけ接続する
    //内蔵SCSI
    //  内蔵SASIと同じページにあるので内蔵SCSIがなくてもページごと切り離すことはできない
    spcSCSIINOn = XEiJ.currentModel.isSCSI ();
    //    拡張  内蔵  unit0-7  unit8-15
    //    -----------------------------
    //    有効  有効    拡張     内蔵
    //    有効  無効    拡張     無効
    //    無効  有効    内蔵     無効
    //    無効  無効    無効     無効
    spcSCSIEXChip.spiReset (spcSCSIEXOn ? 0 : -8);
    spcSCSIINChip.spiReset (spcSCSIINOn ? spcSCSIEXOn ? 8 : 0 : -8);
    for (int u = 0; u < 16; u++) {
      spcUnitArray[u].scuReset (u < 8 ? spcSCSIEXOn ? spcSCSIEXChip : spcSCSIINOn ? spcSCSIINChip : null :
                                spcSCSIEXOn && spcSCSIINOn ? spcSCSIINChip : null);
    }
    if (spcSCSIEXOn) {
      MainMemory.mmrWb (0x00ed0070, MainMemory.mmrRbs (0x00ed0070) | 0x08);  //拡張フラグをセットする
    } else {
      MainMemory.mmrWb (0x00ed0070, MainMemory.mmrRbs (0x00ed0070) & ~0x08);  //拡張フラグをクリアする
    }
  }  //spcReset()

  //spcTini ()
  //  後始末
  public static void spcTini () {
    spcSCSIEXChip.spiTini ();
    spcSCSIINChip.spiTini ();

    //インタフェイスの有無
    Settings.sgsPutOnOff ("scsiex", spcSCSIEXRequest);

    //開くダイアログ
    //  開くダイアログを作らなかったときはパラメータを更新しない
    if (spcOpenDialog != null) {
      Settings.sgsPutOnOff ("screadonly", spcOpenDialog.getReadOnly ());
      Settings.sgsPutOnOff ("scappreboot", spcOpenDialog.getReboot ());
      ArrayList<String> pathsList = spcOpenDialog.getHistory ();
      int n = pathsList.size ();
      for (int i = 0; i < n; i++) {
        Settings.sgsPutString ("schistory" + i, pathsList.get (i));
      }
      for (int i = n; i < 16; i++) {
        Settings.sgsPutString ("schistory" + i, "");
      }
    }

    //ユニット
    for (int u = 0; u < 16; u++) {
      AbstractUnit unit = spcUnitArray[u];
      Settings.sgsPutString (
        "sc" + u,
        unit.abuConnected && unit.abuInserted ?
        unit.abuWriteProtected ? unit.abuPath + ":R" : unit.abuPath :
        "");
    }

  }  //spcTini()

  public static void spcMakeMenu () {

    //アクションリスナー
    ActionListener listener = new ActionListener () {
      @Override public void actionPerformed (ActionEvent ae) {
        Object source = ae.getSource ();
        String command = ae.getActionCommand ();
        switch (command) {
        case "Expansion SCSI port":  //拡張 SCSI ポート
          spcSCSIEXRequest = ((JCheckBoxMenuItem) source).isSelected ();
          break;
        case "Create new SCSI hard disk image files":  //SCSI ハードディスクのイメージファイルの新規作成
          spcOpenFormatDialog ();
          break;
        }
      }
    };

    //SCSIメニュー
    spcMenu = ComponentFactory.createMenu ("SCSI");  //横に長いとサブメニューを開きにくいので短くする
    ComponentFactory.addComponents (
      spcMenu,
      ComponentFactory.createHorizontalBox (
        Multilingual.mlnText (
          ComponentFactory.createLabel ("SCSI hard disk and CD-ROM"),
          "ja", "SCSI ハードディスクと CD-ROM")),
      ComponentFactory.createHorizontalSeparator ()
      );
    for (int u = 0; u < 16; u++) {
      spcMenu.add (spcUnitArray[u].getMenuBox ());
    }
    ComponentFactory.addComponents (
      spcMenu,
      ComponentFactory.createHorizontalSeparator (),
      spcSCSIINMenuItem = ComponentFactory.setEnabled (
        Multilingual.mlnText (
          ComponentFactory.createCheckBoxMenuItem (XEiJ.currentModel.isSCSI (), "Built-in SCSI port", listener), "ja", "内蔵 SCSI ポート"),
        false),  //機種の指定で内蔵SASIと内蔵SCSIを切り替えるので操作できないことにする
      spcSCSIEXMenuItem = Multilingual.mlnText (
        ComponentFactory.createCheckBoxMenuItem (spcSCSIEXRequest, "Expansion SCSI port", listener),
        "ja", "拡張 SCSI ポート"),
      ComponentFactory.createHorizontalSeparator (),
      Multilingual.mlnText (ComponentFactory.createMenuItem ("Create new SCSI hard disk image files", listener),
                            "ja", "SCSI ハードディスクのイメージファイルの新規作成"),
      !HostCDROM.HCD_ENABLED ? null : ComponentFactory.createHorizontalSeparator (),
      !HostCDROM.HCD_ENABLED ? null : HostCDROM.hcdMenu
      );

  }

  //inserted = spcIsInserted (path)
  //  パスで指定したファイルが既に挿入されているか調べる
  public static boolean spcIsInserted (String path) {
    for (SCUnit unit : spcUnitArray) {
      if (unit != null &&
          unit.abuConnected &&  //接続されている
          unit.abuInserted &&  //挿入されている
          unit.abuPath.equals (path)) {  //パスが一致している
        return true;  //既に挿入されている
      }
    }
    return false;  //まだ挿入されていない
  }  //spcIsInserted(String)

  static class OpenDialog extends AbstractOpenDialog {
    public OpenDialog () {
      super (XEiJ.frmFrame,
             "Open SCSI hard disk or CD-ROM image files",
             "SCSI ハードディスクまたは CD-ROM のイメージファイルを開く",
             false,  //ファイル
             spcFileFilter);
    }
    @Override public void openFiles (File[] files, boolean reboot) {
      spcOpenFiles (files, reboot);
    }
  }  //class OpenDialog

  //spcOpenFiles (list, reset)
  //  開くダイアログで選択されたファイルを開く
  public static void spcOpenFiles (File[] list, boolean reset) {
    boolean success = true;
    for (int u = spcOpenUnit, k = 0; k < list.length; ) {
      if (16 <= u) {  //ユニットが足りない
        success = false;  //失敗
        break;
      }
      SCUnit unit = spcUnitArray[u];  //ユニット
      if (!unit.abuConnected) {  //接続されていない
        u++;
        continue;
      }
      File file = list[k++];  //イメージファイル
      if (!file.isFile ()) {  //イメージファイルが存在しない
        success = false;  //失敗
        continue;
      }
      if (!unit.insert (file.getPath (),
                        spcOpenDialog.getReadOnly () || !file.canWrite ())) {  //挿入できない
        success = false;  //失敗
        continue;
      }
      u++;
    }
    if (success) {  //すべて挿入できた
      spcAddHistory (list);  //ヒストリに追加する
      if (reset) {  //ここから再起動
        if (spcOpenUnit < 8) {
          XEiJ.mpuReset (0xa000, SPC_HANDLE_EX + (spcOpenUnit << 2));  //拡張SCSIがなければ内蔵SCSIに読み替えられる
        }
      }
    }
  }  //spcOpenFiles(File[],boolean)

  //spcMakeFormatDialog ()
  //  フォーマットダイアログを作る
  //  コマンドラインのみ
  public static void spcMakeFormatDialog () {

    //アクションリスナー
    ActionListener listener = new ActionListener () {
      @Override public void actionPerformed (ActionEvent ae) {
        switch (ae.getActionCommand ()) {
        case JFileChooser.APPROVE_SELECTION:
        case "Start formatting":  //フォーマットを開始する
          {
            File[] list = spcFormatFileChooser.getSelectedFiles ();
            if (list.length > 0) {
              spcFormatDialog.setVisible (false);
              spcFormatFiles (list);
            }
          }
          break;
        case JFileChooser.CANCEL_SELECTION:
        case "Cancel":  //キャンセル
          spcFormatDialog.setVisible (false);
          break;
        case "256 bytes":  //256 バイト
          spcFormatBytesPerRecord = 256;
          break;
        case "512 bytes":  //512 バイト
          spcFormatBytesPerRecord = 512;
          break;
        case "Partitioning":
          spcFormatPartitioningOn = spcFormatPartitioningCheckBox.isSelected ();  //領域確保
          if (spcFormatPartitioningOn) {  //領域確保する
            spcFormatCopyHumanSysCheckBox.setEnabled (true);  //HUMAN.SYSを書き込むかどうか選択できる
            spcFormatCopyHumanSysCheckBox.setSelected (spcFormatCopyHumanSysOn);  //HUMAN.SYSを書き込む/書き込まない
          } else {  //領域確保しない
            spcFormatCopyHumanSysCheckBox.setEnabled (false);  //HUMAN.SYSを書き込むかどうか選択できない
            spcFormatCopyHumanSysCheckBox.setSelected (false);  //HUMAN.SYSを書き込まない
          }
          if (spcFormatPartitioningOn && spcFormatCopyHumanSysOn) {  //領域確保してHUMAN.SYSを書き込む
            spcFormatCopyCommandXCheckBox.setEnabled (true);  //COMMAND.Xを書き込むかどうか選択できる
            spcFormatCopyCommandXCheckBox.setSelected (spcFormatCopyCommandXOn);  //COMMAND.Xを書き込む/書き込まない
          } else {  //領域確保しないかHUMAN.SYSを書き込まない
            spcFormatCopyCommandXCheckBox.setEnabled (false);  //COMMAND.Xを書き込むかどうか選択できない
            spcFormatCopyCommandXCheckBox.setSelected (false);  //COMMAND.Xを書き込まない
          }
          break;
          //case "Copy System Files":  //システムファイルを転送する
          //  spcFormatCopySystemFiles = ((JCheckBox) ae.getSource ()).isSelected ();
          //  break;
        case "HUMAN.SYS":
          spcFormatCopyHumanSysOn = spcFormatCopyHumanSysCheckBox.isSelected ();  //HUMAN.SYSを書き込む/書き込まない
          if (spcFormatCopyHumanSysOn) {  //HUMAN.SYSを書き込む
            spcFormatCopyCommandXCheckBox.setEnabled (true);  //COMMAND.Xを書き込むかどうか選択できる
            spcFormatCopyCommandXCheckBox.setSelected (spcFormatCopyCommandXOn);  //COMMAND.Xを書き込む/書き込まない
          } else {  //HUMAN.SYSを書き込まない
            spcFormatCopyCommandXCheckBox.setEnabled (false);  //COMMAND.Xを書き込むかどうか選択できない
            spcFormatCopyCommandXCheckBox.setSelected (false);  //COMMAND.Xを書き込まない
          }
          break;
        case "COMMAND.X":
          spcFormatCopyCommandXOn = spcFormatCopyCommandXCheckBox.isSelected ();  //COMMAND.Xを書き込む/書き込まない
          break;
        }
      }
    };

    //ファイルチューザー
    spcMakeFormatFileChooser ();
    spcFormatFileChooser.setFileFilter (spcFormatFilter);
    spcFormatFileChooser.addActionListener (listener);

    //ダイアログ
    spcFormatModel = new SpinnerNumberModel (spcFormatPartitionMegaBytes, 1, 2047, 1);
    ButtonGroup sectorGroup = new ButtonGroup ();
    spcFormatDialog = Multilingual.mlnTitle (
      ComponentFactory.createModalDialog (
        XEiJ.frmFrame,
        "Create new SCSI hard disk image files",
        ComponentFactory.createBorderPanel (
          0, 0,
          ComponentFactory.createVerticalBox (
            spcFormatFileChooser,
            ComponentFactory.createHorizontalBox (
              Box.createHorizontalStrut (12),
              Box.createHorizontalGlue (),
              Multilingual.mlnText (ComponentFactory.createLabel ("Capacity (MB): "), "ja", "容量 (MB): "),
              ComponentFactory.createNumberSpinner (spcFormatModel, 4, new ChangeListener () {
                @Override public void stateChanged (ChangeEvent ce) {
                  spcFormatPartitionMegaBytes = spcFormatModel.getNumber ().intValue ();
                }
              }),
              Box.createHorizontalGlue (),
              Multilingual.mlnText (ComponentFactory.createLabel ("Sector size: "), "ja", "セクタサイズ: "),
              Multilingual.mlnText (ComponentFactory.createRadioButtonMenuItem (sectorGroup, spcFormatBytesPerRecord == 256, "256 bytes", listener), "ja", "256 バイト"),
              Multilingual.mlnText (ComponentFactory.createRadioButtonMenuItem (sectorGroup, spcFormatBytesPerRecord == 512, "512 bytes", listener), "ja", "512 バイト"),
              Box.createHorizontalStrut (12),
              Multilingual.mlnText (spcFormatPartitioningCheckBox = ComponentFactory.createCheckBox (spcFormatPartitioningOn, "Partitioning", listener), "ja", "領域確保"),
              Box.createHorizontalGlue (),
              Box.createHorizontalStrut (12)
              ),
            Box.createVerticalStrut (12),
            ComponentFactory.createHorizontalBox (
              Box.createHorizontalStrut (12),
              Box.createHorizontalGlue (),
              //Multilingual.mlnText (ComponentFactory.createCheckBox (spcFormatCopySystemFiles, "Copy System Files", listener), "ja", "システムファイルを転送する"),
              spcFormatCopyHumanSysCheckBox = ComponentFactory.setEnabled (
                ComponentFactory.createCheckBox (spcFormatCopyHumanSysOn, "HUMAN.SYS", listener),
                spcFormatPartitioningOn),
              Box.createHorizontalStrut (12),
              spcFormatCopyCommandXCheckBox = ComponentFactory.setEnabled (
                ComponentFactory.createCheckBox (spcFormatCopyHumanSysOn && spcFormatCopyCommandXOn, "COMMAND.X", listener),
                spcFormatPartitioningOn && spcFormatCopyHumanSysOn),
              Box.createHorizontalGlue (),
              Box.createHorizontalStrut (12),
              Multilingual.mlnText (ComponentFactory.createButton ("Start formatting", KeyEvent.VK_F, listener), "ja", "フォーマットを開始する"),
              Box.createHorizontalStrut (12),
              Multilingual.mlnText (ComponentFactory.createButton ("Cancel", KeyEvent.VK_C, listener), "ja", "キャンセル"),
              Box.createHorizontalStrut (12)
              ),
            Box.createVerticalStrut (12)
            )
          )
        ),
      "ja", "SCSI ハードディスクのイメージファイルの新規作成");

  }  //spcMakeFormatDialog()

  //spcMakeFormatFileChooser ()
  //  フォーマットファイルチューザーを作る
  public static void spcMakeFormatFileChooser () {
    if (spcFormatFileChooser == null) {
      spcFormatFileChooser = new JFileChooser2 ();
      //spcFormatFileChooser.setMultiSelectionEnabled (true);  //複数選択可能
      spcFormatFileChooser.setControlButtonsAreShown (false);  //デフォルトのボタンを消す
    }
  }

  //spcOpenFormatDialog ()
  //  フォーマットダイアログを開く
  public static void spcOpenFormatDialog () {
    if (spcFormatDialog == null) {
      spcMakeFormatDialog ();
    }
    XEiJ.pnlExitFullScreen (true);
    spcFormatDialog.setVisible (true);
  }  //spcOpenFormatDialog()

  //success = spcFormatFiles (list)
  //  SCSIハードディスクのイメージファイルを作成する
  //  コマンドラインのみ
  //  spcFormatPartitionMegaBytes
  //  spcFormatBytesPerRecord
  //  //spcFormatCopySystemFiles
  //  spcFormatPartitioningOn
  //  spcFormatCopyHumanSysOn
  //  spcFormatCopyCommandXOn
  //  を設定しておくこと
  public static boolean spcFormatFiles (File[] list) {
    boolean success = true;
  format:
    {
      SCMedia media = new SCMedia (spcFormatBytesPerRecord,  //bytesPerRecord
                                   (int) ((32768L + ((long) spcFormatPartitionMegaBytes << 20)) / spcFormatBytesPerRecord));  //diskEndRecord
      //データの配列を作る
      //  セクタ0から先頭のパーティションのシステムファイルまでがすべて収まっている
      //  パーティションの手前に32KB、IPLが1KB、FATが最大で128KB、ルートディレクトリが16KB、HUMAN.SYSが58KB、COMMAND.Xが28KB
      byte[] array = new byte[1024 * 512];  //データの配列。512KB
      media.scmMakeFormatData (array, spcFormatPartitioningOn, spcFormatCopyHumanSysOn, spcFormatCopyCommandXOn);
      //イメージファイルを作る
      int u = 0;
      for (File file : list) {
        String path = file.getPath ();
        if (!path.toUpperCase ().endsWith (".HDS")) {  //適切な拡張子が指定されていない
          path += path.endsWith (".") ? "HDS" : ".HDS";  //拡張子を付ける
          file = new File (path);
        }
        if (spcIsInserted (path)) {  //他のユニットに挿入されている
          success = false;  //失敗
          break format;
        }
        //書き出す
        if (!XEiJ.rscPutFile (path, array, 0, array.length, (int) media.humDiskEndByte)) {  //書き出せない
          success = false;  //失敗
          break format;
        }
        //空いているユニットがあれば挿入する
        while (u < 16) {
          SCUnit unit = spcUnitArray[u++];  //ユニット
          if (unit.abuConnected &&  //接続されていて
              !unit.abuInserted &&  //空いていて
              unit.insert (path,
                           false)) {  //挿入できた
            //フォーマットしたディスクの書き込みを禁止しても意味がないのでここでは書き込みを禁止しない
            break;
          }
        }
      }
    }  //format
    if (success) {  //すべてフォーマットできた
      spcAddHistory (list);  //ヒストリに追加する
    }
    return success;
  }  //spcFormatFiles(File[])

  //spcSetFat (bb, fatOffset, fat12, cluster, next)
  //  FATに書く
  public static void spcSetFat (byte[] bb, int fatOffset, boolean fat12, int cluster, int next) {
    if (fat12) {
      int i = fatOffset + 3 * (cluster >> 1);
      if ((cluster & 1) == 0) {  //偶数クラスタ(HML→ML.H..)
        bb[i    ] = (byte) next;  //ML。next&0x0ff
        bb[i + 1] = (byte) (bb[i + 1] & 0xf0 | next >> 8 & 15);  //.H。(next&0xf00)>>8
      } else {  //奇数クラスタ(hml→..l.hm)
        bb[i + 1] = (byte) (next << 4 | bb[i + 1] & 15);  //l.。(next&0x00f)<<4
        bb[i + 2] = (byte) (next >> 4);  //hm。(next&0xff0)>>4
      }
    } else {
      int i = fatOffset + (cluster << 1);
      bb[i    ] = (byte) (next >> 8);
      bb[i + 1] = (byte) next;
    }
  }  //spcSetFat(byte[],int,boolean,int,int)


  //spcAddHistory (file)
  //  ファイルをヒストリに追加する
  public static void spcAddHistory (File file) {
    spcAddHistory (new File[] { file });
  }

  //spcAddHistory (files)
  //  複数のファイルをヒストリに追加する
  public static void spcAddHistory (File[] files) {
    if (spcOpenDialog == null) {
      spcOpenHistory.add (files);
    } else {
      spcOpenDialog.addHistory (files);
    }
    spcMakeFormatFileChooser ();
    spcFormatFileChooser.addHistory (files);
    spcFormatFileChooser.selectLastFiles ();
  }


  //success = spcIsHds (file, formatted)
  //  ファイルはHDSか
  //  formatted  true=拡張子がHDSでファイルが存在して装置初期化されていなければならない
  //             false=拡張子がHDSならばファイルが存在しなくてもよい
  public static boolean spcIsHds (File file, boolean formatted) {
    //拡張子がHDSか
    String upperName = file.getName ().toUpperCase ();
    if (!(upperName.endsWith (".HDS") ||
          upperName.endsWith (".DIM.001"))) {
      return false;
    }
    //ファイルが存在するか
    if (!formatted) {
      return true;
    }
    if (!file.isFile ()) {
      return false;
    }
    //ファイルサイズが64KB以上かつ256で割り切れるか
    long longLength = file.length ();  //ファイルサイズ
    if (longLength < (long) (1024 * 64) ||
        (longLength & 255) != 0) {
      return false;
    }
    //先頭512バイトを読み込む
    byte[] bb = new byte[512];
    try (BufferedInputStream bis = new BufferedInputStream (new FileInputStream (file))) {
      if (bis.read (bb, 0, 512) != 512) {
        return false;
      }
    } catch (IOException ioe) {
      return false;
    }
    //装置初期化されているか
    //  0000  q  X68SCSI1マジック
    //  0008  w  1レコードあたりのバイト数(2の累乗)
    //  000a  l  ディスクのレコード数
    //  000e  b  Read(10),Write(10)が動作するか
    //  000f  b  デバイスタイプ
    if (ByteArray.byaRls (bb, 0) == ('X' << 24 | '6' << 16 | '8' << 8 | 'S') &&
        ByteArray.byaRls (bb, 4) == ('C' << 24 | 'S' << 16 | 'I' << 8 | '1')) {
      //SxSIか
      boolean sxsi = ByteArray.byaRls (bb, 42) == ('S' << 24 | 'x' << 16 | 'S' << 8 | 'I');
      //1レコードあたりのバイト数
      int bytesPerRecord = ByteArray.byaRwz (bb, 8);
      //1レコードあたりのバイト数が256または512か
      if (bytesPerRecord != 256 &&
          bytesPerRecord != 512) {
        return false;
      }
      //ディスクのレコード数
      long diskEndRecord = (long) ByteArray.byaRls (bb, 10) & 0xffffffffL;
      //SxSIは1レコードあたりのバイト数が512だがディスクのレコード数は1024で計算されている
      if (sxsi) {
        diskEndRecord <<= 1;
      }
      //ディスクのバイト数
      long diskEndByte = (long) bytesPerRecord * diskEndRecord;
      //ディスクのバイト数とファイルサイズが一致しているか
      //  FORMAT.XはRead Capacityが返す最終論理ブロックアドレスを論理ブロック数と誤解して1ブロック少ない容量で装置初期化を行う
      return (longLength == diskEndByte ||
              longLength == diskEndByte + (long) bytesPerRecord ||
              longLength == diskEndByte + (long) (bytesPerRecord << 1));
    }
    //IBMスーパーフロッピーか
    //  0000  b  0xeb
    //  01fe  b  0x55
    //  01ff  b  0xaa
    if (bb[0x0000] == (byte) 0xeb &&
        bb[0x01fe] == (byte) 0x55 &&
        bb[0x01ff] == (byte) 0xaa) {  //IBMスーパーフロッピー
      return true;
    }
    return false;
  }  //spcIsHds(File,boolean)

  //spcIsIso (file)
  //  ファイルはISOか
  //  参考
  //    http://euc.jp/periphs/iso9660.ja.html
  public static boolean spcIsIso (File file) {
    String upperName = file.getName ().toUpperCase ();
    if (upperName.endsWith (".ISO")) {  //拡張子がISO
      if (file.isFile ()) {  //ファイルが存在する
        long longLength = file.length ();  //ファイルサイズ
        if (((int) longLength & 2047) == 0 &&  //ファイルサイズが2048で割り切れる
            (long) (2048 * 18) <= longLength) {  //ファイルサイズが18セクタ以上
          try (DataInputStream dis = new DataInputStream (new BufferedInputStream (new FileInputStream (file)))) {
            dis.skipBytes (2048 * 16);  //セクタ16まで読み飛ばす
            long id = dis.readLong ();  //0  ボリューム記述子種別と規格識別子とボリューム記述子版数とボリュームフラグ
            dis.skipBytes (84 - 8);
            int diskEndRecord = dis.readInt ();  //84  ボリュームのブロック数
            dis.skipBytes (130 - 88);
            int bytesPerRecord = (int) dis.readChar ();  //130  ブロックのバイト数
            long diskEndByte = (long) bytesPerRecord * (long) diskEndRecord;  //ボリュームのバイト数
            if (id >>> 8 == ((long) 0x01 << 48 |
                             (long) 'C' << 40 | (long) 'D' << 32 | (long) '0' << 24 | (long) '0' << 16 | (long) '1' << 8 |
                             (long) 0x01) &&  //ボリューム記述子種別と規格識別子とボリューム記述子版数が一致
                bytesPerRecord == 2048 &&  //ブロックのバイト数が2048
                0 <= diskEndRecord &&  //ボリュームのブロック数が31bitに収まっている
                diskEndByte <= longLength) {  //ボリュームのバイト数がファイルサイズ以下。一致しているとは限らない
              return true;
            }
          } catch (IOException ioe) {
          }
        }  //if ファイルサイズ
      }  //if ファイルが存在する
    }  //if 拡張子がISO
    return false;
  }  //spcIsIso(File)



  //========================================================================================
  //SCSIフォーマットデータ
  //  無償公開されたHuman68k version 3.02のシステムディスクに入っているFORMAT.Xから抽出したデータを使う

  //----------------------------------------------------------------------------------------
  //SCSIディスクID
  //  SCSIディスクのセクタ0に書き込まれる
/*
  public static final int[] SPC_DISK_ID_1 = {
    //  perl -e "do'sjdump.pl';$p=0;$m=2;$o=0x3ce4;$l=8;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    0x58,0x36,0x38,0x53,0x43,0x53,0x49,0x31,                                          //00000000  X68SCSI1        
  };
*/
  //  perl misc/itob.pl xeij/SPC.java SPC_DISK_ID_1
  public static final byte[] SPC_DISK_ID_1 = "X68SCSI1".getBytes (XEiJ.ISO_8859_1);
/*
  public static final int[] SPC_DISK_ID_2 = {
    //  perl -e "do'sjdump.pl';$p=0x10;$m=2;$o=0x669d;$l=35;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    0x48,0x75,0x6d,0x61,0x6e,0x36,0x38,0x4b,0x20,0x53,0x43,0x53,0x49,0x2d,0x44,0x49,  //00000010  Human68K SCSI-DI
    0x53,0x4b,0x20,0x62,0x79,0x20,0x4b,0x65,0x69,0x73,0x6f,0x6b,0x75,0x20,0x47,0x69,  //00000020  SK by Keisoku Gi
    0x6b,0x65,0x6e,                                                                   //00000030  ken             
  };
*/
  //  perl misc/itob.pl xeij/SPC.java SPC_DISK_ID_2
  public static final byte[] SPC_DISK_ID_2 = "Human68K SCSI-DISK by Keisoku Giken".getBytes (XEiJ.ISO_8859_1);

  //----------------------------------------------------------------------------------------
  //SCSIディスクIPL
  //  SCSIディスクのセクタ1に書き込まれる
  //  HELPキーが押されていたらメニューを表示する
  //  選択されたパーティションのパーティションIPLを読み込んで起動する
/*
  public static final int[] SPC_DISK_IPL = {
    //  perl -e "do'sjdump.pl';$p=0x400;$m=2;$o=0x77da;$l=0x7bc2-$o;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    0x60,0x00,0x00,0xca,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000400  `..ﾊ............
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000410  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000420  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000430  ................
    0x1a,0x1b,0x5b,0x36,0x3b,0x33,0x32,0x48,0x58,0x36,0x38,0x30,0x30,0x30,0x20,0x53,  //00000440  ..[6;32HX68000 S
    0x43,0x53,0x49,0x20,0x44,0x49,0x53,0x4b,0x20,0x49,0x50,0x4c,0x20,0x4d,0x45,0x4e,  //00000450  CSI DISK IPL MEN
    0x55,0x1b,0x5b,0x32,0x35,0x3b,0x32,0x32,0x48,0x83,0x4a,0x81,0x5b,0x83,0x5c,0x83,  //00000460  U.[25;22Hカーソル
    0x8b,0x83,0x4c,0x81,0x5b,0x82,0xc5,0x91,0x49,0x91,0xf0,0x82,0xb5,0x82,0xc4,0x83,  //00000470   キーで選択してリ
    0x8a,0x83,0x5e,0x81,0x5b,0x83,0x93,0x83,0x4c,0x81,0x5b,0x82,0xf0,0x89,0x9f,0x82,  //00000480   ターンキーを押し
    0xb5,0x82,0xc4,0x82,0xad,0x82,0xbe,0x82,0xb3,0x82,0xa2,0x00,0x1b,0x5b,0x32,0x36,  //00000490   てください..[26
    0x3b,0x32,0x38,0x48,0x91,0x49,0x91,0xf0,0x82,0xb5,0x82,0xbd,0x82,0xe0,0x82,0xcc,  //000004a0  ;28H選択したもの
    0x82,0xf0,0x8e,0xa9,0x93,0xae,0x8b,0x4e,0x93,0xae,0x82,0xc6,0x82,0xb5,0x82,0xc4,  //000004b0  を自動起動として
    0x93,0x6f,0x98,0x5e,0x82,0xb5,0x82,0xdc,0x82,0xb7,0x00,0x00,0x72,0x00,0x70,0x04,  //000004c0  登録します..r.p.
    0x4e,0x4f,0x08,0x00,0x00,0x01,0x67,0x02,0x4e,0x75,0x4f,0xfa,0xff,0x24,0x42,0x85,  //000004d0  NO....g.NuO..$B.
    0x70,0xf5,0x72,0x25,0x43,0xfa,0x03,0x02,0x4e,0x4f,0x4a,0x00,0x66,0x00,0x01,0x92,  //000004e0  p.r%C...NOJ.f...
    0x22,0x29,0x00,0x04,0xe0,0x89,0xe2,0x89,0x43,0xfa,0x02,0xea,0x22,0x81,0x74,0x02,  //000004f0  ")..煢竕C..."》.
    0x26,0x3c,0x00,0x00,0x04,0x00,0x43,0xfa,0x02,0xe0,0x61,0x00,0x02,0x3a,0x4a,0x00,  //00000500  &<....C..濛..:J.
    0x66,0x00,0x01,0x6e,0x43,0xfa,0x02,0xd2,0x47,0xfa,0xfe,0xea,0x0c,0x91,0x58,0x36,  //00000510  f..nC..ﾒG....噌6
    0x38,0x4b,0x66,0x00,0x01,0x70,0x74,0x0e,0x42,0x43,0x42,0x47,0x42,0x86,0x43,0xe9,  //00000520  8Kf..pt.BCBGB.C.
    0x00,0x10,0x4a,0x11,0x67,0x16,0x52,0x46,0x26,0xc9,0x10,0x29,0x00,0x08,0x08,0x00,  //00000530  ..J.g.RF&ﾉ.)....
    0x00,0x00,0x66,0x08,0x52,0x43,0x4a,0x00,0x66,0x02,0x52,0x47,0x51,0xca,0xff,0xe0,  //00000540  ..f.RCJ.f.RGQﾊ.濳
    0x4a,0x43,0x67,0x00,0x01,0x46,0x72,0x0a,0x70,0x04,0x4e,0x4f,0x08,0x00,0x00,0x04,  //00000550   Cg..Fr.p.NO....
    0x66,0x12,0x4a,0x47,0x67,0x0e,0x53,0x47,0x67,0x1c,0x43,0xfa,0xfe,0xd4,0x61,0x00,  //00000560  f.JGg.SGg.C..ﾔa.
    0x01,0xc8,0x60,0x28,0x43,0xfa,0xfe,0xca,0x61,0x00,0x01,0xbe,0x43,0xfa,0xff,0x1e,  //00000570  .ﾈ`(C..ﾊa..ｾC...
    0x61,0x00,0x01,0xb6,0x60,0x14,0x47,0xfa,0xfe,0x7c,0x20,0x5b,0x24,0x28,0x00,0x08,  //00000580  a..ｶ`.G..| [$(..
    0x4a,0x28,0x00,0x08,0x66,0xf4,0x60,0x00,0x00,0xc2,0x7a,0x02,0x42,0x43,0x45,0xfa,  //00000590  J(..f.`..ﾂz.BCE.
    0xfe,0x64,0x22,0x52,0x10,0x29,0x00,0x08,0x67,0x0a,0xb0,0x05,0x67,0x06,0x72,0x02,  //000005a0  .d"R.)..g.ｰ.g.r.
    0x61,0x00,0x01,0x7a,0x61,0x00,0x01,0x1a,0x58,0x8a,0x52,0x43,0xb6,0x46,0x65,0xe2,  //000005b0  a..za...X崖CｶFe秡
    0x60,0x2a,0x61,0x00,0x01,0x0a,0x61,0x00,0x00,0xf6,0xb0,0x3c,0x00,0x1d,0x67,0x3a,  //000005c0   *a...a...ｰ<..g:
    0xb0,0x3c,0x00,0x35,0x67,0x0c,0xb0,0x3c,0x00,0x3c,0x67,0x1a,0xb0,0x3c,0x00,0x3e,  //000005d0  ｰ<.5g.ｰ<.<g.ｰ<.>
    0x66,0xe4,0x61,0x00,0x00,0xec,0x52,0x43,0xb6,0x46,0x65,0x02,0x42,0x43,0x61,0x00,  //000005e0  f臑...RCｶFe.BCa.
    0x00,0xb6,0x66,0xf2,0x60,0xcc,0x61,0x00,0x00,0xd8,0x53,0x43,0x6a,0x04,0x36,0x06,  //000005f0  .ｶf.`ﾌa..ﾘSCj.6.
    0x53,0x43,0x61,0x00,0x00,0xa2,0x66,0xf2,0x60,0xb8,0x47,0xfa,0xfd,0xf8,0xe5,0x43,  //00000600  SCa..｢f.`ｸG...蕕
    0x20,0x73,0x30,0x00,0x24,0x28,0x00,0x08,0x4a,0x05,0x67,0x3e,0x43,0xfa,0x01,0xd2,  //00000610   s0.$(..J.g>C..ﾒ
    0x72,0x0e,0x43,0xe9,0x00,0x10,0x4a,0x29,0xff,0xf8,0x67,0x12,0x20,0x11,0x08,0x00,  //00000620  r.C...J)..g. ...
    0x00,0x18,0x66,0x0a,0x42,0x11,0xb4,0x80,0x67,0x04,0x12,0xbc,0x00,0x02,0x51,0xc9,  //00000630  ..f.B.ｴ.g..ｼ..Qﾉ
    0xff,0xe2,0x2f,0x02,0x74,0x02,0x26,0x3c,0x00,0x00,0x04,0x00,0x43,0xfa,0x01,0x9a,  //00000640  ../.t.&<....C..啾
    0x61,0x00,0x00,0xec,0x24,0x1f,0x4a,0x00,0x66,0x26,0xc4,0xbc,0x00,0xff,0xff,0xff,  //00000650   ...$.J.f&ﾄｼ....
    0x26,0x3c,0x00,0x00,0x04,0x00,0x43,0xfa,0xfd,0x98,0xd3,0xfc,0x00,0x00,0x04,0x00,  //00000660  &<....C..侖.....
    0x61,0x00,0x00,0xd4,0x4a,0x00,0x66,0x08,0x0c,0x11,0x00,0x60,0x66,0x22,0x4e,0xd1,  //00000670  a..ﾔJ.f....`f"Nﾑ
    0x45,0xfa,0x00,0xea,0x43,0xfa,0x00,0xdc,0x61,0x00,0x00,0xae,0x22,0x4a,0x61,0x00,  //00000680  E..鵑..ﾜa..ｮ"Ja.
    0x00,0xa8,0x60,0xfe,0x45,0xfa,0x00,0xf5,0x60,0xea,0x45,0xfa,0x01,0x0c,0x60,0xe4,  //00000690  .ｨ`.E...`鵙...`胼
    0x45,0xfa,0x01,0x23,0x60,0xde,0x41,0xfa,0xfd,0x5c,0x20,0x03,0xe5,0x40,0x20,0x70,  //000006a0   ..#`ﾞA..\ .蕁 p
    0x00,0x00,0x10,0x28,0x00,0x08,0xb0,0x05,0x67,0x02,0x4a,0x00,0x4e,0x75,0x42,0x80,  //000006b0  ...(..ｰ.g.J.NuB.
    0x4e,0x4f,0xe0,0x48,0xb0,0x3c,0x00,0x4e,0x66,0x02,0x70,0x1d,0x4e,0x75,0x61,0x5a,  //000006c0  NO潯ｰ<.Nf.p.NuaZ
    0x43,0xfa,0xfd,0x32,0x30,0x03,0xe5,0x40,0x43,0xf1,0x00,0x00,0x22,0x51,0x72,0x24,  //000006d0  C..20.蕁C..."Qr$
    0x74,0x09,0xd4,0x43,0x70,0x23,0x4e,0x4f,0x72,0x28,0x61,0x46,0x24,0x09,0x41,0xfa,  //000006e0  t.ﾔCp#NOr(aF$.A.
    0x00,0xf8,0x94,0x88,0xe8,0x8a,0x84,0xfc,0x00,0x0a,0xd4,0xbc,0x00,0x30,0x00,0x30,  //000006f0  ..蝿闃....ﾔｼ.0.0
    0x72,0x20,0xb4,0x7c,0x00,0x30,0x67,0x02,0x32,0x02,0x61,0x26,0x48,0x42,0x32,0x02,  //00000700  r ｴ|.0g.2.a&HB2.
    0x61,0x20,0x72,0x29,0x61,0x1c,0x72,0x20,0x61,0x18,0x74,0x07,0x42,0x41,0x12,0x19,  //00000710  a r)a.r a.t.BA..
    0x61,0x10,0x51,0xca,0xff,0xf8,0x72,0x03,0x60,0x02,0x72,0x0b,0x70,0x22,0x4e,0x4f,  //00000720  a.Qﾊ..r.`.r.p"NO
    0x4e,0x75,0x70,0x20,0x4e,0x4f,0x4e,0x75,0x70,0x21,0x4e,0x4f,0x4e,0x75,0x48,0xe7,  //00000730  Nup NONup!NONuH轎
    0x7c,0x00,0x72,0x22,0x60,0x06,0x48,0xe7,0x7c,0x00,0x72,0x21,0x2a,0x3a,0x00,0x96,  //00000740   .r"`.H轎.r!*:.籾
    0xe0,0x8b,0xea,0xab,0xe5,0x8a,0xea,0xaa,0x70,0xf5,0x4e,0x4f,0x4c,0xdf,0x00,0x3e,  //00000750   苦ｫ蜉.ｪp.NOLﾟ.>
    0x4e,0x75,0x1a,0x1b,0x5b,0x31,0x36,0x3b,0x33,0x33,0x48,0x00,0x20,0x20,0x82,0x72,  //00000760  Nu..[16;33H.  Ｓ
    0x82,0x62,0x82,0x72,0x82,0x68,0x83,0x66,0x83,0x42,0x83,0x58,0x83,0x4e,0x82,0xaa,  //00000770  ＣＳＩディスクが
    0x93,0xc7,0x82,0xdf,0x82,0xdc,0x82,0xb9,0x82,0xf1,0x00,0x20,0x20,0x8a,0xc7,0x97,  //00000780  読めません.  管理
    0x9d,0x83,0x75,0x83,0x8d,0x83,0x62,0x83,0x4e,0x82,0xaa,0x89,0xf3,0x82,0xea,0x82,  //00000790   ブロックが壊れて
    0xc4,0x82,0xa2,0x82,0xdc,0x82,0xb7,0x00,0x20,0x20,0x8b,0x4e,0x93,0xae,0x89,0xc2,  //000007a0   います.  起動可
    0x94,0x5c,0x82,0xc8,0x97,0xcc,0x88,0xe6,0x82,0xaa,0x82,0xa0,0x82,0xe8,0x82,0xdc,  //000007b0  能な領域がありま
    0x82,0xb9,0x82,0xf1,0x00,0x82,0x68,0x82,0x6f,0x82,0x6b,0x83,0x75,0x83,0x8d,0x83,  //000007c0  せん.ＩＰＬブロッ
    0x62,0x83,0x4e,0x82,0xcc,0x93,0xe0,0x97,0x65,0x82,0xaa,0x88,0xd9,0x8f,0xed,0x82,  //000007d0   クの内容が異常で
    0xc5,0x82,0xb7,0x00,0x00,0x00,0x00,0x00,                                          //000007e0   す.....        
  };
*/
  //  perl misc/itob.pl xeij/SPC.java SPC_DISK_IPL
  public static final byte[] SPC_DISK_IPL = "`\0\0\312\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\32\33[6;32HX68000 SCSI DISK IPL MENU\33[25;22H\203J\201[\203\\\203\213\203L\201[\202\305\221I\221\360\202\265\202\304\203\212\203^\201[\203\223\203L\201[\202\360\211\237\202\265\202\304\202\255\202\276\202\263\202\242\0\33[26;28H\221I\221\360\202\265\202\275\202\340\202\314\202\360\216\251\223\256\213N\223\256\202\306\202\265\202\304\223o\230^\202\265\202\334\202\267\0\0r\0p\4NO\b\0\0\1g\2NuO\372\377$B\205p\365r%C\372\3\2NOJ\0f\0\1\222\")\0\4\340\211\342\211C\372\2\352\"\201t\2&<\0\0\4\0C\372\2\340a\0\2:J\0f\0\1nC\372\2\322G\372\376\352\f\221X68Kf\0\1pt\16BCBGB\206C\351\0\20J\21g\26RF&\311\20)\0\b\b\0\0\0f\bRCJ\0f\2RGQ\312\377\340JCg\0\1Fr\np\4NO\b\0\0\4f\22JGg\16SGg\34C\372\376\324a\0\1\310`(C\372\376\312a\0\1\276C\372\377\36a\0\1\266`\24G\372\376| [$(\0\bJ(\0\bf\364`\0\0\302z\2BCE\372\376d\"R\20)\0\bg\n\260\5g\6r\2a\0\1za\0\1\32X\212RC\266Fe\342`*a\0\1\na\0\0\366\260<\0\35g:\260<\0005g\f\260<\0<g\32\260<\0>f\344a\0\0\354RC\266Fe\2BCa\0\0\266f\362`\314a\0\0\330SCj\0046\6SCa\0\0\242f\362`\270G\372\375\370\345C s0\0$(\0\bJ\5g>C\372\1\322r\16C\351\0\20J)\377\370g\22 \21\b\0\0\30f\nB\21\264\200g\4\22\274\0\2Q\311\377\342/\2t\2&<\0\0\4\0C\372\1\232a\0\0\354$\37J\0f&\304\274\0\377\377\377&<\0\0\4\0C\372\375\230\323\374\0\0\4\0a\0\0\324J\0f\b\f\21\0`f\"N\321E\372\0\352C\372\0\334a\0\0\256\"Ja\0\0\250`\376E\372\0\365`\352E\372\1\f`\344E\372\1#`\336A\372\375\\ \3\345@ p\0\0\20(\0\b\260\5g\2J\0NuB\200NO\340H\260<\0Nf\2p\35NuaZC\372\37520\3\345@C\361\0\0\"Qr$t\t\324Cp#NOr(aF$\tA\372\0\370\224\210\350\212\204\374\0\n\324\274\0000\0000r \264|\0000g\0022\2a&HB2\2a r)a\34r a\30t\7BA\22\31a\20Q\312\377\370r\3`\2r\13p\"NONup NONup!NONuH\347|\0r\"`\6H\347|\0r!*:\0\226\340\213\352\253\345\212\352\252p\365NOL\337\0>Nu\32\33[16;33H\0  \202r\202b\202r\202h\203f\203B\203X\203N\202\252\223\307\202\337\202\334\202\271\202\361\0  \212\307\227\235\203u\203\215\203b\203N\202\252\211\363\202\352\202\304\202\242\202\334\202\267\0  \213N\223\256\211\302\224\\\202\310\227\314\210\346\202\252\202\240\202\350\202\334\202\271\202\361\0\202h\202o\202k\203u\203\215\203b\203N\202\314\223\340\227e\202\252\210\331\217\355\202\305\202\267\0\0\0\0\0".getBytes (XEiJ.ISO_8859_1);

  //----------------------------------------------------------------------------------------
  //SCSIデバイスドライバ
  //  ROMのデバイスドライバが古いとき置き換えて使用する
/*
  public static final int[] SPC_DEVICE_DRIVER = {
    //  perl -e "do'sjdump.pl';$p=0xc00;$m=2;$o=0x7e88;$l=0xb1e0-$o;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    0xff,0xff,0xff,0xff,0x40,0x00,0x00,0x00,0x00,0x6e,0x00,0x00,0x00,0x80,0x01,0x53,  //00000c00  ....@....n.....S
    0x43,0x48,0x44,0x49,0x53,0x4b,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x46,0x00,0x00,  //00000c10  CHDISK.......F..
    0x03,0x46,0x00,0x00,0x0d,0x0c,0x00,0x00,0x00,0xf8,0x00,0x00,0x07,0x9e,0x00,0x00,  //00000c20  .F..............
    0x03,0xb0,0x00,0x00,0x00,0xb8,0x00,0x00,0x00,0xb8,0x00,0x00,0x08,0x14,0x00,0x00,  //00000c30  .ｰ...ｸ...ｸ......
    0x08,0x06,0x00,0x00,0x00,0xb8,0x00,0x00,0x00,0xb8,0x00,0x00,0x01,0x2c,0x00,0x00,  //00000c40  .....ｸ...ｸ...,..
    0x00,0x01,0x70,0x02,0x70,0x07,0x70,0x0c,0x70,0x08,0x00,0x01,0x70,0x0d,0x70,0x0c,  //00000c50  ..p.p.p.p...p.p.
    0x70,0x0c,0x70,0x0c,0x70,0x0c,0x70,0x0c,0x70,0x0c,0x70,0x0c,0x70,0x0c,0x48,0xe7,  //00000c60  p.p.p.p.p.p.p.H.
    0x00,0x02,0x4d,0xfa,0x0d,0xc8,0x2d,0x4d,0x00,0x00,0x4c,0xdf,0x40,0x00,0x4e,0x75,  //00000c70  ..M..ﾈ-M..Lﾟ@.Nu
    0x48,0xe7,0x7f,0xfe,0x4d,0xfa,0x0d,0xb6,0x2a,0x6e,0x00,0x00,0x70,0x00,0x10,0x2d,  //00000c80  H...M..ｶ*n..p..-
    0x00,0x02,0x0c,0x00,0x00,0x0c,0x62,0x1a,0x22,0x6d,0x00,0x0e,0x41,0xfa,0xff,0x7c,  //00000c90  ......b."m..A..|
    0xd0,0x40,0xd0,0x40,0xd1,0xc0,0x20,0x50,0x20,0x08,0x41,0xfa,0xff,0x54,0xd1,0xc0,  //00000ca0  ﾐ@ﾐ@ﾑﾀ P .A..Tﾑﾀ
    0x4e,0xd0,0x30,0x3c,0x70,0x0c,0x60,0x02,0x42,0x40,0x32,0x00,0x1b,0x41,0x00,0x03,  //00000cb0  Nﾐ0<p.`.B@2..A..
    0xe0,0x49,0x1b,0x41,0x00,0x04,0x4a,0x40,0x67,0x12,0x4a,0x6e,0x05,0xac,0x66,0x06,  //00000cc0  潛.A..J@g.Jn.ｬf.
    0x4a,0x6e,0x05,0xae,0x67,0x06,0x3d,0x7c,0xff,0xff,0x05,0xb2,0x4c,0xdf,0x7f,0xfe,  //00000cd0  Jn.ｮg.=|...ｲLﾟ..
    0x4e,0x75,0x4a,0xae,0x05,0xca,0x67,0xca,0x22,0x6e,0x05,0xca,0x70,0x80,0x22,0x3c,  //00000ce0  NuJｮ.ﾊgﾊ"n.ﾊp."<
    0x00,0x00,0x01,0xf5,0x4e,0x4f,0x60,0xba,0x20,0x6d,0x00,0x0e,0x20,0x2d,0x00,0x12,  //00000cf0  ....NO`ｺ m.. -..
    0x0c,0x80,0x00,0x00,0x00,0x08,0x67,0x14,0x0c,0x80,0x00,0x00,0x00,0x04,0x66,0xa8,  //00000d00  ......g.......fｨ
    0x30,0xee,0x05,0xc4,0x20,0x2e,0x05,0xa8,0x30,0x80,0x60,0x9c,0x30,0xee,0x05,0xc4,  //00000d10  0..ﾄ ..ｨ0.`.0..ﾄ
    0x20,0x2e,0x05,0xa8,0x30,0xc0,0x20,0xae,0x05,0xc6,0x60,0x8c,0x20,0x6d,0x00,0x0e,  //00000d20   ..ｨ0ﾀ ｮ.ﾆ`. m..
    0x20,0x2d,0x00,0x12,0x0c,0x80,0x00,0x00,0x00,0x02,0x66,0x00,0xff,0x7c,0x3d,0x50,  //00000d30   -........f..|=P
    0x05,0xc4,0x60,0x00,0xff,0x74,0x0c,0x2d,0x00,0x17,0x00,0x16,0x64,0x00,0xff,0x64,  //00000d40  .ﾄ`..t.-....d..d
    0x53,0x82,0x02,0x82,0x00,0x00,0x00,0x0f,0x2d,0x42,0x05,0xa8,0x05,0x39,0x00,0x00,  //00000d50  S.......-B.ｨ.9..
    0x0c,0xec,0x66,0x00,0xff,0x4e,0x61,0x00,0x1a,0xb8,0x2d,0x40,0x05,0xca,0x20,0x3c,  //00000d60  ..f..Na..ｸ-@.ﾊ <
    0x00,0x00,0x00,0xf5,0x72,0x24,0x28,0x2e,0x05,0xa8,0x4e,0x4f,0xb0,0xbc,0x00,0x00,  //00000d70  ....r$(..ｨNOｰｼ..
    0x00,0x00,0x67,0x64,0xb0,0xbc,0xff,0xff,0xff,0xff,0x67,0x00,0xff,0x56,0xb0,0xbc,  //00000d80  ..gdｰｼ....g..Vｰｼ
    0x00,0x00,0x00,0x08,0x67,0x50,0xb0,0xbc,0x00,0x00,0x00,0x02,0x66,0x00,0xff,0x44,  //00000d90  ....gPｰｼ....f..D
    0x20,0x3c,0x00,0x00,0x00,0xf5,0x72,0x2c,0x76,0x0e,0x28,0x2e,0x05,0xa8,0x43,0xee,  //00000da0   <....r,v.(..ｨC.
    0x05,0xd2,0x4e,0x4f,0xb0,0xbc,0x00,0x00,0x00,0x00,0x66,0x00,0xff,0x26,0x43,0xee,  //00000db0  .ﾒNOｰｼ....f..&C.
    0x05,0xd2,0x10,0x11,0x02,0x00,0x00,0x70,0x0c,0x00,0x00,0x70,0x66,0x00,0xff,0x14,  //00000dc0  .ﾒ.....p...pf...
    0x10,0x29,0x00,0x02,0x67,0x10,0xb0,0x3c,0x00,0x01,0x67,0x0a,0xb0,0x3c,0x00,0x06,  //00000dd0  .)..g.ｰ<..g.ｰ<..
    0x67,0x04,0x60,0x00,0xfe,0xfe,0x60,0x86,0x70,0xf5,0x72,0x2b,0x28,0x2e,0x05,0xa8,  //00000de0  g.`...`.p.r+(..ｨ
    0x4e,0x4f,0x4a,0x80,0x66,0x00,0xfe,0xec,0x70,0xf5,0x72,0x25,0x28,0x2e,0x05,0xa8,  //00000df0  NOJ.f...p.r%(..ｨ
    0x43,0xee,0x05,0xd2,0x4e,0x4f,0x4a,0x80,0x66,0x00,0xfe,0xd8,0x43,0xee,0x05,0xd2,  //00000e00  C..ﾒNOJ.f..ﾘC..ﾒ
    0x22,0x29,0x00,0x04,0xb2,0xbc,0x00,0x00,0x04,0x00,0x67,0x18,0xb2,0xbc,0x00,0x00,  //00000e10  ")..ｲｼ....g.ｲｼ..
    0x02,0x00,0x67,0x08,0x3d,0x7c,0x00,0x02,0x05,0xa0,0x60,0x0e,0x3d,0x7c,0x00,0x01,  //00000e20  ..g.=|....`.=|..
    0x05,0xa0,0x60,0x06,0x3d,0x7c,0x00,0x00,0x05,0xa0,0xe0,0x89,0xe2,0x89,0x2d,0x41,  //00000e30  ..`.=|....煢竕-A
    0x05,0xa4,0x43,0xee,0x05,0xd2,0x2a,0x2e,0x05,0xa4,0x74,0x00,0xe5,0x8a,0xea,0xaa,  //00000e40  .､C..ﾒ*..､t.蜉.ｪ
    0x76,0x01,0xe5,0x8b,0xea,0xab,0x28,0x2e,0x05,0xa8,0x70,0xf5,0x72,0x21,0x4e,0x4f,  //00000e50  v.蜍.ｫ(..ｨp.r!NO
    0x4a,0x80,0x66,0x00,0xfe,0x7e,0x0c,0x91,0x58,0x36,0x38,0x53,0x66,0x00,0xfe,0x74,  //00000e60  J.f..~.噌68Sf..t
    0x0c,0xa9,0x43,0x53,0x49,0x31,0x00,0x04,0x66,0x00,0xfe,0x68,0x1d,0x69,0x00,0x0e,  //00000e70  .ｩCSI1..f..h.i..
    0x05,0xb0,0x3d,0x7c,0x00,0x00,0x05,0xac,0x3d,0x7c,0x00,0x00,0x05,0xae,0x0c,0x29,  //00000e80  .ｰ=|...ｬ=|...ｮ.)
    0x00,0x01,0x00,0x0f,0x67,0x10,0x0c,0x29,0x00,0x02,0x00,0x0f,0x66,0x0e,0x3d,0x7c,  //00000e90  ....g..)....f.=|
    0xff,0xff,0x05,0xae,0x60,0x06,0x3d,0x7c,0xff,0xff,0x05,0xac,0x61,0x00,0x03,0xb6,  //00000ea0  ...ｮ`.=|...ｬa..ｶ
    0x4a,0x80,0x66,0x00,0xfe,0x2e,0x4a,0x46,0x67,0x00,0xfe,0x28,0x3d,0x7c,0x00,0x00,  //00000eb0  J.f...JFg..(=|..
    0x05,0xb2,0x3d,0x46,0x05,0xb4,0x4a,0xae,0x05,0xca,0x67,0x06,0x41,0xfa,0x30,0x8a,  //00000ec0  .ｲ=F.ｴJｮ.ﾊg.A.0柿
    0x60,0x04,0x41,0xfa,0x19,0x4c,0x2d,0x48,0x05,0xce,0xd1,0xfc,0x00,0x00,0x10,0x00,  //00000ed0   .A..L-H.ﾎﾑ.....
    0x2b,0x48,0x00,0x0e,0x10,0x2d,0x00,0x16,0xd0,0x06,0x04,0x00,0x00,0x17,0x65,0x02,  //00000ee0  +H...-..ﾐ.....e.
    0x9c,0x00,0x1b,0x46,0x00,0x0d,0x41,0xee,0x05,0x44,0x2b,0x48,0x00,0x12,0x43,0xee,  //00000ef0  ...F..A..D+H..C.
    0x04,0x04,0x70,0x0e,0x20,0xc9,0x43,0xe9,0x00,0x14,0x51,0xc8,0xff,0xf8,0x1b,0x7a,  //00000f00  ..p. ﾉC...Qﾈ...z
    0xfd,0x06,0x00,0x16,0x70,0x0f,0x41,0xee,0x05,0x84,0x10,0xfc,0xff,0xff,0x51,0xc8,  //00000f10  ....p.A.......Qﾈ
    0xff,0xfa,0x24,0x2e,0x05,0xa8,0x05,0xf9,0x00,0x00,0x0c,0xec,0x70,0x00,0x3d,0x40,  //00000f20  ..$..ｨ......p.=@
    0x05,0xc4,0x2d,0x40,0x05,0xc6,0x2d,0x40,0x0d,0xd2,0x2d,0x40,0x0d,0xd6,0x61,0x00,  //00000f30  .ﾄ-@.ﾆ-@.ﾒ-@.ﾖa.
    0x02,0xa6,0x60,0x00,0xfd,0x74,0x4a,0x6e,0x05,0xac,0x66,0x22,0x4a,0x6e,0x05,0xae,  //00000f40  .ｦ`..tJn.ｬf"Jn.ｮ
    0x66,0x1c,0x72,0x09,0x70,0xf5,0x4e,0x4f,0x02,0x00,0x00,0xc0,0x67,0x06,0x72,0x00,  //00000f50  f.r.p.NO...ﾀg.r.
    0x70,0xf5,0x4e,0x4f,0x1b,0x7c,0x00,0x01,0x00,0x0e,0x60,0x00,0xfd,0x4c,0x61,0x00,  //00000f60  p.NO.|....`..La.
    0x01,0xea,0x4a,0x80,0x67,0x02,0x60,0x1c,0x70,0x00,0x10,0x2d,0x00,0x01,0x41,0xee,  //00000f70  .鵯.g.`.p..-..A.
    0x05,0x84,0x41,0xf0,0x00,0x00,0x4a,0x10,0x66,0x0a,0x1b,0x7c,0x00,0x01,0x00,0x0e,  //00000f80  .Б...J.f..|....
    0x60,0x00,0xfd,0x26,0x70,0x00,0x10,0x2d,0x00,0x01,0x41,0xee,0x05,0x84,0x41,0xf0,  //00000f90  `..&p..-..A..Б.
    0x00,0x00,0x10,0xbc,0x00,0x00,0x1b,0x7c,0xff,0xff,0x00,0x0e,0x60,0x00,0xfd,0x0a,  //00000fa0  ...ｼ...|....`...
    0x4a,0x6e,0x05,0xac,0x66,0x40,0x4a,0x6e,0x05,0xae,0x66,0x3a,0x72,0x09,0x70,0xf5,  //00000fb0  Jn.ｬf@Jn.ｮf:r.p.
    0x4e,0x4f,0x02,0x00,0x00,0xc0,0x67,0x06,0x72,0x00,0x70,0xf5,0x4e,0x4f,0x0c,0x2d,  //00000fc0  NO...ﾀg.r.p.NO.-
    0x00,0x08,0x00,0x0d,0x64,0x16,0x1b,0x7c,0x00,0x42,0x00,0x0d,0x4a,0x6e,0x05,0xc4,  //00000fd0  ....d..|.B..Jn.ﾄ
    0x67,0x06,0x08,0xed,0x00,0x03,0x00,0x0d,0x60,0x00,0xfc,0xce,0x1b,0x7c,0xff,0xff,  //00000fe0  g.......`..ﾎ.|..
    0x00,0x0d,0x60,0x00,0xfc,0xc4,0x10,0x2d,0x00,0x0d,0x67,0x2c,0xb0,0x3c,0x00,0x01,  //00000ff0  ..`..ﾄ.-..g,ｰ<..
    0x67,0x62,0xb0,0x3c,0x00,0x02,0x67,0x6e,0xb0,0x3c,0x00,0x03,0x67,0x00,0x00,0x88,  //00001000  gbｰ<..gnｰ<..g..芦
    0xb0,0x3c,0x00,0x06,0x67,0x70,0xb0,0x3c,0x00,0x07,0x67,0x00,0x00,0x90,0x1b,0x7c,  //00001010   <..gpｰ<..g....|
    0xff,0xff,0x00,0x0d,0x60,0x00,0xfc,0x92,0x61,0x00,0x01,0x30,0x4a,0x80,0x67,0x10,  //00001020  ....`..誕..0J.g.
    0x0c,0x40,0x00,0x01,0x67,0x0a,0x0c,0x40,0x70,0x02,0x67,0x1e,0x60,0x00,0xfc,0x7c,  //00001030  .@..g..@p.g.`..|
    0x4a,0x6e,0x0d,0xda,0x66,0x0a,0x1b,0x7c,0x00,0x02,0x00,0x0d,0x60,0x00,0x00,0xa2,  //00001040  Jn.ﾚf..|....`..｢
    0x1b,0x7c,0x00,0x0a,0x00,0x0d,0x60,0x00,0x00,0x98,0x1b,0x7c,0x00,0x04,0x00,0x0d,  //00001050  .|....`....|....
    0x60,0x00,0x00,0x8e,0x4a,0xae,0x05,0xba,0x66,0xbe,0x4a,0xae,0x05,0xb6,0x66,0xb8,  //00001060  `..捌ｮ.ｺfｾJｮ.ｶfｸ
    0x61,0x00,0x00,0xa6,0x60,0xe4,0x2d,0x7c,0xff,0xff,0xff,0xff,0x05,0xb6,0x4a,0xae,  //00001070  a..ｦ`.-|.....ｶJｮ
    0x05,0xba,0x67,0x56,0x60,0xba,0x2d,0x7c,0xff,0xff,0xff,0xff,0x05,0xba,0x4a,0xae,  //00001080  .ｺgV`ｺ-|.....ｺJｮ
    0x05,0xb6,0x67,0x46,0x60,0xaa,0x4a,0xae,0x05,0xb6,0x67,0xa4,0x2d,0x7c,0x00,0x00,  //00001090  .ｶgF`ｪJｮ.ｶg､-|..
    0x00,0x00,0x05,0xb6,0x4a,0xae,0x05,0xba,0x67,0x1a,0x60,0x94,0x4a,0xae,0x05,0xba,  //000010a0  ...ｶJｮ.ｺg.`寧ｮ.ｺ
    0x67,0x8e,0x2d,0x7c,0x00,0x00,0x00,0x00,0x05,0xba,0x4a,0xae,0x05,0xb6,0x67,0x04,  //000010b0  g.-|.....ｺJｮ.ｶg.
    0x60,0x00,0xff,0x7e,0x70,0xf5,0x72,0x32,0x28,0x2e,0x05,0xa8,0x76,0x00,0x4e,0x4f,  //000010c0  `..~p.r2(..ｨv.NO
    0x4a,0x80,0x66,0x00,0xff,0x54,0x60,0x00,0xff,0x68,0x70,0xf5,0x72,0x32,0x28,0x2e,  //000010d0  J.f..T`..hp.r2(.
    0x05,0xa8,0x76,0x01,0x4e,0x4f,0x4a,0x80,0x66,0x00,0xff,0x3e,0x60,0x00,0xff,0x52,  //000010e0  .ｨv.NOJ.f..>`..R
    0x4a,0x6e,0x05,0xc4,0x67,0x06,0x08,0xed,0x00,0x03,0x00,0x0d,0x4a,0xae,0x05,0xb6,  //000010f0  Jn.ﾄg.......Jｮ.ｶ
    0x67,0x06,0x08,0xed,0x00,0x04,0x00,0x0d,0x4a,0xae,0x05,0xba,0x67,0x06,0x08,0xed,  //00001100  g.......Jｮ.ｺg...
    0x00,0x06,0x00,0x0d,0x60,0x00,0xfb,0xa2,0x4a,0x6e,0x05,0xac,0x66,0x08,0x4a,0x6e,  //00001110  ....`..｢Jn.ｬf.Jn
    0x05,0xae,0x66,0x28,0x4e,0x75,0x70,0x02,0x4e,0x4f,0x08,0x00,0x00,0x03,0x66,0x0e,  //00001120  .ｮf(Nup.NO....f.
    0x70,0xf5,0x72,0x30,0x28,0x2e,0x05,0xa8,0x76,0x00,0x4e,0x4f,0x4e,0x75,0x70,0xf5,  //00001130  p.r0(..ｨv.NONup.
    0x72,0x30,0x28,0x2e,0x05,0xa8,0x76,0x01,0x4e,0x4f,0x4e,0x75,0x70,0xf5,0x72,0x2f,  //00001140  r0(..ｨv.NONup.r/
    0x28,0x2e,0x05,0xa8,0x76,0x02,0x4e,0x4f,0x4e,0x75,0x48,0xe7,0x70,0xc0,0x70,0x7f,  //00001150  (..ｨv.NONuH輛ﾀp.
    0x4e,0x4f,0xb2,0xae,0x0d,0xd6,0x66,0x12,0x24,0x2e,0x0d,0xd2,0x26,0x00,0x96,0x82,  //00001160  NOｲｮ.ﾖf.$..ﾒ&.魔
    0x0c,0x83,0x00,0x00,0x00,0x64,0x65,0x00,0x00,0xe2,0x2d,0x40,0x0d,0xd2,0x2d,0x41,  //00001170  .....de...-@.ﾒ-A
    0x0d,0xd6,0x72,0x09,0x70,0xf5,0x4e,0x4f,0x02,0x00,0x00,0xc0,0x67,0x06,0x72,0x00,  //00001180  .ﾖr.p.NO...ﾀg.r.
    0x70,0xf5,0x4e,0x4f,0x20,0x3c,0x00,0x00,0x00,0xf5,0x72,0x24,0x28,0x2e,0x05,0xa8,  //00001190  p.NO <....r$(..ｨ
    0x4e,0x4f,0x4a,0x80,0x67,0x00,0x00,0x92,0xb0,0xbc,0x00,0x00,0x00,0x08,0x67,0xd2,  //000011a0  NOJ.g..腸ｼ....gﾒ
    0xb0,0xbc,0x00,0x00,0x00,0x02,0x66,0x00,0x00,0x8a,0x61,0x00,0x07,0x16,0x4a,0x40,  //000011b0  ｰｼ....f..蛎...J@
    0x67,0xc0,0xb0,0x7c,0x00,0x01,0x66,0x70,0x70,0x0f,0x41,0xee,0x05,0x84,0x10,0xfc,  //000011c0  gﾀｰ|..fpp.A.....
    0xff,0xff,0x51,0xc8,0xff,0xfa,0x61,0x00,0x01,0x26,0x4a,0x80,0x67,0x0c,0x3d,0x7c,  //000011d0  ..Qﾈ..a..&J.g.=|
    0xff,0xff,0x05,0xb2,0x60,0x5c,0x48,0xe7,0x70,0xc0,0x28,0x2e,0x05,0xa8,0x76,0x04,  //000011e0  ...ｲ`\H輛ﾀ(..ｨv.
    0x43,0xee,0x05,0xd2,0x74,0x3f,0x72,0x29,0x70,0xf5,0x4e,0x4f,0x4a,0x80,0x67,0x20,  //000011f0  C..ﾒt?r)p.NOJ.g 
    0x0c,0x80,0x00,0x00,0x00,0x08,0x67,0xe2,0x0c,0x80,0x00,0x00,0x00,0x02,0x66,0x32,  //00001200  ......g.......f2
    0x61,0x00,0x06,0xc0,0x4a,0x40,0x67,0xd2,0xb0,0x7c,0x00,0x01,0x66,0x1a,0x60,0xa8,  //00001210  a..ﾀJ@gﾒｰ|..f.`ｨ
    0x08,0x29,0x00,0x07,0x00,0x02,0x67,0x08,0x3d,0x7c,0xff,0xff,0x0d,0xda,0x60,0x06,  //00001220  .)....g.=|...ﾚ`.
    0x3d,0x7c,0x00,0x00,0x0d,0xda,0x70,0x01,0x2d,0x40,0x0d,0xdc,0x4c,0xdf,0x03,0x0e,  //00001230  =|...ﾚp.-@.ﾜLﾟ..
    0x4e,0x75,0x2d,0x7c,0x00,0x00,0x00,0x00,0x0d,0xd2,0x2d,0x7c,0x00,0x00,0x00,0x00,  //00001240  Nu-|.....ﾒ-|....
    0x0d,0xd6,0x70,0xff,0x4c,0xdf,0x03,0x0e,0x4e,0x75,0x20,0x2e,0x0d,0xdc,0x4c,0xdf,  //00001250  .ﾖp.Lﾟ..Nu ..ﾜLﾟ
    0x03,0x0e,0x4e,0x75,0x43,0xee,0x05,0xd2,0x2a,0x2e,0x05,0xa4,0x74,0x02,0xe5,0x8a,  //00001260  ..NuC..ﾒ*..､t.蜉
    0xea,0xaa,0x76,0x01,0xe5,0x8b,0xea,0xab,0x28,0x2e,0x05,0xa8,0x70,0xf5,0x72,0x21,  //00001270  .ｪv.蜍.ｫ(..ｨp.r!
    0x4e,0x4f,0x4a,0x80,0x66,0x76,0x0c,0x91,0x58,0x36,0x38,0x4b,0x66,0x6c,0x26,0x49,  //00001280  NOJ.fv.噌68Kfl&I
    0x45,0xee,0x04,0x04,0x7c,0x00,0x7e,0x0e,0x47,0xeb,0x00,0x10,0x4a,0x13,0x67,0x52,  //00001290  E...|.~.G...J.gR
    0x0c,0x93,0x48,0x75,0x6d,0x61,0x66,0x4a,0x0c,0xab,0x6e,0x36,0x38,0x6b,0x00,0x04,  //000012a0  .滴umafJ.ｫn68k..
    0x66,0x40,0x10,0x2b,0x00,0x08,0x08,0x00,0x00,0x00,0x66,0x36,0x24,0x2b,0x00,0x08,  //000012b0  f@.+......f6$+..
    0x43,0xee,0x00,0x04,0x2a,0x2e,0x05,0xa4,0xe5,0x8a,0xea,0xaa,0x76,0x01,0xe5,0x8b,  //000012c0  C...*..､蜉.ｪv.蜍
    0xea,0xab,0x28,0x2e,0x05,0xa8,0x70,0xf5,0x72,0x21,0x4e,0x4f,0x4a,0x80,0x66,0x1c,  //000012d0  .ｫ(..ｨp.r!NOJ.f.
    0x43,0xe9,0x00,0x12,0x4a,0x51,0x67,0x0a,0x72,0x04,0x24,0xd9,0x51,0xc9,0xff,0xfc,  //000012e0  C...JQg.r.$ﾙQﾉ..
    0x52,0x46,0x51,0xcf,0xff,0xa4,0x70,0x00,0x4e,0x75,0x70,0xff,0x4e,0x75,0x43,0xee,  //000012f0  RFQﾏ.､p.Nup.NuC.
    0x05,0xd2,0x2a,0x2e,0x05,0xa4,0x74,0x02,0xe5,0x8a,0xea,0xaa,0x76,0x01,0xe5,0x8b,  //00001300  .ﾒ*..､t.蜉.ｪv.蜍
    0xea,0xab,0x28,0x2e,0x05,0xa8,0x70,0xf5,0x72,0x21,0x4e,0x4f,0x4a,0x80,0x66,0xdc,  //00001310  .ｫ(..ｨp.r!NOJ.fﾜ
    0x0c,0x91,0x58,0x36,0x38,0x4b,0x66,0xd2,0x26,0x49,0x45,0xee,0x04,0x04,0x7c,0x00,  //00001320  .噌68Kfﾒ&IE...|.
    0x7e,0x0e,0x47,0xeb,0x00,0x10,0x4a,0x13,0x67,0x54,0x0c,0x93,0x48,0x75,0x6d,0x61,  //00001330  ~.G...J.gT.滴uma
    0x66,0x4c,0x0c,0xab,0x6e,0x36,0x38,0x6b,0x00,0x04,0x66,0x42,0x10,0x2b,0x00,0x08,  //00001340  fL.ｫn68k..fB.+..
    0x08,0x00,0x00,0x00,0x66,0x38,0x24,0x2b,0x00,0x08,0x43,0xee,0x00,0x04,0x2a,0x2e,  //00001350  ....f8$+..C...*.
    0x05,0xa4,0xea,0xaa,0xe5,0x8a,0x76,0x01,0xe5,0x8b,0xea,0xab,0x28,0x2e,0x05,0xa8,  //00001360  .､.ｪ蜉v.蜍.ｫ(..ｨ
    0x70,0xf5,0x72,0x21,0x4e,0x4f,0x4a,0x80,0x66,0x82,0x43,0xe9,0x00,0x12,0x72,0x04,  //00001370  p.r!NOJ.f.C...r.
    0x20,0x19,0xb0,0x9a,0x66,0x00,0xff,0x74,0x51,0xc9,0xff,0xf6,0x52,0x46,0x51,0xcf,  //00001380   .ｰ喃..tQﾉ..RFQﾏ
    0xff,0xa2,0xbc,0x6e,0x05,0xb4,0x66,0x00,0xff,0x62,0x70,0x00,0x4e,0x75,0x3d,0x7c,  //00001390  .｢ｼn.ｴf..bp.Nu=|
    0x00,0x08,0x05,0xc2,0x3d,0x7c,0x00,0x00,0x05,0xc0,0x4a,0x6e,0x05,0xb2,0x67,0x44,  //000013a0  ...ﾂ=|...ﾀJn.ｲgD
    0x61,0x00,0xff,0x4c,0x4a,0x80,0x67,0x2c,0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x2c,  //000013b0  a..LJ.g,......g,
    0x0c,0x80,0x00,0x00,0x00,0x08,0x67,0xe8,0x0c,0x80,0x00,0x00,0x00,0x02,0x66,0x00,  //000013c0  ......g.......f.
    0xf8,0xe2,0x61,0x00,0x04,0xfe,0x4a,0x40,0x67,0x0a,0x0c,0x40,0x00,0x01,0x67,0xd0,  //000013d0  .秣...J@g..@..gﾐ
    0x60,0x00,0xf8,0xd8,0x3d,0x7c,0x00,0x00,0x05,0xb2,0x67,0x08,0x30,0x3c,0x70,0x07,  //000013e0  `..ﾘ=|...ｲg.0<p.
    0x60,0x00,0xf8,0xc8,0x4a,0x2e,0x05,0xb0,0x66,0x06,0x72,0x21,0x60,0x00,0x00,0x8a,  //000013f0  `..ﾈJ..ｰf.r!`..較
    0x72,0x26,0x60,0x00,0x01,0xec,0x3d,0x7c,0x00,0x08,0x05,0xc2,0x3d,0x7c,0xff,0xff,  //00001400   &`...=|...ﾂ=|..
    0x05,0xc0,0x60,0x0c,0x3d,0x7c,0x00,0x08,0x05,0xc2,0x3d,0x7c,0x00,0x00,0x05,0xc0,  //00001410  .ﾀ`.=|...ﾂ=|...ﾀ
    0x4a,0x6e,0x05,0xc4,0x67,0x08,0x30,0x3c,0x70,0x0d,0x60,0x00,0xf8,0x8e,0x4a,0x6e,  //00001420  Jn.ﾄg.0<p.`..捌n
    0x05,0xb2,0x67,0x44,0x61,0x00,0xfe,0xc8,0x4a,0x80,0x67,0x2c,0x0c,0x80,0xff,0xff,  //00001430  .ｲgDa..ﾈJ.g,....
    0xff,0xff,0x67,0x2c,0x0c,0x80,0x00,0x00,0x00,0x08,0x67,0xe8,0x0c,0x80,0x00,0x00,  //00001440  ..g,......g.....
    0x00,0x02,0x66,0x00,0xf8,0x5e,0x61,0x00,0x04,0x7a,0x4a,0x40,0x67,0x0a,0xb0,0x7c,  //00001450  ..f..^a..zJ@g.ｰ|
    0x00,0x01,0x67,0xd0,0x60,0x00,0xf8,0x54,0x3d,0x7c,0x00,0x00,0x05,0xb2,0x67,0x08,  //00001460  ..gﾐ`..T=|...ｲg.
    0x30,0x3c,0x70,0x07,0x60,0x00,0xf8,0x44,0x4a,0x2e,0x05,0xb0,0x66,0x04,0x72,0x22,  //00001470  0<p.`..DJ..ｰf.r"
    0x60,0x06,0x72,0x27,0x60,0x00,0x01,0x6a,0x2d,0x6d,0x00,0x0e,0x05,0x94,0x2d,0x6d,  //00001480  `.r'`..j-m....-m
    0x00,0x12,0x05,0x9c,0x70,0x00,0x10,0x2d,0x00,0x01,0xe5,0x88,0x41,0xee,0x05,0x44,  //00001490  ...徘..-..蛻A..D
    0xd1,0xc0,0x20,0x50,0x20,0x2d,0x00,0x16,0xd0,0xa8,0x00,0x10,0x2d,0x40,0x05,0x98,  //000014a0  ﾑﾀ P -..ﾐｨ..-@..
    0x3c,0x2e,0x05,0xa0,0x24,0x2e,0x05,0x98,0xed,0xaa,0x2e,0x2e,0x05,0x9c,0xed,0xaf,  //000014b0  <...$..倆ｪ...懦ｯ
    0x22,0x6e,0x05,0x94,0x26,0x07,0xb6,0xbc,0x00,0x00,0x01,0x00,0x63,0x06,0x26,0x3c,  //000014c0  "n..&.ｶｼ....c.&<
    0x00,0x00,0x01,0x00,0x70,0xf5,0x28,0x2e,0x05,0xa8,0x2a,0x2e,0x05,0xa4,0x4e,0x4f,  //000014d0  ....p.(..ｨ*..､NO
    0x4a,0x80,0x67,0x00,0x00,0xd2,0xb0,0xbc,0xff,0xff,0xff,0xff,0x67,0x00,0x00,0xea,  //000014e0  J.g..ﾒｰｼ....g...
    0xb0,0xbc,0xff,0xff,0xff,0xfe,0x67,0x00,0x00,0xaa,0xb0,0xbc,0x00,0x00,0x00,0x08,  //000014f0  ｰｼ....g..ｪｰｼ....
    0x67,0x86,0xb0,0xbc,0x00,0x00,0x00,0x02,0x66,0x00,0xf7,0xa8,0x61,0x00,0x03,0xc4,  //00001500  g.ｰｼ....f..ｨa..ﾄ
    0x4a,0x40,0x67,0x00,0xff,0x74,0xb0,0x7c,0x00,0x01,0x67,0x00,0xff,0x6c,0x0c,0x40,  //00001510  J@g..tｰ|..g..l.@
    0x70,0x07,0x66,0x00,0xf7,0x96,0x0c,0x81,0x00,0x00,0x00,0x22,0x66,0x00,0xf7,0x8c,  //00001520  p.f........"f..靴
    0x43,0xee,0x05,0xd2,0x08,0x11,0x00,0x07,0x67,0x00,0xf7,0x80,0x43,0xe9,0x00,0x03,  //00001530   ..ﾒ....g...C...
    0x10,0x19,0xe1,0x88,0x10,0x19,0xe1,0x88,0x10,0x19,0xe1,0x88,0x10,0x19,0x2d,0x40,  //00001540  ..瘉..瘉..瘉..-@
    0x0d,0xe0,0x61,0x00,0x03,0x5a,0x4a,0x80,0x67,0x00,0xff,0x2e,0xb0,0xbc,0xff,0xff,  //00001550  .濛..ZJ.g...ｰｼ..
    0xff,0xff,0x67,0x74,0xb0,0xbc,0xff,0xff,0xff,0xfe,0x67,0x36,0xb0,0xbc,0x00,0x00,  //00001560  ..gtｰｼ....g6ｰｼ..
    0x00,0x08,0x67,0xde,0xb0,0xbc,0x00,0x00,0x00,0x02,0x66,0x00,0xf7,0x36,0x61,0x00,  //00001570  ..gﾞｰｼ....f..6a.
    0x03,0x52,0x4a,0x40,0x67,0x00,0xff,0x02,0xb0,0x7c,0x00,0x01,0x67,0xc4,0x67,0x00,  //00001580  .RJ@g...ｰ|..gﾄg.
    0xf7,0x2a,0x52,0xae,0x05,0xc6,0x53,0x6e,0x05,0xc2,0x66,0x00,0xfe,0xec,0x60,0x00,  //00001590  .*Rｮ.ﾆSn.ﾂf...`.
    0xf7,0x1a,0x52,0xae,0x05,0xc6,0x53,0x6e,0x05,0xc2,0x66,0x00,0xfe,0xdc,0x30,0x3c,  //000015a0  ..Rｮ.ﾆSn.ﾂf..ﾜ0<
    0x70,0x0c,0x60,0x00,0xf7,0x06,0x20,0x03,0xe1,0x88,0xeb,0xa8,0xd3,0xc0,0xd4,0x83,  //000015b0  p.`... .瘉.ｨﾓﾀﾔ.
    0x9e,0x83,0x62,0x00,0xff,0x00,0x4a,0x6e,0x05,0xc0,0x66,0x00,0x01,0x68,0x3d,0x7c,  //000015c0  档b...Jn.ﾀf..h=|
    0x00,0x00,0x05,0xbe,0x60,0x00,0xf6,0xe2,0x4a,0x6e,0x05,0xbe,0x66,0x00,0xf6,0xd4,  //000015d0  ...ｾ`..祀n.ｾf..ﾔ
    0x72,0x00,0x70,0xf5,0x4e,0x4f,0x3d,0x7c,0xff,0xff,0x05,0xbe,0x60,0x00,0xfe,0x9a,  //000015e0  r.p.NO=|...ｾ`...
    0x2d,0x6d,0x00,0x0e,0x05,0x94,0x2d,0x6d,0x00,0x12,0x05,0x9c,0x70,0x00,0x10,0x2d,  //000015f0  -m....-m...徘..-
    0x00,0x01,0xe5,0x88,0x41,0xee,0x05,0x44,0xd1,0xc0,0x20,0x50,0x20,0x2d,0x00,0x16,  //00001600  ..蛻A..Dﾑﾀ P -..
    0xd0,0xa8,0x00,0x10,0x2d,0x40,0x05,0x98,0x3c,0x2e,0x05,0xa0,0x24,0x2e,0x05,0x98,  //00001610  ﾐｨ..-@..<...$..倆
    0xed,0xaa,0x26,0x2e,0x05,0x9c,0xed,0xab,0x22,0x6e,0x05,0x94,0x70,0xf5,0x28,0x2e,  //00001620   ｪ&..懦ｫ"n.廃.(.
    0x05,0xa8,0x2a,0x2e,0x05,0xa4,0x4e,0x4f,0x4a,0x80,0x67,0x00,0x00,0xd0,0xb0,0xbc,  //00001630  .ｨ*..､NOJ.g..ﾐｰｼ
    0xff,0xff,0xff,0xff,0x67,0x00,0x00,0xd6,0xb0,0xbc,0xff,0xff,0xff,0xfe,0x67,0x00,  //00001640  ....g..ﾖｰｼ....g.
    0x00,0xa8,0xb0,0xbc,0x00,0x00,0x00,0x08,0x67,0x96,0xb0,0xbc,0x00,0x00,0x00,0x02,  //00001650  .ｨｰｼ....g眠ｼ....
    0x66,0x00,0xf6,0x50,0x61,0x00,0x02,0x6c,0x4a,0x40,0x67,0x84,0xb0,0x7c,0x00,0x01,  //00001660  f..Pa..lJ@g┣|..
    0x67,0x00,0xff,0x7e,0x0c,0x40,0x70,0x07,0x66,0x00,0xf6,0x40,0x0c,0x81,0x00,0x00,  //00001670  g..~.@p.f..@....
    0x00,0x27,0x66,0x00,0xf6,0x36,0x43,0xee,0x05,0xd2,0x08,0x11,0x00,0x07,0x67,0x00,  //00001680  .'f..6C..ﾒ....g.
    0xf6,0x2a,0x43,0xe9,0x00,0x03,0x10,0x19,0xe1,0x88,0x10,0x19,0xe1,0x88,0x10,0x19,  //00001690  .*C.....瘉..瘉..
    0xe1,0x88,0x10,0x19,0x2d,0x40,0x0d,0xe0,0x61,0x00,0x02,0x04,0x4a,0x80,0x67,0x00,  //000016a0  瘉..-@.濛...J.g.
    0xff,0x40,0xb0,0xbc,0xff,0xff,0xff,0xff,0x67,0x62,0xb0,0xbc,0xff,0xff,0xff,0xfe,  //000016b0  .@ｰｼ....gbｰｼ....
    0x67,0x36,0xb0,0xbc,0x00,0x00,0x00,0x08,0x67,0xde,0xb0,0xbc,0x00,0x00,0x00,0x02,  //000016c0  g6ｰｼ....gﾞｰｼ....
    0x66,0x00,0xf5,0xe0,0x61,0x00,0x01,0xfc,0x4a,0x40,0x67,0x00,0xff,0x14,0xb0,0x7c,  //000016d0  f..濛...J@g...ｰ|
    0x00,0x01,0x67,0xc4,0x60,0x00,0xf5,0xd4,0x52,0xae,0x05,0xc6,0x53,0x6e,0x05,0xc2,  //000016e0  ..gﾄ`..ﾔRｮ.ﾆSn.ﾂ
    0x66,0x00,0xfe,0xfe,0x60,0x00,0xf5,0xc4,0x52,0xae,0x05,0xc6,0x53,0x6e,0x05,0xc2,  //000016f0  f...`..ﾄRｮ.ﾆSn.ﾂ
    0x66,0x00,0xfd,0x86,0x30,0x3c,0x70,0x0c,0x60,0x00,0xf5,0xb0,0x3d,0x7c,0x00,0x00,  //00001700  f...0<p.`..ｰ=|..
    0x05,0xbe,0x4a,0x6e,0x05,0xc0,0x66,0x1c,0x60,0x00,0xf5,0x9e,0x4a,0x6e,0x05,0xbe,  //00001710  .ｾJn.ﾀf.`..曷n.ｾ
    0x66,0x00,0xf5,0x90,0x72,0x00,0x70,0xf5,0x4e,0x4f,0x3d,0x7c,0xff,0xff,0x05,0xbe,  //00001720  f..甚.p.NO=|...ｾ
    0x60,0x00,0xfe,0xbe,0x3d,0x7c,0x00,0x08,0x05,0xc2,0x2d,0x6d,0x00,0x0e,0x05,0x94,  //00001730  `..ｾ=|...ﾂ-m....
    0x2d,0x6d,0x00,0x12,0x05,0x9c,0x70,0x00,0x10,0x2d,0x00,0x01,0xe5,0x88,0x41,0xee,  //00001740  -m...徘..-..蛻A.
    0x05,0x44,0xd1,0xc0,0x20,0x50,0x20,0x2d,0x00,0x16,0xd0,0xa8,0x00,0x10,0x2d,0x40,  //00001750  .Dﾑﾀ P -..ﾐｨ..-@
    0x05,0x98,0x3a,0x2e,0x05,0xa0,0x24,0x2e,0x05,0x98,0xeb,0xaa,0x2e,0x2e,0x05,0x9c,  //00001760  ..:...$..俯ｪ...罹
    0xeb,0xaf,0x7c,0x04,0xeb,0xae,0x28,0x2e,0x05,0xa8,0x2a,0x2e,0x05,0xa4,0x24,0x6e,  //00001770   ｯ|..ｮ(..ｨ*..､$n
    0x05,0x94,0xbe,0x86,0x64,0x04,0x26,0x07,0x60,0x02,0x26,0x06,0x22,0x6e,0x05,0xce,  //00001780  .叛.d.&.`.&."n.ﾎ
    0x72,0x21,0x70,0xf5,0x4e,0x4f,0x4a,0x80,0x67,0x00,0x00,0xc8,0xb0,0xbc,0xff,0xff,  //00001790  r!p.NOJ.g..ﾈｰｼ..
    0xff,0xff,0x67,0x00,0x00,0xe4,0xb0,0xbc,0xff,0xff,0xff,0xfe,0x67,0x00,0x00,0xa0,  //000017a0  ..g..莢ｼ....g...
    0xb0,0xbc,0x00,0x00,0x00,0x08,0x67,0x00,0xff,0x7c,0xb0,0xbc,0x00,0x00,0x00,0x02,  //000017b0  ｰｼ....g..|ｰｼ....
    0x66,0x00,0xf4,0xf0,0x61,0x00,0x01,0x0c,0x4a,0x40,0x67,0x00,0xff,0x68,0xb0,0x7c,  //000017c0  f...a...J@g..hｰ|
    0x00,0x01,0x67,0x00,0xff,0x60,0x0c,0x40,0x70,0x07,0x66,0x00,0xf4,0xde,0x43,0xee,  //000017d0  ..g..`.@p.f..ﾞC.
    0x05,0xd2,0x08,0x11,0x00,0x07,0x67,0x00,0xf4,0xd2,0x43,0xe9,0x00,0x03,0x10,0x19,  //000017e0  .ﾒ....g..ﾒC.....
    0xe1,0x88,0x10,0x19,0xe1,0x88,0x10,0x19,0xe1,0x88,0x10,0x19,0x2d,0x40,0x0d,0xe0,  //000017f0  瘉..瘉..瘉..-@.濛
    0x61,0x00,0x00,0xac,0x4a,0x80,0x67,0x00,0xfc,0x70,0xb0,0xbc,0xff,0xff,0xff,0xff,  //00001800   ..ｬJ.g..pｰｼ....
    0x67,0x76,0xb0,0xbc,0xff,0xff,0xff,0xfe,0x67,0x34,0xb0,0xbc,0x00,0x00,0x00,0x08,  //00001810  gvｰｼ....g4ｰｼ....
    0x67,0xde,0xb0,0xbc,0x00,0x00,0x00,0x02,0x66,0x00,0xf4,0x88,0x61,0x00,0x00,0xa4,  //00001820  gﾞｰｼ....f...a..､
    0x4a,0x40,0x67,0xcc,0xb0,0x7c,0x00,0x01,0x67,0xc6,0x60,0x00,0xf4,0x7e,0x52,0xae,  //00001830  J@gﾌｰ|..gﾆ`..~Rｮ
    0x05,0xc6,0x53,0x6e,0x05,0xc2,0x66,0x00,0xfe,0xec,0x60,0x00,0xf4,0x6e,0x52,0xae,  //00001840  .ﾆSn.ﾂf...`..nRｮ
    0x05,0xc6,0x53,0x6e,0x05,0xc2,0x66,0x00,0xfc,0x30,0x30,0x3c,0x70,0x0c,0x60,0x00,  //00001850  .ﾆSn.ﾂf..00<p.`.
    0xf4,0x5a,0x20,0x03,0xe1,0x88,0xeb,0xa8,0x32,0x00,0x53,0x41,0xb5,0x09,0x66,0x30,  //00001860  .Z .瘉.ｨ2.SAｵ.f0
    0x51,0xc9,0xff,0xfa,0xd3,0xc0,0xd4,0x83,0x9e,0x83,0x62,0x00,0xff,0x06,0x3d,0x7c,  //00001870  Qﾉ..ﾓﾀﾔ.档b...=|
    0x00,0x00,0x05,0xbe,0x60,0x00,0xf4,0x32,0x4a,0x6e,0x05,0xbe,0x66,0x00,0xf4,0x24,  //00001880  ...ｾ`..2Jn.ｾf..$
    0x72,0x00,0x70,0xf5,0x4e,0x4f,0x3d,0x7c,0xff,0xff,0x05,0xbe,0x60,0x00,0xfe,0x96,  //00001890  r.p.NO=|...ｾ`...
    0x3d,0x7c,0x00,0x00,0x05,0xbe,0x30,0x3c,0x70,0x0b,0x60,0x00,0xf4,0x0e,0x48,0xe7,  //000018a0  =|...ｾ0<p.`...H.
    0x10,0x40,0x20,0x2e,0x0d,0xe0,0x43,0xee,0x05,0xd2,0x32,0xfc,0x00,0x00,0x32,0xfc,  //000018b0  .@ ..澆..ﾒ2...2.
    0x00,0x04,0x22,0xc0,0x43,0xee,0x05,0xd2,0x76,0x08,0x61,0x64,0x4c,0xdf,0x02,0x08,  //000018c0  .."ﾀC..ﾒv.adLﾟ..
    0x4e,0x75,0x48,0xe7,0x40,0x40,0x70,0xf5,0x72,0x2c,0x28,0x2e,0x05,0xa8,0x76,0x0e,  //000018d0  NuH蹇@p.r,(..ｨv.
    0x43,0xee,0x05,0xd2,0x4e,0x4f,0x4a,0x80,0x66,0x18,0x42,0x41,0x12,0x29,0x00,0x02,  //000018e0  C..ﾒNOJ.f.BA.)..
    0xe3,0x49,0x43,0xfa,0xf3,0x5a,0x70,0x00,0x30,0x31,0x10,0x00,0x4c,0xdf,0x02,0x02,  //000018f0  紵C..Zp.01..Lﾟ..
    0x4e,0x75,0x30,0x3c,0x70,0x0c,0x4c,0xdf,0x02,0x02,0x4e,0x75,0x70,0x00,0x10,0x2d,  //00001900  Nu0<p.Lﾟ..Nup..-
    0x00,0x01,0xe5,0x88,0x41,0xee,0x05,0x44,0xd1,0xc0,0x2b,0x48,0x00,0x12,0x20,0x50,  //00001910  ..蛻A..Dﾑﾀ+H.. P
    0x1b,0x68,0x00,0x0a,0x00,0x0d,0x60,0x00,0xf3,0x90,0x07,0x00,0x00,0x00,0x00,0x00,  //00001920  .h....`.........
    0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,  //00001930  NU..H躋p&IE..獷.
    0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,0x61,0x28,  //00001940  ..r..ﾚQﾉ..C...a(
    0x4a,0x80,0x66,0x1a,0x22,0x4b,0x72,0x05,0x70,0xf5,0x4e,0x4f,0x0c,0x80,0xff,0xff,  //00001950  J.f."Kr.p.NO....
    0xff,0xff,0x67,0x0a,0x61,0x4c,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x70,0xff,  //00001960  ..g.aLLﾟ.JN]Nup.
    0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x48,0xe7,0x68,0x00,0x32,0x3c,0x00,0x01,  //00001970  Lﾟ.JN]NuH輊.2<..
    0x72,0x01,0x70,0xf5,0x4e,0x4f,0x4a,0x80,0x67,0x06,0x51,0xca,0xff,0xf4,0x60,0x1a,  //00001980  r.p.NOJ.g.Qﾊ..`.
    0x48,0x44,0xeb,0x0c,0x89,0x29,0x00,0x01,0x72,0x03,0x70,0xf5,0x4e,0x4f,0x4a,0x80,  //00001990  HD...)..r.p.NOJ.
    0x66,0x08,0x70,0x00,0x4c,0xdf,0x00,0x16,0x4e,0x75,0x70,0xff,0x4c,0xdf,0x00,0x16,  //000019a0  f.p.Lﾟ..Nup.Lﾟ..
    0x4e,0x75,0x43,0xed,0xff,0xff,0x72,0x06,0x70,0xf5,0x4e,0x4f,0x4a,0x80,0x66,0x1a,  //000019b0  NuC...r.p.NOJ.f.
    0x43,0xed,0xff,0xfe,0x72,0x07,0x70,0xf5,0x4e,0x4f,0x4a,0x80,0x66,0x0c,0x10,0x2d,  //000019c0  C...r.p.NOJ.f..-
    0xff,0xfe,0x48,0x40,0x10,0x2d,0xff,0xff,0x4e,0x75,0x70,0xff,0x4e,0x75,0x0d,0x0a,  //000019d0  ..H@.-..Nup.Nu..
    0x53,0x43,0x53,0x49,0x20,0x44,0x49,0x53,0x4b,0x20,0x44,0x52,0x49,0x56,0x45,0x52,  //000019e0  SCSI DISK DRIVER
    0x20,0x66,0x6f,0x72,0x20,0x58,0x36,0x38,0x30,0x30,0x30,0x20,0x76,0x65,0x72,0x73,  //000019f0   for X68000 vers
    0x69,0x6f,0x6e,0x20,0x31,0x2e,0x30,0x34,0x0d,0x0a,0x43,0x6f,0x70,0x79,0x72,0x69,  //00001a00  ion 1.04..Copyri
    0x67,0x68,0x74,0x20,0x31,0x39,0x39,0x30,0x2d,0x39,0x32,0x20,0x53,0x48,0x41,0x52,  //00001a10  ght 1990-92 SHAR
    0x50,0x2f,0x46,0x69,0x72,0x73,0x74,0x20,0x43,0x6c,0x61,0x73,0x73,0x20,0x54,0x65,  //00001a20  P/First Class Te
    0x63,0x68,0x6e,0x6f,0x6c,0x6f,0x67,0x79,0x0d,0x0a,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a30  chnology........
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001a90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001aa0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ab0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ac0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ad0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ae0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001af0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001b90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ba0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001bb0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001bc0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001bd0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001be0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001bf0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001c90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ca0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001cb0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001cc0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001cd0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ce0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001cf0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001d90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001da0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001db0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001dc0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001dd0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001de0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001df0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001e90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ea0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001eb0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ec0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ed0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ee0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ef0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001f90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001fa0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001fb0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001fc0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001fd0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001fe0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00001ff0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002000  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002010  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002020  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002030  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002040  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002050  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002060  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002070  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002080  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002090  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000020a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000020b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000020c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000020d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000020e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000020f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002100  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002110  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002120  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002130  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002140  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002150  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002160  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002170  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002180  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002190  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000021a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000021b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000021c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000021d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000021e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000021f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002200  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002210  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002220  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002230  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002240  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002250  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002260  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002270  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002280  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002290  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000022a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000022b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000022c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000022d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000022e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000022f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002300  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002310  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002320  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002330  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002340  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002350  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002360  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002370  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002380  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002390  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000023a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000023b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000023c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000023d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000023e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000023f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002400  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002410  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002420  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002430  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002440  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002450  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002460  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002470  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002480  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002490  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000024a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000024b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000024c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000024d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000024e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000024f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002500  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002510  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002520  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002530  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002540  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002550  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002560  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002570  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002580  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002590  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000025a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000025b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000025c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000025d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000025e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000025f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002600  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002610  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002620  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002630  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002640  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002650  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002660  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002670  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002680  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002690  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000026a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000026b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000026c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000026d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000026e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000026f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002700  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002710  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002720  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002730  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002740  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002750  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002760  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002770  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002780  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002790  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000027a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000027b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000027c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000027d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000027e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000027f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002800  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002810  ................
    0x48,0xe7,0x08,0x00,0x70,0xf5,0x72,0x0a,0x4e,0x4f,0x0c,0x80,0x00,0x00,0x00,0x04,  //00002820  H...p.r.NO......
    0x64,0x36,0x61,0x00,0x00,0xce,0x41,0xfa,0x01,0x1c,0x08,0x04,0x00,0x00,0x67,0x08,  //00002830  d6a..ﾎA.......g.
    0x20,0xbc,0x00,0xe9,0x60,0x20,0x60,0x0c,0x08,0x04,0x00,0x01,0x67,0x1a,0x20,0xbc,  //00002840   ｼ.饒 `.....g. ｼ
    0x00,0xea,0x00,0x00,0x70,0x80,0x22,0x3c,0x00,0x00,0x01,0xf5,0x43,0xfa,0x00,0xfa,  //00002850  ....p."<....C...
    0x4e,0x4f,0x4c,0xdf,0x00,0x10,0x4e,0x75,0x70,0x00,0x4c,0xdf,0x00,0x10,0x4e,0x75,  //00002860  NOLﾟ..Nup.Lﾟ..Nu
    0x48,0xe7,0x00,0xa2,0x78,0x00,0x41,0xf9,0x00,0xfc,0x00,0x00,0x61,0x00,0x00,0x9e,  //00002870  H..｢x.A.....a..曷
    0x4a,0x80,0x66,0x16,0x0c,0xa8,0x53,0x43,0x53,0x49,0x00,0x24,0x66,0x0c,0x0c,0x68,  //00002880   .f..ｨSCSI.$f..h
    0x49,0x4e,0x00,0x28,0x66,0x04,0x08,0xc4,0x00,0x00,0x41,0xf9,0x00,0xea,0x00,0x20,  //00002890  IN.(f..ﾄ..A.... 
    0x61,0x7a,0x4a,0x80,0x66,0x16,0x0c,0xa8,0x53,0x43,0x53,0x49,0x00,0x24,0x66,0x0c,  //000028a0  azJ.f..ｨSCSI.$f.
    0x0c,0x68,0x45,0x58,0x00,0x28,0x66,0x04,0x08,0xc4,0x00,0x01,0x13,0xfc,0x00,0x31,  //000028b0  .hEX.(f..ﾄ.....1
    0x00,0xe8,0xe0,0x0d,0x0c,0x39,0x00,0x56,0x00,0xed,0x00,0x6f,0x67,0x18,0x13,0xfc,  //000028c0  .鞨..9.V...og...
    0x00,0x56,0x00,0xed,0x00,0x6f,0x13,0xfc,0x00,0x07,0x00,0xed,0x00,0x70,0x13,0xfc,  //000028d0  .V...o.......p..
    0x00,0x00,0x00,0xed,0x00,0x71,0x08,0x04,0x00,0x00,0x66,0x08,0x08,0xf9,0x00,0x03,  //000028e0  .....q....f.....
    0x00,0xed,0x00,0x70,0x13,0xfc,0x00,0x00,0x00,0xe8,0xe0,0x0d,0x4c,0xdf,0x45,0x00,  //000028f0  ...p.....鞨.LﾟE.
    0x4e,0x75,0x61,0x00,0xff,0x6c,0x08,0x39,0x00,0x03,0x00,0xed,0x00,0x70,0x66,0x06,  //00002900  Nua..l.9.....pf.
    0x08,0x84,0x00,0x01,0x4e,0x75,0x08,0x84,0x00,0x00,0x4e,0x75,0x2c,0x4f,0x43,0xfa,  //00002910  ....Nu....Nu,OC.
    0x00,0x28,0x24,0x79,0x00,0x00,0x00,0x08,0x23,0xc9,0x00,0x00,0x00,0x08,0x20,0x10,  //00002920  .($y....#ﾉ.... .
    0x08,0x00,0x00,0x00,0x66,0x12,0xb0,0xbc,0x00,0x20,0x00,0x00,0x65,0x0a,0x23,0xca,  //00002930  ....f.ｰｼ. ..e.#ﾊ
    0x00,0x00,0x00,0x08,0x70,0x00,0x4e,0x75,0x2e,0x4e,0x23,0xca,0x00,0x00,0x00,0x08,  //00002940  ....p.Nu.N#ﾊ....
    0x70,0xff,0x4e,0x75,0x00,0xe9,0x60,0x20,0x48,0xe7,0x50,0x62,0xb2,0xbc,0x00,0x00,  //00002950  p.Nu.饒 H躅bｲｼ..
    0x00,0x10,0x65,0x18,0xb2,0xbc,0x00,0x00,0x00,0x20,0x65,0x40,0xb2,0xbc,0x00,0x00,  //00002960  ..e.ｲｼ... e@ｲｼ..
    0x00,0x40,0x65,0x0e,0xb2,0xbc,0x00,0x00,0x00,0x20,0x65,0x30,0x45,0xfa,0x00,0x3a,  //00002970  .@e.ｲｼ... e0E..:
    0x60,0x16,0x92,0xbc,0x00,0x00,0x00,0x20,0x45,0xfa,0x00,0x6e,0x60,0x0a,0x92,0xbc,  //00002980  `.直... E..n`.直
    0x00,0x00,0x00,0x40,0x45,0xfa,0x00,0xe2,0xe5,0x89,0x2c,0x7a,0xff,0xb8,0x22,0x32,  //00002990  ...@E..粢.,z.ｸ"2
    0x10,0x00,0xd5,0xc1,0x4e,0x92,0x4c,0xdf,0x46,0x0a,0x4e,0x75,0x70,0xff,0x4c,0xdf,  //000029a0  ..ﾕﾁN鱈ﾟF.Nup.Lﾟ
    0x46,0x0a,0x4e,0x75,0x70,0xff,0x4e,0x75,0x00,0x00,0x0a,0x08,0x00,0x00,0x0b,0x08,  //000029b0  F.Nup.Nu........
    0x00,0x00,0x0a,0xe8,0x00,0x00,0x0c,0x16,0x00,0x00,0x01,0x56,0x00,0x00,0x01,0x00,  //000029c0  ...........V....
    0x00,0x00,0x0d,0x18,0x00,0x00,0x0d,0x64,0x00,0x00,0x0d,0xb0,0x00,0x00,0x0d,0xfa,  //000029d0  .......d...ｰ....
    0x00,0x00,0x0e,0x0e,0x00,0x00,0x0c,0xce,0x00,0x00,0x0c,0x84,0xff,0xff,0xff,0xfc,  //000029e0  .......ﾎ........
    0xff,0xff,0xff,0xfc,0xff,0xff,0xff,0xfc,0x00,0x00,0x01,0x76,0x00,0x00,0x03,0x00,  //000029f0  ...........v....
    0x00,0x00,0x03,0x6e,0x00,0x00,0x04,0xde,0x00,0x00,0x0f,0x4e,0x00,0x00,0x10,0x0a,  //00002a00  ...n...ﾞ...N....
    0x00,0x00,0x03,0xe0,0x00,0x00,0x04,0x4c,0x00,0x00,0x04,0xca,0x00,0x00,0x02,0x12,  //00002a10  .......L...ﾊ....
    0x00,0x00,0x02,0x64,0x00,0x00,0x0f,0x6c,0x00,0x00,0x01,0xc4,0x00,0x00,0x05,0xe2,  //00002a20  ...d...l...ﾄ....
    0x00,0x00,0x0f,0x8a,0x00,0x00,0x05,0x60,0x00,0x00,0x05,0xa0,0x00,0x00,0x02,0xb6,  //00002a30  .......`.......ｶ
    0x00,0x00,0x05,0x20,0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0xbc,  //00002a40  ... ...ｼ...ｼ...ｼ
    0x00,0x00,0x06,0x2c,0x00,0x00,0x06,0x80,0x00,0x00,0x06,0xce,0x00,0x00,0x07,0x1a,  //00002a50  ...,.......ﾎ....
    0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0xbc,  //00002a60  ...ｼ...ｼ...ｼ...ｼ
    0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0xbc,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,  //00002a70  ...ｼ...ｼ...<...<
    0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,  //00002a80  ...<...<...<...<
    0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,  //00002a90  ...<...<...<...<
    0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,  //00002aa0  ...<...<...<...<
    0xff,0xff,0xff,0x3c,0xff,0xff,0xff,0x3c,0x48,0xe7,0x00,0x02,0x2c,0x7a,0xfe,0x96,  //00002ab0  ...<...<H...,z..
    0x10,0x2e,0x00,0x09,0x08,0x00,0x00,0x05,0x66,0x38,0x10,0x2e,0x00,0x0b,0x08,0x00,  //00002ac0  ........f8......
    0x00,0x07,0x67,0xec,0x02,0x00,0x00,0x07,0xb0,0x3c,0x00,0x00,0x66,0x1a,0x61,0x00,  //00002ad0  ..g.....ｰ<..f.a.
    0x06,0x94,0x48,0x40,0x66,0x06,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x4a,0x40,0x67,0x08,  //00002ae0  .禰@f.Lﾟ@.NuJ@g.
    0x48,0x40,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x00,  //00002af0  H@Lﾟ@.Nu....Lﾟ@.
    0x4e,0x75,0x61,0x00,0x08,0xbc,0x70,0xff,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x48,0xe7,  //00002b00  Nua..ｼp.Lﾟ@.NuH.
    0x00,0x02,0x2c,0x7a,0xfe,0x40,0x10,0x2e,0x00,0x09,0x08,0x00,0x00,0x05,0x66,0x3c,  //00002b10  ..,z.@........f<
    0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xec,0x02,0x00,0x00,0x07,0x02,0x00,  //00002b20  ........g.......
    0x00,0x07,0xb0,0x3c,0x00,0x01,0x66,0x1a,0x61,0x00,0x06,0x46,0x48,0x40,0x66,0x06,  //00002b30  ..ｰ<..f.a..FH@f.
    0x4c,0xdf,0x40,0x00,0x4e,0x75,0x4a,0x40,0x67,0x08,0x48,0x40,0x4c,0xdf,0x40,0x00,  //00002b40  Lﾟ@.NuJ@g.H@Lﾟ@.
    0x4e,0x75,0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x61,0x00,0x08,0x62,  //00002b50  Nu....Lﾟ@.Nua..b
    0x70,0xff,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x12,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,  //00002b60  p.Lﾟ@.Nu......NU
    0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,  //00002b70  ..H躋p&IE..獷...
    0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,0x13,0x43,0x00,0x04,  //00002b80  r..ﾚQﾉ..C....C..
    0x61,0x00,0x0e,0xb8,0x4a,0x80,0x66,0x00,0x05,0xd2,0x22,0x4b,0x61,0x00,0x0a,0xe8,  //00002b90  a..ｸJ.f..ﾒ"Ka...
    0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x00,0x05,0xc2,0x61,0x00,0x0e,0xea,0x4c,0xdf,  //00002ba0  ......g..ﾂa..鶚ﾟ
    0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x03,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,  //00002bb0  .JN]Nu......NU..
    0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,0x72,0x05,  //00002bc0  H躋p&IE..獷...r.
    0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,0x13,0x43,0x00,0x04,0x61,0x00,  //00002bd0  .ﾚQﾉ..C....C..a.
    0x0e,0x6a,0x4a,0x80,0x66,0x00,0x05,0x84,0x22,0x4b,0x61,0x00,0x0a,0x9a,0x0c,0x80,  //00002be0  .jJ.f..."Ka.....
    0xff,0xff,0xff,0xff,0x67,0x00,0x05,0x74,0x61,0x00,0x0e,0x9c,0x4c,0xdf,0x0e,0x4a,  //00002bf0  ....g..ta..廰ﾟ.J
    0x4e,0x5d,0x4e,0x75,0x1a,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,  //00002c00  N]Nu......NU..H躋
    0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,  //00002c10   p&IE..獷...r..ﾚ
    0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,0x13,0x43,0x00,0x04,0x13,0x42,0x00,0x02,  //00002c20  Qﾉ..C....C...B..
    0x61,0x00,0x0e,0x18,0x4a,0x80,0x66,0x00,0x05,0x32,0x22,0x4b,0x61,0x00,0x0a,0x48,  //00002c30  a...J.f..2"Ka..H
    0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x00,0x05,0x22,0x61,0x00,0x0e,0x4a,0x4c,0xdf,  //00002c40  ......g.."a..JLﾟ
    0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x15,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,  //00002c50  .JN]Nu......NU..
    0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,0x72,0x05,  //00002c60  H躋p&IE..獷...r.
    0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,0x13,0x43,0x00,0x04,0x13,0x42,  //00002c70  .ﾚQﾉ..C....C...B
    0x00,0x01,0x61,0x00,0x0d,0xc6,0x4a,0x80,0x66,0x00,0x04,0xe0,0x22,0x4b,0x61,0x00,  //00002c80  ..a..ﾆJ.f..."Ka.
    0x09,0xac,0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x00,0x04,0xd0,0x61,0x00,0x0d,0xf8,  //00002c90  .ｬ......g..ﾐa...
    0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x07,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,  //00002ca0  Lﾟ.JN]Nu......NU
    0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,  //00002cb0  ..H躋p&IE..獷...
    0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,0x61,0x00,0x0d,0x7c,  //00002cc0  r..ﾚQﾉ..C...a..|
    0x4a,0x80,0x66,0x00,0x04,0x96,0x22,0x4b,0x61,0x00,0x09,0x62,0x0c,0x80,0xff,0xff,  //00002cd0  J.f..."Ka..b....
    0xff,0xff,0x67,0x00,0x04,0x86,0x61,0x00,0x0d,0xae,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,  //00002ce0  ..g...a..ｮLﾟ.JN]
    0x4e,0x75,0x08,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,  //00002cf0  Nu......NU..H躋p
    0x26,0x49,0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,  //00002d00  &IE..獷...r..ﾚQﾉ
    0xff,0xfc,0x2c,0x02,0x43,0xed,0xff,0xf0,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,  //00002d10  ..,.C....F..燻.F
    0x00,0x02,0xe0,0x8e,0x13,0x46,0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x0d,0x1c,  //00002d20  ..燻.F...C..a...
    0x4a,0x80,0x66,0x00,0x04,0x36,0xe1,0x8b,0xeb,0xab,0x22,0x4b,0x61,0x00,0xfd,0xd0,  //00002d30  J.f..6瘠.ｫ"Ka..ﾐ
    0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x00,0x04,0x22,0x0c,0x80,0xff,0xff,0xff,0xfe,  //00002d40  ......g.."......
    0x67,0x00,0x01,0x54,0x61,0x00,0x0d,0x40,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,  //00002d50  g..Ta..@Lﾟ.JN]Nu
    0x0a,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,  //00002d60  ......NU..H躋p&I
    0x45,0xfa,0xff,0xee,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,  //00002d70  E..獷...r..ﾚQﾉ..
    0x2c,0x02,0x43,0xed,0xff,0xf0,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,0x00,0x02,  //00002d80  ,.C....F..燻.F..
    0xe0,0x8e,0x13,0x46,0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x0c,0xae,0x4a,0x80,  //00002d90  燻.F...C..a..ｮJ.
    0x66,0x00,0x03,0xc8,0xe1,0x8b,0xeb,0xab,0x22,0x4b,0x61,0x00,0xfd,0x0c,0x0c,0x80,  //00002da0  f..ﾈ瘠.ｫ"Ka.....
    0xff,0xff,0xff,0xff,0x67,0x00,0x03,0xb4,0x0c,0x80,0xff,0xff,0xff,0xfe,0x67,0x00,  //00002db0  ....g..ｴ......g.
    0x00,0xe6,0x61,0x00,0x0c,0xd2,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x28,0x00,  //00002dc0  .訛..ﾒLﾟ.JN]Nu(.
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,  //00002dd0  ........NU..H躋p
    0x26,0x49,0x43,0xed,0xff,0xf0,0x45,0xfa,0xff,0xe6,0x72,0x09,0x12,0xda,0x51,0xc9,  //00002de0  &IC...E..誡..ﾚQﾉ
    0xff,0xfc,0x2c,0x03,0x43,0xed,0xff,0xf0,0x23,0x42,0x00,0x02,0x13,0x43,0x00,0x08,  //00002df0  ..,.C...#B...C..
    0xe0,0x8b,0x13,0x43,0x00,0x07,0x61,0x00,0x0c,0x42,0x4a,0x80,0x66,0x00,0x03,0x5c,  //00002e00  煖.C..a..BJ.f..\.
    0x26,0x06,0xe1,0x8b,0xeb,0xab,0x22,0x4b,0x61,0x00,0xfc,0xf4,0x0c,0x80,0xff,0xff,  //00002e10  &.瘠.ｫ"Ka.......
    0xff,0xff,0x67,0x00,0x03,0x46,0x0c,0x80,0xff,0xff,0xff,0xfe,0x67,0x78,0x61,0x00,  //00002e20  ..g..F......gxa.
    0x0c,0x66,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x2a,0x00,0x00,0x00,0x00,0x00,  //00002e30  .fLﾟ.JN]Nu*.....
    0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x45,0xfa,0xff,0xec,  //00002e40  ....NU..H躋pE...
    0x26,0x49,0x43,0xed,0xff,0xf0,0x72,0x09,0x12,0xda,0x51,0xc9,0xff,0xfc,0x2c,0x03,  //00002e50  &IC...r..ﾚQﾉ..,.
    0x43,0xed,0xff,0xf0,0x23,0x42,0x00,0x02,0x13,0x43,0x00,0x08,0xe0,0x8b,0x13,0x43,  //00002e60  C...#B...C..煖.C
    0x00,0x07,0x61,0x00,0x0b,0xd6,0x4a,0x80,0x66,0x00,0x02,0xf0,0x26,0x06,0xe1,0x8b,  //00002e70  ..a..ﾖJ.f...&.瘠
    0xeb,0xab,0x22,0x4b,0x61,0x00,0xfc,0x32,0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x00,  //00002e80  .ｫ"Ka..2......g.
    0x02,0xda,0x0c,0x80,0xff,0xff,0xff,0xfe,0x67,0x0c,0x61,0x00,0x0b,0xfa,0x4c,0xdf,  //00002e90  .ﾚ......g.a...Lﾟ
    0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x61,0x00,0x0b,0xee,0x4a,0x80,0x66,0x02,0x70,0xfe,  //00002ea0  .JN]Nua..珵.f.p.
    0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x2f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00002eb0  Lﾟ.JN]Nu/.......
    0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x45,0xfa,0xff,0xec,0x60,0x80,  //00002ec0  ..NU..H躋pE...`.
    0x04,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x45,0xfa,  //00002ed0  ......NU..H躋pE.
    0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,  //00002ee0  ..C...r..ﾚQﾉ..C.
    0xff,0xf0,0x13,0x43,0x00,0x04,0xe0,0x8b,0x13,0x43,0x00,0x03,0x61,0x00,0x0b,0x4c,  //00002ef0  ...C..煖.C..a..L
    0x4a,0x80,0x66,0x00,0x02,0x66,0x61,0x00,0x0b,0x8e,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,  //00002f00  J.f..fa..鮫ﾟ.JN]
    0x4e,0x75,0x1e,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,  //00002f10  Nu......NU..H躋p
    0x45,0xfa,0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,  //00002f20  E...C...r..ﾚQﾉ..
    0x43,0xed,0xff,0xf0,0x02,0x03,0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x0b,0x0c,  //00002f30  C........C..a...
    0x4a,0x80,0x66,0x00,0x02,0x26,0x61,0x00,0x0b,0x4e,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,  //00002f40  J.f..&a..NLﾟ.JN]
    0x4e,0x75,0x1b,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,  //00002f50  Nu......NU..H躋p
    0x45,0xfa,0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,  //00002f60  E...C...r..ﾚQﾉ..
    0x43,0xed,0xff,0xf0,0x02,0x03,0x00,0x03,0x13,0x43,0x00,0x04,0x61,0x00,0x0a,0xcc,  //00002f70  C........C..a..ﾌ
    0x4a,0x80,0x66,0x00,0x01,0xe6,0x61,0x00,0x0b,0x0e,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,  //00002f80  J.f..訛...Lﾟ.JN]
    0x4e,0x75,0xc1,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,  //00002f90  Nuﾁ.....NU..H躋p
    0x45,0xfa,0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,  //00002fa0  E...C...r..ﾚQﾉ..
    0x43,0xed,0xff,0xf0,0x02,0x03,0x00,0x01,0x13,0x43,0x00,0x04,0x76,0x06,0x61,0x00,  //00002fb0  C........C..v.a.
    0x0a,0x8a,0x4a,0x80,0x66,0x00,0x01,0xa4,0x61,0x00,0x0a,0xcc,0x4c,0xdf,0x0e,0x4a,  //00002fc0  .開.f..､a..ﾌLﾟ.J
    0x4e,0x5d,0x4e,0x75,0x0b,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,  //00002fd0  N]Nu......NU..H躋
    0x52,0x70,0x45,0xfa,0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,  //00002fe0   pE...C...r..ﾚQﾉ
    0xff,0xfc,0x2c,0x02,0x43,0xed,0xff,0xf0,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,  //00002ff0  ..,.C....F..燻.F
    0x00,0x02,0xe0,0x8e,0x13,0x46,0x00,0x01,0x61,0x00,0x0a,0x40,0x4a,0x80,0x66,0x00,  //00003000  ..燻.F..a..@J.f.
    0x01,0x5a,0x61,0x00,0x0a,0x82,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0xc2,0x00,  //00003010  .Za...Lﾟ.JN]Nuﾂ.
    0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,  //00003020  ....NU..H躋p&IE.
    0xff,0xee,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x22,0x03,  //00003030  .獷...r..ﾚQﾉ..".
    0x43,0xed,0xff,0xf0,0x13,0x41,0x00,0x05,0x76,0x06,0x61,0x00,0x09,0xfe,0x4a,0x80,  //00003040  C....A..v.a...J.
    0x66,0x00,0x01,0x18,0x26,0x01,0x22,0x4b,0x61,0x00,0x05,0xe2,0x0c,0x80,0xff,0xff,  //00003050  f...&."Ka.......
    0xff,0xff,0x67,0x00,0x01,0x06,0x61,0x00,0x0a,0x2e,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,  //00003060  ..g...a...Lﾟ.JN]
    0x4e,0x75,0x06,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,  //00003070  Nu......NU..H躋p
    0x45,0xfa,0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,  //00003080  E...C...r..ﾚQﾉ..
    0x43,0xed,0xff,0xf0,0x2c,0x02,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,0x00,0x02,  //00003090  C...,..F..燻.F..
    0xe0,0x8e,0x13,0x46,0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x09,0x9e,0x4a,0x80,  //000030a0  燻.F...C..a..曷.
    0x66,0x00,0x00,0xb8,0x61,0x00,0x09,0xe0,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,  //000030b0  f..ｸa..澂ﾟ.JN]Nu
    0x07,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x45,0xfa,  //000030c0  ......NU..H躋pE.
    0xff,0xf0,0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,  //000030d0  ..C...r..ﾚQﾉ..C.
    0xff,0xf0,0x2c,0x02,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,0x00,0x02,0xe0,0x8e,  //000030e0  ..,..F..燻.F..燻
    0x13,0x46,0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x09,0x50,0x4a,0x80,0x66,0x6a,  //000030f0  .F...C..a..PJ.fj
    0x61,0x00,0x09,0x94,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x0e,0x00,0x00,0x00,  //00003100  a..猫ﾟ.JN]Nu....
    0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,  //00003110  ..NU..H躋p&IE..獷
    0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,  //00003120   ...r..ﾚQﾉ..C...
    0x2c,0x02,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,0x00,0x02,0xe0,0x8e,0x13,0x46,  //00003130  ,..F..燻.F..燻.F
    0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x09,0x02,0x4a,0x80,0x66,0x1c,0x76,0x04,  //00003140  ...C..a...J.f.v.
    0x22,0x4b,0x61,0x00,0x04,0xe8,0x0c,0x80,0xff,0xff,0xff,0xff,0x67,0x0c,0x61,0x00,  //00003150  "Ka.........g.a.
    0x09,0x36,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x70,0xff,0x4c,0xdf,0x0e,0x4a,  //00003160  .6Lﾟ.JN]Nup.Lﾟ.J
    0x4e,0x5d,0x4e,0x75,0x2f,0x0b,0x47,0xfa,0x00,0xd8,0x61,0x38,0x26,0x5f,0x4e,0x75,  //00003170  N]Nu/.G..ﾘa8&_Nu
    0x2f,0x0b,0x47,0xfa,0x01,0xa8,0x61,0x2c,0x48,0xe7,0xc0,0x00,0x10,0x39,0x00,0xe8,  //00003180  /.G..ｨa,H鄲..9.鞨
    0xe0,0x0b,0xe8,0x08,0x0c,0x00,0x00,0x0e,0x64,0x12,0x4e,0x7a,0x00,0x02,0x22,0x00,  //00003190   .......d.Nz..".
    0x08,0xc0,0x00,0x0b,0x4e,0x7b,0x00,0x02,0x4e,0x7b,0x10,0x02,0x4c,0xdf,0x00,0x03,  //000031a0  .ﾀ..N{..N{..Lﾟ..
    0x26,0x5f,0x4e,0x75,0x48,0xe7,0x00,0x60,0x10,0x2e,0x00,0x0b,0x02,0x00,0x00,0x07,  //000031b0  &_NuH..`........
    0x1d,0x40,0x00,0x11,0x20,0x03,0x1d,0x40,0x00,0x1d,0xe0,0x88,0x1d,0x40,0x00,0x1b,  //000031c0  .@.. ..@..煦.@..
    0xe0,0x88,0x1d,0x40,0x00,0x19,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xf6,  //000031d0  煦.@..........g.
    0x45,0xf9,0x00,0xe8,0x40,0x40,0x15,0x7c,0x00,0xff,0x00,0x00,0x35,0x7c,0x00,0x00,  //000031e0  E..錙@.|....5|..
    0x00,0x1a,0x15,0x7c,0x00,0x80,0x00,0x04,0x15,0x7c,0x00,0x04,0x00,0x06,0x41,0xee,  //000031f0  ...|.....|....A.
    0x00,0x15,0x25,0x48,0x00,0x14,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xf6,  //00003200  ..%H..........g.
    0x1d,0x7c,0x00,0x80,0x00,0x05,0x4e,0x93,0x4a,0x80,0x66,0x26,0x08,0x2e,0x00,0x03,  //00003210  .|....N笛.f&....
    0x00,0x09,0x66,0x12,0x08,0x2e,0x00,0x04,0x00,0x09,0x67,0xf0,0x08,0xee,0x00,0x04,  //00003220  ..f.......g.....
    0x00,0x09,0x70,0x00,0x60,0x0c,0x08,0xee,0x00,0x03,0x00,0x09,0x70,0xfd,0x60,0x02,  //00003230  ..p.`.......p.`.
    0x70,0xff,0x25,0x7c,0x00,0xe9,0x60,0x01,0x00,0x14,0x4c,0xdf,0x06,0x00,0x4e,0x75,  //00003240  p.%|.饒...Lﾟ..Nu
    0x48,0xe7,0x7c,0x40,0x15,0x7c,0x00,0x31,0x00,0x05,0x28,0x03,0xb6,0xbc,0x00,0x00,  //00003250  H轎@.|.1..(.ｶｼ..
    0x01,0x00,0x63,0x08,0x2a,0x3c,0x00,0x00,0x01,0x00,0x60,0x02,0x2a,0x03,0x25,0x49,  //00003260  ..c.*<....`.*.%I
    0x00,0x0c,0x35,0x45,0x00,0x0a,0x08,0x2e,0x00,0x03,0x00,0x09,0x66,0x00,0x00,0x98,  //00003270  ..5E........f...
    0x08,0x2e,0x00,0x00,0x00,0x0d,0x67,0xee,0x08,0x2e,0x00,0x03,0x00,0x09,0x66,0x00,  //00003280  ......g.......f.
    0x00,0x86,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xec,0x70,0x00,0x10,0x2e,  //00003290  ..........g.p...
    0x00,0x19,0xe1,0x88,0x10,0x2e,0x00,0x1b,0xe1,0x88,0x10,0x2e,0x00,0x1d,0x22,0x04,  //000032a0  ..瘉....瘉....".
    0x92,0x80,0x70,0x00,0x30,0x2a,0x00,0x0a,0x24,0x05,0x94,0x80,0x92,0x82,0x67,0x08,  //000032b0  逐p.0*..$.楳窒g.
    0xd3,0xaa,0x00,0x0c,0x93,0x6a,0x00,0x0a,0x14,0xbc,0xff,0xff,0x15,0x7c,0x00,0x80,  //000032c0  ﾓｪ..屠...ｼ...|..
    0x00,0x07,0x4e,0x71,0x4e,0x71,0x4e,0x71,0x4e,0x71,0x4e,0x71,0x08,0x2e,0x00,0x03,  //000032d0  ..NqNqNqNqNq....
    0x00,0x09,0x66,0x32,0x08,0x2e,0x00,0x04,0x00,0x09,0x66,0x06,0x08,0x12,0x00,0x07,  //000032e0  ..f2......f.....
    0x67,0xea,0x08,0x2a,0x00,0x01,0x00,0x01,0x66,0x00,0xff,0x7c,0x4a,0x2a,0x00,0x01,  //000032f0  g..*....f..|J*..
    0x66,0x18,0x4a,0x6a,0x00,0x0a,0x66,0x12,0xd3,0xc5,0x98,0x85,0x96,0x85,0x66,0x00,  //00003300  f.Jj..f.ﾓﾅ..妹f.
    0xff,0x4c,0x70,0x00,0x60,0x10,0x70,0xfd,0x60,0x06,0x70,0xfe,0x60,0x02,0x70,0xff,  //00003310  .Lp.`.p.`.p.`.p.
    0x15,0x7c,0x00,0x10,0x00,0x07,0x4c,0xdf,0x02,0x3e,0x4e,0x75,0x48,0xe7,0x7c,0x40,  //00003320  .|....Lﾟ.>NuH轎@
    0x15,0x7c,0x00,0xb1,0x00,0x05,0xb6,0xbc,0x00,0x00,0x01,0x00,0x63,0x08,0x2a,0x3c,  //00003330  .|.ｱ..ｶｼ....c.*<
    0x00,0x00,0x01,0x00,0x60,0x02,0x2a,0x03,0x25,0x49,0x00,0x0c,0x35,0x45,0x00,0x0a,  //00003340  ....`.*.%I..5E..
    0x08,0x2e,0x00,0x03,0x00,0x09,0x66,0x52,0x08,0x2e,0x00,0x00,0x00,0x0d,0x66,0xf0,  //00003350  ......fR......f.
    0x14,0xbc,0xff,0xff,0x15,0x7c,0x00,0x80,0x00,0x07,0x4e,0x71,0x4e,0x71,0x4e,0x71,  //00003360  .ｼ...|....NqNqNq
    0x4e,0x71,0x4e,0x71,0x08,0x2e,0x00,0x03,0x00,0x09,0x66,0x2e,0x08,0x2e,0x00,0x04,  //00003370  NqNq......f.....
    0x00,0x09,0x66,0x06,0x08,0x12,0x00,0x07,0x67,0xea,0x08,0x2a,0x00,0x01,0x00,0x01,  //00003380  ..f.....g..*....
    0x66,0xbe,0x4a,0x2a,0x00,0x01,0x66,0x16,0x4a,0x6a,0x00,0x0a,0x66,0x10,0xd3,0xc5,  //00003390  fｾJ*..f.Jj..f.ﾓﾅ
    0x98,0x85,0x96,0x85,0x66,0x90,0x70,0x00,0x60,0x10,0x70,0xfd,0x60,0x06,0x70,0xfe,  //000033a0  ..妹f壬.`.p.`.p.
    0x60,0x02,0x70,0xff,0x15,0x7c,0x00,0x10,0x00,0x07,0x4c,0xdf,0x02,0x3e,0x4e,0x75,  //000033b0  `.p..|....Lﾟ.>Nu
    0x48,0xe7,0x40,0x42,0x2c,0x7a,0xf5,0x8e,0x1d,0x7c,0x00,0x90,0x00,0x03,0x10,0x39,  //000033c0  H蹇B,z...|.....9
    0x00,0xed,0x00,0x6f,0x0c,0x00,0x00,0x56,0x67,0x3a,0x13,0xfc,0x00,0x31,0x00,0xe8,  //000033d0  ...o...Vg:...1.鞨
    0xe0,0x0d,0xbd,0xfc,0x00,0xe9,0x60,0x20,0x66,0x0a,0x13,0xfc,0x00,0x07,0x00,0xed,  //000033e0   .ｽ..饒 f.......
    0x00,0x70,0x60,0x08,0x13,0xfc,0x00,0x0f,0x00,0xed,0x00,0x70,0x13,0xfc,0x00,0x00,  //000033f0  .p`........p....
    0x00,0xed,0x00,0x71,0x13,0xfc,0x00,0x56,0x00,0xed,0x00,0x6f,0x13,0xfc,0x00,0x00,  //00003400  ...q...V...o....
    0x00,0xe8,0xe0,0x0d,0x10,0x39,0x00,0xed,0x00,0x70,0x02,0x00,0x00,0x07,0x1d,0x40,  //00003410  .鞨..9...p.....@
    0x00,0x01,0x70,0x00,0x1d,0x40,0x00,0x05,0x1d,0x40,0x00,0x11,0x1d,0x40,0x00,0x19,  //00003420  ..p..@...@...@..
    0x1d,0x40,0x00,0x1b,0x1d,0x40,0x00,0x1d,0x1d,0x40,0x00,0x17,0x70,0x80,0xbd,0xfc,  //00003430  .@...@...@..p.ｽ.
    0x00,0xe9,0x60,0x20,0x66,0x04,0x72,0x6c,0x60,0x06,0x22,0x3c,0x00,0x00,0x00,0xf6,  //00003440  .饒 f.rl`."<....
    0x43,0xfa,0x00,0x38,0x4e,0x4f,0x1d,0x7c,0x00,0x10,0x00,0x03,0x1d,0x7c,0x00,0x00,  //00003450  C..8NO.|.....|..
    0x00,0x0b,0x70,0x02,0x61,0x00,0x06,0x58,0x1d,0x7c,0x00,0x10,0x00,0x05,0x70,0x05,  //00003460  ..p.a..X.|....p.
    0x61,0x00,0x06,0x4c,0x1d,0x7c,0x00,0x00,0x00,0x05,0x20,0x3c,0x00,0x00,0x9c,0x40,  //00003470  a..L.|.... <..廖
    0x61,0x00,0x06,0x3c,0x4c,0xdf,0x42,0x02,0x4e,0x75,0x48,0xe7,0xc0,0x02,0x2c,0x7a,  //00003480  a..<LﾟB.NuH鄲.,z
    0xf4,0xc4,0x10,0x2e,0x00,0x09,0x1d,0x40,0x00,0x09,0x4c,0xdf,0x40,0x03,0x4e,0x73,  //00003490  .ﾄ.....@..Lﾟ@.Ns
    0x48,0xe7,0x09,0x02,0x2c,0x7a,0xf4,0xae,0x1d,0x7c,0x00,0x00,0x00,0x11,0x10,0x2e,  //000034a0  H...,z.ｮ.|......
    0x00,0x0d,0x02,0x00,0x00,0xf8,0x66,0xf6,0x1d,0x7c,0x00,0x60,0x00,0x05,0x60,0x18,  //000034b0  ......f..|.`..`.
    0x48,0xe7,0x09,0x02,0x2c,0x7a,0xf4,0x8e,0x1d,0x7c,0x00,0x00,0x00,0x11,0x10,0x2e,  //000034c0  H...,z...|......
    0x00,0x0d,0x02,0x00,0x00,0xf8,0x66,0xf6,0x02,0x44,0x00,0x07,0x10,0x3c,0x00,0x01,  //000034d0  ......f..D...<..
    0xe9,0x28,0x09,0x39,0x00,0xed,0x00,0x71,0x66,0x0c,0x80,0x2e,0x00,0x01,0x1d,0x7c,  //000034e0  .(.9...qf......|
    0x00,0x10,0x00,0x03,0x60,0x06,0x1d,0x7c,0x00,0x00,0x00,0x03,0x1d,0x40,0x00,0x17,  //000034f0  ....`..|.....@..
    0x30,0x3c,0x09,0xc4,0x1d,0x40,0x00,0x1b,0xe0,0x48,0x1d,0x40,0x00,0x19,0x1d,0x7c,  //00003500  0<.ﾄ.@..潯.@...|
    0x00,0x03,0x00,0x1d,0x1d,0x6e,0x00,0x09,0x00,0x09,0x1d,0x7c,0x00,0x20,0x00,0x05,  //00003510  .....n.....|. ..
    0x70,0x01,0x61,0x00,0x05,0x9a,0x10,0x2e,0x00,0x09,0x66,0x08,0x1d,0x7c,0x00,0x05,  //00003520  p.a.......f..|..
    0x00,0x0d,0x66,0xf2,0x10,0x2e,0x00,0x0d,0x08,0x00,0x00,0x07,0x67,0x9a,0x10,0x2e,  //00003530  ..f.........g...
    0x00,0x09,0x67,0xf0,0xb0,0x3c,0x00,0x04,0x67,0x26,0x1d,0x40,0x00,0x09,0xb0,0x3c,  //00003540  ..g.ｰ<..g&.@..ｰ<
    0x00,0x10,0x67,0x0c,0x48,0x40,0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x90,0x4e,0x75,  //00003550  ..g.H@....Lﾟ@侵u
    0x70,0x00,0x4c,0xdf,0x40,0x90,0x4e,0x75,0x70,0xff,0x4c,0xdf,0x40,0x90,0x4e,0x75,  //00003560  p.Lﾟ@侵up.Lﾟ@侵u
    0x70,0x01,0x61,0x00,0x05,0x4a,0x1d,0x7c,0x00,0x00,0x00,0x17,0x20,0x3c,0x00,0x00,  //00003570  p.a..J.|.... <..
    0x02,0x58,0x1d,0x40,0x00,0x1d,0xe0,0x88,0x1d,0x40,0x00,0x1b,0xe0,0x88,0x1d,0x40,  //00003580  .X.@..煦.@..煦.@
    0x00,0x19,0x1d,0x7c,0x00,0x04,0x00,0x09,0x70,0x02,0x61,0x00,0x05,0x22,0x10,0x2e,  //00003590  ...|....p.a.."..
    0x00,0x09,0x67,0xfa,0x1d,0x40,0x00,0x09,0xb0,0x3c,0x00,0x04,0x67,0x08,0xb0,0x3c,  //000035a0  ..g..@..ｰ<..g.ｰ<
    0x00,0x10,0x67,0xac,0x60,0x9e,0x08,0x2e,0x00,0x05,0x00,0x0d,0x66,0xf8,0x1d,0x6e,  //000035b0  ..gｬ`.......f..n
    0x00,0x09,0x00,0x09,0x08,0x2e,0x00,0x07,0x00,0x0d,0x66,0x94,0x60,0x86,0x48,0xe7,  //000035c0  ..........f覗.H.
    0x10,0x02,0x2c,0x7a,0xf3,0x80,0x10,0x11,0x02,0x00,0x00,0xe0,0x0c,0x00,0x00,0x00,  //000035d0  ..,z............
    0x67,0x0e,0xb0,0x3c,0x00,0x20,0x67,0x0c,0xb0,0x3c,0x00,0xa0,0x67,0x0a,0x60,0x0a,  //000035e0  g.ｰ<. g.ｰ<..g.`.
    0x76,0x06,0x60,0x06,0x76,0x0a,0x60,0x02,0x76,0x0c,0x10,0x2e,0x00,0x09,0x08,0x00,  //000035f0  v.`.v.`.v.......
    0x00,0x05,0x66,0x2c,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xec,0x02,0x00,  //00003600  ..f,........g...
    0x00,0x07,0x0c,0x00,0x00,0x02,0x66,0x0e,0x61,0x00,0x02,0xa6,0x48,0x40,0x66,0x06,  //00003610  ......f.a..ｦH@f.
    0x4c,0xdf,0x40,0x08,0x4e,0x75,0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x08,0x4e,0x75,  //00003620  Lﾟ@.Nu....Lﾟ@.Nu
    0x61,0x00,0xfd,0x8e,0x70,0xff,0x4c,0xdf,0x40,0x08,0x4e,0x75,0x48,0xe7,0x00,0x02,  //00003630  a..姿.Lﾟ@.NuH...
    0x2c,0x7a,0xf3,0x12,0x10,0x2e,0x00,0x09,0x08,0x00,0x00,0x05,0x66,0x2c,0x10,0x2e,  //00003640  ,z..........f,..
    0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xec,0x02,0x00,0x00,0x07,0xb0,0x3c,0x00,0x00,  //00003650  ......g.....ｰ<..
    0x66,0x0e,0x61,0x00,0x01,0x66,0x48,0x40,0x66,0x06,0x4c,0xdf,0x40,0x00,0x4e,0x75,  //00003660  f.a..fH@f.Lﾟ@.Nu
    0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x61,0x00,0xfd,0x44,0x70,0xff,  //00003670  ....Lﾟ@.Nua..Dp.
    0x4c,0xdf,0x40,0x00,0x4e,0x75,0x48,0xe7,0x00,0x02,0x2c,0x7a,0xf2,0xc8,0x10,0x2e,  //00003680  Lﾟ@.NuH...,z.ﾈ..
    0x00,0x09,0x08,0x00,0x00,0x05,0x66,0x2c,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,  //00003690  ......f,........
    0x67,0xec,0x02,0x00,0x00,0x07,0xb0,0x3c,0x00,0x01,0x66,0x0e,0x61,0x00,0x01,0x9c,  //000036a0  g.....ｰ<..f.a..廩
    0x48,0x40,0x66,0x06,0x4c,0xdf,0x40,0x00,0x4e,0x75,0x10,0x2e,0x00,0x0b,0x4c,0xdf,  //000036b0   @f.Lﾟ@.Nu....Lﾟ
    0x40,0x00,0x4e,0x75,0x61,0x00,0xfc,0xfa,0x70,0xff,0x4c,0xdf,0x40,0x00,0x4e,0x75,  //000036c0  @.Nua...p.Lﾟ@.Nu
    0x48,0xe7,0x10,0x02,0x2c,0x7a,0xf2,0x7e,0x10,0x2e,0x00,0x09,0x08,0x00,0x00,0x05,  //000036d0  H...,z.~........
    0x66,0x2e,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xec,0x02,0x00,0x00,0x07,  //000036e0  f.........g.....
    0xb0,0x3c,0x00,0x03,0x66,0x10,0x76,0x01,0x61,0x00,0x02,0x06,0x48,0x40,0x66,0x06,  //000036f0  ｰ<..f.v.a...H@f.
    0x4c,0xdf,0x40,0x08,0x4e,0x75,0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x08,0x4e,0x75,  //00003700  Lﾟ@.Nu....Lﾟ@.Nu
    0x61,0x00,0xfc,0xae,0x70,0xff,0x4c,0xdf,0x40,0x08,0x4e,0x75,0x48,0xe7,0x10,0x02,  //00003710  a..ｮp.Lﾟ@.NuH...
    0x2c,0x7a,0xf2,0x32,0x10,0x2e,0x00,0x09,0x08,0x00,0x00,0x05,0x66,0x2e,0x10,0x2e,  //00003720  ,z.2........f...
    0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xec,0x02,0x00,0x00,0x07,0xb0,0x3c,0x00,0x07,  //00003730  ......g.....ｰ<..
    0x66,0x10,0x76,0x01,0x61,0x00,0x01,0xba,0x48,0x40,0x66,0x06,0x4c,0xdf,0x40,0x08,  //00003740  f.v.a..ｺH@f.Lﾟ@.
    0x4e,0x75,0x10,0x2e,0x00,0x0b,0x4c,0xdf,0x40,0x08,0x4e,0x75,0x61,0x00,0xfc,0x62,  //00003750  Nu....Lﾟ@.Nua..b
    0x70,0xff,0x4c,0xdf,0x40,0x08,0x4e,0x75,0x48,0xe7,0x10,0x02,0x2c,0x7a,0xf1,0xe6,  //00003760  p.Lﾟ@.NuH...,z..
    0x10,0x2e,0x00,0x09,0x08,0x00,0x00,0x05,0x66,0x2c,0x10,0x2e,0x00,0x0b,0x08,0x00,  //00003770  ........f,......
    0x00,0x07,0x67,0xec,0x02,0x00,0x00,0x07,0xb0,0x3c,0x00,0x06,0x66,0x0e,0x76,0x01,  //00003780  ..g.....ｰ<..f.v.
    0x61,0x38,0x48,0x40,0x66,0x06,0x4c,0xdf,0x40,0x08,0x4e,0x75,0x10,0x2e,0x00,0x0b,  //00003790  a8H@f.Lﾟ@.Nu....
    0x4c,0xdf,0x40,0x08,0x4e,0x75,0x61,0x00,0xfc,0x18,0x70,0xff,0x4c,0xdf,0x40,0x08,  //000037a0  Lﾟ@.Nua...p.Lﾟ@.
    0x4e,0x75,0x48,0xe7,0x00,0x02,0x2c,0x7a,0xf1,0x9c,0x70,0x00,0x10,0x2e,0x00,0x0b,  //000037b0  NuH...,z.徘.....
    0x4c,0xdf,0x40,0x00,0x4e,0x75,0x70,0x04,0x4e,0x75,0x48,0xe7,0x10,0x40,0x20,0x03,  //000037c0  Lﾟ@.Nup.NuH..@ .
    0x1d,0x40,0x00,0x1d,0xe0,0x88,0x1d,0x40,0x00,0x1b,0xe0,0x88,0x1d,0x40,0x00,0x19,  //000037d0  .@..煦.@..煦.@..
    0x10,0x2e,0x00,0x0b,0x02,0x00,0x00,0x07,0x1d,0x40,0x00,0x11,0x10,0x2e,0x00,0x0b,  //000037e0  .........@......
    0x08,0x00,0x00,0x07,0x67,0xf6,0x1d,0x6e,0x00,0x09,0x00,0x09,0x1d,0x7c,0x00,0x80,  //000037f0  ....g..n.....|..
    0x00,0x05,0x10,0x2e,0x00,0x0d,0x02,0x00,0x00,0xf0,0xb0,0x3c,0x00,0x70,0x67,0x06,  //00003800  ..........ｰ<.pg.
    0xb0,0x3c,0x00,0xb0,0x66,0xec,0x4a,0x2e,0x00,0x09,0x66,0x10,0x08,0x2e,0x00,0x01,  //00003810  ｰ<.ｰf.J...f.....
    0x00,0x0d,0x66,0xf2,0x1d,0x59,0x00,0x15,0x53,0x83,0x66,0xea,0x10,0x2e,0x00,0x09,  //00003820  ..f..Y..Sデ.....
    0x67,0xfa,0x1d,0x40,0x00,0x09,0xb0,0x3c,0x00,0x10,0x67,0x06,0x4c,0xdf,0x02,0x08,  //00003830  g..@..ｰ<..g.Lﾟ..
    0x4e,0x75,0x70,0x00,0x4c,0xdf,0x02,0x08,0x4e,0x75,0x48,0xe7,0x10,0x40,0x10,0x2e,  //00003840  Nup.Lﾟ..NuH..@..
    0x00,0x0b,0x02,0x00,0x00,0x07,0x1d,0x40,0x00,0x11,0x20,0x03,0x1d,0x40,0x00,0x1d,  //00003850  .......@.. ..@..
    0xe0,0x88,0x1d,0x40,0x00,0x1b,0xe0,0x88,0x1d,0x40,0x00,0x19,0x1d,0x6e,0x00,0x09,  //00003860  煦.@..煦.@...n..
    0x00,0x09,0x1d,0x7c,0x00,0x80,0x00,0x05,0x10,0x2e,0x00,0x0d,0x02,0x00,0x00,0xf0,  //00003870  ...|............
    0xb0,0x3c,0x00,0x70,0x67,0x06,0xb0,0x3c,0x00,0xb0,0x66,0xec,0x4a,0x2e,0x00,0x09,  //00003880  ｰ<.pg.ｰ<.ｰf.J...
    0x66,0x10,0x08,0x2e,0x00,0x00,0x00,0x0d,0x66,0xf2,0x12,0xee,0x00,0x15,0x53,0x83,  //00003890  f.......f.....Sデ
    0x66,0xea,0x10,0x2e,0x00,0x09,0x67,0xfa,0x1d,0x40,0x00,0x09,0xb0,0x3c,0x00,0x10,  //000038a0   .....g..@..ｰ<..
    0x67,0x06,0x4c,0xdf,0x02,0x08,0x4e,0x75,0x70,0x00,0x4c,0xdf,0x02,0x08,0x4e,0x75,  //000038b0  g.Lﾟ..Nup.Lﾟ..Nu
    0x48,0xe7,0x10,0x40,0x10,0x2e,0x00,0x0b,0x02,0x00,0x00,0x07,0x1d,0x40,0x00,0x11,  //000038c0  H..@.........@..
    0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xf6,0x1d,0x59,0x00,0x17,0x1d,0x7c,  //000038d0  ........g..Y...|
    0x00,0xec,0x00,0x05,0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x66,0xf6,0x1d,0x7c,  //000038e0  ............f..|
    0x00,0xcc,0x00,0x05,0x53,0x83,0x66,0xcc,0x70,0x00,0x4c,0xdf,0x02,0x08,0x4e,0x75,  //000038f0  .ﾌ..Sデﾌp.Lﾟ..Nu
    0x48,0xe7,0x10,0x40,0x10,0x2e,0x00,0x0b,0x02,0x00,0x00,0x07,0x1d,0x40,0x00,0x11,  //00003900  H..@.........@..
    0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x67,0xf6,0x1d,0x7c,0x00,0xec,0x00,0x05,  //00003910  ........g..|....
    0x10,0x2e,0x00,0x0b,0x08,0x00,0x00,0x07,0x66,0xf6,0x12,0xee,0x00,0x17,0x1d,0x7c,  //00003920  ........f......|
    0x00,0xcc,0x00,0x05,0x53,0x83,0x66,0xcc,0x70,0x00,0x4c,0xdf,0x02,0x08,0x4e,0x75,  //00003930  .ﾌ..Sデﾌp.Lﾟ..Nu
    0x00,0x00,0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x45,0xfa,  //00003940  ......NU..H躋pE.
    0xff,0xf0,0x61,0x00,0x01,0x2c,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x01,0x00,  //00003950  ..a..,Lﾟ.JN]Nu..
    0x00,0x00,0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x45,0xfa,0xff,0xf0,  //00003960  ....NU..H躋pE...
    0x61,0x00,0x01,0x0e,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x08,0x00,0x00,0x00,  //00003970  a...Lﾟ.JN]Nu....
    0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x45,0xfa,0xff,0xee,  //00003980  ..NU..H躋p&IE..獷
    0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x2c,0x02,0x43,0xed,  //00003990   ...r..ﾚQﾉ..,.C.
    0xff,0xf0,0x13,0x46,0x00,0x03,0xe0,0x8e,0x13,0x46,0x00,0x02,0xe0,0x8e,0x13,0x46,  //000039a0  ...F..燻.F..燻.F
    0x00,0x01,0x13,0x43,0x00,0x04,0x61,0x00,0x00,0x92,0x4a,0x80,0x66,0x00,0x00,0x82,  //000039b0  ...C..a..谷.f..ゃ
    0xe1,0x8b,0xeb,0xab,0x22,0x4b,0x61,0x00,0xf1,0x46,0x0c,0x80,0xff,0xff,0xff,0xff,  //000039c0   躯ｫ"Ka..F......
    0x67,0x6e,0x0c,0x80,0xff,0xff,0xff,0xfe,0x67,0x0c,0x61,0x00,0x00,0xba,0x4c,0xdf,  //000039d0  gn......g.a..ｺLﾟ
    0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x61,0x00,0x00,0xae,0x4a,0x80,0x66,0x02,0x70,0xfe,  //000039e0  .JN]Nua..ｮJ.f.p.
    0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x25,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000039f0  Lﾟ.JN]Nu%.......
    0x00,0x00,0x4e,0x55,0xff,0xf0,0x48,0xe7,0x52,0x70,0x26,0x49,0x43,0xed,0xff,0xf0,  //00003a00  ..NU..H躋p&IC...
    0x45,0xfa,0xff,0xe6,0x72,0x09,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,  //00003a10  E..誡..ﾚQﾉ..C...
    0x61,0x28,0x4a,0x80,0x66,0x1a,0x22,0x4b,0x76,0x08,0x61,0x00,0xfc,0x5a,0x0c,0x80,  //00003a20  a(J.f."Kv.a..Z..
    0xff,0xff,0xff,0xff,0x67,0x0a,0x61,0x5e,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,  //00003a30  ....g.a^Lﾟ.JN]Nu
    0x70,0xff,0x4c,0xdf,0x0e,0x4a,0x4e,0x5d,0x4e,0x75,0x48,0xe7,0x48,0x00,0x32,0x3c,  //00003a40  p.Lﾟ.JN]NuH踪.2<
    0x00,0x01,0x61,0x00,0xfa,0x6c,0x4a,0x80,0x67,0x06,0x51,0xc9,0xff,0xf6,0x60,0x18,  //00003a50  ..a..lJ.g.Qﾉ..`.
    0x48,0x44,0xeb,0x0c,0x89,0x29,0x00,0x01,0x61,0x00,0xfb,0x64,0x4a,0x80,0x66,0x08,  //00003a60  HD...)..a..dJ.f.
    0x70,0x00,0x4c,0xdf,0x00,0x12,0x4e,0x75,0x70,0xff,0x4c,0xdf,0x00,0x12,0x4e,0x75,  //00003a70  p.Lﾟ..Nup.Lﾟ..Nu
    0x43,0xed,0xff,0xf0,0x72,0x05,0x12,0xda,0x51,0xc9,0xff,0xfc,0x43,0xed,0xff,0xf0,  //00003a80  C...r..ﾚQﾉ..C...
    0x61,0xb8,0x4a,0x80,0x66,0x24,0x43,0xed,0xff,0xff,0x61,0x00,0xfc,0x34,0x4a,0x80,  //00003a90  aｸJ.f$C...a..4J.
    0x66,0x18,0x43,0xed,0xff,0xfe,0x61,0x00,0xfc,0x74,0x4a,0x80,0x66,0x0c,0x10,0x2d,  //00003aa0  f.C...a..tJ.f..-
    0xff,0xfe,0x48,0x40,0x10,0x2d,0xff,0xff,0x4e,0x75,0x70,0xff,0x4e,0x75,0x48,0xe7,  //00003ab0  ..H@.-..Nup.NuH鈞
    0xe0,0x80,0x41,0xf9,0x00,0xe8,0x80,0x23,0x72,0x00,0x12,0x10,0x12,0x10,0x74,0x00,  //00003ac0   .A..閠#r.....t.
    0x14,0x10,0xb4,0x10,0x65,0xf8,0x92,0x42,0x64,0x04,0xd2,0x7c,0x00,0xc8,0xc3,0x42,  //00003ad0  ..ｴ.e.達d.ﾒ|.ﾈﾃB
    0x90,0x82,0x62,0xea,0x4c,0xdf,0x01,0x07,0x4e,0x75,0x48,0xe7,0xf8,0x42,0x70,0x80,  //00003ae0  垂b鶚ﾟ..NuH鋩Bp.
    0x32,0x3c,0x01,0x40,0x43,0xfa,0x00,0xc6,0x4e,0x4f,0x21,0xc0,0x0c,0xc0,0x70,0x80,  //00003af0  2<.@C..ﾆNO!ﾀ.ﾀp.
    0x32,0x3c,0x01,0x41,0x43,0xfa,0x02,0x92,0x4e,0x4f,0x21,0xc0,0x0c,0xc4,0x70,0x80,  //00003b00  2<.AC..誰O!ﾀ.ﾄp.
    0x32,0x3c,0x01,0x43,0x43,0xfa,0x03,0x04,0x4e,0x4f,0x21,0xc0,0x0c,0xc8,0x70,0x80,  //00003b10  2<.CC...NO!ﾀ.ﾈp.
    0x32,0x3c,0x01,0x44,0x43,0xfa,0x00,0xdc,0x4e,0x4f,0x21,0xc0,0x0c,0xcc,0x70,0x80,  //00003b20  2<.DC..ﾜNO!ﾀ.ﾌp.
    0x32,0x3c,0x01,0x45,0x43,0xfa,0x01,0xdc,0x4e,0x4f,0x21,0xc0,0x0c,0xd0,0x70,0x80,  //00003b30  2<.EC..ﾜNO!ﾀ.ﾐp.
    0x32,0x3c,0x01,0x46,0x43,0xfa,0x01,0xbe,0x4e,0x4f,0x21,0xc0,0x0c,0xd4,0x70,0x80,  //00003b40  2<.FC..ｾNO!ﾀ.ﾔp.
    0x32,0x3c,0x01,0x47,0x43,0xfa,0x00,0x74,0x4e,0x4f,0x21,0xc0,0x0c,0xd8,0x70,0x80,  //00003b50  2<.GC..tNO!ﾀ.ﾘp.
    0x32,0x3c,0x01,0x48,0x43,0xfa,0x00,0x72,0x4e,0x4f,0x21,0xc0,0x0c,0xdc,0x70,0x80,  //00003b60  2<.HC..rNO!ﾀ.ﾜp.
    0x32,0x3c,0x01,0x4b,0x43,0xfa,0x00,0x70,0x4e,0x4f,0x21,0xc0,0x0c,0xe0,0x70,0x80,  //00003b70  2<.KC..pNO!ﾀ.瀾.
    0x32,0x3c,0x01,0x4d,0x43,0xfa,0x00,0x6e,0x4e,0x4f,0x21,0xc0,0x0c,0xe4,0x70,0x80,  //00003b80  2<.MC..nNO!ﾀ.舊.
    0x32,0x3c,0x01,0x4f,0x43,0xfa,0x00,0xca,0x4e,0x4f,0x21,0xc0,0x0c,0xe8,0x32,0x3c,  //00003b90  2<.OC..ﾊNO!ﾀ..2<
    0x80,0x00,0x74,0x0f,0x22,0x7c,0x00,0x00,0x00,0x00,0x61,0x00,0x02,0x6e,0xd2,0x7c,  //00003ba0  ..t."|....a..nﾒ|
    0x01,0x00,0x51,0xca,0xff,0xf0,0x4c,0xdf,0x42,0x1f,0x4e,0x75,0x2f,0x38,0x0c,0xc0,  //00003bb0  ..Qﾊ..LﾟB.Nu/8.ﾀ
    0x48,0xe7,0x48,0x04,0x4b,0xfa,0xf4,0x14,0x60,0x48,0x2f,0x38,0x0c,0xd8,0x48,0xe7,  //00003bc0  H踪.K...`H/8.ﾘH踪
    0x48,0x04,0x4b,0xfa,0xfd,0x90,0x60,0x3a,0x2f,0x38,0x0c,0xdc,0x48,0xe7,0x48,0x04,  //00003bd0   .K..秦:/8.ﾜH踪.
    0x4b,0xfa,0xf5,0x30,0x60,0x2c,0x2f,0x38,0x0c,0xe0,0x48,0xe7,0x48,0x04,0x4b,0xfa,  //00003be0  K..0`,/8.潯踪.K.
    0xf4,0xd6,0x60,0x1e,0x2f,0x38,0x0c,0xe4,0x48,0xe7,0x48,0x04,0x4b,0xfa,0xf4,0x7a,  //00003bf0  .ﾖ`./8.腥踪.K..z
    0x60,0x10,0x2f,0x38,0x0c,0xcc,0x48,0xe7,0x48,0x04,0x4b,0xfa,0xfd,0x3a,0x60,0x00,  //00003c00  `./8.ﾌH踪.K..:`.
    0x00,0x02,0x78,0x00,0x38,0x01,0x02,0x41,0xf0,0x00,0xb2,0x7c,0x80,0x00,0x66,0x3a,  //00003c10  ..x.8..A..ｲ|..f:
    0xe0,0x4c,0xe2,0x4c,0x64,0x04,0x08,0xc4,0x00,0x10,0x02,0x44,0x00,0x07,0x09,0x39,  //00003c20  澂祗d..ﾄ...D...9
    0x00,0xed,0x00,0x71,0x67,0x24,0x4e,0x95,0x02,0x80,0xff,0xff,0xff,0x1e,0x4a,0x80,  //00003c30  ...qg$N.......J.
    0x66,0x0a,0x70,0x00,0x4c,0xdf,0x20,0x12,0x58,0x8f,0x4e,0x75,0x00,0x80,0xff,0xff,  //00003c40  f.p.Lﾟ .X蒐u....
    0xff,0x00,0x4c,0xdf,0x20,0x12,0x58,0x8f,0x4e,0x75,0x4c,0xdf,0x20,0x12,0x4e,0x75,  //00003c50  ..Lﾟ .X蒐uLﾟ .Nu
    0x2f,0x38,0x0c,0xe8,0x48,0xe7,0x7f,0x48,0x78,0x00,0x38,0x01,0x02,0x41,0xf0,0x00,  //00003c60  /8.鍠..Hx.8..A..
    0xb2,0x7c,0x80,0x00,0x66,0x6e,0x22,0x04,0xe0,0x4c,0xe2,0x4c,0x64,0x04,0x08,0xc4,  //00003c70  ｲ|..fn".澂祗d..ﾄ
    0x00,0x10,0x02,0x44,0x00,0x07,0x09,0x39,0x00,0xed,0x00,0x71,0x67,0x4e,0x49,0xf9,  //00003c80  ...D...9...qgNI.
    0x00,0x00,0x09,0xfe,0x20,0x01,0xe0,0x58,0xc0,0xbc,0x00,0x00,0x00,0x0f,0xd9,0xc0,  //00003c90  .... .濆ﾀｼ....ﾙﾀ
    0x10,0x14,0x08,0x00,0x00,0x07,0x66,0x34,0xc0,0x3c,0x00,0x7f,0x67,0x2e,0x24,0x3c,  //00003ca0  ......f4ﾀ<..g.$<
    0x00,0x01,0x56,0x60,0x43,0xfa,0x02,0x84,0xb0,0x3c,0x00,0x14,0x67,0x1a,0x24,0x3c,  //00003cb0  ..V`C..┣<..g.$<
    0x00,0x02,0xac,0xc0,0x43,0xfa,0x02,0x88,0xb0,0x3c,0x00,0x28,0x67,0x0a,0x24,0x3c,  //00003cc0  ..ｬﾀC..芦<.(g.$<
    0x00,0x00,0xaf,0x50,0x43,0xfa,0x02,0x50,0x61,0x00,0xf3,0x00,0x4c,0xdf,0x12,0xfe,  //00003cd0  ..ｯPC..Pa...Lﾟ..
    0x58,0x8f,0x4e,0x75,0x4c,0xdf,0x12,0xfe,0x4e,0x75,0x02,0x41,0xf0,0x00,0xb2,0x7c,  //00003ce0  X蒐uLﾟ..Nu.A..ｲ|
    0x80,0x00,0x66,0x0a,0x4c,0xdf,0x00,0x02,0x58,0x8f,0x70,0x00,0x4e,0x75,0x4c,0xdf,  //00003cf0  ..f.Lﾟ..X術.NuLﾟ
    0x00,0x02,0x4e,0x75,0x2f,0x38,0x0c,0xd4,0x48,0xe7,0x7e,0x64,0x4b,0xfa,0xef,0xea,  //00003d00  ..Nu/8.ﾔH轜dK..鸛
    0x60,0x10,0x2f,0x38,0x0c,0xd0,0x48,0xe7,0x7e,0x64,0x4b,0xfa,0xf0,0x4a,0x60,0x00,  //00003d10   ./8.ﾐH轜dK..J`.
    0x00,0x02,0x78,0x00,0x38,0x01,0x02,0x41,0xf0,0x00,0xb2,0x7c,0x80,0x00,0x66,0x62,  //00003d20  ..x.8..A..ｲ|..fb
    0xe0,0x4c,0xe2,0x4c,0x64,0x04,0x08,0xc4,0x00,0x10,0x02,0x44,0x00,0x07,0x09,0x39,  //00003d30  澂祗d..ﾄ...D...9
    0x00,0xed,0x00,0x71,0x67,0x4c,0x2c,0x03,0x26,0x06,0xd6,0xbc,0x00,0x00,0x00,0xff,  //00003d40  ...qgL,.&.ﾖｼ....
    0xe0,0x8b,0xb6,0xbc,0x00,0x00,0x01,0x00,0x63,0x06,0x26,0x3c,0x00,0x00,0x01,0x00,  //00003d50  煖ｶｼ....c.&<....
    0x7a,0x00,0x4e,0x95,0x02,0x80,0xff,0xff,0xff,0x1e,0x4a,0x80,0x66,0x16,0xd4,0x83,  //00003d60  z.N.......J.f.ﾔ.
    0x22,0x03,0xe1,0x89,0xd3,0xc1,0x9c,0x81,0x62,0xce,0x4c,0xdf,0x26,0x7e,0x58,0x8f,  //00003d70  ".瘟ﾓﾁ怐bﾎLﾟ&~X術
    0x70,0x00,0x4e,0x75,0x4c,0xdf,0x26,0x7e,0x58,0x8f,0x00,0x80,0xff,0xff,0xff,0x00,  //00003d80   .NuLﾟ&~X.......
    0x4e,0x75,0x4c,0xdf,0x26,0x7e,0x4e,0x75,0x4e,0x54,0xff,0x00,0x48,0xe7,0x7e,0x60,  //00003d90  NuLﾟ&~NuNT..H轜`
    0x78,0x00,0x38,0x01,0x02,0x41,0xf0,0x00,0xb2,0x7c,0x80,0x00,0x66,0x60,0x22,0x04,  //00003da0  x.8..A..ｲ|..f`".
    0xe0,0x4c,0xe2,0x4c,0x64,0x04,0x08,0xc4,0x00,0x10,0x02,0x44,0x00,0x07,0x09,0x39,  //00003db0  澂祗d..ﾄ...D...9
    0x00,0xed,0x00,0x71,0x67,0x48,0x24,0x49,0x2c,0x03,0x26,0x06,0xb6,0xbc,0x00,0x00,  //00003dc0  ...qgH$I,.&.ｶｼ..
    0x01,0x00,0x65,0x06,0x26,0x3c,0x00,0x00,0x01,0x00,0x43,0xec,0xff,0x00,0x61,0x00,  //00003dd0  ..e.&<....C...a.
    0xff,0x24,0x2a,0x03,0x53,0x85,0xb5,0x09,0x66,0x14,0x51,0xcd,0xff,0xfa,0x52,0x82,  //00003de0  .$*.S.ｵ.f.Qﾍ..R.
    0x9c,0x83,0x62,0xd6,0x4c,0xdf,0x06,0x7e,0x4e,0x5c,0x70,0x00,0x4e,0x75,0x70,0xfe,  //00003df0  怎bﾖLﾟ.~N\p.Nup.
    0x4c,0xdf,0x06,0x7e,0x4e,0x5c,0x00,0x80,0xff,0xff,0xff,0x00,0x4e,0x75,0x4c,0xdf,  //00003e00  Lﾟ.~N\......NuLﾟ
    0x06,0x7e,0x4e,0x5c,0x2f,0x38,0x0c,0xc4,0x4e,0x75,0x4e,0x54,0xff,0x00,0x48,0xe7,  //00003e10  .~N\/8.ﾄNuNT..H輾
    0x78,0x44,0x78,0x00,0x38,0x01,0x02,0x41,0xf0,0x00,0xb2,0x7c,0x80,0x00,0x66,0x46,  //00003e20   Dx.8..A..ｲ|..fF
    0xe0,0x4c,0xe2,0x4c,0x64,0x04,0x08,0xc4,0x00,0x10,0x02,0x44,0x00,0x07,0x09,0x39,  //00003e30  澂祗d..ﾄ...D...9
    0x00,0xed,0x00,0x71,0x67,0x30,0x20,0x09,0x67,0x38,0x76,0x0a,0x61,0x00,0xf1,0xd6,  //00003e40  ...qg0 .g8v.a..ﾖ
    0x02,0x80,0xff,0xff,0xff,0x1e,0x4a,0x80,0x66,0x0e,0x61,0x00,0x00,0x8c,0x70,0x00,  //00003e50  ......J.f.a..継.
    0x4c,0xdf,0x22,0x1e,0x4e,0x5c,0x4e,0x75,0x00,0x80,0xff,0xff,0xff,0x00,0x4c,0xdf,  //00003e60  Lﾟ".N\Nu......Lﾟ
    0x22,0x1e,0x4e,0x5c,0x4e,0x75,0x4c,0xdf,0x22,0x1e,0x4e,0x5c,0x2f,0x38,0x0c,0xc8,  //00003e70  ".N\NuLﾟ".N\/8.ﾈ
    0x4e,0x75,0x76,0x0a,0x43,0xfa,0x00,0xb4,0x61,0x00,0xf1,0x9a,0x02,0x80,0xff,0xff,  //00003e80  Nuv.C..ｴa.......
    0xff,0x1e,0x4a,0x80,0x66,0xd2,0x43,0xec,0xff,0x00,0x74,0x04,0x76,0x01,0x7a,0x00,  //00003e90  ..J.fﾒC...t.v.z.
    0x61,0x00,0xee,0x56,0x02,0x80,0xff,0xff,0xff,0x1e,0x4a,0x80,0x66,0xba,0x45,0xec,  //00003ea0  a.皞......J.fｺE.
    0xff,0x00,0x43,0xfa,0x00,0x68,0x0c,0xaa,0x58,0x36,0x38,0x4b,0x00,0x00,0x66,0xa8,  //00003eb0  ..C..h.ｪX68K..fｨ
    0x43,0xfa,0x00,0x5a,0x20,0x2a,0x00,0x04,0xb0,0xbc,0x00,0x00,0x9f,0xd9,0x65,0x00,  //00003ec0  C..Z *..ｰｼ..渟e.
    0xff,0x7a,0x43,0xe9,0x00,0x14,0xb0,0xbc,0x00,0x01,0x3d,0x1d,0x65,0x00,0xff,0x6c,  //00003ed0  .zC...ｰｼ..=.e..l
    0x43,0xe9,0x00,0x14,0x60,0x00,0xff,0x64,0x4b,0xf9,0x00,0x00,0x09,0xfe,0x20,0x01,  //00003ee0  C...`..dK..... .
    0xe0,0x58,0xc0,0xbc,0x00,0x00,0x00,0x0f,0xdb,0xc0,0x10,0x3c,0x00,0x28,0x0c,0x29,  //00003ef0  濆ﾀｼ....ﾛﾀ.<.(.)
    0x00,0x07,0x00,0x03,0x67,0x10,0x10,0x3c,0x00,0x14,0x0c,0x29,0x00,0x02,0x00,0x04,  //00003f00  ....g..<...)....
    0x67,0x04,0x10,0x3c,0x00,0x0a,0x1a,0x80,0x42,0x80,0x4e,0x75,0x01,0x01,0x00,0x03,  //00003f10  g..<....B.Nu....
    0x01,0x35,0x80,0x00,0x00,0x00,0x01,0x01,0x00,0x03,0x01,0x54,0x80,0x00,0x00,0x00,  //00003f20  .5.........T....
    0x01,0x01,0x00,0x03,0x02,0x66,0x80,0x00,0x00,0x00,0x01,0x01,0x00,0x03,0x02,0x98,  //00003f30  .....f..........
    0x80,0x00,0x00,0x00,0x01,0x01,0x00,0x07,0x02,0x66,0x80,0x00,0x00,0x00,0x01,0x01,  //00003f40  .........f......
    0x00,0x07,0x02,0x98,0x80,0x00,0x00,0x00,                                          //00003f50  ........        
  };
*/
  //  perl misc/itob.pl xeij/SPC.java SPC_DEVICE_DRIVER
  public static final byte[] SPC_DEVICE_DRIVER = "\377\377\377\377@\0\0\0\0n\0\0\0\200\1SCHDISK\0\0\0\0\0\0\1F\0\0\3F\0\0\r\f\0\0\0\370\0\0\7\236\0\0\3\260\0\0\0\270\0\0\0\270\0\0\b\24\0\0\b\6\0\0\0\270\0\0\0\270\0\0\1,\0\0\0\1p\2p\7p\fp\b\0\1p\rp\fp\fp\fp\fp\fp\fp\fp\fH\347\0\2M\372\r\310-M\0\0L\337@\0NuH\347\177\376M\372\r\266*n\0\0p\0\20-\0\2\f\0\0\fb\32\"m\0\16A\372\377|\320@\320@\321\300 P \bA\372\377T\321\300N\3200<p\f`\2B@2\0\33A\0\3\340I\33A\0\4J@g\22Jn\5\254f\6Jn\5\256g\6=|\377\377\5\262L\337\177\376NuJ\256\5\312g\312\"n\5\312p\200\"<\0\0\1\365NO`\272 m\0\16 -\0\22\f\200\0\0\0\bg\24\f\200\0\0\0\4f\2500\356\5\304 .\5\2500\200`\2340\356\5\304 .\5\2500\300 \256\5\306`\214 m\0\16 -\0\22\f\200\0\0\0\2f\0\377|=P\5\304`\0\377t\f-\0\27\0\26d\0\377dS\202\2\202\0\0\0\17-B\5\250\59\0\0\f\354f\0\377Na\0\32\270-@\5\312 <\0\0\0\365r$(.\5\250NO\260\274\0\0\0\0gd\260\274\377\377\377\377g\0\377V\260\274\0\0\0\bgP\260\274\0\0\0\2f\0\377D <\0\0\0\365r,v\16(.\5\250C\356\5\322NO\260\274\0\0\0\0f\0\377&C\356\5\322\20\21\2\0\0p\f\0\0pf\0\377\24\20)\0\2g\20\260<\0\1g\n\260<\0\6g\4`\0\376\376`\206p\365r+(.\5\250NOJ\200f\0\376\354p\365r%(.\5\250C\356\5\322NOJ\200f\0\376\330C\356\5\322\")\0\4\262\274\0\0\4\0g\30\262\274\0\0\2\0g\b=|\0\2\5\240`\16=|\0\1\5\240`\6=|\0\0\5\240\340\211\342\211-A\5\244C\356\5\322*.\5\244t\0\345\212\352\252v\1\345\213\352\253(.\5\250p\365r!NOJ\200f\0\376~\f\221X68Sf\0\376t\f\251CSI1\0\4f\0\376h\35i\0\16\5\260=|\0\0\5\254=|\0\0\5\256\f)\0\1\0\17g\20\f)\0\2\0\17f\16=|\377\377\5\256`\6=|\377\377\5\254a\0\3\266J\200f\0\376.JFg\0\376(=|\0\0\5\262=F\5\264J\256\5\312g\6A\3720\212`\4A\372\31L-H\5\316\321\374\0\0\20\0+H\0\16\20-\0\26\320\6\4\0\0\27e\2\234\0\33F\0\rA\356\5D+H\0\22C\356\4\4p\16 \311C\351\0\24Q\310\377\370\33z\375\6\0\26p\17A\356\5\204\20\374\377\377Q\310\377\372$.\5\250\5\371\0\0\f\354p\0=@\5\304-@\5\306-@\r\322-@\r\326a\0\2\246`\0\375tJn\5\254f\"Jn\5\256f\34r\tp\365NO\2\0\0\300g\6r\0p\365NO\33|\0\1\0\16`\0\375La\0\1\352J\200g\2`\34p\0\20-\0\1A\356\5\204A\360\0\0J\20f\n\33|\0\1\0\16`\0\375&p\0\20-\0\1A\356\5\204A\360\0\0\20\274\0\0\33|\377\377\0\16`\0\375\nJn\5\254f@Jn\5\256f:r\tp\365NO\2\0\0\300g\6r\0p\365NO\f-\0\b\0\rd\26\33|\0B\0\rJn\5\304g\6\b\355\0\3\0\r`\0\374\316\33|\377\377\0\r`\0\374\304\20-\0\rg,\260<\0\1gb\260<\0\2gn\260<\0\3g\0\0\210\260<\0\6gp\260<\0\7g\0\0\220\33|\377\377\0\r`\0\374\222a\0\0010J\200g\20\f@\0\1g\n\f@p\2g\36`\0\374|Jn\r\332f\n\33|\0\2\0\r`\0\0\242\33|\0\n\0\r`\0\0\230\33|\0\4\0\r`\0\0\216J\256\5\272f\276J\256\5\266f\270a\0\0\246`\344-|\377\377\377\377\5\266J\256\5\272gV`\272-|\377\377\377\377\5\272J\256\5\266gF`\252J\256\5\266g\244-|\0\0\0\0\5\266J\256\5\272g\32`\224J\256\5\272g\216-|\0\0\0\0\5\272J\256\5\266g\4`\0\377~p\365r2(.\5\250v\0NOJ\200f\0\377T`\0\377hp\365r2(.\5\250v\1NOJ\200f\0\377>`\0\377RJn\5\304g\6\b\355\0\3\0\rJ\256\5\266g\6\b\355\0\4\0\rJ\256\5\272g\6\b\355\0\6\0\r`\0\373\242Jn\5\254f\bJn\5\256f(Nup\2NO\b\0\0\3f\16p\365r0(.\5\250v\0NONup\365r0(.\5\250v\1NONup\365r/(.\5\250v\2NONuH\347p\300p\177NO\262\256\r\326f\22$.\r\322&\0\226\202\f\203\0\0\0de\0\0\342-@\r\322-A\r\326r\tp\365NO\2\0\0\300g\6r\0p\365NO <\0\0\0\365r$(.\5\250NOJ\200g\0\0\222\260\274\0\0\0\bg\322\260\274\0\0\0\2f\0\0\212a\0\7\26J@g\300\260|\0\1fpp\17A\356\5\204\20\374\377\377Q\310\377\372a\0\1&J\200g\f=|\377\377\5\262`\\H\347p\300(.\5\250v\4C\356\5\322t?r)p\365NOJ\200g \f\200\0\0\0\bg\342\f\200\0\0\0\2f2a\0\6\300J@g\322\260|\0\1f\32`\250\b)\0\7\0\2g\b=|\377\377\r\332`\6=|\0\0\r\332p\1-@\r\334L\337\3\16Nu-|\0\0\0\0\r\322-|\0\0\0\0\r\326p\377L\337\3\16Nu .\r\334L\337\3\16NuC\356\5\322*.\5\244t\2\345\212\352\252v\1\345\213\352\253(.\5\250p\365r!NOJ\200fv\f\221X68Kfl&IE\356\4\4|\0~\16G\353\0\20J\23gR\f\223HumafJ\f\253n68k\0\4f@\20+\0\b\b\0\0\0f6$+\0\bC\356\0\4*.\5\244\345\212\352\252v\1\345\213\352\253(.\5\250p\365r!NOJ\200f\34C\351\0\22JQg\nr\4$\331Q\311\377\374RFQ\317\377\244p\0Nup\377NuC\356\5\322*.\5\244t\2\345\212\352\252v\1\345\213\352\253(.\5\250p\365r!NOJ\200f\334\f\221X68Kf\322&IE\356\4\4|\0~\16G\353\0\20J\23gT\f\223HumafL\f\253n68k\0\4fB\20+\0\b\b\0\0\0f8$+\0\bC\356\0\4*.\5\244\352\252\345\212v\1\345\213\352\253(.\5\250p\365r!NOJ\200f\202C\351\0\22r\4 \31\260\232f\0\377tQ\311\377\366RFQ\317\377\242\274n\5\264f\0\377bp\0Nu=|\0\b\5\302=|\0\0\5\300Jn\5\262gDa\0\377LJ\200g,\f\200\377\377\377\377g,\f\200\0\0\0\bg\350\f\200\0\0\0\2f\0\370\342a\0\4\376J@g\n\f@\0\1g\320`\0\370\330=|\0\0\5\262g\b0<p\7`\0\370\310J.\5\260f\6r!`\0\0\212r&`\0\1\354=|\0\b\5\302=|\377\377\5\300`\f=|\0\b\5\302=|\0\0\5\300Jn\5\304g\b0<p\r`\0\370\216Jn\5\262gDa\0\376\310J\200g,\f\200\377\377\377\377g,\f\200\0\0\0\bg\350\f\200\0\0\0\2f\0\370^a\0\4zJ@g\n\260|\0\1g\320`\0\370T=|\0\0\5\262g\b0<p\7`\0\370DJ.\5\260f\4r\"`\6r\'`\0\1j-m\0\16\5\224-m\0\22\5\234p\0\20-\0\1\345\210A\356\5D\321\300 P -\0\26\320\250\0\20-@\5\230<.\5\240$.\5\230\355\252..\5\234\355\257\"n\5\224&\7\266\274\0\0\1\0c\6&<\0\0\1\0p\365(.\5\250*.\5\244NOJ\200g\0\0\322\260\274\377\377\377\377g\0\0\352\260\274\377\377\377\376g\0\0\252\260\274\0\0\0\bg\206\260\274\0\0\0\2f\0\367\250a\0\3\304J@g\0\377t\260|\0\1g\0\377l\f@p\7f\0\367\226\f\201\0\0\0\"f\0\367\214C\356\5\322\b\21\0\7g\0\367\200C\351\0\3\20\31\341\210\20\31\341\210\20\31\341\210\20\31-@\r\340a\0\3ZJ\200g\0\377.\260\274\377\377\377\377gt\260\274\377\377\377\376g6\260\274\0\0\0\bg\336\260\274\0\0\0\2f\0\3676a\0\3RJ@g\0\377\2\260|\0\1g\304g\0\367*R\256\5\306Sn\5\302f\0\376\354`\0\367\32R\256\5\306Sn\5\302f\0\376\3340<p\f`\0\367\6 \3\341\210\353\250\323\300\324\203\236\203b\0\377\0Jn\5\300f\0\1h=|\0\0\5\276`\0\366\342Jn\5\276f\0\366\324r\0p\365NO=|\377\377\5\276`\0\376\232-m\0\16\5\224-m\0\22\5\234p\0\20-\0\1\345\210A\356\5D\321\300 P -\0\26\320\250\0\20-@\5\230<.\5\240$.\5\230\355\252&.\5\234\355\253\"n\5\224p\365(.\5\250*.\5\244NOJ\200g\0\0\320\260\274\377\377\377\377g\0\0\326\260\274\377\377\377\376g\0\0\250\260\274\0\0\0\bg\226\260\274\0\0\0\2f\0\366Pa\0\2lJ@g\204\260|\0\1g\0\377~\f@p\7f\0\366@\f\201\0\0\0\'f\0\3666C\356\5\322\b\21\0\7g\0\366*C\351\0\3\20\31\341\210\20\31\341\210\20\31\341\210\20\31-@\r\340a\0\2\4J\200g\0\377@\260\274\377\377\377\377gb\260\274\377\377\377\376g6\260\274\0\0\0\bg\336\260\274\0\0\0\2f\0\365\340a\0\1\374J@g\0\377\24\260|\0\1g\304`\0\365\324R\256\5\306Sn\5\302f\0\376\376`\0\365\304R\256\5\306Sn\5\302f\0\375\2060<p\f`\0\365\260=|\0\0\5\276Jn\5\300f\34`\0\365\236Jn\5\276f\0\365\220r\0p\365NO=|\377\377\5\276`\0\376\276=|\0\b\5\302-m\0\16\5\224-m\0\22\5\234p\0\20-\0\1\345\210A\356\5D\321\300 P -\0\26\320\250\0\20-@\5\230:.\5\240$.\5\230\353\252..\5\234\353\257|\4\353\256(.\5\250*.\5\244$n\5\224\276\206d\4&\7`\2&\6\"n\5\316r!p\365NOJ\200g\0\0\310\260\274\377\377\377\377g\0\0\344\260\274\377\377\377\376g\0\0\240\260\274\0\0\0\bg\0\377|\260\274\0\0\0\2f\0\364\360a\0\1\fJ@g\0\377h\260|\0\1g\0\377`\f@p\7f\0\364\336C\356\5\322\b\21\0\7g\0\364\322C\351\0\3\20\31\341\210\20\31\341\210\20\31\341\210\20\31-@\r\340a\0\0\254J\200g\0\374p\260\274\377\377\377\377gv\260\274\377\377\377\376g4\260\274\0\0\0\bg\336\260\274\0\0\0\2f\0\364\210a\0\0\244J@g\314\260|\0\1g\306`\0\364~R\256\5\306Sn\5\302f\0\376\354`\0\364nR\256\5\306Sn\5\302f\0\37400<p\f`\0\364Z \3\341\210\353\2502\0SA\265\tf0Q\311\377\372\323\300\324\203\236\203b\0\377\6=|\0\0\5\276`\0\3642Jn\5\276f\0\364$r\0p\365NO=|\377\377\5\276`\0\376\226=|\0\0\5\2760<p\13`\0\364\16H\347\20@ .\r\340C\356\5\3222\374\0\0002\374\0\4\"\300C\356\5\322v\badL\337\2\bNuH\347@@p\365r,(.\5\250v\16C\356\5\322NOJ\200f\30BA\22)\0\2\343IC\372\363Zp\00001\20\0L\337\2\2Nu0<p\fL\337\2\2Nup\0\20-\0\1\345\210A\356\5D\321\300+H\0\22 P\33h\0\n\0\r`\0\363\220\7\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360a(J\200f\32\"Kr\5p\365NO\f\200\377\377\377\377g\naLL\337\16JN]Nup\377L\337\16JN]NuH\347h\0002<\0\1r\1p\365NOJ\200g\6Q\312\377\364`\32HD\353\f\211)\0\1r\3p\365NOJ\200f\bp\0L\337\0\26Nup\377L\337\0\26NuC\355\377\377r\6p\365NOJ\200f\32C\355\377\376r\7p\365NOJ\200f\f\20-\377\376H@\20-\377\377Nup\377Nu\r\nSCSI DISK DRIVER for X68000 version 1.04\r\nCopyright 1990-92 SHARP/First Class Technology\r\n\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0H\347\b\0p\365r\nNO\f\200\0\0\0\4d6a\0\0\316A\372\1\34\b\4\0\0g\b \274\0\351` `\f\b\4\0\1g\32 \274\0\352\0\0p\200\"<\0\0\1\365C\372\0\372NOL\337\0\20Nup\0L\337\0\20NuH\347\0\242x\0A\371\0\374\0\0a\0\0\236J\200f\26\f\250SCSI\0$f\f\fhIN\0(f\4\b\304\0\0A\371\0\352\0 azJ\200f\26\f\250SCSI\0$f\f\fhEX\0(f\4\b\304\0\1\23\374\0001\0\350\340\r\f9\0V\0\355\0og\30\23\374\0V\0\355\0o\23\374\0\7\0\355\0p\23\374\0\0\0\355\0q\b\4\0\0f\b\b\371\0\3\0\355\0p\23\374\0\0\0\350\340\rL\337E\0Nua\0\377l\b9\0\3\0\355\0pf\6\b\204\0\1Nu\b\204\0\0Nu,OC\372\0($y\0\0\0\b#\311\0\0\0\b \20\b\0\0\0f\22\260\274\0 \0\0e\n#\312\0\0\0\bp\0Nu.N#\312\0\0\0\bp\377Nu\0\351` H\347Pb\262\274\0\0\0\20e\30\262\274\0\0\0 e@\262\274\0\0\0@e\16\262\274\0\0\0 e0E\372\0:`\26\222\274\0\0\0 E\372\0n`\n\222\274\0\0\0@E\372\0\342\345\211,z\377\270\"2\20\0\325\301N\222L\337F\nNup\377L\337F\nNup\377Nu\0\0\n\b\0\0\13\b\0\0\n\350\0\0\f\26\0\0\1V\0\0\1\0\0\0\r\30\0\0\rd\0\0\r\260\0\0\r\372\0\0\16\16\0\0\f\316\0\0\f\204\377\377\377\374\377\377\377\374\377\377\377\374\0\0\1v\0\0\3\0\0\0\3n\0\0\4\336\0\0\17N\0\0\20\n\0\0\3\340\0\0\4L\0\0\4\312\0\0\2\22\0\0\2d\0\0\17l\0\0\1\304\0\0\5\342\0\0\17\212\0\0\5`\0\0\5\240\0\0\2\266\0\0\5 \377\377\377\274\377\377\377\274\377\377\377\274\0\0\6,\0\0\6\200\0\0\6\316\0\0\7\32\377\377\377\274\377\377\377\274\377\377\377\274\377\377\377\274\377\377\377\274\377\377\377\274\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<\377\377\377<H\347\0\2,z\376\226\20.\0\t\b\0\0\5f8\20.\0\13\b\0\0\7g\354\2\0\0\7\260<\0\0f\32a\0\6\224H@f\6L\337@\0NuJ@g\bH@L\337@\0Nu\20.\0\13L\337@\0Nua\0\b\274p\377L\337@\0NuH\347\0\2,z\376@\20.\0\t\b\0\0\5f<\20.\0\13\b\0\0\7g\354\2\0\0\7\2\0\0\7\260<\0\1f\32a\0\6FH@f\6L\337@\0NuJ@g\bH@L\337@\0Nu\20.\0\13L\337@\0Nua\0\bbp\377L\337@\0Nu\22\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\23C\0\4a\0\16\270J\200f\0\5\322\"Ka\0\n\350\f\200\377\377\377\377g\0\5\302a\0\16\352L\337\16JN]Nu\3\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\23C\0\4a\0\16jJ\200f\0\5\204\"Ka\0\n\232\f\200\377\377\377\377g\0\5ta\0\16\234L\337\16JN]Nu\32\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\23C\0\4\23B\0\2a\0\16\30J\200f\0\0052\"Ka\0\nH\f\200\377\377\377\377g\0\5\"a\0\16JL\337\16JN]Nu\25\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\23C\0\4\23B\0\1a\0\r\306J\200f\0\4\340\"Ka\0\t\254\f\200\377\377\377\377g\0\4\320a\0\r\370L\337\16JN]Nu\7\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360a\0\r|J\200f\0\4\226\"Ka\0\tb\f\200\377\377\377\377g\0\4\206a\0\r\256L\337\16JN]Nu\b\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374,\2C\355\377\360\23F\0\3\340\216\23F\0\2\340\216\23F\0\1\23C\0\4a\0\r\34J\200f\0\0046\341\213\353\253\"Ka\0\375\320\f\200\377\377\377\377g\0\4\"\f\200\377\377\377\376g\0\1Ta\0\r@L\337\16JN]Nu\n\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374,\2C\355\377\360\23F\0\3\340\216\23F\0\2\340\216\23F\0\1\23C\0\4a\0\f\256J\200f\0\3\310\341\213\353\253\"Ka\0\375\f\f\200\377\377\377\377g\0\3\264\f\200\377\377\377\376g\0\0\346a\0\f\322L\337\16JN]Nu(\0\0\0\0\0\0\0\0\0NU\377\360H\347Rp&IC\355\377\360E\372\377\346r\t\22\332Q\311\377\374,\3C\355\377\360#B\0\2\23C\0\b\340\213\23C\0\7a\0\fBJ\200f\0\3\\&\6\341\213\353\253\"Ka\0\374\364\f\200\377\377\377\377g\0\3F\f\200\377\377\377\376gxa\0\ffL\337\16JN]Nu*\0\0\0\0\0\0\0\0\0NU\377\360H\347RpE\372\377\354&IC\355\377\360r\t\22\332Q\311\377\374,\3C\355\377\360#B\0\2\23C\0\b\340\213\23C\0\7a\0\13\326J\200f\0\2\360&\6\341\213\353\253\"Ka\0\3742\f\200\377\377\377\377g\0\2\332\f\200\377\377\377\376g\fa\0\13\372L\337\16JN]Nua\0\13\356J\200f\2p\376L\337\16JN]Nu/\0\0\0\0\0\0\0\0\0NU\377\360H\347RpE\372\377\354`\200\4\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\23C\0\4\340\213\23C\0\3a\0\13LJ\200f\0\2fa\0\13\216L\337\16JN]Nu\36\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\2\3\0\1\23C\0\4a\0\13\fJ\200f\0\2&a\0\13NL\337\16JN]Nu\33\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\2\3\0\3\23C\0\4a\0\n\314J\200f\0\1\346a\0\13\16L\337\16JN]Nu\301\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374C\355\377\360\2\3\0\1\23C\0\4v\6a\0\n\212J\200f\0\1\244a\0\n\314L\337\16JN]Nu\13\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374,\2C\355\377\360\23F\0\3\340\216\23F\0\2\340\216\23F\0\1a\0\n@J\200f\0\1Za\0\n\202L\337\16JN]Nu\302\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374\"\3C\355\377\360\23A\0\5v\6a\0\t\376J\200f\0\1\30&\1\"Ka\0\5\342\f\200\377\377\377\377g\0\1\6a\0\n.L\337\16JN]Nu\6\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374C\355\377\360,\2\23F\0\3\340\216\23F\0\2\340\216\23F\0\1\23C\0\4a\0\t\236J\200f\0\0\270a\0\t\340L\337\16JN]Nu\7\0\0\0\0\0NU\377\360H\347RpE\372\377\360C\355\377\360r\5\22\332Q\311\377\374C\355\377\360,\2\23F\0\3\340\216\23F\0\2\340\216\23F\0\1\23C\0\4a\0\tPJ\200fja\0\t\224L\337\16JN]Nu\16\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374C\355\377\360,\2\23F\0\3\340\216\23F\0\2\340\216\23F\0\1\23C\0\4a\0\t\2J\200f\34v\4\"Ka\0\4\350\f\200\377\377\377\377g\fa\0\t6L\337\16JN]Nup\377L\337\16JN]Nu/\13G\372\0\330a8&_Nu/\13G\372\1\250a,H\347\300\0\209\0\350\340\13\350\b\f\0\0\16d\22Nz\0\2\"\0\b\300\0\13N{\0\2N{\20\2L\337\0\3&_NuH\347\0`\20.\0\13\2\0\0\7\35@\0\21 \3\35@\0\35\340\210\35@\0\33\340\210\35@\0\31\20.\0\13\b\0\0\7g\366E\371\0\350@@\25|\0\377\0\0005|\0\0\0\32\25|\0\200\0\4\25|\0\4\0\6A\356\0\25%H\0\24\20.\0\13\b\0\0\7g\366\35|\0\200\0\5N\223J\200f&\b.\0\3\0\tf\22\b.\0\4\0\tg\360\b\356\0\4\0\tp\0`\f\b\356\0\3\0\tp\375`\2p\377%|\0\351`\1\0\24L\337\6\0NuH\347|@\25|\0001\0\5(\3\266\274\0\0\1\0c\b*<\0\0\1\0`\2*\3%I\0\f5E\0\n\b.\0\3\0\tf\0\0\230\b.\0\0\0\rg\356\b.\0\3\0\tf\0\0\206\20.\0\13\b\0\0\7g\354p\0\20.\0\31\341\210\20.\0\33\341\210\20.\0\35\"\4\222\200p\0000*\0\n$\5\224\200\222\202g\b\323\252\0\f\223j\0\n\24\274\377\377\25|\0\200\0\7NqNqNqNqNq\b.\0\3\0\tf2\b.\0\4\0\tf\6\b\22\0\7g\352\b*\0\1\0\1f\0\377|J*\0\1f\30Jj\0\nf\22\323\305\230\205\226\205f\0\377Lp\0`\20p\375`\6p\376`\2p\377\25|\0\20\0\7L\337\2>NuH\347|@\25|\0\261\0\5\266\274\0\0\1\0c\b*<\0\0\1\0`\2*\3%I\0\f5E\0\n\b.\0\3\0\tfR\b.\0\0\0\rf\360\24\274\377\377\25|\0\200\0\7NqNqNqNqNq\b.\0\3\0\tf.\b.\0\4\0\tf\6\b\22\0\7g\352\b*\0\1\0\1f\276J*\0\1f\26Jj\0\nf\20\323\305\230\205\226\205f\220p\0`\20p\375`\6p\376`\2p\377\25|\0\20\0\7L\337\2>NuH\347@B,z\365\216\35|\0\220\0\3\209\0\355\0o\f\0\0Vg:\23\374\0001\0\350\340\r\275\374\0\351` f\n\23\374\0\7\0\355\0p`\b\23\374\0\17\0\355\0p\23\374\0\0\0\355\0q\23\374\0V\0\355\0o\23\374\0\0\0\350\340\r\209\0\355\0p\2\0\0\7\35@\0\1p\0\35@\0\5\35@\0\21\35@\0\31\35@\0\33\35@\0\35\35@\0\27p\200\275\374\0\351` f\4rl`\6\"<\0\0\0\366C\372\08NO\35|\0\20\0\3\35|\0\0\0\13p\2a\0\6X\35|\0\20\0\5p\5a\0\6L\35|\0\0\0\5 <\0\0\234@a\0\6<L\337B\2NuH\347\300\2,z\364\304\20.\0\t\35@\0\tL\337@\3NsH\347\t\2,z\364\256\35|\0\0\0\21\20.\0\r\2\0\0\370f\366\35|\0`\0\5`\30H\347\t\2,z\364\216\35|\0\0\0\21\20.\0\r\2\0\0\370f\366\2D\0\7\20<\0\1\351(\t9\0\355\0qf\f\200.\0\1\35|\0\20\0\3`\6\35|\0\0\0\3\35@\0\0270<\t\304\35@\0\33\340H\35@\0\31\35|\0\3\0\35\35n\0\t\0\t\35|\0 \0\5p\1a\0\5\232\20.\0\tf\b\35|\0\5\0\rf\362\20.\0\r\b\0\0\7g\232\20.\0\tg\360\260<\0\4g&\35@\0\t\260<\0\20g\fH@\20.\0\13L\337@\220Nup\0L\337@\220Nup\377L\337@\220Nup\1a\0\5J\35|\0\0\0\27 <\0\0\2X\35@\0\35\340\210\35@\0\33\340\210\35@\0\31\35|\0\4\0\tp\2a\0\5\"\20.\0\tg\372\35@\0\t\260<\0\4g\b\260<\0\20g\254`\236\b.\0\5\0\rf\370\35n\0\t\0\t\b.\0\7\0\rf\224`\206H\347\20\2,z\363\200\20\21\2\0\0\340\f\0\0\0g\16\260<\0 g\f\260<\0\240g\n`\nv\6`\6v\n`\2v\f\20.\0\t\b\0\0\5f,\20.\0\13\b\0\0\7g\354\2\0\0\7\f\0\0\2f\16a\0\2\246H@f\6L\337@\bNu\20.\0\13L\337@\bNua\0\375\216p\377L\337@\bNuH\347\0\2,z\363\22\20.\0\t\b\0\0\5f,\20.\0\13\b\0\0\7g\354\2\0\0\7\260<\0\0f\16a\0\1fH@f\6L\337@\0Nu\20.\0\13L\337@\0Nua\0\375Dp\377L\337@\0NuH\347\0\2,z\362\310\20.\0\t\b\0\0\5f,\20.\0\13\b\0\0\7g\354\2\0\0\7\260<\0\1f\16a\0\1\234H@f\6L\337@\0Nu\20.\0\13L\337@\0Nua\0\374\372p\377L\337@\0NuH\347\20\2,z\362~\20.\0\t\b\0\0\5f.\20.\0\13\b\0\0\7g\354\2\0\0\7\260<\0\3f\20v\1a\0\2\6H@f\6L\337@\bNu\20.\0\13L\337@\bNua\0\374\256p\377L\337@\bNuH\347\20\2,z\3622\20.\0\t\b\0\0\5f.\20.\0\13\b\0\0\7g\354\2\0\0\7\260<\0\7f\20v\1a\0\1\272H@f\6L\337@\bNu\20.\0\13L\337@\bNua\0\374bp\377L\337@\bNuH\347\20\2,z\361\346\20.\0\t\b\0\0\5f,\20.\0\13\b\0\0\7g\354\2\0\0\7\260<\0\6f\16v\1a8H@f\6L\337@\bNu\20.\0\13L\337@\bNua\0\374\30p\377L\337@\bNuH\347\0\2,z\361\234p\0\20.\0\13L\337@\0Nup\4NuH\347\20@ \3\35@\0\35\340\210\35@\0\33\340\210\35@\0\31\20.\0\13\2\0\0\7\35@\0\21\20.\0\13\b\0\0\7g\366\35n\0\t\0\t\35|\0\200\0\5\20.\0\r\2\0\0\360\260<\0pg\6\260<\0\260f\354J.\0\tf\20\b.\0\1\0\rf\362\35Y\0\25S\203f\352\20.\0\tg\372\35@\0\t\260<\0\20g\6L\337\2\bNup\0L\337\2\bNuH\347\20@\20.\0\13\2\0\0\7\35@\0\21 \3\35@\0\35\340\210\35@\0\33\340\210\35@\0\31\35n\0\t\0\t\35|\0\200\0\5\20.\0\r\2\0\0\360\260<\0pg\6\260<\0\260f\354J.\0\tf\20\b.\0\0\0\rf\362\22\356\0\25S\203f\352\20.\0\tg\372\35@\0\t\260<\0\20g\6L\337\2\bNup\0L\337\2\bNuH\347\20@\20.\0\13\2\0\0\7\35@\0\21\20.\0\13\b\0\0\7g\366\35Y\0\27\35|\0\354\0\5\20.\0\13\b\0\0\7f\366\35|\0\314\0\5S\203f\314p\0L\337\2\bNuH\347\20@\20.\0\13\2\0\0\7\35@\0\21\20.\0\13\b\0\0\7g\366\35|\0\354\0\5\20.\0\13\b\0\0\7f\366\22\356\0\27\35|\0\314\0\5S\203f\314p\0L\337\2\bNu\0\0\0\0\0\0NU\377\360H\347RpE\372\377\360a\0\1,L\337\16JN]Nu\1\0\0\0\0\0NU\377\360H\347RpE\372\377\360a\0\1\16L\337\16JN]Nu\b\0\0\0\0\0NU\377\360H\347Rp&IE\372\377\356C\355\377\360r\5\22\332Q\311\377\374,\2C\355\377\360\23F\0\3\340\216\23F\0\2\340\216\23F\0\1\23C\0\4a\0\0\222J\200f\0\0\202\341\213\353\253\"Ka\0\361F\f\200\377\377\377\377gn\f\200\377\377\377\376g\fa\0\0\272L\337\16JN]Nua\0\0\256J\200f\2p\376L\337\16JN]Nu%\0\0\0\0\0\0\0\0\0NU\377\360H\347Rp&IC\355\377\360E\372\377\346r\t\22\332Q\311\377\374C\355\377\360a(J\200f\32\"Kv\ba\0\374Z\f\200\377\377\377\377g\na^L\337\16JN]Nup\377L\337\16JN]NuH\347H\0002<\0\1a\0\372lJ\200g\6Q\311\377\366`\30HD\353\f\211)\0\1a\0\373dJ\200f\bp\0L\337\0\22Nup\377L\337\0\22NuC\355\377\360r\5\22\332Q\311\377\374C\355\377\360a\270J\200f$C\355\377\377a\0\3744J\200f\30C\355\377\376a\0\374tJ\200f\f\20-\377\376H@\20-\377\377Nup\377NuH\347\340\200A\371\0\350\200#r\0\22\20\22\20t\0\24\20\264\20e\370\222Bd\4\322|\0\310\303B\220\202b\352L\337\1\7NuH\347\370Bp\2002<\1@C\372\0\306NO!\300\f\300p\2002<\1AC\372\2\222NO!\300\f\304p\2002<\1CC\372\3\4NO!\300\f\310p\2002<\1DC\372\0\334NO!\300\f\314p\2002<\1EC\372\1\334NO!\300\f\320p\2002<\1FC\372\1\276NO!\300\f\324p\2002<\1GC\372\0tNO!\300\f\330p\2002<\1HC\372\0rNO!\300\f\334p\2002<\1KC\372\0pNO!\300\f\340p\2002<\1MC\372\0nNO!\300\f\344p\2002<\1OC\372\0\312NO!\300\f\3502<\200\0t\17\"|\0\0\0\0a\0\2n\322|\1\0Q\312\377\360L\337B\37Nu/8\f\300H\347H\4K\372\364\24`H/8\f\330H\347H\4K\372\375\220`:/8\f\334H\347H\4K\372\3650`,/8\f\340H\347H\4K\372\364\326`\36/8\f\344H\347H\4K\372\364z`\20/8\f\314H\347H\4K\372\375:`\0\0\2x\08\1\2A\360\0\262|\200\0f:\340L\342Ld\4\b\304\0\20\2D\0\7\t9\0\355\0qg$N\225\2\200\377\377\377\36J\200f\np\0L\337 \22X\217Nu\0\200\377\377\377\0L\337 \22X\217NuL\337 \22Nu/8\f\350H\347\177Hx\08\1\2A\360\0\262|\200\0fn\"\4\340L\342Ld\4\b\304\0\20\2D\0\7\t9\0\355\0qgNI\371\0\0\t\376 \1\340X\300\274\0\0\0\17\331\300\20\24\b\0\0\7f4\300<\0\177g.$<\0\1V`C\372\2\204\260<\0\24g\32$<\0\2\254\300C\372\2\210\260<\0(g\n$<\0\0\257PC\372\2Pa\0\363\0L\337\22\376X\217NuL\337\22\376Nu\2A\360\0\262|\200\0f\nL\337\0\2X\217p\0NuL\337\0\2Nu/8\f\324H\347~dK\372\357\352`\20/8\f\320H\347~dK\372\360J`\0\0\2x\08\1\2A\360\0\262|\200\0fb\340L\342Ld\4\b\304\0\20\2D\0\7\t9\0\355\0qgL,\3&\6\326\274\0\0\0\377\340\213\266\274\0\0\1\0c\6&<\0\0\1\0z\0N\225\2\200\377\377\377\36J\200f\26\324\203\"\3\341\211\323\301\234\201b\316L\337&~X\217p\0NuL\337&~X\217\0\200\377\377\377\0NuL\337&~NuNT\377\0H\347~`x\08\1\2A\360\0\262|\200\0f`\"\4\340L\342Ld\4\b\304\0\20\2D\0\7\t9\0\355\0qgH$I,\3&\6\266\274\0\0\1\0e\6&<\0\0\1\0C\354\377\0a\0\377$*\3S\205\265\tf\24Q\315\377\372R\202\234\203b\326L\337\6~N\\p\0Nup\376L\337\6~N\\\0\200\377\377\377\0NuL\337\6~N\\/8\f\304NuNT\377\0H\347xDx\08\1\2A\360\0\262|\200\0fF\340L\342Ld\4\b\304\0\20\2D\0\7\t9\0\355\0qg0 \tg8v\na\0\361\326\2\200\377\377\377\36J\200f\16a\0\0\214p\0L\337\"\36N\\Nu\0\200\377\377\377\0L\337\"\36N\\NuL\337\"\36N\\/8\f\310Nuv\nC\372\0\264a\0\361\232\2\200\377\377\377\36J\200f\322C\354\377\0t\4v\1z\0a\0\356V\2\200\377\377\377\36J\200f\272E\354\377\0C\372\0h\f\252X68K\0\0f\250C\372\0Z *\0\4\260\274\0\0\237\331e\0\377zC\351\0\24\260\274\0\1=\35e\0\377lC\351\0\24`\0\377dK\371\0\0\t\376 \1\340X\300\274\0\0\0\17\333\300\20<\0(\f)\0\7\0\3g\20\20<\0\24\f)\0\2\0\4g\4\20<\0\n\32\200B\200Nu\1\1\0\3\0015\200\0\0\0\1\1\0\3\1T\200\0\0\0\1\1\0\3\2f\200\0\0\0\1\1\0\3\2\230\200\0\0\0\1\1\0\7\2f\200\0\0\0\1\1\0\7\2\230\200\0\0\0".getBytes (XEiJ.ISO_8859_1);

  //----------------------------------------------------------------------------------------
  //SCSIパーティションIPL
  //  各パーティションの先頭に書き込まれる
  //  HUMAN.SYSを読み込んで起動する
/*
  public static final int[] SPC_PARTITION_IPL = {
    //  perl -e "do'sjdump.pl';$p=0x8000;$m=2;$o=0x7bc2;$l=0x7e88-$o;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    0x60,0x24,0x53,0x48,0x41,0x52,0x50,0x2f,0x4b,0x47,0x20,0x20,0x20,0x20,0x31,0x2e,  //00008000  `$SHARP/KG    1.
    0x30,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00008010  00..............
    0x00,0x00,0x00,0x00,0x00,0x00,0x4f,0xfa,0xff,0xd8,0x4e,0x56,0xff,0xfc,0x70,0xf5,  //00008020  ......O..ﾘNV..p.
    0x72,0x25,0x43,0xfa,0x02,0x92,0x4e,0x4f,0x22,0x29,0x00,0x04,0xe0,0x89,0xe2,0x89,  //00008030  r%C..誰O")..煢竕
    0x2d,0x41,0xff,0xfc,0x74,0x01,0xd4,0xba,0xff,0xda,0x26,0x3c,0x00,0x00,0x04,0x00,  //00008040  -A..t.ﾔｺ.ﾚ&<....
    0x43,0xfa,0x02,0x74,0x61,0x00,0x00,0xf2,0xb0,0xbc,0x00,0x00,0x00,0x00,0x66,0x00,  //00008050  C..ta...ｰｼ....f.
    0x01,0x0a,0x42,0x81,0x12,0x3a,0xff,0xaf,0x42,0x82,0x34,0x3a,0xff,0xaa,0x42,0x83,  //00008060  ..B..:.ｯB.4:.ｪB.
    0x16,0x3a,0xff,0xab,0x42,0x85,0x3a,0x3a,0xff,0xa0,0xc2,0xc3,0xd4,0x81,0xd4,0xba,  //00008070  .:.ｫB.::..ﾂﾃﾔ.ﾔｺ
    0xff,0xa2,0x43,0xfa,0x02,0x42,0x26,0x3c,0x00,0x00,0x04,0x00,0x61,0x00,0x00,0xba,  //00008080  .｢C..B&<....a..ｺ
    0x4a,0x80,0x66,0x00,0x00,0xd6,0x3c,0x3c,0x00,0x1f,0x24,0x49,0x47,0xfa,0x02,0x06,  //00008090  J.f..ﾖ<<..$IG...
    0x7e,0x0a,0x10,0x1a,0x80,0x3c,0x00,0x20,0xb0,0x1b,0x66,0x06,0x51,0xcf,0xff,0xf4,  //000080a0  ~....<. ｰ.f.Qﾏ..
    0x60,0x22,0xd3,0xfc,0x00,0x00,0x00,0x20,0x51,0xce,0xff,0xe0,0x43,0xfa,0x01,0x59,  //000080b0  `"ﾓ.... Qﾎ.澆..Y
    0x2f,0x09,0x43,0xfa,0x00,0xe3,0x61,0x00,0x00,0xba,0x22,0x5f,0x61,0x00,0x00,0xb4,  //000080c0  /.C..綢..ｺ"_a..ｴ
    0x70,0xfe,0x4e,0x4f,0xea,0x8d,0xd4,0x85,0x7a,0x00,0x3a,0x29,0x00,0x1a,0xe0,0x5d,  //000080d0  p.NO鼾ﾔ.z.:)..濔
    0x55,0x85,0x10,0x3a,0xff,0x30,0xca,0xc0,0xd4,0x85,0x48,0xe7,0x70,0x00,0x43,0xfa,  //000080e0  U..:.0ﾊﾀﾔ.H輛.C.
    0x01,0xd6,0x26,0x3c,0x00,0x00,0x04,0x00,0x61,0x4e,0x4c,0xdf,0x00,0x0e,0x43,0xfa,  //000080f0  .ﾖ&<....aNLﾟ..C.
    0x01,0xc6,0x0c,0x59,0x48,0x55,0x66,0x6a,0x54,0x89,0x0c,0x99,0x00,0x00,0x68,0x00,  //00008100  .ﾆ.YHUfjT.....h.
    0x66,0x68,0x2f,0x19,0x26,0x19,0xd6,0x99,0x2f,0x03,0x2f,0x19,0x22,0x7c,0x00,0x00,  //00008110  fh/.&.ﾖ././."|..
    0x67,0xc0,0xd6,0xbc,0x00,0x00,0x00,0x40,0x61,0x1e,0x22,0x1f,0x24,0x1f,0x22,0x5f,  //00008120  gﾀﾖｼ...@a.".$."_
    0x4a,0x80,0x66,0x36,0x41,0xf9,0x00,0x00,0x68,0x00,0xd1,0xc2,0x53,0x81,0x65,0x04,  //00008130  J.f6A...h.ﾑﾂS‘.
    0x42,0x18,0x60,0xf8,0x4e,0x5e,0x4e,0xd1,0x48,0xe7,0x3c,0x00,0x2a,0x2e,0xff,0xfc,  //00008140  B.`.N^NﾑH.<.*...
    0xd6,0xbc,0x00,0x00,0x03,0xff,0xe0,0x8b,0xea,0xab,0xe5,0x8a,0xea,0xaa,0x70,0xf5,  //00008150  ﾖｼ....煖.ｫ蜉.ｪp.
    0x72,0x21,0x4e,0x4f,0x4c,0xdf,0x00,0x3c,0x4e,0x75,0x43,0xfa,0x00,0xcf,0x60,0x00,  //00008160  r!NOLﾟ.<NuC..ﾏ`.
    0xff,0x50,0x43,0xfa,0x00,0xe6,0x60,0x00,0xff,0x48,0x43,0xfa,0x01,0x00,0x60,0x00,  //00008170  .PC..訌..HC...`.
    0xff,0x40,0x70,0x21,0x4e,0x4f,0x4e,0x75,0x1a,0x53,0x43,0x53,0x49,0x20,0x49,0x50,  //00008180  .@p!NONu.SCSI IP
    0x4c,0x20,0x43,0x6f,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x31,0x39,0x39,0x30,  //00008190  L Copyright 1990
    0x20,0x53,0x48,0x41,0x52,0x50,0x00,0x1b,0x5b,0x34,0x37,0x6d,0x1b,0x5b,0x31,0x33,  //000081a0   SHARP..[47m.[13
    0x3b,0x32,0x36,0x48,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //000081b0  ;26H            
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //000081c0                  
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //000081d0                  
    0x20,0x1b,0x5b,0x31,0x34,0x3b,0x32,0x36,0x48,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //000081e0   .[14;26H       
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //000081f0                  
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //00008200                  
    0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x1b,0x5b,0x31,0x34,0x3b,0x33,0x35,0x48,0x48,  //00008210        ..[14;35HH
    0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,0x82,0xaa,0x20,0x8c,0xa9,0x82,0xc2,  //00008220  uman.sys が 見つ
    0x82,0xa9,0x82,0xe8,0x82,0xdc,0x82,0xb9,0x82,0xf1,0x00,0x1b,0x5b,0x31,0x34,0x3b,  //00008230  かりません..[14;
    0x33,0x38,0x48,0x83,0x66,0x83,0x42,0x83,0x58,0x83,0x4e,0x82,0xaa,0x81,0x40,0x93,  //00008240  38Hディスクが　読
    0xc7,0x82,0xdf,0x82,0xdc,0x82,0xb9,0x82,0xf1,0x00,0x1b,0x5b,0x31,0x34,0x3b,0x33,  //00008250   めません..[14;3
    0x36,0x48,0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,0x82,0xaa,0x20,0x89,  //00008260  6HHuman.sys が 壊
    0xf3,0x82,0xea,0x82,0xc4,0x82,0xa2,0x82,0xdc,0x82,0xb7,0x00,0x1b,0x5b,0x31,0x34,  //00008270   れています..[14
    0x3b,0x33,0x33,0x48,0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,0x82,0xcc,  //00008280  ;33HHuman.sys の
    0x20,0x83,0x41,0x83,0x68,0x83,0x8c,0x83,0x58,0x82,0xaa,0x88,0xd9,0x8f,0xed,0x82,  //00008290   アドレスが異常で
    0xc5,0x82,0xb7,0x00,0x68,0x75,0x6d,0x61,0x6e,0x20,0x20,0x20,0x73,0x79,0x73,0x00,  //000082a0   す.human   sys.
    0x53,0x43,0x53,0x49,0x20,0x49,0x50,0x4c,0x20,0x76,0x65,0x72,0x73,0x69,0x6f,0x6e,  //000082b0  SCSI IPL version
    0x20,0x31,0x2e,0x30,0x31,0x00,                                                    //000082c0   1.01.          
  };
*/
  //  perl misc/itob.pl xeij/SPC.java SPC_PARTITION_IPL
  public static final byte[] SPC_PARTITION_IPL = "`$SHARP/KG    1.00\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0O\372\377\330NV\377\374p\365r%C\372\2\222NO\")\0\4\340\211\342\211-A\377\374t\1\324\272\377\332&<\0\0\4\0C\372\2ta\0\0\362\260\274\0\0\0\0f\0\1\nB\201\22:\377\257B\2024:\377\252B\203\26:\377\253B\205::\377\240\302\303\324\201\324\272\377\242C\372\2B&<\0\0\4\0a\0\0\272J\200f\0\0\326<<\0\37$IG\372\2\6~\n\20\32\200<\0 \260\33f\6Q\317\377\364`\"\323\374\0\0\0 Q\316\377\340C\372\1Y/\tC\372\0\343a\0\0\272\"_a\0\0\264p\376NO\352\215\324\205z\0:)\0\32\340]U\205\20:\3770\312\300\324\205H\347p\0C\372\1\326&<\0\0\4\0aNL\337\0\16C\372\1\306\fYHUfjT\211\f\231\0\0h\0fh/\31&\31\326\231/\3/\31\"|\0\0g\300\326\274\0\0\0@a\36\"\37$\37\"_J\200f6A\371\0\0h\0\321\302S\201e\4B\30`\370N^N\321H\347<\0*.\377\374\326\274\0\0\3\377\340\213\352\253\345\212\352\252p\365r!NOL\337\0<NuC\372\0\317`\0\377PC\372\0\346`\0\377HC\372\1\0`\0\377@p!NONu\32SCSI IPL Copyright 1990 SHARP\0\33[47m\33[13;26H                                             \33[14;26H                                             \0\33[14;35HHuman.sys \202\252 \214\251\202\302\202\251\202\350\202\334\202\271\202\361\0\33[14;38H\203f\203B\203X\203N\202\252\201@\223\307\202\337\202\334\202\271\202\361\0\33[14;36HHuman.sys \202\252 \211\363\202\352\202\304\202\242\202\334\202\267\0\33[14;33HHuman.sys \202\314 \203A\203h\203\214\203X\202\252\210\331\217\355\202\305\202\267\0human   sys\0SCSI IPL version 1.01\0".getBytes (XEiJ.ISO_8859_1);



  //========================================================================================
  //$$SPI SCSIポート
  public static final class SPCChip {

    public boolean spiExpansion;  //false=内蔵SCSI,true=拡張SCSI
    public int spiDMAChannel;  //使用するDMAのチャンネル。1=内蔵SCSI,2=拡張SCSI

    //レジスタ
    //  ゼロ拡張
    public int spiBdid;  //0x01
    public int spiSctl;  //0x03
    public int spiScmd;  //0x05
    public int spiInts;  //0x09
    public int spiPsns;  //0x0b
    public int spiSsts;  //0x0d
    public int spiSerr;  //0x0f
    public int spiPctl;  //0x11
    public int spiMbc;   //0x13
    public int spiDreg;  //0x15
    public int spiTemp;  //0x17
    public int spiTchTcmTcl;   //0x19,0x1b,0x1d

    public int spiTargetBase;  //現在のspcUnitArrayの開始番号。リセット時に設定される。-8=ユニットなし
    public SCUnit spiTargetUnit;  //現在のターゲット。null=バスフリー

    public byte[] spiReadHandle;  //転送(ターゲット→イニシエータ)の対象の配列
    public byte[] spiWriteHandle;  //転送(イニシエータ→ターゲット)の対象の配列
    public int spiBufferIndex;  //次に転送する位置
    public int spiBufferLimit;  //転送を終了する位置
    public int spiBufferCount;  //バッファの充填または排出を行う残り回数
    public final byte[] spiCommandBuffer = new byte[16];  //コマンドバッファ
    public int spiLUN;  //LUN
    public final byte[] spiStatusBuffer = new byte[1];  //ステータスバッファ
    public final byte[] spiMessageOutBuffer = new byte[1];  //メッセージアウトバッファ
    public final byte[] spiMessageInBuffer = new byte[1];  //メッセージインバッファ
    public final byte[][] spiSenseBuffer = new byte[][] {
      new byte[8],
      new byte[8],
      new byte[8],
      new byte[8],
      new byte[8],
      new byte[8],
      new byte[8],
      new byte[8],
    };  //センスバッファ
    public final byte[] spiDataInBuffer = new byte[804];  //データインバッファ(Inquiry/Mode Sense(6)/Read Capacity/Read TOC)
    public final byte[] spiDataOutBuffer = new byte[255];  //データアウトバッファ(Mode Select(6)/Assign Drive(SASI))

    public SPCChip (boolean expansion) {

      spiExpansion = expansion;
      spiDMAChannel = expansion ? 2 : 1;

      spiReset (-8);

    }  //SPCChip(int)

    //spcChip.spiReset (targetBase)
    //  SPCリセット
    public void spiReset (int targetBase) {

      spiTargetBase = targetBase;

      spiBdid = SPC_BDID_I7;  //自分のSCSI-IDは7
      spiSctl = SPC_SCTL_RD;  //ハードウェアリセット
      spiScmd = 0;
      spiInts = 0;
      spiPsns = 0;
      spiSsts = 0;
      spiSerr = 0;
      spiPctl = 0;
      spiMbc = 0;
      spiDreg = 0;
      spiTemp = 0;
      spiTchTcmTcl = 0;
      spiUpdateSSTS ();

      spiTargetUnit = null;

      spiReadHandle = null;
      spiWriteHandle = null;
      spiBufferIndex = 0;
      spiBufferLimit = 0;

    }  //spiReset(int)

    //spiTini ()
    //  後始末
    public void spiTini () {

      //イメージファイルに書き出す
      for (SCUnit unit : SPC.spcUnitArray) {
        unit.scuTini ();
      }

    }  //spiTini()

    //d = spcChip.spiPeek (a)
    //  SPCポートピーク
    //  ゼロ拡張
    public int spiPeek (int a) {
      int d = 0;
      switch (a & 31) {
      case SPC_BDID:  //0x01
        d = spiBdid;  //8bitで読み出す
        break;
      case SPC_SCTL:  //0x03
        d = spiSctl;
        break;
      case SPC_SCMD:  //0x05
        d = spiScmd;
        break;
      case SPC_INTS:  //0x09
        d = spiInts;
        break;
      case SPC_PSNS:  //0x0b
        d = spiPsns;
        break;
      case SPC_SSTS:  //0x0d
        d = spiSsts;
        break;
      case SPC_SERR:  //0x0f
        d = spiSerr;
        break;
      case SPC_PCTL:  //0x11
        d = spiPctl;
        break;
      case SPC_MBC:   //0x13
        d = spiMbc;
        break;
      case SPC_DREG:  //0x15
        d = spiDreg;
        break;
      case SPC_TEMP:  //0x17
        d = spiTemp;
        break;
      case SPC_TCH:   //0x19
        d = spiTchTcmTcl >>> 16;
        break;
      case SPC_TCM:   //0x1b
        d = (char) spiTchTcmTcl >>> 8;
        break;
      case SPC_TCL:   //0x1d
        d = spiTchTcmTcl & 255;
        break;
      }
      if (SPC_DEBUG_PORT) {
        System.out.printf ("%08x spiPeek(0x%08x(%s))=0x%02x\n", XEiJ.regPC0, a, SPC_REGISTER_NAME[a & 31], d);
      }
      return d;
    }  //spiPeek(int)

    //d = spcChip.spiRead (a)
    //  SPCポートリード
    //  ゼロ拡張
    public int spiRead (int a) {
      int d = 0;
      switch (a & 31) {
      case SPC_BDID:  //0x01
        d = spiBdid;  //8bitで読み出す
        break;
      case SPC_SCTL:  //0x03
        d = spiSctl;
        break;
      case SPC_SCMD:  //0x05
        d = spiScmd;
        break;
      case SPC_INTS:  //0x09
        if (HostCDROM.HCD_ENABLED &&
            HostCDROM.hcdConnected &&
            spiTargetUnit != null &&
            HostCDROM.hcdSCSIId == spiTargetUnit.abuNumber) {
          HostCDROM.hcdReadINTSPSNS ();
        }
        d = spiInts;
        break;
      case SPC_PSNS:  //0x0b
        if (HostCDROM.HCD_ENABLED &&
            HostCDROM.hcdConnected &&
            spiTargetUnit != null &&
            HostCDROM.hcdSCSIId == spiTargetUnit.abuNumber) {
          HostCDROM.hcdReadINTSPSNS ();
        }
        d = spiPsns;
        break;
      case SPC_SSTS:  //0x0d
        d = spiSsts;
        break;
      case SPC_SERR:  //0x0f
        d = spiSerr;
        break;
      case SPC_PCTL:  //0x11
        d = spiPctl;
        break;
      case SPC_MBC:   //0x13
        d = spiMbc;
        break;
      case SPC_DREG:  //0x15
        if ((spiSsts & SPC_SSTS_TRIP) != 0 && spiTchTcmTcl != 0) {  //転送中
          if (spiReadHandle != null && spiBufferIndex < spiBufferLimit) {
            spiDreg = spiReadHandle[spiBufferIndex++] & 255;  //データを入力する
            if (spiBufferIndex == spiBufferLimit && spiBufferCount != 0) {  //バッファを再充填する必要がある
              spiTargetUnit.scuReadImage ();
              spiBufferCount--;
              spiBufferIndex = 0;
            }
          }
          spiTchTcmTcl--;  //0でなかったのだから負になることはない
          spiUpdateSSTS ();
          if (spiBufferCount == 0) {  //最後のバッファ
            if (spiTchTcmTcl == 0) {  //転送終了
              spiTransferComplete ();
            } else if (SPC_EXPOSE_DATAINI_BUG &&
                       spiTchTcmTcl == 8 &&  //残りが8バイトになった
                       (spiPsns & SPC_PHASE_MASK) == SPC_DATA_IN_PHASE) {  //データインフェーズ
              spiPsns = (spiPsns & ~SPC_PHASE_MASK) | SPC_STATUS_PHASE;  //ステータスフェーズに切り替える
              spiSetInterruptStatus (SPC_INTS_SR);  //INTSのSRをセットする
            }
          }
        }
        d = spiDreg;
        break;
      case SPC_TEMP:  //0x17
        d = spiTemp;
        break;
      case SPC_TCH:   //0x19
        d = spiTchTcmTcl >>> 16;
        break;
      case SPC_TCM:   //0x1b
        d = (char) spiTchTcmTcl >>> 8;
        break;
      case SPC_TCL:   //0x1d
        d = spiTchTcmTcl & 255;
        break;
      }
      if (SPC_DEBUG_PORT) {
        System.out.printf ("%08x spiRead(0x%08x(%s))=0x%02x\n", XEiJ.regPC0, a, SPC_REGISTER_NAME[a & 31], d);
      }
      return d;
    }  //spiRead(int)

    //d = spcChip.spiWrite (a, d)
    //  SPCポートライト
    public void spiWrite (int a, int d) {
      d &= 255;
      if (SPC_DEBUG_PORT) {
        System.out.printf ("%08x spiWrite(0x%08x(%s),0x%02x)\n", XEiJ.regPC0, a, SPC_REGISTER_NAME[a & 31], d & 255);
      }
      switch (a & 31) {
      case SPC_BDID:  //0x01
        spiBdid = 1 << (d & 7);  //3bitで書き込む
        break;
      case SPC_SCTL:  //0x03
        spiSctl = d;
        break;
      case SPC_SCMD:  //0x05
        spiScmd = d;
        switch (spiScmd & SPC_SCMD_CC) {
        case SPC_SCMD_CC_BR:  //Bus Release。ターゲットのときバスフリーフェーズへ移行
          if (spiTargetUnit != null) {
            spiBusFreePhase ();  //バスフリーフェーズに移行する
          }
          break;
        case SPC_SCMD_CC_SL:  //Select。セレクション/リセレクションを開始
          {
            if ((spiPctl & SPC_PCTL_SR) == SPC_PCTL_SR_R) {  //リセレクション
              //!!!
              if (SPC_REPORT_UNIMPLEMENTED_COMMAND) {
                System.out.println (String.format ("%08x Unimplemented Command: Reselection\n", XEiJ.regPC0));
              }
              spiSetInterruptStatus (SPC_INTS_RC);  //Reset Conditionで強制終了
              break;
            }
            int u = 0;
            SCUnit unit = null;
            boolean timeOut = false;
            if (spiTargetBase < 0) {  //接続するユニットが存在しない
              timeOut = true;
            } else {
              u = Integer.numberOfTrailingZeros (spiTemp & ~spiBdid);  //ターゲットのID
              if (u > 7) {  //ターゲットのIDが指定されていないか、自分のSCSI-IDと衝突している
                timeOut = true;
              } else {
                unit = SPC.spcUnitArray[spiTargetBase + u];  //ターゲットのユニット
                if (!unit.isConnected ()) {  //ユニットは存在するが接続されていない
                  timeOut = true;
                }
              }
            }
            if (timeOut) {
              //ユニットが存在しないときセレクションフェーズを開始して直ちにタイムアウトにする
              spiSsts |= SPC_SSTS_INIT | SPC_SSTS_BUSY;
              spiPsns |= SPC_PSNS_SEL;
              spiTchTcmTcl = 0;
              spiSetInterruptStatus (SPC_INTS_TO);  //Time Out
              spiUpdateSSTS ();
              break;
            }
            spiTargetUnit = unit;  //接続する
            spiSsts |= SPC_SSTS_INIT;  //自分がイニシエータになる
            spiSetInterruptStatus (SPC_INTS_CC);  //コマンド終了
            if ((spiPsns & SPC_PSNS_ATN) != 0) {  //ATN=1
              spiMessageOutPhase ();  //メッセージアウトフェーズに移行する
            } else {
              spiCommandPhase ();  //コマンドフェーズに移行する
            }
          }
          break;
        case SPC_SCMD_CC_RA:  //Reset ATN。ATNをクリア
          spiPsns &= ~SPC_PSNS_ATN;
          break;
        case SPC_SCMD_CC_SA:  //Set ATN。ATNをセット
          spiPsns |= SPC_PSNS_ATN;
          break;
        case SPC_SCMD_CC_TR:  //Transfer。転送開始
          spiUpdateSSTS ();
          spiSsts |= SPC_SSTS_BUSY | SPC_SSTS_TRIP;  //転送開始
          break;
        case SPC_SCMD_CC_TP:  //Transfer Pause。転送中断
          //!!!
          if (SPC_REPORT_UNIMPLEMENTED_COMMAND) {
            System.out.println (String.format ("%08x Unimplemented Command: Transfer Pause\n", XEiJ.regPC0));
          }
          break;
        case SPC_SCMD_CC_RR:  //Reset ACK/REQ。CPU転送のときACK/REQをクリア
          if ((spiPsns & SPC_PSNS_IO) == 0) {  //Out
            if (spiWriteHandle == null) {  //転送中ではない
              break;
            }
            spiPsns &= ~SPC_PSNS_ACK;  //イニシエータがACKを0にする
            if (spiBufferIndex < spiBufferLimit) {  //継続
              spiPsns |= SPC_PSNS_REQ;  //ターゲットがREQを1にする
              HD63450.dmaFallPCL (spiDMAChannel);
              break;
            }
            spiTransferComplete ();  //転送終了
          } else {  //In
            if (spiReadHandle == null) {  //転送中ではない
              break;
            }
            spiPsns &= ~SPC_PSNS_REQ;  //イニシエータがREQを0にする
            if (spiBufferIndex < spiBufferLimit) {  //継続
              spiPsns |= SPC_PSNS_REQ;  //ターゲットがREQを1にする
              HD63450.dmaFallPCL (spiDMAChannel);
              break;
            }
            spiTransferComplete ();  //転送終了
          }
          break;
        case SPC_SCMD_CC_SR:  //Set ACK/REQ。CPU転送のときACK/REQをセット
          if ((spiPsns & SPC_PSNS_IO) == 0) {  //Out
            if (spiWriteHandle == null) {  //転送中ではない
              break;
            }
            spiPsns |= SPC_PSNS_ACK;  //イニシエータがACKを1にする。spiTempに出力データの準備ができている
            if (spiBufferIndex < spiBufferLimit) {
              spiWriteHandle[spiBufferIndex++] = (byte) spiTemp;  //データを出力する
              if ((spiPctl & SPC_PCTL_TP) == SPC_COMMAND_PHASE &&  //コマンドフェーズの
                  spiBufferIndex == 1) {  //1バイト目
                int oc = spiCommandBuffer[0] & 255;  //オペレーションコード
                spiBufferLimit = SPC_CDB_LENGTH[oc >> 4];  //CDB(Command Descriptor Block)の長さ
              }
              //!!! ディスクイメージ以外のバッファは溢れてはならない
              if (spiBufferIndex == spiBufferLimit && spiBufferCount != 0) {  //バッファから排出する必要がある
                spiTargetUnit.scuWriteImage ();
                if (--spiBufferCount != 0) {
                  spiBufferIndex = 0;
                }
              }
              spiUpdateSSTS ();
            }
            spiPsns &= ~SPC_PSNS_REQ;  //ターゲットがREQを0にする
          } else {  //In
            if (spiReadHandle == null) {  //転送中ではない
              break;
            }
            spiPsns |= SPC_PSNS_ACK;  //イニシエータがACKを1にする。spiReadHandle[spiBufferIndex]に入力データの準備ができている
            if (spiBufferIndex < spiBufferLimit) {
              spiTemp = spiReadHandle[spiBufferIndex++] & 255;  //データを入力する
              if (spiBufferIndex == spiBufferLimit && spiBufferCount != 0) {  //バッファを再充填する必要がある
                spiTargetUnit.scuReadImage ();
                spiBufferCount--;
                spiBufferIndex = 0;
              }
              spiUpdateSSTS ();
            }
            spiPsns &= ~SPC_PSNS_REQ;  //ターゲットがREQを0にする
          }
          break;
        }
        break;
      case SPC_INTS:  //0x09
        //1を書き込んだビットだけ0クリアする
        //  move.b INTS,INTSでクリアできる
        if ((spiPsns & SPC_PSNS_SEL) != 0 &&  //セレクションフェーズで
            (spiInts & d & SPC_INTS_TO) != 0) {  //Time Outをクリアするとき
          spiInts &= ~d;
          if (spiTchTcmTcl != 0) {  //TCH,TCM,TCLが0でないとき
            //再びタイムアウトにする
            spiTchTcmTcl = 0;
            spiSetInterruptStatus (SPC_INTS_TO);  //Time Out
          } else {  //TCH,TCM,TCLが0のとき
            //セレクションフェーズを終了する
            spiPsns &= ~SPC_PSNS_SEL;
            spiSsts &= ~(SPC_SSTS_INIT | SPC_SSTS_BUSY);
            spiUpdateSSTS ();
          }
        } else {
          spiInts &= ~d;
        }
        break;
      case SPC_PSNS:  //0x0b
        spiPsns = d;
        break;
      case SPC_SSTS:  //0x0d
        //Readのみ
        break;
      case SPC_SERR:  //0x0f
        //Readのみ
        break;
      case SPC_PCTL:  //0x11
        spiPctl = d;
        break;
      case SPC_MBC:   //0x13
        //Readのみ
        break;
      case SPC_DREG:  //0x15
        spiDreg = d;
        if ((spiSsts & SPC_SSTS_TRIP) != 0 && spiTchTcmTcl != 0) {  //転送中
          if (spiWriteHandle != null && spiBufferIndex < spiBufferLimit) {
            spiWriteHandle[spiBufferIndex++] = (byte) spiDreg;  //データを出力する
            if ((spiPctl & SPC_PCTL_TP) == SPC_COMMAND_PHASE &&  //コマンドフェーズの
                spiBufferIndex == 1) {  //1バイト目
              int oc = spiCommandBuffer[0] & 255;  //オペレーションコード
              spiBufferLimit = SPC_CDB_LENGTH[oc >> 4];  //CDB(Command Descriptor Block)の長さ
            }
            //!!! ディスクイメージ以外のバッファは溢れてはならない
            if (spiBufferIndex == spiBufferLimit && spiBufferCount != 0) {  //バッファから排出する必要がある
              spiTargetUnit.scuWriteImage ();
              if (--spiBufferCount != 0) {
                spiBufferIndex = 0;
              }
            }
          }
          spiTchTcmTcl--;  //0でなかったのだから負になることはない
          spiUpdateSSTS ();
          if (spiTchTcmTcl == 0 || spiBufferIndex == spiBufferLimit) {  //転送終了。最後のブロックでなければspiBufferIndexは巻き戻されている
            spiTransferComplete ();
          }
        }
        break;
      case SPC_TEMP:  //0x17
        spiTemp = d;
        break;
      case SPC_TCH:   //0x19
        spiTchTcmTcl = d << 16 | (char) spiTchTcmTcl;
        spiUpdateSSTS ();
        break;
      case SPC_TCM:   //0x1b
        spiTchTcmTcl = spiTchTcmTcl & 0xff00ff | d << 8;
        spiUpdateSSTS ();
        break;
      case SPC_TCL:   //0x1d
        spiTchTcmTcl = spiTchTcmTcl & 0xffff00 | d;
        spiUpdateSSTS ();
        break;
      }
    }  //spiWrite(int,int)

    //spiTransferComplete ()
    //  転送終了
    public void spiTransferComplete () {
      if ((spiSsts & SPC_SSTS_TRIP) != 0) {
        spiSetInterruptStatus (SPC_INTS_CC);  //転送終了
        spiSsts &= ~(SPC_SSTS_BUSY | SPC_SSTS_TRIP);
      }
      switch (spiPctl & SPC_PCTL_TP) {  //転送フェーズ
      case SPC_DATA_OUT_PHASE:  //データアウトフェーズの転送が終了した
        {
          int oc = spiCommandBuffer[0] & 255;  //オペレーションコード
          if (HostCDROM.HCD_ENABLED &&
              HostCDROM.hcdConnected &&
              HostCDROM.hcdSCSIId == spiTargetUnit.abuNumber) {
            HostCDROM.hcdDataOutComplete (oc);
            break;
          }
          if (oc == 0x15) {  //Mode Select(6)
            if (SPC_REPORT_UNIMPLEMENTED_COMMAND) {  //未実装コマンドを表示する
              spiTargetUnit.scuPrintDataOut (oc);
            }
          }
        }
        spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
        break;
      case SPC_DATA_IN_PHASE:  //データインフェーズの転送が終了した
        {
          int oc = spiCommandBuffer[0] & 255;  //オペレーションコード
          if (oc == 0x03) {  //Request Sense
            Arrays.fill (spiSenseBuffer[spiLUN], (byte) 0);  //センスデータを消去する
          }
        }
        spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
        break;
      case SPC_COMMAND_PHASE:  //コマンドフェーズの転送が終了した
        {
          int oc = spiCommandBuffer[0] & 255;  //オペレーションコード
          if (spiBufferIndex == 1) {
            spiBufferLimit = SPC_CDB_LENGTH[oc >> 4];  //CDB(Command Descriptor Block)の長さ
            if (spiBufferIndex < spiBufferLimit) {
              //次のデータを要求する
              spiPsns |= SPC_PSNS_REQ;  //ターゲットがREQを1にする
              HD63450.dmaFallPCL (spiDMAChannel);
              break;
            }
          }
          spiLUN = (spiCommandBuffer[1] >> 5) & 7;  //LUN
          if (HostCDROM.HCD_ENABLED &&
              HostCDROM.hcdConnected &&
              HostCDROM.hcdSCSIId == spiTargetUnit.abuNumber) {
            HostCDROM.hcdCommandComplete_1st (spiTargetUnit, oc);
            break;
          }
          spiTargetUnit.scuCommand (oc);  //コマンドを実行する
        }
        break;
      case SPC_STATUS_PHASE:  //ステータスフェーズの転送が終了した
        spiMessageInPhase ();  //メッセージインフェーズに移行する
        break;
      case SPC_MESSAGE_OUT_PHASE:  //メッセージアウトフェーズの転送が終了した
        spiCommandPhase ();  //コマンドフェーズに移行する
        break;
      case SPC_MESSAGE_IN_PHASE:  //メッセージインフェーズの転送が終了した
        spiBusFreePhase ();  //バスフリーフェーズに移行する
        break;
      }
    }  //spiTransferComplete()

    //spiBusFreePhase ()
    //  バスフリーフェーズに移行する
    public void spiBusFreePhase () {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiBusFreePhase()\n", XEiJ.regPC0);
      }
      spiWriteHandle = null;
      spiReadHandle = null;
      spiSsts &= ~SPC_SSTS_INIT;  //自分がイニシエータではなくなる
      spiPsns = 0;
      spiTargetUnit = null;
      if ((spiPctl & SPC_PCTL_IE) != 0) {
        spiSetInterruptStatus (SPC_INTS_DC);
      }
    }  //spiBusFreePhase()

    //spiCommandPhase ()
    //  コマンドフェーズに移行する
    public void spiCommandPhase () {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiCommandPhase()\n", XEiJ.regPC0);
      }
      spiWriteHandle = spiCommandBuffer;
      spiReadHandle = null;
      spiBufferIndex = 0;
      spiBufferLimit = 1;
      spiUpdateSSTS ();
      spiPsns = SPC_PSNS_REQ | SPC_COMMAND_PHASE;
      HD63450.dmaFallPCL (spiDMAChannel);
    }  //spiCommandPhase()

    //spiDataInPhase (handle, index, limit, count)
    //  データインフェーズに移行する
    public void spiDataInPhase (byte[] handle, int index, int limit, int count) {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiDataInPhase(handle,0x%08x,0x%08x,0x%08x)\n", XEiJ.regPC0, index, limit, count);
      }
      spiWriteHandle = null;
      spiReadHandle = handle;
      spiBufferIndex = index;
      spiBufferLimit = limit;
      spiBufferCount = count;
      spiUpdateSSTS ();
      spiPsns = SPC_PSNS_REQ | SPC_DATA_IN_PHASE;
      HD63450.dmaFallPCL (spiDMAChannel);
    }  //spiDataInPhase(byte[],int,int,int)

    //spiDataOutPhase (handle, index, limit, count)
    //  データアウトフェーズに移行する
    public void spiDataOutPhase (byte[] handle, int index, int limit, int count) {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiDataOutPhase(handle,0x%08x,0x%08x,0x%08x)\n", XEiJ.regPC0, index, limit, count);
      }
      spiWriteHandle = handle;
      spiReadHandle = null;
      spiBufferIndex = index;
      spiBufferLimit = limit;
      spiBufferCount = count;
      spiUpdateSSTS ();
      spiPsns = SPC_PSNS_REQ | SPC_DATA_OUT_PHASE;
      HD63450.dmaFallPCL (spiDMAChannel);
    }  //spiDataOutPhase(byte[],int,int,int)

    //spiStatusPhase (status, message)
    //  ステータスフェーズに移行する
    //  status   2=センスデータあり。spiSenseBufferを設定しておくこと
    //  message  常に0
    public void spiStatusPhase (int status, int message) {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiStatusPhase(0x%02x)\n", XEiJ.regPC0, status);
      }
      spiStatusBuffer[0] = (byte) status;
      spiMessageInBuffer[0] = (byte) message;
      spiWriteHandle = null;
      spiReadHandle = spiStatusBuffer;
      spiBufferIndex = 0;
      spiBufferLimit = 1;
      spiTemp = spiStatusBuffer[0] & 255;
      spiUpdateSSTS ();
      spiPsns = SPC_PSNS_REQ | SPC_STATUS_PHASE;
      HD63450.dmaFallPCL (spiDMAChannel);
    }  //spiStatusPhase(int,int)

    //spiMessageInPhase ()
    //  メッセージインフェーズに移行する
    public void spiMessageInPhase () {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiMessageInPhase(0x%02x)\n", XEiJ.regPC0, spiMessageInBuffer[0] & 255);
      }
      spiWriteHandle = null;
      spiReadHandle = spiMessageInBuffer;
      spiBufferIndex = 0;
      spiBufferLimit = 1;
      spiTemp = spiMessageInBuffer[0] & 255;
      spiUpdateSSTS ();
      spiPsns = SPC_PSNS_REQ | SPC_MESSAGE_IN_PHASE;
      HD63450.dmaFallPCL (spiDMAChannel);
    }  //spiMessageInPhase()

    //spiMessageOutPhase ()
    //  メッセージアウトフェーズに移行する
    public void spiMessageOutPhase () {
      if (SPC_DEBUG_PHASE) {
        System.out.printf ("%08x spiMessageOutPhase(0x%02x)\n", XEiJ.regPC0, spiMessageOutBuffer[0] & 255);
      }
      spiWriteHandle = spiMessageOutBuffer;
      spiReadHandle = null;
      spiBufferIndex = 0;
      spiBufferLimit = 1;
      spiUpdateSSTS ();
      spiPsns = SPC_PSNS_REQ | SPC_MESSAGE_OUT_PHASE;
      HD63450.dmaFallPCL (spiDMAChannel);
    }  //spiMessageOutPhase()

    //spiSetInterruptStatus (ints)
    public void spiSetInterruptStatus (int ints) {
      //int oldInts = spiInts;
      if ((ints & ~SPC_INTS_TO) != 0) {  //TO以外をセットするとき
        spiInts &= ~SPC_INTS_TO;  //TOをクリア
      }
      spiInts |= ints;
      if (//oldInts != spiInts &&  //0→1
          (spiSctl & SPC_SCTL_IE) != 0) {  //Interrupt Enable
        if (spiExpansion) {
          XEiJ.eb2Interrupt (XEiJ.EB2_SPC_REQUEST);
        } else {
          IOInterrupt.ioiSpcFall ();
          IOInterrupt.ioiSpcRise ();
        }
      }
    }  //spiSetInterruptStatus(int)

    //spiUpdateSSTS ()
    //  SSTSのTC0,DF,DEを更新する
    public void spiUpdateSSTS () {
      spiSsts = (spiSsts & ~(SPC_SSTS_TC0 | SPC_SSTS_DF | SPC_SSTS_DE) |
                 ((spiPctl & SPC_PCTL_IO) == 0 ?  //Out
                  (spiTchTcmTcl == 0 ? SPC_SSTS_TC0 : 0) |
                  SPC_SSTS_DE  //出力のときFIFOは常にEmptyでFullになることはない
                  :  //In
                  (spiTchTcmTcl != 0 ? 0 : SPC_SSTS_TC0 | SPC_SSTS_DE) |  //入力のとき残りが0バイトでなければFIFOはEmptyではない
                  (spiTchTcmTcl < 8 ? 0 : SPC_SSTS_DF)));  //入力のとき残りが8バイト未満ならばFIFOはFullではない
    }  //spiUpdateSSTS ()

  }  //class SPCChip



  //========================================================================================
  //$$SCU SCSI HDユニット
  //  SCSIハードディスクのユニット
  public static class SCUnit extends AbstractUnit {

    public SPCChip scuChip;  //接続されているSCSIポート
    public int scuMode;  //動作モード。1=SCSI-1,3=SCSI-2
    public RandomAccessFile scuRaf;
    public int scuBytesPerRecord;  //1レコードあたりのバイト数(2の累乗)
    public int scuDiskEndRecord;  //ディスクのレコード数。0=挿入されていない
    public long scuDiskEndByte;  //ディスクのバイト数
    public byte[] scuRecordImage;
    public HDMedia scuSASIMedia;  //SASIモード

    //new SCUnit (number, chip)
    //  コンストラクタ
    public SCUnit (int number, SPCChip chip) {
      super (number);
      scuChip = chip;
      scuMode = 1;  //リセレクションがないのでSCSI-1
      scuRaf = null;
      scuBytesPerRecord = 512;
      scuDiskEndRecord = 0;
      scuDiskEndByte = 0L;
      scuRecordImage = new byte[SPC_MAX_BYTES_PER_BLOCK];
      scuSASIMedia = null;
    }

    public void scuReset (SPCChip chip) {
      scuChip = chip;
    };  //scuReset

    //numberOfModes = unit.abuGetNumberOfModes ()
    //  モードの数を返す
    //  モードボタンを表示するときオーバーライドする
    @Override public int abuGetNumberOfModes () {
      return 2;
    }  //unit.abuGetNumberOfModes()

    //image = unit.abuGetModeIcon (mode, enabled)
    //  モードボタンのアイコンを返す
    //  モードボタンを表示するときオーバーライドする
    @Override public ImageIcon abuGetModeIcon (int mode, boolean enabled) {
      return (enabled ?
              mode == 0 ? LnF.LNF_HD_ICON : LnF.LNF_CD_ICON :
              mode == 0 ? LnF.LNF_HD_DISABLED_ICON : LnF.LNF_CD_DISABLED_ICON);
    }  //unit.abuGetModeIcon(int,boolean)

    //text = unit.abuGetModeTextEn (mode, enabled)
    //  モードボタンの英語のツールチップテキストを返す
    //  モードボタンを表示するときオーバーライドする
    @Override public String abuGetModeTextEn (int mode, boolean enabled) {
      return (enabled ?
              mode == 0 ? "Hard disk mode → CD-ROM mode" : "CD-ROM mode → Hard disk mode" :
              null);
    }  //unit.abuGetModeTextEn(int,boolean)

    //text = unit.abuGetModeTextJa (mode, enabled)
    //  モードボタンの日本語のツールチップテキストを返す
    //  モードボタンを表示するときオーバーライドする
    @Override public String abuGetModeTextJa (int mode, boolean enabled) {
      return (enabled ?
              mode == 0 ? "ハードディスクモード → CD-ROM モード" : "CD-ROM モード → ハードディスクモード" :
              null);
    }  //unit.abuGetModeTextJa(int,boolean)

    //unit.connect (disconnectable)
    //  接続する
    @Override protected void connect (boolean disconnectable) {
      super.connect (disconnectable);
    }

    //unit.disconnect ()
    //  切り離す
    @Override protected void disconnect () {
      super.disconnect ();
    }

    //success = unit.eject ()
    //  イジェクトする
    @Override protected boolean eject () {
      if (scuRaf != null) {  //クローズする
        try {
          scuRaf.close ();
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
        scuRaf = null;
      }
      String path = abuPath;  //イジェクトされたイメージファイルのパス。super.eject()を呼び出す前にコピーすること
      if (!super.eject ()) {  //イジェクトする
        return false;
      }
      if (path.length () != 0) {  //挿入されていたとき
        spcAddHistory (new File (path).getAbsoluteFile ());
        System.out.println (Multilingual.mlnJapanese ?
                            path + " を sc" + abuNumber + " から切り離しました" :
                            path + " was removed from sc" + abuNumber);
      }
      scuDiskEndRecord = 0;
      scuDiskEndByte = 0L;
      return true;
    }

    //success = unit.open ()
    //  開くダイアログを開く
    @Override protected boolean open () {
      if (!super.open ()) {
        return false;
      }
      SPC.spcOpenUnit = abuNumber;
      if (SPC.spcOpenDialog == null) {
        SPC.spcOpenDialog = new OpenDialog ();
        SPC.spcOpenDialog.setReadOnly (Settings.sgsGetOnOff ("screadonly"));
        SPC.spcOpenDialog.setReboot (Settings.sgsGetOnOff ("scappreboot"));
        for (File[] files : SPC.spcOpenHistory) {
          SPC.spcOpenDialog.addHistory (files);
        }
        SPC.spcOpenHistory.clear ();
      }
      SPC.spcOpenDialog.rescanCurrentDirectory ();  //挿入されているファイルが変わると選択できるファイルも変わるのでリストを作り直す
      XEiJ.pnlExitFullScreen (true);
      SPC.spcOpenDialog.setVisible (true);
      return true;
    }  //unit.open()

    //success = unit.insert (path, writeProtected)
    //  挿入する
    @Override protected boolean insert (String path, boolean writeProtected) {
      if (SPC.spcIsInserted (path)) {  //既に挿入されている
        return false;
      }
      if (!super.insert (path, writeProtected)) {  //挿入できなかった
        return false;
      }
      return true;
    }  //unit.insert(String)

    //loaded = unit.load (path)
    //  読み込む
    @Override protected boolean load (String path) {
      File file = new File (path);
      scuSASIMedia = null;
      long longLength = file.length ();
      for (HDMedia media : HDMedia.HDM_ARRAY) {
        if (media.humDiskEndByte == longLength) {  //ファイルサイズが一致
          scuSASIMedia = media;
          break;
        }
      }
      if (scuSASIMedia != null) {  //SASIハードディスク
        abuSetMode (0);
      } else if (SPC.spcIsHds (file, true)) {  //拡張子がHDSで装置初期化されている
        abuSetMode (0);
      } else if (SPC.spcIsIso (file)) {  //拡張子がISO
        abuSetMode (1);
        protect (false);  //開くときに書き込みを禁止した場合はイジェクトするまで書き込みを許可できない
      } else {
        return false;
      }
      try {
        scuRaf = new RandomAccessFile (file, abuWriteProtected ? "r" : "rw");  //RandomAccessFileに"w"というモードはない
      } catch (IOException ioe) {
        return false;  //開けなかった。SPC.spcIsHdsまたはSPC.spcIsISOのチェックを通っても書き込みモードでは開けない可能性がある
      }
      if (scuSASIMedia != null) {  //SASIハードディスク
        scuDiskEndByte = longLength;
        scuBytesPerRecord = 256;
        scuDiskEndRecord = (int) (scuDiskEndByte / scuBytesPerRecord);
      } else if (abuCurrentMode == 0) {  //ハードディスク
        byte[] bb = new byte[512];
        scuDiskEndRecord = 0;
        try {
          scuRaf.seek (0L);
          scuRaf.read (bb, 0, 512);  //セクタ0を読み込む
          if (ByteArray.byaRls (bb, 0) == ('X' << 24 | '6' << 16 | '8' << 8 | 'S') &&
              ByteArray.byaRls (bb, 4) == ('C' << 24 | 'S' << 16 | 'I' << 8 | '1')) {  //X68SCSI1マジックがある
            scuBytesPerRecord = ByteArray.byaRwz (bb, 8);  //1レコードあたりのバイト数(2の累乗)
            scuDiskEndRecord = ByteArray.byaRls (bb, 10);  //ディスクのレコード数
            if (ByteArray.byaRls (bb, 42) == ('S' << 24 | 'x' << 16 | 'S' << 8 | 'I')) {  //SxSI
              scuDiskEndRecord <<= 1;
            }
          } else if (bb[0x0000] == (byte) 0xeb &&
                     bb[0x01fe] == (byte) 0x55 &&
                     bb[0x01ff] == (byte) 0xaa) {  //IBMスーパーフロッピー
            scuBytesPerRecord = 0x0200;  //1レコードあたりのバイト数(2の累乗)
            scuDiskEndRecord = (int) (scuRaf.length () / scuBytesPerRecord);  //ディスクのレコード数
          }
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
        if (scuDiskEndRecord == 0) {
          try {
            scuRaf.close ();
          } catch (IOException ioe) {
            ioe.printStackTrace ();
          }
          scuRaf = null;
          return false;
        }
        scuDiskEndByte = (long) scuBytesPerRecord * scuDiskEndRecord;  //ディスクのバイト数
      } else {  //CD-ROM
        byte[] bb = new byte[2048];
        try {
          scuRaf.seek (2048L * 16);
          scuRaf.read (bb, 0, 2048);  //セクタ16を読み込む
        } catch (IOException ioe) {
          ioe.printStackTrace ();
          try {
            scuRaf.close ();
          } catch (IOException ioe2) {
            ioe.printStackTrace ();
          }
          scuRaf = null;
          return false;  //セクタ16を読み込めなかった
        }
        scuBytesPerRecord = ByteArray.byaRwz (bb, 130);  //ブロックのバイト数
        scuDiskEndRecord = ByteArray.byaRls (bb, 84);  //ボリュームのブロック数
        scuDiskEndByte = (long) scuBytesPerRecord * scuDiskEndRecord;  //ボリュームのバイト数。整合性はSPC.spcIsIsoで確認済みなのでチェックは省略する
      }
      System.out.println (Multilingual.mlnJapanese ?
                          path + " を sc" + abuNumber + " に接続しました" :
                          path + " was connected to sc" + abuNumber);
      return true;
    }

    //scuTini ()
    //  後始末
    public void scuTini () {
      if (scuRaf != null) {
        try {
          scuRaf.close ();
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
        scuRaf = null;
      }
    }  //scuTini()

    //scuCommand (oc)
    //  コマンドを実行する
    //  セレクションフェーズに成功したのだから装置は接続されているはず
    //  ロジカルユニットは存在しない可能性がある
    public void scuCommand (int oc) {
      if (SPC_DEBUG_COMMAND) {
        System.out.printf ("%08x sc%d:scuCommand(0x%02x(%s)) [", XEiJ.regPC0, abuNumber, oc, SPC_COMMAND_NAME[oc]);
        for (int i = 0; i < scuChip.spiBufferLimit; i++) {
          if (i > 0) {
            System.out.print (',');
          }
          System.out.printf ("0x%02x", scuChip.spiCommandBuffer[i] & 255);
        }
        System.out.println (']');
      }
      if (oc != 0x03 &&  //Request Sense以外で
          scuChip.spiLUN != 0) {  //LUNが0でない
        scuNotReady ();
        return;
      }
      switch (oc) {
      case 0x00:  //Test Unit Ready
        scuDoTestUnitReady ();
        break;
      case 0x01:  //Rezero Unit
        scuDoRezeroUnit ();
        break;
      case 0x03:  //Request Sense
        scuDoRequestSense ();
        break;
      case 0x04:  //Format Unit
        scuDoFormatUnit ();
        break;
      case 0x06:  //Format Block(SASI)
        scuDoFormatBlockSASI ();
        break;
        //case 0x07:  //Bad Track Format(SASI)
      case 0x08:  //Read(6)
        scuDoRead6 ();
        break;
      case 0x0a:  //Write(6)
        scuDoWrite6 ();
        break;
      case 0x0b:  //Seek(6)
        scuDoSeek6 ();
        break;
        //case 0x0e:  //Assign Track(SASI)
      case 0x12:  //Inquiry
        scuDoInquiry ();
        break;
      case 0x15:  //Mode Select(6)
        scuDoModeSelect6 ();
        break;
        //case 0x18:  //Copy
      case 0x1a:  //Mode Sense(6)
        scuDoModeSense6 ();
        break;
      case 0x1b:  //Start-Stop Unit
        scuStartStopUnit ();
        break;
        //case 0x1c:  //Receive Diagnostic Results
        //case 0x1d:  //Send Diagnostic
      case 0x1e:  //Prevent-Allow Medium Removal
        scuDoPreventAllowMediumRemoval ();
        break;
      case 0x25:  //Read Capacity
        scuDoReadCapacity ();
        break;
      case 0x28:  //Read(10)
        scuDoRead10 ();
        break;
      case 0x2a:  //Write(10)
        scuDoWrite10 ();
        break;
      case 0x2b:  //Seek(10)
        scuDoSeek10 ();
        break;
      case 0x2e:  //Write and Verify(10)
        scuDoWriteAndVerify10 ();
        break;
      case 0x43:  //Read TOC
        scuDoReadTOC ();
        break;
        //case 0x2f:  //Verify(10)
        //case 0x30:  //Search Data High(10)
        //case 0x31:  //Search Data Equal(10)
        //case 0x32:  //Search Data Low(10)
        //case 0x33:  //Set Limits(10)
        //case 0x34:  //Pre-Fetch
      case 0x35:  //Synchronize Cache
        scuDoSynchronizeCache ();
        break;
        //case 0x36:  //Lock-Unlock Cache
        //case 0x37:  //Read Defect Data(10)
        //case 0x39:  //Compare
        //case 0x3a:  //Copy and Verify
        //case 0x3b:  //Write Buffer
        //case 0x3c:  //Read Buffer
        //case 0x3e:  //Read Long
        //case 0x3f:  //Write Long
        //case 0x40:  //Change Definition
        //case 0x41:  //Write Same
        //case 0x4c:  //Log Select
        //case 0x4d:  //Log Sense
        //case 0x55:  //Mode Select(10)
        //case 0x5a:  //Mode Sense(10)
        //case 0xc1:  //Load/Unload SHARP MO
      case 0xc2:  //Assign Drive(SASI)
        scuDoAssignDriveSASI ();
        break;
        //case 0x50:  //XDWRITE
        //case 0x51:  //XPWRITE
        //case 0x52:  //XDREAD
        //case 0x80:  //XDWRITE EXTENDED
        //case 0x81:  //REBUILD
        //case 0x82:  //REGENERATE
      default:  //Invalid
        if (SPC_REPORT_UNIMPLEMENTED_COMMAND &&  //未実装コマンドを表示するかつ
            (SPC_REPORT_UNIMPLEMENTED_XOR ||  //XOR系の未実装コマンドも表示するまたは
             !((0x50 <= oc && oc <= 0x52) ||  //XDWRITE,XPWRITE,XDREAD
               (0x80 <= oc && oc <= 0x82)))) {  //XDWRITE EXTENDED,REBUILD,REGENERATEではない
          scuPrintCommand (oc);
        }
        scuDoInvalid ();
      }
    }  //scuCommand(int)

    public void scuPrintCommand (int oc) {
      StringBuilder sb = new StringBuilder ();
      sb.append (String.format ("%08x sc%d:scuCommand(0x%02x(%s)) [", XEiJ.regPC0, abuNumber, oc, SPC_COMMAND_NAME[oc]));
      for (int i = 0; i < scuChip.spiBufferLimit; i++) {
        if (i > 0) {
          sb.append (',');
        }
        sb.append (String.format ("0x%02x", scuChip.spiCommandBuffer[i] & 255));
      }
      sb.append (']');
      System.out.println (sb.toString ());
    }  //scuPrintCommand

    public void scuPrintDataOut (int oc) {
      StringBuilder sb = new StringBuilder ();
      sb.append (String.format ("%08x sc%d:scuDataOut(0x%02x(%s)) [", XEiJ.regPC0, abuNumber, oc, SPC_COMMAND_NAME[oc]));
      for (int i = 0; i < scuChip.spiBufferLimit; i++) {
        if (i > 0) {
          sb.append (',');
        }
        sb.append (String.format ("0x%02x", scuChip.spiDataOutBuffer[i] & 255));
      }
      sb.append (']');
      System.out.println (sb.toString ());
    }  //scuPrintDataOut

    //scuDoTestUnitReady ()
    //  [0]  0x00
    //  [1]  |LUN###|-----|
    //  [5]  |..|----|Flag|Link|
    public void scuDoTestUnitReady () {
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoTestUnitReady()

    //scuDoRezeroUnit ()
    //  [0]  0x01
    //  [1]  |LUN###|-----|
    //  [5]  |..|----|Flag|Link|
    public void scuDoRezeroUnit () {
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoRezeroUnit()

    //scuDoPreventAllowMediumRemoval ()
    //  [0]  0x1e
    //  [1]  |LUN###|-----|
    //  [4]  |-------|Prevent|
    //  [5]  |..|----|Flag|Link|
    public void scuDoPreventAllowMediumRemoval () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int prevent = scuChip.spiCommandBuffer[4] & 1;  //0=イジェクト許可,1=イジェクト禁止
      if (prevent == 0) {  //イジェクト許可
        allow ();
      } else {  //イジェクト禁止
        prevent ();
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoPreventAllowMediumRemoval()

    //scuDoRequestSense ()
    //  [0]  0x03
    //  [1]  |LUN###|-----|
    //  [4]  アロケーション長
    //  [5]  |..|----|Flag|Link|
    //  センスデータを最大nバイトまで転送してからクリアする
    //  nが0x00のときSCSI-1は4バイトだけ転送するがSCSI-2は転送しない
    public void scuDoRequestSense () {
      int n = scuChip.spiCommandBuffer[4] & 255;
      if (n == 0) {
        if (scuMode == 1) {  //SCSI-1
          n = 4;
        } else {  //SCSI-2
          Arrays.fill (scuChip.spiSenseBuffer[scuChip.spiLUN], (byte) 0);  //センスデータを消去する
          scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
          return;
        }
      }
      scuChip.spiDataInPhase (scuChip.spiSenseBuffer[scuChip.spiLUN], 0, Math.min (n, scuMode == 1 ? 4 : 8), 0);  //データインフェーズに移行する
    }  //scuDoRequestSense()

    //scuDoFormatUnit ()
    //  [0]  0x04
    //  [1]  |LUN###|FmtData|CmpLst|ディフェクトリスト形式###|
    //  [2]  ベンダ固有
    //  [3][4]  インタリーブ
    //  [5]  |..|----|Flag|Link|
    public void scuDoFormatUnit () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      if (abuWriteProtected) {  //書き込みが禁止されている
        scuDataProtect ();
        return;
      }
      if (true) {
        //イメージファイルをゼロクリアする
        final int step = 1024 * 256;  //256KB
        byte[] zero = new byte[step];
        Arrays.fill (zero, (byte) 0);
        try {
          scuRaf.seek (0L);
          long pos;
          for (pos = 0L; pos + step <= scuDiskEndByte; pos += step) {
            if (SPC_CHECK_BLOCK_ZERO) {
              scuCheckBlockZero (zero);
            }
            scuRaf.write (zero);
          }
          if (pos < scuDiskEndByte) {
            if (SPC_CHECK_BLOCK_ZERO) {
              scuCheckBlockZero (zero, 0, (int) (scuDiskEndByte - pos));
            }
            scuRaf.write (zero, 0, (int) (scuDiskEndByte - pos));
          }
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoFormatUnit()

    //scuDoFormatBlockSASI ()
    //  [0]  0x06
    //  [1][2][3]  LUN<<21|論理ブロックアドレス
    //  [4]  インタリーブ
    public void scuDoFormatBlockSASI () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      if (abuWriteProtected) {  //書き込みが禁止されている
        scuDataProtect ();
        return;
      }
      if (false) {
        //10MBのとき
        //  06 00 00 00 06 00
        //  06 00 00 21 06 00
        //  ...
        //  06 00 9f 12 06 00
        //  06 00 9f 33 06 00
        //  33(レコード/トラック)*4(トラック/シリンダ)*309(シリンダ/ボリューム)=40788(レコード/ボリューム)=0x9f51
        System.out.printf ("%02x %02x %02x %02x %02x %02x\n",
                           scuChip.spiCommandBuffer[0] & 255,
                           scuChip.spiCommandBuffer[1] & 255,
                           scuChip.spiCommandBuffer[2] & 255,
                           scuChip.spiCommandBuffer[3] & 255,
                           scuChip.spiCommandBuffer[4] & 255,
                           scuChip.spiCommandBuffer[5] & 255);
      }
      //指定された範囲をゼロクリアする
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 0) & 0x001fffff;  //論理ブロックアドレス
      final int recordsPerTrack = 33;  //レコード/トラック
      if (a % recordsPerTrack == 0 && a + recordsPerTrack <= scuDiskEndRecord) {
        byte[] zero = new byte[scuBytesPerRecord * recordsPerTrack];
        Arrays.fill (zero, (byte) 0);
        try {
          scuRaf.seek ((long) scuBytesPerRecord * a);
          if (SPC_CHECK_BLOCK_ZERO) {
            scuCheckBlockZero (zero);
          }
          scuRaf.write (zero);
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
      } else {
        scuDoInvalid ();
        return;
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }

    //scuDoRead6 ()
    //  [0]  0x08
    //  [1][2][3]  LUN<<21|論理ブロックアドレス
    //  [4]  論理ブロック数
    //  [5]  |..|----|Flag|Link|
    public void scuDoRead6 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 0) & 0x001fffff;  //論理ブロックアドレス
      int n = scuChip.spiCommandBuffer[4] & 255;  //論理ブロック数
      if (n == 0) {
        n = 256;
      }
      if (scuDiskEndRecord < a + n) {  //範囲外
        scuDoInvalid ();
        return;
      }
      try {
        scuRaf.seek ((long) scuBytesPerRecord * a);
        scuRaf.read (scuRecordImage, 0, scuBytesPerRecord);
      } catch (IOException ioe) {
        ioe.printStackTrace ();
      }
      scuChip.spiDataInPhase (scuRecordImage, 0, scuBytesPerRecord, n - 1);  //データインフェーズに移行する
    }  //scuDoRead6()

    //scuReadImage ()
    public void scuReadImage () {
      try {
        scuRaf.read (scuRecordImage, 0, scuBytesPerRecord);
      } catch (IOException ioe) {
        ioe.printStackTrace ();
      }
    }  //scuReadImage()

    //scuDoWrite6 ()
    //  [0]  0x0a
    //  [1][2][3]  LUN<<21|論理ブロックアドレス
    //  [4]  論理ブロック数
    //  [5]  |..|----|Flag|Link|
    public void scuDoWrite6 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      if (abuWriteProtected) {  //書き込みが禁止されている
        scuDataProtect ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 0) & 0x001fffff;  //論理ブロックアドレス
      int n = scuChip.spiCommandBuffer[4] & 255;  //論理ブロック数
      if (n == 0) {
        n = 256;
      }
      if (scuDiskEndRecord < a + n) {  //範囲外
        scuDoInvalid ();
        return;
      }
      try {
        scuRaf.seek ((long) scuBytesPerRecord * a);
      } catch (IOException ioe) {
        ioe.printStackTrace ();
      }
      scuChip.spiDataOutPhase (scuRecordImage, 0, scuBytesPerRecord, n);  //データアウトフェーズに移行する
    }  //scuDoWrite6()

    //scuWriteImage ()
    public void scuWriteImage () {
      int oc = scuChip.spiCommandBuffer[0] & 255;
      if (oc == 0x0a ||  //Write(6)
          oc == 0x2a ||  //Write(10)
          oc == 0x2e) {  //Write and Verify(10)
        try {
          if (SPC_CHECK_BLOCK_ZERO) {
            scuCheckBlockZero (scuRecordImage, 0, scuBytesPerRecord);
          }
          scuRaf.write (scuRecordImage, 0, scuBytesPerRecord);
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
      }
    }  //scuWriteImage()

    //scuCheckBlockZero (buffer)
    //scuCheckBlockZero (buffer, offset, length)
    //  ブロック0に書き込もうとしたら内容を表示する
    public void scuCheckBlockZero (byte[] buffer) {
      if (SPC_CHECK_BLOCK_ZERO) {
        scuCheckBlockZero (buffer, 0, buffer.length);
      }
    }
    public void scuCheckBlockZero (byte[] buffer, int offset, int length) {
      if (SPC_CHECK_BLOCK_ZERO) {
        length = Math.min (512, length) & -16;
        try {
          if (scuRaf.getFilePointer () == 0L) {
            System.out.println ("writing to block zero");
            for (int index = 0; index < length; index++) {
              if ((index & 15) == 0) {
                System.out.printf ("%08x", index);
              }
              System.out.printf (" %02x", 0xff & buffer[offset + index]);
              if ((index & 15) == 15) {
                System.out.println ();
              }
            }
          }
        } catch (IOException ioe) {
          ioe.printStackTrace ();
        }
      }
    }  //scuCheckBlockZero

    //scuDoSeek6 ()
    //  [0]  0x0b
    //  [1][2][3]  LUN<<21|論理ブロックアドレス
    //  [5]  |..|----|Flag|Link|
    public void scuDoSeek6 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 0) & 0x001fffff;  //論理ブロックアドレス
      if (scuDiskEndRecord < a) {  //範囲外
        scuDoInvalid ();
        return;
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoSeek6()

    //scuDoInquiry ()
    //  [0]  0x12
    //  [1]  |LUN###|----|EVPD|
    //  [2]  ページコード
    //  [4]  アロケーション長
    //  [5]  |..|----|Flag|Link|
    //!!! EVPD==0のみ対応
    //  例
    //    0000  05  クォリファイア=0,デバイスタイプコード=5
    //    0001  80  RMB=1(リムーバブル),デバイスタイプ修飾子=0
    //    0002  02  ISOバージョン=0,ECMAバージョン=0,ANSIバージョン=2
    //    0003  02  AENC=0,TermlOP=0,レスポンスデータ形式=2
    //    0004  1f  追加データ長=31
    //    0005  00
    //    0006  00
    //    0007  18  RelAdr=0,WBus32=0,WBus16=0,Sync=1,Linked=1,CmdQue=0,SftRe=0
    //    0008  53 4f 4e 59 20 20 20 20  ベンダID="SONY    "
    //    0010  43 44 2d 52 4f 4d 20 43 44 55 2d 35 35 53 20 20  プロダクトID="CD-ROM CDU-55S  "
    //    0020  31 2e 30 74  プロダクト版数="1.0t"
    public void scuDoInquiry () {
      int evpd = scuChip.spiCommandBuffer[1] & 1;
      int pagecode = scuChip.spiCommandBuffer[2] & 255;  //ページコード
      int n = scuChip.spiCommandBuffer[4] & 255;  //アロケーション長
      if (evpd == 0) {  //スタンダードInquiry情報
        scuChip.spiDataInBuffer[0] = (byte) (abuCurrentMode == 0 ? SPC_DIRECT_ACCESS_DEVICE :  //ダイレクトアクセスデバイス。クォリファイアは0
                                             SPC_CDROM_DEVICE);  //CD-ROMデバイス
        scuChip.spiDataInBuffer[1] = (byte) (SPC_REMOVABLE_HDD ? 1 << 7 : 0);  //0=固定,1=リムーバブル
        scuChip.spiDataInBuffer[2] = (byte) (scuMode == 1 ? 1 : 2);  //ISO/ECMA/ANSIバージョン。SCSI-1/SCSI-2
        scuChip.spiDataInBuffer[3] = (byte) (scuMode == 1 ? 1 : 2);  //レスポンスデータ形式。SCSI-1/SCSI-2
        scuChip.spiDataInBuffer[4] = 31;  //追加データ長
        scuChip.spiDataInBuffer[5] = 0;  //予約
        scuChip.spiDataInBuffer[6] = 0;  //予約
        scuChip.spiDataInBuffer[7] = 0;  //サポート機能なし
        ByteArray.byaWstr (scuChip.spiDataInBuffer, 8, (
          abuCurrentMode == 0 ?  //ハードディスク
          //         111111
          //123456789012345
          "XEiJ    " +  //ベンダID(ASCII 8文字)
          "Hard Disk       " +  //プロダクトID(ASCII 16文字)。ASCIIに限られるがイメージファイル名を入れてもよい
          "1.0 "  //プロダクト版数(ASCII 4文字)
          :  //CD-ROM
          "XEiJ    " +  //ベンダID(ASCII 8文字)
          "CD-ROM          " +  //プロダクトID(ASCII 16文字)。ASCIIに限られるがイメージファイル名を入れてもよい
          "1.0 "));  //プロダクト版数(ASCII 4文字)
        scuChip.spiDataInPhase (scuChip.spiDataInBuffer, 0, Math.min (36, n), 0);  //データインフェーズに移行する
      } else {  //VPD情報
        //!!!
        if (SPC_REPORT_UNIMPLEMENTED_COMMAND) {
          System.out.println (String.format ("%08x Inquiry with EVPD==1\n", XEiJ.regPC0));
        }
        scuDoInvalid ();
      }
    }  //scuDoInquiry()

    //scuDoModeSelect6 ()
    //  [0]  0x15
    //  [1]  |LUN###|PF|---|SP|
    //  [4]  パラメータリスト長
    //  [5]  コントロールバイト
    public void scuDoModeSelect6 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int n = scuChip.spiCommandBuffer[4] & 255;  //パラメータリスト長
      scuChip.spiDataOutPhase (scuChip.spiDataOutBuffer, 0, n, 0);  //データアウトフェーズに移行する
    }  //scuDoModeSelect6()

    //scuDoModeSense6 ()
    //  [0]  0x1a
    //  [1]  |LUN###|R|DBD|---|
    //  [2]  |PC##|ページコード######|
    //  [4]  アロケーション長
    //  [5]  |..|----|Flag|Link|
    public void scuDoModeSense6 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int page = scuChip.spiCommandBuffer[2] & 63;  //ページコード
      if (!(page == 0x00 || page == 0x3f)) {
        scuDoInvalid ();
        return;
      }
      int n = scuChip.spiCommandBuffer[4] & 255;  //アロケーション長
      scuChip.spiDataInBuffer[0] = 12 - 1;  //センスデータ長
      scuChip.spiDataInBuffer[1] = 0;  //メディアタイプ
      scuChip.spiDataInBuffer[2] = (byte) (abuWriteProtected ? 1 << 7 : 0 << 7);  //ライトプロテクト
      scuChip.spiDataInBuffer[3] = 8;  //ブロックディスクリプタ長
      //ブロックディスクリプタ1
      ByteArray.byaWl (scuChip.spiDataInBuffer, 4, 0 << 24 | scuDiskEndRecord);  //デンシティコード,ブロック数
      ByteArray.byaWl (scuChip.spiDataInBuffer, 8, scuBytesPerRecord);  //ブロック長
      scuChip.spiDataInPhase (scuChip.spiDataInBuffer, 0, Math.min (12, n), 0);  //データインフェーズに移行する
    }  //scuDoModeSense6()

    //scuStartStopUnit ()
    //  [0]  0x1b
    //  [1]  |LUN###|-----|
    //  [4]  |------|LoEj|Start|
    //  [5]  |..|----|Flag|Link|
    public void scuStartStopUnit () {
      int loejStart = scuChip.spiCommandBuffer[4] & 3;  //LoEj|Start
      if (loejStart == 2) {  //イジェクト
        eject ();
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuStartStopUnit()

    //scuDoReadCapacity ()
    //  [0]  0x25
    //  [1]  |LUN###|----|RelAdr|
    //  [2][3][4][5]  論理ブロックアドレス
    //  [8]  |-------|PMI|
    //  [9]  |..|----|Flag|Link|
    //
    //  本来の仕様
    //    Read Capacityが返す値は論理ブロック数ではなくて最終論理ブロックアドレスである
    //    論理ブロックアドレスは0から始まるので、論理ブロック数は最終論理ブロックアドレス+1に等しい
    //  FORMAT.XのSCSIハードディスクに関する動作
    //    装置初期化
    //      先頭セクタの+10にRead Capacityが返した値を書き込む
    //    領域確保
    //      先頭セクタの+10に書かれている値またはRead Capacityが返した値を論理ブロック数と見なして容量を計算する
    //    すなわち、FORMAT.XはRead Capacityが返す最終論理ブロックアドレスを論理ブロック数と誤解している
    //  結論
    //    他の機種では先頭セクタに最終論理ブロックアドレスが書かれていることが多いが、
    //    X68000のSCSIハードディスクの先頭セクタの+10に書かれている値は最終論理ブロックアドレスではなくて論理ブロック数である
    //  辻褄合わせ
    //    Read Capacityが返す値は、実行中のプロセスがFORMAT.Xならば論理ブロック数、それ以外は最終論理ブロックアドレスとする
    public void scuDoReadCapacity () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int pmi = scuChip.spiCommandBuffer[8] & 1;  //0=全体の末尾,1=指定されたアドレスから連続してアクセスできる領域の末尾
      int pmm = MainMemory.mmrHumanPmm ();  //Human68kの実行中のプロセスのメモリ管理テーブルのアドレス
      boolean formatx = (pmi == 0 &&  //_S_READCAPは0
                         0 <= pmm &&  //Human68kのバージョンと実行中のプロセスのメモリ管理テーブルのアドレスの確認
                         ((MC68060.mmuPeekByteSign (pmm + 0x64, 5) == 2 &&  //FORMAT2.Xのモジュール番号
                           MC68060.mmuPeekLong (pmm + 0x30, 5) - (pmm + 256) == 68088 &&  //FORMAT2.X version 2.31のtext+dataのサイズ
                           "FORMAT.X".equalsIgnoreCase (MC68060.mmuPeekStringZ (pmm + 0xc4, 5))) ||  //実行ファイル名
                          (MC68060.mmuPeekLong (pmm + 0x30, 5) - (pmm + 256) == 0x6076 + 0x0004 &&  //SCSIFORMAT.Xのtext+dataのサイズ
                           "SCSIFORMAT.X".equalsIgnoreCase (MC68060.mmuPeekStringZ (pmm + 0xc4, 5)))));  //実行ファイル名
      ByteArray.byaWl (scuChip.spiDataInBuffer, 0, formatx ? scuDiskEndRecord : scuDiskEndRecord - 1);  //実行中のプロセスがFORMAT.XまたはSCSIFORMAT.Xならば論理ブロック数、それ以外は最終論理ブロックアドレスを返す
      ByteArray.byaWl (scuChip.spiDataInBuffer, 4, scuBytesPerRecord);  //ブロック長
      scuChip.spiDataInPhase (scuChip.spiDataInBuffer, 0, 8, 0);  //データインフェーズに移行する
    }  //scuDoReadCapacity

    //scuDoRead10 ()
    //  [0]  0x28
    //  [1]  |LUN###|DPO|FUA|--|RelAdr|
    //  [2][3][4][5]  論理ブロックアドレス
    //  [7][8]  論理ブロック数
    //  [9]  |..|----|Flag|Link|
    public void scuDoRead10 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 2);  //論理ブロックアドレス
      int n = ByteArray.byaRwz (scuChip.spiCommandBuffer, 7);  //論理ブロック数
      if (scuDiskEndRecord < a + n) {  //範囲外
        scuDoInvalid ();
        return;
      }
      if (n == 0) {
        scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
        return;
      }
      try {
        scuRaf.seek ((long) scuBytesPerRecord * a);
        scuRaf.read (scuRecordImage, 0, scuBytesPerRecord);
      } catch (IOException ioe) {
        ioe.printStackTrace ();
      }
      scuChip.spiDataInPhase (scuRecordImage, 0, scuBytesPerRecord, n - 1);  //データインフェーズに移行する
    }  //scuDoRead10()

    //scuDoWrite10 ()
    //  [0]  0x2a
    //  [1]  |LUN###|DPO|FUA|--|RelAdr|
    //  [2][3][4][5]  論理ブロックアドレス
    //  [7][8]  論理ブロック数
    //  [9]  |..|----|Flag|Link|
    public void scuDoWrite10 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      if (abuWriteProtected) {  //書き込みが禁止されている
        scuDataProtect ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 2);  //論理ブロックアドレス
      int n = ByteArray.byaRwz (scuChip.spiCommandBuffer, 7);  //論理ブロック数
      if (scuDiskEndRecord < a + n) {  //範囲外
        scuDoInvalid ();
        return;
      }
      if (n == 0) {
        scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
        return;
      }
      try {
        scuRaf.seek ((long) scuBytesPerRecord * a);
      } catch (IOException ioe) {
        ioe.printStackTrace ();
      }
      scuChip.spiDataOutPhase (scuRecordImage, 0, scuBytesPerRecord, n);  //データアウトフェーズに移行する
    }  //scuDoWrite10()

    //scuDoSeek10 ()
    //  [0]  0x2b
    //  [1]  |LUN###|DPO|FUA|--|RelAdr|
    //  [2][3][4][5]  論理ブロックアドレス
    //  [9]  |..|----|Flag|Link|
    public void scuDoSeek10 () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 2);  //論理ブロックアドレス
      if (scuDiskEndRecord < a) {  //範囲外
        scuDoInvalid ();
        return;
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoSeek10()

    //scuDoWriteAndVerify10 ()
    //  [0]  0x2e
    //  [1]  |LUN###|DPO|Reserved##|BytChk|RelAdr|
    //  [2][3][4][5]  論理ブロックアドレス
    //  [7][8]  論理ブロック数
    //  [9]  |..|----|Flag|Link|
    public void scuDoWriteAndVerify10 () {
      scuDoWrite10 ();
    }  //scuDoWriteAndVerify10

    //scuDoSynchronizeCache ()
    //  [0]  0x35
    //  [1]  |LUN###|---|Immed|RelAdr|
    //  [2][3][4][5]  論理ブロックアドレス
    //  [7][8]  論理ブロック数
    //  [9]  Control
    public void scuDoSynchronizeCache () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int a = ByteArray.byaRls (scuChip.spiCommandBuffer, 2);  //論理ブロックアドレス
      int n = ByteArray.byaRwz (scuChip.spiCommandBuffer, 7);  //論理ブロック数
      if (scuDiskEndRecord < a + n) {  //範囲外
        scuDoInvalid ();
        return;
      }
      //直前のコマンドのデータアウトフェーズが終了した時点でwriteされている
      //アクセスモードが"rw"なので手動で同期をとる
      try {
        scuRaf.getChannel ().force (true);
      } catch (IOException ioe) {
        ioe.printStackTrace ();
      }
      scuChip.spiStatusPhase (SPC_GOOD, SPC_COMMAND_COMPLETE);  //エラーなしでステータスフェーズに移行する
    }  //scuDoSynchronizeCache

    //scuDoReadTOC ()
    //  [0]  0x43
    //  [1]  |LUN###|---|MSF|-|
    //  [6]  開始トラック
    //  [7][8]  アロケーション長
    //  [9]  コントロールバイト
    //  例:
    //    MSF=0
    //      01: 00 3a 01 06 00 14 01 00 00 00 00 00
    //      02: 00 32 01 06 00 10 02 00 00 02 76 b0
    //      03: 00 2a 01 06 00 10 03 00 00 02 bf 7e
    //      04: 00 22 01 06 00 10 04 00 00 03 0e 3b
    //      05: 00 1a 01 06 00 10 05 00 00 03 96 3c
    //      06: 00 12 01 06 00 10 06 00 00 03 d2 e0
    //      aa: 00 0a 01 06 00 10 aa 00 00 04 1b d2
    //    MSF=1
    //      01: 00 3a 01 06 00 14 01 00 00 00 02 00  (0x00*60+(0x02-2))*75+0x00=0x000000
    //      02: 00 32 01 06 00 10 02 00 00 23 36 38  (0x23*60+(0x36-2))*75+0x38=0x0276b0
    //      03: 00 2a 01 06 00 10 03 00 00 28 03 13  (0x28*60+(0x03-2))*75+0x13=0x02bf7e
    //      04: 00 22 01 06 00 10 04 00 00 2c 20 01  (0x2c*60+(0x20-2))*75+0x01=0x030e3b
    //      05: 00 1a 01 06 00 10 05 00 00 34 10 12  (0x34*60+(0x10-2))*75+0x12=0x03963c
    //      06: 00 12 01 06 00 10 06 00 00 37 2b 11  (0x37*60+(0x2b-2))*75+0x11=0x03d2e0
    //      aa: 00 0a 01 06 00 10 aa 00 00 3b 34 10  (0x3b*60+(0x34-2))*75+0x10=0x041bd2
    public void scuDoReadTOC () {
      if (abuCurrentMode == 0) {  //ハードディスク
        scuDoInvalid ();
        return;
      }
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      boolean msf = (scuChip.spiCommandBuffer[1] & 2) != 0;  //true=MSFアドレス形式,false=論理ブロックアドレス形式
      int startTrack = scuChip.spiCommandBuffer[6] & 255;  //開始トラック
      int allocLength = ByteArray.byaRwz (scuChip.spiCommandBuffer, 7);  //アロケーション長
      if (startTrack == 0) {
        startTrack = 1;
      }
      int dataLength;
      if (startTrack <= 1) {  //データトラックとリードアウトトラック
        dataLength = 4 + 8 * 2;  //データの長さ。自分を含む
        ByteArray.byaWw (scuChip.spiDataInBuffer, 0, 2 + 8 * 2);  //TOCデータ長。要求されたサイズに関わらず最後までの長さを返す。自分を含まない
        scuChip.spiDataInBuffer[2] = 1;  //先頭トラック番号
        scuChip.spiDataInBuffer[3] = 1;  //最終トラック番号
        scuChip.spiDataInBuffer[4] = 0;  //Reserved
        scuChip.spiDataInBuffer[5] = 0x14;  //カレントポジションデータ,データトラック
        scuChip.spiDataInBuffer[6] = 1;  //トラック番号
        scuChip.spiDataInBuffer[7] = 0;  //Reserved
        ByteArray.byaWl (scuChip.spiDataInBuffer, 8, 0);  //アブソリュートCD-ROMアドレス
        scuChip.spiDataInBuffer[12] = 0;  //Reserved
        scuChip.spiDataInBuffer[13] = 0x10;  //カレントポジションデータ
        scuChip.spiDataInBuffer[14] = (byte) 0xaa;  //トラック番号
        scuChip.spiDataInBuffer[15] = 0;  //Reserved
        ByteArray.byaWl (scuChip.spiDataInBuffer, 16, msf ? scuDiskEndRecord / 4500 << 16 | scuDiskEndRecord / 75 % 60 + 2 << 8 | scuDiskEndRecord % 75 : scuDiskEndRecord);  //アブソリュートCD-ROMアドレス
      } else if (startTrack <= 0xaa) {  //リードアウトトラック
        dataLength = 4 + 8 * 1;  //データの長さ。自分を含む
        ByteArray.byaWw (scuChip.spiDataInBuffer, 0, 2 + 8 * 1);  //TOCデータ長。要求されたサイズに関わらず最後までの長さを返す。自分を含まない
        scuChip.spiDataInBuffer[2] = 1;  //先頭トラック番号
        scuChip.spiDataInBuffer[3] = 1;  //最終トラック番号
        scuChip.spiDataInBuffer[4] = 0;  //Reserved
        scuChip.spiDataInBuffer[5] = 0x10;  //カレントポジションデータ
        scuChip.spiDataInBuffer[6] = (byte) 0xaa;  //トラック番号
        scuChip.spiDataInBuffer[7] = 0;  //Reserved
        ByteArray.byaWl (scuChip.spiDataInBuffer, 8, msf ? scuDiskEndRecord / 4500 << 16 | scuDiskEndRecord / 75 % 60 + 2 << 8 | scuDiskEndRecord % 75 : scuDiskEndRecord);  //アブソリュートCD-ROMアドレス
      } else {  //開始トラック番号が範囲外
        scuDoInvalid ();
        return;
      }
      scuChip.spiDataInPhase (scuChip.spiDataInBuffer, 0, Math.min (dataLength, allocLength), 0);  //データインフェーズに移行する
    }  //scuDoReadTOC()

    //scuDoAssignDriveSASI ()
    //  [0]  0xc2
    //  [5]  ドライブパラメータの長さ
    public void scuDoAssignDriveSASI () {
      if (!abuInserted || scuDiskEndRecord == 0) {  //挿入されていない
        scuUnitAttention ();
        return;
      }
      int n = scuChip.spiCommandBuffer[5] & 255;  //ドライブパラメータの長さ
      scuChip.spiDataOutPhase (scuChip.spiDataOutBuffer, 0, n, 0);  //データアウトフェーズに移行する
    }  //scuDoAssignDriveSASI

    //scuDoInvalid ()
    public void scuDoInvalid () {
      if (scuMode == 1) {  //SCSI-1
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_INVALID_COMMAND;
      } else {  //SCSI-2
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_EXTENDED_SENSE;
        scuChip.spiSenseBuffer[scuChip.spiLUN][2] = (byte) SPC_ILLEGAL_REQUEST;
      }
      scuChip.spiStatusPhase (SPC_CHECK_CONDITION, SPC_COMMAND_COMPLETE);
    }  //scuDoInvalid()

    //scuNotReady ()
    public void scuNotReady () {
      if (scuMode == 1) {  //SCSI-1
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_INVALID_COMMAND;
      } else {  //SCSI-2
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_EXTENDED_SENSE;
        scuChip.spiSenseBuffer[scuChip.spiLUN][2] = (byte) SPC_NOT_READY;
      }
      scuChip.spiStatusPhase (SPC_CHECK_CONDITION, SPC_COMMAND_COMPLETE);
    }  //scuNotReady

    //scuUnitAttention ()
    public void scuUnitAttention () {
      if (scuMode == 1) {  //SCSI-1
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_INVALID_COMMAND;
      } else {  //SCSI-2
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_EXTENDED_SENSE;
        scuChip.spiSenseBuffer[scuChip.spiLUN][2] = (byte) SPC_UNIT_ATTENTION;
      }
      scuChip.spiStatusPhase (SPC_CHECK_CONDITION, SPC_COMMAND_COMPLETE);
    }  //scuUnitAttention()

    //scuDataProtect ()
    public void scuDataProtect () {
      if (scuMode == 1) {  //SCSI-1
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_INVALID_COMMAND;
      } else {  //SCSI-2
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_EXTENDED_SENSE;
        scuChip.spiSenseBuffer[scuChip.spiLUN][2] = (byte) SPC_DATA_PROTECT;
      }
      scuChip.spiStatusPhase (SPC_CHECK_CONDITION, SPC_COMMAND_COMPLETE);
    }  //scuDataProtect()

    //scuMediumError ()
    public void scuMediumError () {
      if (scuMode == 1) {  //SCSI-1
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_INVALID_COMMAND;
      } else {  //SCSI-2
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_EXTENDED_SENSE;
        scuChip.spiSenseBuffer[scuChip.spiLUN][2] = (byte) SPC_MEDIUM_ERROR;
      }
      scuChip.spiStatusPhase (SPC_CHECK_CONDITION, SPC_COMMAND_COMPLETE);
    }  //scuMediumError

    //scuIllegalRequest ()
    public void scuIllegalRequest () {
      if (scuMode == 1) {  //SCSI-1
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_INVALID_COMMAND;
      } else {  //SCSI-2
        scuChip.spiSenseBuffer[scuChip.spiLUN][0] = (byte) SPC_EXTENDED_SENSE;
        scuChip.spiSenseBuffer[scuChip.spiLUN][2] = (byte) SPC_ILLEGAL_REQUEST;
      }
      scuChip.spiStatusPhase (SPC_CHECK_CONDITION, SPC_COMMAND_COMPLETE);
    }  //scuMediumError

  }  //class SCUnit



}  //class SPC



