//========================================================================================
//  WaitInstruction.java
//    en:Wait instruction
//    ja:待機命令
//  Copyright (C) 2003-2024 Makoto Kamada
//
//  This file is part of the XEiJ (X68000 Emulator in Java).
//  You can use, modify and redistribute the XEiJ if the conditions are met.
//  Read the XEiJ License for more details.
//  https://stdkmd.net/xeij/
//========================================================================================

package xeij;

//待機命令
//  待機ポイントを踏んだときに本来の命令の代わりに実行されるエミュレータ拡張命令
//  終了条件が成立するまで同じアドレスに留まる
//    プログラムカウンタを巻き戻す
//    命令の実行時間を加える
//    トレース例外を処理する
//    デバイスを呼び出す
//    割り込みを受け付ける
//  終了条件が成立したら待機ポイントを撤去する
//  終了条件
//    データの準備ができたとき
//    決められた実時間が経過したとき
//    など
//  終了条件のテストに時間がかかる場合はテストの頻度を下げること
//    デバイス呼び出しと割り込み受け付けの頻度を確保するため待機命令の実行時間は短く設定されている
//    テストの所要時間が待機命令の実行時間より長いと負荷率が100%を超えてしまう

//class WaitInstruction
public class WaitInstruction {

  public static WaitInstruction instruction;  //現在の待機命令

  //execute ()
  //  待機命令を実行する
  public static void execute () {
    XEiJ.regPC = XEiJ.regPC0;  //PCを巻き戻す
    XEiJ.mpuClockTime += XEiJ.TMR_FREQ * 1 / 1000000;  //1us。10MHzのとき10サイクル
    if (instruction.terminate ()) {  //終了する
      InstructionBreakPoint.ibpRemoveWaitPoint (XEiJ.regPC, XEiJ.regSRS);  //待機ポイントを撤去する
    }
  }

  //terminate ()
  //  終了条件
  public boolean terminate () {
    return true;
  }  //terminate

  //REWIND_AR
  //  (Ar)+|-(Ar)のリードで命令を中止したとき変化したArを巻き戻すためのテーブル
  //    R[8+(OC&7)]+=REWIND_AR[OC>>3]
  //  以下の命令には対応していない
  //    FMOVEM  FRESTORE  FSAVE  MOVE16  MOVEM  PFLUSHR  PRESTORE  PSAVE
  public static final byte[] REWIND_AR = "\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\1\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\377\0\377\1\0\0\0\0\376\0\376\2\0\0\0\0\374\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\376\2\0\0\0\0\0\0\374\4\0\0\0\0\0\0\376\2\0\0\0\0\1\0\377\1\0\0\0\0\2\0\376\2\0\0\0\0\4\0\374\4\0\0\0\0\0\0\374\4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\376\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\374\4\0\0\0\0\0\0\377\1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0".getBytes (XEiJ.ISO_8859_1);

}  //class WaitInstruction
