//========================================================================================
//  FDMedia.java
//    en:Floppy disk media
//    ja:フロッピーディスクメディア
//  Copyright (C) 2003-2023 Makoto Kamada
//
//  This file is part of the XEiJ (X68000 Emulator in Java).
//  You can use, modify and redistribute the XEiJ if the conditions are met.
//  Read the XEiJ License for more details.
//  https://stdkmd.net/xeij/
//========================================================================================

//----------------------------------------------------------------------------------------
//フロッピーディスクのメディアの種類
//
//  シリンダ
//    ここではシリンダとトラックと明確に区別する(トラックの項目を参照)
//    シリンダ数は一般的なメディアでは77または80
//    シリンダ番号は最も外側が0、最も内側が(シリンダ数-1)
//      アドレスヘッダのC,H,S,NのC
//
//  サイド
//    サイド数は片面ディスクが1、両面ディスクが2
//    サイド番号は表面が0、裏面が1。2HDEは最初のセクタを除いて表面が128、裏面が129
//      ヘッドアドレスとも呼ばれる
//      アドレスヘッダのC,H,S,NのH
//
//  トラック
//    トラック数は(サイド数*シリンダ数)
//    トラック番号は(サイド数*シリンダ番号+サイド番号)
//    トラック番号はアクセスする順序で数えたトラックの通し番号であり、シリンダ番号とは異なる
//      フロッピーディスクが片面しかなかった時代の名残でシリンダとトラックを明確に区別していない資料もあるが、
//      両面ディスクは1つのシリンダに2つのトラックがあり、トラックを特定できない番号(シリンダ番号)をトラック番号と呼ぶべきではない
//      シリンダとトラックを区別していてもトラック0はシリンダ0にあるのでTrack0信号やトラック0へのシークはそのままになっていたりする
//
//  セクタ
//    セクタ番号はトラックの先頭が1、末尾がEOT。2HSは最初のセクタを除いてトラックの先頭が10、末尾が18
//      レコード番号とも呼ばれる
//      アドレスヘッダのC,H,S,NのS
//    セクタサイズは1セクタあたりのバイト数
//    セクタスケールは(log2(セクタサイズ)-7)
//      アドレスヘッダのC,H,S,NのN
//    セクタサイズと1トラックあたりのセクタ数(EOT)はメディアの種類によって大きく異なる
//! ここで言うセクタはOSレベル(BPB)の論理セクタではなくてIOCSレベルの物理セクタである
//! Human68kではフロッピーディスクの論理セクタと物理セクタのサイズが同じなので区別が曖昧になっている
//
//  セクタの順序
//    シリンダ0サイド0(トラック0)セクタ1、…、シリンダ0サイド0(トラック0)セクタEOT、
//    シリンダ0サイド1(トラック1)セクタ1、…、シリンダ0サイド1(トラック1)セクタEOT、
//    シリンダ1サイド0(トラック2)セクタ1、…、シリンダ1サイド0(トラック2)セクタEOT、
//    シリンダ1サイド1(トラック3)セクタ1、…、シリンダ1サイド1(トラック3)セクタEOT、…
//
//  トラックフォーマット
//    Index Pulseは12cmディスクの場合は円盤に空いている穴で検出される。そこから1回転で1トラック
//        ┏━━━━━━┓                                                                                ┏━━━━━━┓
//        ┃Index Pulse ┃                                                                                ┃            ┃
//    ━━┛            ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛            ┗━
//        │        │        │Index Mark │        │            │            │            │         │
//        │ Gap4a  │  Sync  │    IAM    │  Gap1  │  Sector1   │     …     │ SectorEOT  │  Gap4b  │
//    FM  │$4E x 40│$00 x 6 │    $FC    │$4E x 26│            │            │            │         │
//    MFM │$4E x 80│$00 x 12│$C2 x 3,$FC│$4E x 50│            │            │            │         │
//
//  セクタフォーマット
//    SectorEOTにはGap3がなく、Dataの直後にGap4bが続く
//        │                   ID Field                    │        │                    Data Field                     │         │
//        │        │AddressMark│     Address Header     │        │        │DataAddressMark│         Data           │         │
//        │  Sync  │   IDAM    │ C , H , S , N │  CRC  │  Gap2  │  Sync  │   DAM/DDAM    │               │  CRC  │  Gap3   │
//    FM  │$00 x 6 │    $FE    │$xx,$xx,$xx,$xx│$xx x 2│$4E x 11│$00 x 6 │    $FB/$F8    │$xx x (128<<N) │$xx x 2│$4E x GPL│
//    MFM │$00 x 12│$A1 x 3,$FE│$xx,$xx,$xx,$xx│$xx x 2│$4E x 22│$00 x 12│$A1 x 3,$FB/$F8│$xx x (128<<N) │$xx x 2│$4E x GPL│
//
//  IBMフォーマット
//               N  EOT  GPL  GSL
//    FM    128  0   26   27    7
//          256  1   15   42   14
//          512  2    8   58   27
//    MFM   256  1   26   54   14
//          512  2   15   84   27  2HC
//         1024  3    8  116   53  2HD
//
//  ECMAフォーマット
//               N  EOT  GPL  GSL
//    FM    128  0   16   24    7
//          256  1    9   38   14
//    MFM   256  1   16   50   14
//----------------------------------------------------------------------------------------

package xeij;

import java.io.*;  //BufferedInputStream,BufferedOutputStream,BufferedReader,BufferedWriter,File,FileInputStream,FileNotFoundException,FileReader,InputStream,InputStreamReader,IOException,OutputStreamWriter,RandomAccessFile
import java.lang.*;  //Boolean,Character,Class,Comparable,Double,Exception,Float,IllegalArgumentException,Integer,Long,Math,Number,Object,Runnable,SecurityException,String,StringBuilder,System
import java.util.*;  //ArrayList,Arrays,Calendar,GregorianCalendar,HashMap,Map,Map.Entry,Timer,TimerTask,TreeMap
import java.util.zip.*;  //CRC32,Deflater,GZIPInputStream,GZIPOutputStream,ZipEntry,ZipInputStream

public final class FDMedia extends HumanMedia {

  //  メディアの種類
  //    最初の5種類(2HD/2HC/2DD8/2DD9/2HQ)はFDDEVICE.XとFORMAT.Xに合わせる
  //    資料として参考にしたもの(動作を確認したものではない)
  //      FDDEVICE.X 1.00 (SHARP/Hudson)
  //      FORMAT.X:FORMAT1.X 2.20 (SHARP/Hudson)
  //      FORMAT.X:FORMAT2.X 2.31 (SHARP/Hudson)
  //      9SCDRV.X 3.14+2 (6no8rouさん)
  //      9SCFMT.X 3.14+2 (6no8rouさん)
  //      DIFC.X 1.19+8 (ＧＵＮさん)
  //      DIMG.X 1.10 (H.ATAさん)
  //      dpack.x 1.32.6 (feelさん、m.matsuoさん、SiNさん)
  //      vfic.exe 010621 (REDさん)
  //      READFD.X 1.1 (yamamaさん)
  //    メモ
  //      XDFファイルを作るREADFD.Xは2HDのみの対応なのでXDFは2HDのみと考えられるが、
  //      ここではHuman68kが標準で(FDDEVICE.Xで)対応していてファイルサイズで区別することができる2HD/2HC/2DD8/2DD9/2HQの5種類について、
  //      拡張子がXDFでよいことにする
  //    mediaByte
  //      メディアバイト
  //      FDDEVICE.Xの中のBPBテーブルに書かれている値
  //      メディア判別を行う度にBPBテーブルごとHuman68kに渡され、DPBが作られて_GETDPBで返される
  //    fatID
  //      FDDEVICE.Xの中のメディア判別コードとFORMAT.Xの中のBPBテーブルに書かれている値
  //      IPLの中のBPBテーブルとFAT領域の先頭に書き込まれる
  public static final FDMedia[] FDM_ARRAY = {
    //               -------------論理仕様(BPB)-------------  -------物理仕様(IOCS)------
    //                          rootEntries          sectorsPerTrack tracksPerCylinder
    //                reservedSectors| mediaByte       sectorScale|  |cylindersPerDisk
    //                   fatCount|   |     |  partitionSectors|   |  |   |doubleDensity
    //       sectorsPerCluster|  |   |     |    fatSectors|   |   |  |   |    |
    //       bytesPerSector|  |  |   |     |   fatID |    |   |   |  |   |    |
    //          番号   |   |  |  |   |     |     |   |    |   |   |  |   |    |         名前               拡張子               IOCS
    new FDMedia ( 0, 1024, 1, 2, 1, 192, 0xfe, 0xfe, 2, 1232, 3,  8, 2, 77, false, "2HD (1232KB)", "2HD,HDM,XDF"         ),  //$9xxx
    new FDMedia ( 1,  512, 1, 2, 1, 224, 0xfd, 0xf9, 7, 2400, 2, 15, 2, 80, false, "2HC (1200KB)", "2HC,FS,HD5,XDF"      ),  //$9xxx
    new FDMedia ( 2,  512, 2, 2, 1, 112, 0xfb, 0xfb, 2, 1280, 2,  8, 2, 80, true , "2DD (640KB)",  "2DD,2DD8,DD8,DD6,XDF"),  //$1xxx
    new FDMedia ( 3,  512, 2, 2, 1, 112, 0xfc, 0xf9, 3, 1440, 2,  9, 2, 80, true , "2DD (720KB)",  "2DD,2DD9,DD9,XDF"    ),  //$3xxx
    new FDMedia ( 4,  512, 1, 2, 1, 224, 0xfa, 0xf0, 9, 2880, 2, 18, 2, 80, false, "2HQ (1440KB)", "2HQ,HD4,XDF"         ),  //$7xxx
    new FDMedia ( 5,  512, 2, 2, 1, 112, 0xe0, 0xe0, 3, 1600, 2, 10, 2, 80, true,  "2DD (800KB)",  "2DD,2DD10,DDA"       ),
    new FDMedia ( 6, 1024, 1, 2, 1, 192, 0xec, 0xf8, 3, 1440, 3,  9, 2, 80, false, "2HDE(1440KB)", "2HDE,2HE"            ),
    new FDMedia ( 7, 1024, 1, 1, 1, 192, 0xeb, 0xfb, 3, 1440, 3,  9, 2, 80, false, "2HS (1440KB)", "2HS"                 ),
  };
  public static final FDMedia FDM_2HD   = FDM_ARRAY[0];
  public static final FDMedia FDM_2HC   = FDM_ARRAY[1];
  public static final FDMedia FDM_2DD8  = FDM_ARRAY[2];
  public static final FDMedia FDM_2DD9  = FDM_ARRAY[3];
  public static final FDMedia FDM_2HQ   = FDM_ARRAY[4];
  public static final FDMedia FDM_2DD10 = FDM_ARRAY[5];
  public static final FDMedia FDM_2HDE  = FDM_ARRAY[6];
  public static final FDMedia FDM_2HS   = FDM_ARRAY[7];
  public static final int FDM_BUFFER_SIZE = 256 + 1024 * 1600 + 16;  //FDM_ARRAYにあるfdmBytesPerDiskの最大値+256よりも少し大きい値

  public static final byte[] fdmCache = new byte[FDM_BUFFER_SIZE];

  //media = fdmPathToMedia (path, bb)
  //  パスに合うメディアを返す。null=見つからない
  //  バッファを指定したときはそこにコピーする
  public static FDMedia fdmPathToMedia (String path, byte[] bb) {
    byte[] array = XEiJ.rscGetFile (path);
    if (array == null) {  //読み込めない
      return null;
    }
    FDMedia outputMedia = null;
    byte[] outputArray = null;
    int dotIndex = path.lastIndexOf ('.');
    String upperExt = dotIndex < 0 ? "" : path.substring (dotIndex + 1).toUpperCase ();
    if (upperExt.equals ("DIM")) {  //*.DIM
      if (array.length < 256) {  //短すぎる
        System.out.println (Multilingual.mlnJapanese ?
                            path + " は短すぎます" :
                            path + " is too short");
        return null;
      }
      if (!new String (array, 171, 13, XEiJ.ISO_8859_1).equals ("DIFC HEADER  ")) {  //マジックが一致しない
        System.out.println (Multilingual.mlnJapanese ?
                            path + " に DIFC HEADER がありません" :
                            path + " does not have DIFC HEADER");
        return null;
      }
      //形式を確認する
      switch (array[0]) {
      case 0x00:
        outputMedia = FDM_2HD;
        break;
      case 0x01:
        outputMedia = FDM_2HS;
        break;
      case 0x02:
        outputMedia = FDM_2HC;
        break;
      case 0x03:
        outputMedia = FDM_2HDE;
        break;
      case 0x09:
        outputMedia = FDM_2HQ;
        break;
      default:  //未対応の形式
        System.out.println (Multilingual.mlnJapanese ?
                            path + " のフォーマットが不明です" :
                            "Format of " + path + " is unclear");
        return null;
      }
      //その他のヘッダ情報を確認する
      int tracksPerDisk = outputMedia.fdmTracksPerDisk;  //本来のトラック数
      if (array[0x00FF] != 0x00 &&
          array[0x00FF] != tracksPerDisk) {  //オーバートラック
        System.out.println (Multilingual.mlnJapanese ?
                            path + " のトラック数が違います" :
                            path + " has wrong number of tracks");
        return null;
      }
      for (int i = 0; i < tracksPerDisk; i++) {
        if (array[0x0001 + i] != 0x00 &&
            array[0x0001 + i] != 0x01) {  //存在するトラックの保存状態が0x00または0x01でない
          System.out.println (Multilingual.mlnJapanese ?
                              path + " に不明なトラックがあります" :
                              path + " has unclear tracks");
          return null;
        }
      }
      for (int i = tracksPerDisk; i < 170; i++) {  //存在しないトラックの保存状態が0x00でない
        if (array[0x0001 + i] != 0x00) {
          System.out.println (Multilingual.mlnJapanese ?
                              path + " に不明なトラックがあります" :
                              path + " has unclear tracks");
          return null;
        }
      }
      //トラック保存状態をコピーする
      byte[] savedFlag = new byte[tracksPerDisk];
      System.arraycopy (array, 1, savedFlag, 0, tracksPerDisk);
      //保存されているトラックの数を確認する
      int savedTracks = 0;
      for (int track = 0; track < tracksPerDisk; track++) {
        if (savedFlag[track] == 0x01) {  //保存されている
          savedTracks++;
        }
      }
      int bytesPerTrack = outputMedia.fdmBytesPerSector * outputMedia.fdmSectorsPerTrack;
      if (array.length != 256 + bytesPerTrack * savedTracks) {  //ファイルサイズが一致しない
        System.out.println (Multilingual.mlnJapanese ?
                            path + " のファイルサイズが違います" :
                            path + " has wrong file size");
        return null;
      }
      //保存されているトラックを再配置する
      outputArray = new byte[outputMedia.fdmBytesPerDisk];
      int srcTrack = 0;
      for (int dstTrack = 0; dstTrack < tracksPerDisk; dstTrack++) {
        if (savedFlag[dstTrack] == 0x01) {  //保存されているトラック
          System.arraycopy (array, 256 + bytesPerTrack * srcTrack, outputArray, bytesPerTrack * dstTrack, bytesPerTrack);  //コピーする
          srcTrack++;
        } else {  //保存されていないトラック
          Arrays.fill (outputArray, bytesPerTrack * dstTrack, bytesPerTrack * (dstTrack + 1), (byte) 0xe5);  //0xe5で埋める
        }
      }
    } else {  //*.DIM以外
      //メディアを探す
      //  最初にファイルサイズと拡張子が一致したもの、なければ最初にファイルサイズが一致したもの
    checkMedia:
      for (FDMedia media : FDMedia.FDM_ARRAY) {
        if (media.fdmBytesPerDisk == array.length) {  //ファイルサイズが一致
          if (outputMedia == null) {
            outputMedia = media;
          }
          for (String mediaExt : media.fdmExtensionArray) {
            if (upperExt.equals (mediaExt)) {  //ファイルサイズと拡張子が一致
              outputMedia = media;
              break checkMedia;
            }
          }
        }
      }
      if (outputMedia == null) {  //不明
        System.out.println (Multilingual.mlnJapanese ?
                            path + " はフロッピーディスクのイメージファイルではありません" :
                            path + " is not a floppy disk image file");
        return null;
      }
      outputArray = array;
    }
    System.out.println (Multilingual.mlnJapanese ?
                        path + " は " + outputMedia.fdmName + " です" :
                        path + " is " + outputMedia.fdmName);
    if (bb != null) {  //バッファが指定された
      System.arraycopy (outputArray, 0, bb, 0, outputArray.length);
    }
    return outputMedia;
  }  //fdmPathToMedia(String,byte[])

  public int fdmNumber;  //番号
  public int fdmSectorScale;  //セクタスケール=log2(セクタサイズ)-7。0=128バイト,1=256バイト,2=512バイト,3=1024バイト
  public int fdmSectorsPerTrack;  //セクタ数=セクタ/トラック
  public int fdmTracksPerCylinder;  //サイド数=トラック/シリンダ
  public int fdmCylindersPerDisk;  //シリンダ数=シリンダ/ディスク
  public boolean fdmDoubleDensity;  //false=高密度(8MHz,2HD/2HC/2HQ/2HDE/2HS),true=倍密度(4MHz,2DD8/2DD9/2DD10)
  public int fdmTracksPerDisk;  //総トラック数=トラック/ディスク
  public int fdmSectorsPerDisk;  //総セクタ数=セクタ/ディスク
  public int fdmBytesPerSector;  //セクタサイズ=バイト/セクタ=128<<セクタスケール
  public int fdmBytesPerDisk;  //ディスクサイズ=バイト/ディスク=セクタサイズ*セクタ数*サイド数*シリンダ数
  public int fdmTwoSide;  //Two Side信号。0=片面,FDC.FDC_ST3_TS=両面
  public String fdmName;  //名前
  public String[] fdmExtensionArray;  //拡張子
  public int fdmRootStartSector;  //ルートディレクトリの先頭セクタ番号
  public int fdmDataStartSector;  //データ領域の先頭セクタ番号

  public FDMedia (int number,
                  int bytesPerSector, int sectorsPerCluster, int fatCount, int reservedSectors,
                  int rootEntries, int mediaByte, int fatID, int fatSectors, int partitionSectors,
                  int sectorScale, int sectorsPerTrack, int tracksPerCylinder, int cylindersPerDisk,
                  boolean doubleDensity,
                  String name, String extensions) {
    //super (ByteArray.byaRiwz (FDM_DISK_BPB, (number << 5) + 0x0000),  //bytesPerSector
    //       ByteArray.byaRbz ( FDM_DISK_BPB, (number << 5) + 0x0002),  //sectorsPerCluster
    //       ByteArray.byaRbz ( FDM_DISK_BPB, (number << 5) + 0x0005),  //fatCount
    //       ByteArray.byaRiwz (FDM_DISK_BPB, (number << 5) + 0x0003),  //reservedSectors
    //       ByteArray.byaRiwz (FDM_DISK_BPB, (number << 5) + 0x0006),  //rootEntries
    //       ByteArray.byaRbz ( FDM_DISK_BPB, (number << 5) + 0x000a),  //fatID
    //       ByteArray.byaRbz ( FDM_DISK_BPB, (number << 5) + 0x000b),  //fatSectors
    //       (long) ((128 << sectorScale) * sectorsPerTrack * tracksPerCylinder * cylindersPerDisk));  //diskEndByte
    super (bytesPerSector, sectorsPerCluster, fatCount, reservedSectors,
           rootEntries, mediaByte, fatID, fatSectors, (long) bytesPerSector * partitionSectors);
    fdmNumber = number;  //番号
    fdmSectorScale = sectorScale;  //セクタスケール=log2(セクタサイズ)-7
    fdmSectorsPerTrack = sectorsPerTrack;  //1トラックあたりのセクタ数
    fdmTracksPerCylinder = tracksPerCylinder;  //サイド数=1シリンダあたりのトラック数
    fdmCylindersPerDisk = cylindersPerDisk;  //シリンダ数
    fdmDoubleDensity = doubleDensity;  //高密度/倍密度
    fdmTracksPerDisk = tracksPerCylinder * cylindersPerDisk;  //総トラック数
    fdmSectorsPerDisk = sectorsPerTrack * fdmTracksPerDisk;  //総セクタ数
    fdmBytesPerSector = 128 << sectorScale;  //セクタサイズ=1セクタあたりのバイト数
    fdmBytesPerDisk = fdmBytesPerSector * fdmSectorsPerDisk;  //ディスクサイズ
    fdmTwoSide = tracksPerCylinder == 2 ? FDC.FDC_ST3_TS : 0;  //片面/両面
    fdmName = name;  //名前
    fdmExtensionArray = extensions.split (",");  //拡張子
    fdmRootStartSector = (int) (humRootStartByte >> 7 + fdmSectorScale);  //ルートディレクトリの先頭セクタ番号
    fdmDataStartSector = (int) (humDataStartByte >> 7 + fdmSectorScale);  //データ領域の先頭セクタ番号
    if (FDC.FDC_DEBUG_TRACE && FDC.fdcDebugLogOn) {
      fdmPrintInfo ();
    }
  }  //new FDMedia(...)

  //success = media.fdmMakeFormatData (bb, copySystemFiles, x86SafeOn)
  //success = media.fdmMakeFormatData (bb, humanSysOn, commandXOn, x86SafeOn)
  //  フロッピーディスクのフォーマットデータを作る
  //public boolean fdmMakeFormatData (byte[] bb, boolean copySystemFiles, boolean x86SafeOn) {
  public boolean fdmMakeFormatData (byte[] bb, boolean humanSysOn, boolean commandXOn, boolean x86SafeOn) {
    Arrays.fill (bb, 0, (int) humDataStartByte, (byte) 0x00);  //データ領域以外は0x00で埋める
    Arrays.fill (bb, (int) humDataStartByte, (int) humDiskEndByte, (byte) 0xe5);  //データ領域は0xe5で埋める
    //フロッピーディスクIPLを書き込む
    if (this == FDM_2HDE) {
      System.arraycopy (FDM_2HDE_IPL, 0, bb, 0x00000000, FDM_2HDE_IPL.length);
    } else if (this == FDM_2HS) {
      System.arraycopy (FDM_2HS_IPL, 0, bb, 0x00000000, FDM_2HS_IPL.length);
    } else {
      System.arraycopy (FDM_DISK_IPL, 0, bb, 0x00000000, FDM_DISK_IPL.length);
      //x86セーフ
      //  2HQフローピーディスクのIPLの先頭3バイトを0xeb,0xfe,0x90にする
      //  0xeb,0xfe,0x90は8086の命令でJMP(*);NOP
      //  x86系のPCの起動ディスクではないディスクのIPLの先頭3バイトを0xeb,0xfe,0x90にしておくと、誤ってx86系のPCで起動しようとしても余計なことをする前に無限ループに落ちるので安全(ということらしい)
      //  先頭が0x60でないのでX68000でも起動できなくなる
      //  FORMAT.Xの場合
      //    FORMAT.X /4    x86セーフにする
      //    FORMAT.X /4/i  x86セーフにしない
      if (this == FDM_2HQ && x86SafeOn) {  //2HQでx86セーフにするとき
        bb[0] = (byte) 0xeb;
        bb[1] = (byte) 0xfe;
        //bb[2] = (byte) 0x90;  //元々0x90なので不要
      }
      //フロッピーディスクIPLにBPBを埋め込む
      if (false) {
        System.arraycopy (FDM_DISK_BPB,  fdmNumber << 5      , bb, 0x00000000 + (0x200b - 0x2000), 18);
        System.arraycopy (FDM_DISK_BPB, (fdmNumber << 5) + 18, bb, 0x00000000 + (0x2162 - 0x2000), 14);
      } else {
        ByteArray.byaWiw (bb, 0x00000000 + (0x200b - 0x2000) +  0, humBytesPerSector);
        ByteArray.byaWb ( bb, 0x00000000 + (0x200b - 0x2000) +  2, humSectorsPerCluster);
        ByteArray.byaWiw (bb, 0x00000000 + (0x200b - 0x2000) +  3, humReservedSectors);
        ByteArray.byaWb ( bb, 0x00000000 + (0x200b - 0x2000) +  5, humFatCount);
        ByteArray.byaWiw (bb, 0x00000000 + (0x200b - 0x2000) +  6, humRootEntries);
        ByteArray.byaWiw (bb, 0x00000000 + (0x200b - 0x2000) +  8, humPartitionSectors);
        ByteArray.byaWb ( bb, 0x00000000 + (0x200b - 0x2000) + 10, humFatID);
        ByteArray.byaWb ( bb, 0x00000000 + (0x200b - 0x2000) + 11, humFatSectors);
        ByteArray.byaWw ( bb, 0x00000000 + (0x200b - 0x2000) + 12, fdmSectorsPerTrack);
        ByteArray.byaWw ( bb, 0x00000000 + (0x200b - 0x2000) + 14, fdmTracksPerCylinder);
        ByteArray.byaWb ( bb, 0x00000000 + (0x200b - 0x2000) + 16, 0);
        ByteArray.byaWb ( bb, 0x00000000 + (0x200b - 0x2000) + 17, 0);
        ByteArray.byaWl ( bb, 0x00000000 + (0x2162 - 0x2000) +  0, fdmBytesPerSector);  //1セクタあたりのバイト数
        ByteArray.byaWl ( bb, 0x00000000 + (0x2162 - 0x2000) +  4,
                     fdmSectorScale << 24 |
                     0 << 16 |
                     fdmRootStartSector / fdmSectorsPerTrack << 8 |
                     1 + fdmRootStartSector % fdmSectorsPerTrack);  //ルートディレクトリの位置
        ByteArray.byaWw ( bb, 0x00000000 + (0x2162 - 0x2000) +  8, fdmSectorsPerTrack);  //1トラックあたりのセクタ数
        ByteArray.byaWw ( bb, 0x00000000 + (0x2162 - 0x2000) + 10, (fdmBytesPerSector >> 5) - 1);  //humEntriesPerSector-1。1セクタあたりのエントリ数-1
        ByteArray.byaWw ( bb, 0x00000000 + (0x2162 - 0x2000) + 12, fdmDataStartSector - 2);  //データ領域の先頭セクタ番号-2
        if (FDC.FDC_DEBUG_TRACE && FDC.fdcDebugLogOn) {
          if (fdmNumber < 5) {
            System.out.printf ("fdmNumber=%d\n", fdmNumber);
            for (int k = 0; k < 2; k++) {
              int l = k == 0 ? 18 : 14;
              int b1 = (fdmNumber << 5) + (k == 0 ? 0 : 18);
              int b2 = (k == 0 ? 0x200b : 0x2162) - 0x2000;
              System.out.printf ("%04x ", b1);
              for (int i = 0; i < l; i++) {
                System.out.printf (" %02x", FDM_DISK_BPB[b1 + i] & 255);
              }
              System.out.println ();
              System.out.printf ("%04x ", b2);
              boolean match = true;
              for (int i = 0; i < l; i++) {
                System.out.printf (" %02x", bb[b2 + i] & 255);
                match = match && FDM_DISK_BPB[b1 + i] == bb[b2 + i];
              }
              System.out.println (match ? "  OK" : "  ERROR");
            }
          }
        }
      }
    }
    //FAT領域の先頭にFATマーカーを書き込む
    humWriteFatMarker (bb);
    //システムファイルを転送する
    //if (copySystemFiles) {
    //  if (!humCopyHumanSys (bb) ||
    //      !humCopyCommandX (bb)) {
    //    return false;
    //  }
    //}
    if (humanSysOn) {
      if (!humCopyHumanSys (bb)) {
        return false;
      }
      if (commandXOn) {
        if (!humCopyCommandX (bb)) {
          return false;
        }
      }
    }
    if (false) {
      humDumpFat (bb);
    }
    return true;
  }  //media.fdmMakeFormatData(byte[],boolean)

  //dimSize = media.fdmMakeDimImage (dimBuffer, diskImage)
  //  *.DIMのイメージを作る
  //  dimSize    *.DIMのサイズ
  //  dimBuffer  *.DIMのイメージ
  //  diskImage  ディスクイメージ
  //
  //  DIFCヘッダ
  //  $0000   .b              メディアタイプ($00=2HD,$01=2HS,$02=2HC,$03=2HDE,$09=2HQ,$11=N88)
  //  $0001   .b[170]         トラック保存状態($00=なし,$01=あり)
  //  $00AB   .b[15]  'DIFC HEADER  ',$00,$00
  //  $00BA   .l              ファイル作成日付(_DATEGET)
  //  $00BA   .b              ファイル作成曜日(0～6)
  //  $00BB   .b              ファイル作成西暦年-$1980(BCD)
  //  $00BC   .b              ファイル作成月(BCD)
  //  $00BD   .b              ファイル作成月通日(BCD)
  //  $00BE   .l              ファイル作成時刻(_TIMEGET)
  //  $00BE   .b      $01     $00=12時間計,$01=24時間計
  //  $00BF   .b              ファイル作成時(BCD)
  //  $00C0   .b              ファイル作成分(BCD)
  //  $00C1   .b              ファイル作成秒(BCD)
  //  $00C2   .b[60]          コメント。残りは$00。空白を含むときは/*～*/で囲む
  //  $00FE   .b      $19     DIFC.Xのバージョン(BCD)($19=1.19)
  //  $00FF   .b      $00     オーバートラック($00=NO,その他=/Bのトラック数)
  public int fdmMakeDimImage (byte[] dimImage, byte[] diskImage) {
    if (this == FDM_2HD) {
      dimImage[0x0000] = 0x00;  //メディアタイプ
    } else if (this == FDM_2HS) {
      dimImage[0x0000] = 0x01;
    } else if (this == FDM_2HC) {
      dimImage[0x0000] = 0x02;
    } else if (this == FDM_2HDE) {
      dimImage[0x0000] = 0x03;
    } else if (this == FDM_2HQ) {
      dimImage[0x0000] = 0x09;
    } else {
      return -1;  //非対応
    }
    //                                              111 1 11
    //                                    0123456789012 3 45
    ByteArray.byaWstr (dimImage, 0x00ab, "DIFC HEADER  \0\0");  //マジック
    long dttm = DnT.dntDttmNow ();  //現在の日時。ここではRTCの日時ではなくてホストマシンの日時を使う
    dimImage[0x00ba] = (byte) DnT.dntWdayDttm (dttm);  //曜日
    dimImage[0x00bb] = (byte) XEiJ.fmtBcd4 (DnT.dntYearDttm (dttm) - 1980);  //西暦年-1980。DIFC.Xは20xx年を正しく表示できない
    dimImage[0x00bc] = (byte) XEiJ.fmtBcd4 (DnT.dntMontDttm (dttm));  //月
    dimImage[0x00bd] = (byte) XEiJ.fmtBcd4 (DnT.dntMdayDttm (dttm));  //月通日
    dimImage[0x00be] = 0x01;  //$00=12時間計,$01=24時間計
    dimImage[0x00bf] = (byte) XEiJ.fmtBcd4 (DnT.dntHourDttm (dttm));  //時
    dimImage[0x00c0] = (byte) XEiJ.fmtBcd4 (DnT.dntMinuDttm (dttm));  //分
    dimImage[0x00c1] = (byte) XEiJ.fmtBcd4 (DnT.dntSecoDttm (dttm));  //秒
    Arrays.fill (dimImage, 0x00c2, 0x00c2 + 60, (byte) 0x00);  //コメント
    dimImage[0x00fe] = 0x19;  //DIFC.Xのバージョン
    dimImage[0x00ff] = 0x00;  //オーバートラック
    //
    int bytesPerTrack = (128 << fdmSectorScale) * fdmSectorsPerTrack;  //1トラックあたりのバイト数
    int tracksPerDisk = fdmTracksPerCylinder * fdmCylindersPerDisk;  //総トラック数
    Arrays.fill (dimImage, 0x0001, 0x0001 + 170, (byte) 0x00);  //トラックの保存状態
    int diskPointer = 0;  //ディスクイメージの読み出し位置
    int dimPointer = 256;  //*.DIMのイメージの書き込み位置
    for (int trackNumber = 0; trackNumber < tracksPerDisk; trackNumber++) {  //すべてのトラックについて
      for (int offset = 0; offset < bytesPerTrack; offset++) {  //トラックのすべてのデータについて
        if (diskImage[diskPointer + offset] != (byte) 0xe5) {  //0xe5以外のデータがあるとき。符号に注意
          dimImage[0x0001 + trackNumber] = 0x01;  //トラックの保存状態
          System.arraycopy (diskImage, diskPointer, dimImage, dimPointer, bytesPerTrack);  //トラックをコピーする
          dimPointer += bytesPerTrack;
          break;
        }
      }
      diskPointer += bytesPerTrack;
    }
    return dimPointer;
  }  //media.fdmMakeDimImage(byte[],byte[])

  //media.fdmReviveFiles (bb)
  //  削除ファイルを復元する
  public void fdmReviveFiles (byte[] bb) {
    //int code = Arrays.hashCode (bb);  //Arrays.hashCode()は1.5から。Arrays.hashCode()は範囲を指定できないので配列の長さとメディアの種類が合っていないと使えない
    CRC32 crc32 = new CRC32 ();
    crc32.reset ();
    crc32.update (bb, 0, (int) humDiskEndByte);
    int code = (int) crc32.getValue ();
    switch (code) {
    case 0x6930fe0f:  //HUMAN302.XDF(crc32=0x6930fe0f,hashCode=0xbcebaa12)
      humUndel (bb, 1024 * 110 + 32 * 20, 'O');  //SYS/OPMDRV3.X
      humUndel (bb, 1024 * 418 + 32 * 11, 'D');  //BIN/DUMP.X
      break;
      //case 0x7b0d5df3:  //SXTOOLS.XDF(crc32=0x7b0d5df3,hashCode=0x70cd3082)
      //break;
    case 0x14c461bf:  //SXWIN311.XDF(crc32=0x14c461bf,hashCode=0xd6d3e7cd)
      humUndel (bb, 1024 *  98 + 32 *  7, 'S');  //SHELL/SXCON.X
      humUndel (bb, 1024 * 556 + 32 *  6, 'O');  //SYS/OPMDRV3.X
      break;
    case 0x08c4fe7c:  //SXWIN312.XDF(crc32=0x08c4fe7c,hashCode=0x7da348d8)
      humUndel (bb, 1024 *   5 + 32 *  4, 0x8373 >> 8);  //ピンボール.X
      break;
    case 0x479edc8e:  //SXWIN313.XDF(crc32=0x479edc8e,hashCode=0x0eb43290)
      humUndel (bb, 1024 *   5 + 32 *  1, 'S');  //SAMPLE
      humUndel (bb, 1024 *  11 + 32 *  2, 'S');  //SAMPLE/Sample.PCM
      humUndel (bb, 1024 *  11 + 32 *  3, 'X');  //SAMPLE/X68K.OPM
      humUndel (bb, 1024 *  11 + 32 *  4, 'A');  //SAMPLE/ANGEL.PIX
      humUndel (bb, 1024 *  11 + 32 *  5, 0xcc);  //SAMPLE/ﾌｪﾆｯｸｽ.PIC
      humUndel (bb, 1024 *  11 + 32 *  6, 0x90c2 >> 8);  //SAMPLE/青空.PIC
      humUndel (bb, 1024 *  11 + 32 *  7, 0xcf);  //SAMPLE/ﾏｸﾞｶｯﾌﾟ.PIC
      humUndel (bb, 1024 *  11 + 32 *  8, 0x8946 >> 8);  //SAMPLE/宇宙.PIC
      humUndel (bb, 1024 *  11 + 32 *  9, 0x9190 >> 8);  //SAMPLE/草原.JPG
      humUndel (bb, 1024 *  11 + 32 * 10, 0x9356 >> 8);  //SAMPLE/天球儀.CGA
      humUndel (bb, 1024 *  11 + 32 * 11, 0xb6);  //SAMPLE/ｶﾀﾛｸﾞ.PEN
      break;
      //case 0x9ff5ea29:  //SXWIN315.XDF(crc32=0x9ff5ea29,hashCode=0x56f1f53b)
      //break;
    case 0x1b1ac738:  //SXWIN316.XDF(crc32=0x1b1ac738,hashCode=0x36952122)
      humUndel (bb, 1024 *   5 + 32 *  1, 0xb2);  //ｲﾝｽﾄｰﾗ.X
      humUndel (bb, 1024 *   5 + 32 *  2, 0xb2);  //ｲﾝｽﾄｰﾗ.LB
      humUndel (bb, 1024 *   5 + 32 *  3, 'S');  //SX31_HD.INS
      humUndel (bb, 1024 *   5 + 32 *  4, 'S');  //SX31_FD.INS
      humUndel (bb, 1024 *   5 + 32 *  5, 'S');  //SX31INST.LB
      break;
      //case 0x4b3d5e26:  //SXWORK1.XDF(crc32=0x4b3d5e26,hashCode=0x38c9ee13)
      //break;
      //case 0x4046ff76:  //SXWORK2.XDF(crc32=0x4046ff76,hashCode=0x19fa34e1)
      //break;
      //case 0x80775dd1:  //SXWORK3.XDF(crc32=0x80775dd1,hashCode=0x040501a3)
      //break;
      //case 0x89ff5558:  //XC2101.XDF(crc32=0x89ff5558,hashCode=0x18b24c6d)
      //break;
    case 0xd90f9821:  //XC2102.XDF(crc32=0xd90f9821,hashCode=0x5ce5ffc2)
      humUndel (bb, 1024 *   5 + 32 *  3, 'B');  //BC
      humUndel (bb, 1024 * 438 + 32 * 12, 'G');  //BASIC2/GPIB.FNC
      humUndel (bb, 1024 * 542 + 32 *  2, 'B');  //BC/BC.X
      humUndel (bb, 1024 * 542 + 32 *  3, 'B');  //BC/BASIC.CNF
      humUndel (bb, 1024 * 542 + 32 *  4, 'A');  //BC/AUDIO.DEF
      humUndel (bb, 1024 * 542 + 32 *  5, 'B');  //BC/BASIC.DEF
      humUndel (bb, 1024 * 542 + 32 *  6, 'G');  //BC/GRAPH.DEF
      humUndel (bb, 1024 * 542 + 32 *  7, 'I');  //BC/IMAGE.DEF
      humUndel (bb, 1024 * 542 + 32 *  8, 'M');  //BC/MOUSE.DEF
      humUndel (bb, 1024 * 542 + 32 *  9, 'M');  //BC/MUSIC.DEF
      humUndel (bb, 1024 * 542 + 32 * 10, 'M');  //BC/MUSIC3.DEF
      humUndel (bb, 1024 * 542 + 32 * 11, 'S');  //BC/SPRITE.DEF
      humUndel (bb, 1024 * 542 + 32 * 12, 'S');  //BC/STICK.DEF
      humUndel (bb, 1024 * 542 + 32 * 13, 'G');  //BC/GPIB.DEF
      break;
    case 0x68f09cf0:  //XC2103.XDF(crc32=0x68f09cf0,hashCode=0x809478d0)
      humUndel (bb, 1024 *   5 + 32 *  1, 'S');  //SAMPLE
      humUndel (bb, 1024 *   5 + 32 *  2, 'M');  //MUSIC
      humUndel (bb, 1024 *   5 + 32 *  4, 'S');  //SOURCE
      humUndel (bb, 1024 *  11 + 32 *  2, 'S');  //SAMPLE/SAMPLE.DOC
      humUndel (bb, 1024 *  11 + 32 *  3, 'M');  //SAMPLE/MESH.C
      humUndel (bb, 1024 *  11 + 32 *  4, 'D');  //SAMPLE/DEF.BAS
      humUndel (bb, 1024 *  11 + 32 *  5, 'D');  //SAMPLE/DEF.HLP
      humUndel (bb, 1024 *  11 + 32 *  6, 'S');  //SAMPLE/SCSI1.C
      humUndel (bb, 1024 *  11 + 32 *  7, 'S');  //SAMPLE/SCSI2.C
      humUndel (bb, 1024 *  11 + 32 *  8, 'M');  //SAMPLE/MAKEFILE
      humUndel (bb, 1024 *  69 + 32 *  2, 'O');  //MUSIC/OPMDRV2.X
      humUndel (bb, 1024 *  69 + 32 *  3, 'B');  //MUSIC/BASIC2.CNF
      humUndel (bb, 1024 *  69 + 32 *  4, 'A');  //MUSIC/AUDIO.FNC
      humUndel (bb, 1024 *  69 + 32 *  5, 'M');  //MUSIC/MUSIC2.FNC
      humUndel (bb, 1024 *  69 + 32 *  6, 'B');  //MUSIC/BC.CNF
      humUndel (bb, 1024 *  69 + 32 *  7, 'M');  //MUSIC/MUSIC2.DEF
      humUndel (bb, 1024 *  69 + 32 *  8, 'M');  //MUSIC/MUSIC.H
      humUndel (bb, 1024 *  69 + 32 *  9, 'M');  //MUSIC/MUSIC2.H
      humUndel (bb, 1024 *  69 + 32 * 10, 'B');  //MUSIC/BASLIB.L
      humUndel (bb, 1024 *  69 + 32 * 11, 'B');  //MUSIC/BASLIB.ARC
      humUndel (bb, 1024 * 401 + 32 *  2, 'C');  //SOURCE/CLIB.ARC
      humUndel (bb, 1024 * 401 + 32 *  3, 'B');  //SOURCE/BASLIB.ARC
      humUndel (bb, 1024 * 401 + 32 *  4, 'D');  //SOURCE/DOSLIB.ARC
      humUndel (bb, 1024 * 401 + 32 *  5, 'I');  //SOURCE/IOCSLIB.ARC
      break;
    }  //switch(code)
  }  //fdmReviveFiles(byte[])

  //media.fdmPrintInfo ()
  public void fdmPrintInfo () {
    humPrintInfo ();
    System.out.printf ("             fdmNumber=  %12d\n", fdmNumber);
    System.out.printf ("        fdmSectorScale=  %12d\n", fdmSectorScale);
    System.out.printf ("    fdmSectorsPerTrack=  %12d\n", fdmSectorsPerTrack);
    System.out.printf ("  fdmTracksPerCylinder=  %12d\n", fdmTracksPerCylinder);
    System.out.printf ("   fdmCylindersPerDisk=  %12d\n", fdmCylindersPerDisk);
    System.out.printf ("      fdmDoubleDensity=  %12b\n", fdmDoubleDensity);
    System.out.printf ("      fdmTracksPerDisk=  %12d\n", fdmTracksPerDisk);
    System.out.printf ("     fdmSectorsPerDisk=  %12d\n", fdmSectorsPerDisk);
    System.out.printf ("     fdmBytesPerSector=  %12d\n", fdmBytesPerSector);
    System.out.printf ("       fdmBytesPerDisk=  %12d\n", fdmBytesPerDisk);
    System.out.printf ("            fdmTwoSide=  %12d\n", fdmTwoSide);
    System.out.printf ("               fdmName=\"%s\"\n", fdmName);
    System.out.printf ("     fdmExtensionArray=\"%s\"\n", String.join (",", fdmExtensionArray));
    System.out.printf ("    fdmRootStartSector=  %12d\n", fdmRootStartSector);
    System.out.printf ("    fdmDataStartSector=  %12d\n", fdmDataStartSector);
    if (humBytesPerSector != fdmBytesPerSector) {
      System.out.printf ("FDM_ARRAY[%d]: humBytesPerSector(%d) != fdmBytesPerSector(%d)\n",
                         fdmNumber, humBytesPerSector, fdmBytesPerSector);
    }
    if (humPartitionSectors != fdmSectorsPerDisk) {
      System.out.printf ("FDM_ARRAY[%d]: humPartitionSectors(%d) != fdmSectorsPerDisk(%d)\n",
                         fdmNumber, humPartitionSectors, fdmSectorsPerDisk);
    }
    if (humRootStartSector != fdmRootStartSector) {
      System.out.printf ("FDM_ARRAY[%d]: humRootStartSector(%d) != fdmRootStartSector(%d)\n",
                         fdmNumber, humRootStartSector, fdmRootStartSector);
    }
    if (humDataStartSector != fdmDataStartSector) {
      System.out.printf ("FDM_ARRAY[%d]: humDataStartSector(%d) != fdmDataStartSector(%d)\n",
                         fdmNumber, humDataStartSector, fdmDataStartSector);
    }
  }  //media.fdmPrintInfo()

  //========================================================================================
  //フロッピーディスクフォーマットデータ
  //  無償公開されたHuman68k version 3.02のシステムディスクに入っているFORMAT.Xから抽出したIPLデータを使う

  //----------------------------------------------------------------------------------------
  //フロッピーディスクIPL
  //  フロッピーディスクのセクタ0に書き込まれる
  //  HUMAN.SYSを読み込んで起動する
/*
  public static final int[] FDM_DISK_IPL = {
    //  perl -e "do'sjdump.pl';$p=0;$m=1;$o=0x76e;$l=512;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    0x60,0x3c,0x90,0x58,0x36,0x38,0x49,0x50,0x4c,0x33,0x30,0x00,0x04,0x01,0x01,0x00,  //00000000  `<森68IPL30.....
    0x02,0xc0,0x00,0xd0,0x04,0xfe,0x02,0x00,0x08,0x00,0x02,0x00,0x00,0x00,0x00,0x00,  //00000010  .ﾀ.ﾐ............
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x20,0x20,0x20,0x20,  //00000020  ...........     
    0x20,0x20,0x20,0x20,0x20,0x20,0x46,0x41,0x54,0x31,0x32,0x20,0x20,0x20,0x4f,0xfa,  //00000030        FAT12   O.
    0xff,0xc0,0x4d,0xfa,0x01,0xb8,0x4b,0xfa,0x00,0xe0,0x49,0xfa,0x00,0xea,0x43,0xfa,  //00000040  .ﾀM..ｸK..潛..鵑.
    0x01,0x20,0x4e,0x94,0x70,0x8e,0x4e,0x4f,0x7e,0x70,0xe1,0x48,0x8e,0x40,0x26,0x3a,  //00000050  . N廃晒O~p瓸察&:
    0x01,0x02,0x22,0x4e,0x24,0x3a,0x01,0x00,0x32,0x07,0x4e,0x95,0x66,0x28,0x22,0x4e,  //00000060  .."N$:..2.N蒜("N
    0x32,0x3a,0x00,0xfa,0x20,0x49,0x45,0xfa,0x01,0x78,0x70,0x0a,0x00,0x10,0x00,0x20,  //00000070  2:.. IE..xp.... 
    0xb1,0x0a,0x56,0xc8,0xff,0xf8,0x67,0x38,0xd2,0xfc,0x00,0x20,0x51,0xc9,0xff,0xe6,  //00000080  ｱ.Vﾈ..g8ﾒ.. Qﾉ.襴
    0x45,0xfa,0x00,0xe0,0x60,0x10,0x45,0xfa,0x00,0xfa,0x60,0x0a,0x45,0xfa,0x01,0x10,  //00000090   ..濮.E...`.E...
    0x60,0x04,0x45,0xfa,0x01,0x28,0x61,0x00,0x00,0x94,0x22,0x4a,0x4c,0x99,0x00,0x06,  //000000a0  `.E..(a..."JL...
    0x70,0x23,0x4e,0x4f,0x4e,0x94,0x32,0x07,0x70,0x4f,0x4e,0x4f,0x70,0xfe,0x4e,0x4f,  //000000b0  p#NON.2.pONOp.NO
    0x74,0x00,0x34,0x29,0x00,0x1a,0xe1,0x5a,0xd4,0x7a,0x00,0xa4,0x84,0xfa,0x00,0x9c,  //000000c0  t.4)..畛ﾔz.､...怱
    0x84,0x7a,0x00,0x94,0xe2,0x0a,0x64,0x04,0x08,0xc2,0x00,0x18,0x48,0x42,0x52,0x02,  //000000d0   z.披.d..ﾂ..HBR.
    0x22,0x4e,0x26,0x3a,0x00,0x7e,0x32,0x07,0x4e,0x95,0x34,0x7c,0x68,0x00,0x22,0x4e,  //000000e0  "N&:.~2.N.4|h."N
    0x0c,0x59,0x48,0x55,0x66,0xa6,0x54,0x89,0xb5,0xd9,0x66,0xa6,0x2f,0x19,0x20,0x59,  //000000f0  .YHUfｦT卸ﾙfｦ/. Y
    0xd1,0xd9,0x2f,0x08,0x2f,0x11,0x32,0x7c,0x67,0xc0,0x76,0x40,0xd6,0x88,0x4e,0x95,  //00000100  ﾑﾙ/./.2|gﾀv@ﾖ.N.
    0x22,0x1f,0x24,0x1f,0x22,0x5f,0x4a,0x80,0x66,0x00,0xff,0x7c,0xd5,0xc2,0x53,0x81,  //00000110  ".$."_J.f..|ﾕﾂS‘
    0x65,0x04,0x42,0x1a,0x60,0xf8,0x4e,0xd1,0x70,0x46,0x4e,0x4f,0x08,0x00,0x00,0x1e,  //00000120   .B.`.NﾑpFNO....
    0x66,0x02,0x70,0x00,0x4e,0x75,0x70,0x21,0x4e,0x4f,0x4e,0x75,0x72,0x0f,0x70,0x22,  //00000130  f.p.Nup!NONur.p"
    0x4e,0x4f,0x72,0x19,0x74,0x0c,0x70,0x23,0x4e,0x4f,0x61,0x08,0x72,0x19,0x74,0x0d,  //00000140  NOr.t.p#NOa.r.t.
    0x70,0x23,0x4e,0x4f,0x76,0x2c,0x72,0x20,0x70,0x20,0x4e,0x4f,0x51,0xcb,0xff,0xf8,  //00000150  p#NOv,r p NOQﾋ..
    0x4e,0x75,0x00,0x00,0x04,0x00,0x03,0x00,0x00,0x06,0x00,0x08,0x00,0x1f,0x00,0x09,  //00000160  Nu..............
    0x1a,0x00,0x00,0x22,0x00,0x0d,0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,  //00000170  ..."..Human.sys 
    0x82,0xaa,0x20,0x8c,0xa9,0x82,0xc2,0x82,0xa9,0x82,0xe8,0x82,0xdc,0x82,0xb9,0x82,  //00000180  が 見つかりません
    0xf1,0x00,0x00,0x25,0x00,0x0d,0x83,0x66,0x83,0x42,0x83,0x58,0x83,0x4e,0x82,0xaa,  //00000190   ..%..ディスクが
    0x81,0x40,0x93,0xc7,0x82,0xdf,0x82,0xdc,0x82,0xb9,0x82,0xf1,0x00,0x00,0x00,0x23,  //000001a0  　読めません...#
    0x00,0x0d,0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,0x82,0xaa,0x20,0x89,  //000001b0  ..Human.sys が 壊
    0xf3,0x82,0xea,0x82,0xc4,0x82,0xa2,0x82,0xdc,0x82,0xb7,0x00,0x00,0x20,0x00,0x0d,  //000001c0   れています.. ..
    0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,0x82,0xcc,0x20,0x83,0x41,0x83,  //000001d0  Human.sys の アド
    0x68,0x83,0x8c,0x83,0x58,0x82,0xaa,0x88,0xd9,0x8f,0xed,0x82,0xc5,0x82,0xb7,0x00,  //000001e0   レスが異常です.
    0x68,0x75,0x6d,0x61,0x6e,0x20,0x20,0x20,0x73,0x79,0x73,0x00,0x00,0x00,0x00,0x00,  //000001f0  human   sys.....
  };
*/
  //  perl misc/itob.pl xeij/FDMedia.java FDM_DISK_IPL
  public static final byte[] FDM_DISK_IPL = "`<\220X68IPL30\0\4\1\1\0\2\300\0\320\4\376\2\0\b\0\2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0           FAT12   O\372\377\300M\372\1\270K\372\0\340I\372\0\352C\372\1 N\224p\216NO~p\341H\216@&:\1\2\"N$:\1\0002\7N\225f(\"N2:\0\372 IE\372\1xp\n\0\20\0 \261\nV\310\377\370g8\322\374\0 Q\311\377\346E\372\0\340`\20E\372\0\372`\nE\372\1\20`\4E\372\1(a\0\0\224\"JL\231\0\6p#NON\2242\7pONOp\376NOt\0004)\0\32\341Z\324z\0\244\204\372\0\234\204z\0\224\342\nd\4\b\302\0\30HBR\2\"N&:\0~2\7N\2254|h\0\"N\fYHUf\246T\211\265\331f\246/\31 Y\321\331/\b/\0212|g\300v@\326\210N\225\"\37$\37\"_J\200f\0\377|\325\302S\201e\4B\32`\370N\321pFNO\b\0\0\36f\2p\0Nup!NONur\17p\"NOr\31t\fp#NOa\br\31t\rp#NOv,r p NOQ\313\377\370Nu\0\0\4\0\3\0\0\6\0\b\0\37\0\t\32\0\0\"\0\rHuman.sys \202\252 \214\251\202\302\202\251\202\350\202\334\202\271\202\361\0\0%\0\r\203f\203B\203X\203N\202\252\201@\223\307\202\337\202\334\202\271\202\361\0\0\0#\0\rHuman.sys \202\252 \211\363\202\352\202\304\202\242\202\334\202\267\0\0 \0\rHuman.sys \202\314 \203A\203h\203\214\203X\202\252\210\331\217\355\202\305\202\267\0human   sys\0\0\0\0\0".getBytes (XEiJ.ISO_8859_1);

/*
  public static final int[] FDM_2HDE_IPL = {
    //  perl sjdump.pl 2hde.2hde 0 0x1000 2
    0x60,0x20,0x32,0x48,0x44,0x45,0x20,0x76,0x31,0x2e,0x31,0x00,0x00,0x04,0x01,0x01,  //00000000  ` 2HDE v1.1.....
    0x00,0x02,0xc0,0x00,0xa0,0x05,0x03,0x03,0x00,0x09,0x00,0x02,0x00,0x00,0x00,0x00,  //00000010  ..ﾀ.............
    0x90,0x70,0x60,0x00,0x03,0x5a,0x08,0x01,0x00,0x0c,0x66,0x08,0x4d,0xfa,0xff,0xd2,  //00000020  壬`..Z....f.M..ﾒ
    0x2c,0x56,0x4e,0xd6,0x61,0x00,0x00,0xba,0x48,0xe7,0x4f,0x00,0x61,0x00,0x02,0xf0,  //00000030  ,VNﾖa..ｺH躇.a...
    0x61,0x00,0x00,0xc4,0x08,0x00,0x00,0x1b,0x66,0x4e,0xc2,0x3c,0x00,0xc0,0x82,0x3c,  //00000040  a..ﾄ....fNﾂ<.ﾀ.<
    0x00,0x06,0x61,0x00,0x00,0xd0,0xe1,0x9a,0x54,0x88,0x20,0xc2,0xe0,0x9a,0x10,0xc2,  //00000050  ..a..ﾐ癩T. ﾂ燠.ﾂ
    0x10,0xc7,0x10,0x86,0x61,0x00,0x00,0xf0,0x41,0xf8,0x09,0xee,0x70,0x08,0x61,0x00,  //00000060  .ﾇ..a...A..綠.a.
    0x01,0x0c,0x61,0x00,0x01,0x42,0x61,0x00,0x01,0x60,0x61,0x00,0x01,0x7a,0x08,0x00,  //00000070  ..a..Ba..`a..z..
    0x00,0x0e,0x66,0x0c,0x08,0x00,0x00,0x1e,0x67,0x26,0x08,0x00,0x00,0x1b,0x66,0x08,  //00000080  ..f.....g&....f.
    0x61,0x00,0x01,0x7a,0x51,0xcc,0xff,0xbc,0x4c,0xdf,0x00,0xf2,0x4a,0x38,0x09,0xe1,  //00000090  a..zQﾌ.ｼLﾟ..J8.疊
    0x67,0x0c,0x31,0xf8,0x09,0xc2,0x09,0xc4,0x11,0xfc,0x00,0x40,0x09,0xe1,0x4e,0x75,  //000000a0   .1..ﾂ.ﾄ...@.甎u
    0x08,0x00,0x00,0x1f,0x66,0xe2,0xd3,0xc5,0x96,0x85,0x63,0xdc,0x20,0x04,0x48,0x40,  //000000b0  ....f簽ﾅ妹cﾜ .H@
    0x38,0x00,0x30,0x3c,0x00,0x09,0x52,0x02,0xb0,0x02,0x64,0x86,0x14,0x3c,0x00,0x01,  //000000c0  8.0<..R.ｰ.d..<..
    0x0a,0x42,0x01,0x00,0x08,0x02,0x00,0x08,0x66,0x00,0xff,0x78,0xd4,0xbc,0x00,0x01,  //000000d0  .B......f..xﾔｼ..
    0x00,0x00,0x61,0x00,0x01,0xb8,0x08,0x00,0x00,0x1b,0x66,0xac,0x60,0x00,0xff,0x64,  //000000e0  ..a..ｸ....fｬ`..d
    0x08,0x38,0x00,0x07,0x09,0xe1,0x66,0x0c,0x48,0xe7,0xc0,0x00,0x61,0x00,0x01,0x46,  //000000f0  .8...畴.H鄲.a..F
    0x4c,0xdf,0x00,0x03,0x4e,0x75,0x70,0x00,0x78,0x00,0x08,0x01,0x00,0x05,0x67,0x08,  //00000100  Lﾟ..Nup.x.....g.
    0x78,0x09,0x48,0x44,0x38,0x3c,0x00,0x09,0x08,0x01,0x00,0x04,0x67,0x04,0x61,0x00,  //00000110  x.HD8<......g.a.
    0x01,0x7c,0x4e,0x75,0x2f,0x01,0x41,0xf8,0x09,0xee,0x10,0x81,0xe0,0x99,0xc2,0x3c,  //00000120  .|Nu/.A....≒卍<
    0x00,0x03,0x08,0x02,0x00,0x08,0x67,0x04,0x08,0xc1,0x00,0x02,0x11,0x41,0x00,0x01,  //00000130  ......g..ﾁ...A..
    0x22,0x1f,0x4e,0x75,0x13,0xfc,0x00,0xff,0x00,0xe8,0x40,0x00,0x13,0xfc,0x00,0x32,  //00000140  ".Nu.....錙....2
    0x00,0xe8,0x40,0x05,0x60,0x10,0x13,0xfc,0x00,0xff,0x00,0xe8,0x40,0x00,0x13,0xfc,  //00000150  .錙.`......錙...
    0x00,0xb2,0x00,0xe8,0x40,0x05,0x23,0xc9,0x00,0xe8,0x40,0x0c,0x33,0xc5,0x00,0xe8,  //00000160  .ｲ.錙.#ﾉ.錙.3ﾅ.錙
    0x40,0x0a,0x13,0xfc,0x00,0x80,0x00,0xe8,0x40,0x07,0x4e,0x75,0x48,0xe7,0x40,0x60,  //00000170   ......錙.NuH蹇`
    0x43,0xf9,0x00,0xe9,0x40,0x01,0x45,0xf9,0x00,0xe9,0x40,0x03,0x40,0xe7,0x00,0x7c,  //00000180  C..顱.E..顱.@..|
    0x07,0x00,0x12,0x11,0x08,0x01,0x00,0x04,0x66,0xf8,0x12,0x11,0x08,0x01,0x00,0x07,  //00000190  ........f.......
    0x67,0xf8,0x08,0x01,0x00,0x06,0x66,0xf2,0x14,0x98,0x51,0xc8,0xff,0xee,0x46,0xdf,  //000001a0  g.....f..浪ﾈ.珖ﾟ
    0x4c,0xdf,0x06,0x02,0x4e,0x75,0x10,0x39,0x00,0xe8,0x40,0x00,0x08,0x00,0x00,0x04,  //000001b0  Lﾟ..Nu.9.錙.....
    0x66,0x0e,0x10,0x39,0x00,0xe9,0x40,0x01,0xc0,0x3c,0x00,0x1f,0x66,0xf4,0x4e,0x75,  //000001c0  f..9.顱.ﾀ<..f.Nu
    0x10,0x39,0x00,0xe8,0x40,0x01,0x4e,0x75,0x10,0x39,0x00,0xe8,0x40,0x00,0x08,0x00,  //000001d0  .9.錙.Nu.9.錙...
    0x00,0x07,0x66,0x08,0x13,0xfc,0x00,0x10,0x00,0xe8,0x40,0x07,0x13,0xfc,0x00,0xff,  //000001e0  ..f......錙.....
    0x00,0xe8,0x40,0x00,0x4e,0x75,0x30,0x01,0xe0,0x48,0xc0,0xbc,0x00,0x00,0x00,0x03,  //000001f0  .錙.Nu0.潯ﾀｼ....
    0xe7,0x40,0x41,0xf8,0x0c,0x90,0xd1,0xc0,0x20,0x10,0x4e,0x75,0x2f,0x00,0xc0,0xbc,  //00000200  蹇A..績ﾀ .Nu/.ﾀｼ
    0x00,0x35,0xff,0x00,0x67,0x2a,0xb8,0x3c,0x00,0x05,0x64,0x24,0x2f,0x38,0x09,0xee,  //00000210  .5..g*ｸ<..d$/8..
    0x2f,0x38,0x09,0xf2,0x3f,0x38,0x09,0xf6,0x61,0x00,0x00,0xc4,0x70,0x64,0x51,0xc8,  //00000220  /8..?8..a..ﾄpdQﾈ
    0xff,0xfe,0x61,0x68,0x31,0xdf,0x09,0xf6,0x21,0xdf,0x09,0xf2,0x21,0xdf,0x09,0xee,  //00000230  ..ah1ﾟ..!ﾟ..!ﾟ..
    0x20,0x1f,0x4e,0x75,0x30,0x01,0xe0,0x48,0x4a,0x00,0x67,0x3c,0xc0,0x3c,0x00,0x03,  //00000240   .Nu0.潯J.g<ﾀ<..
    0x80,0x3c,0x00,0x80,0x08,0xf8,0x00,0x07,0x09,0xe1,0x13,0xc0,0x00,0xe9,0x40,0x07,  //00000250  .<.........ﾀ.顱.
    0x08,0xf8,0x00,0x06,0x09,0xe1,0x66,0x18,0x31,0xf8,0x09,0xc2,0x09,0xc4,0x61,0x00,  //00000260  .....畴.1..ﾂ.ﾄa.
    0x00,0x90,0x08,0x00,0x00,0x1d,0x66,0x08,0x0c,0x78,0x00,0x64,0x09,0xc4,0x64,0xee,  //00000270  ......f..x.d.ﾄd.
    0x08,0xb8,0x00,0x06,0x09,0xe1,0x4e,0x75,0x4a,0x38,0x09,0xe1,0x67,0x0c,0x31,0xf8,  //00000280  .ｸ...甎uJ8.疊.1.
    0x09,0xc2,0x09,0xc4,0x11,0xfc,0x00,0x40,0x09,0xe1,0x4e,0x75,0x61,0x12,0x08,0x00,  //00000290  .ﾂ.ﾄ...@.甎ua...
    0x00,0x1b,0x66,0x26,0x48,0x40,0x48,0x42,0xb4,0x00,0x67,0x1a,0x48,0x42,0x61,0x3e,  //000002a0  ..f&H@HBｴ.g.HBa>
    0x2f,0x01,0x12,0x3c,0x00,0x0f,0x61,0x00,0xfe,0x6c,0x48,0x42,0x11,0x42,0x00,0x02,  //000002b0  /..<..a..lHB.B..
    0x48,0x42,0x70,0x02,0x60,0x08,0x48,0x42,0x48,0x40,0x4e,0x75,0x2f,0x01,0x61,0x00,  //000002c0  HBp.`.HBH@Nu/.a.
    0xfe,0xac,0x61,0x00,0xfe,0xee,0x22,0x1f,0x30,0x01,0xe0,0x48,0xc0,0xbc,0x00,0x00,  //000002d0  .ｬa...".0.潯ﾀｼ..
    0x00,0x03,0xe7,0x40,0x41,0xf8,0x0c,0x90,0xd1,0xc0,0x20,0x10,0x4e,0x75,0x2f,0x01,  //000002e0  ..蹇A..績ﾀ .Nu/.
    0x12,0x3c,0x00,0x07,0x61,0x00,0xfe,0x2e,0x70,0x01,0x61,0xd0,0x22,0x1f,0x4e,0x75,  //000002f0  .<..a...p.aﾐ".Nu
    0x2f,0x01,0x12,0x3c,0x00,0x04,0x61,0x00,0xfe,0x1c,0x22,0x1f,0x70,0x01,0x61,0x00,  //00000300  /..<..a...".p.a.
    0xfe,0x6c,0x10,0x39,0x00,0xe9,0x40,0x01,0xc0,0x3c,0x00,0xd0,0xb0,0x3c,0x00,0xd0,  //00000310  .l.9.顱.ﾀ<.ﾐｰ<.ﾐ
    0x66,0xf0,0x70,0x00,0x10,0x39,0x00,0xe9,0x40,0x03,0xe0,0x98,0x4e,0x75,0x53,0x02,  //00000320  f.p..9.顱.燔NuS.
    0x7e,0x00,0x3a,0x02,0xe0,0x5d,0x4a,0x05,0x67,0x04,0x06,0x45,0x08,0x00,0xe0,0x4d,  //00000330  ~.:.濔J.g..E..潼
    0x48,0x42,0x02,0x82,0x00,0x00,0x00,0xff,0xe9,0x8a,0xd4,0x45,0x0c,0x42,0x00,0x04,  //00000340  HB......驫ﾔE.B..
    0x65,0x02,0x54,0x42,0x84,0xfc,0x00,0x12,0x48,0x42,0x3e,0x02,0x8e,0xfc,0x00,0x09,  //00000350  e.TB....HB>.周..
    0x48,0x47,0xe1,0x4f,0xe0,0x8f,0x34,0x07,0x06,0x82,0x03,0x00,0x80,0x01,0x2a,0x3c,  //00000360  HG甍熄4.......*<
    0x00,0x00,0x04,0x00,0x3c,0x3c,0x00,0xff,0x3e,0x3c,0x09,0x28,0x4e,0x75,0x4f,0xfa,  //00000370  ....<<..><.(NuO.
    0xfc,0x80,0x43,0xfa,0xfc,0xa2,0x4d,0xfa,0xfc,0x78,0x2c,0xb9,0x00,0x00,0x05,0x18,  //00000380  ..C..｢M..x,ｹ....
    0x23,0xc9,0x00,0x00,0x05,0x18,0x43,0xfa,0x00,0xda,0x4d,0xfa,0xfc,0x68,0x2c,0xb9,  //00000390  #ﾉ....C..ﾚM..h,ｹ
    0x00,0x00,0x05,0x14,0x23,0xc9,0x00,0x00,0x05,0x14,0x43,0xfa,0x01,0x6e,0x4d,0xfa,  //000003a0  ....#ﾉ....C..nM.
    0xfc,0x58,0x2c,0xb9,0x00,0x00,0x05,0x04,0x23,0xc9,0x00,0x00,0x05,0x04,0x24,0x3c,  //000003b0  .X,ｹ....#ﾉ....$<
    0x03,0x00,0x00,0x04,0x20,0x3c,0x00,0x00,0x00,0x8e,0x4e,0x4f,0x12,0x00,0xe1,0x41,  //000003c0  .... <...晒O..瓣
    0x12,0x3c,0x00,0x70,0x33,0xc1,0x00,0x00,0x00,0x66,0x26,0x3c,0x00,0x00,0x04,0x00,  //000003d0  .<.p3ﾁ...f&<....
    0x43,0xfa,0x00,0x20,0x61,0x04,0x60,0x00,0x01,0xec,0x48,0xe7,0x78,0x40,0x70,0x46,  //000003e0  C.. a.`...H輾@pF
    0x4e,0x4f,0x08,0x00,0x00,0x1e,0x66,0x02,0x70,0x00,0x4c,0xdf,0x02,0x1e,0x4e,0x75,  //000003f0  NO....f.p.Lﾟ..Nu
    0xf8,0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000400  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000410  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000420  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000430  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000440  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000450  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000460  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000470  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000480  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000490  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000500  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000510  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000520  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000530  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000540  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000550  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000560  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000570  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000580  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000590  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000600  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000610  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000620  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000630  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000640  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000650  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000660  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000670  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000680  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000690  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000700  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000710  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000720  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000730  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000740  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000750  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000760  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000770  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000780  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000790  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000800  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000810  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000820  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000830  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000840  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000850  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000860  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000870  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000880  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000890  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000900  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000910  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000920  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000930  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000940  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000950  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000960  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000970  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000980  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000990  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000aa0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ab0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ac0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ad0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ae0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000af0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ba0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bb0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bc0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bd0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000be0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bf0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c60  ................
    0x08,0x01,0x00,0x0c,0x66,0x08,0x4d,0xfa,0xfb,0x8a,0x2c,0x56,0x4e,0xd6,0x61,0x00,  //00000c70  ....f.M...,VNﾖa.
    0xfc,0x6e,0x48,0xe7,0x4f,0x00,0x61,0x00,0xfe,0xa4,0x61,0x00,0xfc,0x78,0x08,0x00,  //00000c80  .nH躇.a..､a..x..
    0x00,0x1b,0x66,0x30,0xc2,0x3c,0x00,0xc0,0x82,0x3c,0x00,0x05,0x60,0x08,0x30,0x3c,  //00000c90  ..f0ﾂ<.ﾀ.<..`.0<
    0x01,0xac,0x51,0xc8,0xff,0xfe,0x61,0x00,0x00,0xfc,0x08,0x00,0x00,0x1e,0x67,0x2c,  //00000ca0  .ｬQﾈ..a.......g,
    0x08,0x00,0x00,0x1b,0x66,0x0e,0x08,0x00,0x00,0x11,0x66,0x08,0x61,0x00,0xfd,0x4c,  //00000cb0  ....f.....f.a..L
    0x51,0xcc,0xff,0xe4,0x4c,0xdf,0x00,0xf2,0x4a,0x38,0x09,0xe1,0x67,0x0c,0x31,0xf8,  //00000cc0  Qﾌ.膈ﾟ..J8.疊.1.
    0x09,0xc2,0x09,0xc4,0x11,0xfc,0x00,0x40,0x09,0xe1,0x4e,0x75,0x08,0x00,0x00,0x1f,  //00000cd0  .ﾂ.ﾄ...@.甎u....
    0x66,0xe2,0xd3,0xc5,0x96,0x85,0x63,0xdc,0x20,0x04,0x48,0x40,0x38,0x00,0x30,0x3c,  //00000ce0  f簽ﾅ妹cﾜ .H@8.0<
    0x00,0x09,0x52,0x02,0xb0,0x02,0x64,0xae,0x14,0x3c,0x00,0x01,0x0a,0x42,0x01,0x00,  //00000cf0  ..R.ｰ.dｮ.<...B..
    0x08,0x02,0x00,0x08,0x66,0x98,0xd4,0xbc,0x00,0x01,0x00,0x00,0x61,0x00,0xfd,0x8c,  //00000d00  ....f儘ｼ....a...
    0x08,0x00,0x00,0x1b,0x66,0xae,0x60,0x8e,0x08,0x01,0x00,0x0c,0x66,0x08,0x4d,0xfa,  //00000d10  ....fｮ`.....f.M.
    0xfa,0xde,0x2c,0x56,0x4e,0xd6,0x61,0x00,0xfb,0xc6,0x48,0xe7,0x4f,0x00,0x61,0x00,  //00000d20  .ﾞ,VNﾖa..ﾆH躇.a.
    0xfd,0xfc,0x61,0x00,0xfb,0xd0,0x08,0x00,0x00,0x1b,0x66,0x24,0xc2,0x3c,0x00,0xc0,  //00000d30  ..a..ﾐ....f$ﾂ<.ﾀ
    0x82,0x3c,0x00,0x11,0x61,0x5e,0x08,0x00,0x00,0x0a,0x66,0x14,0x08,0x00,0x00,0x1e,  //00000d40  .<..a^....f.....
    0x67,0x16,0x08,0x00,0x00,0x1b,0x66,0x08,0x61,0x00,0xfc,0xb0,0x51,0xcc,0xff,0xe6,  //00000d50  g.....f.a..ｰQﾌ.覘
    0x4c,0xdf,0x00,0xf2,0x60,0x00,0xfb,0x34,0x08,0x00,0x00,0x1f,0x66,0xf2,0xd3,0xc5,  //00000d60   ﾟ..`..4....f.ﾓﾅ
    0x96,0x85,0x63,0xec,0x20,0x04,0x48,0x40,0x38,0x00,0x30,0x3c,0x00,0x09,0x52,0x02,  //00000d70  妹c. .H@8.0<..R.
    0xb0,0x02,0x64,0xc0,0x14,0x3c,0x00,0x01,0x0a,0x42,0x01,0x00,0x08,0x02,0x00,0x08,  //00000d80  ｰ.dﾀ.<...B......
    0x66,0xb2,0xd4,0xbc,0x00,0x01,0x00,0x00,0x61,0x00,0xfd,0x00,0x08,0x00,0x00,0x1b,  //00000d90  fｲﾔｼ....a.......
    0x66,0xbe,0x60,0xa0,0x61,0x00,0xfb,0x7c,0xe1,0x9a,0x54,0x88,0x20,0xc2,0xe0,0x9a,  //00000da0  fｾ`.a..|癩T. ﾂ燠
    0x10,0xc2,0x10,0xc7,0x10,0x86,0x61,0x00,0xfb,0x8a,0x41,0xf8,0x09,0xee,0x70,0x08,  //00000db0  .ﾂ.ﾇ..a..晦..綠.
    0x61,0x00,0xfb,0xb8,0x61,0x00,0xfb,0xee,0x61,0x00,0xfc,0x0c,0x61,0x00,0xfc,0x26,  //00000dc0  a..ｸa..礼...a..&
    0x4e,0x75,0x43,0xfa,0x01,0x8c,0x61,0x00,0x01,0x76,0x24,0x3c,0x03,0x00,0x00,0x06,  //00000dd0  NuC..径..v$<....
    0x32,0x39,0x00,0x00,0x00,0x66,0x26,0x3c,0x00,0x00,0x04,0x00,0x43,0xf8,0x28,0x00,  //00000de0  29...f&<....C.(.
    0x61,0x00,0xfd,0xf6,0x4a,0x80,0x66,0x00,0x01,0x20,0x43,0xf8,0x28,0x00,0x49,0xfa,  //00000df0  a...J.f.. C.(.I.
    0x01,0x54,0x78,0x1f,0x24,0x49,0x26,0x4c,0x7a,0x0a,0x10,0x1a,0x80,0x3c,0x00,0x20,  //00000e00  .Tx.$I&Lz....<. 
    0xb0,0x1b,0x66,0x06,0x51,0xcd,0xff,0xf4,0x60,0x0c,0x43,0xe9,0x00,0x20,0x51,0xcc,  //00000e10  ｰ.f.Qﾍ..`.C.. Qﾌ
    0xff,0xe4,0x66,0x00,0x00,0xf4,0x30,0x29,0x00,0x1a,0xe1,0x58,0x55,0x40,0xd0,0x7c,  //00000e20  .臟...0)..畊U@ﾐ|
    0x00,0x0b,0x34,0x00,0xc4,0x7c,0x00,0x07,0x52,0x02,0xe8,0x48,0x64,0x04,0x84,0x7c,  //00000e30  ..4.ﾄ|..R.鍠d.л
    0x01,0x00,0x48,0x42,0x34,0x3c,0x03,0x00,0x14,0x00,0x48,0x42,0x26,0x29,0x00,0x1c,  //00000e40  ..HB4<....HB&)..
    0xe1,0x5b,0x48,0x43,0xe1,0x5b,0x43,0xf8,0x67,0xc0,0x61,0x00,0xfd,0x8c,0x0c,0x51,  //00000e50  畆HC畆C.gﾀa....Q
    0x48,0x55,0x66,0x00,0x00,0xb4,0x4b,0xf8,0x68,0x00,0x49,0xfa,0x00,0x4c,0x22,0x4d,  //00000e60  HUf..ｴK.h.I..L"M
    0x43,0xf1,0x38,0xc0,0x2c,0x3c,0x00,0x04,0x00,0x00,0x0c,0x69,0x4e,0xd4,0xff,0xd2,  //00000e70  C.8ﾀ,<.....iNﾔ.ﾒ
    0x66,0x36,0x0c,0xad,0x4c,0x5a,0x58,0x20,0x00,0x04,0x66,0x16,0x2b,0x46,0x00,0x04,  //00000e80  f6.ｭLZX ..f.+F..
    0x2b,0x4d,0x00,0x08,0x42,0xad,0x00,0x20,0x51,0xf9,0x00,0x00,0x07,0x9c,0x4e,0xed,  //00000e90  +M..Bｭ. Q....廸.
    0x00,0x02,0x0c,0x6d,0x4e,0xec,0x00,0x1a,0x66,0x0e,0x0c,0x6d,0x4e,0xea,0x00,0x2a,  //00000ea0  ...mN...f..mN..*
    0x66,0x06,0x43,0xfa,0x01,0x20,0x60,0x64,0x10,0x3c,0x00,0xc0,0x41,0xf8,0x68,0x00,  //00000eb0  f.C.. `d.<.ﾀA.h.
    0x36,0x3c,0xff,0xff,0xb0,0x18,0x67,0x26,0x51,0xcb,0xff,0xfa,0x43,0xf8,0x68,0x00,  //00000ec0  6<..ｰ.g&Qﾋ..C.h.
    0x4a,0x39,0x00,0x00,0x07,0x9c,0x67,0x14,0x41,0xf8,0x67,0xcc,0x24,0x18,0xd4,0x98,  //00000ed0  J9...徃.A.gﾌ$.ﾔ.
    0x22,0x10,0xd1,0xc2,0x53,0x81,0x65,0x04,0x42,0x18,0x60,0xf8,0x4e,0xd1,0x0c,0x10,  //00000ee0  ".ﾑﾂS‘.B.`.Nﾑ..
    0x00,0x04,0x66,0xd0,0x52,0x88,0x0c,0x10,0x00,0xd0,0x66,0xc8,0x52,0x88,0x0c,0x10,  //00000ef0  ..fﾐR....ﾐfﾈR...
    0x00,0xfe,0x66,0xc0,0x52,0x88,0x0c,0x10,0x00,0x02,0x66,0xb8,0x57,0x88,0x30,0xfc,  //00000f00  ..fﾀR.....fｸW.0.
    0x05,0x9e,0x10,0xbc,0x00,0xfb,0x60,0xac,0x43,0xfa,0x00,0x93,0x2f,0x09,0x43,0xfa,  //00000f10  ...ｼ..`ｬC.../.C.
    0x00,0x48,0x61,0x2a,0x43,0xfa,0x00,0x47,0x61,0x24,0x43,0xfa,0x00,0x53,0x61,0x1e,  //00000f20  .Ha*C..Ga$C..Sa.
    0x43,0xfa,0x00,0x44,0x61,0x18,0x43,0xfa,0x00,0x47,0x61,0x12,0x22,0x5f,0x61,0x0e,  //00000f30  C..Da.C..Ga."_a.
    0x32,0x39,0x00,0x00,0x00,0x66,0x70,0x4f,0x4e,0x4f,0x70,0xfe,0x4e,0x4f,0x70,0x21,  //00000f40  29...fpONOp.NOp!
    0x4e,0x4f,0x4e,0x75,0x68,0x75,0x6d,0x61,0x6e,0x20,0x20,0x20,0x73,0x79,0x73,0x00,  //00000f50  NONuhuman   sys.
    0x32,0x48,0x44,0x45,0x49,0x50,0x4c,0x00,0x1b,0x5b,0x34,0x37,0x6d,0x1b,0x5b,0x31,  //00000f60  2HDEIPL..[47m.[1
    0x33,0x3b,0x32,0x36,0x48,0x00,0x1b,0x5b,0x31,0x34,0x3b,0x32,0x36,0x48,0x00,0x20,  //00000f70  3;26H..[14;26H. 
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //00000f80                  
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //00000f90                  
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,0x1b,0x5b,0x31,  //00000fa0              ..[1
    0x34,0x3b,0x33,0x34,0x48,0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,0x73,0x20,0x82,  //00000fb0  4;34HHuman.sys の
    0xcc,0x93,0xc7,0x82,0xdd,0x8d,0x9e,0x82,0xdd,0x83,0x47,0x83,0x89,0x81,0x5b,0x82,  //00000fc0   読み込みエラーで
    0xc5,0x82,0xb7,0x00,0x1b,0x5b,0x31,0x34,0x3b,0x33,0x34,0x48,0x4c,0x5a,0x58,0x2e,  //00000fd0   す..[14;34HLZX.
    0x58,0x20,0x82,0xcc,0x83,0x6f,0x81,0x5b,0x83,0x57,0x83,0x87,0x83,0x93,0x82,0xaa,  //00000fe0  X のバージョンが
    0x8c,0xc3,0x82,0xb7,0x82,0xac,0x82,0xdc,0x82,0xb7,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ff0  古すぎます......
  };
*/
  //  perl misc/itob.pl xeij/FDMedia.java FDM_2HDE_IPL
  public static final byte[] FDM_2HDE_IPL = "` 2HDE v1.1\0\0\4\1\1\0\2\300\0\240\5\3\3\0\t\0\2\0\0\0\0\220p`\0\3Z\b\1\0\ff\bM\372\377\322,VN\326a\0\0\272H\347O\0a\0\2\360a\0\0\304\b\0\0\33fN\302<\0\300\202<\0\6a\0\0\320\341\232T\210 \302\340\232\20\302\20\307\20\206a\0\0\360A\370\t\356p\ba\0\1\fa\0\1Ba\0\1`a\0\1z\b\0\0\16f\f\b\0\0\36g&\b\0\0\33f\ba\0\1zQ\314\377\274L\337\0\362J8\t\341g\f1\370\t\302\t\304\21\374\0@\t\341Nu\b\0\0\37f\342\323\305\226\205c\334 \4H@8\0000<\0\tR\2\260\2d\206\24<\0\1\nB\1\0\b\2\0\bf\0\377x\324\274\0\1\0\0a\0\1\270\b\0\0\33f\254`\0\377d\b8\0\7\t\341f\fH\347\300\0a\0\1FL\337\0\3Nup\0x\0\b\1\0\5g\bx\tHD8<\0\t\b\1\0\4g\4a\0\1|Nu/\1A\370\t\356\20\201\340\231\302<\0\3\b\2\0\bg\4\b\301\0\2\21A\0\1\"\37Nu\23\374\0\377\0\350@\0\23\374\0002\0\350@\5`\20\23\374\0\377\0\350@\0\23\374\0\262\0\350@\5#\311\0\350@\f3\305\0\350@\n\23\374\0\200\0\350@\7NuH\347@`C\371\0\351@\1E\371\0\351@\3@\347\0|\7\0\22\21\b\1\0\4f\370\22\21\b\1\0\7g\370\b\1\0\6f\362\24\230Q\310\377\356F\337L\337\6\2Nu\209\0\350@\0\b\0\0\4f\16\209\0\351@\1\300<\0\37f\364Nu\209\0\350@\1Nu\209\0\350@\0\b\0\0\7f\b\23\374\0\20\0\350@\7\23\374\0\377\0\350@\0Nu0\1\340H\300\274\0\0\0\3\347@A\370\f\220\321\300 \20Nu/\0\300\274\0005\377\0g*\270<\0\5d$/8\t\356/8\t\362?8\t\366a\0\0\304pdQ\310\377\376ah1\337\t\366!\337\t\362!\337\t\356 \37Nu0\1\340HJ\0g<\300<\0\3\200<\0\200\b\370\0\7\t\341\23\300\0\351@\7\b\370\0\6\t\341f\0301\370\t\302\t\304a\0\0\220\b\0\0\35f\b\fx\0d\t\304d\356\b\270\0\6\t\341NuJ8\t\341g\f1\370\t\302\t\304\21\374\0@\t\341Nua\22\b\0\0\33f&H@HB\264\0g\32HBa>/\1\22<\0\17a\0\376lHB\21B\0\2HBp\2`\bHBH@Nu/\1a\0\376\254a\0\376\356\"\0370\1\340H\300\274\0\0\0\3\347@A\370\f\220\321\300 \20Nu/\1\22<\0\7a\0\376.p\1a\320\"\37Nu/\1\22<\0\4a\0\376\34\"\37p\1a\0\376l\209\0\351@\1\300<\0\320\260<\0\320f\360p\0\209\0\351@\3\340\230NuS\2~\0:\2\340]J\5g\4\6E\b\0\340MHB\2\202\0\0\0\377\351\212\324E\fB\0\4e\2TB\204\374\0\22HB>\2\216\374\0\tHG\341O\340\2174\7\6\202\3\0\200\1*<\0\0\4\0<<\0\377><\t(NuO\372\374\200C\372\374\242M\372\374x,\271\0\0\5\30#\311\0\0\5\30C\372\0\332M\372\374h,\271\0\0\5\24#\311\0\0\5\24C\372\1nM\372\374X,\271\0\0\5\4#\311\0\0\5\4$<\3\0\0\4 <\0\0\0\216NO\22\0\341A\22<\0p3\301\0\0\0f&<\0\0\4\0C\372\0 a\4`\0\1\354H\347x@pFNO\b\0\0\36f\2p\0L\337\2\36Nu\370\377\377\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\b\1\0\ff\bM\372\373\212,VN\326a\0\374nH\347O\0a\0\376\244a\0\374x\b\0\0\33f0\302<\0\300\202<\0\5`\b0<\1\254Q\310\377\376a\0\0\374\b\0\0\36g,\b\0\0\33f\16\b\0\0\21f\ba\0\375LQ\314\377\344L\337\0\362J8\t\341g\f1\370\t\302\t\304\21\374\0@\t\341Nu\b\0\0\37f\342\323\305\226\205c\334 \4H@8\0000<\0\tR\2\260\2d\256\24<\0\1\nB\1\0\b\2\0\bf\230\324\274\0\1\0\0a\0\375\214\b\0\0\33f\256`\216\b\1\0\ff\bM\372\372\336,VN\326a\0\373\306H\347O\0a\0\375\374a\0\373\320\b\0\0\33f$\302<\0\300\202<\0\21a^\b\0\0\nf\24\b\0\0\36g\26\b\0\0\33f\ba\0\374\260Q\314\377\346L\337\0\362`\0\3734\b\0\0\37f\362\323\305\226\205c\354 \4H@8\0000<\0\tR\2\260\2d\300\24<\0\1\nB\1\0\b\2\0\bf\262\324\274\0\1\0\0a\0\375\0\b\0\0\33f\276`\240a\0\373|\341\232T\210 \302\340\232\20\302\20\307\20\206a\0\373\212A\370\t\356p\ba\0\373\270a\0\373\356a\0\374\fa\0\374&NuC\372\1\214a\0\1v$<\3\0\0\00629\0\0\0f&<\0\0\4\0C\370(\0a\0\375\366J\200f\0\1 C\370(\0I\372\1Tx\37$I&Lz\n\20\32\200<\0 \260\33f\6Q\315\377\364`\fC\351\0 Q\314\377\344f\0\0\3640)\0\32\341XU@\320|\0\0134\0\304|\0\7R\2\350Hd\4\204|\1\0HB4<\3\0\24\0HB&)\0\34\341[HC\341[C\370g\300a\0\375\214\fQHUf\0\0\264K\370h\0I\372\0L\"MC\3618\300,<\0\4\0\0\fiN\324\377\322f6\f\255LZX \0\4f\26+F\0\4+M\0\bB\255\0 Q\371\0\0\7\234N\355\0\2\fmN\354\0\32f\16\fmN\352\0*f\6C\372\1 `d\20<\0\300A\370h\0006<\377\377\260\30g&Q\313\377\372C\370h\0J9\0\0\7\234g\24A\370g\314$\30\324\230\"\20\321\302S\201e\4B\30`\370N\321\f\20\0\4f\320R\210\f\20\0\320f\310R\210\f\20\0\376f\300R\210\f\20\0\2f\270W\2100\374\5\236\20\274\0\373`\254C\372\0\223/\tC\372\0Ha*C\372\0Ga$C\372\0Sa\36C\372\0Da\30C\372\0Ga\22\"_a\01629\0\0\0fpONOp\376NOp!NONuhuman   sys\0002HDEIPL\0\33[47m\33[13;26H\0\33[14;26H\0                                             \0\33[14;34HHuman.sys \202\314\223\307\202\335\215\236\202\335\203G\203\211\201[\202\305\202\267\0\33[14;34HLZX.X \202\314\203o\201[\203W\203\207\203\223\202\252\214\303\202\267\202\254\202\334\202\267\0\0\0\0\0\0".getBytes (XEiJ.ISO_8859_1);

/*
  public static final int[] FDM_2HS_IPL = {
    //  perl sjdump.pl 2hs.2hs 0 0x1000 2
    0x60,0x1e,0x39,0x53,0x43,0x46,0x4d,0x54,0x20,0x49,0x50,0x4c,0x20,0x76,0x31,0x2e,  //00000000  `.9SCFMT IPL v1.
    0x30,0x32,0x04,0x00,0x01,0x03,0x00,0x01,0x00,0xc0,0x05,0xa0,0xfb,0x01,0x90,0x70,  //00000010  02.......ﾀ....壬
    0x60,0x00,0x03,0x5a,0x08,0x01,0x00,0x0c,0x66,0x08,0x4d,0xfa,0xff,0xd4,0x2c,0x56,  //00000020  `..Z....f.M..ﾔ,V
    0x4e,0xd6,0x61,0x00,0x00,0xba,0x48,0xe7,0x4f,0x00,0x61,0x00,0x02,0xf0,0x61,0x00,  //00000030  Nﾖa..ｺH躇.a...a.
    0x00,0xc4,0x08,0x00,0x00,0x1b,0x66,0x4e,0xc2,0x3c,0x00,0xc0,0x82,0x3c,0x00,0x06,  //00000040  .ﾄ....fNﾂ<.ﾀ.<..
    0x61,0x00,0x00,0xd0,0xe1,0x9a,0x54,0x88,0x20,0xc2,0xe0,0x9a,0x10,0xc2,0x10,0xc7,  //00000050  a..ﾐ癩T. ﾂ燠.ﾂ.ﾇ
    0x10,0x86,0x61,0x00,0x00,0xf0,0x41,0xf8,0x09,0xee,0x70,0x08,0x61,0x00,0x01,0x0c,  //00000060  ..a...A..綠.a...
    0x61,0x00,0x01,0x42,0x61,0x00,0x01,0x60,0x61,0x00,0x01,0x7a,0x08,0x00,0x00,0x0e,  //00000070  a..Ba..`a..z....
    0x66,0x0c,0x08,0x00,0x00,0x1e,0x67,0x26,0x08,0x00,0x00,0x1b,0x66,0x08,0x61,0x00,  //00000080  f.....g&....f.a.
    0x01,0x7a,0x51,0xcc,0xff,0xbc,0x4c,0xdf,0x00,0xf2,0x4a,0x38,0x09,0xe1,0x67,0x0c,  //00000090  .zQﾌ.ｼLﾟ..J8.疊.
    0x31,0xf8,0x09,0xc2,0x09,0xc4,0x11,0xfc,0x00,0x40,0x09,0xe1,0x4e,0x75,0x08,0x00,  //000000a0  1..ﾂ.ﾄ...@.甎u..
    0x00,0x1f,0x66,0xe2,0xd3,0xc5,0x96,0x85,0x63,0xdc,0x20,0x04,0x48,0x40,0x38,0x00,  //000000b0  ..f簽ﾅ妹cﾜ .H@8.
    0x30,0x3c,0x00,0x12,0x52,0x02,0xb0,0x02,0x64,0x86,0x14,0x3c,0x00,0x0a,0x0a,0x42,  //000000c0  0<..R.ｰ.d..<...B
    0x01,0x00,0x08,0x02,0x00,0x08,0x66,0x00,0xff,0x78,0xd4,0xbc,0x00,0x01,0x00,0x00,  //000000d0  ......f..xﾔｼ....
    0x61,0x00,0x01,0xb8,0x08,0x00,0x00,0x1b,0x66,0xac,0x60,0x00,0xff,0x64,0x08,0x38,  //000000e0  a..ｸ....fｬ`..d.8
    0x00,0x07,0x09,0xe1,0x66,0x0c,0x48,0xe7,0xc0,0x00,0x61,0x00,0x01,0x46,0x4c,0xdf,  //000000f0  ...畴.H鄲.a..FLﾟ
    0x00,0x03,0x4e,0x75,0x70,0x00,0x78,0x00,0x08,0x01,0x00,0x05,0x67,0x08,0x78,0x09,  //00000100  ..Nup.x.....g.x.
    0x48,0x44,0x38,0x3c,0x00,0x09,0x08,0x01,0x00,0x04,0x67,0x04,0x61,0x00,0x01,0x7c,  //00000110  HD8<......g.a..|
    0x4e,0x75,0x2f,0x01,0x41,0xf8,0x09,0xee,0x10,0x81,0xe0,0x99,0xc2,0x3c,0x00,0x03,  //00000120  Nu/.A....≒卍<..
    0x08,0x02,0x00,0x08,0x67,0x04,0x08,0xc1,0x00,0x02,0x11,0x41,0x00,0x01,0x22,0x1f,  //00000130  ....g..ﾁ...A..".
    0x4e,0x75,0x13,0xfc,0x00,0xff,0x00,0xe8,0x40,0x00,0x13,0xfc,0x00,0x32,0x00,0xe8,  //00000140  Nu.....錙....2.錙
    0x40,0x05,0x60,0x10,0x13,0xfc,0x00,0xff,0x00,0xe8,0x40,0x00,0x13,0xfc,0x00,0xb2,  //00000150   .`......錙....ｲ
    0x00,0xe8,0x40,0x05,0x23,0xc9,0x00,0xe8,0x40,0x0c,0x33,0xc5,0x00,0xe8,0x40,0x0a,  //00000160  .錙.#ﾉ.錙.3ﾅ.錙.
    0x13,0xfc,0x00,0x80,0x00,0xe8,0x40,0x07,0x4e,0x75,0x48,0xe7,0x40,0x60,0x43,0xf9,  //00000170  .....錙.NuH蹇`C.
    0x00,0xe9,0x40,0x01,0x45,0xf9,0x00,0xe9,0x40,0x03,0x40,0xe7,0x00,0x7c,0x07,0x00,  //00000180  .顱.E..顱.@..|..
    0x12,0x11,0x08,0x01,0x00,0x04,0x66,0xf8,0x12,0x11,0x08,0x01,0x00,0x07,0x67,0xf8,  //00000190  ......f.......g.
    0x08,0x01,0x00,0x06,0x66,0xf2,0x14,0x98,0x51,0xc8,0xff,0xee,0x46,0xdf,0x4c,0xdf,  //000001a0  ....f..浪ﾈ.珖ﾟLﾟ
    0x06,0x02,0x4e,0x75,0x10,0x39,0x00,0xe8,0x40,0x00,0x08,0x00,0x00,0x04,0x66,0x0e,  //000001b0  ..Nu.9.錙.....f.
    0x10,0x39,0x00,0xe9,0x40,0x01,0xc0,0x3c,0x00,0x1f,0x66,0xf4,0x4e,0x75,0x10,0x39,  //000001c0  .9.顱.ﾀ<..f.Nu.9
    0x00,0xe8,0x40,0x01,0x4e,0x75,0x10,0x39,0x00,0xe8,0x40,0x00,0x08,0x00,0x00,0x07,  //000001d0  .錙.Nu.9.錙.....
    0x66,0x08,0x13,0xfc,0x00,0x10,0x00,0xe8,0x40,0x07,0x13,0xfc,0x00,0xff,0x00,0xe8,  //000001e0  f......錙......錙
    0x40,0x00,0x4e,0x75,0x30,0x01,0xe0,0x48,0xc0,0xbc,0x00,0x00,0x00,0x03,0xe7,0x40,  //000001f0   .Nu0.潯ﾀｼ....蹇
    0x41,0xf8,0x0c,0x90,0xd1,0xc0,0x20,0x10,0x4e,0x75,0x2f,0x00,0xc0,0xbc,0x00,0x35,  //00000200  A..績ﾀ .Nu/.ﾀｼ.5
    0xff,0x00,0x67,0x2a,0xb8,0x3c,0x00,0x05,0x64,0x24,0x2f,0x38,0x09,0xee,0x2f,0x38,  //00000210  ..g*ｸ<..d$/8../8
    0x09,0xf2,0x3f,0x38,0x09,0xf6,0x61,0x00,0x00,0xc4,0x70,0x64,0x51,0xc8,0xff,0xfe,  //00000220  ..?8..a..ﾄpdQﾈ..
    0x61,0x68,0x31,0xdf,0x09,0xf6,0x21,0xdf,0x09,0xf2,0x21,0xdf,0x09,0xee,0x20,0x1f,  //00000230  ah1ﾟ..!ﾟ..!ﾟ.. .
    0x4e,0x75,0x30,0x01,0xe0,0x48,0x4a,0x00,0x67,0x3c,0xc0,0x3c,0x00,0x03,0x80,0x3c,  //00000240  Nu0.潯J.g<ﾀ<...<
    0x00,0x80,0x08,0xf8,0x00,0x07,0x09,0xe1,0x13,0xc0,0x00,0xe9,0x40,0x07,0x08,0xf8,  //00000250  .........ﾀ.顱...
    0x00,0x06,0x09,0xe1,0x66,0x18,0x31,0xf8,0x09,0xc2,0x09,0xc4,0x61,0x00,0x00,0x90,  //00000260  ...畴.1..ﾂ.ﾄa...
    0x08,0x00,0x00,0x1d,0x66,0x08,0x0c,0x78,0x00,0x64,0x09,0xc4,0x64,0xee,0x08,0xb8,  //00000270  ....f..x.d.ﾄd..ｸ
    0x00,0x06,0x09,0xe1,0x4e,0x75,0x4a,0x38,0x09,0xe1,0x67,0x0c,0x31,0xf8,0x09,0xc2,  //00000280  ...甎uJ8.疊.1..ﾂ
    0x09,0xc4,0x11,0xfc,0x00,0x40,0x09,0xe1,0x4e,0x75,0x61,0x12,0x08,0x00,0x00,0x1b,  //00000290  .ﾄ...@.甎ua.....
    0x66,0x26,0x48,0x40,0x48,0x42,0xb4,0x00,0x67,0x1a,0x48,0x42,0x61,0x3e,0x2f,0x01,  //000002a0  f&H@HBｴ.g.HBa>/.
    0x12,0x3c,0x00,0x0f,0x61,0x00,0xfe,0x6c,0x48,0x42,0x11,0x42,0x00,0x02,0x48,0x42,  //000002b0  .<..a..lHB.B..HB
    0x70,0x02,0x60,0x08,0x48,0x42,0x48,0x40,0x4e,0x75,0x2f,0x01,0x61,0x00,0xfe,0xac,  //000002c0  p.`.HBH@Nu/.a..ｬ
    0x61,0x00,0xfe,0xee,0x22,0x1f,0x30,0x01,0xe0,0x48,0xc0,0xbc,0x00,0x00,0x00,0x03,  //000002d0  a...".0.潯ﾀｼ....
    0xe7,0x40,0x41,0xf8,0x0c,0x90,0xd1,0xc0,0x20,0x10,0x4e,0x75,0x2f,0x01,0x12,0x3c,  //000002e0  蹇A..績ﾀ .Nu/..<
    0x00,0x07,0x61,0x00,0xfe,0x2e,0x70,0x01,0x61,0xd0,0x22,0x1f,0x4e,0x75,0x2f,0x01,  //000002f0  ..a...p.aﾐ".Nu/.
    0x12,0x3c,0x00,0x04,0x61,0x00,0xfe,0x1c,0x22,0x1f,0x70,0x01,0x61,0x00,0xfe,0x6c,  //00000300  .<..a...".p.a..l
    0x10,0x39,0x00,0xe9,0x40,0x01,0xc0,0x3c,0x00,0xd0,0xb0,0x3c,0x00,0xd0,0x66,0xf0,  //00000310  .9.顱.ﾀ<.ﾐｰ<.ﾐf.
    0x70,0x00,0x10,0x39,0x00,0xe9,0x40,0x03,0xe0,0x98,0x4e,0x75,0x53,0x02,0x7e,0x00,  //00000320  p..9.顱.燔NuS.~.
    0x3a,0x02,0xe0,0x5d,0x4a,0x05,0x67,0x04,0x06,0x45,0x08,0x00,0xe0,0x4d,0x48,0x42,  //00000330  :.濔J.g..E..潼HB
    0x02,0x82,0x00,0x00,0x00,0xff,0xe9,0x8a,0xd4,0x45,0x0c,0x42,0x00,0x04,0x65,0x02,  //00000340  ......驫ﾔE.B..e.
    0x53,0x42,0x84,0xfc,0x00,0x12,0x48,0x42,0x3e,0x02,0x8e,0xfc,0x00,0x09,0x48,0x47,  //00000350  SB....HB>.周..HG
    0xe1,0x4f,0xe0,0x8f,0x34,0x07,0x06,0x82,0x03,0x00,0x00,0x0a,0x2a,0x3c,0x00,0x00,  //00000360  甍熄4.......*<..
    0x04,0x00,0x3c,0x3c,0x00,0xff,0x3e,0x3c,0x09,0x28,0x4e,0x75,0x4f,0xfa,0xfc,0x82,  //00000370  ..<<..><.(NuO...
    0x43,0xfa,0xfc,0xa2,0x4d,0xfa,0xfc,0x7a,0x2c,0xb9,0x00,0x00,0x05,0x18,0x23,0xc9,  //00000380  C..｢M..z,ｹ....#ﾉ
    0x00,0x00,0x05,0x18,0x43,0xfa,0x00,0xda,0x4d,0xfa,0xfc,0x6a,0x2c,0xb9,0x00,0x00,  //00000390  ....C..ﾚM..j,ｹ..
    0x05,0x14,0x23,0xc9,0x00,0x00,0x05,0x14,0x43,0xfa,0x01,0x6e,0x4d,0xfa,0xfc,0x5a,  //000003a0  ..#ﾉ....C..nM..Z
    0x2c,0xb9,0x00,0x00,0x05,0x04,0x23,0xc9,0x00,0x00,0x05,0x04,0x24,0x3c,0x03,0x00,  //000003b0  ,ｹ....#ﾉ....$<..
    0x00,0x04,0x20,0x3c,0x00,0x00,0x00,0x8e,0x4e,0x4f,0x12,0x00,0xe1,0x41,0x12,0x3c,  //000003c0  .. <...晒O..瓣.<
    0x00,0x70,0x33,0xc1,0x00,0x00,0x00,0x64,0x26,0x3c,0x00,0x00,0x04,0x00,0x43,0xfa,  //000003d0  .p3ﾁ...d&<....C.
    0x00,0x20,0x61,0x04,0x60,0x00,0x01,0xf0,0x48,0xe7,0x78,0x40,0x70,0x46,0x4e,0x4f,  //000003e0  . a.`...H輾@pFNO
    0x08,0x00,0x00,0x1e,0x66,0x02,0x70,0x00,0x4c,0xdf,0x02,0x1e,0x4e,0x75,0x4e,0x75,  //000003f0  ....f.p.Lﾟ..NuNu
    0xfb,0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000400  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000410  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000420  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000430  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000440  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000450  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000460  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000470  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000480  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000490  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000004f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000500  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000510  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000520  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000530  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000540  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000550  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000560  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000570  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000580  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000590  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000005f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000600  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000610  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000620  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000630  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000640  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000650  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000660  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000670  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000680  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000690  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000006f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000700  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000710  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000720  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000730  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000740  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000750  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000760  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000770  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000780  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000790  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000007f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000800  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000810  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000820  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000830  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000840  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000850  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000860  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000870  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000880  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000890  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000008f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000900  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000910  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000920  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000930  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000940  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000950  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000960  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000970  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000980  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000990  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009a0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009b0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009c0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009d0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009e0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //000009f0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000a90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000aa0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ab0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ac0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ad0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ae0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000af0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b60  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b70  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b80  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000b90  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000ba0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bb0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bc0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bd0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000be0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000bf0  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c00  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c10  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c20  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c30  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c40  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c50  ................
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  //00000c60  ................
    0x08,0x01,0x00,0x0c,0x66,0x08,0x4d,0xfa,0xfb,0x8c,0x2c,0x56,0x4e,0xd6,0x61,0x00,  //00000c70  ....f.M...,VNﾖa.
    0xfc,0x6e,0x48,0xe7,0x4f,0x00,0x61,0x00,0xfe,0xa4,0x61,0x00,0xfc,0x78,0x08,0x00,  //00000c80  .nH躇.a..､a..x..
    0x00,0x1b,0x66,0x30,0xc2,0x3c,0x00,0xc0,0x82,0x3c,0x00,0x05,0x60,0x08,0x30,0x3c,  //00000c90  ..f0ﾂ<.ﾀ.<..`.0<
    0x01,0xac,0x51,0xc8,0xff,0xfe,0x61,0x00,0x01,0x00,0x08,0x00,0x00,0x1e,0x67,0x2c,  //00000ca0  .ｬQﾈ..a.......g,
    0x08,0x00,0x00,0x1b,0x66,0x0e,0x08,0x00,0x00,0x11,0x66,0x08,0x61,0x00,0xfd,0x4c,  //00000cb0  ....f.....f.a..L
    0x51,0xcc,0xff,0xe4,0x4c,0xdf,0x00,0xf2,0x4a,0x38,0x09,0xe1,0x67,0x0c,0x31,0xf8,  //00000cc0  Qﾌ.膈ﾟ..J8.疊.1.
    0x09,0xc2,0x09,0xc4,0x11,0xfc,0x00,0x40,0x09,0xe1,0x4e,0x75,0x08,0x00,0x00,0x1f,  //00000cd0  .ﾂ.ﾄ...@.甎u....
    0x66,0xe2,0xd3,0xc5,0x96,0x85,0x63,0xdc,0x20,0x04,0x48,0x40,0x38,0x00,0x30,0x3c,  //00000ce0  f簽ﾅ妹cﾜ .H@8.0<
    0x00,0x12,0x52,0x02,0xb0,0x02,0x64,0xae,0x14,0x3c,0x00,0x0a,0x0a,0x42,0x01,0x00,  //00000cf0  ..R.ｰ.dｮ.<...B..
    0x08,0x02,0x00,0x08,0x66,0x98,0xd4,0xbc,0x00,0x01,0x00,0x00,0x61,0x00,0xfd,0x8c,  //00000d00  ....f儘ｼ....a...
    0x08,0x00,0x00,0x1b,0x66,0xae,0x60,0x8e,0x08,0x01,0x00,0x0c,0x66,0x08,0x4d,0xfa,  //00000d10  ....fｮ`.....f.M.
    0xfa,0xe0,0x2c,0x56,0x4e,0xd6,0x61,0x00,0xfb,0xc6,0x48,0xe7,0x4f,0x00,0x61,0x00,  //00000d20  ..,VNﾖa..ﾆH躇.a.
    0xfd,0xfc,0x61,0x00,0xfb,0xd0,0x08,0x00,0x00,0x1b,0x66,0x24,0xc2,0x3c,0x00,0xc0,  //00000d30  ..a..ﾐ....f$ﾂ<.ﾀ
    0x82,0x3c,0x00,0x11,0x61,0x62,0x08,0x00,0x00,0x0a,0x66,0x14,0x08,0x00,0x00,0x1e,  //00000d40  .<..ab....f.....
    0x67,0x16,0x08,0x00,0x00,0x1b,0x66,0x08,0x61,0x00,0xfc,0xb0,0x51,0xcc,0xff,0xe6,  //00000d50  g.....f.a..ｰQﾌ.覘
    0x4c,0xdf,0x00,0xf2,0x60,0x00,0xfb,0x34,0x08,0x00,0x00,0x1f,0x66,0xf2,0xd3,0xc5,  //00000d60   ﾟ..`..4....f.ﾓﾅ
    0x96,0x85,0x63,0xec,0x20,0x04,0x48,0x40,0x38,0x00,0x30,0x3c,0x00,0x12,0x30,0x3c,  //00000d70  妹c. .H@8.0<..0<
    0x00,0x12,0x52,0x02,0xb0,0x02,0x64,0xbc,0x14,0x3c,0x00,0x0a,0x0a,0x42,0x01,0x00,  //00000d80  ..R.ｰ.dｼ.<...B..
    0x08,0x02,0x00,0x08,0x66,0xae,0xd4,0xbc,0x00,0x01,0x00,0x00,0x61,0x00,0xfc,0xfc,  //00000d90  ....fｮﾔｼ....a...
    0x08,0x00,0x00,0x1b,0x66,0xba,0x60,0x9c,0x61,0x00,0xfb,0x78,0xe1,0x9a,0x54,0x88,  //00000da0  ....fｺ`彗..x癩T.
    0x20,0xc2,0xe0,0x9a,0x10,0xc2,0x10,0xc7,0x10,0x86,0x61,0x00,0xfb,0x86,0x41,0xf8,  //00000db0   ﾂ燠.ﾂ.ﾇ..a...A.
    0x09,0xee,0x70,0x08,0x61,0x00,0xfb,0xb4,0x61,0x00,0xfb,0xea,0x61,0x00,0xfc,0x08,  //00000dc0  .綠.a..ｴa..鸞...
    0x61,0x00,0xfc,0x22,0x4e,0x75,0x43,0xfa,0x01,0x8c,0x61,0x00,0x01,0x76,0x24,0x3c,  //00000dd0  a.."NuC..径..v$<
    0x03,0x00,0x00,0x06,0x32,0x39,0x00,0x00,0x00,0x64,0x26,0x3c,0x00,0x00,0x04,0x00,  //00000de0  ....29...d&<....
    0x43,0xf8,0x28,0x00,0x61,0x00,0xfd,0xf2,0x4a,0x80,0x66,0x00,0x01,0x20,0x43,0xf8,  //00000df0  C.(.a...J.f.. C.
    0x28,0x00,0x49,0xfa,0x01,0x54,0x78,0x1f,0x24,0x49,0x26,0x4c,0x7a,0x0a,0x10,0x1a,  //00000e00  (.I..Tx.$I&Lz...
    0x80,0x3c,0x00,0x20,0xb0,0x1b,0x66,0x06,0x51,0xcd,0xff,0xf4,0x60,0x0c,0x43,0xe9,  //00000e10  .<. ｰ.f.Qﾍ..`.C.
    0x00,0x20,0x51,0xcc,0xff,0xe4,0x66,0x00,0x00,0xf4,0x30,0x29,0x00,0x1a,0xe1,0x58,  //00000e20  . Qﾌ.臟...0)..畊
    0x55,0x40,0xd0,0x7c,0x00,0x0b,0x34,0x00,0xc4,0x7c,0x00,0x07,0x52,0x02,0xe8,0x48,  //00000e30  U@ﾐ|..4.ﾄ|..R.鍠
    0x64,0x04,0x84,0x7c,0x01,0x00,0x48,0x42,0x34,0x3c,0x03,0x00,0x14,0x00,0x48,0x42,  //00000e40  d.л..HB4<....HB
    0x26,0x29,0x00,0x1c,0xe1,0x5b,0x48,0x43,0xe1,0x5b,0x43,0xf8,0x67,0xc0,0x61,0x00,  //00000e50  &)..畆HC畆C.gﾀa.
    0xfd,0x88,0x0c,0x51,0x48,0x55,0x66,0x00,0x00,0xb4,0x4b,0xf8,0x68,0x00,0x49,0xfa,  //00000e60  ...QHUf..ｴK.h.I.
    0x00,0x4c,0x22,0x4d,0x43,0xf1,0x38,0xc0,0x2c,0x3c,0x00,0x04,0x00,0x00,0x0c,0x69,  //00000e70  .L"MC.8ﾀ,<.....i
    0x4e,0xd4,0xff,0xd2,0x66,0x36,0x0c,0xad,0x4c,0x5a,0x58,0x20,0x00,0x04,0x66,0x16,  //00000e80  Nﾔ.ﾒf6.ｭLZX ..f.
    0x2b,0x46,0x00,0x04,0x2b,0x4d,0x00,0x08,0x42,0xad,0x00,0x20,0x51,0xf9,0x00,0x00,  //00000e90  +F..+M..Bｭ. Q...
    0x07,0x9e,0x4e,0xed,0x00,0x02,0x0c,0x6d,0x4e,0xec,0x00,0x1a,0x66,0x0e,0x0c,0x6d,  //00000ea0  .朦....mN...f..m
    0x4e,0xea,0x00,0x2a,0x66,0x06,0x43,0xfa,0x01,0x1f,0x60,0x64,0x10,0x3c,0x00,0xc0,  //00000eb0  N..*f.C...`d.<.ﾀ
    0x41,0xf8,0x68,0x00,0x36,0x3c,0xff,0xff,0xb0,0x18,0x67,0x26,0x51,0xcb,0xff,0xfa,  //00000ec0  A.h.6<..ｰ.g&Qﾋ..
    0x43,0xf8,0x68,0x00,0x4a,0x39,0x00,0x00,0x07,0x9e,0x67,0x14,0x41,0xf8,0x67,0xcc,  //00000ed0  C.h.J9...枷.A.gﾌ
    0x24,0x18,0xd4,0x98,0x22,0x10,0xd1,0xc2,0x53,0x81,0x65,0x04,0x42,0x18,0x60,0xf8,  //00000ee0  $.ﾔ.".ﾑﾂS‘.B.`.
    0x4e,0xd1,0x0c,0x10,0x00,0x04,0x66,0xd0,0x52,0x88,0x0c,0x10,0x00,0xd0,0x66,0xc8,  //00000ef0  Nﾑ....fﾐR....ﾐfﾈ
    0x52,0x88,0x0c,0x10,0x00,0xfe,0x66,0xc0,0x52,0x88,0x0c,0x10,0x00,0x02,0x66,0xb8,  //00000f00  R.....fﾀR.....fｸ
    0x57,0x88,0x30,0xfc,0x05,0xa1,0x10,0xbc,0x00,0xfb,0x60,0xac,0x43,0xfa,0x00,0x92,  //00000f10  W.0..｡.ｼ..`ｬC...
    0x2f,0x09,0x43,0xfa,0x00,0x47,0x61,0x2a,0x43,0xfa,0x00,0x46,0x61,0x24,0x43,0xfa,  //00000f20  /.C..Ga*C..Fa$C.
    0x00,0x52,0x61,0x1e,0x43,0xfa,0x00,0x43,0x61,0x18,0x43,0xfa,0x00,0x46,0x61,0x12,  //00000f30  .Ra.C..Ca.C..Fa.
    0x22,0x5f,0x61,0x0e,0x32,0x39,0x00,0x00,0x00,0x64,0x70,0x4f,0x4e,0x4f,0x70,0xfe,  //00000f40  "_a.29...dpONOp.
    0x4e,0x4f,0x70,0x21,0x4e,0x4f,0x4e,0x75,0x68,0x75,0x6d,0x61,0x6e,0x20,0x20,0x20,  //00000f50  NOp!NONuhuman   
    0x73,0x79,0x73,0x00,0x39,0x53,0x43,0x49,0x50,0x4c,0x00,0x1b,0x5b,0x34,0x37,0x6d,  //00000f60  sys.9SCIPL..[47m
    0x1b,0x5b,0x31,0x33,0x3b,0x32,0x36,0x48,0x00,0x1b,0x5b,0x31,0x34,0x3b,0x32,0x36,  //00000f70  .[13;26H..[14;26
    0x48,0x00,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //00000f80  H.              
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,  //00000f90                  
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x00,  //00000fa0                 .
    0x1b,0x5b,0x31,0x34,0x3b,0x33,0x34,0x48,0x48,0x75,0x6d,0x61,0x6e,0x2e,0x73,0x79,  //00000fb0  .[14;34HHuman.sy
    0x73,0x20,0x82,0xcc,0x93,0xc7,0x82,0xdd,0x8d,0x9e,0x82,0xdd,0x83,0x47,0x83,0x89,  //00000fc0  s の読み込みエラ
    0x81,0x5b,0x82,0xc5,0x82,0xb7,0x00,0x1b,0x5b,0x31,0x34,0x3b,0x33,0x34,0x48,0x4c,  //00000fd0  ーです..[14;34HL
    0x5a,0x58,0x2e,0x58,0x20,0x82,0xcc,0x83,0x6f,0x81,0x5b,0x83,0x57,0x83,0x87,0x83,  //00000fe0  ZX.X のバージョン
    0x93,0x82,0xaa,0x8c,0xc3,0x82,0xb7,0x82,0xac,0x82,0xdc,0x82,0xb7,0x00,0x00,0x00,  //00000ff0   が古すぎます...
  };
*/
  //  perl misc/itob.pl xeij/FDMedia.java FDM_2HS_IPL
  public static final byte[] FDM_2HS_IPL = "`\369SCFMT IPL v1.02\4\0\1\3\0\1\0\300\5\240\373\1\220p`\0\3Z\b\1\0\ff\bM\372\377\324,VN\326a\0\0\272H\347O\0a\0\2\360a\0\0\304\b\0\0\33fN\302<\0\300\202<\0\6a\0\0\320\341\232T\210 \302\340\232\20\302\20\307\20\206a\0\0\360A\370\t\356p\ba\0\1\fa\0\1Ba\0\1`a\0\1z\b\0\0\16f\f\b\0\0\36g&\b\0\0\33f\ba\0\1zQ\314\377\274L\337\0\362J8\t\341g\f1\370\t\302\t\304\21\374\0@\t\341Nu\b\0\0\37f\342\323\305\226\205c\334 \4H@8\0000<\0\22R\2\260\2d\206\24<\0\n\nB\1\0\b\2\0\bf\0\377x\324\274\0\1\0\0a\0\1\270\b\0\0\33f\254`\0\377d\b8\0\7\t\341f\fH\347\300\0a\0\1FL\337\0\3Nup\0x\0\b\1\0\5g\bx\tHD8<\0\t\b\1\0\4g\4a\0\1|Nu/\1A\370\t\356\20\201\340\231\302<\0\3\b\2\0\bg\4\b\301\0\2\21A\0\1\"\37Nu\23\374\0\377\0\350@\0\23\374\0002\0\350@\5`\20\23\374\0\377\0\350@\0\23\374\0\262\0\350@\5#\311\0\350@\f3\305\0\350@\n\23\374\0\200\0\350@\7NuH\347@`C\371\0\351@\1E\371\0\351@\3@\347\0|\7\0\22\21\b\1\0\4f\370\22\21\b\1\0\7g\370\b\1\0\6f\362\24\230Q\310\377\356F\337L\337\6\2Nu\209\0\350@\0\b\0\0\4f\16\209\0\351@\1\300<\0\37f\364Nu\209\0\350@\1Nu\209\0\350@\0\b\0\0\7f\b\23\374\0\20\0\350@\7\23\374\0\377\0\350@\0Nu0\1\340H\300\274\0\0\0\3\347@A\370\f\220\321\300 \20Nu/\0\300\274\0005\377\0g*\270<\0\5d$/8\t\356/8\t\362?8\t\366a\0\0\304pdQ\310\377\376ah1\337\t\366!\337\t\362!\337\t\356 \37Nu0\1\340HJ\0g<\300<\0\3\200<\0\200\b\370\0\7\t\341\23\300\0\351@\7\b\370\0\6\t\341f\0301\370\t\302\t\304a\0\0\220\b\0\0\35f\b\fx\0d\t\304d\356\b\270\0\6\t\341NuJ8\t\341g\f1\370\t\302\t\304\21\374\0@\t\341Nua\22\b\0\0\33f&H@HB\264\0g\32HBa>/\1\22<\0\17a\0\376lHB\21B\0\2HBp\2`\bHBH@Nu/\1a\0\376\254a\0\376\356\"\0370\1\340H\300\274\0\0\0\3\347@A\370\f\220\321\300 \20Nu/\1\22<\0\7a\0\376.p\1a\320\"\37Nu/\1\22<\0\4a\0\376\34\"\37p\1a\0\376l\209\0\351@\1\300<\0\320\260<\0\320f\360p\0\209\0\351@\3\340\230NuS\2~\0:\2\340]J\5g\4\6E\b\0\340MHB\2\202\0\0\0\377\351\212\324E\fB\0\4e\2SB\204\374\0\22HB>\2\216\374\0\tHG\341O\340\2174\7\6\202\3\0\0\n*<\0\0\4\0<<\0\377><\t(NuO\372\374\202C\372\374\242M\372\374z,\271\0\0\5\30#\311\0\0\5\30C\372\0\332M\372\374j,\271\0\0\5\24#\311\0\0\5\24C\372\1nM\372\374Z,\271\0\0\5\4#\311\0\0\5\4$<\3\0\0\4 <\0\0\0\216NO\22\0\341A\22<\0p3\301\0\0\0d&<\0\0\4\0C\372\0 a\4`\0\1\360H\347x@pFNO\b\0\0\36f\2p\0L\337\2\36NuNu\373\377\377\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\b\1\0\ff\bM\372\373\214,VN\326a\0\374nH\347O\0a\0\376\244a\0\374x\b\0\0\33f0\302<\0\300\202<\0\5`\b0<\1\254Q\310\377\376a\0\1\0\b\0\0\36g,\b\0\0\33f\16\b\0\0\21f\ba\0\375LQ\314\377\344L\337\0\362J8\t\341g\f1\370\t\302\t\304\21\374\0@\t\341Nu\b\0\0\37f\342\323\305\226\205c\334 \4H@8\0000<\0\22R\2\260\2d\256\24<\0\n\nB\1\0\b\2\0\bf\230\324\274\0\1\0\0a\0\375\214\b\0\0\33f\256`\216\b\1\0\ff\bM\372\372\340,VN\326a\0\373\306H\347O\0a\0\375\374a\0\373\320\b\0\0\33f$\302<\0\300\202<\0\21ab\b\0\0\nf\24\b\0\0\36g\26\b\0\0\33f\ba\0\374\260Q\314\377\346L\337\0\362`\0\3734\b\0\0\37f\362\323\305\226\205c\354 \4H@8\0000<\0\0220<\0\22R\2\260\2d\274\24<\0\n\nB\1\0\b\2\0\bf\256\324\274\0\1\0\0a\0\374\374\b\0\0\33f\272`\234a\0\373x\341\232T\210 \302\340\232\20\302\20\307\20\206a\0\373\206A\370\t\356p\ba\0\373\264a\0\373\352a\0\374\ba\0\374\"NuC\372\1\214a\0\1v$<\3\0\0\00629\0\0\0d&<\0\0\4\0C\370(\0a\0\375\362J\200f\0\1 C\370(\0I\372\1Tx\37$I&Lz\n\20\32\200<\0 \260\33f\6Q\315\377\364`\fC\351\0 Q\314\377\344f\0\0\3640)\0\32\341XU@\320|\0\0134\0\304|\0\7R\2\350Hd\4\204|\1\0HB4<\3\0\24\0HB&)\0\34\341[HC\341[C\370g\300a\0\375\210\fQHUf\0\0\264K\370h\0I\372\0L\"MC\3618\300,<\0\4\0\0\fiN\324\377\322f6\f\255LZX \0\4f\26+F\0\4+M\0\bB\255\0 Q\371\0\0\7\236N\355\0\2\fmN\354\0\32f\16\fmN\352\0*f\6C\372\1\37`d\20<\0\300A\370h\0006<\377\377\260\30g&Q\313\377\372C\370h\0J9\0\0\7\236g\24A\370g\314$\30\324\230\"\20\321\302S\201e\4B\30`\370N\321\f\20\0\4f\320R\210\f\20\0\320f\310R\210\f\20\0\376f\300R\210\f\20\0\2f\270W\2100\374\5\241\20\274\0\373`\254C\372\0\222/\tC\372\0Ga*C\372\0Fa$C\372\0Ra\36C\372\0Ca\30C\372\0Fa\22\"_a\01629\0\0\0dpONOp\376NOp!NONuhuman   sys\09SCIPL\0\33[47m\33[13;26H\0\33[14;26H\0                                             \0\33[14;34HHuman.sys \202\314\223\307\202\335\215\236\202\335\203G\203\211\201[\202\305\202\267\0\33[14;34HLZX.X \202\314\203o\201[\203W\203\207\203\223\202\252\214\303\202\267\202\254\202\334\202\267\0\0\0".getBytes (XEiJ.ISO_8859_1);

  //----------------------------------------------------------------------------------------
  //フロッピーディスクBPB
  //  フロッピーディスクIPLに埋め込まれる
/*
  public static final int[] FDM_DISK_BPB = {
    //  perl -e "do'sjdump.pl';$p=0;$m=1;$o=0xb6e;$l=160;open IN,'HUMAN302.XDF'or die;binmode IN;seek IN,1024*592,0;read IN,$b,64;seek IN,1024*592+vec($b,15,32)+32*$m,0;read IN,$b,32;seek IN,1024*592+vec($b,7,32)+64+$o,0;read IN,$b,$l;close IN;sjdumpcode($b,0,$l,$p)"
    //bytesPerSector
    //  |sectorsPerCluster               partitionSectors    sectorsPerTrack
    //  |       |reservedSectors rootEntries    |       fatSectors  |
    //  |       |      |   fatCount   |         |    fatID   |      | tracksPerCylinder
    //--+---- --+- ----+---- --+- ----+---- ----+---- --+- --+- ----+---- ----+----
    0x00,0x04,0x01,0x01,0x00,0x02,0xc0,0x00,0xd0,0x04,0xfe,0x02,0x00,0x08,0x00,0x02,  //00000000  ......ﾀ.ﾐ.......
    0x00,0x00,0x00,0x00,0x04,0x00,0x03,0x00,0x00,0x06,0x00,0x08,0x00,0x1f,0x00,0x09,  //00000010  ................
    //+- --+- ---------+--------- ---------+--------- ----+---- ----+---- ----+----
    //PDA  |     bytesPerSector      rootPosition  sectorsPerTrack  | dataStartSector-2
    // ejectFlag                                            entriesPerSector-1
    0x00,0x02,0x01,0x01,0x00,0x02,0xe0,0x00,0x60,0x09,0xf9,0x07,0x00,0x0f,0x00,0x02,  //00000020  ........`.......
    0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x01,0x01,0x00,0x0f,0x00,0x0f,0x00,0x1b,  //00000030  ................
    0x00,0x02,0x02,0x01,0x00,0x02,0x70,0x00,0x00,0x05,0xfb,0x02,0x00,0x08,0x00,0x02,  //00000040  ......p.........
    0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x00,0x06,0x00,0x08,0x00,0x0f,0x00,0x0a,  //00000050  ................
    0x00,0x02,0x02,0x01,0x00,0x02,0x70,0x00,0xa0,0x05,0xf9,0x03,0x00,0x09,0x00,0x02,  //00000060  ......p.........
    0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x00,0x08,0x00,0x09,0x00,0x0f,0x00,0x0c,  //00000070  ................
    0x00,0x02,0x01,0x01,0x00,0x02,0xe0,0x00,0x40,0x0b,0xf0,0x09,0x00,0x12,0x00,0x02,  //00000080  ........@.......
    0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x01,0x02,0x00,0x12,0x00,0x0f,0x00,0x1f,  //00000090  ................
  };
*/
  //  perl misc/itob.pl xeij/FDMedia.java FDM_DISK_BPB
  public static final byte[] FDM_DISK_BPB = "\0\4\1\1\0\2\300\0\320\4\376\2\0\b\0\2\0\0\0\0\4\0\3\0\0\6\0\b\0\37\0\t\0\2\1\1\0\2\340\0`\t\371\7\0\17\0\2\0\0\0\0\2\0\2\0\1\1\0\17\0\17\0\33\0\2\2\1\0\2p\0\0\5\373\2\0\b\0\2\0\0\0\0\2\0\2\0\0\6\0\b\0\17\0\n\0\2\2\1\0\2p\0\240\5\371\3\0\t\0\2\0\0\0\0\2\0\2\0\0\b\0\t\0\17\0\f\0\2\1\1\0\2\340\0@\13\360\t\0\22\0\2\0\0\0\0\2\0\2\0\1\2\0\22\0\17\0\37".getBytes (XEiJ.ISO_8859_1);
  //MS-DOSの2HDディスクのセクタ0の先頭
  //                                                             sectorsPerCluster
  //    JMP(*);NOP                  "NEC  5.0"             bytesPerSector|reservedSectors
  //  -------+------ -------------------+------------------- ----+---- --+- ----+----
  //  0xeb,0x3c,0x90,0x4e,0x45,0x43,0x20,0x20,0x35,0x2e,0x30,0x00,0x04,0x01,0x01,0x00,
  //  0x02,0xc0,0x00,0xd0,0x04,0xfe,0x02,0x00,0x08,0x00,0x02,0x00,0x00,0x00,0x00,0x00,
  //  --+- ----+---- ----+---- --+- ----+---- ----+---- ----+---- ----+----
  //    | rootEntries    |    fatID fatSectors    | tracksPerCylinder |
  //  fatCount    partitionSectors         sectorsPerTrack       hiddenSectors

}  //class FDMedia



