//========================================================================================
//  MC68EC030.java
//    en:MC68EC030 core
//    ja:MC68EC030コア
//  Copyright (C) 2003-2025 Makoto Kamada
//
//  This file is part of the XEiJ (X68000 Emulator in Java).
//  You can use, modify and redistribute the XEiJ if the conditions are met.
//  Read the XEiJ License for more details.
//  https://stdkmd.net/xeij/
//========================================================================================

package xeij;

import java.lang.*;  //Boolean,Character,Class,Comparable,Double,Exception,Float,IllegalArgumentException,Integer,Long,Math,Number,Object,Runnable,SecurityException,String,StringBuilder,System

public class MC68EC030 {

  //ゼロ除算のときの未定義フラグ
  //  MC68030はゼロ除算のときオペランド以外の要因でZとVが変化する
  //  VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
  //  参考
  //    https://twitter.com/moveccr/status/814032098692513792
  //    https://twitter.com/isaki68k/status/814036909030682624
  public static final boolean M30_DIV_ZERO_V_FLAG = true;  //true=ゼロ除算のVフラグの再現を試みる
  public static boolean m30DivZeroVFlag;

  public static void mpuCore () {

    //例外ループ
    //  別のメソッドで検出された例外を命令ループの外側でcatchすることで命令ループを高速化する
  errorLoop:
    while (XEiJ.mpuClockTime < XEiJ.mpuClockLimit) {
      try {
        //命令ループ
        while (XEiJ.mpuClockTime < XEiJ.mpuClockLimit) {
          int t;
          //命令を実行する
          XEiJ.mpuTraceFlag = XEiJ.regSRT1;  //命令実行前のsrT1
          XEiJ.mpuCycleCount = 0;  //第1オペコードからROMのアクセスウエイトを有効にする。命令のサイクル数はすべてXEiJ.mpuCycleCount+=～で加えること
          XEiJ.regPC0 = t = XEiJ.regPC;  //命令の先頭アドレス
          XEiJ.regPC = t + 2;
          XEiJ.regOC = (InstructionBreakPoint.IBP_ON ? InstructionBreakPoint.ibpOp1MemoryMap : DataBreakPoint.DBP_ON ? XEiJ.regSRS != 0 ? XEiJ.busSuperMap : XEiJ.busUserMap : XEiJ.busMemoryMap)[t >>> XEiJ.BUS_PAGE_BITS].mmdRwz (t);  //第1オペコード。必ずゼロ拡張すること。pcに奇数が入っていることはないのでアドレスエラーのチェックを省略する

          //命令の処理
          //  第1オペコードの上位10ビットで分岐する
        irpSwitch:
          switch (XEiJ.regOC >>> 6) {  //第1オペコードの上位10ビット。XEiJ.regOCはゼロ拡張されているので0b1111_111_111&を省略

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ORI.B #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_000_mmm_rrr-{data}
            //OR.B #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_000_mmm_rrr-{data}  [ORI.B #<data>,<ea>]
            //ORI.B #<data>,CCR                               |-|012346|-|*****|*****|          |0000_000_000_111_100-{data}
          case 0b0000_000_000:
            irpOriByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ORI.W #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_001_mmm_rrr-{data}
            //OR.W #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_001_mmm_rrr-{data}  [ORI.W #<data>,<ea>]
            //ORI.W #<data>,SR                                |-|012346|P|*****|*****|          |0000_000_001_111_100-{data}
          case 0b0000_000_001:
            irpOriWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ORI.L #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_010_mmm_rrr-{data}
            //OR.L #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_010_mmm_rrr-{data}  [ORI.L #<data>,<ea>]
          case 0b0000_000_010:
            irpOriLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BITREV.L Dr                                     |-|------|-|-----|-----|D         |0000_000_011_000_rrr (ISA_C)
            //CMP2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn000000000000
            //CHK2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn100000000000
          case 0b0000_000_011:
            irpCmp2Chk2Byte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BTST.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_100_000_rrr
            //MOVEP.W (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_100_001_rrr-{data}
            //BTST.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZPI|0000_qqq_100_mmm_rrr
          case 0b0000_000_100:
          case 0b0000_001_100:
          case 0b0000_010_100:
          case 0b0000_011_100:
          case 0b0000_100_100:
          case 0b0000_101_100:
          case 0b0000_110_100:
          case 0b0000_111_100:
            irpBtstReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCHG.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_101_000_rrr
            //MOVEP.L (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_101_001_rrr-{data}
            //BCHG.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_101_mmm_rrr
          case 0b0000_000_101:
          case 0b0000_001_101:
          case 0b0000_010_101:
          case 0b0000_011_101:
          case 0b0000_100_101:
          case 0b0000_101_101:
          case 0b0000_110_101:
          case 0b0000_111_101:
            irpBchgReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCLR.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_110_000_rrr
            //MOVEP.W Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_110_001_rrr-{data}
            //BCLR.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_110_mmm_rrr
          case 0b0000_000_110:
          case 0b0000_001_110:
          case 0b0000_010_110:
          case 0b0000_011_110:
          case 0b0000_100_110:
          case 0b0000_101_110:
          case 0b0000_110_110:
          case 0b0000_111_110:
            irpBclrReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSET.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_111_000_rrr
            //MOVEP.L Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_111_001_rrr-{data}
            //BSET.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_111_mmm_rrr
          case 0b0000_000_111:
          case 0b0000_001_111:
          case 0b0000_010_111:
          case 0b0000_011_111:
          case 0b0000_100_111:
          case 0b0000_101_111:
          case 0b0000_110_111:
          case 0b0000_111_111:
            irpBsetReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ANDI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_000_mmm_rrr-{data}
            //AND.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_000_mmm_rrr-{data}  [ANDI.B #<data>,<ea>]
            //ANDI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_001_000_111_100-{data}
          case 0b0000_001_000:
            irpAndiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ANDI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_001_mmm_rrr-{data}
            //AND.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_001_mmm_rrr-{data}  [ANDI.W #<data>,<ea>]
            //ANDI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_001_001_111_100-{data}
          case 0b0000_001_001:
            irpAndiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ANDI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_010_mmm_rrr-{data}
            //AND.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_010_mmm_rrr-{data}  [ANDI.L #<data>,<ea>]
          case 0b0000_001_010:
            irpAndiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BYTEREV.L Dr                                    |-|------|-|-----|-----|D         |0000_001_011_000_rrr (ISA_C)
            //CMP2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn000000000000
            //CHK2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn100000000000
          case 0b0000_001_011:
            irpCmp2Chk2Word ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_000_mmm_rrr-{data}
            //SUB.B #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_000_mmm_rrr-{data}  [SUBI.B #<data>,<ea>]
          case 0b0000_010_000:
            irpSubiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_001_mmm_rrr-{data}
            //SUB.W #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_001_mmm_rrr-{data}  [SUBI.W #<data>,<ea>]
          case 0b0000_010_001:
            irpSubiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_010_mmm_rrr-{data}
            //SUB.L #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_010_mmm_rrr-{data}  [SUBI.L #<data>,<ea>]
          case 0b0000_010_010:
            irpSubiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FF1.L Dr                                        |-|------|-|-UUUU|-**00|D         |0000_010_011_000_rrr (ISA_C)
            //CMP2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn000000000000
            //CHK2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn100000000000
          case 0b0000_010_011:
            irpCmp2Chk2Long ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_000_mmm_rrr-{data}
          case 0b0000_011_000:
            irpAddiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_001_mmm_rrr-{data}
          case 0b0000_011_001:
            irpAddiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_010_mmm_rrr-{data}
          case 0b0000_011_010:
            irpAddiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BTST.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_000_000_rrr-{data}
            //BTST.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZP |0000_100_000_mmm_rrr-{data}
          case 0b0000_100_000:
            irpBtstImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCHG.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_001_000_rrr-{data}
            //BCHG.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_001_mmm_rrr-{data}
          case 0b0000_100_001:
            irpBchgImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCLR.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_010_000_rrr-{data}
            //BCLR.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_010_mmm_rrr-{data}
          case 0b0000_100_010:
            irpBclrImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSET.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_011_000_rrr-{data}
            //BSET.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_011_mmm_rrr-{data}
          case 0b0000_100_011:
            irpBsetImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EORI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}
            //EOR.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}  [EORI.B #<data>,<ea>]
            //EORI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_101_000_111_100-{data}
          case 0b0000_101_000:
            irpEoriByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EORI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}
            //EOR.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}  [EORI.W #<data>,<ea>]
            //EORI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_101_001_111_100-{data}
          case 0b0000_101_001:
            irpEoriWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EORI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}
            //EOR.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}  [EORI.L #<data>,<ea>]
          case 0b0000_101_010:
            irpEoriLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CAS.B Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_101_011_mmm_rrr-0000000uuu000ccc
          case 0b0000_101_011:
            irpCasByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPI.B #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_000_mmm_rrr-{data}
            //CMP.B #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_000_mmm_rrr-{data}  [CMPI.B #<data>,<ea>]
          case 0b0000_110_000:
            irpCmpiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPI.W #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_001_mmm_rrr-{data}
            //CMP.W #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_001_mmm_rrr-{data}  [CMPI.W #<data>,<ea>]
          case 0b0000_110_001:
            irpCmpiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPI.L #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_010_mmm_rrr-{data}
            //CMP.L #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_010_mmm_rrr-{data}  [CMPI.L #<data>,<ea>]
          case 0b0000_110_010:
            irpCmpiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CAS.W Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_110_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
            //CAS2.W Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_110_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
          case 0b0000_110_011:
            irpCasWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVES.B <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn000000000000
            //MOVES.B Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn100000000000
          case 0b0000_111_000:
            irpMovesByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVES.W <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn000000000000
            //MOVES.W Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn100000000000
          case 0b0000_111_001:
            irpMovesWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVES.L <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn000000000000
            //MOVES.L Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn100000000000
          case 0b0000_111_010:
            irpMovesLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CAS.L Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_111_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
            //CAS2.L Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_111_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
          case 0b0000_111_011:
            irpCasLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,Dq                                  |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_000_mmm_rrr
          case 0b0001_000_000:
          case 0b0001_001_000:
          case 0b0001_010_000:
          case 0b0001_011_000:
          case 0b0001_100_000:
          case 0b0001_101_000:
          case 0b0001_110_000:
          case 0b0001_111_000:
            irpMoveToDRByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_010_mmm_rrr
          case 0b0001_000_010:
          case 0b0001_001_010:
          case 0b0001_010_010:
          case 0b0001_011_010:
          case 0b0001_100_010:
          case 0b0001_101_010:
          case 0b0001_110_010:
          case 0b0001_111_010:
            irpMoveToMMByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_011_mmm_rrr
          case 0b0001_000_011:
          case 0b0001_001_011:
          case 0b0001_010_011:
          case 0b0001_011_011:
          case 0b0001_100_011:
          case 0b0001_101_011:
          case 0b0001_110_011:
          case 0b0001_111_011:
            irpMoveToMPByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_100_mmm_rrr
          case 0b0001_000_100:
          case 0b0001_001_100:
          case 0b0001_010_100:
          case 0b0001_011_100:
          case 0b0001_100_100:
          case 0b0001_101_100:
          case 0b0001_110_100:
          case 0b0001_111_100:
            irpMoveToMNByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_101_mmm_rrr
          case 0b0001_000_101:
          case 0b0001_001_101:
          case 0b0001_010_101:
          case 0b0001_011_101:
          case 0b0001_100_101:
          case 0b0001_101_101:
          case 0b0001_110_101:
          case 0b0001_111_101:
            irpMoveToMWByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_110_mmm_rrr
          case 0b0001_000_110:
          case 0b0001_001_110:
          case 0b0001_010_110:
          case 0b0001_011_110:
          case 0b0001_100_110:
          case 0b0001_101_110:
          case 0b0001_110_110:
          case 0b0001_111_110:
            irpMoveToMXByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_000_111_mmm_rrr
          case 0b0001_000_111:
            irpMoveToZWByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_001_111_mmm_rrr
          case 0b0001_001_111:
            irpMoveToZLByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_000_mmm_rrr
          case 0b0010_000_000:
          case 0b0010_001_000:
          case 0b0010_010_000:
          case 0b0010_011_000:
          case 0b0010_100_000:
          case 0b0010_101_000:
          case 0b0010_110_000:
          case 0b0010_111_000:
            irpMoveToDRLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVEA.L <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr
            //MOVE.L <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr [MOVEA.L <ea>,Aq]
          case 0b0010_000_001:
          case 0b0010_001_001:
          case 0b0010_010_001:
          case 0b0010_011_001:
          case 0b0010_100_001:
          case 0b0010_101_001:
          case 0b0010_110_001:
          case 0b0010_111_001:
            irpMoveaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_010_mmm_rrr
          case 0b0010_000_010:
          case 0b0010_001_010:
          case 0b0010_010_010:
          case 0b0010_011_010:
          case 0b0010_100_010:
          case 0b0010_101_010:
          case 0b0010_110_010:
          case 0b0010_111_010:
            irpMoveToMMLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_011_mmm_rrr
          case 0b0010_000_011:
          case 0b0010_001_011:
          case 0b0010_010_011:
          case 0b0010_011_011:
          case 0b0010_100_011:
          case 0b0010_101_011:
          case 0b0010_110_011:
          case 0b0010_111_011:
            irpMoveToMPLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_100_mmm_rrr
          case 0b0010_000_100:
          case 0b0010_001_100:
          case 0b0010_010_100:
          case 0b0010_011_100:
          case 0b0010_100_100:
          case 0b0010_101_100:
          case 0b0010_110_100:
          case 0b0010_111_100:
            irpMoveToMNLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_101_mmm_rrr
          case 0b0010_000_101:
          case 0b0010_001_101:
          case 0b0010_010_101:
          case 0b0010_011_101:
          case 0b0010_100_101:
          case 0b0010_101_101:
          case 0b0010_110_101:
          case 0b0010_111_101:
            irpMoveToMWLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_110_mmm_rrr
          case 0b0010_000_110:
          case 0b0010_001_110:
          case 0b0010_010_110:
          case 0b0010_011_110:
          case 0b0010_100_110:
          case 0b0010_101_110:
          case 0b0010_110_110:
          case 0b0010_111_110:
            irpMoveToMXLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_000_111_mmm_rrr
          case 0b0010_000_111:
            irpMoveToZWLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_001_111_mmm_rrr
          case 0b0010_001_111:
            irpMoveToZLLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_000_mmm_rrr
          case 0b0011_000_000:
          case 0b0011_001_000:
          case 0b0011_010_000:
          case 0b0011_011_000:
          case 0b0011_100_000:
          case 0b0011_101_000:
          case 0b0011_110_000:
          case 0b0011_111_000:
            irpMoveToDRWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVEA.W <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr
            //MOVE.W <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr [MOVEA.W <ea>,Aq]
          case 0b0011_000_001:
          case 0b0011_001_001:
          case 0b0011_010_001:
          case 0b0011_011_001:
          case 0b0011_100_001:
          case 0b0011_101_001:
          case 0b0011_110_001:
          case 0b0011_111_001:
            irpMoveaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_010_mmm_rrr
          case 0b0011_000_010:
          case 0b0011_001_010:
          case 0b0011_010_010:
          case 0b0011_011_010:
          case 0b0011_100_010:
          case 0b0011_101_010:
          case 0b0011_110_010:
          case 0b0011_111_010:
            irpMoveToMMWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_011_mmm_rrr
          case 0b0011_000_011:
          case 0b0011_001_011:
          case 0b0011_010_011:
          case 0b0011_011_011:
          case 0b0011_100_011:
          case 0b0011_101_011:
          case 0b0011_110_011:
          case 0b0011_111_011:
            irpMoveToMPWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_100_mmm_rrr
          case 0b0011_000_100:
          case 0b0011_001_100:
          case 0b0011_010_100:
          case 0b0011_011_100:
          case 0b0011_100_100:
          case 0b0011_101_100:
          case 0b0011_110_100:
          case 0b0011_111_100:
            irpMoveToMNWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_101_mmm_rrr
          case 0b0011_000_101:
          case 0b0011_001_101:
          case 0b0011_010_101:
          case 0b0011_011_101:
          case 0b0011_100_101:
          case 0b0011_101_101:
          case 0b0011_110_101:
          case 0b0011_111_101:
            irpMoveToMWWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_110_mmm_rrr
          case 0b0011_000_110:
          case 0b0011_001_110:
          case 0b0011_010_110:
          case 0b0011_011_110:
          case 0b0011_100_110:
          case 0b0011_101_110:
          case 0b0011_110_110:
          case 0b0011_111_110:
            irpMoveToMXWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_000_111_mmm_rrr
          case 0b0011_000_111:
            irpMoveToZWWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_001_111_mmm_rrr
          case 0b0011_001_111:
            irpMoveToZLWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEGX.B <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_000_mmm_rrr
          case 0b0100_000_000:
            irpNegxByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEGX.W <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_001_mmm_rrr
          case 0b0100_000_001:
            irpNegxWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEGX.L <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_010_mmm_rrr
          case 0b0100_000_010:
            irpNegxLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W SR,<ea>                                  |-|-12346|P|*****|-----|D M+-WXZ  |0100_000_011_mmm_rrr
          case 0b0100_000_011:
            irpMoveFromSR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CHK.L <ea>,Dq                                   |-|--2346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_100_mmm_rrr
          case 0b0100_000_100:
          case 0b0100_001_100:
          case 0b0100_010_100:
          case 0b0100_011_100:
          case 0b0100_100_100:
          case 0b0100_101_100:
          case 0b0100_110_100:
          case 0b0100_111_100:
            irpChkLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CHK.W <ea>,Dq                                   |-|012346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_110_mmm_rrr
          case 0b0100_000_110:
          case 0b0100_001_110:
          case 0b0100_010_110:
          case 0b0100_011_110:
          case 0b0100_100_110:
          case 0b0100_101_110:
          case 0b0100_110_110:
          case 0b0100_111_110:
            irpChkWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LEA.L <ea>,Aq                                   |-|012346|-|-----|-----|  M  WXZP |0100_qqq_111_mmm_rrr
            //EXTB.L Dr                                       |-|--2346|-|-UUUU|-**00|D         |0100_100_111_000_rrr
          case 0b0100_000_111:
          case 0b0100_001_111:
          case 0b0100_010_111:
          case 0b0100_011_111:
          case 0b0100_100_111:
          case 0b0100_101_111:
          case 0b0100_110_111:
          case 0b0100_111_111:
            irpLea ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CLR.B <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_000_mmm_rrr (68000 and 68008 read before clear)
          case 0b0100_001_000:
            irpClrByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CLR.W <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_001_mmm_rrr (68000 and 68008 read before clear)
          case 0b0100_001_001:
            irpClrWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CLR.L <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_010_mmm_rrr (68000 and 68008 read before clear)
          case 0b0100_001_010:
            irpClrLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W CCR,<ea>                                 |-|-12346|-|*****|-----|D M+-WXZ  |0100_001_011_mmm_rrr
          case 0b0100_001_011:
            irpMoveFromCCR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEG.B <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_000_mmm_rrr
          case 0b0100_010_000:
            irpNegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEG.W <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_001_mmm_rrr
          case 0b0100_010_001:
            irpNegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEG.L <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_010_mmm_rrr
          case 0b0100_010_010:
            irpNegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,CCR                                 |-|012346|-|UUUUU|*****|D M+-WXZPI|0100_010_011_mmm_rrr
          case 0b0100_010_011:
            irpMoveToCCR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NOT.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_000_mmm_rrr
          case 0b0100_011_000:
            irpNotByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NOT.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_001_mmm_rrr
          case 0b0100_011_001:
            irpNotWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NOT.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_010_mmm_rrr
          case 0b0100_011_010:
            irpNotLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,SR                                  |-|012346|P|UUUUU|*****|D M+-WXZPI|0100_011_011_mmm_rrr
          case 0b0100_011_011:
            irpMoveToSR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NBCD.B <ea>                                     |-|012346|-|UUUUU|*U*U*|D M+-WXZ  |0100_100_000_mmm_rrr
            //LINK.L Ar,#<data>                               |-|--2346|-|-----|-----|          |0100_100_000_001_rrr-{data}
          case 0b0100_100_000:
            irpNbcd ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SWAP.W Dr                                       |-|012346|-|-UUUU|-**00|D         |0100_100_001_000_rrr
            //BKPT #<data>                                    |-|-12346|-|-----|-----|          |0100_100_001_001_ddd
            //PEA.L <ea>                                      |-|012346|-|-----|-----|  M  WXZP |0100_100_001_mmm_rrr
          case 0b0100_100_001:
            irpPea ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXT.W Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_010_000_rrr
            //MOVEM.W <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_010_mmm_rrr-llllllllllllllll
          case 0b0100_100_010:
            irpMovemToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXT.L Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_011_000_rrr
            //MOVEM.L <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_011_mmm_rrr-llllllllllllllll
          case 0b0100_100_011:
            irpMovemToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TST.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_000_mmm_rrr
            //TST.B <ea>                                      |-|--2346|-|-UUUU|-**00|        PI|0100_101_000_mmm_rrr
          case 0b0100_101_000:
            irpTstByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TST.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_001_mmm_rrr
            //TST.W <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_001_mmm_rrr
          case 0b0100_101_001:
            irpTstWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TST.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_010_mmm_rrr
            //TST.L <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_010_mmm_rrr
          case 0b0100_101_010:
            irpTstLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TAS.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_011_mmm_rrr
            //ILLEGAL                                         |-|012346|-|-----|-----|          |0100_101_011_111_100
          case 0b0100_101_011:
            irpTas ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MULU.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll000000000hhh        (h is not used)
            //MULU.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll010000000hhh        (if h=l then result is not defined)
            //MULS.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll100000000hhh        (h is not used)
            //MULS.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll110000000hhh        (if h=l then result is not defined)
          case 0b0100_110_000:
            irpMuluMulsLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DIVU.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000qqq
            //DIVUL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000rrr        (q is not equal to r)
            //DIVU.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq010000000rrr        (q is not equal to r)
            //DIVS.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000qqq
            //DIVSL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000rrr        (q is not equal to r)
            //DIVS.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq110000000rrr        (q is not equal to r)
          case 0b0100_110_001:
            irpDivuDivsLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SATS.L Dr                                       |-|------|-|-UUUU|-**00|D         |0100_110_010_000_rrr (ISA_B)
            //MOVEM.W <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_010_mmm_rrr-llllllllllllllll
          case 0b0100_110_010:
            irpMovemToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVEM.L <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_011_mmm_rrr-llllllllllllllll
          case 0b0100_110_011:
            irpMovemToRegLong ();
            break irpSwitch;

          case 0b0100_111_001:
            switch (XEiJ.regOC & 0b111_111) {

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //TRAP #<vector>                                  |-|012346|-|-----|-----|          |0100_111_001_00v_vvv
            case 0b000_000:
            case 0b000_001:
            case 0b000_010:
            case 0b000_011:
            case 0b000_100:
            case 0b000_101:
            case 0b000_110:
            case 0b000_111:
            case 0b001_000:
            case 0b001_001:
            case 0b001_010:
            case 0b001_011:
            case 0b001_100:
            case 0b001_101:
            case 0b001_110:
              irpTrap ();
              break irpSwitch;
            case 0b001_111:
              irpTrap15 ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //LINK.W Ar,#<data>                               |-|012346|-|-----|-----|          |0100_111_001_010_rrr-{data}
            case 0b010_000:
            case 0b010_001:
            case 0b010_010:
            case 0b010_011:
            case 0b010_100:
            case 0b010_101:
            case 0b010_110:
            case 0b010_111:
              irpLinkWord ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //UNLK Ar                                         |-|012346|-|-----|-----|          |0100_111_001_011_rrr
            case 0b011_000:
            case 0b011_001:
            case 0b011_010:
            case 0b011_011:
            case 0b011_100:
            case 0b011_101:
            case 0b011_110:
            case 0b011_111:
              irpUnlk ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVE.L Ar,USP                                   |-|012346|P|-----|-----|          |0100_111_001_100_rrr
            case 0b100_000:
            case 0b100_001:
            case 0b100_010:
            case 0b100_011:
            case 0b100_100:
            case 0b100_101:
            case 0b100_110:
            case 0b100_111:
              irpMoveToUsp ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVE.L USP,Ar                                   |-|012346|P|-----|-----|          |0100_111_001_101_rrr
            case 0b101_000:
            case 0b101_001:
            case 0b101_010:
            case 0b101_011:
            case 0b101_100:
            case 0b101_101:
            case 0b101_110:
            case 0b101_111:
              irpMoveFromUsp ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RESET                                           |-|012346|P|-----|-----|          |0100_111_001_110_000
            case 0b110_000:
              irpReset ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //NOP                                             |-|012346|-|-----|-----|          |0100_111_001_110_001
            case 0b110_001:
              irpNop ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //STOP #<data>                                    |-|012346|P|UUUUU|*****|          |0100_111_001_110_010-{data}
            case 0b110_010:
              irpStop ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTE                                             |-|012346|P|UUUUU|*****|          |0100_111_001_110_011
            case 0b110_011:
              irpRte ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTD #<data>                                     |-|-12346|-|-----|-----|          |0100_111_001_110_100-{data}
            case 0b110_100:
              irpRtd ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTS                                             |-|012346|-|-----|-----|          |0100_111_001_110_101
            case 0b110_101:
              irpRts ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //TRAPV                                           |-|012346|-|---*-|-----|          |0100_111_001_110_110
            case 0b110_110:
              irpTrapv ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTR                                             |-|012346|-|UUUUU|*****|          |0100_111_001_110_111
            case 0b110_111:
              irpRtr ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVEC.L Rc,Rn                                   |-|-12346|P|-----|-----|          |0100_111_001_111_010-rnnncccccccccccc
            case 0b111_010:
              irpMovecFromControl ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVEC.L Rn,Rc                                   |-|-12346|P|-----|-----|          |0100_111_001_111_011-rnnncccccccccccc
            case 0b111_011:
              irpMovecToControl ();
              break irpSwitch;

            default:
              irpIllegal ();

            }  //switch XEiJ.regOC & 0b111_111
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //JSR <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_010_mmm_rrr
            //JBSR.L <label>                                  |A|012346|-|-----|-----|          |0100_111_010_111_001-{address}       [JSR <label>]
          case 0b0100_111_010:
            irpJsr ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //JMP <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_011_mmm_rrr
            //JBRA.L <label>                                  |A|012346|-|-----|-----|          |0100_111_011_111_001-{address}       [JMP <label>]
          case 0b0100_111_011:
            irpJmp ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_000_mmm_rrr
            //INC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_000_mmm_rrr [ADDQ.B #1,<ea>]
          case 0b0101_000_000:
          case 0b0101_001_000:
          case 0b0101_010_000:
          case 0b0101_011_000:
          case 0b0101_100_000:
          case 0b0101_101_000:
          case 0b0101_110_000:
          case 0b0101_111_000:
            irpAddqByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_001_mmm_rrr
            //ADDQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_001_001_rrr
            //INC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_001_mmm_rrr [ADDQ.W #1,<ea>]
            //INC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_001_001_rrr [ADDQ.W #1,Ar]
          case 0b0101_000_001:
          case 0b0101_001_001:
          case 0b0101_010_001:
          case 0b0101_011_001:
          case 0b0101_100_001:
          case 0b0101_101_001:
          case 0b0101_110_001:
          case 0b0101_111_001:
            irpAddqWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_010_mmm_rrr
            //ADDQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_010_001_rrr
            //INC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_010_mmm_rrr [ADDQ.L #1,<ea>]
            //INC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_010_001_rrr [ADDQ.L #1,Ar]
          case 0b0101_000_010:
          case 0b0101_001_010:
          case 0b0101_010_010:
          case 0b0101_011_010:
          case 0b0101_100_010:
          case 0b0101_101_010:
          case 0b0101_110_010:
          case 0b0101_111_010:
            irpAddqLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ST.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr
            //SNF.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr [ST.B <ea>]
            //DBT.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}
            //DBNF.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}        [DBT.W Dr,<label>]
            //TRAPT.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_010-{data}
            //TPNF.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
            //TPT.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
            //TRAPNF.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
            //TRAPT.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_011-{data}
            //TPNF.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
            //TPT.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
            //TRAPNF.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
            //TRAPT                                           |-|--2346|-|-----|-----|          |0101_000_011_111_100
            //TPNF                                            |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
            //TPT                                             |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
            //TRAPNF                                          |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
          case 0b0101_000_011:
            irpSt ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_100_mmm_rrr
            //DEC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_100_mmm_rrr [SUBQ.B #1,<ea>]
          case 0b0101_000_100:
          case 0b0101_001_100:
          case 0b0101_010_100:
          case 0b0101_011_100:
          case 0b0101_100_100:
          case 0b0101_101_100:
          case 0b0101_110_100:
          case 0b0101_111_100:
            irpSubqByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_101_mmm_rrr
            //SUBQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_101_001_rrr
            //DEC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_101_mmm_rrr [SUBQ.W #1,<ea>]
            //DEC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_101_001_rrr [SUBQ.W #1,Ar]
          case 0b0101_000_101:
          case 0b0101_001_101:
          case 0b0101_010_101:
          case 0b0101_011_101:
          case 0b0101_100_101:
          case 0b0101_101_101:
          case 0b0101_110_101:
          case 0b0101_111_101:
            irpSubqWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_110_mmm_rrr
            //SUBQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_110_001_rrr
            //DEC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_110_mmm_rrr [SUBQ.L #1,<ea>]
            //DEC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_110_001_rrr [SUBQ.L #1,Ar]
          case 0b0101_000_110:
          case 0b0101_001_110:
          case 0b0101_010_110:
          case 0b0101_011_110:
          case 0b0101_100_110:
          case 0b0101_101_110:
          case 0b0101_110_110:
          case 0b0101_111_110:
            irpSubqLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SF.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr
            //SNT.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr [SF.B <ea>]
            //DBF.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}
            //DBNT.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
            //DBRA.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
            //TRAPF.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_010-{data}
            //TPF.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
            //TPNT.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
            //TRAPNT.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
            //TRAPF.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_011-{data}
            //TPF.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
            //TPNT.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
            //TRAPNT.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
            //TRAPF                                           |-|--2346|-|-----|-----|          |0101_000_111_111_100
            //TPF                                             |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
            //TPNT                                            |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
            //TRAPNT                                          |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
          case 0b0101_000_111:
            irpSf ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SHI.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr
            //SNLS.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr [SHI.B <ea>]
            //DBHI.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}
            //DBNLS.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}        [DBHI.W Dr,<label>]
            //TRAPHI.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}
            //TPHI.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
            //TPNLS.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
            //TRAPNLS.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
            //TRAPHI.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}
            //TPHI.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
            //TPNLS.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
            //TRAPNLS.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
            //TRAPHI                                          |-|--2346|-|--*-*|-----|          |0101_001_011_111_100
            //TPHI                                            |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
            //TPNLS                                           |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
            //TRAPNLS                                         |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
          case 0b0101_001_011:
            irpShi ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SLS.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr
            //SNHI.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr [SLS.B <ea>]
            //DBLS.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}
            //DBNHI.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}        [DBLS.W Dr,<label>]
            //TRAPLS.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}
            //TPLS.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}
            //TPNHI.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
            //TRAPNHI.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
            //TRAPLS.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}
            //TPLS.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}
            //TPNHI.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
            //TRAPNHI.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
            //TRAPLS                                          |-|--2346|-|--*-*|-----|          |0101_001_111_111_100
            //TPLS                                            |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
            //TPNHI                                           |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
            //TRAPNHI                                         |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
          case 0b0101_001_111:
            irpSls ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SCC.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr
            //SHS.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
            //SNCS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
            //SNLO.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
            //DBCC.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}
            //DBHS.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
            //DBNCS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
            //DBNLO.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
            //TRAPCC.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_010-{data}
            //TPCC.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TPHS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TPNCS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TPNLO.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPHS.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPNCS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPNLO.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPCC.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_011-{data}
            //TPCC.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TPHS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TPNCS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TPNLO.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPHS.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPNCS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPNLO.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPCC                                          |-|--2346|-|----*|-----|          |0101_010_011_111_100
            //TPCC                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TPHS                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TPNCS                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TPNLO                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TRAPHS                                          |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TRAPNCS                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TRAPNLO                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
          case 0b0101_010_011:
            irpShs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SCS.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr
            //SLO.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
            //SNCC.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
            //SNHS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
            //DBCS.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}
            //DBLO.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
            //DBNCC.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
            //DBNHS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
            //TRAPCS.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_010-{data}
            //TPCS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TPLO.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TPNCC.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TPNHS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPLO.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPNCC.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPNHS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPCS.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_011-{data}
            //TPCS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TPLO.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TPNCC.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TPNHS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPLO.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPNCC.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPNHS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPCS                                          |-|--2346|-|----*|-----|          |0101_010_111_111_100
            //TPCS                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TPLO                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TPNCC                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TPNHS                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TRAPLO                                          |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TRAPNCC                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TRAPNHS                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
          case 0b0101_010_111:
            irpSlo ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SNE.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr
            //SNEQ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
            //SNZ.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
            //SNZE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
            //DBNE.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}
            //DBNEQ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
            //DBNZ.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
            //DBNZE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
            //TRAPNE.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}
            //TPNE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TPNEQ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TPNZ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TPNZE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNEQ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNZ.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNZE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNE.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}
            //TPNE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TPNEQ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TPNZ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TPNZE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNEQ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNZ.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNZE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNE                                          |-|--2346|-|--*--|-----|          |0101_011_011_111_100
            //TPNE                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TPNEQ                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TPNZ                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TPNZE                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TRAPNEQ                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TRAPNZ                                          |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TRAPNZE                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
          case 0b0101_011_011:
            irpSne ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SEQ.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr
            //SNNE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
            //SNNZ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
            //SZE.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
            //DBEQ.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}
            //DBNNE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
            //DBNNZ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
            //DBZE.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
            //TRAPEQ.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}
            //TPEQ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TPNNE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TPNNZ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TPZE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPNNE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPNNZ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPZE.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPEQ.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}
            //TPEQ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TPNNE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TPNNZ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TPZE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPNNE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPNNZ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPZE.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPEQ                                          |-|--2346|-|--*--|-----|          |0101_011_111_111_100
            //TPEQ                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TPNNE                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TPNNZ                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TPZE                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TRAPNNE                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TRAPNNZ                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TRAPZE                                          |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
          case 0b0101_011_111:
            irpSeq ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SVC.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr
            //SNVS.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr [SVC.B <ea>]
            //DBVC.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}
            //DBNVS.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}        [DBVC.W Dr,<label>]
            //TRAPVC.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}
            //TPNVS.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
            //TPVC.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
            //TRAPNVS.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
            //TRAPVC.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}
            //TPNVS.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
            //TPVC.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
            //TRAPNVS.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
            //TRAPVC                                          |-|--2346|-|---*-|-----|          |0101_100_011_111_100
            //TPNVS                                           |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
            //TPVC                                            |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
            //TRAPNVS                                         |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
          case 0b0101_100_011:
            irpSvc ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SVS.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr
            //SNVC.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr [SVS.B <ea>]
            //DBVS.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}
            //DBNVC.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}        [DBVS.W Dr,<label>]
            //TRAPVS.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}
            //TPNVC.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
            //TPVS.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
            //TRAPNVC.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
            //TRAPVS.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}
            //TPNVC.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
            //TPVS.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
            //TRAPNVC.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
            //TRAPVS                                          |-|--2346|-|---*-|-----|          |0101_100_111_111_100
            //TPNVC                                           |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
            //TPVS                                            |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
            //TRAPNVC                                         |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
          case 0b0101_100_111:
            irpSvs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SPL.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr
            //SNMI.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr [SPL.B <ea>]
            //DBPL.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}
            //DBNMI.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}        [DBPL.W Dr,<label>]
            //TRAPPL.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}
            //TPNMI.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
            //TPPL.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
            //TRAPNMI.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
            //TRAPPL.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}
            //TPNMI.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
            //TPPL.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
            //TRAPNMI.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
            //TRAPPL                                          |-|--2346|-|-*---|-----|          |0101_101_011_111_100
            //TPNMI                                           |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
            //TPPL                                            |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
            //TRAPNMI                                         |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
          case 0b0101_101_011:
            irpSpl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SMI.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr
            //SNPL.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr [SMI.B <ea>]
            //DBMI.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}
            //DBNPL.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}        [DBMI.W Dr,<label>]
            //TRAPMI.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}
            //TPMI.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
            //TPNPL.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
            //TRAPNPL.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
            //TRAPMI.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}
            //TPMI.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
            //TPNPL.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
            //TRAPNPL.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
            //TRAPMI                                          |-|--2346|-|-*---|-----|          |0101_101_111_111_100
            //TPMI                                            |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
            //TPNPL                                           |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
            //TRAPNPL                                         |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
          case 0b0101_101_111:
            irpSmi ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SGE.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr
            //SNLT.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr [SGE.B <ea>]
            //DBGE.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}
            //DBNLT.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}        [DBGE.W Dr,<label>]
            //TRAPGE.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}
            //TPGE.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
            //TPNLT.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
            //TRAPNLT.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
            //TRAPGE.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}
            //TPGE.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
            //TPNLT.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
            //TRAPNLT.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
            //TRAPGE                                          |-|--2346|-|-*-*-|-----|          |0101_110_011_111_100
            //TPGE                                            |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
            //TPNLT                                           |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
            //TRAPNLT                                         |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
          case 0b0101_110_011:
            irpSge ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SLT.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr
            //SNGE.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr [SLT.B <ea>]
            //DBLT.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}
            //DBNGE.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}        [DBLT.W Dr,<label>]
            //TRAPLT.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}
            //TPLT.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
            //TPNGE.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
            //TRAPNGE.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
            //TRAPLT.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}
            //TPLT.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
            //TPNGE.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
            //TRAPNGE.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
            //TRAPLT                                          |-|--2346|-|-*-*-|-----|          |0101_110_111_111_100
            //TPLT                                            |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
            //TPNGE                                           |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
            //TRAPNGE                                         |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
          case 0b0101_110_111:
            irpSlt ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SGT.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr
            //SNLE.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr [SGT.B <ea>]
            //DBGT.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}
            //DBNLE.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}        [DBGT.W Dr,<label>]
            //TRAPGT.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}
            //TPGT.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
            //TPNLE.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
            //TRAPNLE.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
            //TRAPGT.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}
            //TPGT.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
            //TPNLE.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
            //TRAPNLE.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
            //TRAPGT                                          |-|--2346|-|-***-|-----|          |0101_111_011_111_100
            //TPGT                                            |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
            //TPNLE                                           |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
            //TRAPNLE                                         |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
          case 0b0101_111_011:
            irpSgt ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SLE.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr
            //SNGT.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr [SLE.B <ea>]
            //DBLE.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}
            //DBNGT.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}        [DBLE.W Dr,<label>]
            //TRAPLE.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}
            //TPLE.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
            //TPNGT.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
            //TRAPNGT.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
            //TRAPLE.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}
            //TPLE.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
            //TPNGT.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
            //TRAPNGT.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
            //TRAPLE                                          |-|--2346|-|-***-|-----|          |0101_111_111_111_100
            //TPLE                                            |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
            //TPNGT                                           |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
            //TRAPNGT                                         |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
          case 0b0101_111_111:
            irpSle ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.W <label>                                   |-|012346|-|-----|-----|          |0110_000_000_000_000-{offset}
            //JBRA.W <label>                                  |A|012346|-|-----|-----|          |0110_000_000_000_000-{offset}        [BRA.W <label>]
            //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)
            //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)   [BRA.S <label>]
          case 0b0110_000_000:
            irpBrasw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_001_sss_sss
            //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_001_sss_sss [BRA.S <label>]
          case 0b0110_000_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_010_sss_sss
            //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_010_sss_sss [BRA.S <label>]
          case 0b0110_000_010:
            irpBras ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)
            //JBRA.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)  [BRA.S <label>]
            //BRA.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_111_111-{offset}
          case 0b0110_000_011:
            irpBrasl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.W <label>                                   |-|012346|-|-----|-----|          |0110_000_100_000_000-{offset}
            //JBSR.W <label>                                  |A|012346|-|-----|-----|          |0110_000_100_000_000-{offset}        [BSR.W <label>]
            //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)
            //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)   [BSR.S <label>]
          case 0b0110_000_100:
            irpBsrsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_101_sss_sss
            //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_101_sss_sss [BSR.S <label>]
          case 0b0110_000_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_110_sss_sss
            //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_110_sss_sss [BSR.S <label>]
          case 0b0110_000_110:
            irpBsrs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)
            //JBSR.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)  [BSR.S <label>]
            //BSR.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_111_111-{offset}
          case 0b0110_000_111:
            irpBsrsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}
            //BNLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
            //JBHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
            //JBNLS.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
            //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)
            //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
            //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
            //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
            //JBLS.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
            //JBNHI.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
          case 0b0110_001_000:
            irpBhisw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_001_sss_sss
            //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
            //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
            //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
          case 0b0110_001_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_010_sss_sss
            //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
            //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
            //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
          case 0b0110_001_010:
            irpBhis ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)
            //BNLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
            //JBHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
            //JBNLS.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
            //BHI.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}
            //BNLS.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}        [BHI.L <label>]
          case 0b0110_001_011:
            irpBhisl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}
            //BNHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
            //JBLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
            //JBNHI.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
            //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)
            //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
            //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
            //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
            //JBHI.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
            //JBNLS.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
          case 0b0110_001_100:
            irpBlssw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_101_sss_sss
            //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
            //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
            //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
          case 0b0110_001_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_110_sss_sss
            //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
            //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
            //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
          case 0b0110_001_110:
            irpBlss ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)
            //BNHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
            //JBLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
            //JBNHI.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
            //BLS.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}
            //BNHI.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}        [BLS.L <label>]
          case 0b0110_001_111:
            irpBlssl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.W <label>                                   |-|012346|-|----*|-----|          |0110_010_000_000_000-{offset}
            //BHS.W <label>                                   |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //BNCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //BNLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBNCS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBNLO.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)
            //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBCS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
            //JBLO.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
            //JBNCC.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
            //JBNHS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
          case 0b0110_010_000:
            irpBhssw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_001_sss_sss
            //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
          case 0b0110_010_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_010_sss_sss
            //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
          case 0b0110_010_010:
            irpBhss ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)
            //BHS.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //BNCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //BNLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBNCS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBNLO.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //BCC.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}
            //BHS.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
            //BNCS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
            //BNLO.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
          case 0b0110_010_011:
            irpBhssl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.W <label>                                   |-|012346|-|----*|-----|          |0110_010_100_000_000-{offset}
            //BLO.W <label>                                   |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //BNCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //BNHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBNCC.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBNHS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)
            //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBCC.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
            //JBHS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
            //JBNCS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
            //JBNLO.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
          case 0b0110_010_100:
            irpBlosw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_101_sss_sss
            //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
          case 0b0110_010_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_110_sss_sss
            //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
          case 0b0110_010_110:
            irpBlos ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)
            //BLO.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //BNCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //BNHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBNCC.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBNHS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //BCS.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}
            //BLO.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
            //BNCC.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
            //BNHS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
          case 0b0110_010_111:
            irpBlosl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}
            //BNEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //BNZ.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //BNZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNEQ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNZE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)
            //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBEQ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNEQ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNNE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNNZ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNZ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNZE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBZE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
          case 0b0110_011_000:
            irpBnesw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_001_sss_sss
            //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
          case 0b0110_011_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_010_sss_sss
            //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
          case 0b0110_011_010:
            irpBnes ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)
            //BNEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //BNZ.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //BNZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNZE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //BNE.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}
            //BNEQ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
            //BNZ.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
            //BNZE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
          case 0b0110_011_011:
            irpBnesl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}
            //BNNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //BNNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //BZE.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBNNE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBNNZ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)
            //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBNE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_110-0100111011111001-{address}      [BEQ.S (*)+8;JMP <label>]
          case 0b0110_011_100:
            irpBeqsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_101_sss_sss
            //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
          case 0b0110_011_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_110_sss_sss
            //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
          case 0b0110_011_110:
            irpBeqs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)
            //BNNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //BZE.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //BEQ.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}
            //BNNE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
            //BNNZ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
            //BZE.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
          case 0b0110_011_111:
            irpBeqsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}
            //BNVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
            //JBNVS.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
            //JBVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
            //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)
            //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
            //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
            //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
            //JBNVC.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
            //JBVS.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
          case 0b0110_100_000:
            irpBvcsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_001_sss_sss
            //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
            //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
            //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
          case 0b0110_100_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_010_sss_sss
            //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
            //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
            //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
          case 0b0110_100_010:
            irpBvcs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)
            //BNVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
            //JBNVS.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
            //JBVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
            //BVC.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}
            //BNVS.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}        [BVC.L <label>]
          case 0b0110_100_011:
            irpBvcsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}
            //BNVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
            //JBNVC.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
            //JBVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
            //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)
            //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
            //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
            //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
            //JBNVS.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
            //JBVC.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
          case 0b0110_100_100:
            irpBvssw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_101_sss_sss
            //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
            //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
            //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
          case 0b0110_100_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_110_sss_sss
            //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
            //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
            //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
          case 0b0110_100_110:
            irpBvss ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)
            //BNVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
            //JBNVC.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
            //JBVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
            //BVS.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}
            //BNVC.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}        [BVS.L <label>]
          case 0b0110_100_111:
            irpBvssl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}
            //BNMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
            //JBNMI.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
            //JBPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
            //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)
            //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
            //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
            //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
            //JBMI.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
            //JBNPL.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
          case 0b0110_101_000:
            irpBplsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_001_sss_sss
            //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
            //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
            //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
          case 0b0110_101_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_010_sss_sss
            //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
            //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
            //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
          case 0b0110_101_010:
            irpBpls ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)
            //BNMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
            //JBNMI.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
            //JBPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
            //BPL.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}
            //BNMI.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}        [BPL.L <label>]
          case 0b0110_101_011:
            irpBplsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}
            //BNPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
            //JBMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
            //JBNPL.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
            //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)
            //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
            //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
            //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
            //JBNMI.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
            //JBPL.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
          case 0b0110_101_100:
            irpBmisw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_101_sss_sss
            //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
            //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
            //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
          case 0b0110_101_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_110_sss_sss
            //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
            //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
            //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
          case 0b0110_101_110:
            irpBmis ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)
            //BNPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
            //JBMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
            //JBNPL.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
            //BMI.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}
            //BNPL.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}        [BMI.L <label>]
          case 0b0110_101_111:
            irpBmisl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}
            //BNLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
            //JBGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
            //JBNLT.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
            //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)
            //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
            //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
            //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
            //JBLT.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
            //JBNGE.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
          case 0b0110_110_000:
            irpBgesw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_001_sss_sss
            //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
            //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
            //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
          case 0b0110_110_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_010_sss_sss
            //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
            //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
            //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
          case 0b0110_110_010:
            irpBges ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)
            //BNLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
            //JBGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
            //JBNLT.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
            //BGE.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}
            //BNLT.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}        [BGE.L <label>]
          case 0b0110_110_011:
            irpBgesl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}
            //BNGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
            //JBLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
            //JBNGE.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
            //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)
            //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
            //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
            //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
            //JBGE.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
            //JBNLT.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
          case 0b0110_110_100:
            irpBltsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_101_sss_sss
            //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
            //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
            //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
          case 0b0110_110_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_110_sss_sss
            //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
            //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
            //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
          case 0b0110_110_110:
            irpBlts ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)
            //BNGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
            //JBLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
            //JBNGE.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
            //BLT.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}
            //BNGE.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}        [BLT.L <label>]
          case 0b0110_110_111:
            irpBltsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}
            //BNLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
            //JBGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
            //JBNLE.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
            //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)
            //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
            //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
            //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
            //JBLE.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
            //JBNGT.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
          case 0b0110_111_000:
            irpBgtsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_001_sss_sss
            //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
            //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
            //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
          case 0b0110_111_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_010_sss_sss
            //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
            //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
            //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
          case 0b0110_111_010:
            irpBgts ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)
            //BNLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
            //JBGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
            //JBNLE.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
            //BGT.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}
            //BNLE.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}        [BGT.L <label>]
          case 0b0110_111_011:
            irpBgtsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}
            //BNGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
            //JBLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
            //JBNGT.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
            //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)
            //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
            //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
            //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
            //JBGT.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
            //JBNLE.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
          case 0b0110_111_100:
            irpBlesw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_101_sss_sss
            //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
            //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
            //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
          case 0b0110_111_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_110_sss_sss
            //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
            //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
            //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
          case 0b0110_111_110:
            irpBles ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)
            //BNGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
            //JBLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
            //JBNGT.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
            //BLE.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}
            //BNGT.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}        [BLE.L <label>]
          case 0b0110_111_111:
            irpBlesl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //IOCS <name>                                     |A|012346|-|UUUUU|UUUUU|          |0111_000_0dd_ddd_ddd-0100111001001111        [MOVEQ.L #<data>,D0;TRAP #15]
            //MOVEQ.L #<data>,Dq                              |-|012346|-|-UUUU|-**00|          |0111_qqq_0dd_ddd_ddd
          case 0b0111_000_000:
          case 0b0111_000_001:
          case 0b0111_000_010:
          case 0b0111_000_011:
          case 0b0111_001_000:
          case 0b0111_001_001:
          case 0b0111_001_010:
          case 0b0111_001_011:
          case 0b0111_010_000:
          case 0b0111_010_001:
          case 0b0111_010_010:
          case 0b0111_010_011:
          case 0b0111_011_000:
          case 0b0111_011_001:
          case 0b0111_011_010:
          case 0b0111_011_011:
          case 0b0111_100_000:
          case 0b0111_100_001:
          case 0b0111_100_010:
          case 0b0111_100_011:
          case 0b0111_101_000:
          case 0b0111_101_001:
          case 0b0111_101_010:
          case 0b0111_101_011:
          case 0b0111_110_000:
          case 0b0111_110_001:
          case 0b0111_110_010:
          case 0b0111_110_011:
          case 0b0111_111_000:
          case 0b0111_111_001:
          case 0b0111_111_010:
          case 0b0111_111_011:
            irpMoveq ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVS.B <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_100_mmm_rrr (ISA_B)
          case 0b0111_000_100:
          case 0b0111_001_100:
          case 0b0111_010_100:
          case 0b0111_011_100:
          case 0b0111_100_100:
          case 0b0111_101_100:
          case 0b0111_110_100:
          case 0b0111_111_100:
            irpMvsByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVS.W <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_101_mmm_rrr (ISA_B)
          case 0b0111_000_101:
          case 0b0111_001_101:
          case 0b0111_010_101:
          case 0b0111_011_101:
          case 0b0111_100_101:
          case 0b0111_101_101:
          case 0b0111_110_101:
          case 0b0111_111_101:
            irpMvsWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVZ.B <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_110_mmm_rrr (ISA_B)
          case 0b0111_000_110:
          case 0b0111_001_110:
          case 0b0111_010_110:
          case 0b0111_011_110:
          case 0b0111_100_110:
          case 0b0111_101_110:
          case 0b0111_110_110:
          case 0b0111_111_110:
            irpMvzByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVZ.W <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_111_mmm_rrr (ISA_B)
          case 0b0111_000_111:
          case 0b0111_001_111:
          case 0b0111_010_111:
          case 0b0111_011_111:
          case 0b0111_100_111:
          case 0b0111_101_111:
          case 0b0111_110_111:
          case 0b0111_111_111:
            irpMvzWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //OR.B <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_000_mmm_rrr
          case 0b1000_000_000:
          case 0b1000_001_000:
          case 0b1000_010_000:
          case 0b1000_011_000:
          case 0b1000_100_000:
          case 0b1000_101_000:
          case 0b1000_110_000:
          case 0b1000_111_000:
            irpOrToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //OR.W <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_001_mmm_rrr
          case 0b1000_000_001:
          case 0b1000_001_001:
          case 0b1000_010_001:
          case 0b1000_011_001:
          case 0b1000_100_001:
          case 0b1000_101_001:
          case 0b1000_110_001:
          case 0b1000_111_001:
            irpOrToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //OR.L <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_010_mmm_rrr
          case 0b1000_000_010:
          case 0b1000_001_010:
          case 0b1000_010_010:
          case 0b1000_011_010:
          case 0b1000_100_010:
          case 0b1000_101_010:
          case 0b1000_110_010:
          case 0b1000_111_010:
            irpOrToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DIVU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_011_mmm_rrr
          case 0b1000_000_011:
          case 0b1000_001_011:
          case 0b1000_010_011:
          case 0b1000_011_011:
          case 0b1000_100_011:
          case 0b1000_101_011:
          case 0b1000_110_011:
          case 0b1000_111_011:
            irpDivuWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SBCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_000_rrr
            //SBCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_001_rrr
            //OR.B Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_100_mmm_rrr
          case 0b1000_000_100:
          case 0b1000_001_100:
          case 0b1000_010_100:
          case 0b1000_011_100:
          case 0b1000_100_100:
          case 0b1000_101_100:
          case 0b1000_110_100:
          case 0b1000_111_100:
            irpOrToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //PACK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_101_000_rrr-{data}
            //PACK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_101_001_rrr-{data}
            //OR.W Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_101_mmm_rrr
          case 0b1000_000_101:
          case 0b1000_001_101:
          case 0b1000_010_101:
          case 0b1000_011_101:
          case 0b1000_100_101:
          case 0b1000_101_101:
          case 0b1000_110_101:
          case 0b1000_111_101:
            irpOrToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //UNPK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_110_000_rrr-{data}
            //UNPK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_110_001_rrr-{data}
            //OR.L Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_110_mmm_rrr
          case 0b1000_000_110:
          case 0b1000_001_110:
          case 0b1000_010_110:
          case 0b1000_011_110:
          case 0b1000_100_110:
          case 0b1000_101_110:
          case 0b1000_110_110:
          case 0b1000_111_110:
            irpOrToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DIVS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_111_mmm_rrr
          case 0b1000_000_111:
          case 0b1000_001_111:
          case 0b1000_010_111:
          case 0b1000_011_111:
          case 0b1000_100_111:
          case 0b1000_101_111:
          case 0b1000_110_111:
          case 0b1000_111_111:
            irpDivsWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUB.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1001_qqq_000_mmm_rrr
          case 0b1001_000_000:
          case 0b1001_001_000:
          case 0b1001_010_000:
          case 0b1001_011_000:
          case 0b1001_100_000:
          case 0b1001_101_000:
          case 0b1001_110_000:
          case 0b1001_111_000:
            irpSubToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUB.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_001_mmm_rrr
          case 0b1001_000_001:
          case 0b1001_001_001:
          case 0b1001_010_001:
          case 0b1001_011_001:
          case 0b1001_100_001:
          case 0b1001_101_001:
          case 0b1001_110_001:
          case 0b1001_111_001:
            irpSubToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUB.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_010_mmm_rrr
          case 0b1001_000_010:
          case 0b1001_001_010:
          case 0b1001_010_010:
          case 0b1001_011_010:
          case 0b1001_100_010:
          case 0b1001_101_010:
          case 0b1001_110_010:
          case 0b1001_111_010:
            irpSubToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr
            //SUB.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr [SUBA.W <ea>,Aq]
            //CLR.W Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_011_001_rrr [SUBA.W Ar,Ar]
          case 0b1001_000_011:
          case 0b1001_001_011:
          case 0b1001_010_011:
          case 0b1001_011_011:
          case 0b1001_100_011:
          case 0b1001_101_011:
          case 0b1001_110_011:
          case 0b1001_111_011:
            irpSubaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_100_000_rrr
            //SUBX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_100_001_rrr
            //SUB.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_100_mmm_rrr
          case 0b1001_000_100:
          case 0b1001_001_100:
          case 0b1001_010_100:
          case 0b1001_011_100:
          case 0b1001_100_100:
          case 0b1001_101_100:
          case 0b1001_110_100:
          case 0b1001_111_100:
            irpSubToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_101_000_rrr
            //SUBX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_101_001_rrr
            //SUB.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_101_mmm_rrr
          case 0b1001_000_101:
          case 0b1001_001_101:
          case 0b1001_010_101:
          case 0b1001_011_101:
          case 0b1001_100_101:
          case 0b1001_101_101:
          case 0b1001_110_101:
          case 0b1001_111_101:
            irpSubToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_110_000_rrr
            //SUBX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_110_001_rrr
            //SUB.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_110_mmm_rrr
          case 0b1001_000_110:
          case 0b1001_001_110:
          case 0b1001_010_110:
          case 0b1001_011_110:
          case 0b1001_100_110:
          case 0b1001_101_110:
          case 0b1001_110_110:
          case 0b1001_111_110:
            irpSubToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr
            //SUB.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr [SUBA.L <ea>,Aq]
            //CLR.L Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_111_001_rrr [SUBA.L Ar,Ar]
          case 0b1001_000_111:
          case 0b1001_001_111:
          case 0b1001_010_111:
          case 0b1001_011_111:
          case 0b1001_100_111:
          case 0b1001_101_111:
          case 0b1001_110_111:
          case 0b1001_111_111:
            irpSubaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SXCALL <name>                                   |A|012346|-|UUUUU|*****|          |1010_0dd_ddd_ddd_ddd [ALINE #<data>]
          case 0b1010_000_000:
          case 0b1010_000_001:
          case 0b1010_000_010:
          case 0b1010_000_011:
          case 0b1010_000_100:
          case 0b1010_000_101:
          case 0b1010_000_110:
          case 0b1010_000_111:
          case 0b1010_001_000:
          case 0b1010_001_001:
          case 0b1010_001_010:
          case 0b1010_001_011:
          case 0b1010_001_100:
          case 0b1010_001_101:
          case 0b1010_001_110:
          case 0b1010_001_111:
          case 0b1010_010_000:
          case 0b1010_010_001:
          case 0b1010_010_010:
          case 0b1010_010_011:
          case 0b1010_010_100:
          case 0b1010_010_101:
          case 0b1010_010_110:
          case 0b1010_010_111:
          case 0b1010_011_000:
          case 0b1010_011_001:
          case 0b1010_011_010:
          case 0b1010_011_011:
          case 0b1010_011_100:
          case 0b1010_011_101:
          case 0b1010_011_110:
          case 0b1010_011_111:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ALINE #<data>                                   |-|012346|-|UUUUU|*****|          |1010_ddd_ddd_ddd_ddd (line 1010 emulator)
          case 0b1010_100_000:
          case 0b1010_100_001:
          case 0b1010_100_010:
          case 0b1010_100_011:
          case 0b1010_100_100:
          case 0b1010_100_101:
          case 0b1010_100_110:
          case 0b1010_100_111:
          case 0b1010_101_000:
          case 0b1010_101_001:
          case 0b1010_101_010:
          case 0b1010_101_011:
          case 0b1010_101_100:
          case 0b1010_101_101:
          case 0b1010_101_110:
          case 0b1010_101_111:
          case 0b1010_110_000:
          case 0b1010_110_001:
          case 0b1010_110_010:
          case 0b1010_110_011:
          case 0b1010_110_100:
          case 0b1010_110_101:
          case 0b1010_110_110:
          case 0b1010_110_111:
          case 0b1010_111_000:
          case 0b1010_111_001:
          case 0b1010_111_010:
          case 0b1010_111_011:
          case 0b1010_111_100:
          case 0b1010_111_101:
          case 0b1010_111_110:
          case 0b1010_111_111:
            irpAline ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMP.B <ea>,Dq                                   |-|012346|-|-UUUU|-****|D M+-WXZPI|1011_qqq_000_mmm_rrr
          case 0b1011_000_000:
          case 0b1011_001_000:
          case 0b1011_010_000:
          case 0b1011_011_000:
          case 0b1011_100_000:
          case 0b1011_101_000:
          case 0b1011_110_000:
          case 0b1011_111_000:
            irpCmpByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMP.W <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_001_mmm_rrr
          case 0b1011_000_001:
          case 0b1011_001_001:
          case 0b1011_010_001:
          case 0b1011_011_001:
          case 0b1011_100_001:
          case 0b1011_101_001:
          case 0b1011_110_001:
          case 0b1011_111_001:
            irpCmpWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMP.L <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_010_mmm_rrr
          case 0b1011_000_010:
          case 0b1011_001_010:
          case 0b1011_010_010:
          case 0b1011_011_010:
          case 0b1011_100_010:
          case 0b1011_101_010:
          case 0b1011_110_010:
          case 0b1011_111_010:
            irpCmpLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPA.W <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr
            //CMP.W <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr [CMPA.W <ea>,Aq]
          case 0b1011_000_011:
          case 0b1011_001_011:
          case 0b1011_010_011:
          case 0b1011_011_011:
          case 0b1011_100_011:
          case 0b1011_101_011:
          case 0b1011_110_011:
          case 0b1011_111_011:
            irpCmpaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EOR.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_100_mmm_rrr
            //CMPM.B (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_100_001_rrr
          case 0b1011_000_100:
          case 0b1011_001_100:
          case 0b1011_010_100:
          case 0b1011_011_100:
          case 0b1011_100_100:
          case 0b1011_101_100:
          case 0b1011_110_100:
          case 0b1011_111_100:
            irpEorByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EOR.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_101_mmm_rrr
            //CMPM.W (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_101_001_rrr
          case 0b1011_000_101:
          case 0b1011_001_101:
          case 0b1011_010_101:
          case 0b1011_011_101:
          case 0b1011_100_101:
          case 0b1011_101_101:
          case 0b1011_110_101:
          case 0b1011_111_101:
            irpEorWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EOR.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_110_mmm_rrr
            //CMPM.L (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_110_001_rrr
          case 0b1011_000_110:
          case 0b1011_001_110:
          case 0b1011_010_110:
          case 0b1011_011_110:
          case 0b1011_100_110:
          case 0b1011_101_110:
          case 0b1011_110_110:
          case 0b1011_111_110:
            irpEorLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPA.L <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr
            //CMP.L <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr [CMPA.L <ea>,Aq]
          case 0b1011_000_111:
          case 0b1011_001_111:
          case 0b1011_010_111:
          case 0b1011_011_111:
          case 0b1011_100_111:
          case 0b1011_101_111:
          case 0b1011_110_111:
          case 0b1011_111_111:
            irpCmpaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //AND.B <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_000_mmm_rrr
          case 0b1100_000_000:
          case 0b1100_001_000:
          case 0b1100_010_000:
          case 0b1100_011_000:
          case 0b1100_100_000:
          case 0b1100_101_000:
          case 0b1100_110_000:
          case 0b1100_111_000:
            irpAndToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //AND.W <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_001_mmm_rrr
          case 0b1100_000_001:
          case 0b1100_001_001:
          case 0b1100_010_001:
          case 0b1100_011_001:
          case 0b1100_100_001:
          case 0b1100_101_001:
          case 0b1100_110_001:
          case 0b1100_111_001:
            irpAndToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //AND.L <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_010_mmm_rrr
          case 0b1100_000_010:
          case 0b1100_001_010:
          case 0b1100_010_010:
          case 0b1100_011_010:
          case 0b1100_100_010:
          case 0b1100_101_010:
          case 0b1100_110_010:
          case 0b1100_111_010:
            irpAndToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MULU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_011_mmm_rrr
          case 0b1100_000_011:
          case 0b1100_001_011:
          case 0b1100_010_011:
          case 0b1100_011_011:
          case 0b1100_100_011:
          case 0b1100_101_011:
          case 0b1100_110_011:
          case 0b1100_111_011:
            irpMuluWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ABCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_000_rrr
            //ABCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_001_rrr
            //AND.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_100_mmm_rrr
          case 0b1100_000_100:
          case 0b1100_001_100:
          case 0b1100_010_100:
          case 0b1100_011_100:
          case 0b1100_100_100:
          case 0b1100_101_100:
          case 0b1100_110_100:
          case 0b1100_111_100:
            irpAndToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXG.L Dq,Dr                                     |-|012346|-|-----|-----|          |1100_qqq_101_000_rrr
            //EXG.L Aq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_101_001_rrr
            //AND.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_101_mmm_rrr
          case 0b1100_000_101:
          case 0b1100_001_101:
          case 0b1100_010_101:
          case 0b1100_011_101:
          case 0b1100_100_101:
          case 0b1100_101_101:
          case 0b1100_110_101:
          case 0b1100_111_101:
            irpAndToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXG.L Dq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_110_001_rrr
            //AND.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_110_mmm_rrr
          case 0b1100_000_110:
          case 0b1100_001_110:
          case 0b1100_010_110:
          case 0b1100_011_110:
          case 0b1100_100_110:
          case 0b1100_101_110:
          case 0b1100_110_110:
          case 0b1100_111_110:
            irpAndToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MULS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_111_mmm_rrr
          case 0b1100_000_111:
          case 0b1100_001_111:
          case 0b1100_010_111:
          case 0b1100_011_111:
          case 0b1100_100_111:
          case 0b1100_101_111:
          case 0b1100_110_111:
          case 0b1100_111_111:
            irpMulsWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADD.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1101_qqq_000_mmm_rrr
          case 0b1101_000_000:
          case 0b1101_001_000:
          case 0b1101_010_000:
          case 0b1101_011_000:
          case 0b1101_100_000:
          case 0b1101_101_000:
          case 0b1101_110_000:
          case 0b1101_111_000:
            irpAddToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADD.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_001_mmm_rrr
          case 0b1101_000_001:
          case 0b1101_001_001:
          case 0b1101_010_001:
          case 0b1101_011_001:
          case 0b1101_100_001:
          case 0b1101_101_001:
          case 0b1101_110_001:
          case 0b1101_111_001:
            irpAddToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADD.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_010_mmm_rrr
          case 0b1101_000_010:
          case 0b1101_001_010:
          case 0b1101_010_010:
          case 0b1101_011_010:
          case 0b1101_100_010:
          case 0b1101_101_010:
          case 0b1101_110_010:
          case 0b1101_111_010:
            irpAddToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr
            //ADD.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr [ADDA.W <ea>,Aq]
          case 0b1101_000_011:
          case 0b1101_001_011:
          case 0b1101_010_011:
          case 0b1101_011_011:
          case 0b1101_100_011:
          case 0b1101_101_011:
          case 0b1101_110_011:
          case 0b1101_111_011:
            irpAddaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_100_000_rrr
            //ADDX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_100_001_rrr
            //ADD.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_100_mmm_rrr
          case 0b1101_000_100:
          case 0b1101_001_100:
          case 0b1101_010_100:
          case 0b1101_011_100:
          case 0b1101_100_100:
          case 0b1101_101_100:
          case 0b1101_110_100:
          case 0b1101_111_100:
            irpAddToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_101_000_rrr
            //ADDX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_101_001_rrr
            //ADD.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_101_mmm_rrr
          case 0b1101_000_101:
          case 0b1101_001_101:
          case 0b1101_010_101:
          case 0b1101_011_101:
          case 0b1101_100_101:
          case 0b1101_101_101:
          case 0b1101_110_101:
          case 0b1101_111_101:
            irpAddToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_110_000_rrr
            //ADDX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_110_001_rrr
            //ADD.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_110_mmm_rrr
          case 0b1101_000_110:
          case 0b1101_001_110:
          case 0b1101_010_110:
          case 0b1101_011_110:
          case 0b1101_100_110:
          case 0b1101_101_110:
          case 0b1101_110_110:
          case 0b1101_111_110:
            irpAddToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr
            //ADD.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr [ADDA.L <ea>,Aq]
          case 0b1101_000_111:
          case 0b1101_001_111:
          case 0b1101_010_111:
          case 0b1101_011_111:
          case 0b1101_100_111:
          case 0b1101_101_111:
          case 0b1101_110_111:
          case 0b1101_111_111:
            irpAddaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_000_rrr
            //LSR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_001_rrr
            //ROXR.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_000_010_rrr
            //ROR.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_011_rrr
            //ASR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_100_rrr
            //LSR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_101_rrr
            //ROXR.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_000_110_rrr
            //ROR.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_111_rrr
            //ASR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_000_rrr [ASR.B #1,Dr]
            //LSR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_001_rrr [LSR.B #1,Dr]
            //ROXR.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_000_010_rrr [ROXR.B #1,Dr]
            //ROR.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_000_011_rrr [ROR.B #1,Dr]
          case 0b1110_000_000:
          case 0b1110_001_000:
          case 0b1110_010_000:
          case 0b1110_011_000:
          case 0b1110_100_000:
          case 0b1110_101_000:
          case 0b1110_110_000:
          case 0b1110_111_000:
            irpXxrToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_000_rrr
            //LSR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_001_rrr
            //ROXR.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_001_010_rrr
            //ROR.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_011_rrr
            //ASR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_100_rrr
            //LSR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_101_rrr
            //ROXR.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_001_110_rrr
            //ROR.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_111_rrr
            //ASR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_000_rrr [ASR.W #1,Dr]
            //LSR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_001_rrr [LSR.W #1,Dr]
            //ROXR.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_001_010_rrr [ROXR.W #1,Dr]
            //ROR.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_001_011_rrr [ROR.W #1,Dr]
          case 0b1110_000_001:
          case 0b1110_001_001:
          case 0b1110_010_001:
          case 0b1110_011_001:
          case 0b1110_100_001:
          case 0b1110_101_001:
          case 0b1110_110_001:
          case 0b1110_111_001:
            irpXxrToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_000_rrr
            //LSR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_001_rrr
            //ROXR.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_010_010_rrr
            //ROR.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_011_rrr
            //ASR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_100_rrr
            //LSR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_101_rrr
            //ROXR.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_010_110_rrr
            //ROR.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_111_rrr
            //ASR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_000_rrr [ASR.L #1,Dr]
            //LSR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_001_rrr [LSR.L #1,Dr]
            //ROXR.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_010_010_rrr [ROXR.L #1,Dr]
            //ROR.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_010_011_rrr [ROR.L #1,Dr]
          case 0b1110_000_010:
          case 0b1110_001_010:
          case 0b1110_010_010:
          case 0b1110_011_010:
          case 0b1110_100_010:
          case 0b1110_101_010:
          case 0b1110_110_010:
          case 0b1110_111_010:
            irpXxrToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.W <ea>                                      |-|012346|-|UUUUU|***0*|  M+-WXZ  |1110_000_011_mmm_rrr
          case 0b1110_000_011:
            irpAsrToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.B #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_100_000_rrr
            //LSL.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_100_001_rrr
            //ROXL.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_100_010_rrr
            //ROL.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_100_011_rrr
            //ASL.B Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_100_100_rrr
            //LSL.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_100_101_rrr
            //ROXL.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_100_110_rrr
            //ROL.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_100_111_rrr
            //ASL.B Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_100_000_rrr [ASL.B #1,Dr]
            //LSL.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_100_001_rrr [LSL.B #1,Dr]
            //ROXL.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_100_010_rrr [ROXL.B #1,Dr]
            //ROL.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_100_011_rrr [ROL.B #1,Dr]
          case 0b1110_000_100:
          case 0b1110_001_100:
          case 0b1110_010_100:
          case 0b1110_011_100:
          case 0b1110_100_100:
          case 0b1110_101_100:
          case 0b1110_110_100:
          case 0b1110_111_100:
            irpXxlToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.W #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_101_000_rrr
            //LSL.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_101_001_rrr
            //ROXL.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_101_010_rrr
            //ROL.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_101_011_rrr
            //ASL.W Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_101_100_rrr
            //LSL.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_101_101_rrr
            //ROXL.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_101_110_rrr
            //ROL.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_101_111_rrr
            //ASL.W Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_101_000_rrr [ASL.W #1,Dr]
            //LSL.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_101_001_rrr [LSL.W #1,Dr]
            //ROXL.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_101_010_rrr [ROXL.W #1,Dr]
            //ROL.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_101_011_rrr [ROL.W #1,Dr]
          case 0b1110_000_101:
          case 0b1110_001_101:
          case 0b1110_010_101:
          case 0b1110_011_101:
          case 0b1110_100_101:
          case 0b1110_101_101:
          case 0b1110_110_101:
          case 0b1110_111_101:
            irpXxlToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.L #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_110_000_rrr
            //LSL.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_110_001_rrr
            //ROXL.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_110_010_rrr
            //ROL.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_110_011_rrr
            //ASL.L Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_110_100_rrr
            //LSL.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_110_101_rrr
            //ROXL.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_110_110_rrr
            //ROL.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_110_111_rrr
            //ASL.L Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_110_000_rrr [ASL.L #1,Dr]
            //LSL.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_110_001_rrr [LSL.L #1,Dr]
            //ROXL.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_110_010_rrr [ROXL.L #1,Dr]
            //ROL.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_110_011_rrr [ROL.L #1,Dr]
          case 0b1110_000_110:
          case 0b1110_001_110:
          case 0b1110_010_110:
          case 0b1110_011_110:
          case 0b1110_100_110:
          case 0b1110_101_110:
          case 0b1110_110_110:
          case 0b1110_111_110:
            irpXxlToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.W <ea>                                      |-|012346|-|UUUUU|*****|  M+-WXZ  |1110_000_111_mmm_rrr
          case 0b1110_000_111:
            irpAslToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LSR.W <ea>                                      |-|012346|-|UUUUU|*0*0*|  M+-WXZ  |1110_001_011_mmm_rrr
          case 0b1110_001_011:
            irpLsrToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LSL.W <ea>                                      |-|012346|-|UUUUU|***0*|  M+-WXZ  |1110_001_111_mmm_rrr
          case 0b1110_001_111:
            irpLslToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROXR.W <ea>                                     |-|012346|-|*UUUU|***0*|  M+-WXZ  |1110_010_011_mmm_rrr
          case 0b1110_010_011:
            irpRoxrToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROXL.W <ea>                                     |-|012346|-|*UUUU|***0*|  M+-WXZ  |1110_010_111_mmm_rrr
          case 0b1110_010_111:
            irpRoxlToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROR.W <ea>                                      |-|012346|-|-UUUU|-**0*|  M+-WXZ  |1110_011_011_mmm_rrr
          case 0b1110_011_011:
            irpRorToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROL.W <ea>                                      |-|012346|-|-UUUU|-**0*|  M+-WXZ  |1110_011_111_mmm_rrr
          case 0b1110_011_111:
            irpRolToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFTST <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-00000ooooo0wwwww
            //BFTST <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-00000ooooo100www
            //BFTST <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-0000100ooo0wwwww
            //BFTST <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-0000100ooo100www
          case 0b1110_100_011:
            irpBftst ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFEXTU <ea>{#o:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFEXTU <ea>{#o:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn0ooooo100www
            //BFEXTU <ea>{Do:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn100ooo0wwwww
            //BFEXTU <ea>{Do:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_100_111:
            irpBfextu ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFCHG <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-00000ooooo0wwwww
            //BFCHG <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-00000ooooo100www
            //BFCHG <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-0000100ooo0wwwww
            //BFCHG <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-0000100ooo100www
          case 0b1110_101_011:
            irpBfchg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFEXTS <ea>{#o:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFEXTS <ea>{#o:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn0ooooo100www
            //BFEXTS <ea>{Do:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn100ooo0wwwww
            //BFEXTS <ea>{Do:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_101_111:
            irpBfexts ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFCLR <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-00000ooooo0wwwww
            //BFCLR <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-00000ooooo100www
            //BFCLR <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-0000100ooo0wwwww
            //BFCLR <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-0000100ooo100www
          case 0b1110_110_011:
            irpBfclr ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFFFO <ea>{#o:#w},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFFFO <ea>{#o:Dw},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn0ooooo100www
            //BFFFO <ea>{Do:#w},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn100ooo0wwwww
            //BFFFO <ea>{Do:Dw},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_110_111:
            irpBfffo ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFSET <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-00000ooooo0wwwww
            //BFSET <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-00000ooooo100www
            //BFSET <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-0000100ooo0wwwww
            //BFSET <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-0000100ooo100www
          case 0b1110_111_011:
            irpBfset ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFINS Dn,<ea>{#o:#w}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFINS Dn,<ea>{#o:Dw}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn0ooooo100www
            //BFINS Dn,<ea>{Do:#w}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn100ooo0wwwww
            //BFINS Dn,<ea>{Do:Dw}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_111_111:
            irpBfins ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //PFLUSHA                                         |-|---3--|P|-----|-----|          |1111_000_000_000_000-0010010000000000
            //PFLUSH SFC,#<mask>                              |-|---3--|P|-----|-----|          |1111_000_000_000_000-00110000mmm00000
            //PFLUSH DFC,#<mask>                              |-|---3--|P|-----|-----|          |1111_000_000_000_000-00110000mmm00001
            //PFLUSH Dn,#<mask>                               |-|---3--|P|-----|-----|          |1111_000_000_000_000-00110000mmm01nnn
            //PFLUSH #<data>,#<mask>                          |-|---3--|P|-----|-----|          |1111_000_000_000_000-00110000mmm10ddd
            //PMOVE.L <ea>,TTn                                |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00001n0000000000
            //PMOVEFD.L <ea>,TTn                              |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00001n0100000000
            //PMOVE.L TTn,<ea>                                |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00001n1000000000
            //PLOADW SFC,<ea>                                 |-|--M3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010000000000000
            //PLOADW DFC,<ea>                                 |-|--M3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010000000000001
            //PLOADW Dn,<ea>                                  |-|--M3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010000000001nnn
            //PLOADW #<data>,<ea>                             |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010000000010ddd
            //PLOADR SFC,<ea>                                 |-|--M3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010001000000000
            //PLOADR DFC,<ea>                                 |-|--M3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010001000000001
            //PLOADR Dn,<ea>                                  |-|--M3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010001000001nnn
            //PLOADR #<data>,<ea>                             |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0010001000010ddd
            //PFLUSH SFC,#<mask>,<ea>                         |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00111000mmm00000
            //PFLUSH DFC,#<mask>,<ea>                         |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00111000mmm00001
            //PFLUSH Dn,#<mask>,<ea>                          |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00111000mmm01nnn
            //PFLUSH #<data>,#<mask>,<ea>                     |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-00111000mmm10ddd
            //PMOVE.L <ea>,TC                                 |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100000000000000
            //PMOVEFD.L <ea>,TC                               |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100000100000000
            //PMOVE.L TC,<ea>                                 |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100001000000000
            //PMOVE.Q <ea>,SRP                                |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100100000000000
            //PMOVEFD.Q <ea>,SRP                              |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100100100000000
            //PMOVE.Q SRP,<ea>                                |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100101000000000
            //PMOVE.Q <ea>,CRP                                |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100110000000000
            //PMOVEFD.Q <ea>,CRP                              |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100110100000000
            //PMOVE.Q CRP,<ea>                                |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0100111000000000
            //PMOVE.W <ea>,MMUSR                              |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0110000000000000
            //PMOVE.W MMUSR,<ea>                              |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-0110001000000000
            //PTESTW SFC,<ea>,#<level>                        |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll0000000000
            //PTESTW DFC,<ea>,#<level>                        |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll0000000001
            //PTESTW Dn,<ea>,#<level>                         |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll0000001nnn
            //PTESTW #<data>,<ea>,#<level>                    |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll0000010ddd
            //PTESTW SFC,<ea>,#<level>,An                     |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll01nnn00000
            //PTESTW DFC,<ea>,#<level>,An                     |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll01nnn00001
            //PTESTW Dn,<ea>,#<level>,An                      |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll01nnn01nnn
            //PTESTW #<data>,<ea>,#<level>,An                 |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll01nnn10ddd
            //PTESTR SFC,<ea>,#<level>                        |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll1000000000
            //PTESTR DFC,<ea>,#<level>                        |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll1000000001
            //PTESTR Dn,<ea>,#<level>                         |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll1000001nnn
            //PTESTR #<data>,<ea>,#<level>                    |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll1000010ddd
            //PTESTR SFC,<ea>,#<level>,An                     |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll11nnn00000
            //PTESTR DFC,<ea>,#<level>,An                     |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll11nnn00001
            //PTESTR Dn,<ea>,#<level>,An                      |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll11nnn01nnn
            //PTESTR #<data>,<ea>,#<level>,An                 |-|---3--|P|-----|-----|  M  WXZ  |1111_000_000_mmm_rrr-100lll11nnn10ddd
          case 0b1111_000_000:
            irpPgen ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FTST.X FPm                                      |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmm0000111010
            //FMOVE.X FPm,FPn                                 |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000000
            //FINT.X FPm,FPn                                  |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000001
            //FSINH.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000010
            //FINTRZ.X FPm,FPn                                |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000011
            //FSQRT.X FPm,FPn                                 |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000100
            //FLOGNP1.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000110
            //FETOXM1.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001000
            //FTANH.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001001
            //FATAN.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001010
            //FASIN.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001100
            //FATANH.X FPm,FPn                                |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001101
            //FSIN.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001110
            //FTAN.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001111
            //FETOX.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010000
            //FTWOTOX.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010001
            //FTENTOX.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010010
            //FLOGN.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010100
            //FLOG10.X FPm,FPn                                |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010101
            //FLOG2.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010110
            //FABS.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011000
            //FCOSH.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011001
            //FNEG.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011010
            //FACOS.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011100
            //FCOS.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011101
            //FGETEXP.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011110
            //FGETMAN.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011111
            //FDIV.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100000
            //FMOD.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100001
            //FADD.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100010
            //FMUL.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100011
            //FSGLDIV.X FPm,FPn                               |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100100
            //FREM.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100101
            //FSCALE.X FPm,FPn                                |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100110
            //FSGLMUL.X FPm,FPn                               |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100111
            //FSUB.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0101000
            //FCMP.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0111000
            //FSINCOS.X FPm,FPc:FPs                           |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmsss0110ccc
            //FMOVECR.X #ccc,FPn                              |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-010111nnn0cccccc
            //FMOVE.L FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011000nnn0000000
            //FMOVE.S FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011001nnn0000000
            //FMOVE.W FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011100nnn0000000
            //FMOVE.B FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011110nnn0000000
            //FMOVE.L FPIAR,<ea>                              |-|--CC46|-|-----|-----|DAM+-WXZ  |1111_001_000_mmm_rrr-1010010000000000
            //FMOVEM.L FPIAR,<ea>                             |-|--CC46|-|-----|-----|DAM+-WXZ  |1111_001_000_mmm_rrr-1010010000000000
            //FMOVE.L FPSR,<ea>                               |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1010100000000000
            //FMOVEM.L FPSR,<ea>                              |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1010100000000000
            //FMOVE.L FPCR,<ea>                               |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1011000000000000
            //FMOVEM.L FPCR,<ea>                              |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1011000000000000
            //FTST.L <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0100000000111010
            //FMOVE.L <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000000
            //FINT.L <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000001
            //FSINH.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000010
            //FINTRZ.L <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000011
            //FSQRT.L <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000100
            //FLOGNP1.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000110
            //FETOXM1.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001000
            //FTANH.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001001
            //FATAN.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001010
            //FASIN.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001100
            //FATANH.L <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001101
            //FSIN.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001110
            //FTAN.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001111
            //FETOX.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010000
            //FTWOTOX.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010001
            //FTENTOX.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010010
            //FLOGN.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010100
            //FLOG10.L <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010101
            //FLOG2.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010110
            //FABS.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011000
            //FCOSH.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011001
            //FNEG.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011010
            //FACOS.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011100
            //FCOS.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011101
            //FGETEXP.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011110
            //FGETMAN.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011111
            //FDIV.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100000
            //FMOD.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100001
            //FADD.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100010
            //FMUL.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100011
            //FSGLDIV.L <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100100
            //FREM.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100101
            //FSCALE.L <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100110
            //FSGLMUL.L <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100111
            //FSUB.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0101000
            //FCMP.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0111000
            //FSINCOS.L <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000sss0110ccc
            //FTST.S <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0100010000111010
            //FMOVE.S <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000000
            //FINT.S <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000001
            //FSINH.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000010
            //FINTRZ.S <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000011
            //FSQRT.S <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000100
            //FLOGNP1.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000110
            //FETOXM1.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001000
            //FTANH.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001001
            //FATAN.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001010
            //FASIN.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001100
            //FATANH.S <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001101
            //FSIN.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001110
            //FTAN.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001111
            //FETOX.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010000
            //FTWOTOX.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010001
            //FTENTOX.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010010
            //FLOGN.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010100
            //FLOG10.S <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010101
            //FLOG2.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010110
            //FABS.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011000
            //FCOSH.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011001
            //FNEG.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011010
            //FACOS.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011100
            //FCOS.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011101
            //FGETEXP.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011110
            //FGETMAN.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011111
            //FDIV.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100000
            //FMOD.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100001
            //FADD.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100010
            //FMUL.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100011
            //FSGLDIV.S <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100100
            //FREM.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100101
            //FSCALE.S <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100110
            //FSGLMUL.S <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100111
            //FSUB.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0101000
            //FCMP.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0111000
            //FSINCOS.S <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001sss0110ccc
            //FTST.W <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0101000000111010
            //FMOVE.W <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000000
            //FINT.W <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000001
            //FSINH.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000010
            //FINTRZ.W <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000011
            //FSQRT.W <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000100
            //FLOGNP1.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000110
            //FETOXM1.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001000
            //FTANH.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001001
            //FATAN.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001010
            //FASIN.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001100
            //FATANH.W <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001101
            //FSIN.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001110
            //FTAN.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001111
            //FETOX.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010000
            //FTWOTOX.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010001
            //FTENTOX.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010010
            //FLOGN.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010100
            //FLOG10.W <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010101
            //FLOG2.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010110
            //FABS.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011000
            //FCOSH.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011001
            //FNEG.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011010
            //FACOS.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011100
            //FCOS.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011101
            //FGETEXP.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011110
            //FGETMAN.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011111
            //FDIV.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100000
            //FMOD.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100001
            //FADD.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100010
            //FMUL.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100011
            //FSGLDIV.W <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100100
            //FREM.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100101
            //FSCALE.W <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100110
            //FSGLMUL.W <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100111
            //FSUB.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0101000
            //FCMP.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0111000
            //FSINCOS.W <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100sss0110ccc
            //FTST.B <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0101100000111010
            //FMOVE.B <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000000
            //FINT.B <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000001
            //FSINH.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000010
            //FINTRZ.B <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000011
            //FSQRT.B <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000100
            //FLOGNP1.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000110
            //FETOXM1.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001000
            //FTANH.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001001
            //FATAN.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001010
            //FASIN.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001100
            //FATANH.B <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001101
            //FSIN.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001110
            //FTAN.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001111
            //FETOX.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010000
            //FTWOTOX.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010001
            //FTENTOX.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010010
            //FLOGN.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010100
            //FLOG10.B <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010101
            //FLOG2.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010110
            //FABS.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011000
            //FCOSH.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011001
            //FNEG.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011010
            //FACOS.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011100
            //FCOS.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011101
            //FGETEXP.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011110
            //FGETMAN.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011111
            //FDIV.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100000
            //FMOD.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100001
            //FADD.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100010
            //FMUL.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100011
            //FSGLDIV.B <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100100
            //FREM.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100101
            //FSCALE.B <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100110
            //FSGLMUL.B <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100111
            //FSUB.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0101000
            //FCMP.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0111000
            //FSINCOS.B <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110sss0110ccc
            //FMOVE.L <ea>,FPIAR                              |-|--CC46|-|-----|-----|DAM+-WXZPI|1111_001_000_mmm_rrr-1000010000000000
            //FMOVEM.L <ea>,FPIAR                             |-|--CC46|-|-----|-----|DAM+-WXZPI|1111_001_000_mmm_rrr-1000010000000000
            //FMOVE.L <ea>,FPSR                               |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1000100000000000
            //FMOVEM.L <ea>,FPSR                              |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1000100000000000
            //FMOVE.L <ea>,FPCR                               |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1001000000000000
            //FMOVEM.L <ea>,FPCR                              |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1001000000000000
            //FMOVE.X FPn,<ea>                                |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011010nnn0000000
            //FMOVE.P FPn,<ea>{#k}                            |-|--CCSS|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011011nnnkkkkkkk
            //FMOVE.D FPn,<ea>                                |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011101nnn0000000
            //FMOVE.P FPn,<ea>{Dk}                            |-|--CCSS|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011111nnnkkk0000
            //FMOVEM.L FPSR/FPIAR,<ea>                        |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1010110000000000
            //FMOVEM.L FPCR/FPIAR,<ea>                        |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1011010000000000
            //FMOVEM.L FPCR/FPSR,<ea>                         |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1011100000000000
            //FMOVEM.L FPCR/FPSR/FPIAR,<ea>                   |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1011110000000000
            //FMOVEM.X #<data>,<ea>                           |-|--CC46|-|-----|-----|  M  WXZ  |1111_001_000_mmm_rrr-11110000dddddddd
            //FMOVEM.X <list>,<ea>                            |-|--CC46|-|-----|-----|  M  WXZ  |1111_001_000_mmm_rrr-11110000llllllll
            //FMOVEM.X Dl,<ea>                                |-|--CC4S|-|-----|-----|  M  WXZ  |1111_001_000_mmm_rrr-111110000lll0000
            //FMOVEM.L <ea>,FPSR/FPIAR                        |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1000110000000000
            //FMOVEM.L <ea>,FPCR/FPIAR                        |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1001010000000000
            //FMOVEM.L <ea>,FPCR/FPSR                         |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1001100000000000
            //FMOVEM.L <ea>,FPCR/FPSR/FPIAR                   |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1001110000000000
            //FMOVEM.X <ea>,#<data>                           |-|--CC46|-|-----|-----|  M+ WXZP |1111_001_000_mmm_rrr-11010000dddddddd
            //FMOVEM.X <ea>,<list>                            |-|--CC46|-|-----|-----|  M+ WXZP |1111_001_000_mmm_rrr-11010000llllllll
            //FMOVEM.X <ea>,Dl                                |-|--CC4S|-|-----|-----|  M+ WXZP |1111_001_000_mmm_rrr-110110000lll0000
            //FTST.X <ea>                                     |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-0100100000111010
            //FMOVE.X <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000000
            //FINT.X <ea>,FPn                                 |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000001
            //FSINH.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000010
            //FINTRZ.X <ea>,FPn                               |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000011
            //FSQRT.X <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000100
            //FLOGNP1.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000110
            //FETOXM1.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001000
            //FTANH.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001001
            //FATAN.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001010
            //FASIN.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001100
            //FATANH.X <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001101
            //FSIN.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001110
            //FTAN.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001111
            //FETOX.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010000
            //FTWOTOX.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010001
            //FTENTOX.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010010
            //FLOGN.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010100
            //FLOG10.X <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010101
            //FLOG2.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010110
            //FABS.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011000
            //FCOSH.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011001
            //FNEG.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011010
            //FACOS.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011100
            //FCOS.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011101
            //FGETEXP.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011110
            //FGETMAN.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011111
            //FDIV.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100000
            //FMOD.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100001
            //FADD.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100010
            //FMUL.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100011
            //FSGLDIV.X <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100100
            //FREM.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100101
            //FSCALE.X <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100110
            //FSGLMUL.X <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100111
            //FSUB.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0101000
            //FCMP.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0111000
            //FSINCOS.X <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010sss0110ccc
            //FTST.P <ea>                                     |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-0100110000111010
            //FMOVE.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000000
            //FINT.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000001
            //FSINH.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000010
            //FINTRZ.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000011
            //FSQRT.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000100
            //FLOGNP1.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000110
            //FETOXM1.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001000
            //FTANH.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001001
            //FATAN.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001010
            //FASIN.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001100
            //FATANH.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001101
            //FSIN.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001110
            //FTAN.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001111
            //FETOX.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010000
            //FTWOTOX.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010001
            //FTENTOX.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010010
            //FLOGN.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010100
            //FLOG10.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010101
            //FLOG2.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010110
            //FABS.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011000
            //FCOSH.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011001
            //FNEG.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011010
            //FACOS.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011100
            //FCOS.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011101
            //FGETEXP.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011110
            //FGETMAN.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011111
            //FDIV.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100000
            //FMOD.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100001
            //FADD.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100010
            //FMUL.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100011
            //FSGLDIV.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100100
            //FREM.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100101
            //FSCALE.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100110
            //FSGLMUL.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100111
            //FSUB.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0101000
            //FCMP.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0111000
            //FSINCOS.P <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011sss0110ccc
            //FTST.D <ea>                                     |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-0101010000111010
            //FMOVE.D <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000000
            //FINT.D <ea>,FPn                                 |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000001
            //FSINH.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000010
            //FINTRZ.D <ea>,FPn                               |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000011
            //FSQRT.D <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000100
            //FLOGNP1.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000110
            //FETOXM1.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001000
            //FTANH.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001001
            //FATAN.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001010
            //FASIN.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001100
            //FATANH.D <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001101
            //FSIN.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001110
            //FTAN.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001111
            //FETOX.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010000
            //FTWOTOX.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010001
            //FTENTOX.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010010
            //FLOGN.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010100
            //FLOG10.D <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010101
            //FLOG2.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010110
            //FABS.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011000
            //FCOSH.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011001
            //FNEG.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011010
            //FACOS.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011100
            //FCOS.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011101
            //FGETEXP.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011110
            //FGETMAN.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011111
            //FDIV.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100000
            //FMOD.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100001
            //FADD.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100010
            //FMUL.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100011
            //FSGLDIV.D <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100100
            //FREM.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100101
            //FSCALE.D <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100110
            //FSGLMUL.D <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100111
            //FSUB.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0101000
            //FCMP.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0111000
            //FSINCOS.D <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101sss0110ccc
            //FMOVEM.X #<data>,-(Ar)                          |-|--CC46|-|-----|-----|    -     |1111_001_000_100_rrr-11100000dddddddd
            //FMOVEM.X <list>,-(Ar)                           |-|--CC46|-|-----|-----|    -     |1111_001_000_100_rrr-11100000llllllll
            //FMOVEM.X Dl,-(Ar)                               |-|--CC4S|-|-----|-----|    -     |1111_001_000_100_rrr-111010000lll0000
            //FMOVEM.L #<data>,#<data>,FPSR/FPIAR             |-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1000110000000000-{data}
            //FMOVEM.L #<data>,#<data>,FPCR/FPIAR             |-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1001010000000000-{data}
            //FMOVEM.L #<data>,#<data>,FPCR/FPSR              |-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1001100000000000-{data}
            //FMOVEM.L #<data>,#<data>,#<data>,FPCR/FPSR/FPIAR|-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1001110000000000-{data}
          case 0b1111_001_000:
            irpFgen ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FSF.B <ea>                                      |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000000
            //FSEQ.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000001
            //FSOGT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000010
            //FSOGE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000011
            //FSOLT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000100
            //FSOLE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000101
            //FSOGL.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000110
            //FSOR.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000111
            //FSUN.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001000
            //FSUEQ.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001001
            //FSUGT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001010
            //FSUGE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001011
            //FSULT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001100
            //FSULE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001101
            //FSNE.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001110
            //FST.B <ea>                                      |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001111
            //FSSF.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010000
            //FSSEQ.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010001
            //FSGT.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010010
            //FSGE.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010011
            //FSLT.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010100
            //FSLE.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010101
            //FSGL.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010110
            //FSGLE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010111
            //FSNGLE.B <ea>                                   |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011000
            //FSNGL.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011001
            //FSNLE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011010
            //FSNLT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011011
            //FSNGE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011100
            //FSNGT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011101
            //FSSNE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011110
            //FSST.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011111
            //FDBF Dr,<label>                                 |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000000-{offset}
            //FDBRA Dr,<label>                                |A|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000000-{offset}       [FDBF Dr,<label>]
            //FDBEQ Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000001-{offset}
            //FDBOGT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000010-{offset}
            //FDBOGE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000011-{offset}
            //FDBOLT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000100-{offset}
            //FDBOLE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000101-{offset}
            //FDBOGL Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000110-{offset}
            //FDBOR Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000111-{offset}
            //FDBUN Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001000-{offset}
            //FDBUEQ Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001001-{offset}
            //FDBUGT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001010-{offset}
            //FDBUGE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001011-{offset}
            //FDBULT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001100-{offset}
            //FDBULE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001101-{offset}
            //FDBNE Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001110-{offset}
            //FDBT Dr,<label>                                 |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001111-{offset}
            //FDBSF Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010000-{offset}
            //FDBSEQ Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010001-{offset}
            //FDBGT Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010010-{offset}
            //FDBGE Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010011-{offset}
            //FDBLT Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010100-{offset}
            //FDBLE Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010101-{offset}
            //FDBGL Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010110-{offset}
            //FDBGLE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010111-{offset}
            //FDBNGLE Dr,<label>                              |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011000-{offset}
            //FDBNGL Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011001-{offset}
            //FDBNLE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011010-{offset}
            //FDBNLT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011011-{offset}
            //FDBNGE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011100-{offset}
            //FDBNGT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011101-{offset}
            //FDBSNE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011110-{offset}
            //FDBST Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011111-{offset}
            //FTRAPF.W #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000000-{data}
            //FTRAPEQ.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000001-{data}
            //FTRAPOGT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000010-{data}
            //FTRAPOGE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000011-{data}
            //FTRAPOLT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000100-{data}
            //FTRAPOLE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000101-{data}
            //FTRAPOGL.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000110-{data}
            //FTRAPOR.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000111-{data}
            //FTRAPUN.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001000-{data}
            //FTRAPUEQ.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001001-{data}
            //FTRAPUGT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001010-{data}
            //FTRAPUGE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001011-{data}
            //FTRAPULT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001100-{data}
            //FTRAPULE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001101-{data}
            //FTRAPNE.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001110-{data}
            //FTRAPT.W #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001111-{data}
            //FTRAPSF.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010000-{data}
            //FTRAPSEQ.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010001-{data}
            //FTRAPGT.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010010-{data}
            //FTRAPGE.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010011-{data}
            //FTRAPLT.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010100-{data}
            //FTRAPLE.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010101-{data}
            //FTRAPGL.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010110-{data}
            //FTRAPGLE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010111-{data}
            //FTRAPNGLE.W #<data>                             |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011000-{data}
            //FTRAPNGL.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011001-{data}
            //FTRAPNLE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011010-{data}
            //FTRAPNLT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011011-{data}
            //FTRAPNGE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011100-{data}
            //FTRAPNGT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011101-{data}
            //FTRAPSNE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011110-{data}
            //FTRAPST.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011111-{data}
            //FTRAPF.L #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000000-{data}
            //FTRAPEQ.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000001-{data}
            //FTRAPOGT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000010-{data}
            //FTRAPOGE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000011-{data}
            //FTRAPOLT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000100-{data}
            //FTRAPOLE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000101-{data}
            //FTRAPOGL.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000110-{data}
            //FTRAPOR.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000111-{data}
            //FTRAPUN.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001000-{data}
            //FTRAPUEQ.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001001-{data}
            //FTRAPUGT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001010-{data}
            //FTRAPUGE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001011-{data}
            //FTRAPULT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001100-{data}
            //FTRAPULE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001101-{data}
            //FTRAPNE.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001110-{data}
            //FTRAPT.L #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001111-{data}
            //FTRAPSF.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010000-{data}
            //FTRAPSEQ.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010001-{data}
            //FTRAPGT.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010010-{data}
            //FTRAPGE.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010011-{data}
            //FTRAPLT.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010100-{data}
            //FTRAPLE.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010101-{data}
            //FTRAPGL.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010110-{data}
            //FTRAPGLE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010111-{data}
            //FTRAPNGLE.L #<data>                             |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011000-{data}
            //FTRAPNGL.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011001-{data}
            //FTRAPNLE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011010-{data}
            //FTRAPNLT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011011-{data}
            //FTRAPNGE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011100-{data}
            //FTRAPNGT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011101-{data}
            //FTRAPSNE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011110-{data}
            //FTRAPST.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011111-{data}
            //FTRAPF                                          |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000000
            //FTRAPEQ                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000001
            //FTRAPOGT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000010
            //FTRAPOGE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000011
            //FTRAPOLT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000100
            //FTRAPOLE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000101
            //FTRAPOGL                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000110
            //FTRAPOR                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000111
            //FTRAPUN                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001000
            //FTRAPUEQ                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001001
            //FTRAPUGT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001010
            //FTRAPUGE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001011
            //FTRAPULT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001100
            //FTRAPULE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001101
            //FTRAPNE                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001110
            //FTRAPT                                          |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001111
            //FTRAPSF                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010000
            //FTRAPSEQ                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010001
            //FTRAPGT                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010010
            //FTRAPGE                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010011
            //FTRAPLT                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010100
            //FTRAPLE                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010101
            //FTRAPGL                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010110
            //FTRAPGLE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010111
            //FTRAPNGLE                                       |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011000
            //FTRAPNGL                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011001
            //FTRAPNLE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011010
            //FTRAPNLT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011011
            //FTRAPNGE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011100
            //FTRAPNGT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011101
            //FTRAPSNE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011110
            //FTRAPST                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011111
          case 0b1111_001_001:
            irpFscc ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FNOP                                            |A|--CC46|-|-----|-----|          |1111_001_010_000_000-0000000000000000        [FBF.W (*)+2]
            //FBF.W <label>                                   |-|--CC46|-|-----|-----|          |1111_001_010_000_000-{offset}
            //FBEQ.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_000_001-{offset}
            //FBOGT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_010-{offset}
            //FBOGE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_011-{offset}
            //FBOLT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_100-{offset}
            //FBOLE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_101-{offset}
            //FBOGL.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_110-{offset}
            //FBOR.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_000_111-{offset}
            //FBUN.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_001_000-{offset}
            //FBUEQ.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_001-{offset}
            //FBUGT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_010-{offset}
            //FBUGE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_011-{offset}
            //FBULT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_100-{offset}
            //FBULE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_101-{offset}
            //FBNE.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_001_110-{offset}
            //FBT.W <label>                                   |-|--CC46|-|-----|-----|          |1111_001_010_001_111-{offset}
            //FBRA.W <label>                                  |A|--CC46|-|-----|-----|          |1111_001_010_001_111-{offset}        [FBT.W <label>]
            //FBSF.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_000-{offset}
            //FBSEQ.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_010_001-{offset}
            //FBGT.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_010-{offset}
            //FBGE.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_011-{offset}
            //FBLT.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_100-{offset}
            //FBLE.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_101-{offset}
            //FBGL.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_110-{offset}
            //FBGLE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_010_111-{offset}
            //FBNGLE.W <label>                                |-|--CC46|-|-----|-----|          |1111_001_010_011_000-{offset}
            //FBNGL.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_001-{offset}
            //FBNLE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_010-{offset}
            //FBNLT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_011-{offset}
            //FBNGE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_100-{offset}
            //FBNGT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_101-{offset}
            //FBSNE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_110-{offset}
            //FBST.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_011_111-{offset}
          case 0b1111_001_010:
            irpFbccWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FBF.L <label>                                   |-|--CC46|-|-----|-----|          |1111_001_011_000_000-{offset}
            //FBEQ.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_000_001-{offset}
            //FBOGT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_010-{offset}
            //FBOGE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_011-{offset}
            //FBOLT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_100-{offset}
            //FBOLE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_101-{offset}
            //FBOGL.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_110-{offset}
            //FBOR.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_000_111-{offset}
            //FBUN.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_001_000-{offset}
            //FBUEQ.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_001-{offset}
            //FBUGT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_010-{offset}
            //FBUGE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_011-{offset}
            //FBULT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_100-{offset}
            //FBULE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_101-{offset}
            //FBNE.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_001_110-{offset}
            //FBT.L <label>                                   |-|--CC46|-|-----|-----|          |1111_001_011_001_111-{offset}
            //FBRA.L <label>                                  |A|--CC46|-|-----|-----|          |1111_001_011_001_111-{offset}        [FBT.L <label>]
            //FBSF.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_000-{offset}
            //FBSEQ.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_010_001-{offset}
            //FBGT.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_010-{offset}
            //FBGE.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_011-{offset}
            //FBLT.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_100-{offset}
            //FBLE.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_101-{offset}
            //FBGL.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_110-{offset}
            //FBGLE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_010_111-{offset}
            //FBNGLE.L <label>                                |-|--CC46|-|-----|-----|          |1111_001_011_011_000-{offset}
            //FBNGL.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_001-{offset}
            //FBNLE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_010-{offset}
            //FBNLT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_011-{offset}
            //FBNGE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_100-{offset}
            //FBNGT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_101-{offset}
            //FBSNE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_110-{offset}
            //FBST.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_011_111-{offset}
          case 0b1111_001_011:
            irpFbccLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FSAVE <ea>                                      |-|--CC46|P|-----|-----|  M -WXZ  |1111_001_100_mmm_rrr
          case 0b1111_001_100:
            irpFsave ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FRESTORE <ea>                                   |-|--CC46|P|-----|-----|  M+ WXZP |1111_001_101_mmm_rrr
          case 0b1111_001_101:
            irpFrestore ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FPACK <data>                                    |A|012346|-|UUUUU|*****|          |1111_111_0dd_ddd_ddd [FLINE #<data>]
          case 0b1111_111_000:
          case 0b1111_111_001:
          case 0b1111_111_010:
          case 0b1111_111_011:
            irpFpack ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DOS <data>                                      |A|012346|-|UUUUU|UUUUU|          |1111_111_1dd_ddd_ddd [FLINE #<data>]
          case 0b1111_111_100:
          case 0b1111_111_101:
          case 0b1111_111_110:
          case 0b1111_111_111:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FLINE #<data>                                   |-|012346|-|UUUUU|UUUUU|          |1111_ddd_ddd_ddd_ddd (line 1111 emulator)
          case 0b1111_000_001:
          case 0b1111_000_010:
          case 0b1111_000_011:
          case 0b1111_000_100:
          case 0b1111_000_101:
          case 0b1111_000_110:
          case 0b1111_000_111:
          case 0b1111_001_110:
          case 0b1111_001_111:
          case 0b1111_010_000:
          case 0b1111_010_001:
          case 0b1111_010_010:
          case 0b1111_010_011:
          case 0b1111_010_100:
          case 0b1111_010_101:
          case 0b1111_010_110:
          case 0b1111_010_111:
          case 0b1111_011_000:
          case 0b1111_011_001:
          case 0b1111_011_010:
          case 0b1111_011_011:
          case 0b1111_011_100:
          case 0b1111_011_101:
          case 0b1111_011_110:
          case 0b1111_011_111:
          case 0b1111_100_000:
          case 0b1111_100_001:
          case 0b1111_100_010:
          case 0b1111_100_011:
          case 0b1111_100_100:
          case 0b1111_100_101:
          case 0b1111_100_110:
          case 0b1111_100_111:
          case 0b1111_101_000:
          case 0b1111_101_001:
          case 0b1111_101_010:
          case 0b1111_101_011:
          case 0b1111_101_100:
          case 0b1111_101_101:
          case 0b1111_101_110:
          case 0b1111_101_111:
          case 0b1111_110_000:
          case 0b1111_110_001:
          case 0b1111_110_010:
          case 0b1111_110_011:
          case 0b1111_110_100:
          case 0b1111_110_101:
          case 0b1111_110_110:
          case 0b1111_110_111:
            irpFline ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //HFSBOOT                                         |-|012346|-|-----|-----|          |0100_111_000_000_000
            //HFSINST                                         |-|012346|-|-----|-----|          |0100_111_000_000_001
            //HFSSTR                                          |-|012346|-|-----|-----|          |0100_111_000_000_010
            //HFSINT                                          |-|012346|-|-----|-----|          |0100_111_000_000_011
            //EMXNOP                                          |-|012346|-|-----|-----|          |0100_111_000_000_100
          case 0b0100_111_000:
            irpEmx ();
            break;

          default:
            irpIllegal ();

          }  //switch XEiJ.regOC >>> 6

          //トレース例外
          //  命令実行前にsrのTビットがセットされていたとき命令実行後にトレース例外が発生する
          //  トレース例外の発動は命令の機能拡張であり、他の例外処理で命令が中断されたときはトレース例外は発生しない
          //  命令例外はトレース例外の前に、割り込み例外はトレース例外の後に処理される
          //  未実装命令のエミュレーションルーチンはrteの直前にsrのTビットを復元することで未実装命令が1個の命令としてトレースされたように見せる
          //    ;DOSコールの終了
          //    ~008616:
          //            btst.b  #$07,(sp)
          //            bne.s   ~00861E
          //            rte
          //    ~00861E:
          //            ori.w   #$8000,sr
          //            rte
          if (XEiJ.mpuTraceFlag != 0) {  //命令実行前にsrのTビットがセットされていた
            XEiJ.mpuCycleCount += 34;
            irpException (M68kException.M6E_TRACE, XEiJ.regPC, XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR, 0x2000, XEiJ.regPC0);  //pcは次の命令
          }
          //クロックをカウントアップする
          //  オペランドをアクセスした時点ではまだXEiJ.mpuClockTimeが更新されていないのでXEiJ.mpuClockTime<xxxClock
          //  xxxTickを呼び出すときはXEiJ.mpuClockTime>=xxxClock
          XEiJ.mpuClockTime += XEiJ.mpuModifiedUnit * XEiJ.mpuCycleCount;
          //デバイスを呼び出す
          TickerQueue.tkqRun (XEiJ.mpuClockTime);
          //割り込みを受け付ける
          if ((t = XEiJ.mpuIMR & XEiJ.mpuIRR) != 0) {  //マスクされているレベルよりも高くて受け付けていない割り込みがあるとき
            if (XEiJ.MPU_INTERRUPT_SWITCH) {
              switch (t) {
              case 0b00000001:
              case 0b00000011:
              case 0b00000101:
              case 0b00000111:
              case 0b00001001:
              case 0b00001011:
              case 0b00001101:
              case 0b00001111:
              case 0b00010001:
              case 0b00010011:
              case 0b00010101:
              case 0b00010111:
              case 0b00011001:
              case 0b00011011:
              case 0b00011101:
              case 0b00011111:
              case 0b00100001:
              case 0b00100011:
              case 0b00100101:
              case 0b00100111:
              case 0b00101001:
              case 0b00101011:
              case 0b00101101:
              case 0b00101111:
              case 0b00110001:
              case 0b00110011:
              case 0b00110101:
              case 0b00110111:
              case 0b00111001:
              case 0b00111011:
              case 0b00111101:
              case 0b00111111:
              case 0b01000001:
              case 0b01000011:
              case 0b01000101:
              case 0b01000111:
              case 0b01001001:
              case 0b01001011:
              case 0b01001101:
              case 0b01001111:
              case 0b01010001:
              case 0b01010011:
              case 0b01010101:
              case 0b01010111:
              case 0b01011001:
              case 0b01011011:
              case 0b01011101:
              case 0b01011111:
              case 0b01100001:
              case 0b01100011:
              case 0b01100101:
              case 0b01100111:
              case 0b01101001:
              case 0b01101011:
              case 0b01101101:
              case 0b01101111:
              case 0b01110001:
              case 0b01110011:
              case 0b01110101:
              case 0b01110111:
              case 0b01111001:
              case 0b01111011:
              case 0b01111101:
              case 0b01111111:
              case 0b10000001:
              case 0b10000011:
              case 0b10000101:
              case 0b10000111:
              case 0b10001001:
              case 0b10001011:
              case 0b10001101:
              case 0b10001111:
              case 0b10010001:
              case 0b10010011:
              case 0b10010101:
              case 0b10010111:
              case 0b10011001:
              case 0b10011011:
              case 0b10011101:
              case 0b10011111:
              case 0b10100001:
              case 0b10100011:
              case 0b10100101:
              case 0b10100111:
              case 0b10101001:
              case 0b10101011:
              case 0b10101101:
              case 0b10101111:
              case 0b10110001:
              case 0b10110011:
              case 0b10110101:
              case 0b10110111:
              case 0b10111001:
              case 0b10111011:
              case 0b10111101:
              case 0b10111111:
              case 0b11000001:
              case 0b11000011:
              case 0b11000101:
              case 0b11000111:
              case 0b11001001:
              case 0b11001011:
              case 0b11001101:
              case 0b11001111:
              case 0b11010001:
              case 0b11010011:
              case 0b11010101:
              case 0b11010111:
              case 0b11011001:
              case 0b11011011:
              case 0b11011101:
              case 0b11011111:
              case 0b11100001:
              case 0b11100011:
              case 0b11100101:
              case 0b11100111:
              case 0b11101001:
              case 0b11101011:
              case 0b11101101:
              case 0b11101111:
              case 0b11110001:
              case 0b11110011:
              case 0b11110101:
              case 0b11110111:
              case 0b11111001:
              case 0b11111011:
              case 0b11111101:
              case 0b11111111:
                //レベル7
                XEiJ.mpuIRR &= ~XEiJ.MPU_SYS_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.sysAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_SYS_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00000010:
              case 0b00000110:
              case 0b00001010:
              case 0b00001110:
              case 0b00010010:
              case 0b00010110:
              case 0b00011010:
              case 0b00011110:
              case 0b00100010:
              case 0b00100110:
              case 0b00101010:
              case 0b00101110:
              case 0b00110010:
              case 0b00110110:
              case 0b00111010:
              case 0b00111110:
              case 0b01000010:
              case 0b01000110:
              case 0b01001010:
              case 0b01001110:
              case 0b01010010:
              case 0b01010110:
              case 0b01011010:
              case 0b01011110:
              case 0b01100010:
              case 0b01100110:
              case 0b01101010:
              case 0b01101110:
              case 0b01110010:
              case 0b01110110:
              case 0b01111010:
              case 0b01111110:
              case 0b10000010:
              case 0b10000110:
              case 0b10001010:
              case 0b10001110:
              case 0b10010010:
              case 0b10010110:
              case 0b10011010:
              case 0b10011110:
              case 0b10100010:
              case 0b10100110:
              case 0b10101010:
              case 0b10101110:
              case 0b10110010:
              case 0b10110110:
              case 0b10111010:
              case 0b10111110:
              case 0b11000010:
              case 0b11000110:
              case 0b11001010:
              case 0b11001110:
              case 0b11010010:
              case 0b11010110:
              case 0b11011010:
              case 0b11011110:
              case 0b11100010:
              case 0b11100110:
              case 0b11101010:
              case 0b11101110:
              case 0b11110010:
              case 0b11110110:
              case 0b11111010:
              case 0b11111110:
                //レベル6
                XEiJ.mpuIRR &= ~XEiJ.MPU_MFP_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = MC68901.mfpAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_MFP_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00000100:
              case 0b00001100:
              case 0b00010100:
              case 0b00011100:
              case 0b00100100:
              case 0b00101100:
              case 0b00110100:
              case 0b00111100:
              case 0b01000100:
              case 0b01001100:
              case 0b01010100:
              case 0b01011100:
              case 0b01100100:
              case 0b01101100:
              case 0b01110100:
              case 0b01111100:
              case 0b10000100:
              case 0b10001100:
              case 0b10010100:
              case 0b10011100:
              case 0b10100100:
              case 0b10101100:
              case 0b10110100:
              case 0b10111100:
              case 0b11000100:
              case 0b11001100:
              case 0b11010100:
              case 0b11011100:
              case 0b11100100:
              case 0b11101100:
              case 0b11110100:
              case 0b11111100:
                //レベル5
                XEiJ.mpuIRR &= ~XEiJ.MPU_SCC_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = Z8530.sccAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_SCC_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00010000:
              case 0b00110000:
              case 0b01010000:
              case 0b01110000:
              case 0b10010000:
              case 0b10110000:
              case 0b11010000:
              case 0b11110000:
                //レベル3
                XEiJ.mpuIRR &= ~XEiJ.MPU_DMA_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = HD63450.dmaAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_DMA_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00100000:
              case 0b01100000:
              case 0b10100000:
              case 0b11100000:
                //レベル2
                XEiJ.mpuIRR &= ~XEiJ.MPU_EB2_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.eb2Acknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_EB2_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b01000000:
              case 0b11000000:
                //レベル1
                XEiJ.mpuIRR &= ~XEiJ.MPU_IOI_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = IOInterrupt.ioiAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_IOI_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              }
            } else {
              t &= -t;
              //  x&=-xはxの最下位の1のビットだけを残す演算
              //  すなわちマスクされているレベルよりも高くて受け付けていない割り込みの中で最高レベルの割り込みのビットだけが残る
              //  最高レベルの割り込みのビットしか残っていないので、割り込みの有無をレベルの高い順ではなく使用頻度の高い順に調べられる
              //  MFPやDMAの割り込みがかかる度にそれより優先度の高いインタラプトスイッチが押されていないかどうかを確かめる必要がない
              if (t == XEiJ.MPU_MFP_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_MFP_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = MC68901.mfpAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_MFP_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_DMA_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_DMA_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = HD63450.dmaAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_DMA_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_SCC_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_SCC_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = Z8530.sccAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_SCC_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_IOI_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_IOI_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = IOInterrupt.ioiAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_IOI_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_EB2_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_EB2_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.eb2Acknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_EB2_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_SYS_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_SYS_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.sysAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t, XEiJ.MPU_SYS_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              }
            }
          }  //if t!=0
          if (MC68901.MFP_DELAYED_INTERRUPT) {
            XEiJ.mpuIRR |= XEiJ.mpuDIRR;  //遅延割り込み要求
            XEiJ.mpuDIRR = 0;
          }
        }  //命令ループ
      } catch (M68kException e) {
        if (M68kException.m6eNumber == M68kException.M6E_WAIT_EXCEPTION) {  //待機例外
          if (irpWaitException ()) {
            continue;
          } else {
            break errorLoop;
          }
        }
        if (M68kException.m6eNumber == M68kException.M6E_INSTRUCTION_BREAK_POINT) {  //命令ブレークポイントによる停止
          XEiJ.regPC = XEiJ.regPC0;
          XEiJ.mpuStop1 (null);  //"Instruction Break Point"
          break errorLoop;
        }
        //例外処理
        //  ここで処理するのはベクタ番号が2～31の例外に限る。TRAP #n命令はインライン展開する
        //  例外処理のサイクル数はBUS_ERRORとADDRESS_ERROR以外は34になっているので必要ならば補正してからthrowする
        //  使用頻度が高いと思われる例外はインライン展開するのでここには来ない
        //    例外処理をインライン展開する場合はMC68000とMC68030のコードを分けなければならずコードが冗長になる
        //    使用頻度が低いと思われる例外はインライン展開しない
        //  セーブされるpcは以下の例外は命令の先頭、これ以外は次の命令
        //     2  BUS_ERROR
        //     3  ADDRESS_ERROR
        //     4  ILLEGAL_INSTRUCTION
        //     8  PRIVILEGE_VIOLATION
        //    10  LINE_1010_EMULATOR
        //    11  LINE_1111_EMULATOR
        //                                      fedcba9876543210fedcba9876543210
        //if ((1 << M68kException.m6eNumber & 0b00000000000000000000110100011100) != 0) {
        //    0123456789abcdef0123456789abcdef
        if (0b00111000101100000000000000000000 << M68kException.m6eNumber < 0) {
          XEiJ.regPC = XEiJ.regPC0;  //セーブされるpcは命令の先頭
        }
        try {
          int save_sr = XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
          int sp = XEiJ.regRn[15];
          XEiJ.regSRT1 = XEiJ.regSRT0 = 0;  //srのTビットを消す
          if (XEiJ.regSRS == 0) {  //ユーザモードのとき
            XEiJ.regSRS = XEiJ.REG_SR_S;  //スーパーバイザモードへ移行する
            XEiJ.mpuUSP = sp;  //USPを保存
            sp = XEiJ.regSRM != 0 ? XEiJ.mpuMSP : XEiJ.mpuISP;  //SSPを復元
            if (DataBreakPoint.DBP_ON) {
              DataBreakPoint.dbpMemoryMap = DataBreakPoint.dbpSuperMap;  //スーパーバイザメモリマップに切り替える
            } else {
              XEiJ.busMemoryMap = XEiJ.busSuperMap;  //スーパーバイザメモリマップに切り替える
            }
            if (InstructionBreakPoint.IBP_ON) {
              InstructionBreakPoint.ibpOp1MemoryMap = InstructionBreakPoint.ibpOp1SuperMap;
            }
          }
          if (M68kException.m6eNumber <= M68kException.M6E_ADDRESS_ERROR) {
            //ホストファイルシステムのデバイスコマンドを強制終了させる
            HFS.hfsState = HFS.HFS_STATE_IDLE;
            XEiJ.mpuClockTime += 50 * XEiJ.mpuModifiedUnit;
            if (false) {
              //FORMAT $Aの例外スタックフレームを作る
              //  命令境界のバスエラーまたはアドレスエラー
              XEiJ.regRn[15] = sp -= 32;
              XEiJ.busWl (sp + 28, 0);  //31-30:内部レジスタ,29-28:内部レジスタ
              XEiJ.busWl (sp + 24, 0);  //27-24:データ出力バッファ
              XEiJ.busWl (sp + 20, 0);  //23-22:内部レジスタ,21-20:内部レジスタ
              XEiJ.busWl (sp + 16, M68kException.m6eAddress);  //19-16:データサイクルフォルトアドレス
              XEiJ.busWl (sp + 12, 0);  //15-14:命令パイプステージB,13-12:命令パイプステージC
              XEiJ.busWw (sp + 10,
                          M68kException.m6eDirection << 6 |
                          (M68kException.m6eSize == XEiJ.MPU_SS_BYTE ? 1 << 4 :
                           M68kException.m6eSize == XEiJ.MPU_SS_WORD ? 2 << 4 : 0));  //11-10:特殊ステータスワード
              XEiJ.busWw (sp + 8, 0);  //9-8:内部レジスタ
              XEiJ.busWw (sp + 6, 0xa000 | M68kException.m6eNumber << 2);  //7-6:フォーマットとベクタオフセット
            } else {
              //FORMAT $Bの例外スタックフレームを作る
              //  命令途中のバスエラーまたはアドレスエラー
              XEiJ.regRn[15] = sp -= 92;
              XEiJ.busWl (sp + 88, 0);  //91-58:内部レジスタ
              XEiJ.busWl (sp + 84, 0);
              XEiJ.busWl (sp + 80, 0);
              XEiJ.busWl (sp + 76, 0);
              XEiJ.busWl (sp + 72, 0);
              XEiJ.busWl (sp + 68, 0);
              XEiJ.busWl (sp + 64, 0);
              XEiJ.busWl (sp + 60, 0);
              XEiJ.busWl (sp + 56, 0);  //57-56:バージョンナンバーと内部情報
              XEiJ.busWl (sp + 52, 0);
              XEiJ.busWl (sp + 48, 0);  //53-48:内部レジスタ
              XEiJ.busWl (sp + 44, 0);  //47-44:データ入力バッファ
              XEiJ.busWl (sp + 40, 0);  //43-40:内部レジスタ
              XEiJ.busWl (sp + 36, 0);  //39-36:ステージBアドレス
              XEiJ.busWl (sp + 32, 0);  //35-28:内部レジスタ
              XEiJ.busWl (sp + 28, 0);
              XEiJ.busWl (sp + 24, 0);  //27-24:データ出力バッファ
              XEiJ.busWl (sp + 20, 0);  //23-22:内部レジスタ,21-20:内部レジスタ
              XEiJ.busWl (sp + 16, M68kException.m6eAddress);  //19-16:データサイクルフォルトアドレス
              XEiJ.busWl (sp + 12, 0);  //15-14:命令パイプステージB,13-12:命令パイプステージC
              XEiJ.busWw (sp + 10,
                          M68kException.m6eDirection << 6 |
                          (M68kException.m6eSize == XEiJ.MPU_SS_BYTE ? 1 << 4 :
                           M68kException.m6eSize == XEiJ.MPU_SS_WORD ? 2 << 4 : 0));  //11-10:特殊ステータスワード
              XEiJ.busWw (sp + 8, 0);  //9-8:内部レジスタ
              XEiJ.busWw (sp + 6, 0xb000 | M68kException.m6eNumber << 2);  //7-6:フォーマットとベクタオフセット
            }
            //                   111111111122222222223333333333444444444455555555556666
            //         0123456789012345678901234567890123456789012345678901234567890123
          } else if (0b0001011101000000000000000000000000000000000000000000000000000000L << M68kException.m6eNumber < 0L) {
            //FORMAT $2の例外スタックフレームを作る
            XEiJ.mpuClockTime += 34 * XEiJ.mpuModifiedUnit;
            XEiJ.regRn[15] = sp -= 12;
            XEiJ.busWl (sp + 8, M68kException.m6eAddress);  //11-8:命令アドレス
            XEiJ.busWw (sp + 6, 0x2000 | M68kException.m6eNumber << 2);  //7-6:フォーマットとベクタオフセット
          } else {
            //FORMAT $0の例外スタックフレームを作る
            XEiJ.mpuClockTime += 34 * XEiJ.mpuModifiedUnit;
            XEiJ.regRn[15] = sp -= 8;
            XEiJ.busWw (sp + 6, 0x0000 | M68kException.m6eNumber << 2);  //7-6:フォーマットとベクタオフセット
          }
          XEiJ.busWl (sp + 2, XEiJ.regPC);  //5-2:プログラムカウンタ
          XEiJ.busWw (sp, save_sr);  //1-0:ステータスレジスタ
          irpSetPC (XEiJ.busRlsf (XEiJ.mpuVBR + (M68kException.m6eNumber << 2)));  //例外ベクタを取り出してジャンプする
          if (XEiJ.dbgStopOnError) {  //エラーで停止する場合
            if (XEiJ.dbgDoStopOnError ()) {
              break errorLoop;
            }
          }
        } catch (M68kException ee) {  //ダブルバスフォルト
          XEiJ.dbgDoubleBusFault ();
          break errorLoop;
        }
      }  //catch M68kException
    }  //例外ループ

    //  通常
    //    pc0  最後に実行した命令
    //    pc  次に実行する命令
    //  バスエラー、アドレスエラー、不当命令、特権違反で停止したとき
    //    pc0  エラーを発生させた命令
    //    pc  例外処理ルーチンの先頭
    //  ダブルバスフォルトで停止したとき
    //    pc0  エラーを発生させた命令
    //    pc  エラーを発生させた命令
    //  命令ブレークポイントで停止したとき
    //    pc0  命令ブレークポイントが設定された、次に実行する命令
    //    pc  命令ブレークポイントが設定された、次に実行する命令
    //  データブレークポイントで停止したとき
    //    pc0  データを書き換えた、最後に実行した命令
    //    pc  次に実行する命令

    //分岐ログに停止レコードを記録する
    if (BranchLog.BLG_ON) {
      BranchLog.blgStop ();
    }

  }  //mpuCore()



  //cont = irpWaitException ()
  //  待機例外をキャッチしたとき
  public static boolean irpWaitException () {
    XEiJ.regPC = XEiJ.regPC0;  //PCを巻き戻す
    XEiJ.regRn[8 + (XEiJ.regOC & 7)] += WaitInstruction.REWIND_AR[XEiJ.regOC >> 3];  //(Ar)+|-(Ar)で変化したArを巻き戻す
    try {
      //トレース例外を処理する
      if (XEiJ.mpuTraceFlag != 0) {  //命令実行前にsrのTビットがセットされていた
        XEiJ.mpuCycleCount += 34;
        irpException (M68kException.M6E_TRACE, XEiJ.regPC, XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR, 0x2000, XEiJ.regPC0);  //pcは次の命令
      }
      //デバイスを呼び出す
      TickerQueue.tkqRun (XEiJ.mpuClockTime);
      //割り込みを受け付ける
      int t;
      if ((t = XEiJ.mpuIMR & XEiJ.mpuIRR) != 0) {  //マスクされているレベルよりも高くて受け付けていない割り込みがあるとき
        t &= -t;
        //  x&=-xはxの最下位の1のビットだけを残す演算
        //  すなわちマスクされているレベルよりも高くて受け付けていない割り込みの中で最高レベルの割り込みのビットだけが残る
        //  最高レベルの割り込みのビットしか残っていないので、割り込みの有無をレベルの高い順ではなく使用頻度の高い順に調べられる
        //  MFPやDMAの割り込みがかかる度にそれより優先度の高いインタラプトスイッチが押されていないかどうかを確かめる必要がない
        if (t == XEiJ.MPU_MFP_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_MFP_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = MC68901.mfpAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t, XEiJ.MPU_MFP_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_DMA_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_DMA_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = HD63450.dmaAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t, XEiJ.MPU_DMA_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_SCC_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_SCC_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = Z8530.sccAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t, XEiJ.MPU_SCC_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_IOI_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_IOI_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = IOInterrupt.ioiAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t, XEiJ.MPU_IOI_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_EB2_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_EB2_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = XEiJ.eb2Acknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t, XEiJ.MPU_EB2_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_SYS_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_SYS_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = XEiJ.sysAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t, XEiJ.MPU_SYS_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        }
      }  //if t!=0
      if (MC68901.MFP_DELAYED_INTERRUPT) {
        XEiJ.mpuIRR |= XEiJ.mpuDIRR;  //遅延割り込み要求
        XEiJ.mpuDIRR = 0;
      }
    } catch (M68kException e) {
      //!!! 待機例外処理中のバスエラーの処理は省略
      XEiJ.dbgDoubleBusFault ();
      return false;
    }  //catch M68kException
    return true;
  }  //irpWaitException



  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ORI.B #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_000_mmm_rrr-{data}
  //OR.B #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_000_mmm_rrr-{data}  [ORI.B #<data>,<ea>]
  //ORI.B #<data>,CCR                               |-|012346|-|*****|*****|          |0000_000_000_111_100-{data}
  public static void irpOriByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      z = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      z = XEiJ.regPC;
      XEiJ.regPC = z + 2;
      z = XEiJ.busRbs (z + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //ORI.B #<data>,Dr
      if (XEiJ.DBG_ORI_BYTE_ZERO_D0) {
        if (z == 0 && ea == 0 && XEiJ.dbgOriByteZeroD0) {  //ORI.B #$00,D0
          M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
          throw M68kException.m6eSignal;
        }
      }
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[ea] |= 255 & z;  //0拡張してからOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea == XEiJ.EA_IM) {  //ORI.B #<data>,CCR
      XEiJ.mpuCycleCount += 20;
      XEiJ.regCCR |= XEiJ.REG_CCR_MASK & z;
    } else {  //ORI.B #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z |= XEiJ.busRbs (a));
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpOriByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ORI.W #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_001_mmm_rrr-{data}
  //OR.W #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_001_mmm_rrr-{data}  [ORI.W #<data>,<ea>]
  //ORI.W #<data>,SR                                |-|012346|P|*****|*****|          |0000_000_001_111_100-{data}
  public static void irpOriWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //ORI.W #<data>,Dr
      int z;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        z = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        z = XEiJ.regPC;
        XEiJ.regPC = z + 2;
        z = XEiJ.busRwse (z);  //pcws
      }
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[ea] |= (char) z;  //0拡張してからOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea == XEiJ.EA_IM) {  //ORI.W #<data>,SR
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
        throw M68kException.m6eSignal;
      }
      //以下はスーパーバイザモード
      XEiJ.mpuCycleCount += 20;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        irpSetSR (XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR | XEiJ.busRwse ((XEiJ.regPC += 2) - 2));  //pcws。特権違反チェックが先
      } else {
        int t = XEiJ.regPC;
        XEiJ.regPC = t + 2;
        irpSetSR (XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR | XEiJ.busRwse (t));  //pcws。特権違反チェックが先
      }
    } else {  //ORI.W #<data>,<mem>
      int z;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        z = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        z = XEiJ.regPC;
        XEiJ.regPC = z + 2;
        z = XEiJ.busRwse (z);  //pcws
      }
      XEiJ.mpuCycleCount += 12;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z |= XEiJ.busRws (a));
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpOriWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ORI.L #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_010_mmm_rrr-{data}
  //OR.L #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_010_mmm_rrr-{data}  [ORI.L #<data>,<ea>]
  public static void irpOriLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 4;
      y = XEiJ.busRlse (y);  //pcls
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //ORI.L #<data>,Dr
      XEiJ.mpuCycleCount += 16;
      z = XEiJ.regRn[ea] |= y;
    } else {  //ORI.L #<data>,<mem>
      XEiJ.mpuCycleCount += 20;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = XEiJ.busRls (a) | y);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpOriLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BITREV.L Dr                                     |-|------|-|-----|-----|D         |0000_000_011_000_rrr (ISA_C)
  //CMP2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn000000000000
  //CHK2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn100000000000
  //
  //BITREV.L Dr
  //  Drのビットの並びを逆順にする。CCRは変化しない
  //
  //CHK2.B <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CHK2.B <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする。このときCHK instruction例外が発生する
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //CMP2.B <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CMP2.B <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  public static void irpCmp2Chk2Byte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //BITREV.L Dr
      XEiJ.mpuCycleCount += 4;
      if (XEiJ.IRP_BITREV_REVERSE) {  //2.83ns  0x0f801f3c
        XEiJ.regRn[ea] = Integer.reverse (XEiJ.regRn[ea]);
      } else if (XEiJ.IRP_BITREV_SHIFT) {  //2.57ns  0x0f801f3c
        int x = XEiJ.regRn[ea];
        x = x << 16 | x >>> 16;
        x = x << 8 & 0xff00ff00 | x >>> 8 & 0x00ff00ff;
        x = x << 4 & 0xf0f0f0f0 | x >>> 4 & 0x0f0f0f0f;
        x = x << 2 & 0xcccccccc | x >>> 2 & 0x33333333;
        XEiJ.regRn[ea] = x << 1 & 0xaaaaaaaa | x >>> 1 & 0x55555555;
      } else if (XEiJ.IRP_BITREV_TABLE) {  //1.57ns  0x0f801f3c
        int x = XEiJ.regRn[ea];
        XEiJ.regRn[ea] = XEiJ.MPU_BITREV_TABLE_0[x & 2047] | XEiJ.MPU_BITREV_TABLE_1[x << 10 >>> 21] | XEiJ.MPU_BITREV_TABLE_2[x >>> 22];
      }
    } else {  //CMP2/CHK2.B <ea>,Rn
      XEiJ.mpuCycleCount += 8;
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 2;
        w = XEiJ.busRwze (w);  //pcwz
      }
      int d = XEiJ.regRn[w >> 12];  //Rn
      if (0 <= (short) w) {  //Dnのとき
        d = (byte) d;  //符号拡張する
      }
      int a = efaCntByte (ea);
      int l = XEiJ.busRbs (a);  //LB
      int u = XEiJ.busRbs (a + 1);  //UB
      //U-D,L-D,D-Lのいずれかに帰着させる
      //  参考
      //    https://twitter.com/moveccr/status/814309539012976640
      //    https://twitter.com/moveccr/status/814309679845109760
      //    https://twitter.com/moveccr/status/814310106598871040
      int x, y;
      if (Integer.compareUnsigned (l, d) < 0 && Integer.compareUnsigned (l, u) <= 0 || d == u) {
        x = u;
        y = d;
      } else if (Integer.compareUnsigned (d, u) < 0 && Integer.compareUnsigned (u, l) < 0) {
        x = l;
        y = d;
      } else {
        x = d;
        y = l;
      }
      int z = x - y;
      int c = (x & (y ^ z) ^ (y | z)) >>> 31;
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //X
                     z >>> 28 & XEiJ.REG_CCR_N |  //N
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |  //Z
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |  //V
                     c);  //C
      if ((w >> 11 & c) != 0) {  //CHK2でCがセットされたとき
        XEiJ.mpuCycleCount += 40 - 8 - 34;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
    }
  }  //irpCmp2Chk2Byte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BTST.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_100_000_rrr
  //MOVEP.W (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_100_001_rrr-{data}
  //BTST.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZPI|0000_qqq_100_mmm_rrr
  public static void irpBtstReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9;  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.W (d16,Ar),Dq
      XEiJ.mpuCycleCount += 16;
      int a;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        a = XEiJ.regRn[ea] + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws。このr[ea]はアドレスレジスタ
      } else {
        a = XEiJ.regPC;
        XEiJ.regPC = a + 2;
        a = XEiJ.regRn[ea] + XEiJ.busRwse (a);  //pcws。このr[ea]はアドレスレジスタ
      }
      XEiJ.regRn[qqq] = ~0xffff & XEiJ.regRn[qqq] | XEiJ.busRbz (a) << 8 | XEiJ.busRbz (a + 2);  //Javaは評価順序が保証されている
    } else {  //BTST.L Dq,Dr/<ea>
      int y = XEiJ.regRn[qqq];
      if (ea < XEiJ.EA_AR) {  //BTST.L Dq,Dr
        XEiJ.mpuCycleCount += 6;
        XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~XEiJ.regRn[ea] >>> y & 1) << 2;  //ccr_btst。intのシフトは5bitでマスクされるので&31を省略
      } else {  //BTST.B Dq,<ea>
        XEiJ.mpuCycleCount += 4;
        XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~XEiJ.busRbs (efaAnyByte (ea)) >>> (y & 7) & 1) << 2;  //ccr_btst
      }
    }
  }  //irpBtstReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCHG.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_101_000_rrr
  //MOVEP.L (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_101_001_rrr-{data}
  //BCHG.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_101_mmm_rrr
  public static void irpBchgReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9;  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.L (d16,Ar),Dq
      XEiJ.mpuCycleCount += 24;
      int a;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        a = XEiJ.regRn[ea] + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws。このr[ea]はアドレスレジスタ
      } else {
        a = XEiJ.regPC;
        XEiJ.regPC = a + 2;
        a = XEiJ.regRn[ea] + XEiJ.busRwse (a);  //pcws。このr[ea]はアドレスレジスタ
      }
      XEiJ.regRn[qqq] = XEiJ.busRbs (a) << 24 | XEiJ.busRbz (a + 2) << 16 | XEiJ.busRbz (a + 4) << 8 | XEiJ.busRbz (a + 6);  //Javaは評価順序が保証されている
    } else {  //BCHG.L Dq,Dr/<ea>
      int x;
      int y = XEiJ.regRn[qqq];
      if (ea < XEiJ.EA_AR) {  //BCHG.L Dq,Dr
        XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) ^ (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
        XEiJ.mpuCycleCount += (char) y != 0 ? 6 : 8;  //(0xffff&y)!=0
      } else {  //BCHG.B Dq,<ea>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltByte (ea);
        XEiJ.busWb (a, (x = XEiJ.busRbs (a)) ^ (y = 1 << (y & 7)));
      }
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
    }
  }  //irpBchgReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCLR.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_110_000_rrr
  //MOVEP.W Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_110_001_rrr-{data}
  //BCLR.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_110_mmm_rrr
  public static void irpBclrReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = XEiJ.regRn[XEiJ.regOC >> 9];  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.W Dq,(d16,Ar)
      XEiJ.mpuCycleCount += 16;
      int a;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        a = XEiJ.regRn[ea] + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws。このr[ea]はアドレスレジスタ
      } else {
        a = XEiJ.regPC;
        XEiJ.regPC = a + 2;
        a = XEiJ.regRn[ea] + XEiJ.busRwse (a);  //pcws。このr[ea]はアドレスレジスタ
      }
      XEiJ.busWb (a, y >> 8);
      XEiJ.busWb (a + 2, y);
    } else {  //BCLR.L Dq,Dr/<ea>
      int x;
      if (ea < XEiJ.EA_AR) {  //BCLR.L Dq,Dr
        XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) & ~(y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
        XEiJ.mpuCycleCount += (char) y != 0 ? 8 : 10;  //(0xffff&y)!=0
      } else {  //BCLR.B Dq,<ea>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltByte (ea);
        XEiJ.busWb (a, (x = XEiJ.busRbs (a)) & ~(y = 1 << (y & 7)));
      }
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
    }
  }  //irpBclrReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSET.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_111_000_rrr
  //MOVEP.L Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_111_001_rrr-{data}
  //BSET.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_111_mmm_rrr
  public static void irpBsetReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = XEiJ.regRn[XEiJ.regOC >> 9];  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.L Dq,(d16,Ar)
      XEiJ.mpuCycleCount += 24;
      int a;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        a = XEiJ.regRn[ea] + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws。このr[ea]はアドレスレジスタ
      } else {
        a = XEiJ.regPC;
        XEiJ.regPC = a + 2;
        a = XEiJ.regRn[ea] + XEiJ.busRwse (a);  //pcws。このr[ea]はアドレスレジスタ
      }
      XEiJ.busWb (a, y >> 24);
      XEiJ.busWb (a + 2, y >> 16);
      XEiJ.busWb (a + 4, y >> 8);
      XEiJ.busWb (a + 6, y);
    } else {  //BSET.L Dq,Dr/<ea>
      int x;
      if (ea < XEiJ.EA_AR) {  //BSET.L Dq,Dr
        XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) | (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
        XEiJ.mpuCycleCount += (char) y != 0 ? 6 : 8;  //(0xffff&y)!=0
      } else {  //BSET.B Dq,<ea>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltByte (ea);
        XEiJ.busWb (a, (x = XEiJ.busRbs (a)) | (y = 1 << (y & 7)));
      }
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
    }
  }  //irpBsetReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ANDI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_000_mmm_rrr-{data}
  //AND.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_000_mmm_rrr-{data}  [ANDI.B #<data>,<ea>]
  //ANDI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_001_000_111_100-{data}
  public static void irpAndiByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      z = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      z = XEiJ.regPC;
      XEiJ.regPC = z + 2;
      z = XEiJ.busRbs (z + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //ANDI.B #<data>,Dr
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[ea] &= ~255 | z;  //1拡張してからAND
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea == XEiJ.EA_IM) {  //ANDI.B #<data>,CCR
      XEiJ.mpuCycleCount += 20;
      XEiJ.regCCR &= z;
    } else {  //ANDI.B #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z &= XEiJ.busRbs (a));
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpAndiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ANDI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_001_mmm_rrr-{data}
  //AND.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_001_mmm_rrr-{data}  [ANDI.W #<data>,<ea>]
  //ANDI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_001_001_111_100-{data}
  public static void irpAndiWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //ANDI.W #<data>,Dr
      int z;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        z = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        z = XEiJ.regPC;
        XEiJ.regPC = z + 2;
        z = XEiJ.busRwse (z);  //pcws
      }
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[ea] &= ~65535 | z;  //1拡張してからAND
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea == XEiJ.EA_IM) {  //ANDI.W #<data>,SR
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
        throw M68kException.m6eSignal;
      }
      //以下はスーパーバイザモード
      XEiJ.mpuCycleCount += 20;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        irpSetSR ((XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR) & XEiJ.busRwse ((XEiJ.regPC += 2) - 2));  //pcws。特権違反チェックが先
      } else {
        int t = XEiJ.regPC;
        XEiJ.regPC = t + 2;
        irpSetSR ((XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR) & XEiJ.busRwse (t));  //pcws。特権違反チェックが先
      }
    } else {  //ANDI.W #<data>,<mem>
      int z;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        z = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        z = XEiJ.regPC;
        XEiJ.regPC = z + 2;
        z = XEiJ.busRwse (z);  //pcws
      }
      XEiJ.mpuCycleCount += 12;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z &= XEiJ.busRws (a));
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpAndiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ANDI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_010_mmm_rrr-{data}
  //AND.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_010_mmm_rrr-{data}  [ANDI.L #<data>,<ea>]
  public static void irpAndiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 4;
      y = XEiJ.busRlse (y);  //pcls
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //ANDI.L #<data>,Dr
      XEiJ.mpuCycleCount += 16;
      z = XEiJ.regRn[ea] &= y;
    } else {  //ANDI.L #<data>,<mem>
      XEiJ.mpuCycleCount += 20;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = XEiJ.busRls (a) & y);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpAndiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BYTEREV.L Dr                                    |-|------|-|-----|-----|D         |0000_001_011_000_rrr (ISA_C)
  //CMP2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn000000000000
  //CHK2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn100000000000
  //
  //BYTEREV.L Dr
  //  Drのバイトの並びを逆順にする。CCRは変化しない
  //
  //CHK2.W <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CHK2.W <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする。このときCHK instruction例外が発生する
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //CMP2.W <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CMP2.W <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  public static void irpCmp2Chk2Word () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //BYTEREV.L Dr
      XEiJ.mpuCycleCount += 4;
      if (true) {  //0.10ns-0.18ns  0x782750ec
        XEiJ.regRn[ea] = Integer.reverseBytes (XEiJ.regRn[ea]);
      } else {  //1.06ns  0x782750ec
        int x = XEiJ.regRn[ea];
        XEiJ.regRn[ea] = x << 24 | x << 8 & 0x00ff0000 | x >>> 8 & 0x0000ff00 | x >>> 24;
      }
    } else {  //CMP2/CHK2.W <ea>,Rn
      XEiJ.mpuCycleCount += 8;
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 2;
        w = XEiJ.busRwze (w);  //pcwz
      }
      int d = XEiJ.regRn[w >> 12];  //Rn
      if (0 <= (short) w) {  //Dnのとき
        d = (short) d;  //符号拡張する
      }
      int a = efaCntWord (ea);
      int l = XEiJ.busRws (a);  //LB
      int u = XEiJ.busRws (a + 2);  //UB
      //U-D,L-D,D-Lのいずれかに帰着させる
      //  参考
      //    https://twitter.com/moveccr/status/814309539012976640
      //    https://twitter.com/moveccr/status/814309679845109760
      //    https://twitter.com/moveccr/status/814310106598871040
      int x, y;
      if (Integer.compareUnsigned (l, d) < 0 && Integer.compareUnsigned (l, u) <= 0 || d == u) {
        x = u;
        y = d;
      } else if (Integer.compareUnsigned (d, u) < 0 && Integer.compareUnsigned (u, l) < 0) {
        x = l;
        y = d;
      } else {
        x = d;
        y = l;
      }
      int z = x - y;
      int c = (x & (y ^ z) ^ (y | z)) >>> 31;
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //X
                     z >>> 28 & XEiJ.REG_CCR_N |  //N
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |  //Z
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |  //V
                     c);  //C
      if ((w >> 11 & c) != 0) {  //CHK2でCがセットされたとき
        XEiJ.mpuCycleCount += 40 - 8 - 34;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
    }
  }  //irpCmp2Chk2Word

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_000_mmm_rrr-{data}
  //SUB.B #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_000_mmm_rrr-{data}  [SUBI.B #<data>,<ea>]
  public static void irpSubiByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBI.B #<data>,Dr
      XEiJ.mpuCycleCount += 8;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) - y);
    } else {  //SUBI.B #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = (byte) ((x = XEiJ.busRbs (a)) - y));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_001_mmm_rrr-{data}
  //SUB.W #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_001_mmm_rrr-{data}  [SUBI.W #<data>,<ea>]
  public static void irpSubiWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRwse (y);  //pcws
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBI.W #<data>,Dr
      XEiJ.mpuCycleCount += 8;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) - y));
    } else {  //SUBI.W #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z = (short) ((x = XEiJ.busRws (a)) - y));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_010_mmm_rrr-{data}
  //SUB.L #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_010_mmm_rrr-{data}  [SUBI.L #<data>,<ea>]
  public static void irpSubiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 4;
      y = XEiJ.busRlse (y);  //pcls
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBI.L #<data>,Dr
      XEiJ.mpuCycleCount += 16;
      XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) - y;
    } else {  //SUBI.L #<data>,<mem>
      XEiJ.mpuCycleCount += 20;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) - y);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //FF1.L Dr                                        |-|------|-|-UUUU|-**00|D         |0000_010_011_000_rrr (ISA_C)
  //CMP2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn000000000000
  //CHK2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn100000000000
  //
  //CHK2.L <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする。このときCHK instruction例外が発生する
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //CMP2.L <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //FF1.L Dr
  //  Drの最上位の1のbit31からのオフセットをDrに格納する
  //  Drが0のときは32になる
  public static void irpCmp2Chk2Long () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //FF1.L Dr
      XEiJ.mpuCycleCount += 4;
      int z = XEiJ.regRn[ea];
      if (true) {
        XEiJ.regRn[ea] = Integer.numberOfLeadingZeros (z);
      } else {
        if (z == 0) {
          XEiJ.regRn[ea] = 32;
        } else {
          int k = -(z >>> 16) >> 16 & 16;
          k += -(z >>> k + 8) >> 8 & 8;
          k += -(z >>> k + 4) >> 4 & 4;
          //     bit3  1  1  1  1  1  1  1  1  0  0  0  0  0  0  0  0
          //     bit2  1  1  1  1  0  0  0  0  1  1  1  1  0  0  0  0
          //     bit1  1  1  0  0  1  1  0  0  1  1  0  0  1  1  0  0
          //     bit0  1  0  1  0  1  0  1  0  1  0  1  0  1  0  1  0
          XEiJ.regRn[ea] = ((0b11_11_11_11_11_11_11_11_10_10_10_10_01_01_00_00 >>> (z >>> k << 1)) & 3) + k;  //intのシフトカウントは下位5bitだけが使用される
        }
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //CMP2/CHK2.L <ea>,Rn
      XEiJ.mpuCycleCount += 8;
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 2;
        w = XEiJ.busRwze (w);  //pcwz
      }
      int d = XEiJ.regRn[w >> 12];  //Rn
      int a = efaCntLong (ea);
      int l = XEiJ.busRls (a);  //LB
      int u = XEiJ.busRls (a + 4);  //UB
      //U-D,L-D,D-Lのいずれかに帰着させる
      //  参考
      //    https://twitter.com/moveccr/status/814309539012976640
      //    https://twitter.com/moveccr/status/814309679845109760
      //    https://twitter.com/moveccr/status/814310106598871040
      int x, y;
      if (Integer.compareUnsigned (l, d) < 0 && Integer.compareUnsigned (l, u) <= 0 || d == u) {
        x = u;
        y = d;
      } else if (Integer.compareUnsigned (d, u) < 0 && Integer.compareUnsigned (u, l) < 0) {
        x = l;
        y = d;
      } else {
        x = d;
        y = l;
      }
      int z = x - y;
      int c = (x & (y ^ z) ^ (y | z)) >>> 31;
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //X
                     z >>> 28 & XEiJ.REG_CCR_N |  //N
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |  //Z
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |  //V
                     c);  //C
      if ((w >> 11 & c) != 0) {  //CHK2でCがセットされたとき
        XEiJ.mpuCycleCount += 40 - 8 - 34;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
    }
  }  //irpCmp2Chk2Long

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_000_mmm_rrr-{data}
  public static void irpAddiByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDI.B #<data>,Dr
      XEiJ.mpuCycleCount += 8;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) + y);
    } else {  //ADDI.B #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = (byte) ((x = XEiJ.busRbs (a)) + y));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >>> 31 << 1 |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_001_mmm_rrr-{data}
  public static void irpAddiWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRwse (y);  //pcws
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDI.W #<data>,Dr
      XEiJ.mpuCycleCount += 8;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) + y));
    } else {  //ADDI.W #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z = (short) ((x = XEiJ.busRws (a)) + y));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >>> 31 << 1 |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_010_mmm_rrr-{data}
  public static void irpAddiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 4;
      y = XEiJ.busRlse (y);  //pcls
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDI.L #<data>,Dr
      XEiJ.mpuCycleCount += 16;
      XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) + y;
    } else {  //ADDI.L #<data>,<mem>
      XEiJ.mpuCycleCount += 20;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) + y);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >>> 31 << 1 |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BTST.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_000_000_rrr-{data}
  //BTST.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZP |0000_100_000_mmm_rrr-{data}
  public static void irpBtstImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //BTST.L #<data>,Dr
      XEiJ.mpuCycleCount += 10;
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~XEiJ.regRn[ea] >>> y & 1) << 2;  //ccr_btst。intのシフトは5bitでマスクされるので&31を省略
    } else {  //BTST.B #<data>,<ea>
      XEiJ.mpuCycleCount += 8;
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~XEiJ.busRbs (efaMemByte (ea)) >>> (y & 7) & 1) << 2;  //ccr_btst
    }
  }  //irpBtstImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCHG.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_001_000_rrr-{data}
  //BCHG.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_001_mmm_rrr-{data}
  public static void irpBchgImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //BCHG.L #<data>,Dr
      XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) ^ (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
      XEiJ.mpuCycleCount += (char) y != 0 ? 10 : 12;  //(0xffff&y)!=0
    } else {  //BCHG.B #<data>,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, (x = XEiJ.busRbs (a)) ^ (y = 1 << (y & 7)));
    }
    XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
  }  //irpBchgImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCLR.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_010_000_rrr-{data}
  //BCLR.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_010_mmm_rrr-{data}
  public static void irpBclrImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //BCLR.L #<data>,Dr
      XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) & ~(y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
      XEiJ.mpuCycleCount += (char) y != 0 ? 12 : 14;  //(0xffff&y)!=0
    } else {  //BCLR.B #<data>,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, (x = XEiJ.busRbs (a)) & ~(y = 1 << (y & 7)));
    }
    XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
  }  //irpBclrImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSET.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_011_000_rrr-{data}
  //BSET.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_011_mmm_rrr-{data}
  public static void irpBsetImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //BSET.L #<data>,Dr
      XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) | (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
      XEiJ.mpuCycleCount += (char) y != 0 ? 10 : 12;  //(0xffff&y)!=0
    } else {  //BSET.B #<data>,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, (x = XEiJ.busRbs (a)) | (y = 1 << (y & 7)));
    }
    XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
  }  //irpBsetImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EORI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}
  //EOR.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}  [EORI.B #<data>,<ea>]
  //EORI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_101_000_111_100-{data}
  public static void irpEoriByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      z = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      z = XEiJ.regPC;
      XEiJ.regPC = z + 2;
      z = XEiJ.busRbs (z + 1);  //pcbs
    }
    if (ea < XEiJ.EA_AR) {  //EORI.B #<data>,Dr
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[ea] ^= 255 & z;  //0拡張してからEOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea == XEiJ.EA_IM) {  //EORI.B #<data>,CCR
      XEiJ.mpuCycleCount += 20;
      XEiJ.regCCR ^= XEiJ.REG_CCR_MASK & z;
    } else {  //EORI.B #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z ^= XEiJ.busRbs (a));
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpEoriByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EORI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}
  //EOR.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}  [EORI.W #<data>,<ea>]
  //EORI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_101_001_111_100-{data}
  public static void irpEoriWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EORI.W #<data>,Dr
      int z;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        z = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        z = XEiJ.regPC;
        XEiJ.regPC = z + 2;
        z = XEiJ.busRwse (z);  //pcws
      }
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[ea] ^= (char) z;  //0拡張してからEOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea == XEiJ.EA_IM) {  //EORI.W #<data>,SR
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
        throw M68kException.m6eSignal;
      }
      //以下はスーパーバイザモード
      XEiJ.mpuCycleCount += 20;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        irpSetSR ((XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR) ^ XEiJ.busRwse ((XEiJ.regPC += 2) - 2));  //pcws。特権違反チェックが先
      } else {
        int t = XEiJ.regPC;
        XEiJ.regPC = t + 2;
        irpSetSR ((XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR) ^ XEiJ.busRwse (t));  //pcws。特権違反チェックが先
      }
    } else {  //EORI.W #<data>,<mem>
      int z;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        z = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        z = XEiJ.regPC;
        XEiJ.regPC = z + 2;
        z = XEiJ.busRwse (z);  //pcws
      }
      XEiJ.mpuCycleCount += 12;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z ^= XEiJ.busRws (a));
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpEoriWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EORI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}
  //EOR.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}  [EORI.L #<data>,<ea>]
  public static void irpEoriLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 4;
      y = XEiJ.busRlse (y);  //pcls
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //EORI.L #<data>,Dr
      XEiJ.mpuCycleCount += 16;
      z = XEiJ.regRn[ea] ^= y;
    } else {  //EORI.L #<data>,<mem>
      XEiJ.mpuCycleCount += 20;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = XEiJ.busRls (a) ^ y);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpEoriLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CAS.B Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_101_011_mmm_rrr-0000000uuu000ccc
  public static void irpCasByte () throws M68kException {
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    if ((w & ~0b0000_000_111_000_111) != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    int c = w & 7;
    int y = (byte) XEiJ.regRn[c];  //y=Dc
    int a = efaMltByte (XEiJ.regOC & 63);
    int x = XEiJ.busRbs (a);  //x=<ea>
    int z = (byte) (x - y);  //z=<ea>-Dc
    XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                   (z < 0 ? XEiJ.REG_CCR_N : 0) |
                   (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                   ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                   (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    if (z == 0) {  //<ea>==Dc
      XEiJ.mpuCycleCount += 16;
      XEiJ.busWb (a, XEiJ.regRn[w >> 6]);  //Du→<ea>
    } else {  //<ea>!=Dc
      XEiJ.mpuCycleCount += 12;
      XEiJ.regRn[c] = ~0xff & XEiJ.regRn[c] | 0xff & x;  //<ea>→Dc
    }
  }  //irpCasByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPI.B #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_000_mmm_rrr-{data}
  //CMP.B #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_000_mmm_rrr-{data}  [CMPI.B #<data>,<ea>]
  public static void irpCmpiByte () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRbs ((XEiJ.regPC += 2) - 1);  //pcbs
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRbs (y + 1);  //pcbs
    }
    int z = (byte) ((x = ea < XEiJ.EA_AR ? (byte) XEiJ.regRn[ea] : XEiJ.busRbs (efaMemByte (ea))) - y);  //アドレッシングモードに注意
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPI.W #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_001_mmm_rrr-{data}
  //CMP.W #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_001_mmm_rrr-{data}  [CMPI.W #<data>,<ea>]
  public static void irpCmpiWord () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 2;
      y = XEiJ.busRwse (y);  //pcws
    }
    int z = (short) ((x = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaMemWord (ea))) - y);  //アドレッシングモードに注意
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPI.L #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_010_mmm_rrr-{data}
  //CMP.L #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_010_mmm_rrr-{data}  [CMPI.L #<data>,<ea>]
  public static void irpCmpiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      y = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
    } else {
      y = XEiJ.regPC;
      XEiJ.regPC = y + 4;
      y = XEiJ.busRlse (y);  //pcls
    }
    int z;
    if (ea < XEiJ.EA_AR) {  //CMPI.L #<data>,Dr
      XEiJ.mpuCycleCount += 14;
      z = (x = XEiJ.regRn[ea]) - y;
    } else {  //CMPI.L #<data>,<mem>
      XEiJ.mpuCycleCount += 12;
      z = (x = XEiJ.busRls (efaMemLong (ea))) - y;  //アドレッシングモードに注意
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CAS.W Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_110_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
  //CAS2.W Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_110_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
  public static void irpCasWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea == XEiJ.EA_IM) {  //CAS2.W Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 4;
        w = XEiJ.busRlse (w);  //pcls
      }
      if ((w & ~0b1111_000_111_000_111_1111_000_111_000_111) != 0) {
        M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int c1 = w >>> 16 & 7;
      int c2 = w        & 7;
      int a1 = XEiJ.regRn[w >>> 16 + 12     ];  //a1=Rn1
      int a2 = XEiJ.regRn[w >>>      12 & 15];  //a2=Rn2
      int x1 = XEiJ.busRws (a1);  //x1=(Rn1)
      int x2 = XEiJ.busRws (a2);  //x2=(Rn2)
      int y = (short) XEiJ.regRn[c1];  //y=Dc1
      int z = (short) (x1 - y);  //z=(Rn1)-Dc1
      if (z == 0) {  //(Rn1)==Dc1
        y = (short) XEiJ.regRn[c2];  //y=Dc2
        z = (short) (x2 - y);  //z=(Rn2)-Dc2
        XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
               ((x2 ^ y) & (x2 ^ z)) >>> 31 << 1 |
               (x2 & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
        if (z == 0) {  //(Rn1)==Dc1&&(Rn2)==Dc2
          XEiJ.mpuCycleCount += 28;
          XEiJ.busWw (a1, XEiJ.regRn[w >>> 16 + 6 & 7]);  //Du1→(Rn1)
          XEiJ.busWw (a2, XEiJ.regRn[w >>>      6 & 7]);  //Du2→(Rn2)
        } else {  //(Rn1)==Dc1&&(Rn2)!=Dc2
          XEiJ.mpuCycleCount += 20;
          XEiJ.regRn[c1] = ~0xffff & XEiJ.regRn[c1] | (char) x1;  //(Rn1)→Dc1
          XEiJ.regRn[c2] = ~0xffff & XEiJ.regRn[c2] | (char) x2;  //(Rn2)→Dc2
        }
      } else {  //(Rn1)!=Dc1
        XEiJ.mpuCycleCount += 20;
        XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
               ((x1 ^ y) & (x1 ^ z)) >>> 31 << 1 |
               (x1 & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
        XEiJ.regRn[c1] = ~0xffff & XEiJ.regRn[c1] | (char) x1;  //(Rn1)→Dc1
        XEiJ.regRn[c2] = ~0xffff & XEiJ.regRn[c2] | (char) x2;  //(Rn2)→Dc2
      }
    } else {  //CAS.W Dc,Du,<ea>
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 2;
        w = XEiJ.busRwze (w);  //pcwz
      }
      if ((w & ~0b0000_000_111_000_111) != 0) {
        M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int c = w & 7;
      int y = (short) XEiJ.regRn[c];  //y=Dc
      int a = efaMltWord (ea);  //a=ea
      int x = XEiJ.busRws (a);  //x=<ea>
      int z = (short) (x - y);  //z=<ea>-Dc
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                     (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
      if (z == 0) {  //<ea>==Dc
        XEiJ.mpuCycleCount += 16;
        XEiJ.busWw (a, XEiJ.regRn[w >> 6]);  //Du→<ea>
      } else {  //<ea>!=Dc
        XEiJ.mpuCycleCount += 12;
        XEiJ.regRn[c] = ~0xffff & XEiJ.regRn[c] | (char) x;  //<ea>→Dc
      }
    }
  }  //irpCasWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVES.B <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn000000000000
  //MOVES.B Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn100000000000
  //
  //MOVES.B <ea>,Rn
  //  MOVES.B <ea>,DnはDnの最下位バイトだけ更新する
  //  MOVES.B <ea>,Anはバイトデータをロングに符号拡張してAnの全体を更新する
  //  SFC=1,2,5,6はアドレス変換あり、SFC=0,3,4はアドレス変換なし、
  //  SFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  //
  //MOVES.B Rn,<ea>
  //  DFC=1,2,5,6はアドレス変換あり、DFC=0,3,4はアドレス変換なし、
  //  DFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  public static void irpMovesByte () throws M68kException {
    int w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
    if (w << -11 != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 4;
    int a = efaMltByte (XEiJ.regOC & 63);
    int n = w >>> 12;  //n
    if (w << 31 - 11 >= 0) {  //MOVES.B <ea>,Rn。リード
      MemoryMappedDevice[] mm;
      if (XEiJ.mpuSFC == 1 || XEiJ.mpuSFC == 2) {  //ユーザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap;
      } else if (XEiJ.mpuSFC == 5 || XEiJ.mpuSFC == 6) {  //スーパーバイザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap;
      } else {  //CPU空間などは不可
        M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
        M68kException.m6eAddress = a;
        M68kException.m6eDirection = XEiJ.MPU_WR_READ;
        M68kException.m6eSize = XEiJ.MPU_SS_BYTE;
        throw M68kException.m6eSignal;
      }
      if (n < 8) {  //MOVES.B <ea>,Dn
        XEiJ.regRn[n] = XEiJ.regRn[n] & ~255 | mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a);
      } else {  //MOVES.B <ea>,An
        XEiJ.regRn[n] = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbs (a);
      }
    } else {  //MOVES.B Rn,<ea>。ライト
      MemoryMappedDevice[] mm;
      if (XEiJ.mpuDFC == 1 || XEiJ.mpuDFC == 2) {  //ユーザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap;
      } else if (XEiJ.mpuDFC == 5 || XEiJ.mpuDFC == 6) {  //スーパーバイザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap;
      } else {  //CPU空間などは不可
        M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
        M68kException.m6eAddress = a;
        M68kException.m6eDirection = XEiJ.MPU_WR_WRITE;
        M68kException.m6eSize = XEiJ.MPU_SS_BYTE;
        throw M68kException.m6eSignal;
      }
      mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, XEiJ.regRn[n]);
    }
  }  //irpMovesByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVES.W <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn000000000000
  //MOVES.W Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn100000000000
  //
  //MOVES.W <ea>,Rn
  //  MOVES.W <ea>,DnはDnの下位ワードだけ更新する
  //  MOVES.W <ea>,Anはワードデータをロングに符号拡張してAnの全体を更新する
  //  SFC=1,2,5,6はアドレス変換あり、SFC=0,3,4はアドレス変換なし、
  //  SFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  //
  //MOVES.W Rn,<ea>
  //  DFC=1,2,5,6はアドレス変換あり、DFC=0,3,4はアドレス変換なし、
  //  DFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  public static void irpMovesWord () throws M68kException {
    int w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
    if (w << -11 != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 4;
    int a = efaMltWord (XEiJ.regOC & 63);
    int n = w >>> 12;  //n
    if (w << 31 - 11 >= 0) {  //MOVES.W <ea>,Rn。リード
      MemoryMappedDevice[] mm;
      if (XEiJ.mpuSFC == 1 || XEiJ.mpuSFC == 2) {  //ユーザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap;
      } else if (XEiJ.mpuSFC == 5 || XEiJ.mpuSFC == 6) {  //スーパーバイザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap;
      } else {  //CPU空間などは不可
        M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
        M68kException.m6eAddress = a;
        M68kException.m6eDirection = XEiJ.MPU_WR_READ;
        M68kException.m6eSize = XEiJ.MPU_SS_WORD;
        throw M68kException.m6eSignal;
      }
      int z;
      if ((a & 1) == 0) {  //偶数
        z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRwz (a);
      } else {  //奇数
        z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a) << 8;
        a++;
        z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a);
      }
      if (n < 8) {  //MOVES.W <ea>,Dn
        XEiJ.regRn[n] = XEiJ.regRn[n] & ~65535 | z;
      } else {  //MOVES.W <ea>,An
        XEiJ.regRn[n] = (short) z;
      }
    } else {  //MOVES.W Rn,<ea>。ライト
      MemoryMappedDevice[] mm;
      if (XEiJ.mpuDFC == 1 || XEiJ.mpuDFC == 2) {  //ユーザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap;
      } else if (XEiJ.mpuDFC == 5 || XEiJ.mpuDFC == 6) {  //スーパーバイザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap;
      } else {  //CPU空間などは不可
        M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
        M68kException.m6eAddress = a;
        M68kException.m6eDirection = XEiJ.MPU_WR_WRITE;
        M68kException.m6eSize = XEiJ.MPU_SS_WORD;
        throw M68kException.m6eSignal;
      }
      int z = XEiJ.regRn[n];
      if ((a & 1) == 0) {  //偶数
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z);
      } else {  //奇数
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z >> 8);
        a++;
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z);
      }
    }
  }  //irpMovesWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVES.L <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn000000000000
  //MOVES.L Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn100000000000
  //
  //MOVES.L <ea>,Rn
  //  SFC=1,2,5,6はアドレス変換あり、SFC=0,3,4はアドレス変換なし、
  //  SFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  //
  //MOVES.L Rn,<ea>
  //  DFC=1,2,5,6はアドレス変換あり、DFC=0,3,4はアドレス変換なし、
  //  DFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  public static void irpMovesLong () throws M68kException {
    int w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
    if (w << -11 != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 4;
    int a = efaMltLong (XEiJ.regOC & 63);
    int n = w >>> 12;  //n
    if (w << 31 - 11 >= 0) {  //MOVES.L <ea>,Rn。リード
      MemoryMappedDevice[] mm;
      if (XEiJ.mpuSFC == 1 || XEiJ.mpuSFC == 2) {  //ユーザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap;
      } else if (XEiJ.mpuSFC == 5 || XEiJ.mpuSFC == 6) {  //スーパーバイザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap;
      } else {  //CPU空間などは不可
        M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
        M68kException.m6eAddress = a;
        M68kException.m6eDirection = XEiJ.MPU_WR_READ;
        M68kException.m6eSize = XEiJ.MPU_SS_LONG;
        throw M68kException.m6eSignal;
      }
      int z;
      if ((a & 3) == 0) {  //4の倍数
        z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRls (a);
      } else if ((a & 1) == 0) {  //4の倍数+2
        z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRws (a) << 16;
        a += 2;
        z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRwz (a);
      } else {  //奇数
        z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbs (a) << 24;
        a++;
        z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRwz (a) << 8;
        a += 2;
        z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a);
      }
      XEiJ.regRn[n] = z;
    } else {  //MOVES.L Rn,<ea>。ライト
      MemoryMappedDevice[] mm;
      if (XEiJ.mpuDFC == 1 || XEiJ.mpuDFC == 2) {  //ユーザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap;
      } else if (XEiJ.mpuDFC == 5 || XEiJ.mpuDFC == 6) {  //スーパーバイザモード
        mm = DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap;
      } else {  //CPU空間などは不可
        M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
        M68kException.m6eAddress = a;
        M68kException.m6eDirection = XEiJ.MPU_WR_WRITE;
        M68kException.m6eSize = XEiJ.MPU_SS_LONG;
        throw M68kException.m6eSignal;
      }
      int z = XEiJ.regRn[n];
      if ((a & 3) == 0) {  //4の倍数
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWl (a, z);
      } else if ((a & 1) == 0) {  //4の倍数+2
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z >> 16);
        a += 2;
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z);
      } else {  //奇数
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z >> 24);
        a++;
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z >> 8);
        a += 2;
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z);
      }
    }
  }  //irpMovesLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CAS.L Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_111_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
  //CAS2.L Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_111_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
  public static void irpCasLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea == XEiJ.EA_IM) {  //CAS2.L Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 4;
        w = XEiJ.busRlse (w);  //pcls
      }
      if ((w & ~0b1111_000_111_000_111_1111_000_111_000_111) != 0) {
        M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int c1 = w >>> 16 & 7;
      int c2 = w        & 7;
      int a1 = XEiJ.regRn[w >>> 16 + 12     ];  //a1=Rn1
      int a2 = XEiJ.regRn[w >>>      12 & 15];  //a2=Rn2
      int x1 = XEiJ.busRls (a1);  //x1=(Rn1)
      int x2 = XEiJ.busRls (a2);  //x2=(Rn2)
      int y = XEiJ.regRn[c1];  //y=Dc1
      int z = x1 - y;  //z=(Rn1)-Dc1
      if (z == 0) {  //(Rn1)==Dc1
        y = XEiJ.regRn[c2];  //y=Dc2
        z = x2 - y;  //z=(Rn2)-Dc2
        XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
               ((x2 ^ y) & (x2 ^ z)) >>> 31 << 1 |
               (x2 & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
        if (z == 0) {  //(Rn1)==Dc1&&(Rn2)==Dc2
          XEiJ.mpuCycleCount += 44;
          XEiJ.busWl (a1, XEiJ.regRn[w >>> 16 + 6 & 7]);  //Du1→(Rn1)
          XEiJ.busWl (a2, XEiJ.regRn[w >>>      6 & 7]);  //Du2→(Rn2)
        } else {  //(Rn1)==Dc1&&(Rn2)!=Dc2
          XEiJ.mpuCycleCount += 28;
          XEiJ.regRn[c1] = x1;  //(Rn1)→Dc1
          XEiJ.regRn[c2] = x2;  //(Rn2)→Dc2
        }
      } else {  //(Rn1)!=Dc1
        XEiJ.mpuCycleCount += 28;
        XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
               ((x1 ^ y) & (x1 ^ z)) >>> 31 << 1 |
               (x1 & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
        XEiJ.regRn[c1] = x1;  //(Rn1)→Dc1
        XEiJ.regRn[c2] = x2;  //(Rn2)→Dc2
      }
    } else {  //CAS.L Dc,Du,<ea>
      int w;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz
      } else {
        w = XEiJ.regPC;
        XEiJ.regPC = w + 2;
        w = XEiJ.busRwze (w);  //pcwz
      }
      if ((w & ~0b0000_000_111_000_111) != 0) {
        M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int c = w & 7;
      int y = XEiJ.regRn[c];  //y=Dc
      int a = efaMltLong (ea);  //a=ea
      int x = XEiJ.busRls (a);  //x=<ea>
      int z = x - y;  //z=<ea>-Dc
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                     (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
      if (z == 0) {  //<ea>==Dc
        XEiJ.mpuCycleCount += 24;
        XEiJ.busWl (a, XEiJ.regRn[w >> 6]);  //Du→<ea>
      } else {  //<ea>!=Dc
        XEiJ.mpuCycleCount += 16;
        XEiJ.regRn[c] = x;  //<ea>→Dc
      }
    }
  }  //irpCasLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,Dq                                  |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_000_mmm_rrr
  public static void irpMoveToDRByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));
    XEiJ.regRn[qqq] = ~255 & XEiJ.regRn[qqq] | 255 & z;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToDRByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_010_mmm_rrr
  public static void irpMoveToMMByte () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));  //ここでAqが変化する可能性があることに注意
    XEiJ.busWb (XEiJ.regRn[XEiJ.regOC >> 9], z);  //1qqq=aqq
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMMByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_011_mmm_rrr
  public static void irpMoveToMPByte () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int aqq = XEiJ.regOC >> 9;  //1qqq=aqq
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));  //ここでAqが変化する可能性があることに注意
    XEiJ.busWb (aqq < 15 ? XEiJ.regRn[aqq]++ : (XEiJ.regRn[15] += 2) - 2, z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMPByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_100_mmm_rrr
  public static void irpMoveToMNByte () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int aqq = XEiJ.regOC >> 9;  //1qqq=aqq
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));  //ここでAqが変化する可能性があることに注意
    XEiJ.busWb (aqq < 15 ? --XEiJ.regRn[aqq] : (XEiJ.regRn[15] -= 2), z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMNByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_101_mmm_rrr
  public static void irpMoveToMWByte () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int aqq = XEiJ.regOC >> 9;  //1qqq=aqq
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));  //ここでAqが変化する可能性があることに注意
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWb (XEiJ.regRn[aqq]  //ベースレジスタ
          + XEiJ.busRwse ((XEiJ.regPC += 2) - 2),  //pcws。ワードディスプレースメント
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.busWb (XEiJ.regRn[aqq]  //ベースレジスタ
          + XEiJ.busRwse (t),  //pcws。ワードディスプレースメント
          z);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMWByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_110_mmm_rrr
  public static void irpMoveToMXByte () throws M68kException {
    XEiJ.mpuCycleCount += 14;
    int ea = XEiJ.regOC & 63;
    int aqq = XEiJ.regOC >> 9;  //1qqq=aqq
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));  //ここでAqが変化する可能性があることに注意
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    XEiJ.mpuCycleCount += XEiJ.EFA_EXTENSION_CLK[w & 511];
    int t = (((~w & 0x0180) == 0 ? 0 :  //ベースレジスタサプレス
              XEiJ.regRn[aqq])  //ベースレジスタ
             + (w << 31 - 8 >= 0 ? (byte) w :  //バイトディスプレースメント
                w << 31 - 5 >= 0 ? 0 :  //ヌルベースディスプレースメント
                w << 31 - 4 >= 0 ? XEiJ.busRwse ((XEiJ.regPC += 2) - 2) :  //pcws。ワードベースディスプレースメント
                XEiJ.busRlse ((XEiJ.regPC += 4) - 4)));  //pcls。ロングベースディスプレースメント
    int x = ((~w & 0x0140) == 0 ? 0 :  //インデックスサプレス
             (w << 31 - 11 >= 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
              XEiJ.regRn[w >> 12])  //ロングインデックス
             << (w >> 9 & 3));  //スケールファクタ。ワードインデックスのときは符号拡張してから掛ける
    XEiJ.busWb ((w & 0x0103) <= 0x0100 ? t + x :  //メモリ間接なし
        ((XEiJ.TEST_BIT_2_SHIFT ? w << 31 - 2 >= 0 : (w & 4) == 0) ? XEiJ.busRls (t + x) :  //プリインデックス
         XEiJ.busRls (t) + x)  //ポストインデックス
        + ((XEiJ.TEST_BIT_1_SHIFT ? w << 31 - 1 >= 0 : (w & 2) == 0) ? 0 :  //ヌルアウタディスプレースメント
           (XEiJ.TEST_BIT_0_SHIFT ? w << 31 - 0 >= 0 : (w & 1) == 0) ? XEiJ.busRwse ((XEiJ.regPC += 2) - 2) :  //pcws。ワードアウタディスプレースメント
           XEiJ.busRlse ((XEiJ.regPC += 4) - 4)),  //pcls。ロングアウタディスプレースメント
        z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMXByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_000_111_mmm_rrr
  public static void irpMoveToZWByte () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWb (XEiJ.busRwse ((XEiJ.regPC += 2) - 2),  //pcws
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.busWb (XEiJ.busRwse (t),  //pcws
          z);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToZWByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_001_111_mmm_rrr
  public static void irpMoveToZLByte () throws M68kException {
    XEiJ.mpuCycleCount += 16;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWb (XEiJ.busRlse ((XEiJ.regPC += 4) - 4),  //pcls
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 4;
      XEiJ.busWb (XEiJ.busRlse (t),  //pcls
          z);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToZLByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_000_mmm_rrr
  public static void irpMoveToDRLong () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToDRLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEA.L <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr
  //MOVE.L <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr [MOVEA.L <ea>,Aq]
  public static void irpMoveaLong () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)] = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。右辺でAqが変化する可能性があることに注意
  }  //irpMoveaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_010_mmm_rrr
  public static void irpMoveToMMLong () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.busWl (XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)], z);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMMLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_011_mmm_rrr
  public static void irpMoveToMPLong () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.busWl ((XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)] += 4) - 4, z);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMPLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_100_mmm_rrr
  public static void irpMoveToMNLong () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.busWl ((XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)] -= 4), z);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMNLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_101_mmm_rrr
  public static void irpMoveToMWLong () throws M68kException {
    XEiJ.mpuCycleCount += 16;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWl (XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)]  //ベースレジスタ
          + XEiJ.busRwse ((XEiJ.regPC += 2) - 2),  //pcws。ワードディスプレースメント
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.busWl (XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)]  //ベースレジスタ
          + XEiJ.busRwse (t),  //pcws。ワードディスプレースメント
          z);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMWLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_110_mmm_rrr
  public static void irpMoveToMXLong () throws M68kException {
    XEiJ.mpuCycleCount += 18;
    int ea = XEiJ.regOC & 63;
    int aqq = (XEiJ.regOC >> 9) - (16 - 8);
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    XEiJ.mpuCycleCount += XEiJ.EFA_EXTENSION_CLK[w & 511];
    int t = (((~w & 0x0180) == 0 ? 0 :  //ベースレジスタサプレス
              XEiJ.regRn[aqq])  //ベースレジスタ
             + (w << 31 - 8 >= 0 ? (byte) w :  //バイトディスプレースメント
                w << 31 - 5 >= 0 ? 0 :  //ヌルベースディスプレースメント
                w << 31 - 4 >= 0 ? XEiJ.busRwse ((XEiJ.regPC += 2) - 2) :  //pcws。ワードベースディスプレースメント
                XEiJ.busRlse ((XEiJ.regPC += 4) - 4)));  //pcls。ロングベースディスプレースメント
    int x = ((~w & 0x0140) == 0 ? 0 :  //インデックスサプレス
             (w << 31 - 11 >= 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
              XEiJ.regRn[w >> 12])  //ロングインデックス
             << (w >> 9 & 3));  //スケールファクタ。ワードインデックスのときは符号拡張してから掛ける
    XEiJ.busWl ((w & 0x0103) <= 0x0100 ? t + x :  //メモリ間接なし
        ((XEiJ.TEST_BIT_2_SHIFT ? w << 31 - 2 >= 0 : (w & 4) == 0) ? XEiJ.busRls (t + x) :  //プリインデックス
         XEiJ.busRls (t) + x)  //ポストインデックス
        + ((XEiJ.TEST_BIT_1_SHIFT ? w << 31 - 1 >= 0 : (w & 2) == 0) ? 0 :  //ヌルアウタディスプレースメント
           (XEiJ.TEST_BIT_0_SHIFT ? w << 31 - 0 >= 0 : (w & 1) == 0) ? XEiJ.busRwse ((XEiJ.regPC += 2) - 2) :  //pcws。ワードアウタディスプレースメント
           XEiJ.busRlse ((XEiJ.regPC += 4) - 4)),  //pcls。ロングアウタディスプレースメント
        z);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMXLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_000_111_mmm_rrr
  public static void irpMoveToZWLong () throws M68kException {
    XEiJ.mpuCycleCount += 16;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWl (XEiJ.busRwse ((XEiJ.regPC += 2) - 2),  //pcws
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.busWl (XEiJ.busRwse (t),  //pcws
          z);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToZWLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_001_111_mmm_rrr
  public static void irpMoveToZLLong () throws M68kException {
    XEiJ.mpuCycleCount += 20;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWl (XEiJ.busRlse ((XEiJ.regPC += 4) - 4),  //pcls
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 4;
      XEiJ.busWl (XEiJ.busRlse (t),  //pcls
          z);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToZLLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_000_mmm_rrr
  public static void irpMoveToDRWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regRn[qqq] = ~65535 & XEiJ.regRn[qqq] | (char) z;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToDRWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEA.W <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr
  //MOVE.W <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr [MOVEA.W <ea>,Aq]
  //
  //MOVEA.W <ea>,Aq
  //  ワードデータをロングに符号拡張してAqの全体を更新する
  public static void irpMoveaWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //符号拡張して32bit全部書き換える。このr[ea]はデータレジスタまたはアドレスレジスタ。右辺でAqが変化する可能性があることに注意
  }  //irpMoveaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_010_mmm_rrr
  public static void irpMoveToMMWord () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.busWw (XEiJ.regRn[XEiJ.regOC >> 9 & 15], z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMMWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_011_mmm_rrr
  public static void irpMoveToMPWord () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.busWw ((XEiJ.regRn[XEiJ.regOC >> 9 & 15] += 2) - 2, z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMPWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_100_mmm_rrr
  public static void irpMoveToMNWord () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.busWw ((XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= 2), z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMNWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_101_mmm_rrr
  public static void irpMoveToMWWord () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int aqq = XEiJ.regOC >> 9 & 15;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWw (XEiJ.regRn[aqq]  //ベースレジスタ
          + XEiJ.busRwse ((XEiJ.regPC += 2) - 2),  //pcws。ワードディスプレースメント
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.busWw (XEiJ.regRn[aqq]  //ベースレジスタ
          + XEiJ.busRwse (t),  //pcws。ワードディスプレースメント
          z);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMWWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_110_mmm_rrr
  public static void irpMoveToMXWord () throws M68kException {
    XEiJ.mpuCycleCount += 14;
    int ea = XEiJ.regOC & 63;
    int aqq = XEiJ.regOC >> 9 & 15;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    XEiJ.mpuCycleCount += XEiJ.EFA_EXTENSION_CLK[w & 511];
    int t = (((~w & 0x0180) == 0 ? 0 :  //ベースレジスタサプレス
              XEiJ.regRn[aqq])  //ベースレジスタ
             + (w << 31 - 8 >= 0 ? (byte) w :  //バイトディスプレースメント
                w << 31 - 5 >= 0 ? 0 :  //ヌルベースディスプレースメント
                w << 31 - 4 >= 0 ? XEiJ.busRwse ((XEiJ.regPC += 2) - 2) :  //pcws。ワードベースディスプレースメント
                XEiJ.busRlse ((XEiJ.regPC += 4) - 4)));  //pcls。ロングベースディスプレースメント
    int x = ((~w & 0x0140) == 0 ? 0 :  //インデックスサプレス
             (w << 31 - 11 >= 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
              XEiJ.regRn[w >> 12])  //ロングインデックス
             << (w >> 9 & 3));  //スケールファクタ。ワードインデックスのときは符号拡張してから掛ける
    XEiJ.busWw ((w & 0x0103) <= 0x0100 ? t + x :  //メモリ間接なし
        ((XEiJ.TEST_BIT_2_SHIFT ? w << 31 - 2 >= 0 : (w & 4) == 0) ? XEiJ.busRls (t + x) :  //プリインデックス
         XEiJ.busRls (t) + x)  //ポストインデックス
        + ((XEiJ.TEST_BIT_1_SHIFT ? w << 31 - 1 >= 0 : (w & 2) == 0) ? 0 :  //ヌルアウタディスプレースメント
           (XEiJ.TEST_BIT_0_SHIFT ? w << 31 - 0 >= 0 : (w & 1) == 0) ? XEiJ.busRwse ((XEiJ.regPC += 2) - 2) :  //pcws。ワードアウタディスプレースメント
           XEiJ.busRlse ((XEiJ.regPC += 4) - 4)),  //pcls。ロングアウタディスプレースメント
        z);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMXWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_000_111_mmm_rrr
  public static void irpMoveToZWWord () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWw (XEiJ.busRwse ((XEiJ.regPC += 2) - 2),  //pcws
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.busWw (XEiJ.busRwse (t),  //pcws
          z);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToZWWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_001_111_mmm_rrr
  public static void irpMoveToZLWord () throws M68kException {
    XEiJ.mpuCycleCount += 16;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.busWw (XEiJ.busRlse ((XEiJ.regPC += 4) - 4),  //pcls
          z);
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 4;
      XEiJ.busWw (XEiJ.busRlse (t),  //pcls
          z);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToZLWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEGX.B <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_000_mmm_rrr
  public static void irpNegxByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEGX.B Dr
      XEiJ.mpuCycleCount += 4;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (y = XEiJ.regRn[ea]) | 0xff & -(y = (byte) y) - (XEiJ.regCCR >> 4));  //Xの左側はすべて0なのでCCR_X&を省略
    } else {  //NEGX.B <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = (byte) (-(y = XEiJ.busRbs (a)) - (XEiJ.regCCR >> 4)));  //Xの左側はすべて0なのでCCR_X&を省略
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_negx
  }  //irpNegxByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEGX.W <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_001_mmm_rrr
  public static void irpNegxWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEGX.W Dr
      XEiJ.mpuCycleCount += 4;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (y = XEiJ.regRn[ea]) | (char) (-(y = (short) y) - (XEiJ.regCCR >> 4)));  //Xの左側はすべて0なのでCCR_X&を省略
    } else {  //NEGX.W <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z = (short) (-(y = XEiJ.busRws (a)) - (XEiJ.regCCR >> 4)));  //Xの左側はすべて0なのでCCR_X&を省略
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_negx
  }  //irpNegxWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEGX.L <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_010_mmm_rrr
  public static void irpNegxLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEGX.L Dr
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[ea] = z = -(y = XEiJ.regRn[ea]) - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
    } else {  //NEGX.L <mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = -(y = XEiJ.busRls (a)) - (XEiJ.regCCR >> 4));  //Xの左側はすべて0なのでCCR_X&を省略
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_negx
  }  //irpNegxLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W SR,<ea>                                  |-|-12346|P|*****|-----|D M+-WXZ  |0100_000_011_mmm_rrr
  public static void irpMoveFromSR () throws M68kException {
    //MC68010以上では特権命令
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //MOVE.W SR,Dr
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[ea] = ~0xffff & XEiJ.regRn[ea] | XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
    } else {  //MOVE.W SR,<mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWw (efaMltWord (ea), XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR);
    }
  }  //irpMoveFromSR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CHK.L <ea>,Dq                                   |-|--2346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_100_mmm_rrr
  public static void irpChkLong () throws M68kException {
    XEiJ.mpuCycleCount += 14;
    int ea = XEiJ.regOC & 63;
    int x = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));
    int y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
    int z = x - y;
    XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                   (y < 0 ? XEiJ.REG_CCR_N : 0) |
                   (y == 0 ? XEiJ.REG_CCR_Z : 0) |
                   ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                   (x & (y ^ z) ^ (y | z)) >>> 31);
    if (y < 0 || x < y) {
      XEiJ.mpuCycleCount += 40 - 14 - 34;
      M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpChkLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CHK.W <ea>,Dq                                   |-|012346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_110_mmm_rrr
  public static void irpChkWord () throws M68kException {
    XEiJ.mpuCycleCount += 10;
    int ea = XEiJ.regOC & 63;
    int x = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));
    int y = (short) XEiJ.regRn[XEiJ.regOC >> 9 & 7];
    int z = (short) (x - y);
    XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                   (y < 0 ? XEiJ.REG_CCR_N : 0) |
                   (y == 0 ? XEiJ.REG_CCR_Z : 0) |
                   ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                   (x & (y ^ z) ^ (y | z)) >>> 31);
    if (y < 0 || x < y) {
      XEiJ.mpuCycleCount += 40 - 10 - 34;
      M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpChkWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //LEA.L <ea>,Aq                                   |-|012346|-|-----|-----|  M  WXZP |0100_qqq_111_mmm_rrr
  //EXTB.L Dr                                       |-|--2346|-|-UUUU|-**00|D         |0100_100_111_000_rrr
  public static void irpLea () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EXTB.L Dr
      XEiJ.mpuCycleCount += 4;
      int z;
      XEiJ.regRn[ea] = z = (byte) XEiJ.regRn[ea];
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //LEA.L <ea>,Aq
      //XEiJ.mpuCycleCount += 4 - 4;
      XEiJ.regRn[(XEiJ.regOC >> 9) - (32 - 8)] = efaLeaPea (ea);
    }
  }  //irpLea

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CLR.B <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_000_mmm_rrr (68000 and 68008 read before clear)
  public static void irpClrByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //CLR.B Dr
      XEiJ.mpuCycleCount += 4;
      XEiJ.regRn[ea] &= ~0xff;
    } else {  //CLR.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), 0);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z;  //ccr_clr
  }  //irpClrByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CLR.W <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_001_mmm_rrr (68000 and 68008 read before clear)
  public static void irpClrWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //CLR.W Dr
      XEiJ.mpuCycleCount += 4;
      XEiJ.regRn[ea] &= ~0xffff;
    } else {  //CLR.W <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWw (efaMltWord (ea), 0);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z;  //ccr_clr
  }  //irpClrWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CLR.L <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_010_mmm_rrr (68000 and 68008 read before clear)
  public static void irpClrLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //CLR.L Dr
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[ea] = 0;
    } else {  //CLR.L <mem>
      XEiJ.mpuCycleCount += 12;
      XEiJ.busWl (efaMltLong (ea), 0);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z;  //ccr_clr
  }  //irpClrLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W CCR,<ea>                                 |-|-12346|-|*****|-----|D M+-WXZ  |0100_001_011_mmm_rrr
  public static void irpMoveFromCCR () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //MOVE.W CCR,Dr
      XEiJ.mpuCycleCount += 4;
      XEiJ.regRn[ea] = ~0xffff & XEiJ.regRn[ea] | XEiJ.regCCR;
    } else {  //MOVE.W CCR,<mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWw (efaMltWord (ea), XEiJ.regCCR);
    }
  }  //irpMoveFromCCR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEG.B <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_000_mmm_rrr
  public static void irpNegByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEG.B Dr
      XEiJ.mpuCycleCount += 4;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (y = XEiJ.regRn[ea]) | 0xff & -(y = (byte) y));
    } else {  //NEG.B <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = (byte) -(y = XEiJ.busRbs (a)));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_neg
  }  //irpNegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEG.W <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_001_mmm_rrr
  public static void irpNegWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEG.W Dr
      XEiJ.mpuCycleCount += 4;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (y = XEiJ.regRn[ea]) | (char) -(y = (short) y));
    } else {  //NEG.W <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z = (short) -(y = XEiJ.busRws (a)));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_neg
  }  //irpNegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEG.L <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_010_mmm_rrr
  public static void irpNegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEG.L Dr
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[ea] = z = -(y = XEiJ.regRn[ea]);
    } else {  //NEG.L <mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = -(y = XEiJ.busRls (a)));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_neg
  }  //irpNegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,CCR                                 |-|012346|-|UUUUU|*****|D M+-WXZPI|0100_010_011_mmm_rrr
  public static void irpMoveToCCR () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.REG_CCR_MASK & (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea)));
  }  //irpMoveToCCR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOT.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_000_mmm_rrr
  public static void irpNotByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //NOT.B Dr
      XEiJ.mpuCycleCount += 4;
      z = XEiJ.regRn[ea] ^= 255;  //0拡張してからEOR
    } else {  //NOT.B <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = ~XEiJ.busRbs (a));
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpNotByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOT.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_001_mmm_rrr
  public static void irpNotWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //NOT.W Dr
      XEiJ.mpuCycleCount += 4;
      z = XEiJ.regRn[ea] ^= 65535;  //0拡張してからEOR
    } else {  //NOT.W <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltWord (ea);
      XEiJ.busWw (a, z = ~XEiJ.busRws (a));
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpNotWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOT.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_010_mmm_rrr
  public static void irpNotLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //NOT.L Dr
      XEiJ.mpuCycleCount += 6;
      z = XEiJ.regRn[ea] ^= 0xffffffff;
    } else {  //NOT.L <mem>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      XEiJ.busWl (a, z = ~XEiJ.busRls (a));
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpNotLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,SR                                  |-|012346|P|UUUUU|*****|D M+-WXZPI|0100_011_011_mmm_rrr
  public static void irpMoveToSR () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    irpSetSR (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRwz (efaAnyWord (ea)));  //特権違反チェックが先
  }  //irpMoveToSR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NBCD.B <ea>                                     |-|012346|-|UUUUU|*U*U*|D M+-WXZ  |0100_100_000_mmm_rrr
  //LINK.L Ar,#<data>                               |-|--2346|-|-----|-----|          |0100_100_000_001_rrr-{data}
  //
  //LINK.L Ar,#<data>
  //  PEA.L (Ar);MOVEA.L A7,Ar;ADDA.L #<data>,A7と同じ
  //  LINK.L A7,#<data>はA7をデクリメントする前の値がプッシュされ、A7に#<data>が加算される
  public static void irpNbcd () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //NBCD.B Dr
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[ea] = ~0xff & XEiJ.regRn[ea] | irpSbcd (0, XEiJ.regRn[ea]);
    } else if (ea < XEiJ.EA_MM) {  //LINK.L Ar,#<data>
      XEiJ.mpuCycleCount += 20;
      int arr = XEiJ.regOC - (0b0100_100_000_001_000 - 8);
      int sp = XEiJ.regRn[15] - 4;
      XEiJ.busWl (sp, XEiJ.regRn[arr]);  //pushl
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        XEiJ.regRn[15] = (XEiJ.regRn[arr] = sp) + XEiJ.busRlse ((XEiJ.regPC += 4) - 4);  //pcls
      } else {
        int t = XEiJ.regPC;
        XEiJ.regPC = t + 4;
        XEiJ.regRn[15] = (XEiJ.regRn[arr] = sp) + XEiJ.busRlse (t);  //pcls
      }
    } else {  //NBCD.B <mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, irpSbcd (0, XEiJ.busRbs (a)));
    }
  }  //irpNbcd

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SWAP.W Dr                                       |-|012346|-|-UUUU|-**00|D         |0100_100_001_000_rrr
  //BKPT #<data>                                    |-|-12346|-|-----|-----|          |0100_100_001_001_ddd
  //PEA.L <ea>                                      |-|012346|-|-----|-----|  M  WXZP |0100_100_001_mmm_rrr
  public static void irpPea () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //SWAP.W Dr
      XEiJ.mpuCycleCount += 4;
      int x;
      int z;
      XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) << 16 | x >>> 16;
      //上位ワードと下位ワードを入れ替えた後のDrをロングでテストする
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //PEA.L <ea>
      XEiJ.mpuCycleCount += 12 - 4;
      int a = efaLeaPea (ea);  //BKPT #<data>はここでillegal instructionになる
      XEiJ.busWl (XEiJ.regRn[15] -= 4, a);  //pushl。評価順序に注意。wl(r[15]-=4,eaz_leapea(ea))は不可
    }
  }  //irpPea

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXT.W Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_010_000_rrr
  //MOVEM.W <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_010_mmm_rrr-llllllllllllllll
  public static void irpMovemToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EXT.W Dr
      XEiJ.mpuCycleCount += 4;
      int z;
      XEiJ.regRn[ea] = ~0xffff & (z = XEiJ.regRn[ea]) | (char) (z = (byte) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //MOVEM.W <list>,<ea>
      int l = XEiJ.busRwze (XEiJ.regPC);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      if (ea >> 3 == XEiJ.MMM_MN) {  //-(Ar)
        //MOVEM.wl <list>,-(Ar)で<list>にArが含まれているとき、000/010は命令開始時のArを、020/030/040/060は命令開始時のAr-オペレーションサイズをメモリに書き込む
        //転送するレジスタが0個のときArは変化しない
        int arr = ea - (XEiJ.EA_MN - 8);
        int a = XEiJ.regRn[arr];
        XEiJ.regRn[arr] = a - 2;
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[15]);
          }
          if ((l & 0x0002) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[14]);
          }
          if ((l & 0x0004) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[13]);
          }
          if ((l & 0x0008) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[12]);
          }
          if ((l & 0x0010) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[11]);
          }
          if ((l & 0x0020) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[10]);
          }
          if ((l & 0x0040) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 9]);
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 8]);
          }
          if ((l & 0x0100) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 7]);
          }
          if ((l & 0x0200) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 6]);
          }
          if ((l & 0x0400) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 5]);
          }
          if ((l & 0x0800) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 4]);
          }
          if ((l & 0x1000) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 3]);
          }
          if ((l & 0x2000) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 2]);
          }
          if ((l & 0x4000) != 0) {
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 1]);
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            XEiJ.busWw (a -= 2, XEiJ.regRn[ 0]);
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 15; i >= 0; i--) {
            if ((l & 0x8000 >>> i) != 0) {
              XEiJ.busWw (a -= 2, XEiJ.regRn[i]);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 15; l != 0; i--, l <<= 1) {
            if (l < 0) {
              XEiJ.busWw (a -= 2, XEiJ.regRn[i]);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 15; l != 0; i--, l >>>= 1) {
            if ((l & 1) != 0) {
              XEiJ.busWw (a -= 2, XEiJ.regRn[i]);
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 15; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            XEiJ.busWw (a -= 2, XEiJ.regRn[i -= k]);
            l = l >>> k & ~1;
          }
        }
        XEiJ.regRn[arr] = a;
        XEiJ.mpuCycleCount += 8 + (t - a << 1);  //2バイト/個→4サイクル/個
      } else {  //-(Ar)以外
        int a = efaCltWord (ea);
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 0]);
            a += 2;
          }
          if ((l & 0x0002) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 1]);
            a += 2;
          }
          if ((l & 0x0004) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 2]);
            a += 2;
          }
          if ((l & 0x0008) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 3]);
            a += 2;
          }
          if ((l & 0x0010) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 4]);
            a += 2;
          }
          if ((l & 0x0020) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 5]);
            a += 2;
          }
          if ((l & 0x0040) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 6]);
            a += 2;
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            XEiJ.busWw (a, XEiJ.regRn[ 7]);
            a += 2;
          }
          if ((l & 0x0100) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 8]);
            a += 2;
          }
          if ((l & 0x0200) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[ 9]);
            a += 2;
          }
          if ((l & 0x0400) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[10]);
            a += 2;
          }
          if ((l & 0x0800) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[11]);
            a += 2;
          }
          if ((l & 0x1000) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[12]);
            a += 2;
          }
          if ((l & 0x2000) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[13]);
            a += 2;
          }
          if ((l & 0x4000) != 0) {
            XEiJ.busWw (a, XEiJ.regRn[14]);
            a += 2;
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            XEiJ.busWw (a, XEiJ.regRn[15]);
            a += 2;
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 0; i <= 15; i++) {
            if ((l & 0x0001 << i) != 0) {
              XEiJ.busWw (a, XEiJ.regRn[i]);
              a += 2;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 0; l != 0; i++, l <<= 1) {
            if (l < 0) {
              XEiJ.busWw (a, XEiJ.regRn[i]);
              a += 2;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 0; l != 0; i++, l >>>= 1) {
            if ((l & 1) != 0) {
              XEiJ.busWw (a, XEiJ.regRn[i]);
              a += 2;
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 0; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            XEiJ.busWw (a, XEiJ.regRn[i += k]);
            a += 2;
            l = l >>> k & ~1;
          }
        }
        XEiJ.mpuCycleCount += 4 + (a - t << 1);  //2バイト/個→4サイクル/個
      }
    }
  }  //irpMovemToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXT.L Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_011_000_rrr
  //MOVEM.L <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_011_mmm_rrr-llllllllllllllll
  public static void irpMovemToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EXT.L Dr
      XEiJ.mpuCycleCount += 4;
      int z;
      XEiJ.regRn[ea] = z = (short) XEiJ.regRn[ea];
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //MOVEM.L <list>,<ea>
      int l = XEiJ.busRwze (XEiJ.regPC);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      if (ea >> 3 == XEiJ.MMM_MN) {  //-(Ar)
        //MOVEM.wl <list>,-(Ar)で<list>にArが含まれているとき、000/010は命令開始時のArを、020/030/040/060は命令開始時のAr-オペレーションサイズをメモリに書き込む
        //転送するレジスタが0個のときArは変化しない
        int arr = ea - (XEiJ.EA_MN - 8);
        int a = XEiJ.regRn[arr];
        XEiJ.regRn[arr] = a - 4;
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[15]);
          }
          if ((l & 0x0002) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[14]);
          }
          if ((l & 0x0004) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[13]);
          }
          if ((l & 0x0008) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[12]);
          }
          if ((l & 0x0010) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[11]);
          }
          if ((l & 0x0020) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[10]);
          }
          if ((l & 0x0040) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 9]);
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 8]);
          }
          if ((l & 0x0100) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 7]);
          }
          if ((l & 0x0200) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 6]);
          }
          if ((l & 0x0400) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 5]);
          }
          if ((l & 0x0800) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 4]);
          }
          if ((l & 0x1000) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 3]);
          }
          if ((l & 0x2000) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 2]);
          }
          if ((l & 0x4000) != 0) {
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 1]);
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            XEiJ.busWl (a -= 4, XEiJ.regRn[ 0]);
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 15; i >= 0; i--) {
            if ((l & 0x8000 >>> i) != 0) {
              XEiJ.busWl (a -= 4, XEiJ.regRn[i]);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 15; l != 0; i--, l <<= 1) {
            if (l < 0) {
              XEiJ.busWl (a -= 4, XEiJ.regRn[i]);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 15; l != 0; i--, l >>>= 1) {
            if ((l & 1) != 0) {
              XEiJ.busWl (a -= 4, XEiJ.regRn[i]);
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 15; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            XEiJ.busWl (a -= 4, XEiJ.regRn[i -= k]);
            l = l >>> k & ~1;
          }
        }
        XEiJ.regRn[arr] = a;
        XEiJ.mpuCycleCount += 8 + (t - a << 1);  //4バイト/個→8サイクル/個
      } else {  //-(Ar)以外
        int a = efaCltLong (ea);
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 0]);
            a += 4;
          }
          if ((l & 0x0002) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 1]);
            a += 4;
          }
          if ((l & 0x0004) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 2]);
            a += 4;
          }
          if ((l & 0x0008) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 3]);
            a += 4;
          }
          if ((l & 0x0010) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 4]);
            a += 4;
          }
          if ((l & 0x0020) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 5]);
            a += 4;
          }
          if ((l & 0x0040) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 6]);
            a += 4;
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            XEiJ.busWl (a, XEiJ.regRn[ 7]);
            a += 4;
          }
          if ((l & 0x0100) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 8]);
            a += 4;
          }
          if ((l & 0x0200) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[ 9]);
            a += 4;
          }
          if ((l & 0x0400) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[10]);
            a += 4;
          }
          if ((l & 0x0800) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[11]);
            a += 4;
          }
          if ((l & 0x1000) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[12]);
            a += 4;
          }
          if ((l & 0x2000) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[13]);
            a += 4;
          }
          if ((l & 0x4000) != 0) {
            XEiJ.busWl (a, XEiJ.regRn[14]);
            a += 4;
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            XEiJ.busWl (a, XEiJ.regRn[15]);
            a += 4;
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 0; i <= 15; i++) {
            if ((l & 0x0001 << i) != 0) {
              XEiJ.busWl (a, XEiJ.regRn[i]);
              a += 4;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 0; l != 0; i++, l <<= 1) {
            if (l < 0) {
              XEiJ.busWl (a, XEiJ.regRn[i]);
              a += 4;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 0; l != 0; i++, l >>>= 1) {
            if ((l & 1) != 0) {
              XEiJ.busWl (a, XEiJ.regRn[i]);
              a += 4;
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 0; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            XEiJ.busWl (a, XEiJ.regRn[i += k]);
            a += 4;
            l = l >>> k & ~1;
          }
        }
        XEiJ.mpuCycleCount += 4 + (a - t << 1);  //4バイト/個→8サイクル/個
      }
    }
  }  //irpMovemToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TST.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_000_mmm_rrr
  //TST.B <ea>                                      |-|--2346|-|-UUUU|-**00|        PI|0100_101_000_mmm_rrr
  public static void irpTstByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea)))];  //ccr_tst_byte。アドレッシングモードに注意
  }  //irpTstByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TST.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_001_mmm_rrr
  //TST.W <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_001_mmm_rrr
  public static void irpTstWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //アドレッシングモードに注意。このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpTstWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TST.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_010_mmm_rrr
  //TST.L <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_010_mmm_rrr
  public static void irpTstLong () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //アドレッシングモードに注意。このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpTstLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TAS.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_011_mmm_rrr
  //ILLEGAL                                         |-|012346|-|-----|-----|          |0100_101_011_111_100
  public static void irpTas () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //TAS.B Dr
      XEiJ.mpuCycleCount += 4;
      XEiJ.regRn[ea] = 0x80 | (z = XEiJ.regRn[ea]);
    } else {  //TAS.B <mem>
      XEiJ.mpuCycleCount += 14;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, 0x80 | (z = XEiJ.busRbs (a)));
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpTas

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MULU.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll000000000hhh        (h is not used)
  //MULU.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll010000000hhh        (if h=l then result is not defined)
  //MULS.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll100000000hhh        (h is not used)
  //MULS.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll110000000hhh        (if h=l then result is not defined)
  public static void irpMuluMulsLong () throws M68kException {
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    if ((w & ~0b0111_110_000_000_111) != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    int l = w >> 12;  //被乗数,積の下位32bit
    int s = w & 0b0000_100_000_000_000;  //0=MULU,1=MULS
    int q = w & 0b0000_010_000_000_000;  //0=32bit,1=64bit
    int h = w & 7;  //積の上位32bit
    XEiJ.mpuCycleCount += 72;  //72*0.6=43.2≒44
    int ea = XEiJ.regOC & 63;
    long yy = (long) (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea)));
    long xx = (long) XEiJ.regRn[l];
    if (s == 0) {  //MULU
      long zz = (0xffffffffL & xx) * (0xffffffffL & yy);
      int z = XEiJ.regRn[l] = (int) zz;
      if (q == 0) {  //32bit
        XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | (zz >>> 32 != 0L ? XEiJ.REG_CCR_V : 0);
      } else {  //64bit
        XEiJ.regRn[h] = (int) (zz >>> 32);
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (zz > 0L ? 0 : zz < 0L ? XEiJ.REG_CCR_N : XEiJ.REG_CCR_Z);
      }
    } else {  //MULS
      long zz = xx * yy;
      int z = XEiJ.regRn[l] = (int) zz;
      if (q == 0) {  //32bit
        XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | (z != zz ? XEiJ.REG_CCR_V : 0);
      } else {  //64bit
        XEiJ.regRn[h] = (int) (zz >> 32);
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (zz > 0L ? 0 : zz < 0L ? XEiJ.REG_CCR_N : XEiJ.REG_CCR_Z);
      }
    }
    if (M30_DIV_ZERO_V_FLAG) {
      m30DivZeroVFlag = false;
    }
  }  //irpMuluMulsLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //DIVU.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000qqq
  //DIVUL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000rrr        (q is not equal to r)
  //DIVU.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq010000000rrr        (q is not equal to r)
  //DIVS.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000qqq
  //DIVSL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000rrr        (q is not equal to r)
  //DIVS.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq110000000rrr        (q is not equal to r)
  //
  //DIVS.L <ea>,Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit商Dq
  //
  //DIVS.L <ea>,Dr:Dq
  //  64bit被除数Dr:Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  //  M68000PRMでDIVS.Lのアドレッシングモードがデータ可変と書かれているのはデータの間違い
  //
  //DIVSL.L <ea>,Dr:Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  //
  //DIVU.L <ea>,Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit商Dq
  //
  //DIVU.L <ea>,Dr:Dq
  //  64bit被除数Dr:Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  //
  //DIVUL.L <ea>,Dr:Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  public static void irpDivuDivsLong () throws M68kException {
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    if ((w & ~0b0111_110_000_000_111) != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    int l = w >> 12;  //被除数の下位32bit,商
    int s = w & 0b0000_100_000_000_000;  //0=DIVU,1=DIVS
    int q = w & 0b0000_010_000_000_000;  //0=32bit被除数,1=64bit被除数
    int h = w & 7;  //被除数の上位32bit,余り
    int ea = XEiJ.regOC & 63;
    int y = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //除数
    if (s == 0) {  //符号なし。DIVU.L <ea>,*
      XEiJ.mpuCycleCount += 130;  //最大。130*0.6=78
      long yy = (long) y & 0xffffffffL;  //除数
      if (q == 0) {  //符号なし、32bit被除数。DIVU.L <ea>,Dq/DIVUL.L <ea>,Dr:Dq
        if (y == 0) {  //ゼロ除算
          if (h == l) {  //DIVU.L <ea>,Dq
            long xx = (long) XEiJ.regRn[h] << 32 | (long) XEiJ.regRn[l] & 0xffffffffL;  //64bit被除数
            XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                           (xx < 0L ? XEiJ.REG_CCR_N : 0) |  //Nは被除数が負のときセット、さもなくばクリア
                           (xx == 0L ? XEiJ.REG_CCR_Z : 0) |  //Zは被除数が0のときセット、さもなくばクリア
                           (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                           );  //Cは常にクリア
          } else {  //DIVUL.L <ea>,Dr:Dq
            int x = XEiJ.regRn[l];  //32bit被除数
            XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                           (x < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは被除数が負のときセット、さもなくばクリア
                           (x == 0 ? XEiJ.REG_CCR_Z : 0) |  //Zは被除数が0のときセット、さもなくばクリア
                           (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                           );  //Cは常にクリア
          }
          XEiJ.mpuCycleCount += 38 - 34;
          M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
          M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
          throw M68kException.m6eSignal;
        }  //if ゼロ除算
        long xx = (long) XEiJ.regRn[l] & 0xffffffffL;  //32bit被除数
        long zz = (long) ((double) xx / (double) yy);  //double→intのキャストは飽和変換で0xffffffff/0x00000001が0x7fffffffになってしまうのでdouble→longとする
        int z = XEiJ.regRn[l] = (int) zz;  //商
        if (h != l) {
          XEiJ.regRn[h] = (int) (xx - yy * zz);  //余り
        }
        XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                       (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                       (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                       );  //VとCは常にクリア
      } else {  //符号なし、64bit被除数。DIVU.L <ea>,Dr:Dq
        if (y == 0) {  //ゼロ除算
          long xx = (long) XEiJ.regRn[h] << 32 | (long) XEiJ.regRn[l] & 0xffffffffL;  //64bit被除数
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         (((int) xx < 0 && (int) xx != 0x7fffffff) || (int) xx == 0x80000000 ? XEiJ.REG_CCR_N : 0) |  //Nは被除数が$xxxxxxxx7fffffffを除く負または$xxxxxxxx80000000のときセット、さもなくばクリア
                         ((int) xx == 0 ? XEiJ.REG_CCR_Z : 0) |  //Zは被除数が$xxxxxxxx00000000のときセット、さもなくばクリア
                         (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                         );  //Cは常にクリア
          XEiJ.mpuCycleCount += 38 - 34;
          M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
          M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
          throw M68kException.m6eSignal;
        }  //if ゼロ除算
        long xx = (long) XEiJ.regRn[h] << 32 | (long) XEiJ.regRn[l] & 0xffffffffL;  //64bit被除数
        long zz = Long.divideUnsigned (xx, yy);  //商。Long.divideUnsigned(long,long)は1.8から
        int z = (int) zz;  //商の下位32bit
        if (zz >>> 32 != 0L) {  //オーバーフローあり
          //Dr:Dqは変化しない
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         ((int) xx < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは被除数の下位32bitが負のときセット、さもなくばクリア
                         ((int) xx == 0 ? XEiJ.REG_CCR_Z : 0) |  //Zは被除数の下位32bitが0のときセット、さもなくばクリア
                         XEiJ.REG_CCR_V  //Vは常にセット
                         );  //Cは常にクリア
        } else {  //オーバーフローなし
          XEiJ.regRn[l] = (int) zz;  //Dr=商
          if (h != l) {
            XEiJ.regRn[h] = (int) (xx - yy * zz);  //Dq=余り
          }
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                         (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                         );  //VとCは常にクリア
        }  //if オーバーフローあり/オーバーフローなし
      }  //if 32bit被除数/64bit被除数
    } else {  //符号あり。DIVS.L <ea>,*
      XEiJ.mpuCycleCount += 150;  //最大。150*0.6=90
      if (q == 0) {  //符号あり、32bit被除数。DIVS.L <ea>,Dq/DIVSL.L <ea>,Dr:Dq
        long yy = (long) y;  //除数
        if (y == 0) {  //ゼロ除算
          long xx = (long) XEiJ.regRn[h] << 32 | (long) XEiJ.regRn[l] & 0xffffffffL;  //64bit被除数
          if (h == l) {  //DIVS.L <ea>,Dq
            XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                           //Nは常にクリア
                           XEiJ.REG_CCR_Z |  //Zは常にセット
                           (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                           );  //Cは常にクリア
          } else {  //DIVSL.L <ea>,Dr:Dq
            XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                           //Nは常にクリア
                           XEiJ.REG_CCR_Z |  //Zは常にセット
                           (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                           );  //Cは常にクリア
          }  //if DIVS.L <ea>,Dq/DIVSL.L <ea>,Dr:Dq
          XEiJ.mpuCycleCount += 38 - 34;
          M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
          M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
          throw M68kException.m6eSignal;
        }  //if ゼロ除算
        long xx = (long) XEiJ.regRn[l];  //32bit被除数
        long zz = xx / yy;  //商
        if ((int) zz != zz) {  //オーバーフローあり
          //Dqは変化しない
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         (xx == 0xffffffff80000000L && y == -1 ? XEiJ.REG_CCR_Z : 0) |  //Zは0x80000000/-1のときセット、さもなくばクリア
                         XEiJ.REG_CCR_V  //Vは常にセット
                         );  //NとCは常にクリア
        } else {  //オーバーフローなし
          int z = XEiJ.regRn[l] = (int) zz;  //商
          if (h != l) {  //DIVSL.L <ea>,Dr:Dq
            XEiJ.regRn[h] = (int) (xx - yy * zz);  //余り
          }
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                         (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                         );  //VとCは常にクリア
        }  //if オーバーフローあり/オーバーフローなし
      } else {  //符号あり、64bit被除数。DIVS.L <ea>,Dr:Dq
        long yy = (long) y;  //除数
        if (y == 0) {  //ゼロ除算
          long xx = (long) XEiJ.regRn[h] << 32 | (long) XEiJ.regRn[l] & 0xffffffffL;  //64bit被除数
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         //Nは常にクリア
                         XEiJ.REG_CCR_Z |  //Zは常にセット
                         (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                         );  //Cは常にクリア
          XEiJ.mpuCycleCount += 38 - 34;
          M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
          M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
          throw M68kException.m6eSignal;
        }  //if ゼロ除算
        long xx = (long) XEiJ.regRn[h] << 32 | (long) XEiJ.regRn[l] & 0xffffffffL;  //64bit被除数
        long zz = xx / yy;  //商
        if ((int) zz != zz) {  //オーバーフローあり
          int zh = (int) (zz >> 32);
          int zl = (int) zz;
          int xh = (int) (xx >> 32);
          int xl = (int) xx;
          //Dr:Dqは変化しない
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         ((zh == 0x00000000 || zh == 0xffffffff) && zl != 0x00000000
                          ?  //商が$0000000000000000を除く$00000000xxxxxxqqまたは$ffffffff00000000を除く$ffffffffxxxxxxqqのとき
                          (zl << 24 < 0 ? XEiJ.REG_CCR_N : 0) |  //qqが負ならばN=1,さもなくばN=0
                          (zl << 24 == 0 ? XEiJ.REG_CCR_Z : 0)  //qqが0ならばZ=1,さもなくばZ=0
                          :  //商が$0000000000000000を除く$00000000xxxxxxqqまたは$ffffffff00000000を除く$ffffffffxxxxxxqqでないとき
                          (xl == 0x80000000 ||  //被除数が$xxxxxxxx80000000または
                           (xh == 0x80000000 && xl != 0x00000000) ||  //被除数が$8000000000000000を除く$80000000xxxxxxxxまたは
                           (xl == 0x7fffffff && xh != 0x7fffffff) ||  //被除数が$7fffffff7fffffffを除く$xxxxxxxx7fffffffまたは
                           (xh == 0x7fffffff && xl != 0x7fffffff) ||  //被除数が$7fffffff7fffffffを除く$7fffffffxxxxxxxxまたは
                           (xl == 0xffffffff && 0x00000000 <= xh) ? XEiJ.REG_CCR_N : 0) |  //被除数が正で$xxxxxxxxffffffffならばN=1,さもなくばN=0
                          (xl == 0x00000000 ? XEiJ.REG_CCR_Z : 0)) |  //被除数が$xxxxxxxx00000000ならばZ=1,さもなくばZ=0
                         XEiJ.REG_CCR_V  //Vは常にセット
                         );  //Cは常にクリア
        } else {  //オーバーフローなし
          int z = XEiJ.regRn[l] = (int) zz;  //商
          if (h != l) {
            XEiJ.regRn[h] = (int) (xx - yy * zz);  //余り
          }
          XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                         (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                         (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                         );  //VとCは常にクリア
        }  //if オーバーフローあり/オーバーフローなし
      }  //if 32bit被除数/64bit被除数
    }  //if 符号なし/符号あり
    if (M30_DIV_ZERO_V_FLAG) {
      m30DivZeroVFlag = false;
    }
  }  //irpDivuDivsLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SATS.L Dr                                       |-|------|-|-UUUU|-**00|D         |0100_110_010_000_rrr (ISA_B)
  //MOVEM.W <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_010_mmm_rrr-llllllllllllllll
  //
  //SATS.L Dr
  //  VがセットされていたらDrを符号が逆で絶対値が最大の値にする(直前のDrに対する演算を飽和演算にする)
  public static void irpMovemToRegWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //SATS.L Dr
      XEiJ.mpuCycleCount += 4;
      int z = XEiJ.regRn[ea];
      if (XEiJ.TEST_BIT_1_SHIFT ? XEiJ.regCCR << 31 - 1 < 0 : (XEiJ.regCCR & XEiJ.REG_CCR_V) != 0) {  //Vがセットされているとき
        XEiJ.regRn[ea] = z = z >> 31 ^ 0x80000000;  //符号が逆で絶対値が最大の値にする
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //MOVEM.W <ea>,<list>
      int l = XEiJ.busRwze (XEiJ.regPC);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      int arr, a;
      if (ea >> 3 == XEiJ.MMM_MP) {  //(Ar)+
        XEiJ.mpuCycleCount += 12;
        arr = ea - (XEiJ.EA_MP - 8);
        a = XEiJ.regRn[arr];
      } else {  //(Ar)+以外
        XEiJ.mpuCycleCount += 8;
        arr = 16;
        a = efaCntWord (ea);
      }
      int t = a;
      if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
        if ((l & 0x0001) != 0) {
          XEiJ.regRn[ 0] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0002) != 0) {
          XEiJ.regRn[ 1] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0004) != 0) {
          XEiJ.regRn[ 2] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0008) != 0) {
          XEiJ.regRn[ 3] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0010) != 0) {
          XEiJ.regRn[ 4] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0020) != 0) {
          XEiJ.regRn[ 5] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0040) != 0) {
          XEiJ.regRn[ 6] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((byte) l < 0) {  //(l & 0x0080) != 0
          XEiJ.regRn[ 7] = XEiJ.busRws (a);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0100) != 0) {
          XEiJ.regRn[ 8] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0200) != 0) {
          XEiJ.regRn[ 9] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0400) != 0) {
          XEiJ.regRn[10] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0800) != 0) {
          XEiJ.regRn[11] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x1000) != 0) {
          XEiJ.regRn[12] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x2000) != 0) {
          XEiJ.regRn[13] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x4000) != 0) {
          XEiJ.regRn[14] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((short) l < 0) {  //(l & 0x8000) != 0
          XEiJ.regRn[15] = XEiJ.busRws (a);  //符号拡張して32bit全部書き換える
          a += 2;
        }
      } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
        for (int i = 0; i <= 15; i++) {
          if ((l & 0x0001 << i) != 0) {
            XEiJ.regRn[i] = XEiJ.busRws (a);  //(データレジスタも)符号拡張して32bit全部書き換える
            a += 2;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
        l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
        for (int i = 0; l != 0; i++, l <<= 1) {
          if (l < 0) {
            XEiJ.regRn[i] = XEiJ.busRws (a);  //(データレジスタも)符号拡張して32bit全部書き換える
            a += 2;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
        for (int i = 0; l != 0; i++, l >>>= 1) {
          if ((l & 1) != 0) {
            XEiJ.regRn[i] = XEiJ.busRws (a);  //(データレジスタも)符号拡張して32bit全部書き換える
            a += 2;
          }
        }
      } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
        for (int i = 0; l != 0; ) {
          int k = Integer.numberOfTrailingZeros (l);
          XEiJ.regRn[i += k] = XEiJ.busRws (a);  //(データレジスタも)符号拡張して32bit全部書き換える
          a += 2;
          l = l >>> k & ~1;
        }
      }
      //MOVEM.W (Ar)+,<list>で<list>にArが含まれているとき、メモリから読み出したデータを捨ててArをインクリメントする
      XEiJ.regRn[arr] = a;
      XEiJ.mpuCycleCount += a - t << 1;  //2バイト/個→4サイクル/個
    }
  }  //irpMovemToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEM.L <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_011_mmm_rrr-llllllllllllllll
  public static void irpMovemToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    {
      int l = XEiJ.busRwze (XEiJ.regPC);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      int arr, a;
      if (ea >> 3 == XEiJ.MMM_MP) {  //(Ar)+
        XEiJ.mpuCycleCount += 12;
        arr = ea - (XEiJ.EA_MP - 8);
        a = XEiJ.regRn[arr];
      } else {  //(Ar)+以外
        XEiJ.mpuCycleCount += 8;
        arr = 16;
        a = efaCntLong (ea);
      }
      int t = a;
      if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
        if ((l & 0x0001) != 0) {
          XEiJ.regRn[ 0] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0002) != 0) {
          XEiJ.regRn[ 1] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0004) != 0) {
          XEiJ.regRn[ 2] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0008) != 0) {
          XEiJ.regRn[ 3] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0010) != 0) {
          XEiJ.regRn[ 4] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0020) != 0) {
          XEiJ.regRn[ 5] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0040) != 0) {
          XEiJ.regRn[ 6] = XEiJ.busRls (a);
          a += 4;
        }
        if ((byte) l < 0) {  //(l & 0x0080) != 0
          XEiJ.regRn[ 7] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0100) != 0) {
          XEiJ.regRn[ 8] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0200) != 0) {
          XEiJ.regRn[ 9] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0400) != 0) {
          XEiJ.regRn[10] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x0800) != 0) {
          XEiJ.regRn[11] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x1000) != 0) {
          XEiJ.regRn[12] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x2000) != 0) {
          XEiJ.regRn[13] = XEiJ.busRls (a);
          a += 4;
        }
        if ((l & 0x4000) != 0) {
          XEiJ.regRn[14] = XEiJ.busRls (a);
          a += 4;
        }
        if ((short) l < 0) {  //(l & 0x8000) != 0
          XEiJ.regRn[15] = XEiJ.busRls (a);
          a += 4;
        }
      } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
        for (int i = 0; i <= 15; i++) {
          if ((l & 0x0001 << i) != 0) {
            XEiJ.regRn[i] = XEiJ.busRls (a);
            a += 4;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
        l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
        for (int i = 0; l != 0; i++, l <<= 1) {
          if (l < 0) {
            XEiJ.regRn[i] = XEiJ.busRls (a);
            a += 4;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
        for (int i = 0; l != 0; i++, l >>>= 1) {
          if ((l & 1) != 0) {
            XEiJ.regRn[i] = XEiJ.busRls (a);
            a += 4;
          }
        }
      } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
        for (int i = 0; l != 0; ) {
          int k = Integer.numberOfTrailingZeros (l);
          XEiJ.regRn[i += k] = XEiJ.busRls (a);
          a += 4;
          l = l >>> k & ~1;
        }
      }
      //MOVEM.L (Ar)+,<list>で<list>にArが含まれているとき、メモリから読み出したデータを捨ててArをインクリメントする
      XEiJ.regRn[arr] = a;
      XEiJ.mpuCycleCount += a - t << 1;  //4バイト/個→8サイクル/個
    }
  }  //irpMovemToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TRAP #<vector>                                  |-|012346|-|-----|-----|          |0100_111_001_00v_vvv
  public static void irpTrap () throws M68kException {
    XEiJ.mpuCycleCount += 34;
    if (XEiJ.MPU_INLINE_EXCEPTION) {
      int save_sr = XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
      int sp = XEiJ.regRn[15];
      XEiJ.regSRT1 = XEiJ.regSRT0 = XEiJ.mpuTraceFlag = 0;  //srのTビットを消す
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        XEiJ.regSRS = XEiJ.REG_SR_S;  //スーパーバイザモードへ移行する
        XEiJ.mpuUSP = sp;  //USPを保存
        sp = XEiJ.regSRM != 0 ? XEiJ.mpuMSP : XEiJ.mpuISP;  //SSPを復元
        if (DataBreakPoint.DBP_ON) {
          DataBreakPoint.dbpMemoryMap = DataBreakPoint.dbpSuperMap;  //スーパーバイザメモリマップに切り替える
        } else {
          XEiJ.busMemoryMap = XEiJ.busSuperMap;  //スーパーバイザメモリマップに切り替える
        }
        if (InstructionBreakPoint.IBP_ON) {
          InstructionBreakPoint.ibpOp1MemoryMap = InstructionBreakPoint.ibpOp1SuperMap;
        }
      }
      XEiJ.regRn[15] = sp -= 8;
      XEiJ.busWw (sp + 6, 0x0000 | XEiJ.regOC - (0x4e40 - M68kException.M6E_TRAP_0_INSTRUCTION_VECTOR) << 2);  //pushw。フォーマットとベクタオフセットをプッシュする
      XEiJ.busWl (sp + 2, XEiJ.regPC);  //pushl。pcをプッシュする
      XEiJ.busWw (sp, save_sr);  //pushw。srをプッシュする
      irpSetPC (XEiJ.busRlsf (XEiJ.mpuVBR + (XEiJ.regOC - (0x4e40 - M68kException.M6E_TRAP_0_INSTRUCTION_VECTOR) << 2)));  //例外ベクタを取り出してジャンプする
    } else {
      irpException (XEiJ.regOC - (0x4e40 - M68kException.M6E_TRAP_0_INSTRUCTION_VECTOR), XEiJ.regPC, XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR, 0x0000, 0);  //pcは次の命令
    }
  }  //irpTrap
  public static void irpTrap15 () throws M68kException {
    if ((XEiJ.regRn[0] & 255) == 0x8e) {  //IOCS _BOOTINF
      MainMemory.mmrCheckHuman ();
    }
    XEiJ.mpuCycleCount += 34;
    if (XEiJ.MPU_INLINE_EXCEPTION) {
      int save_sr = XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
      int sp = XEiJ.regRn[15];
      XEiJ.regSRT1 = XEiJ.regSRT0 = XEiJ.mpuTraceFlag = 0;  //srのTビットを消す
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        XEiJ.regSRS = XEiJ.REG_SR_S;  //スーパーバイザモードへ移行する
        XEiJ.mpuUSP = sp;  //USPを保存
        sp = XEiJ.regSRM != 0 ? XEiJ.mpuMSP : XEiJ.mpuISP;  //SSPを復元
        if (DataBreakPoint.DBP_ON) {
          DataBreakPoint.dbpMemoryMap = DataBreakPoint.dbpSuperMap;  //スーパーバイザメモリマップに切り替える
        } else {
          XEiJ.busMemoryMap = XEiJ.busSuperMap;  //スーパーバイザメモリマップに切り替える
        }
        if (InstructionBreakPoint.IBP_ON) {
          InstructionBreakPoint.ibpOp1MemoryMap = InstructionBreakPoint.ibpOp1SuperMap;
        }
      }
      XEiJ.regRn[15] = sp -= 8;
      XEiJ.busWw (sp + 6, 0x0000 | M68kException.M6E_TRAP_15_INSTRUCTION_VECTOR);  //pushw。フォーマットとベクタオフセットをプッシュする
      XEiJ.busWl (sp + 2, XEiJ.regPC);  //pushl。pcをプッシュする
      XEiJ.busWw (sp, save_sr);  //pushw。srをプッシュする
      irpSetPC (XEiJ.busRlsf (XEiJ.mpuVBR + (M68kException.M6E_TRAP_15_INSTRUCTION_VECTOR << 2)));  //例外ベクタを取り出してジャンプする
    } else {
      irpException (M68kException.M6E_TRAP_15_INSTRUCTION_VECTOR, XEiJ.regPC, XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR, 0x0000, 0);  //pcは次の命令
    }
  }  //irpTrap15

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //LINK.W Ar,#<data>                               |-|012346|-|-----|-----|          |0100_111_001_010_rrr-{data}
  //
  //LINK.W Ar,#<data>
  //  PEA.L (Ar);MOVEA.L A7,Ar;ADDA.W #<data>,A7と同じ
  //  LINK.W A7,#<data>はA7をデクリメントする前の値がプッシュされ、A7に#<data>が加算される
  public static void irpLinkWord () throws M68kException {
    XEiJ.mpuCycleCount += 16;
    int arr = XEiJ.regOC - (0b0100_111_001_010_000 - 8);
    //評価順序に注意
    //  wl(r[15]-=4,r[8+rrr])は不可
    int sp = XEiJ.regRn[15] - 4;
    XEiJ.busWl (sp, XEiJ.regRn[arr]);  //pushl
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.regRn[15] = (XEiJ.regRn[arr] = sp) + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.regRn[15] = (XEiJ.regRn[arr] = sp) + XEiJ.busRwse (t);  //pcws
    }
  }  //irpLinkWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //UNLK Ar                                         |-|012346|-|-----|-----|          |0100_111_001_011_rrr
  //
  //UNLK Ar
  //  MOVEA.L Ar,A7;MOVEA.L (A7)+,Arと同じ
  //  UNLK A7はMOVEA.L A7,A7;MOVEA.L (A7)+,A7すなわちMOVEA.L (A7),A7と同じ
  //  ソースオペランドのポストインクリメントはデスティネーションオペランドが評価される前に完了しているとみなされる
  //    例えばMOVE.L (A0)+,(A0)+はMOVE.L (A0),(4,A0);ADDQ.L #8,A0と同じ
  //    MOVEA.L (A0)+,A0はポストインクリメントされたA0が(A0)から読み出された値で上書きされるのでMOVEA.L (A0),A0と同じ
  //  M68000PRMにUNLK Anの動作はAn→SP;(SP)→An;SP+4→SPだと書かれているがこれはn=7の場合に当てはまらない
  //  余談だが68040の初期のマスクセットはUNLK A7を実行すると固まるらしい
  public static void irpUnlk () throws M68kException {
    XEiJ.mpuCycleCount += 12;
    int arr = XEiJ.regOC - (0b0100_111_001_011_000 - 8);
    //評価順序に注意
    int sp = XEiJ.regRn[arr];
    XEiJ.regRn[15] = sp + 4;
    XEiJ.regRn[arr] = XEiJ.busRls (sp);  //popls
  }  //irpUnlk

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L Ar,USP                                   |-|012346|P|-----|-----|          |0100_111_001_100_rrr
  public static void irpMoveToUsp () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 4;
    XEiJ.mpuUSP = XEiJ.regRn[XEiJ.regOC - (0b0100_111_001_100_000 - 8)];
  }  //irpMoveToUsp

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L USP,Ar                                   |-|012346|P|-----|-----|          |0100_111_001_101_rrr
  public static void irpMoveFromUsp () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 4;
    XEiJ.regRn[XEiJ.regOC - (0b0100_111_001_101_000 - 8)] = XEiJ.mpuUSP;
  }  //irpMoveFromUsp

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RESET                                           |-|012346|P|-----|-----|          |0100_111_001_110_000
  public static void irpReset () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 132;
    XEiJ.irpReset ();
  }  //irpReset

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOP                                             |-|012346|-|-----|-----|          |0100_111_001_110_001
  public static void irpNop () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    //何もしない
  }  //irpNop

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //STOP #<data>                                    |-|012346|P|UUUUU|*****|          |0100_111_001_110_010-{data}
  //
  //STOP #<data>
  //    1. #<data>をsrに設定する
  //    2. pcを進める
  //    3. 以下のいずれかの条件が成立するまで停止する
  //      3a. トレース
  //      3b. マスクされているレベルよりも高い割り込み要求
  //      3c. リセット
  //  コアと一緒にデバイスを止めるわけにいかないので、ここでは条件が成立するまで同じ命令を繰り返すループ命令として実装する
  public static void irpStop () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 4;
    irpSetSR (XEiJ.busRwse ((XEiJ.regPC += 2) - 2));  //pcws。特権違反チェックが先
    if (XEiJ.mpuTraceFlag == 0) {  //トレースまたはマスクされているレベルよりも高い割り込み要求がない
      XEiJ.regPC = XEiJ.regPC0;  //ループ
      //任意の負荷率を100%に設定しているときSTOP命令が軽すぎると動作周波数が大きくなりすぎて割り込みがかかったとき次に進めなくなる
      //負荷率の計算にSTOP命令で止まっていた時間を含めないことにする
      XEiJ.mpuClockTime += XEiJ.TMR_FREQ * 4 / 1000000;  //4μs。25MHzのとき100clk
      XEiJ.mpuLastNano += 4000L;
    }
  }  //irpStop

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTE                                             |-|012346|P|UUUUU|*****|          |0100_111_001_110_011
  public static void irpRte () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 20;
    int sp = XEiJ.regRn[15];
    int format = XEiJ.busRws (sp + 6) & 0xf000;
    XEiJ.regRn[15] = sp + (format == 0x0000 ? 8 :  //010,020,030,040,060
                     format == 0x1000 ? 8 :  //020,030,040
                     format == 0x2000 ? 12 :  //020,030,040,060
                     //format == 0x3000 ? 12 :  //040,060
                     //format == 0x4000 ? 16 :  //060
                     //format == 0x7000 ? 60 :  //040
                     //format == 0x8000 ? 58 :  //010
                     format == 0x9000 ? 20 :  //020,030
                     format == 0xa000 ? 32 :  //020,030
                     format == 0xb000 ? 92 :  //020,030
                     8);  //???
    int newSR = XEiJ.busRwz (sp);  //popwz。ここでバスエラーが生じる可能性がある
    int newPC = XEiJ.busRls (sp + 2);  //popls
    //irpSetSRでモードが切り替わる場合があるのでその前にr[15]を更新しておくこと
    irpSetSR (newSR);  //ここでユーザモードに戻る場合がある。特権違反チェックが先
    irpSetPC (newPC);  //分岐ログが新しいsrを使う。順序に注意
    if (format == 0x1000) {  //スローアウェイフレームだったとき
      sp = XEiJ.regRn[15];
      format = XEiJ.busRws (sp + 6) & 0xf000;
      XEiJ.regRn[15] = sp + (format == 0x0000 ? 8 :  //010,020,030,040,060
                       format == 0x1000 ? 8 :  //020,030,040
                       format == 0x2000 ? 12 :  //020,030,040,060
                       //format == 0x3000 ? 12 :  //040,060
                       //format == 0x4000 ? 16 :  //060
                       //format == 0x7000 ? 60 :  //040
                       //format == 0x8000 ? 58 :  //010
                       format == 0x9000 ? 20 :  //020,030
                       format == 0xa000 ? 32 :  //020,030
                       format == 0xb000 ? 92 :  //020,030
                       8);  //???
      newSR = XEiJ.busRwz (sp);  //popwz。ここでバスエラーが生じる可能性がある
      newPC = XEiJ.busRlse (sp + 2);  //popls
      //irpSetSRでモードが切り替わる場合があるのでその前にr[15]を更新しておくこと
      irpSetSR (newSR);  //ここでユーザモードに戻る場合がある。特権違反チェックが先
      irpSetPC (newPC);  //分岐ログが新しいsrを使う。順序に注意
    }
  }  //irpRte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTD #<data>                                     |-|-12346|-|-----|-----|          |0100_111_001_110_100-{data}
  public static void irpRtd () throws M68kException {
    XEiJ.mpuCycleCount += 20;
    int sp = XEiJ.regRn[15];
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      XEiJ.regRn[15] = sp + 4 + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
    } else {
      int t = XEiJ.regPC;
      XEiJ.regPC = t + 2;
      XEiJ.regRn[15] = sp + 4 + XEiJ.busRwse (t);  //pcws
    }
    irpSetPC (XEiJ.busRls (sp));  //popls
  }  //irpRtd

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTS                                             |-|012346|-|-----|-----|          |0100_111_001_110_101
  public static void irpRts () throws M68kException {
    XEiJ.mpuCycleCount += 16;
    int sp = XEiJ.regRn[15];
    XEiJ.regRn[15] = sp + 4;
    irpSetPC (XEiJ.busRls (sp));  //popls
  }  //irpRts

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TRAPV                                           |-|012346|-|---*-|-----|          |0100_111_001_110_110
  public static void irpTrapv () throws M68kException {
    if (XEiJ.TEST_BIT_1_SHIFT ? XEiJ.regCCR << 31 - 1 >= 0 : (XEiJ.regCCR & XEiJ.REG_CCR_V) == 0) {  //通過
      XEiJ.mpuCycleCount += 4;
    } else {
      M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpTrapv

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTR                                             |-|012346|-|UUUUU|*****|          |0100_111_001_110_111
  public static void irpRtr () throws M68kException {
    XEiJ.mpuCycleCount += 20;
    int sp = XEiJ.regRn[15];
    XEiJ.regRn[15] = sp + 6;
    XEiJ.regCCR = XEiJ.REG_CCR_MASK & XEiJ.busRwz (sp);  //popwz
    irpSetPC (XEiJ.busRls (sp + 2));  //popls
  }  //irpRtr

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEC.L Rc,Rn                                   |-|-12346|P|-----|-----|          |0100_111_001_111_010-rnnncccccccccccc
  public static void irpMovecFromControl () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 10;
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    switch (w & 0x0fff) {
    case 0x000:  //SFC
      XEiJ.regRn[w >> 12] = XEiJ.mpuSFC;
      break;
    case 0x001:  //DFC
      XEiJ.regRn[w >> 12] = XEiJ.mpuDFC;
      break;
    case 0x002:  //CACR
      XEiJ.regRn[w >> 12] = XEiJ.mpuCACR;
      break;
    case 0x800:  //USP
      XEiJ.regRn[w >> 12] = XEiJ.mpuUSP;
      break;
    case 0x801:  //VBR
      XEiJ.regRn[w >> 12] = XEiJ.mpuVBR;
      break;
    case 0x802:  //CAAR
      XEiJ.regRn[w >> 12] = XEiJ.mpuCAAR;
      break;
    case 0x803:  //MSP
      XEiJ.regRn[w >> 12] = XEiJ.regSRM != 0 ? XEiJ.regRn[15] : XEiJ.mpuMSP;
      break;
    case 0x804:  //ISP
      XEiJ.regRn[w >> 12] = XEiJ.regSRM != 0 ? XEiJ.mpuISP : XEiJ.regRn[15];
      break;
    default:
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpMovecFromControl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEC.L Rn,Rc                                   |-|-12346|P|-----|-----|          |0100_111_001_111_011-rnnncccccccccccc
  public static void irpMovecToControl () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 12;
    int w;
    if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
      w = XEiJ.busRwze ((XEiJ.regPC += 2) - 2);  //pcwz。拡張ワード
    } else {
      w = XEiJ.regPC;
      XEiJ.regPC = w + 2;
      w = XEiJ.busRwze (w);  //pcwz。拡張ワード
    }
    int d = XEiJ.regRn[w >> 12];
    switch (w & 0x0fff) {
    case 0x000:  //SFC
      XEiJ.mpuSFC = d & 0x00000007;
      break;
    case 0x001:  //DFC
      XEiJ.mpuDFC = d & 0x00000007;
      break;
    case 0x002:  //CACR
      XEiJ.mpuCACR = d & 0x00003f1f;
      {
        boolean cacheOn = (XEiJ.mpuCACR & 0x00000101) != 0;
        if (XEiJ.mpuCacheOn != cacheOn) {
          XEiJ.mpuCacheOn = cacheOn;
          XEiJ.mpuSetWait ();
        }
      }
      break;
    case 0x800:  //USP
      XEiJ.mpuUSP = d;
      break;
    case 0x801:  //VBR
      XEiJ.mpuVBR = d & -4;  //4の倍数でないと困る
      break;
    case 0x802:  //CAAR
      XEiJ.mpuCAAR = d;
      break;
    case 0x803:  //MSP
      if (XEiJ.regSRM != 0) {
        XEiJ.regRn[15] = d;
      } else {
        XEiJ.mpuMSP = d;
      }
      break;
    case 0x804:  //ISP
      if (XEiJ.regSRM != 0) {
        XEiJ.mpuISP = d;
      } else {
        XEiJ.regRn[15] = d;
      }
      break;
    default:
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpMovecToControl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //JSR <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_010_mmm_rrr
  //JBSR.L <label>                                  |A|012346|-|-----|-----|          |0100_111_010_111_001-{address}       [JSR <label>]
  public static void irpJsr () throws M68kException {
    XEiJ.mpuCycleCount += 16 - 8;
    int a = efaJmpJsr (XEiJ.regOC & 63);  //プッシュする前に実効アドレスを計算する
    XEiJ.busWl (XEiJ.regRn[15] -= 4, XEiJ.regPC);  //pushl
    irpSetPC (a);
  }  //irpJsr

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //JMP <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_011_mmm_rrr
  //JBRA.L <label>                                  |A|012346|-|-----|-----|          |0100_111_011_111_001-{address}       [JMP <label>]
  public static void irpJmp () throws M68kException {
    //XEiJ.mpuCycleCount += 8 - 8;
    irpSetPC (efaJmpJsr (XEiJ.regOC & 63));
  }  //irpJmp

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_000_mmm_rrr
  //INC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_000_mmm_rrr [ADDQ.B #1,<ea>]
  public static void irpAddqByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDQ.B #<data>,Dr
      XEiJ.mpuCycleCount += 4;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) + y);
    } else {  //ADDQ.B #<data>,<mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = (byte) ((x = XEiJ.busRbs (a)) + y));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (~x & z) >>> 31 << 1 |
           (x & ~z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addq
  }  //irpAddqByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_001_mmm_rrr
  //ADDQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_001_001_rrr
  //INC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_001_mmm_rrr [ADDQ.W #1,<ea>]
  //INC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_001_001_rrr [ADDQ.W #1,Ar]
  //
  //ADDQ.W #<data>,Ar
  //  ソースを符号拡張してロングで加算する
  public static void irpAddqWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //ADDQ.W #<data>,Ar
      XEiJ.mpuCycleCount += 8;  //MC68000 User's Manualに4と書いてあるのは8の間違い
      XEiJ.regRn[ea] += y;  //ロングで計算する。このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //ADDQ.W #<data>,Dr
        XEiJ.mpuCycleCount += 4;
        z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) + y));
      } else {  //ADDQ.W #<data>,<mem>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltWord (ea);
        XEiJ.busWw (a, z = (short) ((x = XEiJ.busRws (a)) + y));
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (~x & z) >>> 31 << 1 |
             (x & ~z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addq
    }
  }  //irpAddqWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_010_mmm_rrr
  //ADDQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_010_001_rrr
  //INC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_010_mmm_rrr [ADDQ.L #1,<ea>]
  //INC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_010_001_rrr [ADDQ.L #1,Ar]
  public static void irpAddqLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //ADDQ.L #<data>,Ar
      XEiJ.mpuCycleCount += 8;
      XEiJ.regRn[ea] += y;  //このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //ADDQ.L #<data>,Dr
        XEiJ.mpuCycleCount += 8;
        XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) + y;
      } else {  //ADDQ.L #<data>,<mem>
        XEiJ.mpuCycleCount += 12;
        int a = efaMltLong (ea);
        XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) + y);
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (~x & z) >>> 31 << 1 |
             (x & ~z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addq
    }
  }  //irpAddqLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ST.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr
  //SNF.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr [ST.B <ea>]
  //DBT.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}
  //DBNF.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}        [DBT.W Dr,<label>]
  //TRAPT.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_010-{data}
  //TPNF.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
  //TPT.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
  //TRAPNF.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
  //TRAPT.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_011-{data}
  //TPNF.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
  //TPT.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
  //TRAPNF.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
  //TRAPT                                           |-|--2346|-|-----|-----|          |0101_000_011_111_100
  //TPNF                                            |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
  //TPT                                             |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
  //TRAPNF                                          |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
  public static void irpSt () throws M68kException {
    int ea = XEiJ.regOC & 63;
    //DBT.W Dr,<label>よりもST.B Drを優先する
    if (ea < XEiJ.EA_AR) {  //ST.B Dr
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[ea] |= 0xff;
    } else if (ea < XEiJ.EA_MM) {  //DBT.W Dr,<label>
      //条件が成立しているので通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPT.W/TRAPT.L/TRAPT
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      //条件が成立しているのでTRAPする
      XEiJ.mpuCycleCount += t << 1;
      M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //ST.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), 0xff);
    }
  }  //irpSt

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_100_mmm_rrr
  //DEC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_100_mmm_rrr [SUBQ.B #1,<ea>]
  public static void irpSubqByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBQ.B #<data>,Dr
      XEiJ.mpuCycleCount += 4;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) - y);
    } else {  //SUBQ.B #<data>,<mem>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      XEiJ.busWb (a, z = (byte) ((x = XEiJ.busRbs (a)) - y));
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (x & ~z) >>> 31 << 1 |
           (~x & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subq
  }  //irpSubqByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_101_mmm_rrr
  //SUBQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_101_001_rrr
  //DEC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_101_mmm_rrr [SUBQ.W #1,<ea>]
  //DEC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_101_001_rrr [SUBQ.W #1,Ar]
  //
  //SUBQ.W #<data>,Ar
  //  ソースを符号拡張してロングで減算する
  public static void irpSubqWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //SUBQ.W #<data>,Ar
      XEiJ.mpuCycleCount += 8;
      XEiJ.regRn[ea] -= y;  //ロングで計算する。このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //SUBQ.W #<data>,Dr
        XEiJ.mpuCycleCount += 4;
        z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) - y));
      } else {  //SUBQ.W #<data>,<mem>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltWord (ea);
        XEiJ.busWw (a, z = (short) ((x = XEiJ.busRws (a)) - y));
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (x & ~z) >>> 31 << 1 |
             (~x & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subq
    }
  }  //irpSubqWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_110_mmm_rrr
  //SUBQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_110_001_rrr
  //DEC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_110_mmm_rrr [SUBQ.L #1,<ea>]
  //DEC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_110_001_rrr [SUBQ.L #1,Ar]
  public static void irpSubqLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //SUBQ.L #<data>,Ar
      XEiJ.mpuCycleCount += 8;
      XEiJ.regRn[ea] -= y;  //このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //SUBQ.L #<data>,Dr
        XEiJ.mpuCycleCount += 8;
        XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) - y;
      } else {  //SUBQ.L #<data>,<mem>
        XEiJ.mpuCycleCount += 12;
        int a = efaMltLong (ea);
        XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) - y);
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (x & ~z) >>> 31 << 1 |
             (~x & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subq
    }
  }  //irpSubqLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SF.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr
  //SNT.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr [SF.B <ea>]
  //DBF.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}
  //DBNT.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
  //DBRA.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
  //TRAPF.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_010-{data}
  //TPF.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
  //TPNT.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
  //TRAPNT.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
  //TRAPF.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_011-{data}
  //TPF.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
  //TPNT.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
  //TRAPNT.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
  //TRAPF                                           |-|--2346|-|-----|-----|          |0101_000_111_111_100
  //TPF                                             |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
  //TPNT                                            |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
  //TRAPNT                                          |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
  public static void irpSf () throws M68kException {
    int ea = XEiJ.regOC & 63;
    //DBRA.W Dr,<label>よりもSF.B Drを優先する
    if (ea < XEiJ.EA_AR) {  //SF.B Dr
      XEiJ.mpuCycleCount += 4;
      XEiJ.regRn[ea] &= ~0xff;
    } else if (ea < XEiJ.EA_MM) {  //DBRA.W Dr,<label>
      //条件が成立していないのでデクリメント
      int rrr = XEiJ.regOC & 7;
      int t = XEiJ.regRn[rrr];
      if ((short) t == 0) {  //Drの下位16bitが0なので通過
        XEiJ.mpuCycleCount += 14;
        XEiJ.regRn[rrr] = t + 65535;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {  //Drの下位16bitが0でないのでジャンプ
        XEiJ.mpuCycleCount += 10;
        XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
        irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPF.W/TRAPF.L/TRAPF
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      //条件が成立していないのでTRAPしない
      XEiJ.mpuCycleCount += 4 + (t << 1);
    } else {  //SF.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), 0x00);
    }
  }  //irpSf

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SHI.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr
  //SNLS.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr [SHI.B <ea>]
  //DBHI.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}
  //DBNLS.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}        [DBHI.W Dr,<label>]
  //TRAPHI.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}
  //TPHI.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
  //TPNLS.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
  //TRAPNLS.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
  //TRAPHI.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}
  //TPHI.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
  //TPNLS.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
  //TRAPNLS.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
  //TRAPHI                                          |-|--2346|-|--*-*|-----|          |0101_001_011_111_100
  //TPHI                                            |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
  //TPNLS                                           |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
  //TRAPNLS                                         |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
  public static void irpShi () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBHI.W Dr,<label>
      if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SHI.B Dr
      if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPHI.W/TRAPHI.L/TRAPHI
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SHI.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_HI << XEiJ.regCCR >> 31);
    }
  }  //irpShi

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SLS.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr
  //SNHI.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr [SLS.B <ea>]
  //DBLS.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}
  //DBNHI.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}        [DBLS.W Dr,<label>]
  //TRAPLS.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}
  //TPLS.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
  //TPNHI.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
  //TRAPNHI.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
  //TRAPLS.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}
  //TPLS.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
  //TPNHI.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
  //TRAPNHI.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
  //TRAPLS                                          |-|--2346|-|--*-*|-----|          |0101_001_111_111_100
  //TPLS                                            |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
  //TPNHI                                           |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
  //TRAPNHI                                         |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
  public static void irpSls () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLS.W Dr,<label>
      if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SLS.B Dr
      if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLS.W/TRAPLS.L/TRAPLS
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SLS.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_LS << XEiJ.regCCR >> 31);
    }
  }  //irpSls

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SCC.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr
  //SHS.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
  //SNCS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
  //SNLO.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
  //DBCC.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}
  //DBHS.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
  //DBNCS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
  //DBNLO.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
  //TRAPCC.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_010-{data}
  //TPCC.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TPHS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TPNCS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TPNLO.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPHS.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPNCS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPNLO.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPCC.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_011-{data}
  //TPCC.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TPHS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TPNCS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TPNLO.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPHS.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPNCS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPNLO.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPCC                                          |-|--2346|-|----*|-----|          |0101_010_011_111_100
  //TPCC                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TPHS                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TPNCS                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TPNLO                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TRAPHS                                          |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TRAPNCS                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TRAPNLO                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  public static void irpShs () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBHS.W Dr,<label>
      if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SHS.B Dr
      if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPHS.W/TRAPHS.L/TRAPHS
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SHS.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_HS << XEiJ.regCCR >> 31);
    }
  }  //irpShs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SCS.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr
  //SLO.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
  //SNCC.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
  //SNHS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
  //DBCS.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}
  //DBLO.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
  //DBNCC.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
  //DBNHS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
  //TRAPCS.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_010-{data}
  //TPCS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TPLO.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TPNCC.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TPNHS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPLO.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPNCC.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPNHS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPCS.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_011-{data}
  //TPCS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TPLO.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TPNCC.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TPNHS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPLO.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPNCC.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPNHS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPCS                                          |-|--2346|-|----*|-----|          |0101_010_111_111_100
  //TPCS                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TPLO                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TPNCC                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TPNHS                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TRAPLO                                          |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TRAPNCC                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TRAPNHS                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  public static void irpSlo () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLO.W Dr,<label>
      if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SLO.B Dr
      if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLO.W/TRAPLO.L/TRAPLO
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SLO.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_LO << XEiJ.regCCR >> 31);
    }
  }  //irpSlo

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SNE.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr
  //SNEQ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
  //SNZ.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
  //SNZE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
  //DBNE.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}
  //DBNEQ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
  //DBNZ.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
  //DBNZE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
  //TRAPNE.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}
  //TPNE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TPNEQ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TPNZ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TPNZE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNEQ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNZ.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNZE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNE.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}
  //TPNE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TPNEQ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TPNZ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TPNZE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNEQ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNZ.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNZE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNE                                          |-|--2346|-|--*--|-----|          |0101_011_011_111_100
  //TPNE                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TPNEQ                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TPNZ                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TPNZE                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TRAPNEQ                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TRAPNZ                                          |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TRAPNZE                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  public static void irpSne () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBNE.W Dr,<label>
      if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SNE.B Dr
      if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPNE.W/TRAPNE.L/TRAPNE
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SNE.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_NE << XEiJ.regCCR >> 31);
    }
  }  //irpSne

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SEQ.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr
  //SNNE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
  //SNNZ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
  //SZE.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
  //DBEQ.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}
  //DBNNE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
  //DBNNZ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
  //DBZE.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
  //TRAPEQ.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}
  //TPEQ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TPNNE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TPNNZ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TPZE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPNNE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPNNZ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPZE.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPEQ.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}
  //TPEQ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TPNNE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TPNNZ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TPZE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPNNE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPNNZ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPZE.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPEQ                                          |-|--2346|-|--*--|-----|          |0101_011_111_111_100
  //TPEQ                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TPNNE                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TPNNZ                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TPZE                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TRAPNNE                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TRAPNNZ                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TRAPZE                                          |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  public static void irpSeq () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBEQ.W Dr,<label>
      if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SEQ.B Dr
      if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPEQ.W/TRAPEQ.L/TRAPEQ
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SEQ.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_EQ << XEiJ.regCCR >> 31);
    }
  }  //irpSeq

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SVC.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr
  //SNVS.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr [SVC.B <ea>]
  //DBVC.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}
  //DBNVS.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}        [DBVC.W Dr,<label>]
  //TRAPVC.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}
  //TPNVS.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
  //TPVC.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
  //TRAPNVS.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
  //TRAPVC.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}
  //TPNVS.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
  //TPVC.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
  //TRAPNVS.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
  //TRAPVC                                          |-|--2346|-|---*-|-----|          |0101_100_011_111_100
  //TPNVS                                           |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
  //TPVC                                            |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
  //TRAPNVS                                         |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
  public static void irpSvc () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBVC.W Dr,<label>
      if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SVC.B Dr
      if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPVC.W/TRAPVC.L/TRAPVC
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SVC.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_VC << XEiJ.regCCR >> 31);
    }
  }  //irpSvc

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SVS.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr
  //SNVC.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr [SVS.B <ea>]
  //DBVS.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}
  //DBNVC.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}        [DBVS.W Dr,<label>]
  //TRAPVS.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}
  //TPNVC.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
  //TPVS.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
  //TRAPNVC.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
  //TRAPVS.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}
  //TPNVC.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
  //TPVS.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
  //TRAPNVC.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
  //TRAPVS                                          |-|--2346|-|---*-|-----|          |0101_100_111_111_100
  //TPNVC                                           |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
  //TPVS                                            |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
  //TRAPNVC                                         |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
  public static void irpSvs () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBVS.W Dr,<label>
      if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SVS.B Dr
      if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPVS.W/TRAPVS.L/TRAPVS
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SVS.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_VS << XEiJ.regCCR >> 31);
    }
  }  //irpSvs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SPL.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr
  //SNMI.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr [SPL.B <ea>]
  //DBPL.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}
  //DBNMI.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}        [DBPL.W Dr,<label>]
  //TRAPPL.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}
  //TPNMI.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
  //TPPL.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
  //TRAPNMI.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
  //TRAPPL.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}
  //TPNMI.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
  //TPPL.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
  //TRAPNMI.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
  //TRAPPL                                          |-|--2346|-|-*---|-----|          |0101_101_011_111_100
  //TPNMI                                           |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
  //TPPL                                            |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
  //TRAPNMI                                         |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
  public static void irpSpl () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBPL.W Dr,<label>
      if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SPL.B Dr
      if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPPL.W/TRAPPL.L/TRAPPL
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SPL.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_PL << XEiJ.regCCR >> 31);
    }
  }  //irpSpl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SMI.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr
  //SNPL.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr [SMI.B <ea>]
  //DBMI.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}
  //DBNPL.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}        [DBMI.W Dr,<label>]
  //TRAPMI.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}
  //TPMI.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
  //TPNPL.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
  //TRAPNPL.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
  //TRAPMI.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}
  //TPMI.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
  //TPNPL.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
  //TRAPNPL.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
  //TRAPMI                                          |-|--2346|-|-*---|-----|          |0101_101_111_111_100
  //TPMI                                            |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
  //TPNPL                                           |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
  //TRAPNPL                                         |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
  public static void irpSmi () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBMI.W Dr,<label>
      if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SMI.B Dr
      if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPMI.W/TRAPMI.L/TRAPMI
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SMI.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_MI << XEiJ.regCCR >> 31);
    }
  }  //irpSmi

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SGE.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr
  //SNLT.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr [SGE.B <ea>]
  //DBGE.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}
  //DBNLT.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}        [DBGE.W Dr,<label>]
  //TRAPGE.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}
  //TPGE.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
  //TPNLT.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
  //TRAPNLT.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
  //TRAPGE.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}
  //TPGE.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
  //TPNLT.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
  //TRAPNLT.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
  //TRAPGE                                          |-|--2346|-|-*-*-|-----|          |0101_110_011_111_100
  //TPGE                                            |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
  //TPNLT                                           |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
  //TRAPNLT                                         |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
  public static void irpSge () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBGE.W Dr,<label>
      if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SGE.B Dr
      if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPGE.W/TRAPGE.L/TRAPGE
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SGE.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_GE << XEiJ.regCCR >> 31);
    }
  }  //irpSge

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SLT.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr
  //SNGE.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr [SLT.B <ea>]
  //DBLT.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}
  //DBNGE.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}        [DBLT.W Dr,<label>]
  //TRAPLT.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}
  //TPLT.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
  //TPNGE.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
  //TRAPNGE.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
  //TRAPLT.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}
  //TPLT.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
  //TPNGE.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
  //TRAPNGE.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
  //TRAPLT                                          |-|--2346|-|-*-*-|-----|          |0101_110_111_111_100
  //TPLT                                            |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
  //TPNGE                                           |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
  //TRAPNGE                                         |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
  public static void irpSlt () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLT.W Dr,<label>
      if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SLT.B Dr
      if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLT.W/TRAPLT.L/TRAPLT
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SLT.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_LT << XEiJ.regCCR >> 31);
    }
  }  //irpSlt

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SGT.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr
  //SNLE.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr [SGT.B <ea>]
  //DBGT.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}
  //DBNLE.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}        [DBGT.W Dr,<label>]
  //TRAPGT.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}
  //TPGT.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
  //TPNLE.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
  //TRAPNLE.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
  //TRAPGT.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}
  //TPGT.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
  //TPNLE.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
  //TRAPNLE.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
  //TRAPGT                                          |-|--2346|-|-***-|-----|          |0101_111_011_111_100
  //TPGT                                            |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
  //TPNLE                                           |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
  //TRAPNLE                                         |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
  public static void irpSgt () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBGT.W Dr,<label>
      if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SGT.B Dr
      if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPGT.W/TRAPGT.L/TRAPGT
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SGT.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_GT << XEiJ.regCCR >> 31);
    }
  }  //irpSgt

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SLE.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr
  //SNGT.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr [SLE.B <ea>]
  //DBLE.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}
  //DBNGT.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}        [DBLE.W Dr,<label>]
  //TRAPLE.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}
  //TPLE.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
  //TPNGT.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
  //TRAPNGT.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
  //TRAPLE.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}
  //TPLE.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
  //TPNGT.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
  //TRAPNGT.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
  //TRAPLE                                          |-|--2346|-|-***-|-----|          |0101_111_111_111_100
  //TPLE                                            |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
  //TPNGT                                           |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
  //TRAPNGT                                         |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
  public static void irpSle () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLE.W Dr,<label>
      if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {
        //条件が成立しているので通過
        XEiJ.mpuCycleCount += 12;
        XEiJ.regPC += 2;  //オフセットを読み飛ばす
      } else {
        //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int t = XEiJ.regRn[rrr];
        if ((short) t == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 14;
          XEiJ.regRn[rrr] = t + 65535;
          XEiJ.regPC += 2;  //オフセットを読み飛ばす
        } else {  //Drの下位16bitが0でないのでジャンプ
          XEiJ.mpuCycleCount += 10;
          XEiJ.regRn[rrr] = t - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (XEiJ.regPC + XEiJ.busRws (XEiJ.regPC));  //pc==pc0+2
        }
      }
      if (M30_DIV_ZERO_V_FLAG) {
        m30DivZeroVFlag = !m30DivZeroVFlag;
      }
    } else if (ea < XEiJ.EA_AR) {  //SLE.B Dr
      if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount += 6;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount += 4;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLE.W/TRAPLE.L/TRAPLE
      int t = (ea & 3) + (ea & 1);  //111_010→2,111_011→4,111_100→0
      XEiJ.regPC += t;
      if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        XEiJ.mpuCycleCount += t << 1;
        M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount += 4 + (t << 1);
      }
    } else {  //SLE.B <mem>
      XEiJ.mpuCycleCount += 8;
      XEiJ.busWb (efaMltByte (ea), XEiJ.MPU_CC_LE << XEiJ.regCCR >> 31);
    }
  }  //irpSle

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.W <label>                                   |-|012346|-|-----|-----|          |0110_000_000_000_000-{offset}
  //JBRA.W <label>                                  |A|012346|-|-----|-----|          |0110_000_000_000_000-{offset}        [BRA.W <label>]
  //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)
  //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)   [BRA.S <label>]
  public static void irpBrasw () throws M68kException {
    XEiJ.mpuCycleCount += 10;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == 0) {  //BRA.W
      XEiJ.regPC = t + 2;
      s = XEiJ.busRwse (t);  //pcws
    }
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBrasw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_001_sss_sss
  //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_001_sss_sss [BRA.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_010_sss_sss
  //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_010_sss_sss [BRA.S <label>]
  public static void irpBras () throws M68kException {
    XEiJ.mpuCycleCount += 10;
    irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
  }  //irpBras

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)
  //JBRA.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)  [BRA.S <label>]
  //BRA.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_111_111-{offset}
  public static void irpBrasl () throws M68kException {
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == -1) {  //BRA.L
      XEiJ.mpuCycleCount += 14;
      XEiJ.regPC = t + 4;
      s = XEiJ.busRlse (t);  //pcls
    } else {  //BRA.S
      XEiJ.mpuCycleCount += 10;
    }
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBrasl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.W <label>                                   |-|012346|-|-----|-----|          |0110_000_100_000_000-{offset}
  //JBSR.W <label>                                  |A|012346|-|-----|-----|          |0110_000_100_000_000-{offset}        [BSR.W <label>]
  //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)
  //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)   [BSR.S <label>]
  public static void irpBsrsw () throws M68kException {
    XEiJ.mpuCycleCount += 18;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == 0) {  //BSR.W
      XEiJ.regPC = t + 2;
      s = XEiJ.busRwse (t);  //pcws
    }
    XEiJ.busWl (XEiJ.regRn[15] -= 4, XEiJ.regPC);  //pushl
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBsrsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_101_sss_sss
  //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_101_sss_sss [BSR.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_110_sss_sss
  //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_110_sss_sss [BSR.S <label>]
  public static void irpBsrs () throws M68kException {
    XEiJ.mpuCycleCount += 18;
    XEiJ.busWl (XEiJ.regRn[15] -= 4, XEiJ.regPC);  //pushl
    irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
  }  //irpBsrs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)
  //JBSR.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)  [BSR.S <label>]
  //BSR.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_111_111-{offset}
  public static void irpBsrsl () throws M68kException {
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == -1) {  //BSR.L
      XEiJ.mpuCycleCount += 22;
      XEiJ.regPC = t + 4;
      s = XEiJ.busRlse (t);  //pcls
    } else {  //BSR.S
      XEiJ.mpuCycleCount += 18;
    }
    XEiJ.busWl (XEiJ.regRn[15] -= 4, XEiJ.regPC);  //pushl
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBsrsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}
  //BNLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
  //JBHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
  //JBNLS.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
  //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)
  //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
  //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
  //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
  //JBLS.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
  //JBNHI.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
  public static void irpBhisw () throws M68kException {
    if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6200) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBhisw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_001_sss_sss
  //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
  //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
  //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_010_sss_sss
  //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
  //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
  //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
  public static void irpBhis () throws M68kException {
    if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBhis

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)
  //BNLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
  //JBHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
  //JBNLS.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
  //BHI.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}
  //BNLS.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}        [BHI.L <label>]
  public static void irpBhisl () throws M68kException {
    if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x62ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBhisl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}
  //BNHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
  //JBLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
  //JBNHI.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
  //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)
  //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
  //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
  //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
  //JBHI.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
  //JBNLS.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
  public static void irpBlssw () throws M68kException {
    if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6300) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlssw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_101_sss_sss
  //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
  //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
  //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_110_sss_sss
  //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
  //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
  //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
  public static void irpBlss () throws M68kException {
    if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlss

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)
  //BNHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
  //JBLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
  //JBNHI.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
  //BLS.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}
  //BNHI.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}        [BLS.L <label>]
  public static void irpBlssl () throws M68kException {
    if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x63ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlssl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.W <label>                                   |-|012346|-|----*|-----|          |0110_010_000_000_000-{offset}
  //BHS.W <label>                                   |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //BNCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //BNLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBNCS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBNLO.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)
  //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBCS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  //JBLO.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  //JBNCC.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  //JBNHS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  public static void irpBhssw () throws M68kException {
    if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6400) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBhssw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_001_sss_sss
  //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_010_sss_sss
  //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  public static void irpBhss () throws M68kException {
    if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBhss

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)
  //BHS.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //BNCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //BNLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBNCS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBNLO.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //BCC.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}
  //BHS.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
  //BNCS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
  //BNLO.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
  public static void irpBhssl () throws M68kException {
    if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x64ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBhssl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.W <label>                                   |-|012346|-|----*|-----|          |0110_010_100_000_000-{offset}
  //BLO.W <label>                                   |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //BNCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //BNHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBNCC.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBNHS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)
  //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBCC.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  //JBHS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  //JBNCS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  //JBNLO.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  public static void irpBlosw () throws M68kException {
    if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6500) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlosw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_101_sss_sss
  //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_110_sss_sss
  //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  public static void irpBlos () throws M68kException {
    if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlos

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)
  //BLO.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //BNCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //BNHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBNCC.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBNHS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //BCS.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}
  //BLO.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
  //BNCC.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
  //BNHS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
  public static void irpBlosl () throws M68kException {
    if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x65ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlosl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}
  //BNEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //BNZ.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //BNZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNEQ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNZE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)
  //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBEQ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNEQ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNNE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNNZ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNZ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNZE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBZE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  public static void irpBnesw () throws M68kException {
    if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6600) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBnesw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_001_sss_sss
  //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_010_sss_sss
  //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  public static void irpBnes () throws M68kException {
    if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBnes

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)
  //BNEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //BNZ.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //BNZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNZE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //BNE.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}
  //BNEQ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
  //BNZ.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
  //BNZE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
  public static void irpBnesl () throws M68kException {
    if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x66ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBnesl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}
  //BNNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //BNNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //BZE.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBNNE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBNNZ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)
  //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBNE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_110-0100111011111001-{address}      [BEQ.S (*)+8;JMP <label>]
  public static void irpBeqsw () throws M68kException {
    if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6700) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBeqsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_101_sss_sss
  //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_110_sss_sss
  //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  public static void irpBeqs () throws M68kException {
    if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBeqs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)
  //BNNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //BZE.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //BEQ.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}
  //BNNE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
  //BNNZ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
  //BZE.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
  public static void irpBeqsl () throws M68kException {
    if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x67ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBeqsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}
  //BNVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
  //JBNVS.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
  //JBVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
  //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)
  //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
  //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
  //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
  //JBNVC.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
  //JBVS.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
  public static void irpBvcsw () throws M68kException {
    if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6800) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBvcsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_001_sss_sss
  //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
  //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
  //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_010_sss_sss
  //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
  //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
  //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
  public static void irpBvcs () throws M68kException {
    if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBvcs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)
  //BNVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
  //JBNVS.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
  //JBVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
  //BVC.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}
  //BNVS.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}        [BVC.L <label>]
  public static void irpBvcsl () throws M68kException {
    if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x68ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBvcsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}
  //BNVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
  //JBNVC.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
  //JBVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
  //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)
  //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
  //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
  //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
  //JBNVS.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
  //JBVC.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
  public static void irpBvssw () throws M68kException {
    if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6900) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBvssw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_101_sss_sss
  //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
  //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
  //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_110_sss_sss
  //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
  //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
  //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
  public static void irpBvss () throws M68kException {
    if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBvss

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)
  //BNVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
  //JBNVC.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
  //JBVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
  //BVS.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}
  //BNVC.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}        [BVS.L <label>]
  public static void irpBvssl () throws M68kException {
    if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x69ff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBvssl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}
  //BNMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
  //JBNMI.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
  //JBPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
  //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)
  //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
  //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
  //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
  //JBMI.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
  //JBNPL.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
  public static void irpBplsw () throws M68kException {
    if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6a00) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBplsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_001_sss_sss
  //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
  //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
  //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_010_sss_sss
  //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
  //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
  //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
  public static void irpBpls () throws M68kException {
    if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBpls

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)
  //BNMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
  //JBNMI.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
  //JBPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
  //BPL.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}
  //BNMI.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}        [BPL.L <label>]
  public static void irpBplsl () throws M68kException {
    if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6aff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBplsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}
  //BNPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
  //JBMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
  //JBNPL.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
  //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)
  //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
  //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
  //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
  //JBNMI.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
  //JBPL.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
  public static void irpBmisw () throws M68kException {
    if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6b00) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBmisw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_101_sss_sss
  //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
  //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
  //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_110_sss_sss
  //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
  //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
  //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
  public static void irpBmis () throws M68kException {
    if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBmis

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)
  //BNPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
  //JBMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
  //JBNPL.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
  //BMI.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}
  //BNPL.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}        [BMI.L <label>]
  public static void irpBmisl () throws M68kException {
    if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6bff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBmisl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}
  //BNLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
  //JBGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
  //JBNLT.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
  //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)
  //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
  //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
  //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
  //JBLT.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
  //JBNGE.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
  public static void irpBgesw () throws M68kException {
    if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6c00) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBgesw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_001_sss_sss
  //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
  //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
  //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_010_sss_sss
  //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
  //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
  //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
  public static void irpBges () throws M68kException {
    if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBges

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)
  //BNLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
  //JBGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
  //JBNLT.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
  //BGE.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}
  //BNLT.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}        [BGE.L <label>]
  public static void irpBgesl () throws M68kException {
    if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6cff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBgesl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}
  //BNGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
  //JBLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
  //JBNGE.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
  //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)
  //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
  //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
  //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
  //JBGE.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
  //JBNLT.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
  public static void irpBltsw () throws M68kException {
    if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6d00) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBltsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_101_sss_sss
  //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
  //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
  //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_110_sss_sss
  //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
  //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
  //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
  public static void irpBlts () throws M68kException {
    if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlts

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)
  //BNGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
  //JBLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
  //JBNGE.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
  //BLT.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}
  //BNGE.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}        [BLT.L <label>]
  public static void irpBltsl () throws M68kException {
    if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6dff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBltsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}
  //BNLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
  //JBGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
  //JBNLE.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
  //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)
  //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
  //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
  //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
  //JBLE.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
  //JBNGT.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
  public static void irpBgtsw () throws M68kException {
    if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6e00) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBgtsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_001_sss_sss
  //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
  //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
  //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_010_sss_sss
  //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
  //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
  //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
  public static void irpBgts () throws M68kException {
    if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBgts

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)
  //BNLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
  //JBGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
  //JBNLE.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
  //BGT.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}
  //BNLE.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}        [BGT.L <label>]
  public static void irpBgtsl () throws M68kException {
    if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6eff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBgtsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}
  //BNGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
  //JBLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
  //JBNGT.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
  //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)
  //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
  //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
  //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
  //JBGT.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
  //JBNLE.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
  public static void irpBlesw () throws M68kException {
    if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == 0) {  //Bcc.Wでジャンプ
        XEiJ.regPC = t + 2;
        s = XEiJ.busRwse (t);  //pcws
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6f00) {  //Bcc.Wで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 2;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlesw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_101_sss_sss
  //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
  //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
  //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_110_sss_sss
  //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
  //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
  //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
  public static void irpBles () throws M68kException {
    if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      XEiJ.mpuCycleCount += 10;
      irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBles

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)
  //BNGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
  //JBLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
  //JBNGT.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
  //BLE.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}
  //BNGT.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}        [BLE.L <label>]
  public static void irpBlesl () throws M68kException {
    if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //Bccでジャンプ
      int t = XEiJ.regPC;  //pc0+2
      int s = (byte) XEiJ.regOC;  //オフセット
      if (s == -1) {  //Bcc.Lでジャンプ
        XEiJ.mpuCycleCount += 14;
        XEiJ.regPC = t + 4;
        s = XEiJ.busRlse (t);  //pcls
      } else {  //Bcc.Sでジャンプ
        XEiJ.mpuCycleCount += 10;
      }
      irpSetPC (t + s);  //pc0+2+オフセット
    } else if (XEiJ.regOC == 0x6fff) {  //Bcc.Lで通過
      XEiJ.mpuCycleCount += 12;
      XEiJ.regPC += 4;  //オフセットを読み飛ばす。リードを省略
    } else {  //Bcc.Sで通過
      XEiJ.mpuCycleCount += 8;
    }
  }  //irpBlesl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //IOCS <name>                                     |A|012346|-|UUUUU|UUUUU|          |0111_000_0dd_ddd_ddd-0100111001001111        [MOVEQ.L #<data>,D0;TRAP #15]
  //MOVEQ.L #<data>,Dq                              |-|012346|-|-UUUU|-**00|          |0111_qqq_0dd_ddd_ddd
  public static void irpMoveq () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = (byte) XEiJ.regOC;
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveq

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVS.B <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_100_mmm_rrr (ISA_B)
  //
  //MVS.B <ea>,Dq
  //  バイトデータをロングに符号拡張してDqの全体を更新する
  public static void irpMvsByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? (byte) XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMvsByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVS.W <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_101_mmm_rrr (ISA_B)
  //
  //MVS.W <ea>,Dq
  //  ワードデータをロングに符号拡張してDqの全体を更新する
  public static void irpMvsWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMvsWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVZ.B <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_110_mmm_rrr (ISA_B)
  //
  //MVZ.B <ea>,Dq
  //  バイトデータをロングにゼロ拡張してDqの全体を更新する
  public static void irpMvzByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? 0xff & XEiJ.regRn[ea] : XEiJ.busRbz (efaAnyByte (ea));
    XEiJ.regCCR = XEiJ.REG_CCR_X & XEiJ.regCCR | (z == 0 ? XEiJ.REG_CCR_Z : 0);
  }  //irpMvzByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVZ.W <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_111_mmm_rrr (ISA_B)
  //
  //MVZ.W <ea>,Dq
  //  ワードデータをロングにゼロ拡張してDqの全体を更新する
  public static void irpMvzWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : XEiJ.busRwz (efaAnyWord (ea));
    XEiJ.regCCR = XEiJ.REG_CCR_X & XEiJ.regCCR | (z == 0 ? XEiJ.REG_CCR_Z : 0);
  }  //irpMvzWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //OR.B <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_000_mmm_rrr
  public static void irpOrToRegByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & (XEiJ.regRn[XEiJ.regOC >> 9 & 7] |= 255 & (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea))))];  //ccr_tst_byte。0拡張してからOR
  }  //irpOrToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //OR.W <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_001_mmm_rrr
  public static void irpOrToRegWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z = (short) (XEiJ.regRn[XEiJ.regOC >> 9 & 7] |= ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : XEiJ.busRwz (efaAnyWord (ea)));  //0拡張してからOR
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpOrToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //OR.L <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_010_mmm_rrr
  public static void irpOrToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int z;
    if (ea < XEiJ.EA_AR) {  //OR.L Dr,Dq
      XEiJ.mpuCycleCount += 8;
      XEiJ.regRn[qqq] = z = XEiJ.regRn[qqq] | XEiJ.regRn[ea];
    } else {  //OR.L <mem>,Dq
      XEiJ.mpuCycleCount += ea == XEiJ.EA_IM ? 8 : 6;  //ソースが#<data>のとき2増やす
      XEiJ.regRn[qqq] = z = XEiJ.regRn[qqq] | XEiJ.busRls (efaAnyLong (ea));
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpOrToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //DIVU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_011_mmm_rrr
  //
  //DIVU.W <ea>,Dq
  //  M68000PRMでDIVU.Wのオーバーフローの条件が16bit符号あり整数と書かれているのは16bit符号なし整数の間違い
  public static void irpDivuWord () throws M68kException {
    //  X  変化しない
    //  N  ゼロ除算またはオーバーフローのとき不定。商が負のときセット。それ以外はクリア
    //  Z  ゼロ除算またはオーバーフローのとき不定。商が0のときセット。それ以外はクリア
    //  V  ゼロ除算のとき不定。オーバーフローのときセット。それ以外はクリア
    //  C  常にクリア
    XEiJ.mpuCycleCount += 140;  //最大
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : XEiJ.busRwz (efaAnyWord (ea));  //除数
    int x = XEiJ.regRn[qqq];  //被除数
    if (y == 0) {  //ゼロ除算
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (x < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは被除数が負のときセット、さもなくばクリア
                     (x >> 16 == 0 ? XEiJ.REG_CCR_Z : 0) |  //Zは被除数が$0000xxxxのときセット、さもなくばクリア
                     (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                     );  //Cは常にクリア
      XEiJ.mpuCycleCount += 38 - 140 - 34;
      M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
      throw M68kException.m6eSignal;
    }
    //無理にintで符号なし除算をやろうとするよりもdoubleにキャストしてから割ったほうが速い
    //  intの除算をdoubleの除算器で行うプロセッサならばなおさら
    //被除数を符号なし32ビットとみなすためlongを経由してdoubleに変換する
    //doubleからlongやintへのキャストは小数点以下が切り捨てられ、オーバーフローは表現できる絶対値最大の値になる
    //doubleから直接intに戻しているので0xffffffff/0x0001=0xffffffffが絶対値最大の0x7fffffffになってしまうが、
    //DIVU.Wではオーバーフローになることに変わりはないのでよいことにする
    //  符号なし32ビットの0xffffffffにしたいときは戻すときもlongを経由すればよい
    int z = (int) ((double) ((long) x & 0xffffffffL) / (double) y);  //商
    if (z >>> 16 != 0) {  //オーバーフローあり
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (x < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは被除数が負のときセット、さもなくばクリア
                     //Zは常にクリア
                     XEiJ.REG_CCR_V  //Vは常にセット
                     );  //Cは常にクリア
    } else {  //オーバーフローなし
      XEiJ.regRn[qqq] = x - y * z << 16 | z;  //余り<<16|商
      z = (short) z;
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                     (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                     //Vは常にクリア
                     );  //Cは常にクリア
    }  //if オーバーフローあり/オーバーフローなし
    if (M30_DIV_ZERO_V_FLAG) {
      m30DivZeroVFlag = false;
    }
  }  //irpDivuWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SBCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_000_rrr
  //SBCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_001_rrr
  //OR.B Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_100_mmm_rrr
  public static void irpOrToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //OR.B Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] | XEiJ.busRbs (a);
      XEiJ.busWb (a, z);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea < XEiJ.EA_AR) {  //SBCD.B Dr,Dq
      int qqq = XEiJ.regOC >> 9 & 7;
      XEiJ.mpuCycleCount += 6;
      int x;
      XEiJ.regRn[qqq] = ~0xff & (x = XEiJ.regRn[qqq]) | irpSbcd (x, XEiJ.regRn[ea]);
    } else {  //SBCD.B -(Ar),-(Aq)
      XEiJ.mpuCycleCount += 18;
      int y = XEiJ.busRbz (--XEiJ.regRn[ea]);  //このr[ea]はアドレスレジスタ
      int a = --XEiJ.regRn[(XEiJ.regOC >> 9) - (64 - 8)];
      XEiJ.busWb (a, irpSbcd (XEiJ.busRbz (a), y));
    }
  }  //irpOrToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //PACK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_101_000_rrr-{data}
  //PACK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_101_001_rrr-{data}
  //OR.W Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_101_mmm_rrr
  //
  //PACK Dr,Dq,#<data>
  //PACK -(Ar),-(Aq),#<data>
  //  PACK/UNPKは第1オペランドのソースと第2オペランドのデスティネーションのサイズが違う。パックされていない方がワードでされている方がバイト
  //  10の位を4ビット右または左にシフトする。第3オペランドの補正値はワードでパックされていない方に加算する。CCRは変化しない
  public static void irpOrToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //OR.W Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltWord (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] | XEiJ.busRws (a);
      XEiJ.busWw (a, z);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea < XEiJ.EA_AR) {  //PACK Dr,Dq,#<data>
      XEiJ.mpuCycleCount += 8;
      int qqq = XEiJ.regOC >> 9 & 7;
      int t;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        t = XEiJ.regRn[ea] + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws
      } else {
        t = XEiJ.regPC;
        XEiJ.regPC = t + 2;
        t = XEiJ.regRn[ea] + XEiJ.busRwse (t);  //pcws
      }
      XEiJ.regRn[qqq] = ~0xff & XEiJ.regRn[qqq] | t >> 4 & 0xf0 | t & 15;
    } else {  //PACK -(Ar),-(Aq),#<data>
      XEiJ.mpuCycleCount += 16;
      int t;
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        t = XEiJ.busRws (XEiJ.regRn[ea] -= 2) + XEiJ.busRwse ((XEiJ.regPC += 2) - 2);  //pcws。020以上なのでアドレスエラーは出ない
      } else {
        t = XEiJ.regPC;
        XEiJ.regPC = t + 2;
        t = XEiJ.busRws (XEiJ.regRn[ea] -= 2) + XEiJ.busRwse (t);  //pcws。020以上なのでアドレスエラーは出ない
      }
      XEiJ.busWb (--XEiJ.regRn[(XEiJ.regOC >> 9) - (64 - 8)], t >> 4 & 0xf0 | t & 15);
    }
  }  //irpOrToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //UNPK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_110_000_rrr-{data}
  //UNPK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_110_001_rrr-{data}
  //OR.L Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_110_mmm_rrr
  //
  //UNPK Dr,Dq,#<data>
  //UNPK -(Ar),-(Aq),#<data>
  //  PACK/UNPKは第1オペランドのソースと第2オペランドのデスティネーションのサイズが違う。パックされていない方がワードでされている方がバイト
  //  10の位を4ビット右または左にシフトする。第3オペランドの補正値はワードでパックされていない方に加算する。CCRは変化しない
  public static void irpOrToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //OR.L Dq,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      int z;
      XEiJ.busWl (a, z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] | XEiJ.busRls (a));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else if (ea < XEiJ.EA_AR) {  //UNPK Dr,Dq,#<data>
      int qqq = XEiJ.regOC >> 9 & 7;
      int t = XEiJ.regRn[ea];
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        XEiJ.regRn[qqq] = ~0xffff & XEiJ.regRn[qqq] | (char) ((t << 4 & 0x0f00 | t & 15) + XEiJ.busRwse ((XEiJ.regPC += 2) - 2));  //pcws
      } else {
        int s = XEiJ.regPC;
        XEiJ.regPC = s + 2;
        XEiJ.regRn[qqq] = ~0xffff & XEiJ.regRn[qqq] | (char) ((t << 4 & 0x0f00 | t & 15) + XEiJ.busRwse (s));  //pcws
      }
    } else {  //UNPK -(Ar),-(Aq),#<data>
      int t = XEiJ.busRbs (--XEiJ.regRn[ea]);
      if (XEiJ.MPU_COMPOUND_POSTINCREMENT) {
        XEiJ.busWw (XEiJ.regRn[(XEiJ.regOC >> 9) - (64 - 8)] -= 2, (t << 4 & 0x0f00 | t & 15) + XEiJ.busRwse ((XEiJ.regPC += 2) - 2));  //pcws。020以上なのでアドレスエラーは出ない
      } else {
        int s = XEiJ.regPC;
        XEiJ.regPC = s + 2;
        XEiJ.busWw (XEiJ.regRn[(XEiJ.regOC >> 9) - (64 - 8)] -= 2, (t << 4 & 0x0f00 | t & 15) + XEiJ.busRwse (s));  //pcws。020以上なのでアドレスエラーは出ない
      }
    }
  }  //irpOrToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //DIVS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_111_mmm_rrr
  //
  //DIVS.W <ea>,Dq
  //  DIVSの余りの符号は被除数と一致
  //  M68000PRMでDIVS.Wのアドレッシングモードがデータ可変と書かれているのはデータの間違い
  public static void irpDivsWord () throws M68kException {
    //  X  変化しない
    //  N  ゼロ除算またはオーバーフローのとき不定。商が負のときセット。それ以外はクリア
    //  Z  ゼロ除算またはオーバーフローのとき不定。商が0のときセット。それ以外はクリア
    //  V  ゼロ除算のとき不定。オーバーフローのときセット。それ以外はクリア
    //  C  常にクリア
    //divsの余りの符号は被除数と一致
    //Javaの除算演算子の挙動
    //   10 /  3 ==  3   10 %  3 ==  1   10 =  3 *  3 +  1
    //   10 / -3 == -3   10 % -3 ==  1   10 = -3 * -3 +  1
    //  -10 /  3 == -3  -10 %  3 == -1  -10 =  3 * -3 + -1
    //  -10 / -3 ==  3  -10 % -3 == -1  -10 = -3 *  3 + -1
    XEiJ.mpuCycleCount += 158;  //最大
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //除数
    int x = XEiJ.regRn[qqq];  //被除数
    if (y == 0) {  //ゼロ除算
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     //Nは常にクリア
                     XEiJ.REG_CCR_Z |  //Zは常にセット
                     (M30_DIV_ZERO_V_FLAG && m30DivZeroVFlag ? XEiJ.REG_CCR_V : 0)  //VはDIV/MULの正常終了後ゼロ除算までにDBccが奇数回実行されたときセット
                     );  //Cは常にクリア
      XEiJ.mpuCycleCount += 38 - 158 - 34;
      M68kException.m6eAddress = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
      throw M68kException.m6eSignal;
    }
    int z = x / y;  //商
    if ((short) z != z) {  //オーバーフローあり
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (x == 0x80000000 || (z & 0xffff0080) == 0x00000080 || (z & 0xffff0080) == 0xffff0080 ? XEiJ.REG_CCR_N : 0) |  //Nは被除数が$80000000または商が$0000xxyyまたは$ffffxxyyでyyが負のときセット、さもなくばクリア
                     (z == 0x00008000 || (((z & 0xffff00ff) == 0x00000000 || (z & 0xffff00ff) == 0xffff0000) && (z & 0x0000ff00) != 0) ? XEiJ.REG_CCR_Z : 0) |  //Zは商が$00008000または商が$0000xxyyまたは$ffffxxyyでxxが0でなくてyyが0のときセット、さもなくばクリア
                     XEiJ.REG_CCR_V  //Vは常にセット
                     );  //Cは常にクリア
    } else {  //オーバーフローなし
      XEiJ.regRn[qqq] = x - y * z << 16 | (char) z;  //Dqは余り<<16|商&$ffff
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                     (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                     //Vは常にクリア
                     );  //Cは常にクリア
    }  //if オーバーフローあり/オーバーフローなし
    if (M30_DIV_ZERO_V_FLAG) {
      m30DivZeroVFlag = false;
    }
  }  //irpDivsWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUB.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1001_qqq_000_mmm_rrr
  public static void irpSubToRegByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));
    x = XEiJ.regRn[qqq];
    z = x - y;
    XEiJ.regRn[qqq] = ~255 & x | 255 & z;
    XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
           ((x ^ y) & (x ^ z)) >> 6 & XEiJ.REG_CCR_V |
           (byte) (x & (y ^ z) ^ (y | z)) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_byte
  }  //irpSubToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUB.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_001_mmm_rrr
  public static void irpSubToRegWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    x = XEiJ.regRn[qqq];
    z = x - y;
    XEiJ.regRn[qqq] = ~65535 & x | (char) z;
    XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
           ((x ^ y) & (x ^ z)) >> 14 & XEiJ.REG_CCR_V |
           (short) (x & (y ^ z) ^ (y | z)) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_word
  }  //irpSubToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUB.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_010_mmm_rrr
  public static void irpSubToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    XEiJ.mpuCycleCount += ea < XEiJ.EA_MM || ea == XEiJ.EA_IM ? 8 : 6;  //ソースが#<data>のとき2増やす
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    x = XEiJ.regRn[qqq];
    z = x - y;
    XEiJ.regRn[qqq] = z;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >> 30 & XEiJ.REG_CCR_V |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr
  //SUB.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr [SUBA.W <ea>,Aq]
  //CLR.W Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_011_001_rrr [SUBA.W Ar,Ar]
  //
  //SUBA.W <ea>,Aq
  //  ソースを符号拡張してロングで減算する
  public static void irpSubaWord () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= z;  //r[op >> 9 & 15] -= ea < XEiJ.EA_MM ? (short) r[ea] : rws (efaAnyWord (ea));は不可
    //ccrは変化しない
  }  //irpSubaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_100_000_rrr
  //SUBX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_100_001_rrr
  //SUB.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_100_mmm_rrr
  public static void irpSubToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //SUBX.B Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount += 4;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~255 & x | 255 & z;
      } else {  //SUBX.B -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 18;
        y = XEiJ.busRbs (--XEiJ.regRn[ea]);  //このr[ea]はアドレスレジスタ
        a = --XEiJ.regRn[XEiJ.regOC >> 9 & 15];  //1qqq=aqq
        x = XEiJ.busRbs (a);
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.busWb (a, z);
      }
      XEiJ.regCCR = (z >> 4 & XEiJ.REG_CCR_N | (255 & z) - 1 >> 6 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //SUBXはZをクリアすることはあるがセットすることはない
             ((x ^ y) & (x ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) (x & (y ^ z) ^ (y | z)) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subx_byte
    } else {  //SUB.B Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      a = efaMltByte (ea);
      x = XEiJ.busRbs (a);
      z = x - y;
      XEiJ.busWb (a, z);
      XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
             ((x ^ y) & (x ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) (x & (y ^ z) ^ (y | z)) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_byte
    }
  }  //irpSubToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_101_000_rrr
  //SUBX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_101_001_rrr
  //SUB.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_101_mmm_rrr
  public static void irpSubToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //SUBX.W Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount += 4;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~65535 & x | (char) z;
      } else {  //SUBX.W -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 18;
        y = XEiJ.busRws (XEiJ.regRn[ea] -= 2);  //このr[ea]はアドレスレジスタ
        a = XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= 2;
        x = XEiJ.busRws (a);
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.busWw (a, z);
      }
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //ADDXはZをクリアすることはあるがセットすることはない
             ((x ^ y) & (x ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) (x & (y ^ z) ^ (y | z)) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subx_word
    } else {  //SUB.W Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      y = (short) XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      a = efaMltWord (ea);
      x = XEiJ.busRws (a);
      z = x - y;
      XEiJ.busWw (a, z);
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
             ((x ^ y) & (x ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) (x & (y ^ z) ^ (y | z)) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_word
    }
  }  //irpSubToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_110_000_rrr
  //SUBX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_110_001_rrr
  //SUB.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_110_mmm_rrr
  public static void irpSubToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_MM) {
      int x;
      int y;
      int z;
      if (ea < XEiJ.EA_AR) {  //SUBX.L Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount += 8;
        XEiJ.regRn[qqq] = z = (x = XEiJ.regRn[qqq]) - (y = XEiJ.regRn[ea]) - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
      } else {  //SUBX.L -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 30;
        y = XEiJ.busRls (XEiJ.regRn[ea] -= 4);  //このr[ea]はアドレスレジスタ
        int a = XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= 4;
        XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) - y - (XEiJ.regCCR >> 4));  //Xの左側はすべて0なのでCCR_X&を省略
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subx
    } else {  //SUB.L Dq,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      int x;
      int y;
      int z;
      XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) - (y = XEiJ.regRn[XEiJ.regOC >> 9 & 7]));
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
    }
  }  //irpSubToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr
  //SUB.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr [SUBA.L <ea>,Aq]
  //CLR.L Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_111_001_rrr [SUBA.L Ar,Ar]
  public static void irpSubaLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    XEiJ.mpuCycleCount += ea < XEiJ.EA_MM || ea == XEiJ.EA_IM ? 8 : 6;  //Dr/Ar/#<data>のとき8+、それ以外は6+
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= z;  //r[op >> 9 & 15] -= ea < XEiJ.EA_MM ? r[ea] : rls (efaAnyLong (ea));は不可
    //ccrは変化しない
  }  //irpSubaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SXCALL <name>                                   |A|012346|-|UUUUU|*****|          |1010_0dd_ddd_ddd_ddd [ALINE #<data>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ALINE #<data>                                   |-|012346|-|UUUUU|*****|          |1010_ddd_ddd_ddd_ddd (line 1010 emulator)
  public static void irpAline () throws M68kException {
    XEiJ.mpuCycleCount += 34;
    if (XEiJ.MPU_INLINE_EXCEPTION) {
      int save_sr = XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
      int sp = XEiJ.regRn[15];
      XEiJ.regSRT1 = XEiJ.regSRT0 = XEiJ.mpuTraceFlag = 0;  //srのTビットを消す
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        XEiJ.regSRS = XEiJ.REG_SR_S;  //スーパーバイザモードへ移行する
        XEiJ.mpuUSP = sp;  //USPを保存
        sp = XEiJ.regSRM != 0 ? XEiJ.mpuMSP : XEiJ.mpuISP;  //SSPを復元
        if (DataBreakPoint.DBP_ON) {
          DataBreakPoint.dbpMemoryMap = DataBreakPoint.dbpSuperMap;  //スーパーバイザメモリマップに切り替える
        } else {
          XEiJ.busMemoryMap = XEiJ.busSuperMap;  //スーパーバイザメモリマップに切り替える
        }
        if (InstructionBreakPoint.IBP_ON) {
          InstructionBreakPoint.ibpOp1MemoryMap = InstructionBreakPoint.ibpOp1SuperMap;
        }
      }
      XEiJ.regRn[15] = sp -= 8;
      XEiJ.busWw (sp + 6, 0x0000 | M68kException.M6E_LINE_1010_EMULATOR << 2);  //pushw。フォーマットとベクタオフセットをプッシュする
      XEiJ.busWl (sp + 2, XEiJ.regPC0);  //pushl。pcをプッシュする
      XEiJ.busWw (sp, save_sr);  //pushw。srをプッシュする
      irpSetPC (XEiJ.busRlsf (XEiJ.mpuVBR + (M68kException.M6E_LINE_1010_EMULATOR << 2)));  //例外ベクタを取り出してジャンプする
    } else {
      irpException (M68kException.M6E_LINE_1010_EMULATOR, XEiJ.regPC0, XEiJ.regSRT1 | XEiJ.regSRT0 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR, 0x0000, 0);  //pcは命令の先頭
    }
  }  //irpAline

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMP.B <ea>,Dq                                   |-|012346|-|-UUUU|-****|D M+-WXZPI|1011_qqq_000_mmm_rrr
  public static void irpCmpByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    int z = (byte) ((x = (byte) XEiJ.regRn[XEiJ.regOC >> 9 & 7]) - (y = ea < XEiJ.EA_AR ? (byte) XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea))));
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMP.W <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_001_mmm_rrr
  public static void irpCmpWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    int z = (short) ((x = (short) XEiJ.regRn[XEiJ.regOC >> 9 & 7]) - (y = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea))));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMP.L <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_010_mmm_rrr
  public static void irpCmpLong () throws M68kException {
    XEiJ.mpuCycleCount += 6;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    int z = (x = XEiJ.regRn[XEiJ.regOC >> 9 & 7]) - (y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea)));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPA.W <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr
  //CMP.W <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr [CMPA.W <ea>,Aq]
  //
  //CMPA.W <ea>,Aq
  //  ソースを符号拡張してロングで比較する
  public static void irpCmpaWord () throws M68kException {
    XEiJ.mpuCycleCount += 6;
    int ea = XEiJ.regOC & 63;
    //ソースを符号拡張してからロングで比較する
    int y = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int x;
    int z = (x = XEiJ.regRn[XEiJ.regOC >> 9 & 15]) - y;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EOR.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_100_mmm_rrr
  //CMPM.B (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_100_001_rrr
  public static void irpEorByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //CMPM.B (Ar)+,(Aq)+
      XEiJ.mpuCycleCount += 12;
      int y = XEiJ.busRbs (XEiJ.regRn[ea]++);  //このr[ea]はアドレスレジスタ
      int x;
      int z = (byte) ((x = XEiJ.busRbs (XEiJ.regRn[XEiJ.regOC >> 9 & 15]++)) - y);
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    } else {
      int qqq = XEiJ.regOC >> 9 & 7;
      int z;
      if (ea < XEiJ.EA_AR) {  //EOR.B Dq,Dr
        XEiJ.mpuCycleCount += 4;
        z = XEiJ.regRn[ea] ^= 255 & XEiJ.regRn[qqq];  //0拡張してからEOR
      } else {  //EOR.B Dq,<mem>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltByte (ea);
        XEiJ.busWb (a, z = XEiJ.regRn[qqq] ^ XEiJ.busRbs (a));
      }
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpEorByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EOR.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_101_mmm_rrr
  //CMPM.W (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_101_001_rrr
  public static void irpEorWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int rrr = XEiJ.regOC & 7;
    int mmm = ea >> 3;
    if (mmm == XEiJ.MMM_AR) {  //CMPM.W (Ar)+,(Aq)+
      XEiJ.mpuCycleCount += 12;
      int y = XEiJ.busRws ((XEiJ.regRn[ea] += 2) - 2);  //このr[ea]はアドレスレジスタ
      int x;
      int z = (short) ((x = XEiJ.busRws ((XEiJ.regRn[XEiJ.regOC >> 9 & 15] += 2) - 2)) - y);
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    } else {
      int qqq = XEiJ.regOC >> 9 & 7;
      int z;
      if (ea < XEiJ.EA_AR) {  //EOR.W Dq,Dr
        XEiJ.mpuCycleCount += 4;
        z = XEiJ.regRn[rrr] ^= (char) XEiJ.regRn[qqq];  //0拡張してからEOR
      } else {  //EOR.W Dq,<mem>
        XEiJ.mpuCycleCount += 8;
        int a = efaMltWord (ea);
        XEiJ.busWw (a, z = XEiJ.regRn[qqq] ^ XEiJ.busRws (a));
      }
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpEorWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EOR.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_110_mmm_rrr
  //CMPM.L (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_110_001_rrr
  public static void irpEorLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //CMPM.L (Ar)+,(Aq)+
      XEiJ.mpuCycleCount += 20;
      int y = XEiJ.busRls ((XEiJ.regRn[ea] += 4) - 4);  //このr[ea]はアドレスレジスタ
      int x;
      int z = (x = XEiJ.busRls ((XEiJ.regRn[XEiJ.regOC >> 9 & 15] += 4) - 4)) - y;
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    } else {
      int qqq = XEiJ.regOC >> 9 & 7;
      int z;
      if (ea < XEiJ.EA_AR) {  //EOR.L Dq,Dr
        XEiJ.mpuCycleCount += 8;
        XEiJ.regRn[ea] = z = XEiJ.regRn[ea] ^ XEiJ.regRn[qqq];
      } else {  //EOR.L Dq,<mem>
        XEiJ.mpuCycleCount += 12;
        int a = efaMltLong (ea);
        XEiJ.busWl (a, z = XEiJ.busRls (a) ^ XEiJ.regRn[qqq]);
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    }
  }  //irpEorLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPA.L <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr
  //CMP.L <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr [CMPA.L <ea>,Aq]
  public static void irpCmpaLong () throws M68kException {
    XEiJ.mpuCycleCount += 6;
    int ea = XEiJ.regOC & 63;
    int y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int x;
    int z = (x = XEiJ.regRn[XEiJ.regOC >> 9 & 15]) - y;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //AND.B <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_000_mmm_rrr
  public static void irpAndToRegByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & (XEiJ.regRn[XEiJ.regOC >> 9 & 7] &= ~255 | (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea))))];  //ccr_tst_byte。1拡張してからAND
  }  //irpAndToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //AND.W <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_001_mmm_rrr
  public static void irpAndToRegWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] &= ~65535 | (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea)));  //1拡張してからAND
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpAndToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //AND.L <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_010_mmm_rrr
  public static void irpAndToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int z;
    if (ea < XEiJ.EA_AR) {  //AND.L Dr,Dq
      XEiJ.mpuCycleCount += 8;
      z = XEiJ.regRn[qqq] &= XEiJ.regRn[ea];
    } else {  //AND.L <mem>,Dq
      XEiJ.mpuCycleCount += ea == XEiJ.EA_IM ? 8 : 6;  //ソースが#<data>のとき2増やす
      z = XEiJ.regRn[qqq] &= XEiJ.busRls (efaAnyLong (ea));
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpAndToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MULU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_011_mmm_rrr
  public static void irpMuluWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : XEiJ.busRwz (efaAnyWord (ea));
    //muluの所要サイクル数は38+2n
    //nはソースに含まれる1の数
    int s = y & 0x5555;
    s += y - s >> 1;
    int t = s & 0x3333;
    t += s - t >> 2;
    t += t >> 4;
    XEiJ.mpuCycleCount += 38 + ((t & 15) + (t >> 8 & 15)) << 1;  //38+2n
    //XEiJ.mpuCycleCount += 38 + (Integer.bitCount (y) << 1);  //少し遅くなる
    int z;
    XEiJ.regRn[qqq] = z = (char) XEiJ.regRn[qqq] * y;  //積の下位32ビット。オーバーフローは無視
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    if (M30_DIV_ZERO_V_FLAG) {
      m30DivZeroVFlag = false;
    }
  }  //irpMuluWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ABCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_000_rrr
  //ABCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_001_rrr
  //AND.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_100_mmm_rrr
  public static void irpAndToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //AND.B Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltByte (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & XEiJ.busRbs (a);
      XEiJ.busWb (a, z);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea < XEiJ.EA_AR) {  //ABCD.B Dr,Dq
      int qqq = XEiJ.regOC >> 9 & 7;
      XEiJ.mpuCycleCount += 6;
      XEiJ.regRn[qqq] = ~0xff & XEiJ.regRn[qqq] | irpAbcd (XEiJ.regRn[qqq], XEiJ.regRn[ea]);
    } else {  //ABCD.B -(Ar),-(Aq)
      XEiJ.mpuCycleCount += 18;
      int y = XEiJ.busRbz (--XEiJ.regRn[ea]);  //このr[ea]はアドレスレジスタ
      int a = --XEiJ.regRn[(XEiJ.regOC >> 9) - (96 - 8)];
      XEiJ.busWb (a, irpAbcd (XEiJ.busRbz (a), y));
    }
  }  //irpAndToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXG.L Dq,Dr                                     |-|012346|-|-----|-----|          |1100_qqq_101_000_rrr
  //EXG.L Aq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_101_001_rrr
  //AND.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_101_mmm_rrr
  public static void irpAndToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_MM) {  //EXG
      XEiJ.mpuCycleCount += 6;
      if (ea < XEiJ.EA_AR) {  //EXG.L Dq,Dr
        int qqq = XEiJ.regOC >> 9 & 7;
        int t = XEiJ.regRn[qqq];
        XEiJ.regRn[qqq] = XEiJ.regRn[ea];
        XEiJ.regRn[ea] = t;
      } else {  //EXG.L Aq,Ar
        int aqq = (XEiJ.regOC >> 9) - (96 - 8);
        int t = XEiJ.regRn[aqq];
        XEiJ.regRn[aqq] = XEiJ.regRn[ea];  //このr[ea]アドレスレジスタ
        XEiJ.regRn[ea] = t;  //このr[ea]はアドレスレジスタ
      }
    } else {  //AND.W Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      int a = efaMltWord (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & XEiJ.busRws (a);
      XEiJ.busWw (a, z);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpAndToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXG.L Dq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_110_001_rrr
  //AND.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_110_mmm_rrr
  public static void irpAndToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    if (ea >> 3 == XEiJ.MMM_AR) {  //EXG.L Dq,Ar
      XEiJ.mpuCycleCount += 6;
      int t = XEiJ.regRn[qqq];
      XEiJ.regRn[qqq] = XEiJ.regRn[ea];  //このr[ea]はアドレスレジスタ
      XEiJ.regRn[ea] = t;  //このr[ea]はアドレスレジスタ
    } else {  //AND.L Dq,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      int z;
      XEiJ.busWl (a, z = XEiJ.busRls (a) & XEiJ.regRn[qqq]);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    }
  }  //irpAndToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MULS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_111_mmm_rrr
  public static void irpMulsWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));
    int t = y << 1 ^ y;  //右側が1である0と右側が0または末尾である1は1、それ以外は0。ソースは符号拡張されているので上位16ビットはすべて0
    //mulsの所要サイクル数は38+2n
    //nはソースの末尾に0を付け加えた17ビットに含まれる10または01の数
    int s = t & 0x5555;
    s += t - s >> 1;
    t = s & 0x3333;
    t += s - t >> 2;
    t += t >> 4;
    XEiJ.mpuCycleCount += 38 + ((t & 15) + (t >> 8 & 15)) << 1;  //38+2n
    int z;
    XEiJ.regRn[qqq] = z = (short) XEiJ.regRn[qqq] * y;  //積の下位32ビット。オーバーフローは無視
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    if (M30_DIV_ZERO_V_FLAG) {
      m30DivZeroVFlag = false;
    }
  }  //irpMulsWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADD.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1101_qqq_000_mmm_rrr
  public static void irpAddToRegByte () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : XEiJ.busRbs (efaAnyByte (ea));
    x = XEiJ.regRn[qqq];
    z = x + y;
    XEiJ.regRn[qqq] = ~255 & x | 255 & z;
    XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
           ((x ^ z) & (y ^ z)) >> 6 & XEiJ.REG_CCR_V |
           (byte) ((x | y) ^ (x ^ y) & z) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_byte
  }  //irpAddToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADD.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_001_mmm_rrr
  public static void irpAddToRegWord () throws M68kException {
    XEiJ.mpuCycleCount += 4;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    x = XEiJ.regRn[qqq];
    z = x + y;
    XEiJ.regRn[qqq] = ~65535 & x | (char) z;
    XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
           ((x ^ z) & (y ^ z)) >> 14 & XEiJ.REG_CCR_V |
           (short) ((x | y) ^ (x ^ y) & z) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_word
  }  //irpAddToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADD.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_010_mmm_rrr
  public static void irpAddToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    XEiJ.mpuCycleCount += ea < XEiJ.EA_MM || ea == XEiJ.EA_IM ? 8 : 6;  //ソースが#<data>のとき2増やす
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ
    x = XEiJ.regRn[qqq];
    z = x + y;
    XEiJ.regRn[qqq] = z;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >> 30 & XEiJ.REG_CCR_V |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr
  //ADD.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr [ADDA.W <ea>,Aq]
  //
  //ADDA.W <ea>,Aq
  //  ソースを符号拡張してロングで加算する
  public static void irpAddaWord () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : XEiJ.busRws (efaAnyWord (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] += z;  //r[op >> 9 & 15] += ea < XEiJ.EA_MM ? (short) r[ea] : rws (efaAnyWord (ea));は不可
    //ccrは変化しない
  }  //irpAddaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_100_000_rrr
  //ADDX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_100_001_rrr
  //ADD.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_100_mmm_rrr
  public static void irpAddToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //ADDX.B Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount += 4;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~255 & x | 255 & z;
      } else {  //ADDX.B -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 18;
        y = XEiJ.busRbs (--XEiJ.regRn[ea]);  //このr[ea]はアドレスレジスタ
        a = --XEiJ.regRn[XEiJ.regOC >> 9 & 15];  //1qqq=aqq
        x = XEiJ.busRbs (a);
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.busWb (a, z);
      }
      XEiJ.regCCR = (z >> 4 & XEiJ.REG_CCR_N | (255 & z) - 1 >> 6 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //ADDXはZをクリアすることはあるがセットすることはない
             ((x ^ z) & (y ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) ((x | y) ^ (x ^ y) & z) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addx_byte
    } else {  //ADD.B Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      a = efaMltByte (ea);
      x = XEiJ.busRbs (a);
      z = x + y;
      XEiJ.busWb (a, z);
      XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
             ((x ^ z) & (y ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) ((x | y) ^ (x ^ y) & z) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_byte
    }
  }  //irpAddToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_101_000_rrr
  //ADDX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_101_001_rrr
  //ADD.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_101_mmm_rrr
  public static void irpAddToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //ADDX.W Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount += 4;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~65535 & x | (char) z;
      } else {  //ADDX.W -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 18;
        y = XEiJ.busRws (XEiJ.regRn[ea] -= 2);  //このr[ea]はアドレスレジスタ
        a = XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= 2;
        x = XEiJ.busRws (a);
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.busWw (a, z);
      }
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //ADDXはZをクリアすることはあるがセットすることはない
             ((x ^ z) & (y ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) ((x | y) ^ (x ^ y) & z) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addx_word
    } else {  //ADD.W Dq,<ea>
      XEiJ.mpuCycleCount += 8;
      a = efaMltWord (ea);
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      x = XEiJ.busRws (a);
      z = x + y;
      XEiJ.busWw (a, z);
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
             ((x ^ z) & (y ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) ((x | y) ^ (x ^ y) & z) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_word
    }
  }  //irpAddToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_110_000_rrr
  //ADDX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_110_001_rrr
  //ADD.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_110_mmm_rrr
  public static void irpAddToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_MM) {
      int x;
      int y;
      int z;
      if (ea < XEiJ.EA_AR) {  //ADDX.L Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount += 8;
        XEiJ.regRn[qqq] = z = (x = XEiJ.regRn[qqq]) + (y = XEiJ.regRn[ea]) + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
      } else {  //ADDX.L -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 30;
        y = XEiJ.busRls (XEiJ.regRn[ea] -= 4);  //このr[ea]はアドレスレジスタ
        int a = XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= 4;
        XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) + y + (XEiJ.regCCR >> 4));  //Xの左側はすべて0なのでCCR_X&を省略
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
             ((x ^ z) & (y ^ z)) >>> 31 << 1 |
             ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addx
    } else {  //ADD.L Dq,<ea>
      XEiJ.mpuCycleCount += 12;
      int a = efaMltLong (ea);
      int x;
      int y;
      int z;
      XEiJ.busWl (a, z = (x = XEiJ.busRls (a)) + (y = XEiJ.regRn[XEiJ.regOC >> 9 & 7]));
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             ((x ^ z) & (y ^ z)) >>> 31 << 1 |
             ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
    }
  }  //irpAddToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr
  //ADD.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr [ADDA.L <ea>,Aq]
  public static void irpAddaLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    XEiJ.mpuCycleCount += ea < XEiJ.EA_MM || ea == XEiJ.EA_IM ? 8 : 6;  //Dr/Ar/#<data>のとき8+、それ以外は6+
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : XEiJ.busRls (efaAnyLong (ea));  //このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] += z;  //r[op >> 9 & 15] += ea < XEiJ.EA_MM ? r[ea] : rls (efaAnyLong (ea));は不可
    //ccrは変化しない
  }  //irpAddaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_000_rrr
  //LSR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_001_rrr
  //ROXR.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_000_010_rrr
  //ROR.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_011_rrr
  //ASR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_100_rrr
  //LSR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_101_rrr
  //ROXR.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_000_110_rrr
  //ROR.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_111_rrr
  //ASR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_000_rrr [ASR.B #1,Dr]
  //LSR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_001_rrr [LSR.B #1,Dr]
  //ROXR.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_000_010_rrr [ROXR.B #1,Dr]
  //ROR.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_000_011_rrr [ROR.B #1,Dr]
  //
  //ASR.B #<data>,Dr
  //ASR.B Dq,Dr
  //  算術右シフトバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｱｱｲｳｴｵｶｷ ｸｱ*0ｸ Z=ｱｲｳｴｵｶｷ==0
  //     2 ........................ｱｱｱｲｳｴｵｶ ｷｱ*0ｷ Z=ｱｲｳｴｵｶ==0
  //     3 ........................ｱｱｱｱｲｳｴｵ ｶｱ*0ｶ Z=ｱｲｳｴｵ==0
  //     4 ........................ｱｱｱｱｱｲｳｴ ｵｱ*0ｵ Z=ｱｲｳｴ==0
  //     5 ........................ｱｱｱｱｱｱｲｳ ｴｱ*0ｴ Z=ｱｲｳ==0
  //     6 ........................ｱｱｱｱｱｱｱｲ ｳｱ*0ｳ Z=ｱｲ==0
  //     7 ........................ｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //     8 ........................ｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSR.B #<data>,Dr
  //LSR.B Dq,Dr
  //  論理右シフトバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................0ｱｲｳｴｵｶｷ ｸ0*0ｸ Z=ｱｲｳｴｵｶｷ==0
  //     2 ........................00ｱｲｳｴｵｶ ｷ0*0ｷ Z=ｱｲｳｴｵｶ==0
  //     3 ........................000ｱｲｳｴｵ ｶ0*0ｶ Z=ｱｲｳｴｵ==0
  //     4 ........................0000ｱｲｳｴ ｵ0*0ｵ Z=ｱｲｳｴ==0
  //     5 ........................00000ｱｲｳ ｴ0*0ｴ Z=ｱｲｳ==0
  //     6 ........................000000ｱｲ ｳ0*0ｳ Z=ｱｲ==0
  //     7 ........................0000000ｱ ｲ0*0ｲ Z=ｱ==0
  //     8 ........................00000000 ｱ010ｱ
  //     9 ........................00000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROR.B #<data>,Dr
  //ROR.B Dq,Dr
  //  右ローテートバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｸｱｲｳｴｵｶｷ Xｸ*0ｸ Z=ｱｲｳｴｵｶｷｸ==0
  //     :
  //     7 ........................ｲｳｴｵｶｷｸｱ Xｲ*0ｲ Z=ｱｲｳｴｵｶｷｸ==0
  //     8 ........................ｱｲｳｴｵｶｷｸ Xｱ*0ｱ Z=ｱｲｳｴｵｶｷｸ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最上位ビット
  //
  //ROXR.B #<data>,Dr
  //ROXR.B Dq,Dr
  //  拡張右ローテートバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*0X Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................Xｱｲｳｴｵｶｷ ｸX*0ｸ Z=ｱｲｳｴｵｶｷX==0
  //     2 ........................ｸXｱｲｳｴｵｶ ｷｸ*0ｷ Z=ｱｲｳｴｵｶｸX==0
  //     3 ........................ｷｸXｱｲｳｴｵ ｶｷ*0ｶ Z=ｱｲｳｴｵｷｸX==0
  //     4 ........................ｶｷｸXｱｲｳｴ ｵｶ*0ｵ Z=ｱｲｳｴｶｷｸX==0
  //     5 ........................ｵｶｷｸXｱｲｳ ｴｵ*0ｴ Z=ｱｲｳｵｶｷｸX==0
  //     6 ........................ｴｵｶｷｸXｱｲ ｳｴ*0ｳ Z=ｱｲｴｵｶｷｸX==0
  //     7 ........................ｳｴｵｶｷｸXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸX==0
  //     8 ........................ｲｳｴｵｶｷｸX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸX==0
  //     9 ........................ｱｲｳｴｵｶｷｸ Xｱ*0X Z=ｱｲｳｴｵｶｷｸ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxrToRegByte () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASR.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (t = (byte) x >> y) >> 1);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSR.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xff & x | (z = (t = (0xff & x) >>> y) >>> 1);
      XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXR.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 7 - 4 | (0xff & x) >>> 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = x << 9 - 1 - y | (t = z >>> y - (2 - 1)) >>> 1;
      }
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROR.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + ((y = XEiJ.regOC >> 9 & 7) << 1);  //y=data&7
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) (x << 8 - y | (0xff & x) >>> y));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z >>> 7 & 1;  //Xは変化しない。Cは結果の最上位ビット
      break;
    case 0b100_000 >> 3:  //ASR.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = (byte) x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (t = (byte) x >> (y <= 8 ? y - 1 : 7)) >> 1);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b101_000 >> 3:  //LSR.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = (byte) x;
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (z < 0 ? XEiJ.REG_CCR_N : z == 0 ? XEiJ.REG_CCR_Z : 0);  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xff & x | (z = (t = y <= 8 ? (0xff & x) >>> y - 1 : 0) >>> 1);
        XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      break;
    case 0b110_000 >> 3:  //ROXR.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      //y %= 9;
      y = (y & 7) - (y >> 3);  //y=data=-7～7
      y += y >> 3 & 9;  //y=data=0～8
      if (y == 0) {  //y=data=0
        z = (byte) x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～8
        z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 7 - 4 | (0xff & x) >>> 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～8
          z = x << 9 - y | (t = z >>> y - 2) >>> 1;
        }
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) z);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROR.B Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {
        z = (byte) x;
        t = 0;  //Cはクリア
      } else {
        y &= 7;  //y=data=0～7
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) (x << 8 - y | (0xff & x) >>> y));
        t = z >>> 7 & 1;  //Cは結果の最上位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxrToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_000_rrr
  //LSR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_001_rrr
  //ROXR.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_001_010_rrr
  //ROR.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_011_rrr
  //ASR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_100_rrr
  //LSR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_101_rrr
  //ROXR.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_001_110_rrr
  //ROR.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_111_rrr
  //ASR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_000_rrr [ASR.W #1,Dr]
  //LSR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_001_rrr [LSR.W #1,Dr]
  //ROXR.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_001_010_rrr [ROXR.W #1,Dr]
  //ROR.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_001_011_rrr [ROR.W #1,Dr]
  //
  //ASR.W #<data>,Dr
  //ASR.W Dq,Dr
  //ASR.W <ea>
  //  算術右シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｱｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀｱ*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ==0
  //     :
  //    15 ................ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //    16 ................ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSR.W #<data>,Dr
  //LSR.W Dq,Dr
  //LSR.W <ea>
  //  論理右シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................0ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀ0*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ==0
  //     :
  //    15 ................000000000000000ｱ ｲ0*0ｲ Z=ｱ==0
  //    16 ................0000000000000000 ｱ010ｱ
  //    17 ................0000000000000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROR.W #<data>,Dr
  //ROR.W Dq,Dr
  //ROR.W <ea>
  //  右ローテートワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ﾀｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ Xﾀ*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     :
  //    15 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀｱ Xｲ*0ｲ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //    16 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0ｱ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最上位ビット
  //
  //ROXR.W #<data>,Dr
  //ROXR.W Dq,Dr
  //ROXR.W <ea>
  //  拡張右ローテートワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................Xｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀX*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿX==0
  //     2 ................ﾀXｱｲｳｴｵｶｷｸｹｺｻｼｽｾ ｿﾀ*0ｿ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾﾀX==0
  //     :
  //    15 ................ｳｴｵｶｷｸｹｺｻｼｽｾｿﾀXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX==0
  //    16 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX==0
  //    17 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxrToRegWord () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASR.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (t = (short) x >> y) >> 1);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSR.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (z = (t = (char) x >>> y) >>> 1);
      XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXR.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 15 - 4 | (char) x >>> 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = x << 17 - 1 - y | (t = z >>> y - (2 - 1)) >>> 1;
      }
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROR.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) (x << 16 - 1 - y | (char) x >>> y + 1));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z >>> 15 & 1;  //Xは変化しない。Cは結果の最上位ビット
      break;
    case 0b100_000 >> 3:  //ASR.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = (short) x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (t = (short) x >> (y <= 16 ? y - 1 : 15)) >> 1);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b101_000 >> 3:  //LSR.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = (short) x;
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (z < 0 ? XEiJ.REG_CCR_N : z == 0 ? XEiJ.REG_CCR_Z : 0);  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xffff & x | (z = (t = y <= 16 ? (char) x >>> y - 1 : 0) >>> 1);
        XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      break;
    case 0b110_000 >> 3:  //ROXR.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      //y %= 17;
      y = (y & 15) - (y >> 4);  //y=data=-3～15
      y += y >> 4 & 17;  //y=data=0～16
      if (y == 0) {  //y=data=0
        z = (short) x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～16
        z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 15 - 4 | (char) x >>> 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～16
          z = x << 17 - y | (t = z >>> y - 2) >>> 1;
        }
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) z);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROR.W Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {
        z = (short) x;
        t = 0;  //Cはクリア
      } else {
        y &= 15;  //y=data=0～15
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) (x << 16 - y | (char) x >>> y));
        t = z >>> 15 & 1;  //Cは結果の最上位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxrToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_000_rrr
  //LSR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_001_rrr
  //ROXR.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_010_010_rrr
  //ROR.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_011_rrr
  //ASR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_100_rrr
  //LSR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_101_rrr
  //ROXR.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_010_110_rrr
  //ROR.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_111_rrr
  //ASR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_000_rrr [ASR.L #1,Dr]
  //LSR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_001_rrr [LSR.L #1,Dr]
  //ROXR.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_010_010_rrr [ROXR.L #1,Dr]
  //ROR.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_010_011_rrr [ROR.L #1,Dr]
  //
  //ASR.L #<data>,Dr
  //ASR.L Dq,Dr
  //  算術右シフトロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ｱｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ ﾐｱ*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ==0
  //     :
  //    31 ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //    32 ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSR.L #<data>,Dr
  //LSR.L Dq,Dr
  //  論理右シフトロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 0ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ ﾐ0*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ==0
  //     :
  //    31 0000000000000000000000000000000ｱ ｲ0*0ｲ Z=ｱ==0
  //    32 00000000000000000000000000000000 ｱ010ｱ
  //    33 00000000000000000000000000000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROR.L #<data>,Dr
  //ROR.L Dq,Dr
  //  右ローテートロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ﾐｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ Xﾐ*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     :
  //    31 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐｱ Xｲ*0ｲ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //    32 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*0ｱ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最上位ビット
  //
  //ROXR.L #<data>,Dr
  //ROXR.L Dq,Dr
  //  拡張右ローテートロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 Xｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ ﾐX*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏX==0
  //     2 ﾐXｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎ ﾏﾐ*0ﾏ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾐX==0
  //     :
  //    31 ｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐX==0
  //    32 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐX==0
  //    33 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxrToRegLong () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASR.L #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = z = (t = x >> y) >> 1;
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSR.L #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = z = (t = x >>> y) >>> 1;
      XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXR.L #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 31 - 4 | x >>> 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = x << -y | (t = z >>> y - (2 - 1)) >>> 1;  //Javaのシフト演算子は5ビットでマスクされるので33-1-yを-yに省略
      }
      XEiJ.regRn[rrr] = z;
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROR.L #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = z = x << ~y | x >>> y + 1;  //Javaのシフト演算子は5ビットでマスクされるので32-1-yを~yに省略
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z >>> 31;  //Xは変化しない。Cは結果の最上位ビット
      break;
    case 0b100_000 >> 3:  //ASR.L Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = z = (t = x >> (y <= 32 ? y - 1 : 31)) >> 1;
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b101_000 >> 3:  //LSR.L Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = x;
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (z < 0 ? XEiJ.REG_CCR_N : z == 0 ? XEiJ.REG_CCR_Z : 0);  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = z = (t = y <= 32 ? x >>> y - 1 : 0) >>> 1;
        XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      break;
    case 0b110_000 >> 3:  //ROXR.L Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      //y %= 33;
      y -= 32 - y >> 6 & 33;  //y=data=0～32
      if (y == 0) {  //y=data=0
        z = x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～32
        z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 31 - 4 | x >>> 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～32
          z = x << 33 - y | (t = z >>> y - 2) >>> 1;
        }
        XEiJ.regRn[rrr] = z;
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROR.L Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {
        z = x;
        t = 0;  //Cはクリア
      } else {
        y &= 31;  //y=data=0～31
        XEiJ.regRn[rrr] = z = x << -y | x >>> y;  //Javaのシフト演算子は5ビットでマスクされるので32-yを-yに省略。y=32のときx|xになるが問題ない
        t = z >>> 31;  //Cは結果の最上位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxrToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.W <ea>                                      |-|012346|-|UUUUU|***0*|  M+-WXZ  |1110_000_011_mmm_rrr
  //
  //ASR.W #<data>,Dr
  //ASR.W Dq,Dr
  //ASR.W <ea>
  //  算術右シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｱｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀｱ*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ==0
  //     :
  //    15 ................ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //    16 ................ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  public static void irpAsrToMem () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int ea = XEiJ.regOC & 63;
    int a = efaMltWord (ea);
    int x = XEiJ.busRws (a);
    int z = x >> 1;
    XEiJ.busWw (a, z);
    XEiJ.regCCR = ((z < 0 ? XEiJ.REG_CCR_N : 0) |
                   (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                   -(x & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //XとCは最後に押し出されたビット
  }  //irpAsrToMem

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASL.B #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_100_000_rrr
  //LSL.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_100_001_rrr
  //ROXL.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_100_010_rrr
  //ROL.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_100_011_rrr
  //ASL.B Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_100_100_rrr
  //LSL.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_100_101_rrr
  //ROXL.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_100_110_rrr
  //ROL.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_100_111_rrr
  //ASL.B Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_100_000_rrr [ASL.B #1,Dr]
  //LSL.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_100_001_rrr [LSL.B #1,Dr]
  //ROXL.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_100_010_rrr [ROXL.B #1,Dr]
  //ROL.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_100_011_rrr [ROL.B #1,Dr]
  //
  //ASL.B #<data>,Dr
  //ASL.B Dq,Dr
  //  算術左シフトバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｲｳｴｵｶｷｸ0 ｱｲ**ｱ Z=ｲｳｴｵｶｷｸ==0,V=ｱｲ!=0/-1
  //     :
  //     7 ........................ｸ0000000 ｷｸ**ｷ Z=ｸ==0,V=ｱｲｳｴｵｶｷｸ!=0/-1
  //     8 ........................00000000 ｸ01*ｸ V=ｱｲｳｴｵｶｷｸ!=0
  //     9 ........................00000000 001*0 V=ｱｲｳｴｵｶｷｸ!=0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  ASRで元に戻せないときセット。他はクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSL.B #<data>,Dr
  //LSL.B Dq,Dr
  //  論理左シフトバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｲｳｴｵｶｷｸ0 ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸ==0
  //     :
  //     7 ........................ｸ0000000 ｷｸ*0ｷ Z=ｸ==0
  //     8 ........................00000000 ｸ010ｸ
  //     9 ........................00000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROL.B #<data>,Dr
  //ROL.B Dq,Dr
  //  左ローテートバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｲｳｴｵｶｷｸｱ Xｲ*0ｱ Z=ｱｲｳｴｵｶｷｸ==0
  //     :
  //     7 ........................ｸｱｲｳｴｵｶｷ Xｸ*0ｷ Z=ｱｲｳｴｵｶｷｸ==0
  //     8 ........................ｱｲｳｴｵｶｷｸ Xｱ*0ｸ Z=ｱｲｳｴｵｶｷｸ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最下位ビット
  //
  //ROXL.B #<data>,Dr
  //ROXL.B Dq,Dr
  //  拡張左ローテートバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*0X Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｲｳｴｵｶｷｸX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸX==0
  //     2 ........................ｳｴｵｶｷｸXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸX==0
  //     :
  //     7 ........................ｸXｱｲｳｴｵｶ ｷｸ*0ｷ Z=ｱｲｳｴｵｶｸX==0
  //     8 ........................Xｱｲｳｴｵｶｷ ｸX*0ｸ Z=ｱｲｳｴｵｶｷX==0
  //     9 ........................ｱｲｳｴｵｶｷｸ Xｱ*0X Z=ｱｲｳｴｵｶｷｸ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxlToRegByte () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASL.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) ((t = x << y) << 1));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | (z >> y + 1 != (byte) x ? XEiJ.REG_CCR_V : 0) | (byte) t >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //VはASRで元に戻せないときセット。XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSL.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) ((t = x << y) << 1));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | (byte) t >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXL.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      z = x << 1 | XEiJ.regCCR >> 4 & 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = (t = z << y - (2 - 1)) << 1 | (0xff & x) >>> 9 - 1 - y;
      }
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | (byte) t >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROL.B #<data>,Dr
      XEiJ.mpuCycleCount += 6 + ((y = XEiJ.regOC >> 9 & 7) << 1);  //y=data&7
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) (x << y | (0xff & x) >>> 8 - y));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z & 1;  //Xは変化しない。Cは結果の最下位ビット
      break;
    case 0b100_000 >> 3:  //ASL.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y <= 7) {  //y=data=0～7
        if (y == 0) {  //y=data=0
          z = (byte) x;
          t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。VとCはクリア
        } else {  //y=data=1～7
          XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) ((t = x << y - 1) << 1));
          t = (z >> y != (byte) x ? XEiJ.REG_CCR_V : 0) | (byte) t >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //VはASRで元に戻せないときセット。XとCは最後に押し出されたビット
        }
        XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      } else {  //y=data=8～63
        XEiJ.regRn[rrr] = ~0xff & x;
        XEiJ.regCCR = XEiJ.REG_CCR_Z | ((byte) x != 0 ? XEiJ.REG_CCR_V : 0) | (y == 8 ? -(x & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C) : 0);
      }
      break;
    case 0b101_000 >> 3:  //LSL.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = (byte) x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) ((t = y <= 8 ? x << y - 1 : 0) << 1));
        t = (byte) t >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b110_000 >> 3:  //ROXL.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      //y %= 9;
      y = (y & 7) - (y >> 3);  //y=data=-7～7
      y += y >> 3 & 9;  //y=data=0～8
      if (y == 0) {  //y=data=0
        z = (byte) x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～8
        z = x << 1 | XEiJ.regCCR >> 4 & 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～8
          z = (t = z << y - 2) << 1 | (0xff & x) >>> 9 - y;
        }
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) z);
        t = (byte) t >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROL.B Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {
        z = (byte) x;
        t = 0;  //Cはクリア
      } else {
        y &= 7;  //y=data=0～7
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) (x << y | (0xff & x) >>> 8 - y));
        t = z & 1;  //Cは結果の最下位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxlToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASL.W #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_101_000_rrr
  //LSL.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_101_001_rrr
  //ROXL.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_101_010_rrr
  //ROL.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_101_011_rrr
  //ASL.W Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_101_100_rrr
  //LSL.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_101_101_rrr
  //ROXL.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_101_110_rrr
  //ROL.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_101_111_rrr
  //ASL.W Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_101_000_rrr [ASL.W #1,Dr]
  //LSL.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_101_001_rrr [LSL.W #1,Dr]
  //ROXL.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_101_010_rrr [ROXL.W #1,Dr]
  //ROL.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_101_011_rrr [ROL.W #1,Dr]
  //
  //ASL.W #<data>,Dr
  //ASL.W Dq,Dr
  //ASL.W <ea>
  //  算術左シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ0 ｱｲ**ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0,V=ｱｲ!=0/-1
  //     :
  //    15 ................ﾀ000000000000000 ｿﾀ**ｿ Z=ﾀ==0,V=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ!=0/-1
  //    16 ................0000000000000000 ﾀ01*ﾀ V=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ!=0
  //    17 ................0000000000000000 001*0 V=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ!=0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  ASRで元に戻せないときセット。他はクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSL.W #<data>,Dr
  //LSL.W Dq,Dr
  //LSL.W <ea>
  //  論理左シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ0 ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     :
  //    15 ................ﾀ000000000000000 ｿﾀ*0ｿ Z=ﾀ==0
  //    16 ................0000000000000000 ﾀ010ﾀ
  //    17 ................0000000000000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROL.W #<data>,Dr
  //ROL.W Dq,Dr
  //ROL.W <ea>
  //  左ローテートワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀｱ Xｲ*0ｱ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     :
  //    15 ................ﾀｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ Xﾀ*0ｿ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //    16 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最下位ビット
  //
  //ROXL.W #<data>,Dr
  //ROXL.W Dq,Dr
  //ROXL.W <ea>
  //  拡張左ローテートワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX==0
  //     2 ................ｳｴｵｶｷｸｹｺｻｼｽｾｿﾀXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX==0
  //     :
  //    15 ................ﾀXｱｲｳｴｵｶｷｸｹｺｻｼｽｾ ｿﾀ*0ｿ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾﾀX==0
  //    16 ................Xｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀX*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿX==0
  //    17 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxlToRegWord () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASL.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) ((t = x << y) << 1));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | (z >> y + 1 != (short) x ? XEiJ.REG_CCR_V : 0) | (short) t >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //VはASRで元に戻せないときセット。XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSL.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) ((t = x << y) << 1));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | (short) t >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXL.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      z = x << 1 | XEiJ.regCCR >> 4 & 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = (t = z << y - (2 - 1)) << 1 | (char) x >>> 17 - 1 - y;
      }
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | (short) t >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROL.W #<data>,Dr
      XEiJ.mpuCycleCount += 6 + 2 + ((y = (XEiJ.regOC >> 9) - 1 & 7) << 1);  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) (x << y + 1 | (char) x >>> 16 - 1 - y));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z & 1;  //Xは変化しない。Cは結果の最下位ビット
      break;
    case 0b100_000 >> 3:  //ASL.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y <= 15) {  //y=data=0～15
        if (y == 0) {  //y=data=0
          z = (short) x;
          t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。VとCはクリア
        } else {  //y=data=1～15
          XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) ((t = x << y - 1) << 1));
          t = (z >> y != (short) x ? XEiJ.REG_CCR_V : 0) | (short) t >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //VはASRで元に戻せないときセット。XとCは最後に押し出されたビット
        }
        XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      } else {  //y=data=16～63
        XEiJ.regRn[rrr] = ~0xffff & x;
        XEiJ.regCCR = XEiJ.REG_CCR_Z | ((short) x != 0 ? XEiJ.REG_CCR_V : 0) | (y == 16 ? -(x & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C) : 0);
      }
      break;
    case 0b101_000 >> 3:  //LSL.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {  //y=data=0
        z = (short) x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) ((t = y <= 16 ? x << y - 1 : 0) << 1));
        t = (short) t >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b110_000 >> 3:  //ROXL.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      //y %= 17;
      y = (y & 15) - (y >> 4);  //y=data=-3～15
      y += y >> 4 & 17;  //y=data=0～16
      if (y == 0) {  //y=data=0
        z = (short) x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～16
        z = x << 1 | XEiJ.regCCR >> 4 & 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～16
          z = (t = z << y - 2) << 1 | (char) x >>> 17 - y;
        }
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) z);
        t = (short) t >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROL.W Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      XEiJ.mpuCycleCount += 6 + (y << 1);
      if (y == 0) {
        z = (short) x;
        t = 0;  //Cはクリア
      } else {
        y &= 15;  //y=data=0～15
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) (x << y | (char) x >>> 16 - y));
        t = z & 1;  //Cは結果の最下位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxlToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASL.L #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_110_000_rrr
  //LSL.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_110_001_rrr
  //ROXL.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_110_010_rrr
  //ROL.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_110_011_rrr
  //ASL.L Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_110_100_rrr
  //LSL.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_110_101_rrr
  //ROXL.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_110_110_rrr
  //ROL.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_110_111_rrr
  //ASL.L Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_110_000_rrr [ASL.L #1,Dr]
  //LSL.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_110_001_rrr [LSL.L #1,Dr]
  //ROXL.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_110_010_rrr [ROXL.L #1,Dr]
  //ROL.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_110_011_rrr [ROL.L #1,Dr]
  //
  //ASL.L #<data>,Dr
  //ASL.L Dq,Dr
  //  算術左シフトロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ**0 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ0 ｱｲ**ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0,V=ｱｲ!=0/-1
  //     :
  //    31 ﾐ0000000000000000000000000000000 ﾏﾐ**ﾏ Z=ﾐ==0,V=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ!=0/-1
  //    32 00000000000000000000000000000000 ﾐ01*ﾐ V=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ!=0
  //    33 00000000000000000000000000000000 001*0 V=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ!=0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  ASRで元に戻せないときセット。他はクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSL.L #<data>,Dr
  //LSL.L Dq,Dr
  //  論理左シフトロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ0 ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     :
  //    31 ﾐ0000000000000000000000000000000 ﾏﾐ*0ﾏ Z=ﾐ==0
  //    32 00000000000000000000000000000000 ﾐ010ﾐ
  //    33 00000000000000000000000000000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROL.L #<data>,Dr
  //ROL.L Dq,Dr
  //  左ローテートロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐｱ Xｲ*0ｱ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     :
  //    31 ﾐｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ Xﾐ*0ﾏ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //    3