//========================================================================================
//  MC68060.java
//    en:MC68060 core
//    ja:MC68060コア
//  Copyright (C) 2003-2025 Makoto Kamada
//
//  This file is part of the XEiJ (X68000 Emulator in Java).
//  You can use, modify and redistribute the XEiJ if the conditions are met.
//  Read the XEiJ License for more details.
//  https://stdkmd.net/xeij/
//========================================================================================

package xeij;

import java.lang.*;  //Boolean,Character,Class,Comparable,Double,Exception,Float,IllegalArgumentException,Integer,Long,Math,Number,Object,Runnable,SecurityException,String,StringBuilder,System
import java.util.*;  //ArrayList,Arrays,Calendar,GregorianCalendar,HashMap,Map,Map.Entry,Timer,TimerTask,TreeMap

public class MC68060 {

  public static void mpuCore () {

    //例外ループ
    //  別のメソッドで検出された例外を命令ループの外側でcatchすることで命令ループを高速化する
  errorLoop:
    while (XEiJ.mpuClockTime < XEiJ.mpuClockLimit) {
      try {
        //命令ループ
        while (XEiJ.mpuClockTime < XEiJ.mpuClockLimit) {
          int t;
          //命令を実行する
          m60Incremented = 0L;  //アドレスレジスタの増分
          XEiJ.mpuTraceFlag = XEiJ.regSRT1;  //命令実行前のsrT1
          XEiJ.mpuCycleCount = 0;  //第1オペコードからROMのアクセスウエイトを有効にする。命令のサイクル数はすべてXEiJ.mpuCycleCount+=～で加えること
          XEiJ.regPC0 = t = m60Address = XEiJ.regPC;  //命令の先頭アドレス
          XEiJ.regPC = t + 2;
          //XEiJ.regOC = mmuReadWordZeroOpword (t, XEiJ.regSRS);  //第1オペコード。必ずゼロ拡張すること。pcに奇数が入っていることはないのでアドレスエラーのチェックを省略する
          if (XEiJ.regSRS != 0) {  //スーパーバイザモード
            m60FSLW = M60_FSLW_IOMA_OPWORD | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | M60_FSLW_TM_SUPER_CODE;
            t = mmuTranslateReadSuperCode (t);
            XEiJ.regOC = (InstructionBreakPoint.IBP_ON ? InstructionBreakPoint.ibpOp1SuperMap : DataBreakPoint.DBP_ON ? DataBreakPoint.dbpSuperMap : XEiJ.busSuperMap)[t >>> XEiJ.BUS_PAGE_BITS].mmdRwz (t);
          } else {  //ユーザモード
            m60FSLW = M60_FSLW_IOMA_OPWORD | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | M60_FSLW_TM_USER_CODE;
            t = mmuTranslateReadUserCode (t);
            XEiJ.regOC = (InstructionBreakPoint.IBP_ON ? InstructionBreakPoint.ibpOp1UserMap : DataBreakPoint.DBP_ON ? DataBreakPoint.dbpUserMap : XEiJ.busUserMap)[t >>> XEiJ.BUS_PAGE_BITS].mmdRwz (t);
          }

          //命令の処理
          //  第1オペコードの上位10ビットで分岐する
        irpSwitch:
          switch (XEiJ.regOC >>> 6) {  //第1オペコードの上位10ビット。XEiJ.regOCはゼロ拡張されているので0b1111_111_111&を省略

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ORI.B #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_000_mmm_rrr-{data}
            //OR.B #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_000_mmm_rrr-{data}  [ORI.B #<data>,<ea>]
            //ORI.B #<data>,CCR                               |-|012346|-|*****|*****|          |0000_000_000_111_100-{data}
          case 0b0000_000_000:
            irpOriByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ORI.W #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_001_mmm_rrr-{data}
            //OR.W #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_001_mmm_rrr-{data}  [ORI.W #<data>,<ea>]
            //ORI.W #<data>,SR                                |-|012346|P|*****|*****|          |0000_000_001_111_100-{data}
          case 0b0000_000_001:
            irpOriWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ORI.L #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_010_mmm_rrr-{data}
            //OR.L #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_010_mmm_rrr-{data}  [ORI.L #<data>,<ea>]
          case 0b0000_000_010:
            irpOriLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BITREV.L Dr                                     |-|------|-|-----|-----|D         |0000_000_011_000_rrr (ISA_C)
            //CMP2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn000000000000
            //CHK2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn100000000000
          case 0b0000_000_011:
            irpCmp2Chk2Byte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BTST.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_100_000_rrr
            //MOVEP.W (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_100_001_rrr-{data}
            //BTST.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZPI|0000_qqq_100_mmm_rrr
          case 0b0000_000_100:
          case 0b0000_001_100:
          case 0b0000_010_100:
          case 0b0000_011_100:
          case 0b0000_100_100:
          case 0b0000_101_100:
          case 0b0000_110_100:
          case 0b0000_111_100:
            irpBtstReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCHG.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_101_000_rrr
            //MOVEP.L (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_101_001_rrr-{data}
            //BCHG.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_101_mmm_rrr
          case 0b0000_000_101:
          case 0b0000_001_101:
          case 0b0000_010_101:
          case 0b0000_011_101:
          case 0b0000_100_101:
          case 0b0000_101_101:
          case 0b0000_110_101:
          case 0b0000_111_101:
            irpBchgReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCLR.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_110_000_rrr
            //MOVEP.W Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_110_001_rrr-{data}
            //BCLR.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_110_mmm_rrr
          case 0b0000_000_110:
          case 0b0000_001_110:
          case 0b0000_010_110:
          case 0b0000_011_110:
          case 0b0000_100_110:
          case 0b0000_101_110:
          case 0b0000_110_110:
          case 0b0000_111_110:
            irpBclrReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSET.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_111_000_rrr
            //MOVEP.L Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_111_001_rrr-{data}
            //BSET.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_111_mmm_rrr
          case 0b0000_000_111:
          case 0b0000_001_111:
          case 0b0000_010_111:
          case 0b0000_011_111:
          case 0b0000_100_111:
          case 0b0000_101_111:
          case 0b0000_110_111:
          case 0b0000_111_111:
            irpBsetReg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ANDI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_000_mmm_rrr-{data}
            //AND.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_000_mmm_rrr-{data}  [ANDI.B #<data>,<ea>]
            //ANDI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_001_000_111_100-{data}
          case 0b0000_001_000:
            irpAndiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ANDI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_001_mmm_rrr-{data}
            //AND.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_001_mmm_rrr-{data}  [ANDI.W #<data>,<ea>]
            //ANDI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_001_001_111_100-{data}
          case 0b0000_001_001:
            irpAndiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ANDI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_010_mmm_rrr-{data}
            //AND.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_010_mmm_rrr-{data}  [ANDI.L #<data>,<ea>]
          case 0b0000_001_010:
            irpAndiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BYTEREV.L Dr                                    |-|------|-|-----|-----|D         |0000_001_011_000_rrr (ISA_C)
            //CMP2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn000000000000
            //CHK2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn100000000000
          case 0b0000_001_011:
            irpCmp2Chk2Word ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_000_mmm_rrr-{data}
            //SUB.B #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_000_mmm_rrr-{data}  [SUBI.B #<data>,<ea>]
          case 0b0000_010_000:
            irpSubiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_001_mmm_rrr-{data}
            //SUB.W #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_001_mmm_rrr-{data}  [SUBI.W #<data>,<ea>]
          case 0b0000_010_001:
            irpSubiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_010_mmm_rrr-{data}
            //SUB.L #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_010_mmm_rrr-{data}  [SUBI.L #<data>,<ea>]
          case 0b0000_010_010:
            irpSubiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FF1.L Dr                                        |-|------|-|-UUUU|-**00|D         |0000_010_011_000_rrr (ISA_C)
            //CMP2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn000000000000
            //CHK2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn100000000000
          case 0b0000_010_011:
            irpCmp2Chk2Long ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_000_mmm_rrr-{data}
          case 0b0000_011_000:
            irpAddiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_001_mmm_rrr-{data}
          case 0b0000_011_001:
            irpAddiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_010_mmm_rrr-{data}
          case 0b0000_011_010:
            irpAddiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BTST.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_000_000_rrr-{data}
            //BTST.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZP |0000_100_000_mmm_rrr-{data}
          case 0b0000_100_000:
            irpBtstImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCHG.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_001_000_rrr-{data}
            //BCHG.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_001_mmm_rrr-{data}
          case 0b0000_100_001:
            irpBchgImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCLR.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_010_000_rrr-{data}
            //BCLR.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_010_mmm_rrr-{data}
          case 0b0000_100_010:
            irpBclrImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSET.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_011_000_rrr-{data}
            //BSET.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_011_mmm_rrr-{data}
          case 0b0000_100_011:
            irpBsetImm ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EORI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}
            //EOR.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}  [EORI.B #<data>,<ea>]
            //EORI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_101_000_111_100-{data}
          case 0b0000_101_000:
            irpEoriByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EORI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}
            //EOR.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}  [EORI.W #<data>,<ea>]
            //EORI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_101_001_111_100-{data}
          case 0b0000_101_001:
            irpEoriWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EORI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}
            //EOR.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}  [EORI.L #<data>,<ea>]
          case 0b0000_101_010:
            irpEoriLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CAS.B Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_101_011_mmm_rrr-0000000uuu000ccc
          case 0b0000_101_011:
            irpCasByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPI.B #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_000_mmm_rrr-{data}
            //CMP.B #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_000_mmm_rrr-{data}  [CMPI.B #<data>,<ea>]
          case 0b0000_110_000:
            irpCmpiByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPI.W #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_001_mmm_rrr-{data}
            //CMP.W #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_001_mmm_rrr-{data}  [CMPI.W #<data>,<ea>]
          case 0b0000_110_001:
            irpCmpiWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPI.L #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_010_mmm_rrr-{data}
            //CMP.L #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_010_mmm_rrr-{data}  [CMPI.L #<data>,<ea>]
          case 0b0000_110_010:
            irpCmpiLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CAS.W Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_110_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
            //CAS2.W Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_110_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
          case 0b0000_110_011:
            irpCasWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVES.B <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn000000000000
            //MOVES.B Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn100000000000
          case 0b0000_111_000:
            irpMovesByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVES.W <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn000000000000
            //MOVES.W Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn100000000000
          case 0b0000_111_001:
            irpMovesWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVES.L <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn000000000000
            //MOVES.L Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn100000000000
          case 0b0000_111_010:
            irpMovesLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CAS.L Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_111_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
            //CAS2.L Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_111_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
          case 0b0000_111_011:
            irpCasLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,Dq                                  |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_000_mmm_rrr
          case 0b0001_000_000:
          case 0b0001_001_000:
          case 0b0001_010_000:
          case 0b0001_011_000:
          case 0b0001_100_000:
          case 0b0001_101_000:
          case 0b0001_110_000:
          case 0b0001_111_000:
            irpMoveToDRByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_010_mmm_rrr
          case 0b0001_000_010:
          case 0b0001_001_010:
          case 0b0001_010_010:
          case 0b0001_011_010:
          case 0b0001_100_010:
          case 0b0001_101_010:
          case 0b0001_110_010:
          case 0b0001_111_010:
            irpMoveToMMByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_011_mmm_rrr
          case 0b0001_000_011:
          case 0b0001_001_011:
          case 0b0001_010_011:
          case 0b0001_011_011:
          case 0b0001_100_011:
          case 0b0001_101_011:
          case 0b0001_110_011:
          case 0b0001_111_011:
            irpMoveToMPByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_100_mmm_rrr
          case 0b0001_000_100:
          case 0b0001_001_100:
          case 0b0001_010_100:
          case 0b0001_011_100:
          case 0b0001_100_100:
          case 0b0001_101_100:
          case 0b0001_110_100:
          case 0b0001_111_100:
            irpMoveToMNByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_101_mmm_rrr
          case 0b0001_000_101:
          case 0b0001_001_101:
          case 0b0001_010_101:
          case 0b0001_011_101:
          case 0b0001_100_101:
          case 0b0001_101_101:
          case 0b0001_110_101:
          case 0b0001_111_101:
            irpMoveToMWByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_110_mmm_rrr
          case 0b0001_000_110:
          case 0b0001_001_110:
          case 0b0001_010_110:
          case 0b0001_011_110:
          case 0b0001_100_110:
          case 0b0001_101_110:
          case 0b0001_110_110:
          case 0b0001_111_110:
            irpMoveToMXByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_000_111_mmm_rrr
          case 0b0001_000_111:
            irpMoveToZWByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.B <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_001_111_mmm_rrr
          case 0b0001_001_111:
            irpMoveToZLByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_000_mmm_rrr
          case 0b0010_000_000:
          case 0b0010_001_000:
          case 0b0010_010_000:
          case 0b0010_011_000:
          case 0b0010_100_000:
          case 0b0010_101_000:
          case 0b0010_110_000:
          case 0b0010_111_000:
            irpMoveToDRLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVEA.L <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr
            //MOVE.L <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr [MOVEA.L <ea>,Aq]
          case 0b0010_000_001:
          case 0b0010_001_001:
          case 0b0010_010_001:
          case 0b0010_011_001:
          case 0b0010_100_001:
          case 0b0010_101_001:
          case 0b0010_110_001:
          case 0b0010_111_001:
            irpMoveaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_010_mmm_rrr
          case 0b0010_000_010:
          case 0b0010_001_010:
          case 0b0010_010_010:
          case 0b0010_011_010:
          case 0b0010_100_010:
          case 0b0010_101_010:
          case 0b0010_110_010:
          case 0b0010_111_010:
            irpMoveToMMLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_011_mmm_rrr
          case 0b0010_000_011:
          case 0b0010_001_011:
          case 0b0010_010_011:
          case 0b0010_011_011:
          case 0b0010_100_011:
          case 0b0010_101_011:
          case 0b0010_110_011:
          case 0b0010_111_011:
            irpMoveToMPLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_100_mmm_rrr
          case 0b0010_000_100:
          case 0b0010_001_100:
          case 0b0010_010_100:
          case 0b0010_011_100:
          case 0b0010_100_100:
          case 0b0010_101_100:
          case 0b0010_110_100:
          case 0b0010_111_100:
            irpMoveToMNLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_101_mmm_rrr
          case 0b0010_000_101:
          case 0b0010_001_101:
          case 0b0010_010_101:
          case 0b0010_011_101:
          case 0b0010_100_101:
          case 0b0010_101_101:
          case 0b0010_110_101:
          case 0b0010_111_101:
            irpMoveToMWLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_110_mmm_rrr
          case 0b0010_000_110:
          case 0b0010_001_110:
          case 0b0010_010_110:
          case 0b0010_011_110:
          case 0b0010_100_110:
          case 0b0010_101_110:
          case 0b0010_110_110:
          case 0b0010_111_110:
            irpMoveToMXLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_000_111_mmm_rrr
          case 0b0010_000_111:
            irpMoveToZWLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.L <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_001_111_mmm_rrr
          case 0b0010_001_111:
            irpMoveToZLLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_000_mmm_rrr
          case 0b0011_000_000:
          case 0b0011_001_000:
          case 0b0011_010_000:
          case 0b0011_011_000:
          case 0b0011_100_000:
          case 0b0011_101_000:
          case 0b0011_110_000:
          case 0b0011_111_000:
            irpMoveToDRWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVEA.W <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr
            //MOVE.W <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr [MOVEA.W <ea>,Aq]
          case 0b0011_000_001:
          case 0b0011_001_001:
          case 0b0011_010_001:
          case 0b0011_011_001:
          case 0b0011_100_001:
          case 0b0011_101_001:
          case 0b0011_110_001:
          case 0b0011_111_001:
            irpMoveaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_010_mmm_rrr
          case 0b0011_000_010:
          case 0b0011_001_010:
          case 0b0011_010_010:
          case 0b0011_011_010:
          case 0b0011_100_010:
          case 0b0011_101_010:
          case 0b0011_110_010:
          case 0b0011_111_010:
            irpMoveToMMWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_011_mmm_rrr
          case 0b0011_000_011:
          case 0b0011_001_011:
          case 0b0011_010_011:
          case 0b0011_011_011:
          case 0b0011_100_011:
          case 0b0011_101_011:
          case 0b0011_110_011:
          case 0b0011_111_011:
            irpMoveToMPWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_100_mmm_rrr
          case 0b0011_000_100:
          case 0b0011_001_100:
          case 0b0011_010_100:
          case 0b0011_011_100:
          case 0b0011_100_100:
          case 0b0011_101_100:
          case 0b0011_110_100:
          case 0b0011_111_100:
            irpMoveToMNWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_101_mmm_rrr
          case 0b0011_000_101:
          case 0b0011_001_101:
          case 0b0011_010_101:
          case 0b0011_011_101:
          case 0b0011_100_101:
          case 0b0011_101_101:
          case 0b0011_110_101:
          case 0b0011_111_101:
            irpMoveToMWWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_110_mmm_rrr
          case 0b0011_000_110:
          case 0b0011_001_110:
          case 0b0011_010_110:
          case 0b0011_011_110:
          case 0b0011_100_110:
          case 0b0011_101_110:
          case 0b0011_110_110:
          case 0b0011_111_110:
            irpMoveToMXWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_000_111_mmm_rrr
          case 0b0011_000_111:
            irpMoveToZWWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_001_111_mmm_rrr
          case 0b0011_001_111:
            irpMoveToZLWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEGX.B <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_000_mmm_rrr
          case 0b0100_000_000:
            irpNegxByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEGX.W <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_001_mmm_rrr
          case 0b0100_000_001:
            irpNegxWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEGX.L <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_010_mmm_rrr
          case 0b0100_000_010:
            irpNegxLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W SR,<ea>                                  |-|-12346|P|*****|-----|D M+-WXZ  |0100_000_011_mmm_rrr
          case 0b0100_000_011:
            irpMoveFromSR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CHK.L <ea>,Dq                                   |-|--2346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_100_mmm_rrr
          case 0b0100_000_100:
          case 0b0100_001_100:
          case 0b0100_010_100:
          case 0b0100_011_100:
          case 0b0100_100_100:
          case 0b0100_101_100:
          case 0b0100_110_100:
          case 0b0100_111_100:
            irpChkLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CHK.W <ea>,Dq                                   |-|012346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_110_mmm_rrr
          case 0b0100_000_110:
          case 0b0100_001_110:
          case 0b0100_010_110:
          case 0b0100_011_110:
          case 0b0100_100_110:
          case 0b0100_101_110:
          case 0b0100_110_110:
          case 0b0100_111_110:
            irpChkWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LEA.L <ea>,Aq                                   |-|012346|-|-----|-----|  M  WXZP |0100_qqq_111_mmm_rrr
            //EXTB.L Dr                                       |-|--2346|-|-UUUU|-**00|D         |0100_100_111_000_rrr
          case 0b0100_000_111:
          case 0b0100_001_111:
          case 0b0100_010_111:
          case 0b0100_011_111:
          case 0b0100_100_111:
          case 0b0100_101_111:
          case 0b0100_110_111:
          case 0b0100_111_111:
            irpLea ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CLR.B <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_000_mmm_rrr (68000 and 68008 read before clear)
          case 0b0100_001_000:
            irpClrByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CLR.W <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_001_mmm_rrr (68000 and 68008 read before clear)
          case 0b0100_001_001:
            irpClrWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CLR.L <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_010_mmm_rrr (68000 and 68008 read before clear)
          case 0b0100_001_010:
            irpClrLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W CCR,<ea>                                 |-|-12346|-|*****|-----|D M+-WXZ  |0100_001_011_mmm_rrr
          case 0b0100_001_011:
            irpMoveFromCCR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEG.B <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_000_mmm_rrr
          case 0b0100_010_000:
            irpNegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEG.W <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_001_mmm_rrr
          case 0b0100_010_001:
            irpNegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NEG.L <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_010_mmm_rrr
          case 0b0100_010_010:
            irpNegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,CCR                                 |-|012346|-|UUUUU|*****|D M+-WXZPI|0100_010_011_mmm_rrr
          case 0b0100_010_011:
            irpMoveToCCR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NOT.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_000_mmm_rrr
          case 0b0100_011_000:
            irpNotByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NOT.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_001_mmm_rrr
          case 0b0100_011_001:
            irpNotWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NOT.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_010_mmm_rrr
          case 0b0100_011_010:
            irpNotLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE.W <ea>,SR                                  |-|012346|P|UUUUU|*****|D M+-WXZPI|0100_011_011_mmm_rrr
          case 0b0100_011_011:
            irpMoveToSR ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //NBCD.B <ea>                                     |-|012346|-|UUUUU|*U*U*|D M+-WXZ  |0100_100_000_mmm_rrr
            //LINK.L Ar,#<data>                               |-|--2346|-|-----|-----|          |0100_100_000_001_rrr-{data}
          case 0b0100_100_000:
            irpNbcd ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SWAP.W Dr                                       |-|012346|-|-UUUU|-**00|D         |0100_100_001_000_rrr
            //BKPT #<data>                                    |-|-12346|-|-----|-----|          |0100_100_001_001_ddd
            //PEA.L <ea>                                      |-|012346|-|-----|-----|  M  WXZP |0100_100_001_mmm_rrr
          case 0b0100_100_001:
            irpPea ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXT.W Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_010_000_rrr
            //MOVEM.W <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_010_mmm_rrr-llllllllllllllll
          case 0b0100_100_010:
            irpMovemToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXT.L Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_011_000_rrr
            //MOVEM.L <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_011_mmm_rrr-llllllllllllllll
          case 0b0100_100_011:
            irpMovemToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TST.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_000_mmm_rrr
            //TST.B <ea>                                      |-|--2346|-|-UUUU|-**00|        PI|0100_101_000_mmm_rrr
          case 0b0100_101_000:
            irpTstByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TST.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_001_mmm_rrr
            //TST.W <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_001_mmm_rrr
          case 0b0100_101_001:
            irpTstWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TST.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_010_mmm_rrr
            //TST.L <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_010_mmm_rrr
          case 0b0100_101_010:
            irpTstLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //TAS.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_011_mmm_rrr
            //ILLEGAL                                         |-|012346|-|-----|-----|          |0100_101_011_111_100
          case 0b0100_101_011:
            irpTas ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MULU.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll000000000hhh        (h is not used)
            //MULU.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll010000000hhh        (if h=l then result is not defined)
            //MULS.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll100000000hhh        (h is not used)
            //MULS.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll110000000hhh        (if h=l then result is not defined)
          case 0b0100_110_000:
            irpMuluMulsLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DIVU.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000qqq
            //DIVUL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000rrr        (q is not equal to r)
            //DIVU.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq010000000rrr        (q is not equal to r)
            //DIVS.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000qqq
            //DIVSL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000rrr        (q is not equal to r)
            //DIVS.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq110000000rrr        (q is not equal to r)
          case 0b0100_110_001:
            irpDivuDivsLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SATS.L Dr                                       |-|------|-|-UUUU|-**00|D         |0100_110_010_000_rrr (ISA_B)
            //MOVEM.W <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_010_mmm_rrr-llllllllllllllll
          case 0b0100_110_010:
            irpMovemToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVEM.L <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_011_mmm_rrr-llllllllllllllll
          case 0b0100_110_011:
            irpMovemToRegLong ();
            break irpSwitch;

          case 0b0100_111_001:
            switch (XEiJ.regOC & 0b111_111) {

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //TRAP #<vector>                                  |-|012346|-|-----|-----|          |0100_111_001_00v_vvv
            case 0b000_000:
            case 0b000_001:
            case 0b000_010:
            case 0b000_011:
            case 0b000_100:
            case 0b000_101:
            case 0b000_110:
            case 0b000_111:
            case 0b001_000:
            case 0b001_001:
            case 0b001_010:
            case 0b001_011:
            case 0b001_100:
            case 0b001_101:
            case 0b001_110:
              irpTrap ();
              break irpSwitch;
            case 0b001_111:
              irpTrap15 ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //LINK.W Ar,#<data>                               |-|012346|-|-----|-----|          |0100_111_001_010_rrr-{data}
            case 0b010_000:
            case 0b010_001:
            case 0b010_010:
            case 0b010_011:
            case 0b010_100:
            case 0b010_101:
            case 0b010_110:
            case 0b010_111:
              irpLinkWord ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //UNLK Ar                                         |-|012346|-|-----|-----|          |0100_111_001_011_rrr
            case 0b011_000:
            case 0b011_001:
            case 0b011_010:
            case 0b011_011:
            case 0b011_100:
            case 0b011_101:
            case 0b011_110:
            case 0b011_111:
              irpUnlk ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVE.L Ar,USP                                   |-|012346|P|-----|-----|          |0100_111_001_100_rrr
            case 0b100_000:
            case 0b100_001:
            case 0b100_010:
            case 0b100_011:
            case 0b100_100:
            case 0b100_101:
            case 0b100_110:
            case 0b100_111:
              irpMoveToUsp ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVE.L USP,Ar                                   |-|012346|P|-----|-----|          |0100_111_001_101_rrr
            case 0b101_000:
            case 0b101_001:
            case 0b101_010:
            case 0b101_011:
            case 0b101_100:
            case 0b101_101:
            case 0b101_110:
            case 0b101_111:
              irpMoveFromUsp ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RESET                                           |-|012346|P|-----|-----|          |0100_111_001_110_000
            case 0b110_000:
              irpReset ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //NOP                                             |-|012346|-|-----|-----|          |0100_111_001_110_001
            case 0b110_001:
              irpNop ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //STOP #<data>                                    |-|012346|P|UUUUU|*****|          |0100_111_001_110_010-{data}
            case 0b110_010:
              irpStop ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTE                                             |-|012346|P|UUUUU|*****|          |0100_111_001_110_011
            case 0b110_011:
              irpRte ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTD #<data>                                     |-|-12346|-|-----|-----|          |0100_111_001_110_100-{data}
            case 0b110_100:
              irpRtd ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTS                                             |-|012346|-|-----|-----|          |0100_111_001_110_101
            case 0b110_101:
              irpRts ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //TRAPV                                           |-|012346|-|---*-|-----|          |0100_111_001_110_110
            case 0b110_110:
              irpTrapv ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //RTR                                             |-|012346|-|UUUUU|*****|          |0100_111_001_110_111
            case 0b110_111:
              irpRtr ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVEC.L Rc,Rn                                   |-|-12346|P|-----|-----|          |0100_111_001_111_010-rnnncccccccccccc
            case 0b111_010:
              irpMovecFromControl ();
              break irpSwitch;

              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
              //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
              //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
              //MOVEC.L Rn,Rc                                   |-|-12346|P|-----|-----|          |0100_111_001_111_011-rnnncccccccccccc
            case 0b111_011:
              irpMovecToControl ();
              break irpSwitch;

            default:
              irpIllegal ();

            }  //switch XEiJ.regOC & 0b111_111
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //JSR <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_010_mmm_rrr
            //JBSR.L <label>                                  |A|012346|-|-----|-----|          |0100_111_010_111_001-{address}       [JSR <label>]
          case 0b0100_111_010:
            irpJsr ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //JMP <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_011_mmm_rrr
            //JBRA.L <label>                                  |A|012346|-|-----|-----|          |0100_111_011_111_001-{address}       [JMP <label>]
          case 0b0100_111_011:
            irpJmp ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_000_mmm_rrr
            //INC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_000_mmm_rrr [ADDQ.B #1,<ea>]
          case 0b0101_000_000:
          case 0b0101_001_000:
          case 0b0101_010_000:
          case 0b0101_011_000:
          case 0b0101_100_000:
          case 0b0101_101_000:
          case 0b0101_110_000:
          case 0b0101_111_000:
            irpAddqByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_001_mmm_rrr
            //ADDQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_001_001_rrr
            //INC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_001_mmm_rrr [ADDQ.W #1,<ea>]
            //INC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_001_001_rrr [ADDQ.W #1,Ar]
          case 0b0101_000_001:
          case 0b0101_001_001:
          case 0b0101_010_001:
          case 0b0101_011_001:
          case 0b0101_100_001:
          case 0b0101_101_001:
          case 0b0101_110_001:
          case 0b0101_111_001:
            irpAddqWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_010_mmm_rrr
            //ADDQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_010_001_rrr
            //INC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_010_mmm_rrr [ADDQ.L #1,<ea>]
            //INC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_010_001_rrr [ADDQ.L #1,Ar]
          case 0b0101_000_010:
          case 0b0101_001_010:
          case 0b0101_010_010:
          case 0b0101_011_010:
          case 0b0101_100_010:
          case 0b0101_101_010:
          case 0b0101_110_010:
          case 0b0101_111_010:
            irpAddqLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ST.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr
            //SNF.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr [ST.B <ea>]
            //DBT.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}
            //DBNF.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}        [DBT.W Dr,<label>]
            //TRAPT.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_010-{data}
            //TPNF.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
            //TPT.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
            //TRAPNF.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
            //TRAPT.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_011-{data}
            //TPNF.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
            //TPT.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
            //TRAPNF.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
            //TRAPT                                           |-|--2346|-|-----|-----|          |0101_000_011_111_100
            //TPNF                                            |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
            //TPT                                             |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
            //TRAPNF                                          |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
          case 0b0101_000_011:
            irpSt ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_100_mmm_rrr
            //DEC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_100_mmm_rrr [SUBQ.B #1,<ea>]
          case 0b0101_000_100:
          case 0b0101_001_100:
          case 0b0101_010_100:
          case 0b0101_011_100:
          case 0b0101_100_100:
          case 0b0101_101_100:
          case 0b0101_110_100:
          case 0b0101_111_100:
            irpSubqByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_101_mmm_rrr
            //SUBQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_101_001_rrr
            //DEC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_101_mmm_rrr [SUBQ.W #1,<ea>]
            //DEC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_101_001_rrr [SUBQ.W #1,Ar]
          case 0b0101_000_101:
          case 0b0101_001_101:
          case 0b0101_010_101:
          case 0b0101_011_101:
          case 0b0101_100_101:
          case 0b0101_101_101:
          case 0b0101_110_101:
          case 0b0101_111_101:
            irpSubqWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_110_mmm_rrr
            //SUBQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_110_001_rrr
            //DEC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_110_mmm_rrr [SUBQ.L #1,<ea>]
            //DEC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_110_001_rrr [SUBQ.L #1,Ar]
          case 0b0101_000_110:
          case 0b0101_001_110:
          case 0b0101_010_110:
          case 0b0101_011_110:
          case 0b0101_100_110:
          case 0b0101_101_110:
          case 0b0101_110_110:
          case 0b0101_111_110:
            irpSubqLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SF.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr
            //SNT.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr [SF.B <ea>]
            //DBF.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}
            //DBNT.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
            //DBRA.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
            //TRAPF.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_010-{data}
            //TPF.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
            //TPNT.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
            //TRAPNT.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
            //TRAPF.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_011-{data}
            //TPF.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
            //TPNT.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
            //TRAPNT.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
            //TRAPF                                           |-|--2346|-|-----|-----|          |0101_000_111_111_100
            //TPF                                             |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
            //TPNT                                            |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
            //TRAPNT                                          |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
          case 0b0101_000_111:
            irpSf ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SHI.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr
            //SNLS.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr [SHI.B <ea>]
            //DBHI.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}
            //DBNLS.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}        [DBHI.W Dr,<label>]
            //TRAPHI.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}
            //TPHI.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
            //TPNLS.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
            //TRAPNLS.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
            //TRAPHI.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}
            //TPHI.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
            //TPNLS.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
            //TRAPNLS.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
            //TRAPHI                                          |-|--2346|-|--*-*|-----|          |0101_001_011_111_100
            //TPHI                                            |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
            //TPNLS                                           |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
            //TRAPNLS                                         |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
          case 0b0101_001_011:
            irpShi ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SLS.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr
            //SNHI.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr [SLS.B <ea>]
            //DBLS.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}
            //DBNHI.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}        [DBLS.W Dr,<label>]
            //TRAPLS.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}
            //TPLS.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}
            //TPNHI.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
            //TRAPNHI.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
            //TRAPLS.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}
            //TPLS.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}
            //TPNHI.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
            //TRAPNHI.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
            //TRAPLS                                          |-|--2346|-|--*-*|-----|          |0101_001_111_111_100
            //TPLS                                            |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
            //TPNHI                                           |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
            //TRAPNHI                                         |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
          case 0b0101_001_111:
            irpSls ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SCC.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr
            //SHS.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
            //SNCS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
            //SNLO.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
            //DBCC.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}
            //DBHS.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
            //DBNCS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
            //DBNLO.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
            //TRAPCC.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_010-{data}
            //TPCC.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TPHS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TPNCS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TPNLO.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPHS.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPNCS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPNLO.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
            //TRAPCC.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_011-{data}
            //TPCC.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TPHS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TPNCS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TPNLO.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPHS.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPNCS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPNLO.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
            //TRAPCC                                          |-|--2346|-|----*|-----|          |0101_010_011_111_100
            //TPCC                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TPHS                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TPNCS                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TPNLO                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TRAPHS                                          |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TRAPNCS                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
            //TRAPNLO                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
          case 0b0101_010_011:
            irpShs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SCS.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr
            //SLO.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
            //SNCC.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
            //SNHS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
            //DBCS.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}
            //DBLO.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
            //DBNCC.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
            //DBNHS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
            //TRAPCS.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_010-{data}
            //TPCS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TPLO.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TPNCC.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TPNHS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPLO.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPNCC.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPNHS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
            //TRAPCS.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_011-{data}
            //TPCS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TPLO.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TPNCC.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TPNHS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPLO.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPNCC.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPNHS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
            //TRAPCS                                          |-|--2346|-|----*|-----|          |0101_010_111_111_100
            //TPCS                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TPLO                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TPNCC                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TPNHS                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TRAPLO                                          |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TRAPNCC                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
            //TRAPNHS                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
          case 0b0101_010_111:
            irpSlo ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SNE.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr
            //SNEQ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
            //SNZ.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
            //SNZE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
            //DBNE.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}
            //DBNEQ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
            //DBNZ.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
            //DBNZE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
            //TRAPNE.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}
            //TPNE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TPNEQ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TPNZ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TPNZE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNEQ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNZ.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNZE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
            //TRAPNE.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}
            //TPNE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TPNEQ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TPNZ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TPNZE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNEQ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNZ.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNZE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
            //TRAPNE                                          |-|--2346|-|--*--|-----|          |0101_011_011_111_100
            //TPNE                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TPNEQ                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TPNZ                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TPNZE                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TRAPNEQ                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TRAPNZ                                          |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
            //TRAPNZE                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
          case 0b0101_011_011:
            irpSne ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SEQ.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr
            //SNNE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
            //SNNZ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
            //SZE.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
            //DBEQ.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}
            //DBNNE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
            //DBNNZ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
            //DBZE.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
            //TRAPEQ.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}
            //TPEQ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TPNNE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TPNNZ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TPZE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPNNE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPNNZ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPZE.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
            //TRAPEQ.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}
            //TPEQ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TPNNE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TPNNZ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TPZE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPNNE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPNNZ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPZE.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
            //TRAPEQ                                          |-|--2346|-|--*--|-----|          |0101_011_111_111_100
            //TPEQ                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TPNNE                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TPNNZ                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TPZE                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TRAPNNE                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TRAPNNZ                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
            //TRAPZE                                          |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
          case 0b0101_011_111:
            irpSeq ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SVC.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr
            //SNVS.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr [SVC.B <ea>]
            //DBVC.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}
            //DBNVS.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}        [DBVC.W Dr,<label>]
            //TRAPVC.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}
            //TPNVS.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
            //TPVC.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
            //TRAPNVS.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
            //TRAPVC.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}
            //TPNVS.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
            //TPVC.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
            //TRAPNVS.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
            //TRAPVC                                          |-|--2346|-|---*-|-----|          |0101_100_011_111_100
            //TPNVS                                           |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
            //TPVC                                            |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
            //TRAPNVS                                         |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
          case 0b0101_100_011:
            irpSvc ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SVS.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr
            //SNVC.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr [SVS.B <ea>]
            //DBVS.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}
            //DBNVC.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}        [DBVS.W Dr,<label>]
            //TRAPVS.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}
            //TPNVC.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
            //TPVS.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
            //TRAPNVC.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
            //TRAPVS.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}
            //TPNVC.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
            //TPVS.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
            //TRAPNVC.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
            //TRAPVS                                          |-|--2346|-|---*-|-----|          |0101_100_111_111_100
            //TPNVC                                           |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
            //TPVS                                            |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
            //TRAPNVC                                         |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
          case 0b0101_100_111:
            irpSvs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SPL.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr
            //SNMI.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr [SPL.B <ea>]
            //DBPL.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}
            //DBNMI.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}        [DBPL.W Dr,<label>]
            //TRAPPL.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}
            //TPNMI.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
            //TPPL.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
            //TRAPNMI.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
            //TRAPPL.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}
            //TPNMI.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
            //TPPL.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
            //TRAPNMI.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
            //TRAPPL                                          |-|--2346|-|-*---|-----|          |0101_101_011_111_100
            //TPNMI                                           |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
            //TPPL                                            |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
            //TRAPNMI                                         |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
          case 0b0101_101_011:
            irpSpl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SMI.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr
            //SNPL.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr [SMI.B <ea>]
            //DBMI.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}
            //DBNPL.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}        [DBMI.W Dr,<label>]
            //TRAPMI.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}
            //TPMI.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
            //TPNPL.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
            //TRAPNPL.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
            //TRAPMI.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}
            //TPMI.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
            //TPNPL.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
            //TRAPNPL.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
            //TRAPMI                                          |-|--2346|-|-*---|-----|          |0101_101_111_111_100
            //TPMI                                            |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
            //TPNPL                                           |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
            //TRAPNPL                                         |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
          case 0b0101_101_111:
            irpSmi ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SGE.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr
            //SNLT.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr [SGE.B <ea>]
            //DBGE.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}
            //DBNLT.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}        [DBGE.W Dr,<label>]
            //TRAPGE.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}
            //TPGE.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
            //TPNLT.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
            //TRAPNLT.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
            //TRAPGE.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}
            //TPGE.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
            //TPNLT.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
            //TRAPNLT.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
            //TRAPGE                                          |-|--2346|-|-*-*-|-----|          |0101_110_011_111_100
            //TPGE                                            |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
            //TPNLT                                           |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
            //TRAPNLT                                         |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
          case 0b0101_110_011:
            irpSge ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SLT.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr
            //SNGE.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr [SLT.B <ea>]
            //DBLT.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}
            //DBNGE.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}        [DBLT.W Dr,<label>]
            //TRAPLT.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}
            //TPLT.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
            //TPNGE.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
            //TRAPNGE.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
            //TRAPLT.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}
            //TPLT.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
            //TPNGE.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
            //TRAPNGE.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
            //TRAPLT                                          |-|--2346|-|-*-*-|-----|          |0101_110_111_111_100
            //TPLT                                            |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
            //TPNGE                                           |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
            //TRAPNGE                                         |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
          case 0b0101_110_111:
            irpSlt ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SGT.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr
            //SNLE.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr [SGT.B <ea>]
            //DBGT.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}
            //DBNLE.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}        [DBGT.W Dr,<label>]
            //TRAPGT.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}
            //TPGT.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
            //TPNLE.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
            //TRAPNLE.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
            //TRAPGT.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}
            //TPGT.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
            //TPNLE.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
            //TRAPNLE.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
            //TRAPGT                                          |-|--2346|-|-***-|-----|          |0101_111_011_111_100
            //TPGT                                            |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
            //TPNLE                                           |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
            //TRAPNLE                                         |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
          case 0b0101_111_011:
            irpSgt ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SLE.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr
            //SNGT.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr [SLE.B <ea>]
            //DBLE.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}
            //DBNGT.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}        [DBLE.W Dr,<label>]
            //TRAPLE.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}
            //TPLE.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
            //TPNGT.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
            //TRAPNGT.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
            //TRAPLE.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}
            //TPLE.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
            //TPNGT.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
            //TRAPNGT.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
            //TRAPLE                                          |-|--2346|-|-***-|-----|          |0101_111_111_111_100
            //TPLE                                            |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
            //TPNGT                                           |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
            //TRAPNGT                                         |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
          case 0b0101_111_111:
            irpSle ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.W <label>                                   |-|012346|-|-----|-----|          |0110_000_000_000_000-{offset}
            //JBRA.W <label>                                  |A|012346|-|-----|-----|          |0110_000_000_000_000-{offset}        [BRA.W <label>]
            //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)
            //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)   [BRA.S <label>]
          case 0b0110_000_000:
            irpBrasw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_001_sss_sss
            //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_001_sss_sss [BRA.S <label>]
          case 0b0110_000_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_010_sss_sss
            //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_010_sss_sss [BRA.S <label>]
          case 0b0110_000_010:
            irpBras ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BRA.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)
            //JBRA.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)  [BRA.S <label>]
            //BRA.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_111_111-{offset}
          case 0b0110_000_011:
            irpBrasl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.W <label>                                   |-|012346|-|-----|-----|          |0110_000_100_000_000-{offset}
            //JBSR.W <label>                                  |A|012346|-|-----|-----|          |0110_000_100_000_000-{offset}        [BSR.W <label>]
            //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)
            //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)   [BSR.S <label>]
          case 0b0110_000_100:
            irpBsrsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_101_sss_sss
            //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_101_sss_sss [BSR.S <label>]
          case 0b0110_000_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_110_sss_sss
            //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_110_sss_sss [BSR.S <label>]
          case 0b0110_000_110:
            irpBsrs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BSR.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)
            //JBSR.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)  [BSR.S <label>]
            //BSR.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_111_111-{offset}
          case 0b0110_000_111:
            irpBsrsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}
            //BNLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
            //JBHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
            //JBNLS.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
            //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)
            //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
            //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
            //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
            //JBLS.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
            //JBNHI.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
          case 0b0110_001_000:
            irpBhisw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_001_sss_sss
            //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
            //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
            //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
          case 0b0110_001_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_010_sss_sss
            //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
            //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
            //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
          case 0b0110_001_010:
            irpBhis ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BHI.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)
            //BNLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
            //JBHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
            //JBNLS.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
            //BHI.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}
            //BNLS.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}        [BHI.L <label>]
          case 0b0110_001_011:
            irpBhisl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}
            //BNHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
            //JBLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
            //JBNHI.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
            //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)
            //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
            //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
            //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
            //JBHI.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
            //JBNLS.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
          case 0b0110_001_100:
            irpBlssw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_101_sss_sss
            //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
            //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
            //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
          case 0b0110_001_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_110_sss_sss
            //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
            //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
            //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
          case 0b0110_001_110:
            irpBlss ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLS.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)
            //BNHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
            //JBLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
            //JBNHI.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
            //BLS.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}
            //BNHI.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}        [BLS.L <label>]
          case 0b0110_001_111:
            irpBlssl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.W <label>                                   |-|012346|-|----*|-----|          |0110_010_000_000_000-{offset}
            //BHS.W <label>                                   |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //BNCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //BNLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBNCS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //JBNLO.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
            //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)
            //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
            //JBCS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
            //JBLO.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
            //JBNCC.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
            //JBNHS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
          case 0b0110_010_000:
            irpBhssw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_001_sss_sss
            //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
            //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
          case 0b0110_010_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_010_sss_sss
            //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
            //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
          case 0b0110_010_010:
            irpBhss ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCC.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)
            //BHS.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //BNCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //BNLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBNCS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //JBNLO.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
            //BCC.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}
            //BHS.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
            //BNCS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
            //BNLO.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
          case 0b0110_010_011:
            irpBhssl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.W <label>                                   |-|012346|-|----*|-----|          |0110_010_100_000_000-{offset}
            //BLO.W <label>                                   |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //BNCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //BNHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBNCC.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //JBNHS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
            //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)
            //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
            //JBCC.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
            //JBHS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
            //JBNCS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
            //JBNLO.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
          case 0b0110_010_100:
            irpBlosw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_101_sss_sss
            //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
            //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
          case 0b0110_010_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_110_sss_sss
            //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
            //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
          case 0b0110_010_110:
            irpBlos ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BCS.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)
            //BLO.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //BNCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //BNHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBNCC.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //JBNHS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
            //BCS.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}
            //BLO.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
            //BNCC.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
            //BNHS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
          case 0b0110_010_111:
            irpBlosl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}
            //BNEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //BNZ.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //BNZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNEQ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //JBNZE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
            //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)
            //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
            //JBEQ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNEQ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNNE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNNZ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNZ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBNZE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
            //JBZE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
          case 0b0110_011_000:
            irpBnesw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_001_sss_sss
            //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
            //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
          case 0b0110_011_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_010_sss_sss
            //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
            //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
          case 0b0110_011_010:
            irpBnes ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BNE.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)
            //BNEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //BNZ.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //BNZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNEQ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //JBNZE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
            //BNE.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}
            //BNEQ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
            //BNZ.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
            //BNZE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
          case 0b0110_011_011:
            irpBnesl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}
            //BNNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //BNNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //BZE.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBNNE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBNNZ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //JBZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
            //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)
            //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
            //JBNE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_110-0100111011111001-{address}      [BEQ.S (*)+8;JMP <label>]
          case 0b0110_011_100:
            irpBeqsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_101_sss_sss
            //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
            //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
          case 0b0110_011_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_110_sss_sss
            //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
            //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
          case 0b0110_011_110:
            irpBeqs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BEQ.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)
            //BNNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //BNNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //BZE.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBNNE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBNNZ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //JBZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
            //BEQ.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}
            //BNNE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
            //BNNZ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
            //BZE.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
          case 0b0110_011_111:
            irpBeqsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}
            //BNVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
            //JBNVS.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
            //JBVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
            //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)
            //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
            //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
            //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
            //JBNVC.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
            //JBVS.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
          case 0b0110_100_000:
            irpBvcsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_001_sss_sss
            //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
            //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
            //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
          case 0b0110_100_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_010_sss_sss
            //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
            //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
            //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
          case 0b0110_100_010:
            irpBvcs ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVC.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)
            //BNVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
            //JBNVS.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
            //JBVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
            //BVC.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}
            //BNVS.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}        [BVC.L <label>]
          case 0b0110_100_011:
            irpBvcsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}
            //BNVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
            //JBNVC.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
            //JBVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
            //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)
            //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
            //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
            //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
            //JBNVS.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
            //JBVC.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
          case 0b0110_100_100:
            irpBvssw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_101_sss_sss
            //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
            //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
            //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
          case 0b0110_100_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_110_sss_sss
            //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
            //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
            //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
          case 0b0110_100_110:
            irpBvss ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BVS.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)
            //BNVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
            //JBNVC.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
            //JBVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
            //BVS.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}
            //BNVC.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}        [BVS.L <label>]
          case 0b0110_100_111:
            irpBvssl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}
            //BNMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
            //JBNMI.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
            //JBPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
            //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)
            //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
            //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
            //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
            //JBMI.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
            //JBNPL.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
          case 0b0110_101_000:
            irpBplsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_001_sss_sss
            //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
            //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
            //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
          case 0b0110_101_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_010_sss_sss
            //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
            //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
            //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
          case 0b0110_101_010:
            irpBpls ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BPL.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)
            //BNMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
            //JBNMI.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
            //JBPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
            //BPL.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}
            //BNMI.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}        [BPL.L <label>]
          case 0b0110_101_011:
            irpBplsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}
            //BNPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
            //JBMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
            //JBNPL.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
            //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)
            //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
            //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
            //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
            //JBNMI.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
            //JBPL.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
          case 0b0110_101_100:
            irpBmisw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_101_sss_sss
            //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
            //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
            //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
          case 0b0110_101_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_110_sss_sss
            //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
            //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
            //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
          case 0b0110_101_110:
            irpBmis ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BMI.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)
            //BNPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
            //JBMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
            //JBNPL.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
            //BMI.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}
            //BNPL.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}        [BMI.L <label>]
          case 0b0110_101_111:
            irpBmisl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}
            //BNLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
            //JBGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
            //JBNLT.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
            //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)
            //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
            //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
            //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
            //JBLT.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
            //JBNGE.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
          case 0b0110_110_000:
            irpBgesw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_001_sss_sss
            //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
            //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
            //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
          case 0b0110_110_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_010_sss_sss
            //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
            //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
            //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
          case 0b0110_110_010:
            irpBges ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGE.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)
            //BNLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
            //JBGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
            //JBNLT.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
            //BGE.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}
            //BNLT.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}        [BGE.L <label>]
          case 0b0110_110_011:
            irpBgesl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}
            //BNGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
            //JBLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
            //JBNGE.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
            //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)
            //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
            //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
            //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
            //JBGE.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
            //JBNLT.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
          case 0b0110_110_100:
            irpBltsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_101_sss_sss
            //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
            //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
            //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
          case 0b0110_110_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_110_sss_sss
            //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
            //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
            //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
          case 0b0110_110_110:
            irpBlts ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLT.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)
            //BNGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
            //JBLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
            //JBNGE.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
            //BLT.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}
            //BNGE.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}        [BLT.L <label>]
          case 0b0110_110_111:
            irpBltsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}
            //BNLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
            //JBGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
            //JBNLE.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
            //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)
            //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
            //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
            //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
            //JBLE.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
            //JBNGT.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
          case 0b0110_111_000:
            irpBgtsw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_001_sss_sss
            //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
            //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
            //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
          case 0b0110_111_001:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_010_sss_sss
            //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
            //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
            //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
          case 0b0110_111_010:
            irpBgts ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BGT.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)
            //BNLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
            //JBGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
            //JBNLE.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
            //BGT.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}
            //BNLE.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}        [BGT.L <label>]
          case 0b0110_111_011:
            irpBgtsl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}
            //BNGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
            //JBLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
            //JBNGT.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
            //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)
            //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
            //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
            //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
            //JBGT.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
            //JBNLE.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
          case 0b0110_111_100:
            irpBlesw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_101_sss_sss
            //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
            //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
            //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
          case 0b0110_111_101:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_110_sss_sss
            //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
            //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
            //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
          case 0b0110_111_110:
            irpBles ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BLE.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)
            //BNGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
            //JBLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
            //JBNGT.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
            //BLE.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}
            //BNGT.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}        [BLE.L <label>]
          case 0b0110_111_111:
            irpBlesl ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //IOCS <name>                                     |A|012346|-|UUUUU|UUUUU|          |0111_000_0dd_ddd_ddd-0100111001001111        [MOVEQ.L #<data>,D0;TRAP #15]
            //MOVEQ.L #<data>,Dq                              |-|012346|-|-UUUU|-**00|          |0111_qqq_0dd_ddd_ddd
          case 0b0111_000_000:
          case 0b0111_000_001:
          case 0b0111_000_010:
          case 0b0111_000_011:
          case 0b0111_001_000:
          case 0b0111_001_001:
          case 0b0111_001_010:
          case 0b0111_001_011:
          case 0b0111_010_000:
          case 0b0111_010_001:
          case 0b0111_010_010:
          case 0b0111_010_011:
          case 0b0111_011_000:
          case 0b0111_011_001:
          case 0b0111_011_010:
          case 0b0111_011_011:
          case 0b0111_100_000:
          case 0b0111_100_001:
          case 0b0111_100_010:
          case 0b0111_100_011:
          case 0b0111_101_000:
          case 0b0111_101_001:
          case 0b0111_101_010:
          case 0b0111_101_011:
          case 0b0111_110_000:
          case 0b0111_110_001:
          case 0b0111_110_010:
          case 0b0111_110_011:
          case 0b0111_111_000:
          case 0b0111_111_001:
          case 0b0111_111_010:
          case 0b0111_111_011:
            irpMoveq ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVS.B <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_100_mmm_rrr (ISA_B)
          case 0b0111_000_100:
          case 0b0111_001_100:
          case 0b0111_010_100:
          case 0b0111_011_100:
          case 0b0111_100_100:
          case 0b0111_101_100:
          case 0b0111_110_100:
          case 0b0111_111_100:
            irpMvsByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVS.W <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_101_mmm_rrr (ISA_B)
          case 0b0111_000_101:
          case 0b0111_001_101:
          case 0b0111_010_101:
          case 0b0111_011_101:
          case 0b0111_100_101:
          case 0b0111_101_101:
          case 0b0111_110_101:
          case 0b0111_111_101:
            irpMvsWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVZ.B <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_110_mmm_rrr (ISA_B)
          case 0b0111_000_110:
          case 0b0111_001_110:
          case 0b0111_010_110:
          case 0b0111_011_110:
          case 0b0111_100_110:
          case 0b0111_101_110:
          case 0b0111_110_110:
          case 0b0111_111_110:
            irpMvzByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MVZ.W <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_111_mmm_rrr (ISA_B)
          case 0b0111_000_111:
          case 0b0111_001_111:
          case 0b0111_010_111:
          case 0b0111_011_111:
          case 0b0111_100_111:
          case 0b0111_101_111:
          case 0b0111_110_111:
          case 0b0111_111_111:
            irpMvzWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //OR.B <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_000_mmm_rrr
          case 0b1000_000_000:
          case 0b1000_001_000:
          case 0b1000_010_000:
          case 0b1000_011_000:
          case 0b1000_100_000:
          case 0b1000_101_000:
          case 0b1000_110_000:
          case 0b1000_111_000:
            irpOrToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //OR.W <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_001_mmm_rrr
          case 0b1000_000_001:
          case 0b1000_001_001:
          case 0b1000_010_001:
          case 0b1000_011_001:
          case 0b1000_100_001:
          case 0b1000_101_001:
          case 0b1000_110_001:
          case 0b1000_111_001:
            irpOrToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //OR.L <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_010_mmm_rrr
          case 0b1000_000_010:
          case 0b1000_001_010:
          case 0b1000_010_010:
          case 0b1000_011_010:
          case 0b1000_100_010:
          case 0b1000_101_010:
          case 0b1000_110_010:
          case 0b1000_111_010:
            irpOrToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DIVU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_011_mmm_rrr
          case 0b1000_000_011:
          case 0b1000_001_011:
          case 0b1000_010_011:
          case 0b1000_011_011:
          case 0b1000_100_011:
          case 0b1000_101_011:
          case 0b1000_110_011:
          case 0b1000_111_011:
            irpDivuWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SBCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_000_rrr
            //SBCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_001_rrr
            //OR.B Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_100_mmm_rrr
          case 0b1000_000_100:
          case 0b1000_001_100:
          case 0b1000_010_100:
          case 0b1000_011_100:
          case 0b1000_100_100:
          case 0b1000_101_100:
          case 0b1000_110_100:
          case 0b1000_111_100:
            irpOrToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //PACK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_101_000_rrr-{data}
            //PACK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_101_001_rrr-{data}
            //OR.W Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_101_mmm_rrr
          case 0b1000_000_101:
          case 0b1000_001_101:
          case 0b1000_010_101:
          case 0b1000_011_101:
          case 0b1000_100_101:
          case 0b1000_101_101:
          case 0b1000_110_101:
          case 0b1000_111_101:
            irpOrToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //UNPK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_110_000_rrr-{data}
            //UNPK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_110_001_rrr-{data}
            //OR.L Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_110_mmm_rrr
          case 0b1000_000_110:
          case 0b1000_001_110:
          case 0b1000_010_110:
          case 0b1000_011_110:
          case 0b1000_100_110:
          case 0b1000_101_110:
          case 0b1000_110_110:
          case 0b1000_111_110:
            irpOrToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DIVS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_111_mmm_rrr
          case 0b1000_000_111:
          case 0b1000_001_111:
          case 0b1000_010_111:
          case 0b1000_011_111:
          case 0b1000_100_111:
          case 0b1000_101_111:
          case 0b1000_110_111:
          case 0b1000_111_111:
            irpDivsWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUB.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1001_qqq_000_mmm_rrr
          case 0b1001_000_000:
          case 0b1001_001_000:
          case 0b1001_010_000:
          case 0b1001_011_000:
          case 0b1001_100_000:
          case 0b1001_101_000:
          case 0b1001_110_000:
          case 0b1001_111_000:
            irpSubToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUB.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_001_mmm_rrr
          case 0b1001_000_001:
          case 0b1001_001_001:
          case 0b1001_010_001:
          case 0b1001_011_001:
          case 0b1001_100_001:
          case 0b1001_101_001:
          case 0b1001_110_001:
          case 0b1001_111_001:
            irpSubToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUB.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_010_mmm_rrr
          case 0b1001_000_010:
          case 0b1001_001_010:
          case 0b1001_010_010:
          case 0b1001_011_010:
          case 0b1001_100_010:
          case 0b1001_101_010:
          case 0b1001_110_010:
          case 0b1001_111_010:
            irpSubToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr
            //SUB.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr [SUBA.W <ea>,Aq]
            //CLR.W Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_011_001_rrr [SUBA.W Ar,Ar]
          case 0b1001_000_011:
          case 0b1001_001_011:
          case 0b1001_010_011:
          case 0b1001_011_011:
          case 0b1001_100_011:
          case 0b1001_101_011:
          case 0b1001_110_011:
          case 0b1001_111_011:
            irpSubaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_100_000_rrr
            //SUBX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_100_001_rrr
            //SUB.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_100_mmm_rrr
          case 0b1001_000_100:
          case 0b1001_001_100:
          case 0b1001_010_100:
          case 0b1001_011_100:
          case 0b1001_100_100:
          case 0b1001_101_100:
          case 0b1001_110_100:
          case 0b1001_111_100:
            irpSubToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_101_000_rrr
            //SUBX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_101_001_rrr
            //SUB.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_101_mmm_rrr
          case 0b1001_000_101:
          case 0b1001_001_101:
          case 0b1001_010_101:
          case 0b1001_011_101:
          case 0b1001_100_101:
          case 0b1001_101_101:
          case 0b1001_110_101:
          case 0b1001_111_101:
            irpSubToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_110_000_rrr
            //SUBX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_110_001_rrr
            //SUB.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_110_mmm_rrr
          case 0b1001_000_110:
          case 0b1001_001_110:
          case 0b1001_010_110:
          case 0b1001_011_110:
          case 0b1001_100_110:
          case 0b1001_101_110:
          case 0b1001_110_110:
          case 0b1001_111_110:
            irpSubToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SUBA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr
            //SUB.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr [SUBA.L <ea>,Aq]
            //CLR.L Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_111_001_rrr [SUBA.L Ar,Ar]
          case 0b1001_000_111:
          case 0b1001_001_111:
          case 0b1001_010_111:
          case 0b1001_011_111:
          case 0b1001_100_111:
          case 0b1001_101_111:
          case 0b1001_110_111:
          case 0b1001_111_111:
            irpSubaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //SXCALL <name>                                   |A|012346|-|UUUUU|*****|          |1010_0dd_ddd_ddd_ddd [ALINE #<data>]
          case 0b1010_000_000:
          case 0b1010_000_001:
          case 0b1010_000_010:
          case 0b1010_000_011:
          case 0b1010_000_100:
          case 0b1010_000_101:
          case 0b1010_000_110:
          case 0b1010_000_111:
          case 0b1010_001_000:
          case 0b1010_001_001:
          case 0b1010_001_010:
          case 0b1010_001_011:
          case 0b1010_001_100:
          case 0b1010_001_101:
          case 0b1010_001_110:
          case 0b1010_001_111:
          case 0b1010_010_000:
          case 0b1010_010_001:
          case 0b1010_010_010:
          case 0b1010_010_011:
          case 0b1010_010_100:
          case 0b1010_010_101:
          case 0b1010_010_110:
          case 0b1010_010_111:
          case 0b1010_011_000:
          case 0b1010_011_001:
          case 0b1010_011_010:
          case 0b1010_011_011:
          case 0b1010_011_100:
          case 0b1010_011_101:
          case 0b1010_011_110:
          case 0b1010_011_111:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ALINE #<data>                                   |-|012346|-|UUUUU|*****|          |1010_ddd_ddd_ddd_ddd (line 1010 emulator)
          case 0b1010_100_000:
          case 0b1010_100_001:
          case 0b1010_100_010:
          case 0b1010_100_011:
          case 0b1010_100_100:
          case 0b1010_100_101:
          case 0b1010_100_110:
          case 0b1010_100_111:
          case 0b1010_101_000:
          case 0b1010_101_001:
          case 0b1010_101_010:
          case 0b1010_101_011:
          case 0b1010_101_100:
          case 0b1010_101_101:
          case 0b1010_101_110:
          case 0b1010_101_111:
          case 0b1010_110_000:
          case 0b1010_110_001:
          case 0b1010_110_010:
          case 0b1010_110_011:
          case 0b1010_110_100:
          case 0b1010_110_101:
          case 0b1010_110_110:
          case 0b1010_110_111:
          case 0b1010_111_000:
          case 0b1010_111_001:
          case 0b1010_111_010:
          case 0b1010_111_011:
          case 0b1010_111_100:
          case 0b1010_111_101:
          case 0b1010_111_110:
          case 0b1010_111_111:
            irpAline ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMP.B <ea>,Dq                                   |-|012346|-|-UUUU|-****|D M+-WXZPI|1011_qqq_000_mmm_rrr
          case 0b1011_000_000:
          case 0b1011_001_000:
          case 0b1011_010_000:
          case 0b1011_011_000:
          case 0b1011_100_000:
          case 0b1011_101_000:
          case 0b1011_110_000:
          case 0b1011_111_000:
            irpCmpByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMP.W <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_001_mmm_rrr
          case 0b1011_000_001:
          case 0b1011_001_001:
          case 0b1011_010_001:
          case 0b1011_011_001:
          case 0b1011_100_001:
          case 0b1011_101_001:
          case 0b1011_110_001:
          case 0b1011_111_001:
            irpCmpWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMP.L <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_010_mmm_rrr
          case 0b1011_000_010:
          case 0b1011_001_010:
          case 0b1011_010_010:
          case 0b1011_011_010:
          case 0b1011_100_010:
          case 0b1011_101_010:
          case 0b1011_110_010:
          case 0b1011_111_010:
            irpCmpLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPA.W <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr
            //CMP.W <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr [CMPA.W <ea>,Aq]
          case 0b1011_000_011:
          case 0b1011_001_011:
          case 0b1011_010_011:
          case 0b1011_011_011:
          case 0b1011_100_011:
          case 0b1011_101_011:
          case 0b1011_110_011:
          case 0b1011_111_011:
            irpCmpaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EOR.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_100_mmm_rrr
            //CMPM.B (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_100_001_rrr
          case 0b1011_000_100:
          case 0b1011_001_100:
          case 0b1011_010_100:
          case 0b1011_011_100:
          case 0b1011_100_100:
          case 0b1011_101_100:
          case 0b1011_110_100:
          case 0b1011_111_100:
            irpEorByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EOR.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_101_mmm_rrr
            //CMPM.W (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_101_001_rrr
          case 0b1011_000_101:
          case 0b1011_001_101:
          case 0b1011_010_101:
          case 0b1011_011_101:
          case 0b1011_100_101:
          case 0b1011_101_101:
          case 0b1011_110_101:
          case 0b1011_111_101:
            irpEorWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EOR.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_110_mmm_rrr
            //CMPM.L (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_110_001_rrr
          case 0b1011_000_110:
          case 0b1011_001_110:
          case 0b1011_010_110:
          case 0b1011_011_110:
          case 0b1011_100_110:
          case 0b1011_101_110:
          case 0b1011_110_110:
          case 0b1011_111_110:
            irpEorLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CMPA.L <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr
            //CMP.L <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr [CMPA.L <ea>,Aq]
          case 0b1011_000_111:
          case 0b1011_001_111:
          case 0b1011_010_111:
          case 0b1011_011_111:
          case 0b1011_100_111:
          case 0b1011_101_111:
          case 0b1011_110_111:
          case 0b1011_111_111:
            irpCmpaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //AND.B <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_000_mmm_rrr
          case 0b1100_000_000:
          case 0b1100_001_000:
          case 0b1100_010_000:
          case 0b1100_011_000:
          case 0b1100_100_000:
          case 0b1100_101_000:
          case 0b1100_110_000:
          case 0b1100_111_000:
            irpAndToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //AND.W <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_001_mmm_rrr
          case 0b1100_000_001:
          case 0b1100_001_001:
          case 0b1100_010_001:
          case 0b1100_011_001:
          case 0b1100_100_001:
          case 0b1100_101_001:
          case 0b1100_110_001:
          case 0b1100_111_001:
            irpAndToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //AND.L <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_010_mmm_rrr
          case 0b1100_000_010:
          case 0b1100_001_010:
          case 0b1100_010_010:
          case 0b1100_011_010:
          case 0b1100_100_010:
          case 0b1100_101_010:
          case 0b1100_110_010:
          case 0b1100_111_010:
            irpAndToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MULU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_011_mmm_rrr
          case 0b1100_000_011:
          case 0b1100_001_011:
          case 0b1100_010_011:
          case 0b1100_011_011:
          case 0b1100_100_011:
          case 0b1100_101_011:
          case 0b1100_110_011:
          case 0b1100_111_011:
            irpMuluWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ABCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_000_rrr
            //ABCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_001_rrr
            //AND.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_100_mmm_rrr
          case 0b1100_000_100:
          case 0b1100_001_100:
          case 0b1100_010_100:
          case 0b1100_011_100:
          case 0b1100_100_100:
          case 0b1100_101_100:
          case 0b1100_110_100:
          case 0b1100_111_100:
            irpAndToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXG.L Dq,Dr                                     |-|012346|-|-----|-----|          |1100_qqq_101_000_rrr
            //EXG.L Aq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_101_001_rrr
            //AND.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_101_mmm_rrr
          case 0b1100_000_101:
          case 0b1100_001_101:
          case 0b1100_010_101:
          case 0b1100_011_101:
          case 0b1100_100_101:
          case 0b1100_101_101:
          case 0b1100_110_101:
          case 0b1100_111_101:
            irpAndToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //EXG.L Dq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_110_001_rrr
            //AND.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_110_mmm_rrr
          case 0b1100_000_110:
          case 0b1100_001_110:
          case 0b1100_010_110:
          case 0b1100_011_110:
          case 0b1100_100_110:
          case 0b1100_101_110:
          case 0b1100_110_110:
          case 0b1100_111_110:
            irpAndToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MULS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_111_mmm_rrr
          case 0b1100_000_111:
          case 0b1100_001_111:
          case 0b1100_010_111:
          case 0b1100_011_111:
          case 0b1100_100_111:
          case 0b1100_101_111:
          case 0b1100_110_111:
          case 0b1100_111_111:
            irpMulsWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADD.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1101_qqq_000_mmm_rrr
          case 0b1101_000_000:
          case 0b1101_001_000:
          case 0b1101_010_000:
          case 0b1101_011_000:
          case 0b1101_100_000:
          case 0b1101_101_000:
          case 0b1101_110_000:
          case 0b1101_111_000:
            irpAddToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADD.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_001_mmm_rrr
          case 0b1101_000_001:
          case 0b1101_001_001:
          case 0b1101_010_001:
          case 0b1101_011_001:
          case 0b1101_100_001:
          case 0b1101_101_001:
          case 0b1101_110_001:
          case 0b1101_111_001:
            irpAddToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADD.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_010_mmm_rrr
          case 0b1101_000_010:
          case 0b1101_001_010:
          case 0b1101_010_010:
          case 0b1101_011_010:
          case 0b1101_100_010:
          case 0b1101_101_010:
          case 0b1101_110_010:
          case 0b1101_111_010:
            irpAddToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr
            //ADD.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr [ADDA.W <ea>,Aq]
          case 0b1101_000_011:
          case 0b1101_001_011:
          case 0b1101_010_011:
          case 0b1101_011_011:
          case 0b1101_100_011:
          case 0b1101_101_011:
          case 0b1101_110_011:
          case 0b1101_111_011:
            irpAddaWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_100_000_rrr
            //ADDX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_100_001_rrr
            //ADD.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_100_mmm_rrr
          case 0b1101_000_100:
          case 0b1101_001_100:
          case 0b1101_010_100:
          case 0b1101_011_100:
          case 0b1101_100_100:
          case 0b1101_101_100:
          case 0b1101_110_100:
          case 0b1101_111_100:
            irpAddToMemByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_101_000_rrr
            //ADDX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_101_001_rrr
            //ADD.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_101_mmm_rrr
          case 0b1101_000_101:
          case 0b1101_001_101:
          case 0b1101_010_101:
          case 0b1101_011_101:
          case 0b1101_100_101:
          case 0b1101_101_101:
          case 0b1101_110_101:
          case 0b1101_111_101:
            irpAddToMemWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_110_000_rrr
            //ADDX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_110_001_rrr
            //ADD.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_110_mmm_rrr
          case 0b1101_000_110:
          case 0b1101_001_110:
          case 0b1101_010_110:
          case 0b1101_011_110:
          case 0b1101_100_110:
          case 0b1101_101_110:
          case 0b1101_110_110:
          case 0b1101_111_110:
            irpAddToMemLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ADDA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr
            //ADD.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr [ADDA.L <ea>,Aq]
          case 0b1101_000_111:
          case 0b1101_001_111:
          case 0b1101_010_111:
          case 0b1101_011_111:
          case 0b1101_100_111:
          case 0b1101_101_111:
          case 0b1101_110_111:
          case 0b1101_111_111:
            irpAddaLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_000_rrr
            //LSR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_001_rrr
            //ROXR.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_000_010_rrr
            //ROR.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_011_rrr
            //ASR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_100_rrr
            //LSR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_101_rrr
            //ROXR.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_000_110_rrr
            //ROR.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_111_rrr
            //ASR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_000_rrr [ASR.B #1,Dr]
            //LSR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_001_rrr [LSR.B #1,Dr]
            //ROXR.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_000_010_rrr [ROXR.B #1,Dr]
            //ROR.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_000_011_rrr [ROR.B #1,Dr]
          case 0b1110_000_000:
          case 0b1110_001_000:
          case 0b1110_010_000:
          case 0b1110_011_000:
          case 0b1110_100_000:
          case 0b1110_101_000:
          case 0b1110_110_000:
          case 0b1110_111_000:
            irpXxrToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_000_rrr
            //LSR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_001_rrr
            //ROXR.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_001_010_rrr
            //ROR.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_011_rrr
            //ASR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_100_rrr
            //LSR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_101_rrr
            //ROXR.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_001_110_rrr
            //ROR.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_111_rrr
            //ASR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_000_rrr [ASR.W #1,Dr]
            //LSR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_001_rrr [LSR.W #1,Dr]
            //ROXR.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_001_010_rrr [ROXR.W #1,Dr]
            //ROR.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_001_011_rrr [ROR.W #1,Dr]
          case 0b1110_000_001:
          case 0b1110_001_001:
          case 0b1110_010_001:
          case 0b1110_011_001:
          case 0b1110_100_001:
          case 0b1110_101_001:
          case 0b1110_110_001:
          case 0b1110_111_001:
            irpXxrToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_000_rrr
            //LSR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_001_rrr
            //ROXR.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_010_010_rrr
            //ROR.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_011_rrr
            //ASR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_100_rrr
            //LSR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_101_rrr
            //ROXR.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_010_110_rrr
            //ROR.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_111_rrr
            //ASR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_000_rrr [ASR.L #1,Dr]
            //LSR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_001_rrr [LSR.L #1,Dr]
            //ROXR.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_010_010_rrr [ROXR.L #1,Dr]
            //ROR.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_010_011_rrr [ROR.L #1,Dr]
          case 0b1110_000_010:
          case 0b1110_001_010:
          case 0b1110_010_010:
          case 0b1110_011_010:
          case 0b1110_100_010:
          case 0b1110_101_010:
          case 0b1110_110_010:
          case 0b1110_111_010:
            irpXxrToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASR.W <ea>                                      |-|012346|-|UUUUU|***0*|  M+-WXZ  |1110_000_011_mmm_rrr
          case 0b1110_000_011:
            irpAsrToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.B #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_100_000_rrr
            //LSL.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_100_001_rrr
            //ROXL.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_100_010_rrr
            //ROL.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_100_011_rrr
            //ASL.B Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_100_100_rrr
            //LSL.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_100_101_rrr
            //ROXL.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_100_110_rrr
            //ROL.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_100_111_rrr
            //ASL.B Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_100_000_rrr [ASL.B #1,Dr]
            //LSL.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_100_001_rrr [LSL.B #1,Dr]
            //ROXL.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_100_010_rrr [ROXL.B #1,Dr]
            //ROL.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_100_011_rrr [ROL.B #1,Dr]
          case 0b1110_000_100:
          case 0b1110_001_100:
          case 0b1110_010_100:
          case 0b1110_011_100:
          case 0b1110_100_100:
          case 0b1110_101_100:
          case 0b1110_110_100:
          case 0b1110_111_100:
            irpXxlToRegByte ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.W #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_101_000_rrr
            //LSL.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_101_001_rrr
            //ROXL.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_101_010_rrr
            //ROL.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_101_011_rrr
            //ASL.W Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_101_100_rrr
            //LSL.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_101_101_rrr
            //ROXL.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_101_110_rrr
            //ROL.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_101_111_rrr
            //ASL.W Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_101_000_rrr [ASL.W #1,Dr]
            //LSL.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_101_001_rrr [LSL.W #1,Dr]
            //ROXL.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_101_010_rrr [ROXL.W #1,Dr]
            //ROL.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_101_011_rrr [ROL.W #1,Dr]
          case 0b1110_000_101:
          case 0b1110_001_101:
          case 0b1110_010_101:
          case 0b1110_011_101:
          case 0b1110_100_101:
          case 0b1110_101_101:
          case 0b1110_110_101:
          case 0b1110_111_101:
            irpXxlToRegWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.L #<data>,Dr                                |-|012346|-|UUUUU|*****|          |1110_qqq_110_000_rrr
            //LSL.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_110_001_rrr
            //ROXL.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_110_010_rrr
            //ROL.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_110_011_rrr
            //ASL.L Dq,Dr                                     |-|012346|-|UUUUU|*****|          |1110_qqq_110_100_rrr
            //LSL.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_110_101_rrr
            //ROXL.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_110_110_rrr
            //ROL.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_110_111_rrr
            //ASL.L Dr                                        |A|012346|-|UUUUU|*****|          |1110_001_110_000_rrr [ASL.L #1,Dr]
            //LSL.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_110_001_rrr [LSL.L #1,Dr]
            //ROXL.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_110_010_rrr [ROXL.L #1,Dr]
            //ROL.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_110_011_rrr [ROL.L #1,Dr]
          case 0b1110_000_110:
          case 0b1110_001_110:
          case 0b1110_010_110:
          case 0b1110_011_110:
          case 0b1110_100_110:
          case 0b1110_101_110:
          case 0b1110_110_110:
          case 0b1110_111_110:
            irpXxlToRegLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ASL.W <ea>                                      |-|012346|-|UUUUU|*****|  M+-WXZ  |1110_000_111_mmm_rrr
          case 0b1110_000_111:
            irpAslToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LSR.W <ea>                                      |-|012346|-|UUUUU|*0*0*|  M+-WXZ  |1110_001_011_mmm_rrr
          case 0b1110_001_011:
            irpLsrToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LSL.W <ea>                                      |-|012346|-|UUUUU|***0*|  M+-WXZ  |1110_001_111_mmm_rrr
          case 0b1110_001_111:
            irpLslToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROXR.W <ea>                                     |-|012346|-|*UUUU|***0*|  M+-WXZ  |1110_010_011_mmm_rrr
          case 0b1110_010_011:
            irpRoxrToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROXL.W <ea>                                     |-|012346|-|*UUUU|***0*|  M+-WXZ  |1110_010_111_mmm_rrr
          case 0b1110_010_111:
            irpRoxlToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROR.W <ea>                                      |-|012346|-|-UUUU|-**0*|  M+-WXZ  |1110_011_011_mmm_rrr
          case 0b1110_011_011:
            irpRorToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //ROL.W <ea>                                      |-|012346|-|-UUUU|-**0*|  M+-WXZ  |1110_011_111_mmm_rrr
          case 0b1110_011_111:
            irpRolToMem ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFTST <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-00000ooooo0wwwww
            //BFTST <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-00000ooooo100www
            //BFTST <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-0000100ooo0wwwww
            //BFTST <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_011_mmm_rrr-0000100ooo100www
          case 0b1110_100_011:
            irpBftst ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFEXTU <ea>{#o:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFEXTU <ea>{#o:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn0ooooo100www
            //BFEXTU <ea>{Do:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn100ooo0wwwww
            //BFEXTU <ea>{Do:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_100_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_100_111:
            irpBfextu ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFCHG <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-00000ooooo0wwwww
            //BFCHG <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-00000ooooo100www
            //BFCHG <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-0000100ooo0wwwww
            //BFCHG <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_101_011_mmm_rrr-0000100ooo100www
          case 0b1110_101_011:
            irpBfchg ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFEXTS <ea>{#o:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFEXTS <ea>{#o:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn0ooooo100www
            //BFEXTS <ea>{Do:#w},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn100ooo0wwwww
            //BFEXTS <ea>{Do:Dw},Dn                           |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_101_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_101_111:
            irpBfexts ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFCLR <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-00000ooooo0wwwww
            //BFCLR <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-00000ooooo100www
            //BFCLR <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-0000100ooo0wwwww
            //BFCLR <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_110_011_mmm_rrr-0000100ooo100www
          case 0b1110_110_011:
            irpBfclr ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFFFO <ea>{#o:#w},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFFFO <ea>{#o:Dw},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn0ooooo100www
            //BFFFO <ea>{Do:#w},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn100ooo0wwwww
            //BFFFO <ea>{Do:Dw},Dn                            |-|--2346|-|-UUUU|-**00|D M  WXZP |1110_110_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_110_111:
            irpBfffo ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFSET <ea>{#o:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-00000ooooo0wwwww
            //BFSET <ea>{#o:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-00000ooooo100www
            //BFSET <ea>{Do:#w}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-0000100ooo0wwwww
            //BFSET <ea>{Do:Dw}                               |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_011_mmm_rrr-0000100ooo100www
          case 0b1110_111_011:
            irpBfset ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //BFINS Dn,<ea>{#o:#w}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn0ooooo0wwwww
            //BFINS Dn,<ea>{#o:Dw}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn0ooooo100www
            //BFINS Dn,<ea>{Do:#w}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn100ooo0wwwww
            //BFINS Dn,<ea>{Do:Dw}                            |-|--2346|-|-UUUU|-**00|D M  WXZ  |1110_111_111_mmm_rrr-0nnn100ooo100www
          case 0b1110_111_111:
            irpBfins ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FTST.X FPm                                      |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmm0000111010
            //FMOVE.X FPm,FPn                                 |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000000
            //FINT.X FPm,FPn                                  |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000001
            //FSINH.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000010
            //FINTRZ.X FPm,FPn                                |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000011
            //FSQRT.X FPm,FPn                                 |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000100
            //FLOGNP1.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0000110
            //FETOXM1.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001000
            //FTANH.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001001
            //FATAN.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001010
            //FASIN.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001100
            //FATANH.X FPm,FPn                                |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001101
            //FSIN.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001110
            //FTAN.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0001111
            //FETOX.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010000
            //FTWOTOX.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010001
            //FTENTOX.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010010
            //FLOGN.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010100
            //FLOG10.X FPm,FPn                                |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010101
            //FLOG2.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0010110
            //FABS.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011000
            //FCOSH.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011001
            //FNEG.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011010
            //FACOS.X FPm,FPn                                 |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011100
            //FCOS.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011101
            //FGETEXP.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011110
            //FGETMAN.X FPm,FPn                               |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0011111
            //FDIV.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100000
            //FMOD.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100001
            //FADD.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100010
            //FMUL.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100011
            //FSGLDIV.X FPm,FPn                               |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100100
            //FREM.X FPm,FPn                                  |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100101
            //FSCALE.X FPm,FPn                                |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100110
            //FSGLMUL.X FPm,FPn                               |-|--CCS6|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0100111
            //FSUB.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0101000
            //FCMP.X FPm,FPn                                  |-|--CC46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn0111000
            //FSMOVE.X FPm,FPn                                |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1000000
            //FSSQRT.X FPm,FPn                                |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1000001
            //FDMOVE.X FPm,FPn                                |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1000100
            //FDSQRT.X FPm,FPn                                |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1000101
            //FSABS.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1011000
            //FSNEG.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1011010
            //FDABS.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1011100
            //FDNEG.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1011110
            //FSDIV.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1100000
            //FSADD.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1100010
            //FSMUL.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1100011
            //FDDIV.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1100100
            //FDADD.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1100110
            //FDMUL.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1100111
            //FSSUB.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1101000
            //FDSUB.X FPm,FPn                                 |-|----46|-|-----|-----|          |1111_001_000_000_000-000mmmnnn1101100
            //FSINCOS.X FPm,FPc:FPs                           |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-000mmmsss0110ccc
            //FMOVECR.X #ccc,FPn                              |-|--CCSS|-|-----|-----|          |1111_001_000_000_000-010111nnn0cccccc
            //FMOVE.L FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011000nnn0000000
            //FMOVE.S FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011001nnn0000000
            //FMOVE.W FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011100nnn0000000
            //FMOVE.B FPn,<ea>                                |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-011110nnn0000000
            //FMOVE.L FPIAR,<ea>                              |-|--CC46|-|-----|-----|DAM+-WXZ  |1111_001_000_mmm_rrr-1010010000000000
            //FMOVEM.L FPIAR,<ea>                             |-|--CC46|-|-----|-----|DAM+-WXZ  |1111_001_000_mmm_rrr-1010010000000000
            //FMOVE.L FPSR,<ea>                               |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1010100000000000
            //FMOVEM.L FPSR,<ea>                              |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1010100000000000
            //FMOVE.L FPCR,<ea>                               |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1011000000000000
            //FMOVEM.L FPCR,<ea>                              |-|--CC46|-|-----|-----|D M+-WXZ  |1111_001_000_mmm_rrr-1011000000000000
            //FTST.L <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0100000000111010
            //FMOVE.L <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000000
            //FINT.L <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000001
            //FSINH.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000010
            //FINTRZ.L <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000011
            //FSQRT.L <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000100
            //FLOGNP1.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0000110
            //FETOXM1.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001000
            //FTANH.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001001
            //FATAN.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001010
            //FASIN.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001100
            //FATANH.L <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001101
            //FSIN.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001110
            //FTAN.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0001111
            //FETOX.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010000
            //FTWOTOX.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010001
            //FTENTOX.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010010
            //FLOGN.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010100
            //FLOG10.L <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010101
            //FLOG2.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0010110
            //FABS.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011000
            //FCOSH.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011001
            //FNEG.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011010
            //FACOS.L <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011100
            //FCOS.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011101
            //FGETEXP.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011110
            //FGETMAN.L <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0011111
            //FDIV.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100000
            //FMOD.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100001
            //FADD.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100010
            //FMUL.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100011
            //FSGLDIV.L <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100100
            //FREM.L <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100101
            //FSCALE.L <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100110
            //FSGLMUL.L <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0100111
            //FSUB.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0101000
            //FCMP.L <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn0111000
            //FSMOVE.L <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1000000
            //FSSQRT.L <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1000001
            //FDMOVE.L <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1000100
            //FDSQRT.L <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1000101
            //FSABS.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1011000
            //FSNEG.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1011010
            //FDABS.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1011100
            //FDNEG.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1011110
            //FSDIV.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1100000
            //FSADD.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1100010
            //FSMUL.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1100011
            //FDDIV.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1100100
            //FDADD.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1100110
            //FDMUL.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1100111
            //FSSUB.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1101000
            //FDSUB.L <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000nnn1101100
            //FSINCOS.L <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010000sss0110ccc
            //FTST.S <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0100010000111010
            //FMOVE.S <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000000
            //FINT.S <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000001
            //FSINH.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000010
            //FINTRZ.S <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000011
            //FSQRT.S <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000100
            //FLOGNP1.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0000110
            //FETOXM1.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001000
            //FTANH.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001001
            //FATAN.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001010
            //FASIN.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001100
            //FATANH.S <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001101
            //FSIN.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001110
            //FTAN.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0001111
            //FETOX.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010000
            //FTWOTOX.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010001
            //FTENTOX.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010010
            //FLOGN.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010100
            //FLOG10.S <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010101
            //FLOG2.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0010110
            //FABS.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011000
            //FCOSH.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011001
            //FNEG.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011010
            //FACOS.S <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011100
            //FCOS.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011101
            //FGETEXP.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011110
            //FGETMAN.S <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0011111
            //FDIV.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100000
            //FMOD.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100001
            //FADD.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100010
            //FMUL.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100011
            //FSGLDIV.S <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100100
            //FREM.S <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100101
            //FSCALE.S <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100110
            //FSGLMUL.S <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0100111
            //FSUB.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0101000
            //FCMP.S <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn0111000
            //FSMOVE.S <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1000000
            //FSSQRT.S <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1000001
            //FDMOVE.S <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1000100
            //FDSQRT.S <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1000101
            //FSABS.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1011000
            //FSNEG.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1011010
            //FDABS.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1011100
            //FDNEG.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1011110
            //FSDIV.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1100000
            //FSADD.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1100010
            //FSMUL.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1100011
            //FDDIV.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1100100
            //FDADD.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1100110
            //FDMUL.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1100111
            //FSSUB.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1101000
            //FDSUB.S <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001nnn1101100
            //FSINCOS.S <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010001sss0110ccc
            //FTST.W <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0101000000111010
            //FMOVE.W <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000000
            //FINT.W <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000001
            //FSINH.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000010
            //FINTRZ.W <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000011
            //FSQRT.W <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000100
            //FLOGNP1.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0000110
            //FETOXM1.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001000
            //FTANH.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001001
            //FATAN.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001010
            //FASIN.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001100
            //FATANH.W <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001101
            //FSIN.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001110
            //FTAN.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0001111
            //FETOX.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010000
            //FTWOTOX.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010001
            //FTENTOX.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010010
            //FLOGN.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010100
            //FLOG10.W <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010101
            //FLOG2.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0010110
            //FABS.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011000
            //FCOSH.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011001
            //FNEG.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011010
            //FACOS.W <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011100
            //FCOS.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011101
            //FGETEXP.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011110
            //FGETMAN.W <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0011111
            //FDIV.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100000
            //FMOD.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100001
            //FADD.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100010
            //FMUL.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100011
            //FSGLDIV.W <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100100
            //FREM.W <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100101
            //FSCALE.W <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100110
            //FSGLMUL.W <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0100111
            //FSUB.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0101000
            //FCMP.W <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn0111000
            //FSMOVE.W <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1000000
            //FSSQRT.W <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1000001
            //FDMOVE.W <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1000100
            //FDSQRT.W <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1000101
            //FSABS.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1011000
            //FSNEG.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1011010
            //FDABS.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1011100
            //FDNEG.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1011110
            //FSDIV.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1100000
            //FSADD.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1100010
            //FSMUL.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1100011
            //FDDIV.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1100100
            //FDADD.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1100110
            //FDMUL.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1100111
            //FSSUB.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1101000
            //FDSUB.W <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100nnn1101100
            //FSINCOS.W <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010100sss0110ccc
            //FTST.B <ea>                                     |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-0101100000111010
            //FMOVE.B <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000000
            //FINT.B <ea>,FPn                                 |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000001
            //FSINH.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000010
            //FINTRZ.B <ea>,FPn                               |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000011
            //FSQRT.B <ea>,FPn                                |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000100
            //FLOGNP1.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0000110
            //FETOXM1.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001000
            //FTANH.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001001
            //FATAN.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001010
            //FASIN.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001100
            //FATANH.B <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001101
            //FSIN.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001110
            //FTAN.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0001111
            //FETOX.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010000
            //FTWOTOX.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010001
            //FTENTOX.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010010
            //FLOGN.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010100
            //FLOG10.B <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010101
            //FLOG2.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0010110
            //FABS.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011000
            //FCOSH.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011001
            //FNEG.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011010
            //FACOS.B <ea>,FPn                                |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011100
            //FCOS.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011101
            //FGETEXP.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011110
            //FGETMAN.B <ea>,FPn                              |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0011111
            //FDIV.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100000
            //FMOD.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100001
            //FADD.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100010
            //FMUL.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100011
            //FSGLDIV.B <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100100
            //FREM.B <ea>,FPn                                 |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100101
            //FSCALE.B <ea>,FPn                               |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100110
            //FSGLMUL.B <ea>,FPn                              |-|--CCS6|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0100111
            //FSUB.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0101000
            //FCMP.B <ea>,FPn                                 |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn0111000
            //FSMOVE.B <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1000000
            //FSSQRT.B <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1000001
            //FDMOVE.B <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1000100
            //FDSQRT.B <ea>,FPn                               |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1000101
            //FSABS.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1011000
            //FSNEG.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1011010
            //FDABS.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1011100
            //FDNEG.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1011110
            //FSDIV.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1100000
            //FSADD.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1100010
            //FSMUL.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1100011
            //FDDIV.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1100100
            //FDADD.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1100110
            //FDMUL.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1100111
            //FSSUB.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1101000
            //FDSUB.B <ea>,FPn                                |-|----46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110nnn1101100
            //FSINCOS.B <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-010110sss0110ccc
            //FMOVE.L <ea>,FPIAR                              |-|--CC46|-|-----|-----|DAM+-WXZPI|1111_001_000_mmm_rrr-1000010000000000
            //FMOVEM.L <ea>,FPIAR                             |-|--CC46|-|-----|-----|DAM+-WXZPI|1111_001_000_mmm_rrr-1000010000000000
            //FMOVE.L <ea>,FPSR                               |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1000100000000000
            //FMOVEM.L <ea>,FPSR                              |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1000100000000000
            //FMOVE.L <ea>,FPCR                               |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1001000000000000
            //FMOVEM.L <ea>,FPCR                              |-|--CC46|-|-----|-----|D M+-WXZPI|1111_001_000_mmm_rrr-1001000000000000
            //FMOVE.X FPn,<ea>                                |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011010nnn0000000
            //FMOVE.P FPn,<ea>{#k}                            |-|--CCSS|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011011nnnkkkkkkk
            //FMOVE.D FPn,<ea>                                |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011101nnn0000000
            //FMOVE.P FPn,<ea>{Dk}                            |-|--CCSS|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-011111nnnkkk0000
            //FMOVEM.L FPSR/FPIAR,<ea>                        |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1010110000000000
            //FMOVEM.L FPCR/FPIAR,<ea>                        |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1011010000000000
            //FMOVEM.L FPCR/FPSR,<ea>                         |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1011100000000000
            //FMOVEM.L FPCR/FPSR/FPIAR,<ea>                   |-|--CC46|-|-----|-----|  M+-WXZ  |1111_001_000_mmm_rrr-1011110000000000
            //FMOVEM.X #<data>,<ea>                           |-|--CC46|-|-----|-----|  M  WXZ  |1111_001_000_mmm_rrr-11110000dddddddd
            //FMOVEM.X <list>,<ea>                            |-|--CC46|-|-----|-----|  M  WXZ  |1111_001_000_mmm_rrr-11110000llllllll
            //FMOVEM.X Dl,<ea>                                |-|--CC4S|-|-----|-----|  M  WXZ  |1111_001_000_mmm_rrr-111110000lll0000
            //FMOVEM.L <ea>,FPSR/FPIAR                        |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1000110000000000
            //FMOVEM.L <ea>,FPCR/FPIAR                        |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1001010000000000
            //FMOVEM.L <ea>,FPCR/FPSR                         |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1001100000000000
            //FMOVEM.L <ea>,FPCR/FPSR/FPIAR                   |-|--CC46|-|-----|-----|  M+-WXZP |1111_001_000_mmm_rrr-1001110000000000
            //FMOVEM.X <ea>,#<data>                           |-|--CC46|-|-----|-----|  M+ WXZP |1111_001_000_mmm_rrr-11010000dddddddd
            //FMOVEM.X <ea>,<list>                            |-|--CC46|-|-----|-----|  M+ WXZP |1111_001_000_mmm_rrr-11010000llllllll
            //FMOVEM.X <ea>,Dl                                |-|--CC4S|-|-----|-----|  M+ WXZP |1111_001_000_mmm_rrr-110110000lll0000
            //FTST.X <ea>                                     |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-0100100000111010
            //FMOVE.X <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000000
            //FINT.X <ea>,FPn                                 |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000001
            //FSINH.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000010
            //FINTRZ.X <ea>,FPn                               |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000011
            //FSQRT.X <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000100
            //FLOGNP1.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0000110
            //FETOXM1.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001000
            //FTANH.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001001
            //FATAN.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001010
            //FASIN.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001100
            //FATANH.X <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001101
            //FSIN.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001110
            //FTAN.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0001111
            //FETOX.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010000
            //FTWOTOX.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010001
            //FTENTOX.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010010
            //FLOGN.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010100
            //FLOG10.X <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010101
            //FLOG2.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0010110
            //FABS.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011000
            //FCOSH.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011001
            //FNEG.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011010
            //FACOS.X <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011100
            //FCOS.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011101
            //FGETEXP.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011110
            //FGETMAN.X <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0011111
            //FDIV.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100000
            //FMOD.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100001
            //FADD.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100010
            //FMUL.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100011
            //FSGLDIV.X <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100100
            //FREM.X <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100101
            //FSCALE.X <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100110
            //FSGLMUL.X <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0100111
            //FSUB.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0101000
            //FCMP.X <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn0111000
            //FSMOVE.X <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1000000
            //FSSQRT.X <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1000001
            //FDMOVE.X <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1000100
            //FDSQRT.X <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1000101
            //FSABS.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1011000
            //FSNEG.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1011010
            //FDABS.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1011100
            //FDNEG.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1011110
            //FSDIV.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1100000
            //FSADD.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1100010
            //FSMUL.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1100011
            //FDDIV.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1100100
            //FDADD.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1100110
            //FDMUL.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1100111
            //FSSUB.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1101000
            //FDSUB.X <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010nnn1101100
            //FSINCOS.X <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010010sss0110ccc
            //FTST.P <ea>                                     |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-0100110000111010
            //FMOVE.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000000
            //FINT.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000001
            //FSINH.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000010
            //FINTRZ.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000011
            //FSQRT.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000100
            //FLOGNP1.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0000110
            //FETOXM1.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001000
            //FTANH.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001001
            //FATAN.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001010
            //FASIN.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001100
            //FATANH.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001101
            //FSIN.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001110
            //FTAN.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0001111
            //FETOX.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010000
            //FTWOTOX.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010001
            //FTENTOX.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010010
            //FLOGN.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010100
            //FLOG10.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010101
            //FLOG2.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0010110
            //FABS.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011000
            //FCOSH.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011001
            //FNEG.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011010
            //FACOS.P <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011100
            //FCOS.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011101
            //FGETEXP.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011110
            //FGETMAN.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0011111
            //FDIV.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100000
            //FMOD.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100001
            //FADD.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100010
            //FMUL.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100011
            //FSGLDIV.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100100
            //FREM.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100101
            //FSCALE.P <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100110
            //FSGLMUL.P <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0100111
            //FSUB.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0101000
            //FCMP.P <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn0111000
            //FSMOVE.P <ea>,FPn                               |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1000000
            //FSSQRT.P <ea>,FPn                               |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1000001
            //FDMOVE.P <ea>,FPn                               |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1000100
            //FDSQRT.P <ea>,FPn                               |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1000101
            //FSABS.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1011000
            //FSNEG.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1011010
            //FDABS.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1011100
            //FDNEG.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1011110
            //FSDIV.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1100000
            //FSADD.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1100010
            //FSMUL.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1100011
            //FDDIV.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1100100
            //FDADD.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1100110
            //FDMUL.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1100111
            //FSSUB.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1101000
            //FDSUB.P <ea>,FPn                                |-|----SS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011nnn1101100
            //FSINCOS.P <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010011sss0110ccc
            //FTST.D <ea>                                     |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-0101010000111010
            //FMOVE.D <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000000
            //FINT.D <ea>,FPn                                 |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000001
            //FSINH.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000010
            //FINTRZ.D <ea>,FPn                               |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000011
            //FSQRT.D <ea>,FPn                                |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000100
            //FLOGNP1.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0000110
            //FETOXM1.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001000
            //FTANH.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001001
            //FATAN.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001010
            //FASIN.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001100
            //FATANH.D <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001101
            //FSIN.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001110
            //FTAN.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0001111
            //FETOX.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010000
            //FTWOTOX.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010001
            //FTENTOX.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010010
            //FLOGN.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010100
            //FLOG10.D <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010101
            //FLOG2.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0010110
            //FABS.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011000
            //FCOSH.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011001
            //FNEG.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011010
            //FACOS.D <ea>,FPn                                |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011100
            //FCOS.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011101
            //FGETEXP.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011110
            //FGETMAN.D <ea>,FPn                              |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0011111
            //FDIV.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100000
            //FMOD.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100001
            //FADD.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100010
            //FMUL.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100011
            //FSGLDIV.D <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100100
            //FREM.D <ea>,FPn                                 |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100101
            //FSCALE.D <ea>,FPn                               |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100110
            //FSGLMUL.D <ea>,FPn                              |-|--CCS6|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0100111
            //FSUB.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0101000
            //FCMP.D <ea>,FPn                                 |-|--CC46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn0111000
            //FSMOVE.D <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1000000
            //FSSQRT.D <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1000001
            //FDMOVE.D <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1000100
            //FDSQRT.D <ea>,FPn                               |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1000101
            //FSABS.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1011000
            //FSNEG.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1011010
            //FDABS.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1011100
            //FDNEG.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1011110
            //FSDIV.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1100000
            //FSADD.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1100010
            //FSMUL.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1100011
            //FDDIV.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1100100
            //FDADD.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1100110
            //FDMUL.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1100111
            //FSSUB.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1101000
            //FDSUB.D <ea>,FPn                                |-|----46|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101nnn1101100
            //FSINCOS.D <ea>,FPc:FPs                          |-|--CCSS|-|-----|-----|  M+-WXZPI|1111_001_000_mmm_rrr-010101sss0110ccc
            //FMOVEM.X #<data>,-(Ar)                          |-|--CC46|-|-----|-----|    -     |1111_001_000_100_rrr-11100000dddddddd
            //FMOVEM.X <list>,-(Ar)                           |-|--CC46|-|-----|-----|    -     |1111_001_000_100_rrr-11100000llllllll
            //FMOVEM.X Dl,-(Ar)                               |-|--CC4S|-|-----|-----|    -     |1111_001_000_100_rrr-111010000lll0000
            //FMOVEM.L #<data>,#<data>,FPSR/FPIAR             |-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1000110000000000-{data}
            //FMOVEM.L #<data>,#<data>,FPCR/FPIAR             |-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1001010000000000-{data}
            //FMOVEM.L #<data>,#<data>,FPCR/FPSR              |-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1001100000000000-{data}
            //FMOVEM.L #<data>,#<data>,#<data>,FPCR/FPSR/FPIAR|-|--CC4S|-|-----|-----|         I|1111_001_000_111_100-1001110000000000-{data}
          case 0b1111_001_000:
            irpFgen ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FSF.B <ea>                                      |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000000
            //FSEQ.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000001
            //FSOGT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000010
            //FSOGE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000011
            //FSOLT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000100
            //FSOLE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000101
            //FSOGL.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000110
            //FSOR.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000000111
            //FSUN.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001000
            //FSUEQ.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001001
            //FSUGT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001010
            //FSUGE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001011
            //FSULT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001100
            //FSULE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001101
            //FSNE.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001110
            //FST.B <ea>                                      |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000001111
            //FSSF.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010000
            //FSSEQ.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010001
            //FSGT.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010010
            //FSGE.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010011
            //FSLT.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010100
            //FSLE.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010101
            //FSGL.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010110
            //FSGLE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000010111
            //FSNGLE.B <ea>                                   |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011000
            //FSNGL.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011001
            //FSNLE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011010
            //FSNLT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011011
            //FSNGE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011100
            //FSNGT.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011101
            //FSSNE.B <ea>                                    |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011110
            //FSST.B <ea>                                     |-|--CC4S|-|-----|-----|D M+-WXZ  |1111_001_001_mmm_rrr-0000000000011111
            //FDBF Dr,<label>                                 |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000000-{offset}
            //FDBRA Dr,<label>                                |A|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000000-{offset}       [FDBF Dr,<label>]
            //FDBEQ Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000001-{offset}
            //FDBOGT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000010-{offset}
            //FDBOGE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000011-{offset}
            //FDBOLT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000100-{offset}
            //FDBOLE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000101-{offset}
            //FDBOGL Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000110-{offset}
            //FDBOR Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000000111-{offset}
            //FDBUN Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001000-{offset}
            //FDBUEQ Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001001-{offset}
            //FDBUGT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001010-{offset}
            //FDBUGE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001011-{offset}
            //FDBULT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001100-{offset}
            //FDBULE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001101-{offset}
            //FDBNE Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001110-{offset}
            //FDBT Dr,<label>                                 |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000001111-{offset}
            //FDBSF Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010000-{offset}
            //FDBSEQ Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010001-{offset}
            //FDBGT Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010010-{offset}
            //FDBGE Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010011-{offset}
            //FDBLT Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010100-{offset}
            //FDBLE Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010101-{offset}
            //FDBGL Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010110-{offset}
            //FDBGLE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000010111-{offset}
            //FDBNGLE Dr,<label>                              |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011000-{offset}
            //FDBNGL Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011001-{offset}
            //FDBNLE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011010-{offset}
            //FDBNLT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011011-{offset}
            //FDBNGE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011100-{offset}
            //FDBNGT Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011101-{offset}
            //FDBSNE Dr,<label>                               |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011110-{offset}
            //FDBST Dr,<label>                                |-|--CC4S|-|-----|-----|          |1111_001_001_001_rrr-0000000000011111-{offset}
            //FTRAPF.W #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000000-{data}
            //FTRAPEQ.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000001-{data}
            //FTRAPOGT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000010-{data}
            //FTRAPOGE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000011-{data}
            //FTRAPOLT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000100-{data}
            //FTRAPOLE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000101-{data}
            //FTRAPOGL.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000110-{data}
            //FTRAPOR.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000000111-{data}
            //FTRAPUN.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001000-{data}
            //FTRAPUEQ.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001001-{data}
            //FTRAPUGT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001010-{data}
            //FTRAPUGE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001011-{data}
            //FTRAPULT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001100-{data}
            //FTRAPULE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001101-{data}
            //FTRAPNE.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001110-{data}
            //FTRAPT.W #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000001111-{data}
            //FTRAPSF.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010000-{data}
            //FTRAPSEQ.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010001-{data}
            //FTRAPGT.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010010-{data}
            //FTRAPGE.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010011-{data}
            //FTRAPLT.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010100-{data}
            //FTRAPLE.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010101-{data}
            //FTRAPGL.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010110-{data}
            //FTRAPGLE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000010111-{data}
            //FTRAPNGLE.W #<data>                             |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011000-{data}
            //FTRAPNGL.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011001-{data}
            //FTRAPNLE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011010-{data}
            //FTRAPNLT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011011-{data}
            //FTRAPNGE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011100-{data}
            //FTRAPNGT.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011101-{data}
            //FTRAPSNE.W #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011110-{data}
            //FTRAPST.W #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_010-0000000000011111-{data}
            //FTRAPF.L #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000000-{data}
            //FTRAPEQ.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000001-{data}
            //FTRAPOGT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000010-{data}
            //FTRAPOGE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000011-{data}
            //FTRAPOLT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000100-{data}
            //FTRAPOLE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000101-{data}
            //FTRAPOGL.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000110-{data}
            //FTRAPOR.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000000111-{data}
            //FTRAPUN.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001000-{data}
            //FTRAPUEQ.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001001-{data}
            //FTRAPUGT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001010-{data}
            //FTRAPUGE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001011-{data}
            //FTRAPULT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001100-{data}
            //FTRAPULE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001101-{data}
            //FTRAPNE.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001110-{data}
            //FTRAPT.L #<data>                                |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000001111-{data}
            //FTRAPSF.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010000-{data}
            //FTRAPSEQ.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010001-{data}
            //FTRAPGT.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010010-{data}
            //FTRAPGE.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010011-{data}
            //FTRAPLT.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010100-{data}
            //FTRAPLE.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010101-{data}
            //FTRAPGL.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010110-{data}
            //FTRAPGLE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000010111-{data}
            //FTRAPNGLE.L #<data>                             |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011000-{data}
            //FTRAPNGL.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011001-{data}
            //FTRAPNLE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011010-{data}
            //FTRAPNLT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011011-{data}
            //FTRAPNGE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011100-{data}
            //FTRAPNGT.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011101-{data}
            //FTRAPSNE.L #<data>                              |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011110-{data}
            //FTRAPST.L #<data>                               |-|--CC4S|-|-----|-----|          |1111_001_001_111_011-0000000000011111-{data}
            //FTRAPF                                          |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000000
            //FTRAPEQ                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000001
            //FTRAPOGT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000010
            //FTRAPOGE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000011
            //FTRAPOLT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000100
            //FTRAPOLE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000101
            //FTRAPOGL                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000110
            //FTRAPOR                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000000111
            //FTRAPUN                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001000
            //FTRAPUEQ                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001001
            //FTRAPUGT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001010
            //FTRAPUGE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001011
            //FTRAPULT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001100
            //FTRAPULE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001101
            //FTRAPNE                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001110
            //FTRAPT                                          |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000001111
            //FTRAPSF                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010000
            //FTRAPSEQ                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010001
            //FTRAPGT                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010010
            //FTRAPGE                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010011
            //FTRAPLT                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010100
            //FTRAPLE                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010101
            //FTRAPGL                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010110
            //FTRAPGLE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000010111
            //FTRAPNGLE                                       |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011000
            //FTRAPNGL                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011001
            //FTRAPNLE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011010
            //FTRAPNLT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011011
            //FTRAPNGE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011100
            //FTRAPNGT                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011101
            //FTRAPSNE                                        |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011110
            //FTRAPST                                         |-|--CC4S|-|-----|-----|          |1111_001_001_111_100-0000000000011111
          case 0b1111_001_001:
            irpFscc ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FNOP                                            |A|--CC46|-|-----|-----|          |1111_001_010_000_000-0000000000000000        [FBF.W (*)+2]
            //FBF.W <label>                                   |-|--CC46|-|-----|-----|          |1111_001_010_000_000-{offset}
            //FBEQ.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_000_001-{offset}
            //FBOGT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_010-{offset}
            //FBOGE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_011-{offset}
            //FBOLT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_100-{offset}
            //FBOLE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_101-{offset}
            //FBOGL.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_000_110-{offset}
            //FBOR.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_000_111-{offset}
            //FBUN.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_001_000-{offset}
            //FBUEQ.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_001-{offset}
            //FBUGT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_010-{offset}
            //FBUGE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_011-{offset}
            //FBULT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_100-{offset}
            //FBULE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_001_101-{offset}
            //FBNE.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_001_110-{offset}
            //FBT.W <label>                                   |-|--CC46|-|-----|-----|          |1111_001_010_001_111-{offset}
            //FBRA.W <label>                                  |A|--CC46|-|-----|-----|          |1111_001_010_001_111-{offset}        [FBT.W <label>]
            //FBSF.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_000-{offset}
            //FBSEQ.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_010_001-{offset}
            //FBGT.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_010-{offset}
            //FBGE.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_011-{offset}
            //FBLT.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_100-{offset}
            //FBLE.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_101-{offset}
            //FBGL.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_010_110-{offset}
            //FBGLE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_010_111-{offset}
            //FBNGLE.W <label>                                |-|--CC46|-|-----|-----|          |1111_001_010_011_000-{offset}
            //FBNGL.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_001-{offset}
            //FBNLE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_010-{offset}
            //FBNLT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_011-{offset}
            //FBNGE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_100-{offset}
            //FBNGT.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_101-{offset}
            //FBSNE.W <label>                                 |-|--CC46|-|-----|-----|          |1111_001_010_011_110-{offset}
            //FBST.W <label>                                  |-|--CC46|-|-----|-----|          |1111_001_010_011_111-{offset}
          case 0b1111_001_010:
            irpFbccWord ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FBF.L <label>                                   |-|--CC46|-|-----|-----|          |1111_001_011_000_000-{offset}
            //FBEQ.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_000_001-{offset}
            //FBOGT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_010-{offset}
            //FBOGE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_011-{offset}
            //FBOLT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_100-{offset}
            //FBOLE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_101-{offset}
            //FBOGL.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_000_110-{offset}
            //FBOR.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_000_111-{offset}
            //FBUN.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_001_000-{offset}
            //FBUEQ.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_001-{offset}
            //FBUGT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_010-{offset}
            //FBUGE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_011-{offset}
            //FBULT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_100-{offset}
            //FBULE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_001_101-{offset}
            //FBNE.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_001_110-{offset}
            //FBT.L <label>                                   |-|--CC46|-|-----|-----|          |1111_001_011_001_111-{offset}
            //FBRA.L <label>                                  |A|--CC46|-|-----|-----|          |1111_001_011_001_111-{offset}        [FBT.L <label>]
            //FBSF.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_000-{offset}
            //FBSEQ.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_010_001-{offset}
            //FBGT.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_010-{offset}
            //FBGE.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_011-{offset}
            //FBLT.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_100-{offset}
            //FBLE.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_101-{offset}
            //FBGL.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_010_110-{offset}
            //FBGLE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_010_111-{offset}
            //FBNGLE.L <label>                                |-|--CC46|-|-----|-----|          |1111_001_011_011_000-{offset}
            //FBNGL.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_001-{offset}
            //FBNLE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_010-{offset}
            //FBNLT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_011-{offset}
            //FBNGE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_100-{offset}
            //FBNGT.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_101-{offset}
            //FBSNE.L <label>                                 |-|--CC46|-|-----|-----|          |1111_001_011_011_110-{offset}
            //FBST.L <label>                                  |-|--CC46|-|-----|-----|          |1111_001_011_011_111-{offset}
          case 0b1111_001_011:
            irpFbccLong ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FSAVE <ea>                                      |-|--CC46|P|-----|-----|  M -WXZ  |1111_001_100_mmm_rrr
          case 0b1111_001_100:
            irpFsave ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FRESTORE <ea>                                   |-|--CC46|P|-----|-----|  M+ WXZP |1111_001_101_mmm_rrr
          case 0b1111_001_101:
            irpFrestore ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CINVL NC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_000_001_rrr
            //CINVP NC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_000_010_rrr
            //CINVA NC                                        |-|----46|P|-----|-----|          |1111_010_000_011_000
            //CPUSHL NC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_000_101_rrr
            //CPUSHP NC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_000_110_rrr
            //CPUSHA NC                                       |-|----46|P|-----|-----|          |1111_010_000_111_000
          case 0b1111_010_000:
            irpCinvCpushNC ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CINVL DC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_001_001_rrr
            //CINVP DC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_001_010_rrr
            //CINVA DC                                        |-|----46|P|-----|-----|          |1111_010_001_011_000
            //CPUSHL DC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_001_101_rrr
            //CPUSHP DC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_001_110_rrr
            //CPUSHA DC                                       |-|----46|P|-----|-----|          |1111_010_001_111_000
          case 0b1111_010_001:
            irpCinvCpushDC ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CINVL IC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_010_001_rrr
            //CINVP IC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_010_010_rrr
            //CINVA IC                                        |-|----46|P|-----|-----|          |1111_010_010_011_000
            //CPUSHL IC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_010_101_rrr
            //CPUSHP IC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_010_110_rrr
            //CPUSHA IC                                       |-|----46|P|-----|-----|          |1111_010_010_111_000
          case 0b1111_010_010:
            irpCinvCpushIC ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //CINVL BC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_011_001_rrr
            //CINVP BC,(Ar)                                   |-|----46|P|-----|-----|          |1111_010_011_010_rrr
            //CINVA BC                                        |-|----46|P|-----|-----|          |1111_010_011_011_000
            //CPUSHL BC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_011_101_rrr
            //CPUSHP BC,(Ar)                                  |-|----46|P|-----|-----|          |1111_010_011_110_rrr
            //CPUSHA BC                                       |-|----46|P|-----|-----|          |1111_010_011_111_000
          case 0b1111_010_011:
            irpCinvCpushBC ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //PFLUSHN (Ar)                                    |-|----46|P|-----|-----|          |1111_010_100_000_rrr
            //PFLUSH (Ar)                                     |-|----46|P|-----|-----|          |1111_010_100_001_rrr
            //PFLUSHAN                                        |-|----46|P|-----|-----|          |1111_010_100_010_000
            //PFLUSHA                                         |-|----46|P|-----|-----|          |1111_010_100_011_000
          case 0b1111_010_100:
            irpPflush ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //PLPAW (Ar)                                      |-|-----6|P|-----|-----|          |1111_010_110_001_rrr
          case 0b1111_010_110:
            irpPlpaw ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //PLPAR (Ar)                                      |-|-----6|P|-----|-----|          |1111_010_111_001_rrr
          case 0b1111_010_111:
            irpPlpar ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //MOVE16 (Ar)+,xxx.L                              |-|----46|-|-----|-----|          |1111_011_000_000_rrr-{address}
            //MOVE16 xxx.L,(Ar)+                              |-|----46|-|-----|-----|          |1111_011_000_001_rrr-{address}
            //MOVE16 (Ar),xxx.L                               |-|----46|-|-----|-----|          |1111_011_000_010_rrr-{address}
            //MOVE16 xxx.L,(Ar)                               |-|----46|-|-----|-----|          |1111_011_000_011_rrr-{address}
            //MOVE16 (Ar)+,(An)+                              |-|----46|-|-----|-----|          |1111_011_000_100_rrr-1nnn000000000000
          case 0b1111_011_000:
            irpMove16 ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //LPSTOP.W #<data>                                |-|-----6|P|-----|-----|          |1111_100_000_000_000-0000000111000000-{data}
          case 0b1111_100_000:
            irpLpstop ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FPACK <data>                                    |A|012346|-|UUUUU|*****|          |1111_111_0dd_ddd_ddd [FLINE #<data>]
          case 0b1111_111_000:
          case 0b1111_111_001:
          case 0b1111_111_010:
          case 0b1111_111_011:
            irpFpack ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //DOS <data>                                      |A|012346|-|UUUUU|UUUUU|          |1111_111_1dd_ddd_ddd [FLINE #<data>]
          case 0b1111_111_100:
          case 0b1111_111_101:
          case 0b1111_111_110:
          case 0b1111_111_111:
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //FLINE #<data>                                   |-|012346|-|UUUUU|UUUUU|          |1111_ddd_ddd_ddd_ddd (line 1111 emulator)
          case 0b1111_000_000:
          case 0b1111_000_001:
          case 0b1111_000_010:
          case 0b1111_000_011:
          case 0b1111_000_100:
          case 0b1111_000_101:
          case 0b1111_000_110:
          case 0b1111_000_111:
          case 0b1111_001_110:
          case 0b1111_001_111:
          case 0b1111_010_101:
          case 0b1111_011_001:
          case 0b1111_011_010:
          case 0b1111_011_011:
          case 0b1111_011_100:
          case 0b1111_011_101:
          case 0b1111_011_110:
          case 0b1111_011_111:
          case 0b1111_100_001:
          case 0b1111_100_010:
          case 0b1111_100_011:
          case 0b1111_100_100:
          case 0b1111_100_101:
          case 0b1111_100_110:
          case 0b1111_100_111:
          case 0b1111_101_000:
          case 0b1111_101_001:
          case 0b1111_101_010:
          case 0b1111_101_011:
          case 0b1111_101_100:
          case 0b1111_101_101:
          case 0b1111_101_110:
          case 0b1111_101_111:
          case 0b1111_110_000:
          case 0b1111_110_001:
          case 0b1111_110_010:
          case 0b1111_110_011:
          case 0b1111_110_100:
          case 0b1111_110_101:
          case 0b1111_110_110:
          case 0b1111_110_111:
            irpFline ();
            break irpSwitch;

            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
            //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
            //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
            //HFSBOOT                                         |-|012346|-|-----|-----|          |0100_111_000_000_000
            //HFSINST                                         |-|012346|-|-----|-----|          |0100_111_000_000_001
            //HFSSTR                                          |-|012346|-|-----|-----|          |0100_111_000_000_010
            //HFSINT                                          |-|012346|-|-----|-----|          |0100_111_000_000_011
            //EMXNOP                                          |-|012346|-|-----|-----|          |0100_111_000_000_100
          case 0b0100_111_000:
            irpEmx ();
            break;

          default:
            irpIllegal ();

          }  //switch XEiJ.regOC >>> 6

          //トレース例外
          //  命令実行前にsrのTビットがセットされていたとき命令実行後にトレース例外が発生する
          //  トレース例外の発動は命令の機能拡張であり、他の例外処理で命令が中断されたときはトレース例外は発生しない
          //  命令例外はトレース例外の前に、割り込み例外はトレース例外の後に処理される
          //  未実装命令のエミュレーションルーチンはrteの直前にsrのTビットを復元することで未実装命令が1個の命令としてトレースされたように見せる
          //    ;DOSコールの終了
          //    ~008616:
          //            btst.b  #$07,(sp)
          //            bne.s   ~00861E
          //            rte
          //    ~00861E:
          //            ori.w   #$8000,sr
          //            rte
          if (XEiJ.mpuTraceFlag != 0) {  //命令実行前にsrのTビットがセットされていた
            irpExceptionFormat2 (M68kException.M6E_TRACE << 2, XEiJ.regPC, XEiJ.regPC0);  //pcは次の命令
          }
          //クロックをカウントアップする
          //  オペランドをアクセスした時点ではまだXEiJ.mpuClockTimeが更新されていないのでXEiJ.mpuClockTime<xxxClock
          //  xxxTickを呼び出すときはXEiJ.mpuClockTime>=xxxClock
          XEiJ.mpuClockTime += XEiJ.mpuModifiedUnit * XEiJ.mpuCycleCount;
          //デバイスを呼び出す
          TickerQueue.tkqRun (XEiJ.mpuClockTime);
          //割り込みを受け付ける
          if ((t = XEiJ.mpuIMR & XEiJ.mpuIRR) != 0) {  //マスクされているレベルよりも高くて受け付けていない割り込みがあるとき
            if (XEiJ.MPU_INTERRUPT_SWITCH) {
              switch (t) {
              case 0b00000001:
              case 0b00000011:
              case 0b00000101:
              case 0b00000111:
              case 0b00001001:
              case 0b00001011:
              case 0b00001101:
              case 0b00001111:
              case 0b00010001:
              case 0b00010011:
              case 0b00010101:
              case 0b00010111:
              case 0b00011001:
              case 0b00011011:
              case 0b00011101:
              case 0b00011111:
              case 0b00100001:
              case 0b00100011:
              case 0b00100101:
              case 0b00100111:
              case 0b00101001:
              case 0b00101011:
              case 0b00101101:
              case 0b00101111:
              case 0b00110001:
              case 0b00110011:
              case 0b00110101:
              case 0b00110111:
              case 0b00111001:
              case 0b00111011:
              case 0b00111101:
              case 0b00111111:
              case 0b01000001:
              case 0b01000011:
              case 0b01000101:
              case 0b01000111:
              case 0b01001001:
              case 0b01001011:
              case 0b01001101:
              case 0b01001111:
              case 0b01010001:
              case 0b01010011:
              case 0b01010101:
              case 0b01010111:
              case 0b01011001:
              case 0b01011011:
              case 0b01011101:
              case 0b01011111:
              case 0b01100001:
              case 0b01100011:
              case 0b01100101:
              case 0b01100111:
              case 0b01101001:
              case 0b01101011:
              case 0b01101101:
              case 0b01101111:
              case 0b01110001:
              case 0b01110011:
              case 0b01110101:
              case 0b01110111:
              case 0b01111001:
              case 0b01111011:
              case 0b01111101:
              case 0b01111111:
              case 0b10000001:
              case 0b10000011:
              case 0b10000101:
              case 0b10000111:
              case 0b10001001:
              case 0b10001011:
              case 0b10001101:
              case 0b10001111:
              case 0b10010001:
              case 0b10010011:
              case 0b10010101:
              case 0b10010111:
              case 0b10011001:
              case 0b10011011:
              case 0b10011101:
              case 0b10011111:
              case 0b10100001:
              case 0b10100011:
              case 0b10100101:
              case 0b10100111:
              case 0b10101001:
              case 0b10101011:
              case 0b10101101:
              case 0b10101111:
              case 0b10110001:
              case 0b10110011:
              case 0b10110101:
              case 0b10110111:
              case 0b10111001:
              case 0b10111011:
              case 0b10111101:
              case 0b10111111:
              case 0b11000001:
              case 0b11000011:
              case 0b11000101:
              case 0b11000111:
              case 0b11001001:
              case 0b11001011:
              case 0b11001101:
              case 0b11001111:
              case 0b11010001:
              case 0b11010011:
              case 0b11010101:
              case 0b11010111:
              case 0b11011001:
              case 0b11011011:
              case 0b11011101:
              case 0b11011111:
              case 0b11100001:
              case 0b11100011:
              case 0b11100101:
              case 0b11100111:
              case 0b11101001:
              case 0b11101011:
              case 0b11101101:
              case 0b11101111:
              case 0b11110001:
              case 0b11110011:
              case 0b11110101:
              case 0b11110111:
              case 0b11111001:
              case 0b11111011:
              case 0b11111101:
              case 0b11111111:
                //レベル7
                XEiJ.mpuIRR &= ~XEiJ.MPU_SYS_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.sysAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_SYS_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00000010:
              case 0b00000110:
              case 0b00001010:
              case 0b00001110:
              case 0b00010010:
              case 0b00010110:
              case 0b00011010:
              case 0b00011110:
              case 0b00100010:
              case 0b00100110:
              case 0b00101010:
              case 0b00101110:
              case 0b00110010:
              case 0b00110110:
              case 0b00111010:
              case 0b00111110:
              case 0b01000010:
              case 0b01000110:
              case 0b01001010:
              case 0b01001110:
              case 0b01010010:
              case 0b01010110:
              case 0b01011010:
              case 0b01011110:
              case 0b01100010:
              case 0b01100110:
              case 0b01101010:
              case 0b01101110:
              case 0b01110010:
              case 0b01110110:
              case 0b01111010:
              case 0b01111110:
              case 0b10000010:
              case 0b10000110:
              case 0b10001010:
              case 0b10001110:
              case 0b10010010:
              case 0b10010110:
              case 0b10011010:
              case 0b10011110:
              case 0b10100010:
              case 0b10100110:
              case 0b10101010:
              case 0b10101110:
              case 0b10110010:
              case 0b10110110:
              case 0b10111010:
              case 0b10111110:
              case 0b11000010:
              case 0b11000110:
              case 0b11001010:
              case 0b11001110:
              case 0b11010010:
              case 0b11010110:
              case 0b11011010:
              case 0b11011110:
              case 0b11100010:
              case 0b11100110:
              case 0b11101010:
              case 0b11101110:
              case 0b11110010:
              case 0b11110110:
              case 0b11111010:
              case 0b11111110:
                //レベル6
                XEiJ.mpuIRR &= ~XEiJ.MPU_MFP_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = MC68901.mfpAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_MFP_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00000100:
              case 0b00001100:
              case 0b00010100:
              case 0b00011100:
              case 0b00100100:
              case 0b00101100:
              case 0b00110100:
              case 0b00111100:
              case 0b01000100:
              case 0b01001100:
              case 0b01010100:
              case 0b01011100:
              case 0b01100100:
              case 0b01101100:
              case 0b01110100:
              case 0b01111100:
              case 0b10000100:
              case 0b10001100:
              case 0b10010100:
              case 0b10011100:
              case 0b10100100:
              case 0b10101100:
              case 0b10110100:
              case 0b10111100:
              case 0b11000100:
              case 0b11001100:
              case 0b11010100:
              case 0b11011100:
              case 0b11100100:
              case 0b11101100:
              case 0b11110100:
              case 0b11111100:
                //レベル5
                XEiJ.mpuIRR &= ~XEiJ.MPU_SCC_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = Z8530.sccAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_SCC_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00010000:
              case 0b00110000:
              case 0b01010000:
              case 0b01110000:
              case 0b10010000:
              case 0b10110000:
              case 0b11010000:
              case 0b11110000:
                //レベル3
                XEiJ.mpuIRR &= ~XEiJ.MPU_DMA_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = HD63450.dmaAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_DMA_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b00100000:
              case 0b01100000:
              case 0b10100000:
              case 0b11100000:
                //レベル2
                XEiJ.mpuIRR &= ~XEiJ.MPU_EB2_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.eb2Acknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_EB2_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              case 0b01000000:
              case 0b11000000:
                //レベル1
                XEiJ.mpuIRR &= ~XEiJ.MPU_IOI_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = IOInterrupt.ioiAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_IOI_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
                break;
              }
            } else {
              t &= -t;
              //  x&=-xはxの最下位の1のビットだけを残す演算
              //  すなわちマスクされているレベルよりも高くて受け付けていない割り込みの中で最高レベルの割り込みのビットだけが残る
              //  最高レベルの割り込みのビットしか残っていないので、割り込みの有無をレベルの高い順ではなく使用頻度の高い順に調べられる
              //  MFPやDMAの割り込みがかかる度にそれより優先度の高いインタラプトスイッチが押されていないかどうかを確かめる必要がない
              if (t == XEiJ.MPU_MFP_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_MFP_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = MC68901.mfpAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_MFP_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_DMA_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_DMA_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = HD63450.dmaAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_DMA_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_SCC_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_SCC_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = Z8530.sccAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_SCC_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_IOI_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_IOI_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = IOInterrupt.ioiAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_IOI_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_EB2_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_EB2_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.eb2Acknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_EB2_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              } else if (t == XEiJ.MPU_SYS_INTERRUPT_MASK) {
                XEiJ.mpuIRR &= ~XEiJ.MPU_SYS_INTERRUPT_MASK;  //割り込みを受け付ける
                if ((t = XEiJ.sysAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
                  irpInterrupt (t << 2, XEiJ.MPU_SYS_INTERRUPT_LEVEL);  //割り込み処理を開始する
                }
              }
            }
          }  //if t!=0
          if (MC68901.MFP_DELAYED_INTERRUPT) {
            XEiJ.mpuIRR |= XEiJ.mpuDIRR;  //遅延割り込み要求
            XEiJ.mpuDIRR = 0;
          }
        }  //命令ループ
      } catch (M68kException e) {
        if (M68kException.m6eNumber == M68kException.M6E_WAIT_EXCEPTION) {  //待機例外
          if (irpWaitException ()) {
            continue;
          } else {
            break errorLoop;
          }
        }
        if (M68kException.m6eNumber == M68kException.M6E_INSTRUCTION_BREAK_POINT) {  //命令ブレークポイントによる停止
          XEiJ.regPC = XEiJ.regPC0;
          XEiJ.mpuStop1 (null);  //"Instruction Break Point"
          break errorLoop;
        }
        //例外処理
        //  ここで処理するのはベクタ番号が2～63の例外に限る
        //  例外処理のサイクル数はACCESS_FAULTとADDRESS_ERROR以外は19になっているので必要ならば補正してからthrowする
        //  使用頻度が高いと思われる例外はインライン展開するのでここには来ない
        //  セーブされるpcは以下の例外は命令の先頭、これ以外は次の命令
        //     2  ACCESS_FAULT
        //     3  ADDRESS_ERROR
        //     4  ILLEGAL_INSTRUCTION
        //     8  PRIVILEGE_VIOLATION
        //    10  LINE_1010_EMULATOR
        //    11  LINE_1111_EMULATOR
        //    14  FORMAT_ERROR
        //    48  FP_BRANCH_SET_UNORDERED
        //    60  UNIMPLEMENTED_EFFECTIVE
        //    61  UNIMPLEMENTED_INSTRUCTION
        //              111111111122222222223333333333444444444455555555556666
        //    0123456789012345678901234567890123456789012345678901234567890123
        if (0b0011100010110010000000000000000000000000000000001000000000001100L << M68kException.m6eNumber < 0L) {
          XEiJ.regPC = XEiJ.regPC0;  //セーブされるpcは命令の先頭
          //アドレスレジスタを巻き戻す
          //  A7を含むのでユーザモードのときはスーパーバイザモードに移行する前に巻き戻すこと
          for (int arr = 8; m60Incremented != 0L; arr++) {
            XEiJ.regRn[arr] -= (byte) m60Incremented;
            m60Incremented = (m60Incremented + 0x80L) >> 8;
          }
        }
        //FSLWのTTRを設定する
        //  透過変換でアドレス変換キャッシュがヒットしてバスエラーが発生したときFSLWのTTRが設定されていない
        //!!! SECONDのときFIRSTと同じページか確認していない。ページフォルトのときは次のページだがバスエラーのときは同じページかもしれない
        if ((m60FSLW & (M60_FSLW_BUS_ERROR_ON_READ | M60_FSLW_BUS_ERROR_ON_WRITE)) != 0) {  //バスエラーのとき
          if (((m60FSLW & M60_FSLW_TM_SUPERVISOR) != 0 ?
               (m60FSLW & M60_FSLW_TM_CODE) != 0 ? mmuSuperCodeTransparent : mmuSuperDataTransparent :
               (m60FSLW & M60_FSLW_TM_CODE) != 0 ? mmuUserCodeTransparent : mmuUserDataTransparent)
              [m60Address >>> 24] != 0) {  //透過変換
            m60FSLW |= M60_FSLW_TRANSPARENT;
          }
        }
        if (false) {
          System.out.println (m60ErrorToString ());  //srを表示するのでsrを更新する前に呼び出すこと
        }
        try {
          int save_sr = XEiJ.regSRT1 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
          XEiJ.regSRT1 = XEiJ.mpuTraceFlag = 0;  //srのTビットを消す
          int sp;
          if (XEiJ.regSRS != 0) {  //スーパーバイザモード
            sp = XEiJ.regRn[15];
          } else {  //ユーザモード
            XEiJ.regSRS = XEiJ.REG_SR_S;  //スーパーバイザモードへ移行する
            XEiJ.mpuUSP = XEiJ.regRn[15];  //USPを保存
            sp = XEiJ.mpuISP;  //SSPを復元
            if (DataBreakPoint.DBP_ON) {
              DataBreakPoint.dbpMemoryMap = DataBreakPoint.dbpSuperMap;  //スーパーバイザメモリマップに切り替える
            } else {
              XEiJ.busMemoryMap = XEiJ.busSuperMap;  //スーパーバイザメモリマップに切り替える
            }
            if (InstructionBreakPoint.IBP_ON) {
              InstructionBreakPoint.ibpOp1MemoryMap = InstructionBreakPoint.ibpOp1SuperMap;
            }
          }
          //以下はスーパーバイザモード
          XEiJ.mpuClockTime += XEiJ.mpuModifiedUnit * 19;
          //  同じオフセットで異なるフォーマットになるものはここでは処理できない
          if (M68kException.m6eNumber == M68kException.M6E_ACCESS_FAULT) {
            //ホストファイルシステムのデバイスコマンドを強制終了させる
            HFS.hfsState = HFS.HFS_STATE_IDLE;
            //FORMAT $4の例外スタックフレームを作る
            XEiJ.regRn[15] = sp -= 16;
            mmuWriteLongData (sp + 12, m60FSLW, 1);  //15-12:フォルトステータスロングワード(FSLW)
            mmuWriteLongData (sp + 8, m60Address, 1);  //11-8:フォルトアドレス
            mmuWriteWordData (sp + 6, 0x4000 | M68kException.M6E_ACCESS_FAULT << 2, 1);  //7-6:フォーマットとベクタオフセット
            //                   111111111122222222223333333333444444444455555555556666
            //         0123456789012345678901234567890123456789012345678901234567890123
          } else if (0b0001011101000000000000000000000000000000000000000000000000000000L << M68kException.m6eNumber < 0L) {
            //FORMAT $2の例外スタックフレームを作る
            XEiJ.regRn[15] = sp -= 12;
            mmuWriteLongData (sp + 8, m60Address, 1);  //11-8:命令アドレス
            mmuWriteWordData (sp + 6, 0x2000 | M68kException.m6eNumber << 2, 1);  //7-6:フォーマットとベクタオフセット
          } else {
            //FORMAT $0の例外スタックフレームを作る
            XEiJ.regRn[15] = sp -= 8;
            mmuWriteWordData (sp + 6, M68kException.m6eNumber << 2, 1);  //7-6:フォーマットとベクタオフセット
          }
          mmuWriteLongData (sp + 2, XEiJ.regPC, 1);  //5-2:プログラムカウンタ
          mmuWriteWordData (sp, save_sr, 1);  //1-0:ステータスレジスタ
          irpSetPC (mmuReadLongFour (XEiJ.mpuVBR + (M68kException.m6eNumber << 2), 1));  //例外ベクタを取り出してジャンプする
          if (XEiJ.dbgStopOnError) {  //エラーで停止する場合
            if (XEiJ.dbgDoStopOnError ()) {
              break errorLoop;
            }
          }
        } catch (M68kException ee) {  //ダブルバスフォルト
          XEiJ.dbgDoubleBusFault ();
          break errorLoop;
        }
      }  //catch M68kException
    }  //例外ループ

    //  通常
    //    pc0  最後に実行した命令
    //    pc  次に実行する命令
    //  バスエラー、アドレスエラー、不当命令、特権違反で停止したとき
    //    pc0  エラーを発生させた命令
    //    pc  例外処理ルーチンの先頭
    //  ダブルバスフォルトで停止したとき
    //    pc0  エラーを発生させた命令
    //    pc  エラーを発生させた命令
    //  命令ブレークポイントで停止したとき
    //    pc0  命令ブレークポイントが設定された、次に実行する命令
    //    pc  命令ブレークポイントが設定された、次に実行する命令
    //  データブレークポイントで停止したとき
    //    pc0  データを書き換えた、最後に実行した命令
    //    pc  次に実行する命令

    //分岐ログに停止レコードを記録する
    if (BranchLog.BLG_ON) {
      BranchLog.blgStop ();
    }

  }  //mpuCore()



  //cont = irpWaitException ()
  //  待機例外をキャッチしたとき
  public static boolean irpWaitException () {
    XEiJ.regPC = XEiJ.regPC0;  //PCを巻き戻す
    XEiJ.regRn[8 + (XEiJ.regOC & 7)] += WaitInstruction.REWIND_AR[XEiJ.regOC >> 3];  //(Ar)+|-(Ar)で変化したArを巻き戻す
    try {
      //トレース例外を処理する
      if (XEiJ.mpuTraceFlag != 0) {  //命令実行前にsrのTビットがセットされていた
        irpExceptionFormat2 (M68kException.M6E_TRACE << 2, XEiJ.regPC, XEiJ.regPC0);  //pcは次の命令
      }
      //デバイスを呼び出す
      TickerQueue.tkqRun (XEiJ.mpuClockTime);
      //割り込みを受け付ける
      int t;
      if ((t = XEiJ.mpuIMR & XEiJ.mpuIRR) != 0) {  //マスクされているレベルよりも高くて受け付けていない割り込みがあるとき
        t &= -t;
        //  x&=-xはxの最下位の1のビットだけを残す演算
        //  すなわちマスクされているレベルよりも高くて受け付けていない割り込みの中で最高レベルの割り込みのビットだけが残る
        //  最高レベルの割り込みのビットしか残っていないので、割り込みの有無をレベルの高い順ではなく使用頻度の高い順に調べられる
        //  MFPやDMAの割り込みがかかる度にそれより優先度の高いインタラプトスイッチが押されていないかどうかを確かめる必要がない
        if (t == XEiJ.MPU_MFP_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_MFP_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = MC68901.mfpAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t << 2, XEiJ.MPU_MFP_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_DMA_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_DMA_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = HD63450.dmaAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t << 2, XEiJ.MPU_DMA_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_SCC_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_SCC_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = Z8530.sccAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t << 2, XEiJ.MPU_SCC_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_IOI_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_IOI_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = IOInterrupt.ioiAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t << 2, XEiJ.MPU_IOI_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_EB2_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_EB2_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = XEiJ.eb2Acknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t << 2, XEiJ.MPU_EB2_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        } else if (t == XEiJ.MPU_SYS_INTERRUPT_MASK) {
          XEiJ.mpuIRR &= ~XEiJ.MPU_SYS_INTERRUPT_MASK;  //割り込みを受け付ける
          if ((t = XEiJ.sysAcknowledge ()) != 0) {  //デバイスにベクタ番号を要求して割り込み処理中の状態になったとき
            irpInterrupt (t << 2, XEiJ.MPU_SYS_INTERRUPT_LEVEL);  //割り込み処理を開始する
          }
        }
      }  //if t!=0
      if (MC68901.MFP_DELAYED_INTERRUPT) {
        XEiJ.mpuIRR |= XEiJ.mpuDIRR;  //遅延割り込み要求
        XEiJ.mpuDIRR = 0;
      }
    } catch (M68kException e) {
      //!!! 待機例外処理中のバスエラーの処理は省略
      XEiJ.dbgDoubleBusFault ();
      return false;
    }  //catch M68kException
    return true;
  }  //irpWaitException



  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ORI.B #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_000_mmm_rrr-{data}
  //OR.B #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_000_mmm_rrr-{data}  [ORI.B #<data>,<ea>]
  //ORI.B #<data>,CCR                               |-|012346|-|*****|*****|          |0000_000_000_111_100-{data}
  public static void irpOriByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //ORI.B #<data>,Dr
      if (XEiJ.DBG_ORI_BYTE_ZERO_D0) {
        if (z == 0 && ea == 0 && XEiJ.dbgOriByteZeroD0) {  //ORI.B #$00,D0
          M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
          throw M68kException.m6eSignal;
        }
      }
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] |= 255 & z;  //0拡張してからOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea == XEiJ.EA_IM) {  //ORI.B #<data>,CCR
      XEiJ.mpuCycleCount++;
      XEiJ.regCCR |= XEiJ.REG_CCR_MASK & z;
    } else {  //ORI.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z |= mmuModifyByteSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpOriByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ORI.W #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_001_mmm_rrr-{data}
  //OR.W #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_001_mmm_rrr-{data}  [ORI.W #<data>,<ea>]
  //ORI.W #<data>,SR                                |-|012346|P|*****|*****|          |0000_000_001_111_100-{data}
  public static void irpOriWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //ORI.W #<data>,Dr
      int z = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] |= (char) z;  //0拡張してからOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea == XEiJ.EA_IM) {  //ORI.W #<data>,SR
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
        throw M68kException.m6eSignal;
      }
      //以下はスーパーバイザモード
      XEiJ.mpuCycleCount += 5;
      irpSetSR (XEiJ.regSRT1 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR | mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, 1));  //pcws。特権違反チェックが先
    } else {  //ORI.W #<data>,<mem>
      int z = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z |= mmuModifyWordSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpOriWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ORI.L #<data>,<ea>                              |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_000_010_mmm_rrr-{data}
  //OR.L #<data>,<ea>                               |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_000_010_mmm_rrr-{data}  [ORI.L #<data>,<ea>]
  public static void irpOriLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
    int z;
    if (ea < XEiJ.EA_AR) {  //ORI.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] |= y;
    } else {  //ORI.L #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = mmuModifyLongData (a, XEiJ.regSRS) | y, XEiJ.regSRS);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpOriLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BITREV.L Dr                                     |-|------|-|-----|-----|D         |0000_000_011_000_rrr (ISA_C)
  //CMP2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn000000000000
  //CHK2.B <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_000_011_mmm_rrr-rnnn100000000000
  //
  //BITREV.L Dr
  //  Drのビットの並びを逆順にする。CCRは変化しない
  //
  //CHK2.B <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CHK2.B <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする。このときCHK instruction例外が発生する
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //CMP2.B <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CMP2.B <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  public static void irpCmp2Chk2Byte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //BITREV.L Dr
      XEiJ.mpuCycleCount++;
      int x = XEiJ.regRn[ea];
      XEiJ.regRn[ea] = XEiJ.MPU_BITREV_TABLE_0[x & 2047] | XEiJ.MPU_BITREV_TABLE_1[x << 10 >>> 21] | XEiJ.MPU_BITREV_TABLE_2[x >>> 22];
    } else {  //CMP2/CHK2.B <ea>,Rn
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpCmp2Chk2Byte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BTST.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_100_000_rrr
  //MOVEP.W (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_100_001_rrr-{data}
  //BTST.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZPI|0000_qqq_100_mmm_rrr
  public static void irpBtstReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9;  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.W (d16,Ar),Dq
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //BTST.L Dq,Dr/<ea>
      int y = XEiJ.regRn[qqq];
      if (ea < XEiJ.EA_AR) {  //BTST.L Dq,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~XEiJ.regRn[ea] >>> y & 1) << 2;  //ccr_btst。intのシフトは5bitでマスクされるので&31を省略
      } else {  //BTST.B Dq,<ea>
        XEiJ.mpuCycleCount++;
        XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~(ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS)) >>> (y & 7) & 1) << 2;  //ccr_btst。pcbs。イミディエイトを分離
      }
    }
  }  //irpBtstReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCHG.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_101_000_rrr
  //MOVEP.L (d16,Ar),Dq                             |-|01234S|-|-----|-----|          |0000_qqq_101_001_rrr-{data}
  //BCHG.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_101_mmm_rrr
  public static void irpBchgReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9;  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.L (d16,Ar),Dq
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //BCHG.L Dq,Dr/<ea>
      int x;
      int y = XEiJ.regRn[qqq];
      if (ea < XEiJ.EA_AR) {  //BCHG.L Dq,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) ^ (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
      } else {  //BCHG.B Dq,<ea>
        XEiJ.mpuCycleCount++;
        int a = efaMltByte (ea);
        mmuWriteByteData (a, (x = mmuModifyByteSignData (a, XEiJ.regSRS)) ^ (y = 1 << (y & 7)), XEiJ.regSRS);
      }
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
    }
  }  //irpBchgReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCLR.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_110_000_rrr
  //MOVEP.W Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_110_001_rrr-{data}
  //BCLR.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_110_mmm_rrr
  public static void irpBclrReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = XEiJ.regRn[XEiJ.regOC >> 9];  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.W Dq,(d16,Ar)
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //BCLR.L Dq,Dr/<ea>
      int x;
      if (ea < XEiJ.EA_AR) {  //BCLR.L Dq,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) & ~(y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
      } else {  //BCLR.B Dq,<ea>
        XEiJ.mpuCycleCount++;
        int a = efaMltByte (ea);
        mmuWriteByteData (a, (x = mmuModifyByteSignData (a, XEiJ.regSRS)) & ~(y = 1 << (y & 7)), XEiJ.regSRS);
      }
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
    }
  }  //irpBclrReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSET.L Dq,Dr                                    |-|012346|-|--U--|--*--|D         |0000_qqq_111_000_rrr
  //MOVEP.L Dq,(d16,Ar)                             |-|01234S|-|-----|-----|          |0000_qqq_111_001_rrr-{data}
  //BSET.B Dq,<ea>                                  |-|012346|-|--U--|--*--|  M+-WXZ  |0000_qqq_111_mmm_rrr
  public static void irpBsetReg () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = XEiJ.regRn[XEiJ.regOC >> 9];  //qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //MOVEP.L Dq,(d16,Ar)
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //BSET.L Dq,Dr/<ea>
      int x;
      if (ea < XEiJ.EA_AR) {  //BSET.L Dq,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) | (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
      } else {  //BSET.B Dq,<ea>
        XEiJ.mpuCycleCount++;
        int a = efaMltByte (ea);
        mmuWriteByteData (a, (x = mmuModifyByteSignData (a, XEiJ.regSRS)) | (y = 1 << (y & 7)), XEiJ.regSRS);
      }
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
    }
  }  //irpBsetReg

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ANDI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_000_mmm_rrr-{data}
  //AND.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_000_mmm_rrr-{data}  [ANDI.B #<data>,<ea>]
  //ANDI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_001_000_111_100-{data}
  public static void irpAndiByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //ANDI.B #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] &= ~255 | z;  //1拡張してからAND
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea == XEiJ.EA_IM) {  //ANDI.B #<data>,CCR
      XEiJ.mpuCycleCount++;
      XEiJ.regCCR &= z;
    } else {  //ANDI.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z &= mmuModifyByteSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpAndiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ANDI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_001_mmm_rrr-{data}
  //AND.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_001_mmm_rrr-{data}  [ANDI.W #<data>,<ea>]
  //ANDI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_001_001_111_100-{data}
  public static void irpAndiWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //ANDI.W #<data>,Dr
      int z = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] &= ~65535 | z;  //1拡張してからAND
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea == XEiJ.EA_IM) {  //ANDI.W #<data>,SR
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
        throw M68kException.m6eSignal;
      }
      //以下はスーパーバイザモード
      XEiJ.mpuCycleCount += 12;
      irpSetSR ((XEiJ.regSRT1 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR) & mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, 1));  //pcws。特権違反チェックが先
    } else {  //ANDI.W #<data>,<mem>
      int z = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z &= mmuModifyWordSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpAndiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ANDI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_001_010_mmm_rrr-{data}
  //AND.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|  M+-WXZ  |0000_001_010_mmm_rrr-{data}  [ANDI.L #<data>,<ea>]
  public static void irpAndiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
    int z;
    if (ea < XEiJ.EA_AR) {  //ANDI.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] &= y;
    } else {  //ANDI.L #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = mmuModifyLongData (a, XEiJ.regSRS) & y, XEiJ.regSRS);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpAndiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BYTEREV.L Dr                                    |-|------|-|-----|-----|D         |0000_001_011_000_rrr (ISA_C)
  //CMP2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn000000000000
  //CHK2.W <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_001_011_mmm_rrr-rnnn100000000000
  //
  //BYTEREV.L Dr
  //  Drのバイトの並びを逆順にする。CCRは変化しない
  //
  //CHK2.W <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CHK2.W <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする。このときCHK instruction例外が発生する
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //CMP2.W <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  CMP2.W <ea>,Anは下限と上限をそれぞれロングに符号拡張してロングで比較する
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  public static void irpCmp2Chk2Word () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //BYTEREV.L Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = Integer.reverseBytes (XEiJ.regRn[ea]);
    } else {  //CMP2/CHK2.W <ea>,Rn
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpCmp2Chk2Word

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_000_mmm_rrr-{data}
  //SUB.B #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_000_mmm_rrr-{data}  [SUBI.B #<data>,<ea>]
  public static void irpSubiByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBI.B #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) - y);
    } else {  //SUBI.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = (byte) ((x = mmuModifyByteSignData (a, XEiJ.regSRS)) - y), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_001_mmm_rrr-{data}
  //SUB.W #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_001_mmm_rrr-{data}  [SUBI.W #<data>,<ea>]
  public static void irpSubiWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBI.W #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) - y));
    } else {  //SUBI.W #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z = (short) ((x = mmuModifyWordSignData (a, XEiJ.regSRS)) - y), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_010_010_mmm_rrr-{data}
  //SUB.L #<data>,<ea>                              |A|012346|-|UUUUU|*****|  M+-WXZ  |0000_010_010_mmm_rrr-{data}  [SUBI.L #<data>,<ea>]
  public static void irpSubiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBI.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) - y;
    } else {  //SUBI.L #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) - y, XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //FF1.L Dr                                        |-|------|-|-UUUU|-**00|D         |0000_010_011_000_rrr (ISA_C)
  //CMP2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn000000000000
  //CHK2.L <ea>,Rn                                  |-|--234S|-|-UUUU|-U*U*|  M  WXZP |0000_010_011_mmm_rrr-rnnn100000000000
  //
  //CHK2.L <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする。このときCHK instruction例外が発生する
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //CMP2.L <ea>,Rn
  //  <ea>から下限と上限をリードしてRnが範囲内か調べる
  //  Rnが下限または上限と等しいときZをセットする
  //  Rnが範囲外のときCをセットする
  //  060ISPのソースは注釈に誤りが多いので注釈ではなくコードを参考にする
  //  CCR
  //    X  変化しない
  //    N  変化しない(M68000PRMでは未定義)
  //    Z  Rn-LB==0||Rn-LB==UB-LB
  //    V  変化しない(M68000PRMでは未定義)
  //    C  Rn-LB>UB-LB(符号なし比較)
  //
  //FF1.L Dr
  //  Drの最上位の1のbit31からのオフセットをDrに格納する
  //  Drが0のときは32になる
  public static void irpCmp2Chk2Long () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //FF1.L Dr
      XEiJ.mpuCycleCount++;
      int z = XEiJ.regRn[ea];
      XEiJ.regRn[ea] = Integer.numberOfLeadingZeros (z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //CMP2/CHK2.L <ea>,Rn
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpCmp2Chk2Long

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDI.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_000_mmm_rrr-{data}
  public static void irpAddiByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDI.B #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) + y);
    } else {  //ADDI.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = (byte) ((x = mmuModifyByteSignData (a, XEiJ.regSRS)) + y), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >>> 31 << 1 |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDI.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_001_mmm_rrr-{data}
  public static void irpAddiWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDI.W #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) + y));
    } else {  //ADDI.W #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z = (short) ((x = mmuModifyWordSignData (a, XEiJ.regSRS)) + y), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >>> 31 << 1 |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDI.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0000_011_010_mmm_rrr-{data}
  public static void irpAddiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDI.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) + y;
    } else {  //ADDI.L #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) + y, XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >>> 31 << 1 |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BTST.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_000_000_rrr-{data}
  //BTST.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZP |0000_100_000_mmm_rrr-{data}
  public static void irpBtstImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //BTST.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~XEiJ.regRn[ea] >>> y & 1) << 2;  //ccr_btst。intのシフトは5bitでマスクされるので&31を省略
    } else {  //BTST.B #<data>,<ea>
      XEiJ.mpuCycleCount++;
      XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (~mmuReadByteSignData (efaMemByte (ea), XEiJ.regSRS) >>> (y & 7) & 1) << 2;  //ccr_btst
    }
  }  //irpBtstImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCHG.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_001_000_rrr-{data}
  //BCHG.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_001_mmm_rrr-{data}
  public static void irpBchgImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //BCHG.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) ^ (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
    } else {  //BCHG.B #<data>,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, (x = mmuModifyByteSignData (a, XEiJ.regSRS)) ^ (y = 1 << (y & 7)), XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
  }  //irpBchgImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCLR.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_010_000_rrr-{data}
  //BCLR.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_010_mmm_rrr-{data}
  public static void irpBclrImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //BCLR.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) & ~(y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
    } else {  //BCLR.B #<data>,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, (x = mmuModifyByteSignData (a, XEiJ.regSRS)) & ~(y = 1 << (y & 7)), XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
  }  //irpBclrImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSET.L #<data>,Dr                               |-|012346|-|--U--|--*--|D         |0000_100_011_000_rrr-{data}
  //BSET.B #<data>,<ea>                             |-|012346|-|--U--|--*--|  M+-WXZ  |0000_100_011_mmm_rrr-{data}
  public static void irpBsetImm () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //BSET.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = (x = XEiJ.regRn[ea]) | (y = 1 << y);  //intのシフトは5bitでマスクされるので1<<(y&0x1f)の&0x1fを省略
    } else {  //BSET.B #<data>,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, (x = mmuModifyByteSignData (a, XEiJ.regSRS)) | (y = 1 << (y & 7)), XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) | (x & y) - 1 >>> 31 << 2;  //ccr_btst
  }  //irpBsetImm

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EORI.B #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}
  //EOR.B #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_000_mmm_rrr-{data}  [EORI.B #<data>,<ea>]
  //EORI.B #<data>,CCR                              |-|012346|-|*****|*****|          |0000_101_000_111_100-{data}
  public static void irpEoriByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    if (ea < XEiJ.EA_AR) {  //EORI.B #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] ^= 255 & z;  //0拡張してからEOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea == XEiJ.EA_IM) {  //EORI.B #<data>,CCR
      XEiJ.mpuCycleCount++;
      XEiJ.regCCR ^= XEiJ.REG_CCR_MASK & z;
    } else {  //EORI.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z ^= mmuModifyByteSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpEoriByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EORI.W #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}
  //EOR.W #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_001_mmm_rrr-{data}  [EORI.W #<data>,<ea>]
  //EORI.W #<data>,SR                               |-|012346|P|*****|*****|          |0000_101_001_111_100-{data}
  public static void irpEoriWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EORI.W #<data>,Dr
      int z = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] ^= (char) z;  //0拡張してからEOR
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea == XEiJ.EA_IM) {  //EORI.W #<data>,SR
      if (XEiJ.regSRS == 0) {  //ユーザモードのとき
        M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
        throw M68kException.m6eSignal;
      }
      //以下はスーパーバイザモード
      XEiJ.mpuCycleCount += 12;
      irpSetSR ((XEiJ.regSRT1 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR) ^ mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, 1));  //pcws。特権違反チェックが先
    } else {  //EORI.W #<data>,<mem>
      int z = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z ^= mmuModifyWordSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpEoriWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EORI.L #<data>,<ea>                             |-|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}
  //EOR.L #<data>,<ea>                              |A|012346|-|-UUUU|-**00|D M+-WXZ  |0000_101_010_mmm_rrr-{data}  [EORI.L #<data>,<ea>]
  public static void irpEoriLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
    int z;
    if (ea < XEiJ.EA_AR) {  //EORI.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] ^= y;
    } else {  //EORI.L #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = mmuModifyLongData (a, XEiJ.regSRS) ^ y, XEiJ.regSRS);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpEoriLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CAS.B Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_101_011_mmm_rrr-0000000uuu000ccc
  public static void irpCasByte () throws M68kException {
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz。拡張ワード
    if ((w & ~0b0000_000_111_000_111) != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    int c = w & 7;
    int y = (byte) XEiJ.regRn[c];  //y=Dc
    int a = efaMltByte (XEiJ.regOC & 63);
    int x = mmuReadByteSignData (a, XEiJ.regSRS);  //x=<ea>
    int z = (byte) (x - y);  //z=<ea>-Dc
    XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                   (z < 0 ? XEiJ.REG_CCR_N : 0) |
                   (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                   ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                   (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    if (z == 0) {  //<ea>==Dc
      XEiJ.mpuCycleCount += 19;
      mmuWriteByteData (a, XEiJ.regRn[w >> 6], XEiJ.regSRS);  //Du→<ea>
    } else {  //<ea>!=Dc
      XEiJ.mpuCycleCount += 19;
      XEiJ.regRn[c] = ~0xff & XEiJ.regRn[c] | 0xff & x;  //<ea>→Dc
    }
  }  //irpCasByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPI.B #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_000_mmm_rrr-{data}
  //CMP.B #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_000_mmm_rrr-{data}  [CMPI.B #<data>,<ea>]
  public static void irpCmpiByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS);  //pcbs
    int z = (byte) ((x = ea < XEiJ.EA_AR ? (byte) XEiJ.regRn[ea] : mmuReadByteSignData (efaMemByte (ea), XEiJ.regSRS)) - y);  //アドレッシングモードに注意
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpiByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPI.W #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_001_mmm_rrr-{data}
  //CMP.W #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_001_mmm_rrr-{data}  [CMPI.W #<data>,<ea>]
  public static void irpCmpiWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
    int z = (short) ((x = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : mmuReadWordSignData (efaMemWord (ea), XEiJ.regSRS)) - y);  //アドレッシングモードに注意
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpiWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPI.L #<data>,<ea>                             |-|--2346|-|-UUUU|-****|D M+-WXZP |0000_110_010_mmm_rrr-{data}
  //CMP.L #<data>,<ea>                              |A|--2346|-|-UUUU|-****|  M+-WXZP |0000_110_010_mmm_rrr-{data}  [CMPI.L #<data>,<ea>]
  public static void irpCmpiLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
    int z;
    if (ea < XEiJ.EA_AR) {  //CMPI.L #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (x = XEiJ.regRn[ea]) - y;
    } else {  //CMPI.L #<data>,<mem>
      XEiJ.mpuCycleCount++;
      z = (x = mmuReadLongData (efaMemLong (ea), XEiJ.regSRS)) - y;  //アドレッシングモードに注意
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpiLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CAS.W Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_110_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
  //CAS2.W Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_110_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
  public static void irpCasWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea == XEiJ.EA_IM) {  //CAS2.W Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //CAS.W Dc,Du,<ea>
      int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz
      if ((w & ~0b0000_000_111_000_111) != 0) {
        M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int a = efaMltWord (ea);  //a=ea
      if ((a & 1) != 0) {  //misaligned <ea>
        M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int c = w & 7;
      int y = (short) XEiJ.regRn[c];  //y=Dc
      int x = mmuReadWordSignData (a, XEiJ.regSRS);  //x=<ea>
      int z = (short) (x - y);  //z=<ea>-Dc
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                     (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
      if (z == 0) {  //<ea>==Dc
        XEiJ.mpuCycleCount += 19;
        mmuWriteWordData (a, XEiJ.regRn[w >> 6], XEiJ.regSRS);  //Du→<ea>
      } else {  //<ea>!=Dc
        XEiJ.mpuCycleCount += 19;
        XEiJ.regRn[c] = ~0xffff & XEiJ.regRn[c] | (char) x;  //<ea>→Dc
      }
    }
  }  //irpCasWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVES.B <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn000000000000
  //MOVES.B Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_000_mmm_rrr-rnnn100000000000
  //
  //MOVES.B <ea>,Rn
  //  MOVES.B <ea>,DnはDnの最下位バイトだけ更新する
  //  MOVES.B <ea>,Anはバイトデータをロングに符号拡張してAnの全体を更新する
  //  SFC=1,2,5,6はアドレス変換あり、SFC=0,3,4はアドレス変換なし、
  //  SFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  //
  //MOVES.B Rn,<ea>
  //  DFC=1,2,5,6はアドレス変換あり、DFC=0,3,4はアドレス変換なし、
  //  DFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  public static void irpMovesByte () throws M68kException {
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz
    if (w << -11 != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount++;
    int a = efaMltByte (XEiJ.regOC & 63);
    int n = w >>> 12;  //n
    if (w << 31 - 11 >= 0) {  //MOVES.B <ea>,Rn。リード
      boolean supervisor = (0b10011111 << 24 << XEiJ.mpuSFC) < 0;
      boolean instruction = (0b00101010 << 24 << XEiJ.mpuSFC) < 0;
      MemoryMappedDevice[] mm = (DataBreakPoint.DBP_ON ?
                                 supervisor ? DataBreakPoint.dbpSuperMap : DataBreakPoint.dbpUserMap :
                                 supervisor ? XEiJ.busSuperMap : XEiJ.busUserMap);
      int z;
      //    01234567
      if (0b01100110 << 24 << XEiJ.mpuSFC < 0) {  //SFC=1,2,5,6。アドレス変換あり
        m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_BYTE | XEiJ.mpuSFC << 16;
        int pa = (supervisor ?
                  instruction ? mmuTranslateReadSuperCode (a) : mmuTranslateReadSuperData (a) :
                  instruction ? mmuTranslateReadUserCode (a) : mmuTranslateReadUserData (a));
        //z = XEiJ.busRbz (pa);
        z = mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRbz (pa);
      } else if (XEiJ.mpuSFC != 7) {  //SFC=0,3,4。アドレス変換なし
        m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_BYTE | XEiJ.mpuSFC << 16;
        //z = XEiJ.busRbz (a);
        z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a);
      } else {  //SFC=7。CPU空間
        if (0x00022000 <= a && a <= 0x0002201f) {  //コプロセッサID=1
          z = XEiJ.fpuMotherboardCoprocessor.cirReadByteZero (a);
        } else {
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_BYTE | XEiJ.mpuSFC << 16 | M60_FSLW_BUS_ERROR_ON_READ;
          M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
          M68kException.m6eAddress = a;
          M68kException.m6eDirection = XEiJ.MPU_WR_READ;
          M68kException.m6eSize = XEiJ.MPU_SS_BYTE;
          throw M68kException.m6eSignal;
        }
      }
      if (n < 8) {  //MOVES.B <ea>,Dn
        XEiJ.regRn[n] = XEiJ.regRn[n] & ~255 | z;
      } else {  //MOVES.B <ea>,An
        XEiJ.regRn[n] = (byte) z;
      }
      if (MMU_DEBUG_COMMAND) {
        System.out.printf ("%08x movesReadByte(%d,0x%08x)=0x%02x\n", XEiJ.regPC0, XEiJ.mpuSFC, a, XEiJ.regRn[n] & 255);
      }
    } else {  //MOVES.B Rn,<ea>。ライト
      if (MMU_DEBUG_COMMAND) {
        System.out.printf ("%08x movesWriteByte(%d,0x%08x,0x%02x)\n", XEiJ.regPC0, XEiJ.mpuDFC, a, XEiJ.regRn[n] & 255);
      }
      boolean supervisor = (0b10011111 << 24 << XEiJ.mpuDFC) < 0;
      boolean instruction = (0b00101010 << 24 << XEiJ.mpuDFC) < 0;
      MemoryMappedDevice mm[] = (DataBreakPoint.DBP_ON ?
                                 supervisor ? DataBreakPoint.dbpSuperMap : DataBreakPoint.dbpUserMap :
                                 supervisor ? XEiJ.busSuperMap : XEiJ.busUserMap);
      int z = XEiJ.regRn[n];
      //    01234567
      if (0b01100110 << 24 << XEiJ.mpuDFC < 0) {  //DFC=1,2,5,6。アドレス変換あり
        m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_BYTE | XEiJ.mpuDFC << 16;
        int pa = (supervisor ?
                  instruction ? mmuTranslateWriteSuperCode (a) : mmuTranslateWriteSuperData (a) :
                  instruction ? mmuTranslateWriteUserCode (a) : mmuTranslateWriteUserData (a));
        //XEiJ.busWb (pa, z);
        mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWb (pa, z);
      } else if (XEiJ.mpuDFC != 7) {  //DFC=0,3,4。アドレス変換なし
        m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_BYTE | XEiJ.mpuDFC << 16;
        //XEiJ.busWb (a, z);
        mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z);
      } else {  //DFC=7。CPU空間
        if (0x00022000 <= a && a <= 0x0002201f) {  //コプロセッサID=1
          XEiJ.fpuMotherboardCoprocessor.cirWriteByte (a, z);
        } else {
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_BYTE | XEiJ.mpuDFC << 16 | M60_FSLW_BUS_ERROR_ON_WRITE;
          M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
          M68kException.m6eAddress = a;
          M68kException.m6eDirection = XEiJ.MPU_WR_WRITE;
          M68kException.m6eSize = XEiJ.MPU_SS_BYTE;
          throw M68kException.m6eSignal;
        }
      }
    }
  }  //irpMovesByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVES.W <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn000000000000
  //MOVES.W Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_001_mmm_rrr-rnnn100000000000
  //
  //MOVES.W <ea>,Rn
  //  MOVES.W <ea>,DnはDnの下位ワードだけ更新する
  //  MOVES.W <ea>,Anはワードデータをロングに符号拡張してAnの全体を更新する
  //  SFC=1,2,5,6はアドレス変換あり、SFC=0,3,4はアドレス変換なし、
  //  SFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  //
  //MOVES.W Rn,<ea>
  //  DFC=1,2,5,6はアドレス変換あり、DFC=0,3,4はアドレス変換なし、
  //  DFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  public static void irpMovesWord () throws M68kException {
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz
    if (w << -11 != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount++;
    int a = efaMltWord (XEiJ.regOC & 63);
    int n = w >>> 12;  //n
    if (w << 31 - 11 >= 0) {  //MOVES.W <ea>,Rn。リード
      boolean supervisor = (0b10011111 << 24 << XEiJ.mpuSFC) < 0;
      boolean instruction = (0b00101010 << 24 << XEiJ.mpuSFC) < 0;
      MemoryMappedDevice[] mm = (DataBreakPoint.DBP_ON ?
                                 supervisor ? DataBreakPoint.dbpSuperMap : DataBreakPoint.dbpUserMap :
                                 supervisor ? XEiJ.busSuperMap : XEiJ.busUserMap);
      int z;
      //    01234567
      if (0b01100110 << 24 << XEiJ.mpuSFC < 0) {  //SFC=1,2,5,6。アドレス変換あり
        if ((a & 1) == 0) {  //偶数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | XEiJ.mpuSFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateReadSuperCode (a) : mmuTranslateReadSuperData (a) :
                    instruction ? mmuTranslateReadUserCode (a) : mmuTranslateReadUserData (a));
          //z = XEiJ.busRwze (pa);
          z = mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRwz (pa);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | XEiJ.mpuSFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateReadSuperCode (a) : mmuTranslateReadSuperData (a) :
                    instruction ? mmuTranslateReadUserCode (a) : mmuTranslateReadUserData (a));
          //z = XEiJ.busRbz (pa) << 8;
          z = mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRbz (pa) << 8;
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          pa = (supervisor ?
                instruction ? mmuTranslateReadSuperCode (a + 1) : mmuTranslateReadSuperData (a + 1) :
                instruction ? mmuTranslateReadUserCode (a + 1) : mmuTranslateReadUserData (a + 1));
          //z |= XEiJ.busRbz (pa);
          z |= mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRbz (pa);
        }
      } else if (XEiJ.mpuSFC != 7) {  //SFC=0,3,4。アドレス変換なし
        if ((a & 1) == 0) {  //偶数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | XEiJ.mpuSFC << 16;
          //z = XEiJ.busRwze (a);
          z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRwz (a);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | XEiJ.mpuSFC << 16;
          //z = XEiJ.busRbz (a) << 8;
          z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a) << 8;
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          a++;
          //z |= XEiJ.busRbz (a);
          z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a);
        }
      } else {  //SFC=7。CPU空間
        if (0x00022000 <= a && a <= 0x0002201f) {  //コプロセッサID=1
          z = XEiJ.fpuMotherboardCoprocessor.cirReadWordZero (a);
        } else {
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_WORD | XEiJ.mpuSFC << 16 | M60_FSLW_BUS_ERROR_ON_READ;
          M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
          M68kException.m6eAddress = a;
          M68kException.m6eDirection = XEiJ.MPU_WR_READ;
          M68kException.m6eSize = XEiJ.MPU_SS_WORD;
          throw M68kException.m6eSignal;
        }
      }
      if (n < 8) {  //MOVES.W <ea>,Dn
        XEiJ.regRn[n] = XEiJ.regRn[n] & ~65535 | z;
      } else {  //MOVES.W <ea>,An
        XEiJ.regRn[n] = (short) z;
      }
      if (MMU_DEBUG_COMMAND) {
        System.out.printf ("%08x movesReadWord(%d,0x%08x)=0x%04x\n", XEiJ.regPC0, XEiJ.mpuSFC, a, XEiJ.regRn[n] & 65535);
      }
    } else {  //MOVES.W Rn,<ea>。ライト
      if (MMU_DEBUG_COMMAND) {
        System.out.printf ("%08x movesWriteWord(%d,0x%08x,0x%04x)\n", XEiJ.regPC0, XEiJ.mpuDFC, a, XEiJ.regRn[n] & 65535);
      }
      boolean supervisor = (0b10011111 << 24 << XEiJ.mpuDFC) < 0;
      boolean instruction = (0b00101010 << 24 << XEiJ.mpuDFC) < 0;
      MemoryMappedDevice[] mm = (DataBreakPoint.DBP_ON ?
                                 supervisor ? DataBreakPoint.dbpSuperMap : DataBreakPoint.dbpUserMap :
                                 supervisor ? XEiJ.busSuperMap : XEiJ.busUserMap);
      int z = XEiJ.regRn[n];
      //    01234567
      if (0b01100110 << 24 << XEiJ.mpuDFC < 0) {  //DFC=1,2,5,6。アドレス変換あり
        if ((a & 1) == 0) {  //偶数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_WORD | XEiJ.mpuDFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateWriteSuperCode (a) : mmuTranslateWriteSuperData (a) :
                    instruction ? mmuTranslateWriteUserCode (a) : mmuTranslateWriteUserData (a));
          //XEiJ.busWwe (pa, z);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWw (pa, z);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_WORD | XEiJ.mpuDFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateWriteSuperCode (a) : mmuTranslateWriteSuperData (a) :
                    instruction ? mmuTranslateWriteUserCode (a) : mmuTranslateWriteUserData (a));
          //XEiJ.busWb (pa, z >> 8);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWb (pa, z >> 8);
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          pa = (supervisor ?
                instruction ? mmuTranslateWriteSuperCode (a + 1) : mmuTranslateWriteSuperData (a + 1) :
                instruction ? mmuTranslateWriteUserCode (a + 1) : mmuTranslateWriteUserData (a + 1));
          //XEiJ.busWb (pa, z);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWb (pa, z);
        }
      } else if (XEiJ.mpuDFC != 7) {  //DFC=0,3,4。アドレス変換なし
        if ((a & 1) == 0) {  //偶数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_WORD | XEiJ.mpuDFC << 16;
          //XEiJ.busWwe (a, z);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_WORD | XEiJ.mpuDFC << 16;
          //XEiJ.busWb (a, z >> 8);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z >> 8);
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          a++;
          //XEiJ.busWb (a, z);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z);
        }
      } else {  //DFC=7。CPU空間
        if (0x00022000 <= a && a <= 0x0002201f) {  //コプロセッサID=1
          XEiJ.fpuMotherboardCoprocessor.cirWriteWord (a, z);
        } else {
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_WORD | XEiJ.mpuDFC << 16 | M60_FSLW_BUS_ERROR_ON_WRITE;
          M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
          M68kException.m6eAddress = a;
          M68kException.m6eDirection = XEiJ.MPU_WR_WRITE;
          M68kException.m6eSize = XEiJ.MPU_SS_WORD;
          throw M68kException.m6eSignal;
        }
      }
    }
  }  //irpMovesWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVES.L <ea>,Rn                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn000000000000
  //MOVES.L Rn,<ea>                                 |-|-12346|P|-----|-----|  M+-WXZ  |0000_111_010_mmm_rrr-rnnn100000000000
  //
  //MOVES.L <ea>,Rn
  //  SFC=1,2,5,6はアドレス変換あり、SFC=0,3,4はアドレス変換なし、
  //  SFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  //
  //MOVES.L Rn,<ea>
  //  DFC=1,2,5,6はアドレス変換あり、DFC=0,3,4はアドレス変換なし、
  //  DFC=7はCPU空間なのでコプロセッサが割り当てられている領域以外はバスエラーになる
  public static void irpMovesLong () throws M68kException {
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz
    if (w << -11 != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount++;
    int a = efaMltLong (XEiJ.regOC & 63);
    int n = w >>> 12;  //n
    if (w << 31 - 11 >= 0) {  //MOVES.L <ea>,Rn。リード
      boolean supervisor = (0b10011111 << 24 << XEiJ.mpuSFC) < 0;
      boolean instruction = (0b00101010 << 24 << XEiJ.mpuSFC) < 0;
      MemoryMappedDevice[] mm = (DataBreakPoint.DBP_ON ?
                                 supervisor ? DataBreakPoint.dbpSuperMap : DataBreakPoint.dbpUserMap :
                                 supervisor ? XEiJ.busSuperMap : XEiJ.busUserMap);
      int z;
      //    01234567
      if (0b01100110 << 24 << XEiJ.mpuSFC < 0) {  //SFC=1,2,5,6。アドレス変換あり
        if ((a & 3) == 0) {  //4の倍数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateReadSuperCode (a) : mmuTranslateReadSuperData (a) :
                    instruction ? mmuTranslateReadUserCode (a) : mmuTranslateReadUserData (a));
          //z = XEiJ.busRlsf (pa);
          z = mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRls (pa);
        } else if ((a & 1) == 0) {  //4の倍数+2
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateReadSuperCode (a) : mmuTranslateReadSuperData (a) :
                    instruction ? mmuTranslateReadUserCode (a) : mmuTranslateReadUserData (a));
          //z = XEiJ.busRwse (pa) << 16;
          z = mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRws (pa) << 16;
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          pa = (supervisor ?
                instruction ? mmuTranslateReadSuperCode (a + 2) : mmuTranslateReadSuperData (a + 2) :
                instruction ? mmuTranslateReadUserCode (a + 2) : mmuTranslateReadUserData (a + 2));
          //z |= XEiJ.busRwze (pa);
          z |= mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRwz (pa);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateReadSuperCode (a) : mmuTranslateReadSuperData (a) :
                    instruction ? mmuTranslateReadUserCode (a) : mmuTranslateReadUserData (a));
          //z = XEiJ.busRbs (pa) << 24;
          z = mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRbs (pa) << 24;
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          pa = (supervisor ?
                instruction ? mmuTranslateReadSuperCode (a + 1) : mmuTranslateReadSuperData (a + 1) :
                instruction ? mmuTranslateReadUserCode (a + 1) : mmuTranslateReadUserData (a + 1));
          //z |= XEiJ.busRwze (pa) << 8;
          z |= mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRwz (pa) << 8;
          pa = (supervisor ?
                instruction ? mmuTranslateReadSuperCode (a + 3) : mmuTranslateReadSuperData (a + 3) :
                instruction ? mmuTranslateReadUserCode (a + 3) : mmuTranslateReadUserData (a + 3));
          //z |= XEiJ.busRbz (pa);
          z |= mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdRbz (pa);
        }
      } else if (XEiJ.mpuSFC != 7) {  //SFC=0,3,4。アドレス変換なし
        if ((a & 3) == 0) {  //4の倍数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16;
          //z = XEiJ.busRlsf (a);
          z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRls (a);
        } else if ((a & 1) == 0) {  //4の倍数+2
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16;
          //z = XEiJ.busRwse (a) << 16;
          z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRws (a) << 16;
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          a += 2;
          //z |= XEiJ.busRwze (a);
          z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRwz (a);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16;
          //z = XEiJ.busRbs (a) << 24;
          z = mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbs (a) << 24;
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          a++;
          //z |= XEiJ.busRwze (a) << 8;
          z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRwz (a) << 8;
          a += 2;
          //z |= XEiJ.busRbz (a);
          z |= mm[a >>> XEiJ.BUS_PAGE_BITS].mmdRbz (a);
        }
      } else {  //SFC=7。CPU空間
        if (0x00022000 <= a && a <= 0x0002201f) {  //コプロセッサID=1
          z = XEiJ.fpuMotherboardCoprocessor.cirReadLong (a);
        } else {
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_READ | M60_FSLW_SIZE_LONG | XEiJ.mpuSFC << 16 | M60_FSLW_BUS_ERROR_ON_READ;
          M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
          M68kException.m6eAddress = a;
          M68kException.m6eDirection = XEiJ.MPU_WR_READ;
          M68kException.m6eSize = XEiJ.MPU_SS_LONG;
          throw M68kException.m6eSignal;
        }
      }
      XEiJ.regRn[n] = z;
      if (MMU_DEBUG_COMMAND) {
        System.out.printf ("%08x movesReadLong(%d,0x%08x)=0x%08x\n", XEiJ.regPC0, XEiJ.mpuSFC, a, XEiJ.regRn[n]);
      }
    } else {  //MOVES.L Rn,<ea>。ライト
      if (MMU_DEBUG_COMMAND) {
        System.out.printf ("%08x movesWriteLong(%d,0x%08x,0x%08x)\n", XEiJ.regPC0, XEiJ.mpuDFC, a, XEiJ.regRn[n]);
      }
      boolean supervisor = (0b10011111 << 24 << XEiJ.mpuDFC) < 0;
      boolean instruction = (0b00101010 << 24 << XEiJ.mpuDFC) < 0;
      MemoryMappedDevice[] mm = (DataBreakPoint.DBP_ON ?
                                 supervisor ? DataBreakPoint.dbpSuperMap : DataBreakPoint.dbpUserMap :
                                 supervisor ? XEiJ.busSuperMap : XEiJ.busUserMap);
      int z = XEiJ.regRn[n];
      //    01234567
      if (0b01100110 << 24 << XEiJ.mpuDFC < 0) {  //DFC=1,2,5,6。アドレス変換あり
        if ((a & 3) == 0) {  //4の倍数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateWriteSuperCode (a) : mmuTranslateWriteSuperData (a) :
                    instruction ? mmuTranslateWriteUserCode (a) : mmuTranslateWriteUserData (a));
          //XEiJ.busWlf (pa, z);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWl (pa, z);
        } else if ((a & 1) == 0) {  //4の倍数+2
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateWriteSuperCode (a) : mmuTranslateWriteSuperData (a) :
                    instruction ? mmuTranslateWriteUserCode (a) : mmuTranslateWriteUserData (a));
          //XEiJ.busWwe (pa, z >> 16);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWw (pa, z >> 16);
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          pa = (supervisor ?
                instruction ? mmuTranslateWriteSuperCode (a + 2) : mmuTranslateWriteSuperData (a + 2) :
                instruction ? mmuTranslateWriteUserCode (a + 2) : mmuTranslateWriteUserData (a + 2));
          //XEiJ.busWwe (pa, z);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWw (pa, z);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16;
          int pa = (supervisor ?
                    instruction ? mmuTranslateWriteSuperCode (a) : mmuTranslateWriteSuperData (a) :
                    instruction ? mmuTranslateWriteUserCode (a) : mmuTranslateWriteUserData (a));
          //XEiJ.busWb (pa, z >> 24);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWb (pa, z >> 24);
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          pa = (supervisor ?
                instruction ? mmuTranslateWriteSuperCode (a + 1) : mmuTranslateWriteSuperData (a + 1) :
                instruction ? mmuTranslateWriteUserCode (a + 1) : mmuTranslateWriteUserData (a + 1));
          //XEiJ.busWwe (pa, z >> 8);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWw (pa, z >> 8);
          pa = (supervisor ?
                instruction ? mmuTranslateWriteSuperCode (a + 3) : mmuTranslateWriteSuperData (a + 3) :
                instruction ? mmuTranslateWriteUserCode (a + 3) : mmuTranslateWriteUserData (a + 3));
          //XEiJ.busWb (pa, z);
          mm[pa >>> XEiJ.BUS_PAGE_BITS].mmdWb (pa, z);
        }
      } else if (XEiJ.mpuDFC != 7) {  //DFC=0,3,4。アドレス変換なし
        if ((a & 3) == 0) {  //4の倍数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16;
          //XEiJ.busWlf (a, z);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWl (a, z);
        } else if ((a & 1) == 0) {  //4の倍数+2
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16;
          //XEiJ.busWwe (a, z >> 16);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z >> 16);
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          a += 2;
          //XEiJ.busWwe (a, z);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z);
        } else {  //奇数
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16;
          //XEiJ.busWb (a, z >> 24);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z >> 24);
          m60FSLW ^= M60_FSLW_IOMA_FIRST ^ M60_FSLW_IOMA_SECOND;
          a++;
          //XEiJ.busWwe (a, z >> 8);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWw (a, z >> 8);
          a += 2;
          //XEiJ.busWb (a, z);
          mm[a >>> XEiJ.BUS_PAGE_BITS].mmdWb (a, z);
        }
      } else {  //DFC=7。CPU空間
        if (0x00022000 <= a && a <= 0x0002201f) {  //コプロセッサID=1
          XEiJ.fpuMotherboardCoprocessor.cirWriteLong (a, z);
        } else {
          m60FSLW = M60_FSLW_IOMA_FIRST | M60_FSLW_RW_WRITE | M60_FSLW_SIZE_LONG | XEiJ.mpuDFC << 16 | M60_FSLW_BUS_ERROR_ON_WRITE;
          M68kException.m6eNumber = M68kException.M6E_ACCESS_FAULT;
          M68kException.m6eAddress = a;
          M68kException.m6eDirection = XEiJ.MPU_WR_WRITE;
          M68kException.m6eSize = XEiJ.MPU_SS_LONG;
          throw M68kException.m6eSignal;
        }
      }
    }
  }  //irpMovesLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CAS.L Dc,Du,<ea>                                |-|--2346|-|-UUUU|-****|  M+-WXZ  |0000_111_011_mmm_rrr-0000000uuu000ccc        (68060 software emulate misaligned <ea>)
  //CAS2.L Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)              |-|--234S|-|-UUUU|-****|          |0000_111_011_111_100-rnnn000uuu000ccc(1)-rnnn_000_uuu_000_ccc(2)
  public static void irpCasLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea == XEiJ.EA_IM) {  //CAS2.L Dc1:Dc2,Du1:Du2,(Rn1):(Rn2)
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //CAS.L Dc,Du,<ea>
      int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz
      if ((w & ~0b0000_000_111_000_111) != 0) {
        M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int a = efaMltLong (ea);  //a=ea
      if ((a & 1) != 0) {  //misaligned <ea>
        M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
        throw M68kException.m6eSignal;
      }
      int c = w & 7;
      int y = XEiJ.regRn[c];  //y=Dc
      int x = mmuReadLongData (a, XEiJ.regSRS);  //x=<ea>
      int z = x - y;  //z=<ea>-Dc
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |
                     (z == 0 ? XEiJ.REG_CCR_Z : 0) |
                     ((x ^ y) & (x ^ z)) >>> 31 << 1 |
                     (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
      if (z == 0) {  //<ea>==Dc
        XEiJ.mpuCycleCount += 19;
        mmuWriteLongData (a, XEiJ.regRn[w >> 6], XEiJ.regSRS);  //Du→<ea>
      } else {  //<ea>!=Dc
        XEiJ.mpuCycleCount += 19;
        XEiJ.regRn[c] = x;  //<ea>→Dc
      }
    }
  }  //irpCasLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,Dq                                  |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_000_mmm_rrr
  public static void irpMoveToDRByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離
    XEiJ.regRn[qqq] = ~255 & XEiJ.regRn[qqq] | 255 & z;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToDRByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_010_mmm_rrr
  public static void irpMoveToMMByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0001_000 - 8);
    int a = m60Address = XEiJ.regRn[aqq];
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMMByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_011_mmm_rrr
  public static void irpMoveToMPByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0001_000 - 8);
    int a;
    if (aqq < 15) {
      m60Incremented += 1L << (aqq << 3);  //longのシフトカウントは6bitでマスクされる
      a = m60Address = XEiJ.regRn[aqq]++;
    } else {
      m60Incremented += 2L << (7 << 3);
      a = m60Address = (XEiJ.regRn[15] += 2) - 2;
    }
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMPByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_100_mmm_rrr
  public static void irpMoveToMNByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0001_000 - 8);
    int a;
    if (aqq < 15) {
      m60Incremented -= 1L << (aqq << 3);  //longのシフトカウントは6bitでマスクされる
      a = m60Address = --XEiJ.regRn[aqq];
    } else {
      m60Incremented -= 2L << (7 << 3);
      a = m60Address = XEiJ.regRn[15] -= 2;
    }
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMNByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_101_mmm_rrr
  public static void irpMoveToMWByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0001_000 - 8);
    int t = XEiJ.regRn[aqq];  //ベースレジスタ
    int a = m60Address = t + mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //ワードディスプレースメント
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMWByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_qqq_110_mmm_rrr
  public static void irpMoveToMXByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0001_000 - 8);
    int a;
    int t = XEiJ.regRn[aqq];  //ベースレジスタ
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //拡張ワード
    if ((0x0100 & w) == 0) {  //ブリーフフォーマット
      a = m60Address =
        (t  //ベースレジスタ
         + (byte) w  //バイトディスプレースメント
         + (((0x0800 & w) == 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
             XEiJ.regRn[w >> 12])  //ロングインデックス
            << ((0x0600 & w) >> 9)));  //スケールファクタ
    } else {  //フルフォーマット
      XEiJ.mpuCycleCount += ((0x0003 & w) == 0 ? 1 :  //インダイレクトなし
                             3);  //インダイレクトあり
      t = (((0x0080 & w) != 0 ? 0 :  //ベースレジスタなし
            t) +  //ベースレジスタあり
           ((0x0020 & w) == 0 ? 0 :  //ベースディスプレースメントなし
            (0x0010 & w) == 0 ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) :  //ワードベースディスプレースメント
            mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS)));  //ロングベースディスプレースメント
      int x = ((0x0040 & w) != 0 ? 0 :  //インデックスなし
               ((0x0800 & w) == 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
                XEiJ.regRn[w >> 12])  //ロングインデックス
               << ((0x0600 & w) >> 9));  //スケールファクタ
      a = m60Address =
        ((0x0003 & w) == 0 ? t + x :  //インダイレクトなし
         (((0x0004 & w) == 0 ? mmuReadLongData (m60Address = t + x, XEiJ.regSRS) :  //プリインデックス
           mmuReadLongData (m60Address = t, XEiJ.regSRS) + x)  //ポストインデックス
          + ((0x0002 & w) == 0 ? 0 :  //アウタディスプレースメントなし
             (0x0001 & w) == 0 ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) :  //ワードアウタディスプレースメント
             mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS))));  //ロングアウタディスプレースメント
    }
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToMXByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_000_111_mmm_rrr
  public static void irpMoveToZWByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離
    int a = m60Address = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToZWByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.B <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|D M+-WXZPI|0001_001_111_mmm_rrr
  public static void irpMoveToZLByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離
    int a = m60Address = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
    mmuWriteByteData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpMoveToZLByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_000_mmm_rrr
  public static void irpMoveToDRLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToDRLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEA.L <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr
  //MOVE.L <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0010_qqq_001_mmm_rrr [MOVEA.L <ea>,Aq]
  public static void irpMoveaLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    XEiJ.regRn[(XEiJ.regOC >> 9) - (16 - 8)] = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。右辺でAqが変化する可能性があることに注意
  }  //irpMoveaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_010_mmm_rrr
  public static void irpMoveToMMLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0010_000 - 8);
    int a = m60Address = XEiJ.regRn[aqq];
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMMLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_011_mmm_rrr
  public static void irpMoveToMPLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0010_000 - 8);
    m60Incremented += 4L << (aqq << 3);  //longのシフトカウントは6bitでマスクされる
    int a = m60Address = (XEiJ.regRn[aqq] += 4) - 4;
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMPLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_100_mmm_rrr
  public static void irpMoveToMNLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0010_000 - 8);
    m60Incremented -= 4L << (aqq << 3);  //longのシフトカウントは6bitでマスクされる
    int a = m60Address = XEiJ.regRn[aqq] -= 4;
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMNLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_101_mmm_rrr
  public static void irpMoveToMWLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0010_000 - 8);
    int t = XEiJ.regRn[aqq];  //ベースレジスタ
    int a = m60Address = t + mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //ワードディスプレースメント
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMWLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_qqq_110_mmm_rrr
  public static void irpMoveToMXLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0010_000 - 8);
    int t = XEiJ.regRn[aqq];  //ベースレジスタ
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //拡張ワード
    int a;
    if ((0x0100 & w) == 0) {  //ブリーフフォーマット
      a = m60Address =
        (t  //ベースレジスタ
         + (byte) w  //バイトディスプレースメント
         + (((0x0800 & w) == 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
             XEiJ.regRn[w >> 12])  //ロングインデックス
            << ((0x0600 & w) >> 9)));  //スケールファクタ
    } else {  //フルフォーマット
      XEiJ.mpuCycleCount += ((0x0003 & w) == 0 ? 1 :  //インダイレクトなし
                             3);  //インダイレクトあり
      t = (((0x0080 & w) != 0 ? 0 :  //ベースレジスタなし
            t) +  //ベースレジスタあり
           ((0x0020 & w) == 0 ? 0 :  //ベースディスプレースメントなし
            (0x0010 & w) == 0 ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) :  //ワードベースディスプレースメント
            mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS)));  //ロングベースディスプレースメント
      int x = ((0x0040 & w) != 0 ? 0 :  //インデックスなし
               ((0x0800 & w) == 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
                XEiJ.regRn[w >> 12])  //ロングインデックス
               << ((0x0600 & w) >> 9));  //スケールファクタ
      a = m60Address =
        ((0x0003 & w) == 0 ? t + x :  //インダイレクトなし
         (((0x0004 & w) == 0 ? mmuReadLongData (m60Address = t + x, XEiJ.regSRS) :  //プリインデックス
           mmuReadLongData (m60Address = t, XEiJ.regSRS) + x)  //ポストインデックス
          + ((0x0002 & w) == 0 ? 0 :  //アウタディスプレースメントなし
             (0x0001 & w) == 0 ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) :  //ワードアウタディスプレースメント
             mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS))));  //ロングアウタディスプレースメント
    }
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToMXLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_000_111_mmm_rrr
  public static void irpMoveToZWLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ
    int a = m60Address = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToZWLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0010_001_111_mmm_rrr
  public static void irpMoveToZLLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ
    int a = m60Address = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
    mmuWriteLongData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveToZLLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,Dq                                  |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_000_mmm_rrr
  public static void irpMoveToDRWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //pcws。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regRn[qqq] = ~65535 & XEiJ.regRn[qqq] | (char) z;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToDRWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEA.W <ea>,Aq                                 |-|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr
  //MOVE.W <ea>,Aq                                  |A|012346|-|-----|-----|DAM+-WXZPI|0011_qqq_001_mmm_rrr [MOVEA.W <ea>,Aq]
  //
  //MOVEA.W <ea>,Aq
  //  ワードデータをロングに符号拡張してAqの全体を更新する
  public static void irpMoveaWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //符号拡張して32bit全部書き換える。pcws。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。右辺でAqが変化する可能性があることに注意
  }  //irpMoveaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(Aq)                                |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_010_mmm_rrr
  public static void irpMoveToMMWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //pcws。イミディエイトを分離。このr[ea]はデータレジスタまたはアドレスレジスタ。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0011_000 - 8);
    int a = m60Address = XEiJ.regRn[aqq];
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMMWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(Aq)+                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_011_mmm_rrr
  public static void irpMoveToMPWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0011_000 - 8);
    m60Incremented += 2L << (aqq << 3);  //longのシフトカウントは6bitでマスクされる
    int a = m60Address = (XEiJ.regRn[aqq] += 2) - 2;
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMPWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,-(Aq)                               |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_100_mmm_rrr
  public static void irpMoveToMNWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0011_000 - 8);
    m60Incremented -= 2L << (aqq << 3);  //longのシフトカウントは6bitでマスクされる
    int a = m60Address = XEiJ.regRn[aqq] -= 2;
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMNWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(d16,Aq)                            |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_101_mmm_rrr
  public static void irpMoveToMWWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0011_000 - 8);
    int t = XEiJ.regRn[aqq];  //ベースレジスタ
    int a = m60Address = t + mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //ワードディスプレースメント
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMWWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(d8,Aq,Rn.wl)                       |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_qqq_110_mmm_rrr
  public static void irpMoveToMXWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int aqq = (XEiJ.regOC >> 9) - (0b0011_000 - 8);
    int t = XEiJ.regRn[aqq];  //ベースレジスタ
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //拡張ワード
    int a;
    if ((0x0100 & w) == 0) {  //ブリーフフォーマット
      a = m60Address =
        (t  //ベースレジスタ
         + (byte) w  //バイトディスプレースメント
         + (((0x0800 & w) == 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
             XEiJ.regRn[w >> 12])  //ロングインデックス
            << ((0x0600 & w) >> 9)));  //スケールファクタ
    } else {  //フルフォーマット
      XEiJ.mpuCycleCount += ((0x0003 & w) == 0 ? 1 :  //インダイレクトなし
                             3);  //インダイレクトあり
      t = (((0x0080 & w) != 0 ? 0 :  //ベースレジスタなし
            t) +  //ベースレジスタあり
           ((0x0020 & w) == 0 ? 0 :  //ベースディスプレースメントなし
            (0x0010 & w) == 0 ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) :  //ワードベースディスプレースメント
            mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS)));  //ロングベースディスプレースメント
      int x = ((0x0040 & w) != 0 ? 0 :  //インデックスなし
               ((0x0800 & w) == 0 ? (short) XEiJ.regRn[w >> 12] :  //ワードインデックス
                XEiJ.regRn[w >> 12])  //ロングインデックス
               << ((0x0600 & w) >> 9));  //スケールファクタ
      a = m60Address =
        ((0x0003 & w) == 0 ? t + x :  //インダイレクトなし
         (((0x0004 & w) == 0 ? mmuReadLongData (m60Address = t + x, XEiJ.regSRS) :  //プリインデックス
           mmuReadLongData (m60Address = t, XEiJ.regSRS) + x)  //ポストインデックス
          + ((0x0002 & w) == 0 ? 0 :  //アウタディスプレースメントなし
             (0x0001 & w) == 0 ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) :  //ワードアウタディスプレースメント
             mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS))));  //ロングアウタディスプレースメント
    }
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToMXWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(xxx).W                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_000_111_mmm_rrr
  public static void irpMoveToZWWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離
    int a = m60Address = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToZWWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,(xxx).L                             |-|012346|-|-UUUU|-**00|DAM+-WXZPI|0011_001_111_mmm_rrr
  public static void irpMoveToZLWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離
    int a = m60Address = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
    mmuWriteWordData (a, z, XEiJ.regSRS);
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpMoveToZLWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEGX.B <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_000_mmm_rrr
  public static void irpNegxByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEGX.B Dr
      XEiJ.mpuCycleCount++;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (y = XEiJ.regRn[ea]) | 0xff & -(y = (byte) y) - (XEiJ.regCCR >> 4));  //Xの左側はすべて0なのでCCR_X&を省略
    } else {  //NEGX.B <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = (byte) (-(y = mmuModifyByteSignData (a, XEiJ.regSRS)) - (XEiJ.regCCR >> 4)), XEiJ.regSRS);  //Xの左側はすべて0なのでCCR_X&を省略
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_negx
  }  //irpNegxByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEGX.W <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_001_mmm_rrr
  public static void irpNegxWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEGX.W Dr
      XEiJ.mpuCycleCount++;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (y = XEiJ.regRn[ea]) | (char) (-(y = (short) y) - (XEiJ.regCCR >> 4)));  //Xの左側はすべて0なのでCCR_X&を省略
    } else {  //NEGX.W <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z = (short) (-(y = mmuModifyWordSignData (a, XEiJ.regSRS)) - (XEiJ.regCCR >> 4)), XEiJ.regSRS);  //Xの左側はすべて0なのでCCR_X&を省略
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_negx
  }  //irpNegxWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEGX.L <ea>                                     |-|012346|-|*UUUU|*****|D M+-WXZ  |0100_000_010_mmm_rrr
  public static void irpNegxLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEGX.L Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = z = -(y = XEiJ.regRn[ea]) - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
    } else {  //NEGX.L <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = -(y = mmuModifyLongData (a, XEiJ.regSRS)) - (XEiJ.regCCR >> 4), XEiJ.regSRS);  //Xの左側はすべて0なのでCCR_X&を省略
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_negx
  }  //irpNegxLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W SR,<ea>                                  |-|-12346|P|*****|-----|D M+-WXZ  |0100_000_011_mmm_rrr
  public static void irpMoveFromSR () throws M68kException {
    //MC68010以上では特権命令
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //MOVE.W SR,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = ~0xffff & XEiJ.regRn[ea] | XEiJ.regSRT1 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR;
    } else {  //MOVE.W SR,<mem>
      XEiJ.mpuCycleCount++;
      mmuWriteWordData (efaMltWord (ea), XEiJ.regSRT1 | XEiJ.regSRS | XEiJ.regSRM | XEiJ.regSRI | XEiJ.regCCR, 1);
    }
  }  //irpMoveFromSR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CHK.L <ea>,Dq                                   |-|--2346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_100_mmm_rrr
  public static void irpChkLong () throws M68kException {
    XEiJ.mpuCycleCount += 2;
    int ea = XEiJ.regOC & 63;
    int x = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離
    int y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
    int z = x - y;
    XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) |
                   (y < 0 ? XEiJ.REG_CCR_N : 0));
    if (y < 0 || x < y) {
      XEiJ.mpuCycleCount += 20 - 19;
      m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpChkLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CHK.W <ea>,Dq                                   |-|012346|-|-UUUU|-*UUU|D M+-WXZPI|0100_qqq_110_mmm_rrr
  public static void irpChkWord () throws M68kException {
    XEiJ.mpuCycleCount += 2;
    int ea = XEiJ.regOC & 63;
    int x = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //pcws。イミディエイトを分離
    int y = (short) XEiJ.regRn[XEiJ.regOC >> 9 & 7];
    int z = (short) (x - y);
    XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z | XEiJ.REG_CCR_V | XEiJ.REG_CCR_C) |
                   (y < 0 ? XEiJ.REG_CCR_N : 0));
    if (y < 0 || x < y) {
      XEiJ.mpuCycleCount += 20 - 19;
      m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_CHK_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpChkWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //LEA.L <ea>,Aq                                   |-|012346|-|-----|-----|  M  WXZP |0100_qqq_111_mmm_rrr
  //EXTB.L Dr                                       |-|--2346|-|-UUUU|-**00|D         |0100_100_111_000_rrr
  public static void irpLea () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EXTB.L Dr
      XEiJ.mpuCycleCount++;
      int z;
      XEiJ.regRn[ea] = z = (byte) XEiJ.regRn[ea];
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //LEA.L <ea>,Aq
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[(XEiJ.regOC >> 9) - (32 - 8)] = efaLeaPea (ea);
    }
  }  //irpLea

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CLR.B <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_000_mmm_rrr (68000 and 68008 read before clear)
  public static void irpClrByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //CLR.B Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] &= ~0xff;
    } else {  //CLR.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), 0, XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z;  //ccr_clr
  }  //irpClrByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CLR.W <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_001_mmm_rrr (68000 and 68008 read before clear)
  public static void irpClrWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //CLR.W Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] &= ~0xffff;
    } else {  //CLR.W <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteWordData (efaMltWord (ea), 0, XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z;  //ccr_clr
  }  //irpClrWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CLR.L <ea>                                      |-|012346|-|-UUUU|-0100|D M+-WXZ  |0100_001_010_mmm_rrr (68000 and 68008 read before clear)
  public static void irpClrLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //CLR.L Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = 0;
    } else {  //CLR.L <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteLongData (efaMltLong (ea), 0, XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z;  //ccr_clr
  }  //irpClrLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W CCR,<ea>                                 |-|-12346|-|*****|-----|D M+-WXZ  |0100_001_011_mmm_rrr
  public static void irpMoveFromCCR () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //MOVE.W CCR,Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = ~0xffff & XEiJ.regRn[ea] | XEiJ.regCCR;
    } else {  //MOVE.W CCR,<mem>
      XEiJ.mpuCycleCount++;
      mmuWriteWordData (efaMltWord (ea), XEiJ.regCCR, XEiJ.regSRS);
    }
  }  //irpMoveFromCCR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEG.B <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_000_mmm_rrr
  public static void irpNegByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEG.B Dr
      XEiJ.mpuCycleCount++;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (y = XEiJ.regRn[ea]) | 0xff & -(y = (byte) y));
    } else {  //NEG.B <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = (byte) -(y = mmuModifyByteSignData (a, XEiJ.regSRS)), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_neg
  }  //irpNegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEG.W <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_001_mmm_rrr
  public static void irpNegWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEG.W Dr
      XEiJ.mpuCycleCount++;
      z = (short) (XEiJ.regRn[ea] = ~0xffff & (y = XEiJ.regRn[ea]) | (char) -(y = (short) y));
    } else {  //NEG.W <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z = (short) -(y = mmuModifyWordSignData (a, XEiJ.regSRS)), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_neg
  }  //irpNegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NEG.L <ea>                                      |-|012346|-|UUUUU|*****|D M+-WXZ  |0100_010_010_mmm_rrr
  public static void irpNegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y;
    int z;
    if (ea < XEiJ.EA_AR) {  //NEG.L Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = z = -(y = XEiJ.regRn[ea]);
    } else {  //NEG.L <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = -(y = mmuModifyLongData (a, XEiJ.regSRS)), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (y & z) >>> 31 << 1 |
           (y | z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_neg
  }  //irpNegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,CCR                                 |-|012346|-|UUUUU|*****|D M+-WXZPI|0100_010_011_mmm_rrr
  public static void irpMoveToCCR () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.REG_CCR_MASK & (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS));  //pcws。イミディエイトを分離
  }  //irpMoveToCCR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOT.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_000_mmm_rrr
  public static void irpNotByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //NOT.B Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] ^= 255;  //0拡張してからEOR
    } else {  //NOT.B <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = ~mmuModifyByteSignData (a, XEiJ.regSRS), XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpNotByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOT.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_001_mmm_rrr
  public static void irpNotWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //NOT.W Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] ^= 65535;  //0拡張してからEOR
    } else {  //NOT.W <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      mmuWriteWordData (a, z = ~mmuModifyWordSignData (a, XEiJ.regSRS), XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpNotWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOT.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_011_010_mmm_rrr
  public static void irpNotLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //NOT.L Dr
      XEiJ.mpuCycleCount++;
      z = XEiJ.regRn[ea] ^= 0xffffffff;
    } else {  //NOT.L <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      mmuWriteLongData (a, z = ~mmuModifyLongData (a, XEiJ.regSRS), XEiJ.regSRS);
    }
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpNotLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.W <ea>,SR                                  |-|012346|P|UUUUU|*****|D M+-WXZPI|0100_011_011_mmm_rrr
  public static void irpMoveToSR () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 12;
    int ea = XEiJ.regOC & 63;
    irpSetSR (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, 1) : mmuReadWordZeroData (efaAnyWord (ea), 1));  //特権違反チェックが先。pcwz。イミディエイトを分離
  }  //irpMoveToSR

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NBCD.B <ea>                                     |-|012346|-|UUUUU|*U*U*|D M+-WXZ  |0100_100_000_mmm_rrr
  //LINK.L Ar,#<data>                               |-|--2346|-|-----|-----|          |0100_100_000_001_rrr-{data}
  //
  //LINK.L Ar,#<data>
  //  PEA.L (Ar);MOVEA.L A7,Ar;ADDA.L #<data>,A7と同じ
  //  LINK.L A7,#<data>はA7をデクリメントする前の値がプッシュされ、A7に#<data>が加算される
  public static void irpNbcd () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //NBCD.B Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = ~0xff & XEiJ.regRn[ea] | irpSbcd (0, XEiJ.regRn[ea]);
    } else if (ea < XEiJ.EA_MM) {  //LINK.L Ar,#<data>
      XEiJ.mpuCycleCount += 2;
      int o = mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);  //pcls
      int arr = XEiJ.regOC - (0b0100_100_000_001_000 - 8);
      //評価順序に注意。LINK.L A7,#<data>のときプッシュするのはA7をデクリメントする前の値。wl(r[15]-=4,r[8+rrr])は不可
      int a = XEiJ.regRn[arr];
      m60Incremented -= 4L << (7 << 3);
      int sp = m60Address = XEiJ.regRn[15] -= 4;
      mmuWriteLongData (sp, a, XEiJ.regSRS);  //pushl
      XEiJ.regRn[arr] = sp;
      XEiJ.regRn[15] = sp + o;
    } else {  //NBCD.B <mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, irpSbcd (0, mmuModifyByteSignData (a, XEiJ.regSRS)), XEiJ.regSRS);
    }
  }  //irpNbcd

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SWAP.W Dr                                       |-|012346|-|-UUUU|-**00|D         |0100_100_001_000_rrr
  //BKPT #<data>                                    |-|-12346|-|-----|-----|          |0100_100_001_001_ddd
  //PEA.L <ea>                                      |-|012346|-|-----|-----|  M  WXZP |0100_100_001_mmm_rrr
  public static void irpPea () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //SWAP.W Dr
      XEiJ.mpuCycleCount++;
      int x;
      int z;
      XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) << 16 | x >>> 16;
      //上位ワードと下位ワードを入れ替えた後のDrをロングでテストする
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //PEA.L <ea>
      XEiJ.mpuCycleCount++;
      //評価順序に注意。実効アドレスを求めてからspをデクリメントすること
      int a = efaLeaPea (ea);  //BKPT #<data>はここでillegal instructionになる
      m60Incremented -= 4L << (7 << 3);
      int sp = m60Address = XEiJ.regRn[15] -= 4;
      mmuWriteLongData (sp, a, XEiJ.regSRS);
    }
  }  //irpPea

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXT.W Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_010_000_rrr
  //MOVEM.W <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_010_mmm_rrr-llllllllllllllll
  public static void irpMovemToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EXT.W Dr
      XEiJ.mpuCycleCount++;
      int z;
      XEiJ.regRn[ea] = ~0xffff & (z = XEiJ.regRn[ea]) | (char) (z = (byte) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //MOVEM.W <list>,<ea>
      int l = mmuReadWordZeroExword (XEiJ.regPC, XEiJ.regSRS);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      if (ea >> 3 == XEiJ.MMM_MN) {  //-(Ar)
        //MOVEM.wl <list>,-(Ar)で<list>にArが含まれているとき、000/010は命令開始時のArを、020/030/040/060は命令開始時のAr-オペレーションサイズをメモリに書き込む
        //転送するレジスタが0個のときArは変化しない
        int arr = ea - (XEiJ.EA_MN - 8);
        m60Incremented -= 2L << (arr << 3);  //longのシフトカウントは6bitでマスクされる
        int a = m60Address = XEiJ.regRn[arr];
        XEiJ.regRn[arr] = a - 2;
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[15], XEiJ.regSRS);
          }
          if ((l & 0x0002) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[14], XEiJ.regSRS);
          }
          if ((l & 0x0004) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[13], XEiJ.regSRS);
          }
          if ((l & 0x0008) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[12], XEiJ.regSRS);
          }
          if ((l & 0x0010) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[11], XEiJ.regSRS);
          }
          if ((l & 0x0020) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[10], XEiJ.regSRS);
          }
          if ((l & 0x0040) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 9], XEiJ.regSRS);
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 8], XEiJ.regSRS);
          }
          if ((l & 0x0100) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 7], XEiJ.regSRS);
          }
          if ((l & 0x0200) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 6], XEiJ.regSRS);
          }
          if ((l & 0x0400) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 5], XEiJ.regSRS);
          }
          if ((l & 0x0800) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 4], XEiJ.regSRS);
          }
          if ((l & 0x1000) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 3], XEiJ.regSRS);
          }
          if ((l & 0x2000) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 2], XEiJ.regSRS);
          }
          if ((l & 0x4000) != 0) {
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 1], XEiJ.regSRS);
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[ 0], XEiJ.regSRS);
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 15; i >= 0; i--) {
            if ((l & 0x8000 >>> i) != 0) {
              mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[i], XEiJ.regSRS);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 15; l != 0; i--, l <<= 1) {
            if (l < 0) {
              mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[i], XEiJ.regSRS);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 15; l != 0; i--, l >>>= 1) {
            if ((l & 1) != 0) {
              mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[i], XEiJ.regSRS);
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 15; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            mmuWriteWordData (m60Address = a -= 2, XEiJ.regRn[i -= k], XEiJ.regSRS);
            l = l >>> k & ~1;
          }
        }
        m60Incremented += 2L << (arr << 3);  //元に戻しておく。longのシフトカウントは6bitでマスクされる
        XEiJ.regRn[arr] = a;
        XEiJ.mpuCycleCount += t - a >> 1;  //2バイト/個→1サイクル/個
      } else {  //-(Ar)以外
        int a = efaCltWord (ea);
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 0], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0002) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 1], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0004) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 2], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0008) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 3], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0010) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 4], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0020) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 5], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0040) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 6], XEiJ.regSRS);
            a += 2;
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 7], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0100) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 8], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0200) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[ 9], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0400) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[10], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x0800) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[11], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x1000) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[12], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x2000) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[13], XEiJ.regSRS);
            a += 2;
          }
          if ((l & 0x4000) != 0) {
            mmuWriteWordData (m60Address = a, XEiJ.regRn[14], XEiJ.regSRS);
            a += 2;
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            mmuWriteWordData (m60Address = a, XEiJ.regRn[15], XEiJ.regSRS);
            a += 2;
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 0; i <= 15; i++) {
            if ((l & 0x0001 << i) != 0) {
              mmuWriteWordData (m60Address = a, XEiJ.regRn[i], XEiJ.regSRS);
              a += 2;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 0; l != 0; i++, l <<= 1) {
            if (l < 0) {
              mmuWriteWordData (m60Address = a, XEiJ.regRn[i], XEiJ.regSRS);
              a += 2;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 0; l != 0; i++, l >>>= 1) {
            if ((l & 1) != 0) {
              mmuWriteWordData (m60Address = a, XEiJ.regRn[i], XEiJ.regSRS);
              a += 2;
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 0; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            mmuWriteWordData (m60Address = a, XEiJ.regRn[i += k], XEiJ.regSRS);
            a += 2;
            l = l >>> k & ~1;
          }
        }
        XEiJ.mpuCycleCount += a - t >> 1;  //2バイト/個→1サイクル/個
      }
    }
  }  //irpMovemToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXT.L Dr                                        |-|012346|-|-UUUU|-**00|D         |0100_100_011_000_rrr
  //MOVEM.L <list>,<ea>                             |-|012346|-|-----|-----|  M -WXZ  |0100_100_011_mmm_rrr-llllllllllllllll
  public static void irpMovemToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //EXT.L Dr
      XEiJ.mpuCycleCount++;
      int z;
      XEiJ.regRn[ea] = z = (short) XEiJ.regRn[ea];
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //MOVEM.L <list>,<ea>
      int l = mmuReadWordZeroExword (XEiJ.regPC, XEiJ.regSRS);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      if (ea >> 3 == XEiJ.MMM_MN) {  //-(Ar)
        //MOVEM.wl <list>,-(Ar)で<list>にArが含まれているとき、000/010は命令開始時のArを、020/030/040/060は命令開始時のAr-オペレーションサイズをメモリに書き込む
        //転送するレジスタが0個のときArは変化しない
        int arr = ea - (XEiJ.EA_MN - 8);
        m60Incremented -= 4L << (arr << 3);  //longのシフトカウントは6bitでマスクされる
        int a = m60Address = XEiJ.regRn[arr];
        XEiJ.regRn[arr] = a - 4;
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[15], XEiJ.regSRS);
          }
          if ((l & 0x0002) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[14], XEiJ.regSRS);
          }
          if ((l & 0x0004) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[13], XEiJ.regSRS);
          }
          if ((l & 0x0008) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[12], XEiJ.regSRS);
          }
          if ((l & 0x0010) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[11], XEiJ.regSRS);
          }
          if ((l & 0x0020) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[10], XEiJ.regSRS);
          }
          if ((l & 0x0040) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 9], XEiJ.regSRS);
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 8], XEiJ.regSRS);
          }
          if ((l & 0x0100) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 7], XEiJ.regSRS);
          }
          if ((l & 0x0200) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 6], XEiJ.regSRS);
          }
          if ((l & 0x0400) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 5], XEiJ.regSRS);
          }
          if ((l & 0x0800) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 4], XEiJ.regSRS);
          }
          if ((l & 0x1000) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 3], XEiJ.regSRS);
          }
          if ((l & 0x2000) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 2], XEiJ.regSRS);
          }
          if ((l & 0x4000) != 0) {
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 1], XEiJ.regSRS);
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[ 0], XEiJ.regSRS);
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 15; i >= 0; i--) {
            if ((l & 0x8000 >>> i) != 0) {
              mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[i], XEiJ.regSRS);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 15; l != 0; i--, l <<= 1) {
            if (l < 0) {
              mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[i], XEiJ.regSRS);
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 15; l != 0; i--, l >>>= 1) {
            if ((l & 1) != 0) {
              mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[i], XEiJ.regSRS);
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 15; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            mmuWriteLongData (m60Address = a -= 4, XEiJ.regRn[i -= k], XEiJ.regSRS);
            l = l >>> k & ~1;
          }
        }
        m60Incremented += 4L << (arr << 3);  //元に戻しておく。longのシフトカウントは6bitでマスクされる
        XEiJ.regRn[arr] = a;
        XEiJ.mpuCycleCount += t - a >> 2;  //4バイト/個→1サイクル/個
      } else {  //-(Ar)以外
        int a = efaCltLong (ea);
        int t = a;
        if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
          if ((l & 0x0001) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 0], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0002) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 1], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0004) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 2], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0008) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 3], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0010) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 4], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0020) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 5], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0040) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 6], XEiJ.regSRS);
            a += 4;
          }
          if ((byte) l < 0) {  //(l & 0x0080) != 0
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 7], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0100) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 8], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0200) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[ 9], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0400) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[10], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x0800) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[11], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x1000) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[12], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x2000) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[13], XEiJ.regSRS);
            a += 4;
          }
          if ((l & 0x4000) != 0) {
            mmuWriteLongData (m60Address = a, XEiJ.regRn[14], XEiJ.regSRS);
            a += 4;
          }
          if ((short) l < 0) {  //(l & 0x8000) != 0
            mmuWriteLongData (m60Address = a, XEiJ.regRn[15], XEiJ.regSRS);
            a += 4;
          }
        } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
          for (int i = 0; i <= 15; i++) {
            if ((l & 0x0001 << i) != 0) {
              mmuWriteLongData (m60Address = a, XEiJ.regRn[i], XEiJ.regSRS);
              a += 4;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
          l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
          for (int i = 0; l != 0; i++, l <<= 1) {
            if (l < 0) {
              mmuWriteLongData (m60Address = a, XEiJ.regRn[i], XEiJ.regSRS);
              a += 4;
            }
          }
        } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
          for (int i = 0; l != 0; i++, l >>>= 1) {
            if ((l & 1) != 0) {
              mmuWriteLongData (m60Address = a, XEiJ.regRn[i], XEiJ.regSRS);
              a += 4;
            }
          }
        } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
          for (int i = 0; l != 0; ) {
            int k = Integer.numberOfTrailingZeros (l);
            mmuWriteLongData (m60Address = a, XEiJ.regRn[i += k], XEiJ.regSRS);
            a += 4;
            l = l >>> k & ~1;
          }
        }
        XEiJ.mpuCycleCount += a - t >> 2;  //4バイト/個→1サイクル/個
      }
    }
  }  //irpMovemToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TST.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_000_mmm_rrr
  //TST.B <ea>                                      |-|--2346|-|-UUUU|-**00|        PI|0100_101_000_mmm_rrr
  public static void irpTstByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS))];  //ccr_tst_byte。pcbs。イミディエイトを分離。アドレッシングモードに注意
  }  //irpTstByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TST.W <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_001_mmm_rrr
  //TST.W <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_001_mmm_rrr
  public static void irpTstWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //pcws。イミディエイトを分離。アドレッシングモードに注意。このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpTstWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TST.L <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_010_mmm_rrr
  //TST.L <ea>                                      |-|--2346|-|-UUUU|-**00| A      PI|0100_101_010_mmm_rrr
  public static void irpTstLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離。アドレッシングモードに注意。このr[ea]はデータレジスタまたはアドレスレジスタ
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpTstLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TAS.B <ea>                                      |-|012346|-|-UUUU|-**00|D M+-WXZ  |0100_101_011_mmm_rrr
  //ILLEGAL                                         |-|012346|-|-----|-----|          |0100_101_011_111_100
  public static void irpTas () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z;
    if (ea < XEiJ.EA_AR) {  //TAS.B Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] = 0x80 | (z = XEiJ.regRn[ea]);
    } else {  //TAS.B <mem>
      XEiJ.mpuCycleCount += 17;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, 0x80 | (z = mmuModifyByteSignData (a, XEiJ.regSRS)), XEiJ.regSRS);
    }
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
  }  //irpTas

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MULU.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll000000000hhh        (h is not used)
  //MULU.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll010000000hhh        (if h=l then result is not defined)
  //MULS.L <ea>,Dl                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll100000000hhh        (h is not used)
  //MULS.L <ea>,Dh:Dl                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_000_mmm_rrr-0lll110000000hhh        (if h=l then result is not defined)
  public static void irpMuluMulsLong () throws M68kException {
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz。拡張ワード
    if ((w & ~0b0111_110_000_000_111) != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if ((w & 0b0000_010_000_000_000) != 0) {  //64bit積
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    //32bit積
    int s = w & 0b0000_100_000_000_000;  //0=MULU,1=MULS
    int l = w >> 12;  //被乗数,積
    XEiJ.mpuCycleCount += 2;
    int ea = XEiJ.regOC & 63;
    long yy = (long) (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS));  //pcls。イミディエイトを分離
    long xx = (long) XEiJ.regRn[l];
    if (s == 0) {  //MULU
      long zz = (0xffffffffL & xx) * (0xffffffffL & yy);
      int z = XEiJ.regRn[l] = (int) zz;
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | (zz >>> 32 != 0L ? XEiJ.REG_CCR_V : 0);
    } else {  //MULS
      long zz = xx * yy;
      int z = XEiJ.regRn[l] = (int) zz;
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | (z != zz ? XEiJ.REG_CCR_V : 0);
    }
  }  //irpMuluMulsLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //DIVU.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000qqq
  //DIVUL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq000000000rrr        (q is not equal to r)
  //DIVU.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq010000000rrr        (q is not equal to r)
  //DIVS.L <ea>,Dq                                  |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000qqq
  //DIVSL.L <ea>,Dr:Dq                              |-|--2346|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq100000000rrr        (q is not equal to r)
  //DIVS.L <ea>,Dr:Dq                               |-|--234S|-|-UUUU|-***0|D M+-WXZPI|0100_110_001_mmm_rrr-0qqq110000000rrr        (q is not equal to r)
  //
  //DIVS.L <ea>,Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit商Dq
  //
  //DIVS.L <ea>,Dr:Dq
  //  64bit被除数Dr:Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  //  M68000PRMでDIVS.Lのアドレッシングモードがデータ可変と書かれているのはデータの間違い
  //
  //DIVSL.L <ea>,Dr:Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  //
  //DIVU.L <ea>,Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit商Dq
  //
  //DIVU.L <ea>,Dr:Dq
  //  64bit被除数Dr:Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  //
  //DIVUL.L <ea>,Dr:Dq
  //  32bit被除数Dq/32bit除数<ea>→32bit余りDr:32bit商Dq
  public static void irpDivuDivsLong () throws M68kException {
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcwz。拡張ワード
    if ((w & ~0b0111_110_000_000_111) != 0) {
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    if ((w & 0b0000_010_000_000_000) != 0) {  //64bit被除数
      M68kException.m6eNumber = M68kException.M6E_UNIMPLEMENTED_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
    //32bit被除数
    int s = w & 0b0000_100_000_000_000;  //0=DIVU,1=DIVS
    int h = w & 7;  //余り
    int l = w >> 12;  //被除数,商
    int ea = XEiJ.regOC & 63;
    int y = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //除数。pcls。イミディエイトを分離
    if (s == 0) {  //符号なし。DIVU.L <ea>,*
      XEiJ.mpuCycleCount += 38;  //最大
      long yy = (long) y & 0xffffffffL;  //除数
      if (y == 0) {  //ゼロ除算
        XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z | XEiJ.REG_CCR_V)  //XとNとZとVは変化しない
                       );  //Cは常にクリア
        XEiJ.mpuCycleCount += 38 - 34;
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
        throw M68kException.m6eSignal;
      }  //if ゼロ除算
      long xx = (long) XEiJ.regRn[l] & 0xffffffffL;  //被除数
      long zz = (long) ((double) xx / (double) yy);  //double→intのキャストは飽和変換で0xffffffff/0x00000001が0x7fffffffになってしまうのでdouble→longとする
      int z = XEiJ.regRn[l] = (int) zz;  //商
      if (h != l) {
        XEiJ.regRn[h] = (int) (xx - yy * zz);  //余り
      }
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                     (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                     );  //VとCは常にクリア
    } else {  //符号あり。DIVS.L <ea>,*
      XEiJ.mpuCycleCount += 38;  //最大
      long yy = (long) y;  //除数
      if (y == 0) {  //ゼロ除算
        XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z | XEiJ.REG_CCR_V)  //XとNとZとVは変化しない
                       );  //Cは常にクリア
        XEiJ.mpuCycleCount += 38 - 34;
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
        throw M68kException.m6eSignal;
      }  //if ゼロ除算
      long xx = (long) XEiJ.regRn[l];  //被除数
      long zz = xx / yy;  //商
      if ((int) zz != zz) {  //オーバーフローあり
        //Dqは変化しない
        XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z) |  //XとNとZは変化しない
                       XEiJ.REG_CCR_V  //Vは常にセット
                       );  //Cは常にクリア
      } else {  //オーバーフローなし
        int z = XEiJ.regRn[l] = (int) zz;  //商
        if (h != l) {  //DIVSL.L <ea>,Dr:Dq
          XEiJ.regRn[h] = (int) (xx - yy * zz);  //余り
        }
        XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                       (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                       (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                       );  //VとCは常にクリア
      }  //if オーバーフローあり/オーバーフローなし
    }  //if 符号なし/符号あり
  }  //irpDivuDivsLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SATS.L Dr                                       |-|------|-|-UUUU|-**00|D         |0100_110_010_000_rrr (ISA_B)
  //MOVEM.W <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_010_mmm_rrr-llllllllllllllll
  //
  //SATS.L Dr
  //  VがセットされていたらDrを符号が逆で絶対値が最大の値にする(直前のDrに対する演算を飽和演算にする)
  public static void irpMovemToRegWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_AR) {  //SATS.L Dr
      XEiJ.mpuCycleCount++;
      int z = XEiJ.regRn[ea];
      if (XEiJ.TEST_BIT_1_SHIFT ? XEiJ.regCCR << 31 - 1 < 0 : (XEiJ.regCCR & XEiJ.REG_CCR_V) != 0) {  //Vがセットされているとき
        XEiJ.regRn[ea] = z = z >> 31 ^ 0x80000000;  //符号が逆で絶対値が最大の値にする
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else {  //MOVEM.W <ea>,<list>
      int l = mmuReadWordZeroExword (XEiJ.regPC, XEiJ.regSRS);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      int arr, a;
      if (ea >> 3 == XEiJ.MMM_MP) {  //(Ar)+
        arr = ea - (XEiJ.EA_MP - 8);
        a = m60Address = XEiJ.regRn[arr];
      } else {  //(Ar)+以外
        arr = 16;
        a = efaCntWord (ea);
      }
      int t = a;
      if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
        if ((l & 0x0001) != 0) {
          XEiJ.regRn[ 0] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0002) != 0) {
          XEiJ.regRn[ 1] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0004) != 0) {
          XEiJ.regRn[ 2] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0008) != 0) {
          XEiJ.regRn[ 3] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0010) != 0) {
          XEiJ.regRn[ 4] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0020) != 0) {
          XEiJ.regRn[ 5] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0040) != 0) {
          XEiJ.regRn[ 6] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((byte) l < 0) {  //(l & 0x0080) != 0
          XEiJ.regRn[ 7] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //データレジスタも符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0100) != 0) {
          XEiJ.regRn[ 8] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0200) != 0) {
          XEiJ.regRn[ 9] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0400) != 0) {
          XEiJ.regRn[10] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x0800) != 0) {
          XEiJ.regRn[11] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x1000) != 0) {
          XEiJ.regRn[12] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x2000) != 0) {
          XEiJ.regRn[13] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((l & 0x4000) != 0) {
          XEiJ.regRn[14] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
        if ((short) l < 0) {  //(l & 0x8000) != 0
          XEiJ.regRn[15] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //符号拡張して32bit全部書き換える
          a += 2;
        }
      } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
        for (int i = 0; i <= 15; i++) {
          if ((l & 0x0001 << i) != 0) {
            XEiJ.regRn[i] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //(データレジスタも)符号拡張して32bit全部書き換える
            a += 2;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
        l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
        for (int i = 0; l != 0; i++, l <<= 1) {
          if (l < 0) {
            XEiJ.regRn[i] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //(データレジスタも)符号拡張して32bit全部書き換える
            a += 2;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
        for (int i = 0; l != 0; i++, l >>>= 1) {
          if ((l & 1) != 0) {
            XEiJ.regRn[i] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //(データレジスタも)符号拡張して32bit全部書き換える
            a += 2;
          }
        }
      } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
        for (int i = 0; l != 0; ) {
          int k = Integer.numberOfTrailingZeros (l);
          XEiJ.regRn[i += k] = mmuReadWordSignData (m60Address = a, XEiJ.regSRS);  //(データレジスタも)符号拡張して32bit全部書き換える
          a += 2;
          l = l >>> k & ~1;
        }
      }
      //MOVEM.W (Ar)+,<list>で<list>にArが含まれているとき、メモリから読み出したデータを捨ててArをインクリメントする
      XEiJ.regRn[arr] = a;
      XEiJ.mpuCycleCount += a - t >> 1;  //2バイト/個→1サイクル/個
    }
  }  //irpMovemToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEM.L <ea>,<list>                             |-|012346|-|-----|-----|  M+ WXZP |0100_110_011_mmm_rrr-llllllllllllllll
  public static void irpMovemToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    {
      int l = mmuReadWordZeroExword (XEiJ.regPC, XEiJ.regSRS);  //pcwze。レジスタリスト。ゼロ拡張
      XEiJ.regPC += 2;
      int arr, a;
      if (ea >> 3 == XEiJ.MMM_MP) {  //(Ar)+
        arr = ea - (XEiJ.EA_MP - 8);
        a = m60Address = XEiJ.regRn[arr];
      } else {  //(Ar)+以外
        arr = 16;
        a = efaCntLong (ea);
      }
      int t = a;
      if (XEiJ.IRP_MOVEM_EXPAND) {  //16回展開する
        if ((l & 0x0001) != 0) {
          XEiJ.regRn[ 0] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0002) != 0) {
          XEiJ.regRn[ 1] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0004) != 0) {
          XEiJ.regRn[ 2] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0008) != 0) {
          XEiJ.regRn[ 3] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0010) != 0) {
          XEiJ.regRn[ 4] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0020) != 0) {
          XEiJ.regRn[ 5] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0040) != 0) {
          XEiJ.regRn[ 6] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((byte) l < 0) {  //(l & 0x0080) != 0
          XEiJ.regRn[ 7] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0100) != 0) {
          XEiJ.regRn[ 8] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0200) != 0) {
          XEiJ.regRn[ 9] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0400) != 0) {
          XEiJ.regRn[10] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x0800) != 0) {
          XEiJ.regRn[11] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x1000) != 0) {
          XEiJ.regRn[12] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x2000) != 0) {
          XEiJ.regRn[13] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((l & 0x4000) != 0) {
          XEiJ.regRn[14] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
        if ((short) l < 0) {  //(l & 0x8000) != 0
          XEiJ.regRn[15] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
        }
      } else if (XEiJ.IRP_MOVEM_LOOP) {  //16回ループする。コンパイラが展開する
        for (int i = 0; i <= 15; i++) {
          if ((l & 0x0001 << i) != 0) {
            XEiJ.regRn[i] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
            a += 4;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_LEFT) {  //0になるまで左にシフトする
        l = XEiJ.MPU_BITREV_TABLE_0[l & 2047] | XEiJ.MPU_BITREV_TABLE_1[l << 10 >>> 21];  //Integer.reverse(l)
        for (int i = 0; l != 0; i++, l <<= 1) {
          if (l < 0) {
            XEiJ.regRn[i] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
            a += 4;
          }
        }
      } else if (XEiJ.IRP_MOVEM_SHIFT_RIGHT) {  //0になるまで右にシフトする
        for (int i = 0; l != 0; i++, l >>>= 1) {
          if ((l & 1) != 0) {
            XEiJ.regRn[i] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
            a += 4;
          }
        }
      } else if (XEiJ.IRP_MOVEM_ZEROS) {  //Integer.numberOfTrailingZerosを使う
        for (int i = 0; l != 0; ) {
          int k = Integer.numberOfTrailingZeros (l);
          XEiJ.regRn[i += k] = mmuReadLongData (m60Address = a, XEiJ.regSRS);
          a += 4;
          l = l >>> k & ~1;
        }
      }
      //MOVEM.L (Ar)+,<list>で<list>にArが含まれているとき、メモリから読み出したデータを捨ててArをインクリメントする
      XEiJ.regRn[arr] = a;
      XEiJ.mpuCycleCount += a - t >> 2;  //4バイト/個→1サイクル/個
    }
  }  //irpMovemToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TRAP #<vector>                                  |-|012346|-|-----|-----|          |0100_111_001_00v_vvv
  public static void irpTrap () throws M68kException {
    irpExceptionFormat0 (XEiJ.regOC - (0b0100_111_001_000_000 - M68kException.M6E_TRAP_0_INSTRUCTION_VECTOR) << 2, XEiJ.regPC);  //pcは次の命令
  }  //irpTrap
  public static void irpTrap15 () throws M68kException {
    if ((XEiJ.regRn[0] & 255) == 0x8e) {  //IOCS _BOOTINF
      MainMemory.mmrCheckHuman ();
    }
    irpExceptionFormat0 (M68kException.M6E_TRAP_15_INSTRUCTION_VECTOR << 2, XEiJ.regPC);  //pcは次の命令
  }  //irpTrap15

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //LINK.W Ar,#<data>                               |-|012346|-|-----|-----|          |0100_111_001_010_rrr-{data}
  //
  //LINK.W Ar,#<data>
  //  PEA.L (Ar);MOVEA.L A7,Ar;ADDA.W #<data>,A7と同じ
  //  LINK.W A7,#<data>はA7をデクリメントする前の値がプッシュされ、A7に#<data>が加算される
  public static void irpLinkWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int o = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
    int arr = XEiJ.regOC - (0b0100_111_001_010_000 - 8);
    //評価順序に注意。LINK.W A7,#<data>のときプッシュするのはA7をデクリメントする前の値。wl(r[15]-=4,r[8+rrr])は不可
    int a = XEiJ.regRn[arr];
    m60Incremented -= 4L << (7 << 3);
    int sp = m60Address = XEiJ.regRn[15] -= 4;
    mmuWriteLongData (sp, a, XEiJ.regSRS);  //pushl
    XEiJ.regRn[arr] = sp;
    XEiJ.regRn[15] = sp + o;
  }  //irpLinkWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //UNLK Ar                                         |-|012346|-|-----|-----|          |0100_111_001_011_rrr
  //
  //UNLK Ar
  //  MOVEA.L Ar,A7;MOVEA.L (A7)+,Arと同じ
  //  UNLK A7はMOVEA.L A7,A7;MOVEA.L (A7)+,A7すなわちMOVEA.L (A7),A7と同じ
  //  ソースオペランドのポストインクリメントはデスティネーションオペランドが評価される前に完了しているとみなされる
  //    例えばMOVE.L (A0)+,(A0)+はMOVE.L (A0),(4,A0);ADDQ.L #8,A0と同じ
  //    MOVEA.L (A0)+,A0はポストインクリメントされたA0が(A0)から読み出された値で上書きされるのでMOVEA.L (A0),A0と同じ
  //  M68000PRMにUNLK Anの動作はAn→SP;(SP)→An;SP+4→SPだと書かれているがこれはn=7の場合に当てはまらない
  //  余談だが68040の初期のマスクセットはUNLK A7を実行すると固まるらしい
  public static void irpUnlk () throws M68kException {
    XEiJ.mpuCycleCount += 2;
    int arr = XEiJ.regOC - (0b0100_111_001_011_000 - 8);
    //評価順序に注意
    int sp = XEiJ.regRn[arr];
    //  UNLK ArはMOVEA.L Ar,A7;MOVEA.L (A7)+,Arと同じ
    //  (A7)+がページフォルトになってリトライするとき
    //    Arはまだ更新されておらず、リトライでMOVEA.L Ar,A7が再実行されるので、A7を巻き戻す必要はない
    m60Incremented += 4L << (7 << 3);  //UNLK A7でページフォルトが発生したときA7が増えすぎないようにする
    XEiJ.regRn[15] = sp + 4;
    XEiJ.regRn[arr] = mmuReadLongData (m60Address = sp, XEiJ.regSRS);  //popls
  }  //irpUnlk

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L Ar,USP                                   |-|012346|P|-----|-----|          |0100_111_001_100_rrr
  public static void irpMoveToUsp () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 2;
    XEiJ.mpuUSP = XEiJ.regRn[XEiJ.regOC - (0b0100_111_001_100_000 - 8)];
  }  //irpMoveToUsp

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVE.L USP,Ar                                   |-|012346|P|-----|-----|          |0100_111_001_101_rrr
  public static void irpMoveFromUsp () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount++;
    XEiJ.regRn[XEiJ.regOC - (0b0100_111_001_101_000 - 8)] = XEiJ.mpuUSP;
  }  //irpMoveFromUsp

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RESET                                           |-|012346|P|-----|-----|          |0100_111_001_110_000
  public static void irpReset () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 45;
    XEiJ.irpReset ();
  }  //irpReset

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //NOP                                             |-|012346|-|-----|-----|          |0100_111_001_110_001
  public static void irpNop () throws M68kException {
    XEiJ.mpuCycleCount += 9;
    //何もしない
  }  //irpNop

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //STOP #<data>                                    |-|012346|P|UUUUU|*****|          |0100_111_001_110_010-{data}
  //
  //STOP #<data>
  //    1. #<data>をsrに設定する
  //    2. pcを進める
  //    3. 以下のいずれかの条件が成立するまで停止する
  //      3a. トレース
  //      3b. マスクされているレベルよりも高い割り込み要求
  //      3c. リセット
  //  コアと一緒にデバイスを止めるわけにいかないので、ここでは条件が成立するまで同じ命令を繰り返すループ命令として実装する
  public static void irpStop () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount++;
    irpSetSR (mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, 1));  //pcws。特権違反チェックが先
    if (XEiJ.mpuTraceFlag == 0) {  //トレースまたはマスクされているレベルよりも高い割り込み要求がない
      XEiJ.regPC = XEiJ.regPC0;  //ループ
      //任意の負荷率を100%に設定しているときSTOP命令が軽すぎると動作周波数が大きくなりすぎて割り込みがかかったとき次に進めなくなる
      //負荷率の計算にSTOP命令で止まっていた時間を含めないことにする
      XEiJ.mpuClockTime += XEiJ.TMR_FREQ * 4 / 1000000;  //4μs。50MHzのとき200clk
      XEiJ.mpuLastNano += 4000L;
    }
  }  //irpStop

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTE                                             |-|012346|P|UUUUU|*****|          |0100_111_001_110_011
  public static void irpRte () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    XEiJ.mpuCycleCount += 17;
    int sp = XEiJ.regRn[15];
    int newSR = mmuReadWordZeroData (m60Address = sp, 1);  //popwz
    int newPC = mmuReadLongData (m60Address = sp + 2, 1);  //popls
    int format = mmuReadWordZeroData (m60Address = sp + 6, 1) >> 12;
    if (format == 0) {  //010,020,030,040,060
      m60Incremented += 8L << (7 << 3);
      XEiJ.regRn[15] = sp + 8;
    } else if (format == 2 ||  //020,030,040,060
               format == 3) {  //040,060
      m60Incremented += 12L << (7 << 3);
      XEiJ.regRn[15] = sp + 12;
    } else if (format == 4) {  //060
      m60Incremented += 16L << (7 << 3);
      XEiJ.regRn[15] = sp + 16;
    } else {
      M68kException.m6eNumber = M68kException.M6E_FORMAT_ERROR;
      throw M68kException.m6eSignal;
    }
    //irpSetSRでモードが切り替わる場合があるのでその前にr[15]を更新しておくこと
    irpSetSR (newSR);  //ここでユーザモードに戻る場合がある。特権違反チェックが先
    irpSetPC (newPC);  //分岐ログが新しいsrを使う。順序に注意。ここでアドレスエラーが発生する場合がある
  }  //irpRte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTD #<data>                                     |-|-12346|-|-----|-----|          |0100_111_001_110_100-{data}
  public static void irpRtd () throws M68kException {
    XEiJ.mpuCycleCount += 7;
    int sp = XEiJ.regRn[15];
    int o = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
    int pc = mmuReadLongData (m60Address = sp, XEiJ.regSRS);  //popls
    m60Incremented += 4L << (7 << 3);
    XEiJ.regRn[15] = sp + 4 + o;
    irpSetPC (pc);
  }  //irpRtd

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTS                                             |-|012346|-|-----|-----|          |0100_111_001_110_101
  public static void irpRts () throws M68kException {
    XEiJ.mpuCycleCount += 7;
    int sp = XEiJ.regRn[15];
    int pc = mmuReadLongData (m60Address = sp, XEiJ.regSRS);  //popls
    m60Incremented += 4L << (7 << 3);
    XEiJ.regRn[15] = sp + 4;
    irpSetPC (pc);
  }  //irpRts

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //TRAPV                                           |-|012346|-|---*-|-----|          |0100_111_001_110_110
  public static void irpTrapv () throws M68kException {
    if (XEiJ.TEST_BIT_1_SHIFT ? XEiJ.regCCR << 31 - 1 >= 0 : (XEiJ.regCCR & XEiJ.REG_CCR_V) == 0) {  //通過
      XEiJ.mpuCycleCount++;
    } else {
      m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpTrapv

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //RTR                                             |-|012346|-|UUUUU|*****|          |0100_111_001_110_111
  public static void irpRtr () throws M68kException {
    XEiJ.mpuCycleCount += 8;
    int sp = XEiJ.regRn[15];
    int w = mmuReadWordZeroData (m60Address = sp, XEiJ.regSRS);  //popwz
    int pc = mmuReadLongData (m60Address = sp + 2, XEiJ.regSRS);  //popls
    XEiJ.regCCR = XEiJ.REG_CCR_MASK & w;
    m60Incremented += 6L << (7 << 3);
    XEiJ.regRn[15] = sp + 6;
    irpSetPC (pc);
  }  //irpRtr

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEC.L Rc,Rn                                   |-|-12346|P|-----|-----|          |0100_111_001_111_010-rnnncccccccccccc
  public static void irpMovecFromControl () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, 1);  //pcwz。拡張ワード
    switch (w & 0x0fff) {
    case 0x000:  //SFC
      XEiJ.mpuCycleCount += 12;
      XEiJ.regRn[w >> 12] = XEiJ.mpuSFC;
      break;
    case 0x001:  //DFC
      XEiJ.mpuCycleCount += 12;
      XEiJ.regRn[w >> 12] = XEiJ.mpuDFC;
      break;
    case 0x002:  //CACR
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = XEiJ.mpuCACR & 0xf880e000;  //CABCとCUBCのリードは常に0
      break;
    case 0x003:  //TCR
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetTCR ();
      break;
    case 0x004:  //ITT0
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetITT0 ();
      break;
    case 0x005:  //ITT1
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetITT1 ();
      break;
    case 0x006:  //DTT0
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetDTT0 ();
      break;
    case 0x007:  //DTT1
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetDTT1 ();
      break;
    case 0x008:  //BUSCR
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = XEiJ.mpuBUSCR;
      break;
    case 0x800:  //USP
      XEiJ.mpuCycleCount += 12;
      XEiJ.regRn[w >> 12] = XEiJ.mpuUSP;
      break;
    case 0x801:  //VBR
      XEiJ.mpuCycleCount += 12;
      XEiJ.regRn[w >> 12] = XEiJ.mpuVBR;
      break;
    case 0x806:  //URP
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetURP ();;
      break;
    case 0x807:  //SRP
      XEiJ.mpuCycleCount += 15;
      XEiJ.regRn[w >> 12] = mmuGetSRP ();;
      break;
    case 0x808:  //PCR
      XEiJ.mpuCycleCount += 12;
      XEiJ.regRn[w >> 12] = XEiJ.mpuPCR;
      break;
    default:
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpMovecFromControl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MOVEC.L Rn,Rc                                   |-|-12346|P|-----|-----|          |0100_111_001_111_011-rnnncccccccccccc
  public static void irpMovecToControl () throws M68kException {
    if (XEiJ.regSRS == 0) {  //ユーザモードのとき
      M68kException.m6eNumber = M68kException.M6E_PRIVILEGE_VIOLATION;
      throw M68kException.m6eSignal;
    }
    //以下はスーパーバイザモード
    int w = mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, 1);  //pcwz。拡張ワード
    int d = XEiJ.regRn[w >> 12];
    switch (w & 0x0fff) {
    case 0x000:  //SFC
      XEiJ.mpuCycleCount += 11;
      XEiJ.mpuSFC = d & 0x00000007;
      break;
    case 0x001:  //DFC
      XEiJ.mpuCycleCount += 11;
      XEiJ.mpuDFC = d & 0x00000007;
      break;
    case 0x002:  //CACR
      //  CACR
      //   31  30  29  28  27 26 25 24   23   22   21 20 19 18 17 16   15  14  13 12 11 10 9 8  7 6 5 4 3 2 1 0
      //  EDC NAD ESB DPI FOC  0  0  0  EBC CABC CUBC  0  0  0  0  0  EIC NAI FIC  0  0  0 0 0  0 0 0 0 0 0 0 0
      //    bit31  EDC   Enable Data Cache
      //                 データキャッシュ有効
      //                 4ウェイセットアソシアティブ。16バイト/ライン*128ライン/セット*4セット=8KB
      //    bit30  NAD   No Allocate Mode (Data Cache)
      //                 データキャッシュでミスしても新しいキャッシュラインをアロケートしない
      //    bit29  ESB   Enable Store Buffer
      //                 ストアバッファ有効
      //                 ライトスルーおよびキャッシュ禁止インプリサイスのページの書き込みを4エントリ(16バイト)のFIFOバッファで遅延させる
      //                 例えば4の倍数のアドレスから始まる4バイトに連続して書き込むと1回のロングの書き込みにまとめられる
      //    bit28  DPI   Disable CPUSH Invalidation
      //                 CPUSHでプッシュされたキャッシュラインを無効化しない
      //    bit27  FOC   1/2 Cache Operation Mode Enable (Data Cache)
      //                 データキャッシュを1/2キャッシュモードにする
      //    bit23  EBC   Enable Branch Cache
      //                 分岐キャッシュ有効
      //                 256エントリの分岐キャッシュを用いて分岐予測を行う
      //                 正しく予測された分岐は前後の命令に隠れて実質0サイクルで実行される
      //                   MC68060は最大3個の命令(1個の分岐命令と2個の整数命令)を1サイクルで実行できる
      //                   MC68000(10MHz)とMC68060(50MHz)の処理速度の比は局所的に100倍を超えることがある
      //    bit22  CABC  Clear All Entries in the Branch Cache
      //                 分岐キャッシュのすべてのエントリをクリアする
      //                 分岐命令以外の場所で分岐キャッシュがヒットしてしまったときに発生する分岐予測エラーから復帰するときに使う
      //                 CABCはライトオンリーでリードは常に0
      //    bit21  CUBC  Clear All User Entries in the Branch Cache
      //                 分岐キャッシュのすべてのユーザエントリをクリアする
      //                 CUBCはライトオンリーでリードは常に0
      //    bit15  EIC   Enable Instruction Cache
      //                 命令キャッシュ有効
      //                 4ウェイセットアソシアティブ。16バイト/ライン*128ライン/セット*4セット=8KB
      //    bit14  NAI   No Allocate Mode (Instruction Cache)
      //                 命令キャッシュでミスしても新しいキャッシュラインをアロケートしない
      //    bit13  FIC   1/2 Cache Operation Mode Enable (Instruction Cache)
      //                 命令キャッシュを1/2キャッシュモードにする
      //! 非対応
      XEiJ.mpuCycleCount += 14;
      XEiJ.mpuCACR = d & 0xf8e0e000;  //CABCとCUBCは保存しておいてリードするときにマスクする
      {
        boolean cacheOn = (XEiJ.mpuCACR & 0x80008000) != 0;
        if (XEiJ.mpuCacheOn != cacheOn) {
          XEiJ.mpuCacheOn = cacheOn;
          XEiJ.mpuSetWait ();
        }
      }
      break;
    case 0x003:  //TCR
      //  TCR
      //  31 30 29 28 27 26 25 24  23 22 21 20 19 18 17 16  15 14  13  12   11   10 9 8  7 6   5 4 3 2 1 0
      //   0  0  0  0  0  0  0  0   0  0  0  0  0  0  0  0   E  P NAD NAI FOTC FITC DCO  DUO DWO DCI DUI 0
      //  bit15   E     Enable
      //  bit14   P     Page Size
      //  bit13   NAD   No Allocate Mode (Data ATC)
      //  bit12   NAI   No Allocate Mode (Instruction ATC)
      //  bit11   FOTC  1/2-Cache Mode (Data ATC)
      //  bit10   FITC  1/2-Cache Mode (Instruction ATC)
      //  bit9-8  DCO   Default Cache Mode (Data Cache)
      //  bit7-6  DUO   Default UPA bits (Data Cache)
      //  bit5    DWO   Default Write Protect (Data Cache)
      //  bit4-3  DCI   Default Cache Mode (Instruction Cache)
      //  bit2-1  DUI   Default UPA bits (Instruction Cache)
      //MMUを参照
      XEiJ.mpuCycleCount += 14;
      mmuSetTCR (d);
      break;
    case 0x004:  //ITT0
      XEiJ.mpuCycleCount += 14;
      mmuSetITT0 (d);
      break;
    case 0x005:  //ITT1
      XEiJ.mpuCycleCount += 14;
      mmuSetITT1 (d);
      break;
    case 0x006:  //DTT0
      XEiJ.mpuCycleCount += 14;
      mmuSetDTT0 (d);
      break;
    case 0x007:  //DTT1
      XEiJ.mpuCycleCount += 14;
      mmuSetDTT1 (d);
      break;
    case 0x008:  //BUSCR
      XEiJ.mpuCycleCount += 14;
      XEiJ.mpuBUSCR = d & 0xf0000000;
      break;
    case 0x800:  //USP
      XEiJ.mpuCycleCount += 11;
      XEiJ.mpuUSP = d;
      break;
    case 0x801:  //VBR
      XEiJ.mpuCycleCount += 11;
      XEiJ.mpuVBR = d & -4;  //4の倍数でないと困る
      break;
    case 0x806:  //URP
      XEiJ.mpuCycleCount += 14;
      mmuSetURP (d);
      break;
    case 0x807:  //SRP
      XEiJ.mpuCycleCount += 14;
      mmuSetSRP (d);
      break;
    case 0x808:  //PCR
      //  PCR
      //  31 30 29 28 27 26 25 24  23 22 21 20 19 18 17 16  15 14 13 12 11 10 9 8       7 6 5 4 3 2   1   0
      //   0  0  0  0  0  1  0  0   0  0  1  1  0  0  0  0     Revision Number     EDEBUG  Reserved DFP ESS
      //  bit31-16  Identification   0x0430
      //  bit15-8   Revision Number  1=F43G,5=G65V,6=E41J。偽物もあるらしい
      //  bit7      EDEBUG           Enable Debug Features
      //  bit6-2    Reserved
      //  bit1      DFP              Disable Floating-Point Unit。浮動小数点ユニット無効
      //  bit0      ESS              Enable Superscalar Dispatch。スーパースカラ有効
      XEiJ.mpuCycleCount += 11;
      XEiJ.mpuPCR = 0x04300000 | XEiJ.MPU_060_REV << 8 | d & 0x00000083;
      break;
    default:
      M68kException.m6eNumber = M68kException.M6E_ILLEGAL_INSTRUCTION;
      throw M68kException.m6eSignal;
    }
  }  //irpMovecToControl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //JSR <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_010_mmm_rrr
  //JBSR.L <label>                                  |A|012346|-|-----|-----|          |0100_111_010_111_001-{address}       [JSR <label>]
  public static void irpJsr () throws M68kException {
    XEiJ.mpuCycleCount++;
    //評価順序に注意。実効アドレスを求めてからspをデクリメントすること
    int a = efaJmpJsr (XEiJ.regOC & 63);
    m60Incremented -= 4L << (7 << 3);
    int sp = m60Address = XEiJ.regRn[15] -= 4;
    mmuWriteLongData (sp, XEiJ.regPC, XEiJ.regSRS);  //pushl
    irpSetPC (a);
  }  //irpJsr

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //JMP <ea>                                        |-|012346|-|-----|-----|  M  WXZP |0100_111_011_mmm_rrr
  //JBRA.L <label>                                  |A|012346|-|-----|-----|          |0100_111_011_111_001-{address}       [JMP <label>]
  public static void irpJmp () throws M68kException {
    XEiJ.mpuCycleCount++;  //0clkにしない
    irpSetPC (efaJmpJsr (XEiJ.regOC & 63));
  }  //irpJmp

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_000_mmm_rrr
  //INC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_000_mmm_rrr [ADDQ.B #1,<ea>]
  public static void irpAddqByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    int z;
    if (ea < XEiJ.EA_AR) {  //ADDQ.B #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) + y);
    } else {  //ADDQ.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = (byte) ((x = mmuModifyByteSignData (a, XEiJ.regSRS)) + y), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (~x & z) >>> 31 << 1 |
           (x & ~z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addq
  }  //irpAddqByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_001_mmm_rrr
  //ADDQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_001_001_rrr
  //INC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_001_mmm_rrr [ADDQ.W #1,<ea>]
  //INC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_001_001_rrr [ADDQ.W #1,Ar]
  //
  //ADDQ.W #<data>,Ar
  //  ソースを符号拡張してロングで加算する
  public static void irpAddqWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //ADDQ.W #<data>,Ar
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] += y;  //ロングで計算する。このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //ADDQ.W #<data>,Dr
        XEiJ.mpuCycleCount++;
        z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) + y));
      } else {  //ADDQ.W #<data>,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltWord (ea);
        mmuWriteWordData (a, z = (short) ((x = mmuModifyWordSignData (a, XEiJ.regSRS)) + y), XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (~x & z) >>> 31 << 1 |
             (x & ~z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addq
    }
  }  //irpAddqWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_010_mmm_rrr
  //ADDQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_010_001_rrr
  //INC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_010_mmm_rrr [ADDQ.L #1,<ea>]
  //INC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_010_001_rrr [ADDQ.L #1,Ar]
  public static void irpAddqLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //ADDQ.L #<data>,Ar
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] += y;  //このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //ADDQ.L #<data>,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) + y;
      } else {  //ADDQ.L #<data>,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltLong (ea);
        mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) + y, XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (~x & z) >>> 31 << 1 |
             (x & ~z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addq
    }
  }  //irpAddqLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ST.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr
  //SNF.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_011_mmm_rrr [ST.B <ea>]
  //DBT.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}
  //DBNF.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_011_001_rrr-{offset}        [DBT.W Dr,<label>]
  //TRAPT.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_010-{data}
  //TPNF.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
  //TPT.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
  //TRAPNF.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_010-{data}  [TRAPT.W #<data>]
  //TRAPT.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_011_111_011-{data}
  //TPNF.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
  //TPT.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
  //TRAPNF.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_011_111_011-{data}  [TRAPT.L #<data>]
  //TRAPT                                           |-|--2346|-|-----|-----|          |0101_000_011_111_100
  //TPNF                                            |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
  //TPT                                             |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
  //TRAPNF                                          |A|--2346|-|-----|-----|          |0101_000_011_111_100 [TRAPT]
  public static void irpSt () throws M68kException {
    int ea = XEiJ.regOC & 63;
    //DBT.W Dr,<label>よりもST.B Drを優先する
    if (ea < XEiJ.EA_AR) {  //ST.B Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] |= 0xff;
    } else if (ea < XEiJ.EA_MM) {  //DBT.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      //条件が成立しているので通過
      XEiJ.mpuCycleCount += 2;
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPT.W/TRAPT.L/TRAPT
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      //条件が成立しているのでTRAPする
      m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
      throw M68kException.m6eSignal;
    } else {  //ST.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), 0xff, XEiJ.regSRS);
    }
  }  //irpSt

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBQ.B #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_100_mmm_rrr
  //DEC.B <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_100_mmm_rrr [SUBQ.B #1,<ea>]
  public static void irpSubqByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int x;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    int z;
    if (ea < XEiJ.EA_AR) {  //SUBQ.B #<data>,Dr
      XEiJ.mpuCycleCount++;
      z = (byte) (XEiJ.regRn[ea] = ~0xff & (x = XEiJ.regRn[ea]) | 0xff & (x = (byte) x) - y);
    } else {  //SUBQ.B #<data>,<mem>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      mmuWriteByteData (a, z = (byte) ((x = mmuModifyByteSignData (a, XEiJ.regSRS)) - y), XEiJ.regSRS);
    }
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           (x & ~z) >>> 31 << 1 |
           (~x & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subq
  }  //irpSubqByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBQ.W #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_101_mmm_rrr
  //SUBQ.W #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_101_001_rrr
  //DEC.W <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_101_mmm_rrr [SUBQ.W #1,<ea>]
  //DEC.W Ar                                        |A|012346|-|-----|-----| A        |0101_001_101_001_rrr [SUBQ.W #1,Ar]
  //
  //SUBQ.W #<data>,Ar
  //  ソースを符号拡張してロングで減算する
  public static void irpSubqWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //SUBQ.W #<data>,Ar
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] -= y;  //ロングで計算する。このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //SUBQ.W #<data>,Dr
        XEiJ.mpuCycleCount++;
        z = (short) (XEiJ.regRn[ea] = ~0xffff & (x = XEiJ.regRn[ea]) | (char) ((x = (short) x) - y));
      } else {  //SUBQ.W #<data>,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltWord (ea);
        mmuWriteWordData (a, z = (short) ((x = mmuModifyWordSignData (a, XEiJ.regSRS)) - y), XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (x & ~z) >>> 31 << 1 |
             (~x & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subq
    }
  }  //irpSubqWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBQ.L #<data>,<ea>                             |-|012346|-|UUUUU|*****|D M+-WXZ  |0101_qqq_110_mmm_rrr
  //SUBQ.L #<data>,Ar                               |-|012346|-|-----|-----| A        |0101_qqq_110_001_rrr
  //DEC.L <ea>                                      |A|012346|-|UUUUU|*****|D M+-WXZ  |0101_001_110_mmm_rrr [SUBQ.L #1,<ea>]
  //DEC.L Ar                                        |A|012346|-|-----|-----| A        |0101_001_110_001_rrr [SUBQ.L #1,Ar]
  public static void irpSubqLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int y = ((XEiJ.regOC >> 9) - 1 & 7) + 1;  //qqq==0?8:qqq
    if (ea >> 3 == XEiJ.MMM_AR) {  //SUBQ.L #<data>,Ar
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] -= y;  //このr[ea]はアドレスレジスタ
      //ccrは操作しない
    } else {
      int x;
      int z;
      if (ea < XEiJ.EA_AR) {  //SUBQ.L #<data>,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] = z = (x = XEiJ.regRn[ea]) - y;
      } else {  //SUBQ.L #<data>,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltLong (ea);
        mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) - y, XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             (x & ~z) >>> 31 << 1 |
             (~x & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subq
    }
  }  //irpSubqLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SF.B <ea>                                       |-|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr
  //SNT.B <ea>                                      |A|012346|-|-----|-----|D M+-WXZ  |0101_000_111_mmm_rrr [SF.B <ea>]
  //DBF.W Dr,<label>                                |-|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}
  //DBNT.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
  //DBRA.W Dr,<label>                               |A|012346|-|-----|-----|          |0101_000_111_001_rrr-{offset}        [DBF.W Dr,<label>]
  //TRAPF.W #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_010-{data}
  //TPF.W #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
  //TPNT.W #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
  //TRAPNT.W #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_010-{data}  [TRAPF.W #<data>]
  //TRAPF.L #<data>                                 |-|--2346|-|-----|-----|          |0101_000_111_111_011-{data}
  //TPF.L #<data>                                   |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
  //TPNT.L #<data>                                  |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
  //TRAPNT.L #<data>                                |A|--2346|-|-----|-----|          |0101_000_111_111_011-{data}  [TRAPF.L #<data>]
  //TRAPF                                           |-|--2346|-|-----|-----|          |0101_000_111_111_100
  //TPF                                             |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
  //TPNT                                            |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
  //TRAPNT                                          |A|--2346|-|-----|-----|          |0101_000_111_111_100 [TRAPF]
  public static void irpSf () throws M68kException {
    int ea = XEiJ.regOC & 63;
    //DBRA.W Dr,<label>よりもSF.B Drを優先する
    if (ea < XEiJ.EA_AR) {  //SF.B Dr
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[ea] &= ~0xff;
    } else if (ea < XEiJ.EA_MM) {  //DBRA.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      //条件が成立していないのでデクリメント
      int rrr = XEiJ.regOC & 7;
      int s = XEiJ.regRn[rrr];
      if ((short) s == 0) {  //Drの下位16bitが0なので通過
        XEiJ.mpuCycleCount += 2;
        XEiJ.regRn[rrr] = s + 65535;
      } else {  //Drの下位16bitが0でないので分岐
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
        irpSetPC (t);
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPF.W/TRAPF.L/TRAPF
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      //条件が成立していないのでTRAPしない
      XEiJ.mpuCycleCount++;
    } else {  //SF.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), 0x00, XEiJ.regSRS);
    }
  }  //irpSf

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SHI.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr
  //SNLS.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_011_mmm_rrr [SHI.B <ea>]
  //DBHI.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}
  //DBNLS.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_011_001_rrr-{offset}        [DBHI.W Dr,<label>]
  //TRAPHI.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}
  //TPHI.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
  //TPNLS.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
  //TRAPNLS.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_010-{data}  [TRAPHI.W #<data>]
  //TRAPHI.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}
  //TPHI.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
  //TPNLS.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
  //TRAPNLS.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_011_111_011-{data}  [TRAPHI.L #<data>]
  //TRAPHI                                          |-|--2346|-|--*-*|-----|          |0101_001_011_111_100
  //TPHI                                            |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
  //TPNLS                                           |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
  //TRAPNLS                                         |A|--2346|-|--*-*|-----|          |0101_001_011_111_100 [TRAPHI]
  public static void irpShi () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBHI.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SHI.B Dr
      if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPHI.W/TRAPHI.L/TRAPHI
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SHI.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_HI << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpShi

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SLS.B <ea>                                      |-|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr
  //SNHI.B <ea>                                     |A|012346|-|--*-*|-----|D M+-WXZ  |0101_001_111_mmm_rrr [SLS.B <ea>]
  //DBLS.W Dr,<label>                               |-|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}
  //DBNHI.W Dr,<label>                              |A|012346|-|--*-*|-----|          |0101_001_111_001_rrr-{offset}        [DBLS.W Dr,<label>]
  //TRAPLS.W #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}
  //TPLS.W #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
  //TPNHI.W #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
  //TRAPNHI.W #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_010-{data}  [TRAPLS.W #<data>]
  //TRAPLS.L #<data>                                |-|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}
  //TPLS.L #<data>                                  |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
  //TPNHI.L #<data>                                 |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
  //TRAPNHI.L #<data>                               |A|--2346|-|--*-*|-----|          |0101_001_111_111_011-{data}  [TRAPLS.L #<data>]
  //TRAPLS                                          |-|--2346|-|--*-*|-----|          |0101_001_111_111_100
  //TPLS                                            |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
  //TPNHI                                           |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
  //TRAPNHI                                         |A|--2346|-|--*-*|-----|          |0101_001_111_111_100 [TRAPLS]
  public static void irpSls () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLS.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SLS.B Dr
      if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLS.W/TRAPLS.L/TRAPLS
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SLS.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_LS << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSls

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SCC.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr
  //SHS.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
  //SNCS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
  //SNLO.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_011_mmm_rrr [SCC.B <ea>]
  //DBCC.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}
  //DBHS.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
  //DBNCS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
  //DBNLO.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_011_001_rrr-{offset}        [DBCC.W Dr,<label>]
  //TRAPCC.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_010-{data}
  //TPCC.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TPHS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TPNCS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TPNLO.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPHS.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPNCS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPNLO.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_010-{data}  [TRAPCC.W #<data>]
  //TRAPCC.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_011_111_011-{data}
  //TPCC.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TPHS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TPNCS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TPNLO.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPHS.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPNCS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPNLO.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_011_111_011-{data}  [TRAPCC.L #<data>]
  //TRAPCC                                          |-|--2346|-|----*|-----|          |0101_010_011_111_100
  //TPCC                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TPHS                                            |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TPNCS                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TPNLO                                           |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TRAPHS                                          |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TRAPNCS                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  //TRAPNLO                                         |A|--2346|-|----*|-----|          |0101_010_011_111_100 [TRAPCC]
  public static void irpShs () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBHS.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SHS.B Dr
      if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPHS.W/TRAPHS.L/TRAPHS
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SHS.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_HS << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpShs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SCS.B <ea>                                      |-|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr
  //SLO.B <ea>                                      |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
  //SNCC.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
  //SNHS.B <ea>                                     |A|012346|-|----*|-----|D M+-WXZ  |0101_010_111_mmm_rrr [SCS.B <ea>]
  //DBCS.W Dr,<label>                               |-|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}
  //DBLO.W Dr,<label>                               |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
  //DBNCC.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
  //DBNHS.W Dr,<label>                              |A|012346|-|----*|-----|          |0101_010_111_001_rrr-{offset}        [DBCS.W Dr,<label>]
  //TRAPCS.W #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_010-{data}
  //TPCS.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TPLO.W #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TPNCC.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TPNHS.W #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPLO.W #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPNCC.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPNHS.W #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_010-{data}  [TRAPCS.W #<data>]
  //TRAPCS.L #<data>                                |-|--2346|-|----*|-----|          |0101_010_111_111_011-{data}
  //TPCS.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TPLO.L #<data>                                  |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TPNCC.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TPNHS.L #<data>                                 |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPLO.L #<data>                                |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPNCC.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPNHS.L #<data>                               |A|--2346|-|----*|-----|          |0101_010_111_111_011-{data}  [TRAPCS.L #<data>]
  //TRAPCS                                          |-|--2346|-|----*|-----|          |0101_010_111_111_100
  //TPCS                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TPLO                                            |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TPNCC                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TPNHS                                           |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TRAPLO                                          |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TRAPNCC                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  //TRAPNHS                                         |A|--2346|-|----*|-----|          |0101_010_111_111_100 [TRAPCS]
  public static void irpSlo () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLO.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SLO.B Dr
      if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLO.W/TRAPLO.L/TRAPLO
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SLO.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_LO << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSlo

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SNE.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr
  //SNEQ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
  //SNZ.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
  //SNZE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_011_mmm_rrr [SNE.B <ea>]
  //DBNE.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}
  //DBNEQ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
  //DBNZ.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
  //DBNZE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_011_001_rrr-{offset}        [DBNE.W Dr,<label>]
  //TRAPNE.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}
  //TPNE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TPNEQ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TPNZ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TPNZE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNEQ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNZ.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNZE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_010-{data}  [TRAPNE.W #<data>]
  //TRAPNE.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}
  //TPNE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TPNEQ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TPNZ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TPNZE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNEQ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNZ.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNZE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_011_111_011-{data}  [TRAPNE.L #<data>]
  //TRAPNE                                          |-|--2346|-|--*--|-----|          |0101_011_011_111_100
  //TPNE                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TPNEQ                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TPNZ                                            |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TPNZE                                           |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TRAPNEQ                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TRAPNZ                                          |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  //TRAPNZE                                         |A|--2346|-|--*--|-----|          |0101_011_011_111_100 [TRAPNE]
  public static void irpSne () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBNE.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SNE.B Dr
      if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPNE.W/TRAPNE.L/TRAPNE
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SNE.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_NE << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSne

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SEQ.B <ea>                                      |-|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr
  //SNNE.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
  //SNNZ.B <ea>                                     |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
  //SZE.B <ea>                                      |A|012346|-|--*--|-----|D M+-WXZ  |0101_011_111_mmm_rrr [SEQ.B <ea>]
  //DBEQ.W Dr,<label>                               |-|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}
  //DBNNE.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
  //DBNNZ.W Dr,<label>                              |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
  //DBZE.W Dr,<label>                               |A|012346|-|--*--|-----|          |0101_011_111_001_rrr-{offset}        [DBEQ.W Dr,<label>]
  //TRAPEQ.W #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}
  //TPEQ.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TPNNE.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TPNNZ.W #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TPZE.W #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPNNE.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPNNZ.W #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPZE.W #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_010-{data}  [TRAPEQ.W #<data>]
  //TRAPEQ.L #<data>                                |-|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}
  //TPEQ.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TPNNE.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TPNNZ.L #<data>                                 |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TPZE.L #<data>                                  |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPNNE.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPNNZ.L #<data>                               |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPZE.L #<data>                                |A|--2346|-|--*--|-----|          |0101_011_111_111_011-{data}  [TRAPEQ.L #<data>]
  //TRAPEQ                                          |-|--2346|-|--*--|-----|          |0101_011_111_111_100
  //TPEQ                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TPNNE                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TPNNZ                                           |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TPZE                                            |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TRAPNNE                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TRAPNNZ                                         |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  //TRAPZE                                          |A|--2346|-|--*--|-----|          |0101_011_111_111_100 [TRAPEQ]
  public static void irpSeq () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBEQ.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SEQ.B Dr
      if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPEQ.W/TRAPEQ.L/TRAPEQ
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SEQ.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_EQ << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSeq

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SVC.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr
  //SNVS.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_011_mmm_rrr [SVC.B <ea>]
  //DBVC.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}
  //DBNVS.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_011_001_rrr-{offset}        [DBVC.W Dr,<label>]
  //TRAPVC.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}
  //TPNVS.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
  //TPVC.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
  //TRAPNVS.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_010-{data}  [TRAPVC.W #<data>]
  //TRAPVC.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}
  //TPNVS.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
  //TPVC.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
  //TRAPNVS.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_011_111_011-{data}  [TRAPVC.L #<data>]
  //TRAPVC                                          |-|--2346|-|---*-|-----|          |0101_100_011_111_100
  //TPNVS                                           |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
  //TPVC                                            |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
  //TRAPNVS                                         |A|--2346|-|---*-|-----|          |0101_100_011_111_100 [TRAPVC]
  public static void irpSvc () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBVC.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SVC.B Dr
      if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPVC.W/TRAPVC.L/TRAPVC
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SVC.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_VC << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSvc

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SVS.B <ea>                                      |-|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr
  //SNVC.B <ea>                                     |A|012346|-|---*-|-----|D M+-WXZ  |0101_100_111_mmm_rrr [SVS.B <ea>]
  //DBVS.W Dr,<label>                               |-|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}
  //DBNVC.W Dr,<label>                              |A|012346|-|---*-|-----|          |0101_100_111_001_rrr-{offset}        [DBVS.W Dr,<label>]
  //TRAPVS.W #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}
  //TPNVC.W #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
  //TPVS.W #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
  //TRAPNVC.W #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_010-{data}  [TRAPVS.W #<data>]
  //TRAPVS.L #<data>                                |-|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}
  //TPNVC.L #<data>                                 |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
  //TPVS.L #<data>                                  |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
  //TRAPNVC.L #<data>                               |A|--2346|-|---*-|-----|          |0101_100_111_111_011-{data}  [TRAPVS.L #<data>]
  //TRAPVS                                          |-|--2346|-|---*-|-----|          |0101_100_111_111_100
  //TPNVC                                           |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
  //TPVS                                            |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
  //TRAPNVC                                         |A|--2346|-|---*-|-----|          |0101_100_111_111_100 [TRAPVS]
  public static void irpSvs () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBVS.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SVS.B Dr
      if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPVS.W/TRAPVS.L/TRAPVS
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SVS.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_VS << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSvs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SPL.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr
  //SNMI.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_011_mmm_rrr [SPL.B <ea>]
  //DBPL.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}
  //DBNMI.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_011_001_rrr-{offset}        [DBPL.W Dr,<label>]
  //TRAPPL.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}
  //TPNMI.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
  //TPPL.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
  //TRAPNMI.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_010-{data}  [TRAPPL.W #<data>]
  //TRAPPL.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}
  //TPNMI.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
  //TPPL.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
  //TRAPNMI.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_011_111_011-{data}  [TRAPPL.L #<data>]
  //TRAPPL                                          |-|--2346|-|-*---|-----|          |0101_101_011_111_100
  //TPNMI                                           |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
  //TPPL                                            |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
  //TRAPNMI                                         |A|--2346|-|-*---|-----|          |0101_101_011_111_100 [TRAPPL]
  public static void irpSpl () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBPL.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SPL.B Dr
      if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPPL.W/TRAPPL.L/TRAPPL
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SPL.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_PL << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSpl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SMI.B <ea>                                      |-|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr
  //SNPL.B <ea>                                     |A|012346|-|-*---|-----|D M+-WXZ  |0101_101_111_mmm_rrr [SMI.B <ea>]
  //DBMI.W Dr,<label>                               |-|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}
  //DBNPL.W Dr,<label>                              |A|012346|-|-*---|-----|          |0101_101_111_001_rrr-{offset}        [DBMI.W Dr,<label>]
  //TRAPMI.W #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}
  //TPMI.W #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
  //TPNPL.W #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
  //TRAPNPL.W #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_010-{data}  [TRAPMI.W #<data>]
  //TRAPMI.L #<data>                                |-|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}
  //TPMI.L #<data>                                  |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
  //TPNPL.L #<data>                                 |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
  //TRAPNPL.L #<data>                               |A|--2346|-|-*---|-----|          |0101_101_111_111_011-{data}  [TRAPMI.L #<data>]
  //TRAPMI                                          |-|--2346|-|-*---|-----|          |0101_101_111_111_100
  //TPMI                                            |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
  //TPNPL                                           |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
  //TRAPNPL                                         |A|--2346|-|-*---|-----|          |0101_101_111_111_100 [TRAPMI]
  public static void irpSmi () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBMI.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SMI.B Dr
      if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPMI.W/TRAPMI.L/TRAPMI
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SMI.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_MI << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSmi

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SGE.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr
  //SNLT.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_011_mmm_rrr [SGE.B <ea>]
  //DBGE.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}
  //DBNLT.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_011_001_rrr-{offset}        [DBGE.W Dr,<label>]
  //TRAPGE.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}
  //TPGE.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
  //TPNLT.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
  //TRAPNLT.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_010-{data}  [TRAPGE.W #<data>]
  //TRAPGE.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}
  //TPGE.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
  //TPNLT.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
  //TRAPNLT.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_011_111_011-{data}  [TRAPGE.L #<data>]
  //TRAPGE                                          |-|--2346|-|-*-*-|-----|          |0101_110_011_111_100
  //TPGE                                            |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
  //TPNLT                                           |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
  //TRAPNLT                                         |A|--2346|-|-*-*-|-----|          |0101_110_011_111_100 [TRAPGE]
  public static void irpSge () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBGE.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SGE.B Dr
      if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPGE.W/TRAPGE.L/TRAPGE
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SGE.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_GE << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSge

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SLT.B <ea>                                      |-|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr
  //SNGE.B <ea>                                     |A|012346|-|-*-*-|-----|D M+-WXZ  |0101_110_111_mmm_rrr [SLT.B <ea>]
  //DBLT.W Dr,<label>                               |-|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}
  //DBNGE.W Dr,<label>                              |A|012346|-|-*-*-|-----|          |0101_110_111_001_rrr-{offset}        [DBLT.W Dr,<label>]
  //TRAPLT.W #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}
  //TPLT.W #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
  //TPNGE.W #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
  //TRAPNGE.W #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_010-{data}  [TRAPLT.W #<data>]
  //TRAPLT.L #<data>                                |-|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}
  //TPLT.L #<data>                                  |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
  //TPNGE.L #<data>                                 |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
  //TRAPNGE.L #<data>                               |A|--2346|-|-*-*-|-----|          |0101_110_111_111_011-{data}  [TRAPLT.L #<data>]
  //TRAPLT                                          |-|--2346|-|-*-*-|-----|          |0101_110_111_111_100
  //TPLT                                            |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
  //TPNGE                                           |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
  //TRAPNGE                                         |A|--2346|-|-*-*-|-----|          |0101_110_111_111_100 [TRAPLT]
  public static void irpSlt () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLT.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SLT.B Dr
      if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLT.W/TRAPLT.L/TRAPLT
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SLT.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_LT << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSlt

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SGT.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr
  //SNLE.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_011_mmm_rrr [SGT.B <ea>]
  //DBGT.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}
  //DBNLE.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_011_001_rrr-{offset}        [DBGT.W Dr,<label>]
  //TRAPGT.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}
  //TPGT.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
  //TPNLE.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
  //TRAPNLE.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_010-{data}  [TRAPGT.W #<data>]
  //TRAPGT.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}
  //TPGT.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
  //TPNLE.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
  //TRAPNLE.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_011_111_011-{data}  [TRAPGT.L #<data>]
  //TRAPGT                                          |-|--2346|-|-***-|-----|          |0101_111_011_111_100
  //TPGT                                            |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
  //TPNLE                                           |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
  //TRAPNLE                                         |A|--2346|-|-***-|-----|          |0101_111_011_111_100 [TRAPGT]
  public static void irpSgt () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBGT.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SGT.B Dr
      if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPGT.W/TRAPGT.L/TRAPGT
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SGT.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_GT << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSgt

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SLE.B <ea>                                      |-|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr
  //SNGT.B <ea>                                     |A|012346|-|-***-|-----|D M+-WXZ  |0101_111_111_mmm_rrr [SLE.B <ea>]
  //DBLE.W Dr,<label>                               |-|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}
  //DBNGT.W Dr,<label>                              |A|012346|-|-***-|-----|          |0101_111_111_001_rrr-{offset}        [DBLE.W Dr,<label>]
  //TRAPLE.W #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}
  //TPLE.W #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
  //TPNGT.W #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
  //TRAPNGT.W #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_010-{data}  [TRAPLE.W #<data>]
  //TRAPLE.L #<data>                                |-|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}
  //TPLE.L #<data>                                  |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
  //TPNGT.L #<data>                                 |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
  //TRAPNGT.L #<data>                               |A|--2346|-|-***-|-----|          |0101_111_111_111_011-{data}  [TRAPLE.L #<data>]
  //TRAPLE                                          |-|--2346|-|-***-|-----|          |0101_111_111_111_100
  //TPLE                                            |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
  //TPNGT                                           |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
  //TRAPNGT                                         |A|--2346|-|-***-|-----|          |0101_111_111_111_100 [TRAPLE]
  public static void irpSle () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //DBLE.W Dr,<label>
      int t = XEiJ.regPC;  //pc0+2
      XEiJ.regPC = t + 2;  //pc0+4
      t += mmuReadWordSignExword (t, XEiJ.regSRS);  //pc0+2+16bitディスプレースメント
      if ((t & 1) != 0) {  //分岐先のアドレスが奇数
        //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
        irpBccAddressError (t);
      }
      if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //条件が成立しているので通過
        XEiJ.mpuCycleCount += 2;
      } else {  //条件が成立していないのでデクリメント
        int rrr = XEiJ.regOC & 7;
        int s = XEiJ.regRn[rrr];
        if ((short) s == 0) {  //Drの下位16bitが0なので通過
          XEiJ.mpuCycleCount += 2;
          XEiJ.regRn[rrr] = s + 65535;
        } else {  //Drの下位16bitが0でないので分岐
          XEiJ.mpuCycleCount++;
          XEiJ.regRn[rrr] = s - 1;  //下位16bitが0でないので上位16bitは変化しない
          irpSetPC (t);
        }
      }
    } else if (ea < XEiJ.EA_AR) {  //SLE.B Dr
      if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //セット
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] |= 0xff;
      } else {  //クリア
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] &= ~0xff;
      }
    } else if ((XEiJ.EAM_PW | XEiJ.EAM_PX | XEiJ.EAM_IM) << ea < 0L) {  //TRAPLE.W/TRAPLE.L/TRAPLE
      if (ea == 072) {  //.W
        mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);
      } else if (ea == 073) {  //.L
        mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS);
      }
      if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {
        //条件が成立しているのでTRAPする
        m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
        M68kException.m6eNumber = M68kException.M6E_TRAPV_INSTRUCTION;
        throw M68kException.m6eSignal;
      } else {
        //条件が成立していないのでTRAPしない
        XEiJ.mpuCycleCount++;
      }
    } else {  //SLE.B <mem>
      XEiJ.mpuCycleCount++;
      mmuWriteByteData (efaMltByte (ea), XEiJ.MPU_CC_LE << XEiJ.regCCR >> 31, XEiJ.regSRS);
    }
  }  //irpSle

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.W <label>                                   |-|012346|-|-----|-----|          |0110_000_000_000_000-{offset}
  //JBRA.W <label>                                  |A|012346|-|-----|-----|          |0110_000_000_000_000-{offset}        [BRA.W <label>]
  //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)
  //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_000_sss_sss (s is not equal to 0)   [BRA.S <label>]
  public static void irpBrasw () throws M68kException {
    XEiJ.mpuCycleCount++;  //0clkにしない
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == 0) {  //BRA.W
      XEiJ.regPC = t + 2;
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //pcws
    }
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBrasw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_001_sss_sss
  //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_001_sss_sss [BRA.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.S <label>                                   |-|012346|-|-----|-----|          |0110_000_010_sss_sss
  //JBRA.S <label>                                  |A|012346|-|-----|-----|          |0110_000_010_sss_sss [BRA.S <label>]
  public static void irpBras () throws M68kException {
    XEiJ.mpuCycleCount++;  //0clkにしない
    irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
  }  //irpBras

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BRA.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)
  //JBRA.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_011_sss_sss (s is not equal to 63)  [BRA.S <label>]
  //BRA.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_011_111_111-{offset}
  public static void irpBrasl () throws M68kException {
    XEiJ.mpuCycleCount++;  //0clkにしない
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == -1) {  //BRA.L
      XEiJ.regPC = t + 4;
      s = mmuReadLongExword (t, XEiJ.regSRS);  //pcls
    }
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBrasl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.W <label>                                   |-|012346|-|-----|-----|          |0110_000_100_000_000-{offset}
  //JBSR.W <label>                                  |A|012346|-|-----|-----|          |0110_000_100_000_000-{offset}        [BSR.W <label>]
  //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)
  //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_100_sss_sss (s is not equal to 0)   [BSR.S <label>]
  public static void irpBsrsw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == 0) {  //BSR.W
      XEiJ.regPC = t + 2;
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //pcws
    }
    m60Incremented -= 4L << (7 << 3);
    int sp = m60Address = XEiJ.regRn[15] -= 4;
    mmuWriteLongData (sp, XEiJ.regPC, XEiJ.regSRS);  //pushl
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBsrsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_101_sss_sss
  //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_101_sss_sss [BSR.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.S <label>                                   |-|012346|-|-----|-----|          |0110_000_110_sss_sss
  //JBSR.S <label>                                  |A|012346|-|-----|-----|          |0110_000_110_sss_sss [BSR.S <label>]
  public static void irpBsrs () throws M68kException {
    XEiJ.mpuCycleCount++;
    m60Incremented -= 4L << (7 << 3);
    int sp = m60Address = XEiJ.regRn[15] -= 4;
    mmuWriteLongData (sp, XEiJ.regPC, XEiJ.regSRS);  //pushl
    irpSetPC (XEiJ.regPC + (byte) XEiJ.regOC);  //pc0+2+オフセット
  }  //irpBsrs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BSR.S <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)
  //JBSR.S <label>                                  |A|--2346|-|-----|-----|          |0110_000_111_sss_sss (s is not equal to 63)  [BSR.S <label>]
  //BSR.L <label>                                   |-|--2346|-|-----|-----|          |0110_000_111_111_111-{offset}
  public static void irpBsrsl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //オフセット
    if (s == -1) {  //BSR.L
      XEiJ.regPC = t + 4;
      s = mmuReadLongExword (t, XEiJ.regSRS);  //pcls
    }
    m60Incremented -= 4L << (7 << 3);
    int sp = m60Address = XEiJ.regRn[15] -= 4;
    mmuWriteLongData (sp, XEiJ.regPC, XEiJ.regSRS);  //pushl
    irpSetPC (t + s);  //pc0+2+オフセット
  }  //irpBsrsl

  //irpBccAddressError (int t)
  public static void irpBccAddressError (int t) throws M68kException {
    M68kException.m6eNumber = M68kException.M6E_ADDRESS_ERROR;
    m60Address = t & -2;  //偶数にする
    M68kException.m6eDirection = XEiJ.MPU_WR_READ;
    M68kException.m6eSize = XEiJ.MPU_SS_WORD;
    throw M68kException.m6eSignal;
  }

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}
  //BNLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
  //JBHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
  //JBNLS.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_000-{offset}        [BHI.W <label>]
  //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)
  //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
  //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
  //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_sss_sss (s is not equal to 0)   [BHI.S <label>]
  //JBLS.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
  //JBNHI.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_000_000_110-0100111011111001-{address}      [BHI.S (*)+8;JMP <label>]
  public static void irpBhisw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBhisw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_001_sss_sss
  //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
  //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
  //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_001_sss_sss [BHI.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_010_sss_sss
  //BNLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
  //JBHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
  //JBNLS.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_010_sss_sss [BHI.S <label>]
  public static void irpBhis () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBhis

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BHI.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)
  //BNLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
  //JBHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
  //JBNLS.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_011_sss_sss (s is not equal to 63)  [BHI.S <label>]
  //BHI.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}
  //BNLS.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_011_111_111-{offset}        [BHI.L <label>]
  public static void irpBhisl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_HI << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBhisl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.W <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}
  //BNHI.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
  //JBLS.W <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
  //JBNHI.W <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_000-{offset}        [BLS.W <label>]
  //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)
  //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
  //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
  //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_sss_sss (s is not equal to 0)   [BLS.S <label>]
  //JBHI.L <label>                                  |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
  //JBNLS.L <label>                                 |A|012346|-|--*-*|-----|          |0110_001_100_000_110-0100111011111001-{address}      [BLS.S (*)+8;JMP <label>]
  public static void irpBlssw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlssw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_101_sss_sss
  //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
  //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
  //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_101_sss_sss [BLS.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.S <label>                                   |-|012346|-|--*-*|-----|          |0110_001_110_sss_sss
  //BNHI.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
  //JBLS.S <label>                                  |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
  //JBNHI.S <label>                                 |A|012346|-|--*-*|-----|          |0110_001_110_sss_sss [BLS.S <label>]
  public static void irpBlss () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlss

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLS.S <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)
  //BNHI.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
  //JBLS.S <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
  //JBNHI.S <label>                                 |A|--2346|-|--*-*|-----|          |0110_001_111_sss_sss (s is not equal to 63)  [BLS.S <label>]
  //BLS.L <label>                                   |-|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}
  //BNHI.L <label>                                  |A|--2346|-|--*-*|-----|          |0110_001_111_111_111-{offset}        [BLS.L <label>]
  public static void irpBlssl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlssl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.W <label>                                   |-|012346|-|----*|-----|          |0110_010_000_000_000-{offset}
  //BHS.W <label>                                   |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //BNCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //BNLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBNCS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //JBNLO.W <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_000-{offset}        [BCC.W <label>]
  //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)
  //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_000_sss_sss (s is not equal to 0)   [BCC.S <label>]
  //JBCS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  //JBLO.L <label>                                  |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  //JBNCC.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  //JBNHS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_000_000_110-0100111011111001-{address}      [BCC.S (*)+8;JMP <label>]
  public static void irpBhssw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBhssw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_001_sss_sss
  //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_001_sss_sss [BCC.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.S <label>                                   |-|012346|-|----*|-----|          |0110_010_010_sss_sss
  //BHS.S <label>                                   |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //BNCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //BNLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBNCS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  //JBNLO.S <label>                                 |A|012346|-|----*|-----|          |0110_010_010_sss_sss [BCC.S <label>]
  public static void irpBhss () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBhss

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCC.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)
  //BHS.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //BNCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //BNLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBNCS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //JBNLO.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_011_sss_sss (s is not equal to 63)  [BCC.S <label>]
  //BCC.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}
  //BHS.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
  //BNCS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
  //BNLO.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_011_111_111-{offset}        [BCC.L <label>]
  public static void irpBhssl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_HS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBhssl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.W <label>                                   |-|012346|-|----*|-----|          |0110_010_100_000_000-{offset}
  //BLO.W <label>                                   |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //BNCC.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //BNHS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBCS.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBLO.W <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBNCC.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //JBNHS.W <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_000-{offset}        [BCS.W <label>]
  //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)
  //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_100_sss_sss (s is not equal to 0)   [BCS.S <label>]
  //JBCC.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  //JBHS.L <label>                                  |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  //JBNCS.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  //JBNLO.L <label>                                 |A|012346|-|----*|-----|          |0110_010_100_000_110-0100111011111001-{address}      [BCS.S (*)+8;JMP <label>]
  public static void irpBlosw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlosw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_101_sss_sss
  //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_101_sss_sss [BCS.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.S <label>                                   |-|012346|-|----*|-----|          |0110_010_110_sss_sss
  //BLO.S <label>                                   |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //BNCC.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //BNHS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBCS.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBLO.S <label>                                  |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBNCC.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  //JBNHS.S <label>                                 |A|012346|-|----*|-----|          |0110_010_110_sss_sss [BCS.S <label>]
  public static void irpBlos () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlos

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BCS.S <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)
  //BLO.S <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //BNCC.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //BNHS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBCS.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBLO.S <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBNCC.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //JBNHS.S <label>                                 |A|--2346|-|----*|-----|          |0110_010_111_sss_sss (s is not equal to 63)  [BCS.S <label>]
  //BCS.L <label>                                   |-|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}
  //BLO.L <label>                                   |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
  //BNCC.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
  //BNHS.L <label>                                  |A|--2346|-|----*|-----|          |0110_010_111_111_111-{offset}        [BCS.L <label>]
  public static void irpBlosl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LO << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlosl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}
  //BNEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //BNZ.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //BNZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNEQ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //JBNZE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_000-{offset}        [BNE.W <label>]
  //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)
  //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_sss_sss (s is not equal to 0)   [BNE.S <label>]
  //JBEQ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNEQ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNNE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNNZ.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNZ.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBNZE.L <label>                                 |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  //JBZE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_000_000_110-0100111011111001-{address}      [BNE.S (*)+8;JMP <label>]
  public static void irpBnesw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBnesw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_001_sss_sss
  //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_001_sss_sss [BNE.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_010_sss_sss
  //BNEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //BNZ.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //BNZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  //JBNZE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_010_sss_sss [BNE.S <label>]
  public static void irpBnes () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBnes

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BNE.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)
  //BNEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //BNZ.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //BNZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNEQ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //JBNZE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_011_sss_sss (s is not equal to 63)  [BNE.S <label>]
  //BNE.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}
  //BNEQ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
  //BNZ.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
  //BNZE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_011_111_111-{offset}        [BNE.L <label>]
  public static void irpBnesl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_NE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBnesl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.W <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}
  //BNNE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //BNNZ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //BZE.W <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBEQ.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBNNE.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBNNZ.W <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //JBZE.W <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_000-{offset}        [BEQ.W <label>]
  //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)
  //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_sss_sss (s is not equal to 0)   [BEQ.S <label>]
  //JBNE.L <label>                                  |A|012346|-|--*--|-----|          |0110_011_100_000_110-0100111011111001-{address}      [BEQ.S (*)+8;JMP <label>]
  public static void irpBeqsw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBeqsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_101_sss_sss
  //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_101_sss_sss [BEQ.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.S <label>                                   |-|012346|-|--*--|-----|          |0110_011_110_sss_sss
  //BNNE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //BZE.S <label>                                   |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  //JBZE.S <label>                                  |A|012346|-|--*--|-----|          |0110_011_110_sss_sss [BEQ.S <label>]
  public static void irpBeqs () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBeqs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BEQ.S <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)
  //BNNE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //BNNZ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //BZE.S <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBEQ.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBNNE.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBNNZ.S <label>                                 |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //JBZE.S <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_sss_sss (s is not equal to 63)  [BEQ.S <label>]
  //BEQ.L <label>                                   |-|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}
  //BNNE.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
  //BNNZ.L <label>                                  |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
  //BZE.L <label>                                   |A|--2346|-|--*--|-----|          |0110_011_111_111_111-{offset}        [BEQ.L <label>]
  public static void irpBeqsl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_EQ << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBeqsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}
  //BNVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
  //JBNVS.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
  //JBVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_000-{offset}        [BVC.W <label>]
  //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)
  //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
  //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
  //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_sss_sss (s is not equal to 0)   [BVC.S <label>]
  //JBNVC.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
  //JBVS.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_000_000_110-0100111011111001-{address}      [BVC.S (*)+8;JMP <label>]
  public static void irpBvcsw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBvcsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_001_sss_sss
  //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
  //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
  //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_001_sss_sss [BVC.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_010_sss_sss
  //BNVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
  //JBNVS.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
  //JBVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_010_sss_sss [BVC.S <label>]
  public static void irpBvcs () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBvcs

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVC.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)
  //BNVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
  //JBNVS.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
  //JBVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_sss_sss (s is not equal to 63)  [BVC.S <label>]
  //BVC.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}
  //BNVS.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_011_111_111-{offset}        [BVC.L <label>]
  public static void irpBvcsl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_VC << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBvcsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.W <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}
  //BNVC.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
  //JBNVC.W <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
  //JBVS.W <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_000-{offset}        [BVS.W <label>]
  //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)
  //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
  //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
  //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_sss_sss (s is not equal to 0)   [BVS.S <label>]
  //JBNVS.L <label>                                 |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
  //JBVC.L <label>                                  |A|012346|-|---*-|-----|          |0110_100_100_000_110-0100111011111001-{address}      [BVS.S (*)+8;JMP <label>]
  public static void irpBvssw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBvssw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_101_sss_sss
  //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
  //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
  //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_101_sss_sss [BVS.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.S <label>                                   |-|012346|-|---*-|-----|          |0110_100_110_sss_sss
  //BNVC.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
  //JBNVC.S <label>                                 |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
  //JBVS.S <label>                                  |A|012346|-|---*-|-----|          |0110_100_110_sss_sss [BVS.S <label>]
  public static void irpBvss () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBvss

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BVS.S <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)
  //BNVC.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
  //JBNVC.S <label>                                 |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
  //JBVS.S <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_sss_sss (s is not equal to 63)  [BVS.S <label>]
  //BVS.L <label>                                   |-|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}
  //BNVC.L <label>                                  |A|--2346|-|---*-|-----|          |0110_100_111_111_111-{offset}        [BVS.L <label>]
  public static void irpBvssl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_VS << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBvssl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}
  //BNMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
  //JBNMI.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
  //JBPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_000-{offset}        [BPL.W <label>]
  //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)
  //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
  //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
  //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_sss_sss (s is not equal to 0)   [BPL.S <label>]
  //JBMI.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
  //JBNPL.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_000_000_110-0100111011111001-{address}      [BPL.S (*)+8;JMP <label>]
  public static void irpBplsw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBplsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_001_sss_sss
  //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
  //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
  //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_001_sss_sss [BPL.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_010_sss_sss
  //BNMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
  //JBNMI.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
  //JBPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_010_sss_sss [BPL.S <label>]
  public static void irpBpls () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBpls

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BPL.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)
  //BNMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
  //JBNMI.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
  //JBPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_sss_sss (s is not equal to 63)  [BPL.S <label>]
  //BPL.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}
  //BNMI.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_011_111_111-{offset}        [BPL.L <label>]
  public static void irpBplsl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_PL << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBplsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.W <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}
  //BNPL.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
  //JBMI.W <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
  //JBNPL.W <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_000-{offset}        [BMI.W <label>]
  //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)
  //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
  //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
  //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_sss_sss (s is not equal to 0)   [BMI.S <label>]
  //JBNMI.L <label>                                 |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
  //JBPL.L <label>                                  |A|012346|-|-*---|-----|          |0110_101_100_000_110-0100111011111001-{address}      [BMI.S (*)+8;JMP <label>]
  public static void irpBmisw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBmisw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_101_sss_sss
  //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
  //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
  //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_101_sss_sss [BMI.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.S <label>                                   |-|012346|-|-*---|-----|          |0110_101_110_sss_sss
  //BNPL.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
  //JBMI.S <label>                                  |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
  //JBNPL.S <label>                                 |A|012346|-|-*---|-----|          |0110_101_110_sss_sss [BMI.S <label>]
  public static void irpBmis () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBmis

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BMI.S <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)
  //BNPL.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
  //JBMI.S <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
  //JBNPL.S <label>                                 |A|--2346|-|-*---|-----|          |0110_101_111_sss_sss (s is not equal to 63)  [BMI.S <label>]
  //BMI.L <label>                                   |-|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}
  //BNPL.L <label>                                  |A|--2346|-|-*---|-----|          |0110_101_111_111_111-{offset}        [BMI.L <label>]
  public static void irpBmisl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_MI << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBmisl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}
  //BNLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
  //JBGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
  //JBNLT.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_000-{offset}        [BGE.W <label>]
  //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)
  //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
  //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
  //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_sss_sss (s is not equal to 0)   [BGE.S <label>]
  //JBLT.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
  //JBNGE.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_000_000_110-0100111011111001-{address}      [BGE.S (*)+8;JMP <label>]
  public static void irpBgesw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBgesw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_001_sss_sss
  //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
  //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
  //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_001_sss_sss [BGE.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_010_sss_sss
  //BNLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
  //JBGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
  //JBNLT.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_010_sss_sss [BGE.S <label>]
  public static void irpBges () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBges

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGE.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)
  //BNLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
  //JBGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
  //JBNLT.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_011_sss_sss (s is not equal to 63)  [BGE.S <label>]
  //BGE.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}
  //BNLT.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_011_111_111-{offset}        [BGE.L <label>]
  public static void irpBgesl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_GE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBgesl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.W <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}
  //BNGE.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
  //JBLT.W <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
  //JBNGE.W <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_000-{offset}        [BLT.W <label>]
  //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)
  //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
  //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
  //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_sss_sss (s is not equal to 0)   [BLT.S <label>]
  //JBGE.L <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
  //JBNLT.L <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_100_000_110-0100111011111001-{address}      [BLT.S (*)+8;JMP <label>]
  public static void irpBltsw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBltsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_101_sss_sss
  //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
  //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
  //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_101_sss_sss [BLT.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.S <label>                                   |-|012346|-|-*-*-|-----|          |0110_110_110_sss_sss
  //BNGE.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
  //JBLT.S <label>                                  |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
  //JBNGE.S <label>                                 |A|012346|-|-*-*-|-----|          |0110_110_110_sss_sss [BLT.S <label>]
  public static void irpBlts () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlts

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLT.S <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)
  //BNGE.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
  //JBLT.S <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
  //JBNGE.S <label>                                 |A|--2346|-|-*-*-|-----|          |0110_110_111_sss_sss (s is not equal to 63)  [BLT.S <label>]
  //BLT.L <label>                                   |-|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}
  //BNGE.L <label>                                  |A|--2346|-|-*-*-|-----|          |0110_110_111_111_111-{offset}        [BLT.L <label>]
  public static void irpBltsl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LT << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBltsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}
  //BNLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
  //JBGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
  //JBNLE.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_000-{offset}        [BGT.W <label>]
  //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)
  //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
  //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
  //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_sss_sss (s is not equal to 0)   [BGT.S <label>]
  //JBLE.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
  //JBNGT.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_000_000_110-0100111011111001-{address}      [BGT.S (*)+8;JMP <label>]
  public static void irpBgtsw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBgtsw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_001_sss_sss
  //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
  //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
  //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_001_sss_sss [BGT.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_010_sss_sss
  //BNLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
  //JBGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
  //JBNLE.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_010_sss_sss [BGT.S <label>]
  public static void irpBgts () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBgts

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BGT.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)
  //BNLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
  //JBGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
  //JBNLE.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_011_sss_sss (s is not equal to 63)  [BGT.S <label>]
  //BGT.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}
  //BNLE.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_011_111_111-{offset}        [BGT.L <label>]
  public static void irpBgtsl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_GT << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBgtsl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.W <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}
  //BNGT.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
  //JBLE.W <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
  //JBNGT.W <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_000-{offset}        [BLE.W <label>]
  //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)
  //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
  //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
  //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_sss_sss (s is not equal to 0)   [BLE.S <label>]
  //JBGT.L <label>                                  |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
  //JBNLE.L <label>                                 |A|012346|-|-***-|-----|          |0110_111_100_000_110-0100111011111001-{address}      [BLE.S (*)+8;JMP <label>]
  public static void irpBlesw () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == 0) {  //Bcc.W
      XEiJ.regPC = t + 2;  //pc0+4
      s = mmuReadWordSignExword (t, XEiJ.regSRS);  //16bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlesw

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_101_sss_sss
  //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
  //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
  //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_101_sss_sss [BLE.S <label>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.S <label>                                   |-|012346|-|-***-|-----|          |0110_111_110_sss_sss
  //BNGT.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
  //JBLE.S <label>                                  |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
  //JBNGT.S <label>                                 |A|012346|-|-***-|-----|          |0110_111_110_sss_sss [BLE.S <label>]
  public static void irpBles () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBles

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //BLE.S <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)
  //BNGT.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
  //JBLE.S <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
  //JBNGT.S <label>                                 |A|--2346|-|-***-|-----|          |0110_111_111_sss_sss (s is not equal to 63)  [BLE.S <label>]
  //BLE.L <label>                                   |-|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}
  //BNGT.L <label>                                  |A|--2346|-|-***-|-----|          |0110_111_111_111_111-{offset}        [BLE.L <label>]
  public static void irpBlesl () throws M68kException {
    XEiJ.mpuCycleCount++;
    int t = XEiJ.regPC;  //pc0+2
    int s = (byte) XEiJ.regOC;  //8bitディスプレースメント
    if (s == -1) {  //Bcc.L
      XEiJ.regPC = t + 4;  //pc0+6
      s = mmuReadLongExword (t, XEiJ.regSRS);  //32bitディスプレースメント
    }
    t += s;  //pc0+2+ディスプレースメント
    if ((t & 1) != 0) {  //分岐先のアドレスが奇数
      //MC68060のBcc/DBcc/FBccは分岐先のアドレスが奇数のとき分岐しなくてもアドレスエラーになる。FDBccは分岐するときだけ
      irpBccAddressError (t);
    }
    if (XEiJ.MPU_CC_LE << XEiJ.regCCR < 0) {  //分岐する
      irpSetPC (t);
    }
  }  //irpBlesl

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //IOCS <name>                                     |A|012346|-|UUUUU|UUUUU|          |0111_000_0dd_ddd_ddd-0100111001001111        [MOVEQ.L #<data>,D0;TRAP #15]
  //MOVEQ.L #<data>,Dq                              |-|012346|-|-UUUU|-**00|          |0111_qqq_0dd_ddd_ddd
  public static void irpMoveq () throws M68kException {
    XEiJ.mpuCycleCount++;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = (byte) XEiJ.regOC;
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMoveq

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVS.B <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_100_mmm_rrr (ISA_B)
  //
  //MVS.B <ea>,Dq
  //  バイトデータをロングに符号拡張してDqの全体を更新する
  public static void irpMvsByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? (byte) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMvsByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVS.W <ea>,Dq                                   |-|------|-|-UUUU|-**00|D M+-WXZPI|0111_qqq_101_mmm_rrr (ISA_B)
  //
  //MVS.W <ea>,Dq
  //  ワードデータをロングに符号拡張してDqの全体を更新する
  public static void irpMvsWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //pcws。イミディエイトを分離
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMvsWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVZ.B <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_110_mmm_rrr (ISA_B)
  //
  //MVZ.B <ea>,Dq
  //  バイトデータをロングにゼロ拡張してDqの全体を更新する
  public static void irpMvzByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? 0xff & XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteZeroExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteZeroData (efaAnyByte (ea), XEiJ.regSRS);  //pcbz。イミディエイトを分離
    XEiJ.regCCR = XEiJ.REG_CCR_X & XEiJ.regCCR | (z == 0 ? XEiJ.REG_CCR_Z : 0);
  }  //irpMvzByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MVZ.W <ea>,Dq                                   |-|------|-|-UUUU|-0*00|D M+-WXZPI|0111_qqq_111_mmm_rrr (ISA_B)
  //
  //MVZ.W <ea>,Dq
  //  ワードデータをロングにゼロ拡張してDqの全体を更新する
  public static void irpMvzWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z;
    XEiJ.regRn[XEiJ.regOC >> 9 & 7] = z = ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordZeroData (efaAnyWord (ea), XEiJ.regSRS);  //pcwz。イミディエイトを分離
    XEiJ.regCCR = XEiJ.REG_CCR_X & XEiJ.regCCR | (z == 0 ? XEiJ.REG_CCR_Z : 0);
  }  //irpMvzWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //OR.B <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_000_mmm_rrr
  public static void irpOrToRegByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & (XEiJ.regRn[XEiJ.regOC >> 9 & 7] |= 255 & (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS)))];  //ccr_tst_byte。pcbs。イミディエイトを分離。0拡張してからOR
  }  //irpOrToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //OR.W <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_001_mmm_rrr
  public static void irpOrToRegWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = (short) (XEiJ.regRn[XEiJ.regOC >> 9 & 7] |= ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordZeroData (efaAnyWord (ea), XEiJ.regSRS));  //pcwz。イミディエイトを分離。0拡張してからOR
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpOrToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //OR.L <ea>,Dq                                    |-|012346|-|-UUUU|-**00|D M+-WXZPI|1000_qqq_010_mmm_rrr
  public static void irpOrToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] |= ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpOrToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //DIVU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_011_mmm_rrr
  //
  //DIVU.W <ea>,Dq
  //  M68000PRMでDIVU.Wのオーバーフローの条件が16bit符号あり整数と書かれているのは16bit符号なし整数の間違い
  public static void irpDivuWord () throws M68kException {
    //  X  変化しない
    //  N  ゼロ除算またはオーバーフローのとき不定。商が負のときセット。それ以外はクリア
    //  Z  ゼロ除算またはオーバーフローのとき不定。商が0のときセット。それ以外はクリア
    //  V  ゼロ除算のとき不定。オーバーフローのときセット。それ以外はクリア
    //  C  常にクリア
    XEiJ.mpuCycleCount += 22;  //最大
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordZeroData (efaAnyWord (ea), XEiJ.regSRS);  //除数。pcwz。イミディエイトを分離
    int x = XEiJ.regRn[qqq];  //被除数
    if (y == 0) {  //ゼロ除算
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z | XEiJ.REG_CCR_V)  //XとNとZとVは変化しない
                     );  //Cは常にクリア
      m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
      throw M68kException.m6eSignal;
    }
    //無理にintで符号なし除算をやろうとするよりもdoubleにキャストしてから割ったほうが速い
    //  intの除算をdoubleの除算器で行うプロセッサならばなおさら
    //被除数を符号なし32ビットとみなすためlongを経由してdoubleに変換する
    //doubleからlongやintへのキャストは小数点以下が切り捨てられ、オーバーフローは表現できる絶対値最大の値になる
    //doubleから直接intに戻しているので0xffffffff/0x0001=0xffffffffが絶対値最大の0x7fffffffになってしまうが、
    //DIVU.Wではオーバーフローになることに変わりはないのでよいことにする
    //  符号なし32ビットの0xffffffffにしたいときは戻すときもlongを経由すればよい
    int z = (int) ((double) ((long) x & 0xffffffffL) / (double) y);  //商
    if (z >>> 16 != 0) {  //オーバーフローあり
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z) |  //XとNとZは変化しない
                     XEiJ.REG_CCR_V  //Vは常にセット
                     );  //Cは常にクリア
    } else {  //オーバーフローなし
      XEiJ.regRn[qqq] = x - y * z << 16 | z;  //余り<<16|商
      z = (short) z;
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                     (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                     //Vは常にクリア
                     );  //Cは常にクリア
    }  //if オーバーフローあり/オーバーフローなし
  }  //irpDivuWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SBCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_000_rrr
  //SBCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1000_qqq_100_001_rrr
  //OR.B Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_100_mmm_rrr
  public static void irpOrToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //OR.B Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] | mmuReadByteSignData (a, XEiJ.regSRS);
      mmuWriteByteData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea < XEiJ.EA_AR) {  //SBCD.B Dr,Dq
      int qqq = XEiJ.regOC >> 9 & 7;
      XEiJ.mpuCycleCount++;
      int x;
      XEiJ.regRn[qqq] = ~0xff & (x = XEiJ.regRn[qqq]) | irpSbcd (x, XEiJ.regRn[ea]);
    } else {  //SBCD.B -(Ar),-(Aq)
      XEiJ.mpuCycleCount += 2;
      m60Incremented -= 1L << (ea << 3);
      int a = m60Address = --XEiJ.regRn[ea];  //このr[ea]はアドレスレジスタ
      int y = mmuReadByteZeroData (a, XEiJ.regSRS);
      int aqq = (XEiJ.regOC >> 9) - (64 - 8);
      m60Incremented -= 1L << (aqq << 3);
      a = m60Address = --XEiJ.regRn[aqq];
      mmuWriteByteData (a, irpSbcd (mmuModifyByteZeroData (a, XEiJ.regSRS), y), XEiJ.regSRS);
    }
  }  //irpOrToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //PACK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_101_000_rrr-{data}
  //PACK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_101_001_rrr-{data}
  //OR.W Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_101_mmm_rrr
  //
  //PACK Dr,Dq,#<data>
  //PACK -(Ar),-(Aq),#<data>
  //  PACK/UNPKは第1オペランドのソースと第2オペランドのデスティネーションのサイズが違う。パックされていない方がワードでされている方がバイト
  //  10の位を4ビット右または左にシフトする。第3オペランドの補正値はワードでパックされていない方に加算する。CCRは変化しない
  public static void irpOrToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //OR.W Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      int z;
      mmuWriteWordData (a, z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] | mmuModifyWordSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    } else if (ea < XEiJ.EA_AR) {  //PACK Dr,Dq,#<data>
      XEiJ.mpuCycleCount += 2;
      int qqq = XEiJ.regOC >> 9 & 7;
      int t = XEiJ.regRn[ea] + mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      XEiJ.regRn[qqq] = ~0xff & XEiJ.regRn[qqq] | t >> 4 & 0xf0 | t & 15;
    } else {  //PACK -(Ar),-(Aq),#<data>
      int o = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      m60Incremented -= 2L << (ea << 3);
      int a = m60Address = XEiJ.regRn[ea] -= 2;
      int t = mmuReadWordSignData (a, XEiJ.regSRS) + o;  //020以上なのでアドレスエラーは出ない
      int aqq = (XEiJ.regOC >> 9) - (64 - 8);
      m60Incremented -= 1L << (aqq << 3);
      a = m60Address = --XEiJ.regRn[aqq];
      mmuWriteByteData (a, t >> 4 & 0xf0 | t & 15, XEiJ.regSRS);
    }
  }  //irpOrToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //UNPK Dr,Dq,#<data>                              |-|--2346|-|-----|-----|          |1000_qqq_110_000_rrr-{data}
  //UNPK -(Ar),-(Aq),#<data>                        |-|--2346|-|-----|-----|          |1000_qqq_110_001_rrr-{data}
  //OR.L Dq,<ea>                                    |-|012346|-|-UUUU|-**00|  M+-WXZ  |1000_qqq_110_mmm_rrr
  //
  //UNPK Dr,Dq,#<data>
  //UNPK -(Ar),-(Aq),#<data>
  //  PACK/UNPKは第1オペランドのソースと第2オペランドのデスティネーションのサイズが違う。パックされていない方がワードでされている方がバイト
  //  10の位を4ビット右または左にシフトする。第3オペランドの補正値はワードでパックされていない方に加算する。CCRは変化しない
  public static void irpOrToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //OR.L Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      int z;
      mmuWriteLongData (a, z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] | mmuModifyLongData (a, XEiJ.regSRS), XEiJ.regSRS);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    } else if (ea < XEiJ.EA_AR) {  //UNPK Dr,Dq,#<data>
      int qqq = XEiJ.regOC >> 9 & 7;
      int t = XEiJ.regRn[ea];
      XEiJ.regRn[qqq] = ~0xffff & XEiJ.regRn[qqq] | (char) ((t << 4 & 0x0f00 | t & 15) + mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS));  //pcws
    } else {  //UNPK -(Ar),-(Aq),#<data>
      int o = mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS);  //pcws
      m60Incremented -= 1L << (ea << 3);
      int a = m60Address = --XEiJ.regRn[ea];
      int t = mmuReadByteSignData (a, XEiJ.regSRS);
      int aqq = (XEiJ.regOC >> 9) - (64 - 8);
      m60Incremented -= 2L << (aqq << 3);
      a = m60Address = XEiJ.regRn[aqq] -= 2;
      mmuWriteWordData (a, (t << 4 & 0x0f00 | t & 15) + o, XEiJ.regSRS);  //020以上なのでアドレスエラーは出ない
    }
  }  //irpOrToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //DIVS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1000_qqq_111_mmm_rrr
  //
  //DIVS.W <ea>,Dq
  //  DIVSの余りの符号は被除数と一致
  //  M68000PRMでDIVS.Wのアドレッシングモードがデータ可変と書かれているのはデータの間違い
  public static void irpDivsWord () throws M68kException {
    //  X  変化しない
    //  N  ゼロ除算またはオーバーフローのとき不定。商が負のときセット。それ以外はクリア
    //  Z  ゼロ除算またはオーバーフローのとき不定。商が0のときセット。それ以外はクリア
    //  V  ゼロ除算のとき不定。オーバーフローのときセット。それ以外はクリア
    //  C  常にクリア
    //divsの余りの符号は被除数と一致
    //Javaの除算演算子の挙動
    //   10 /  3 ==  3   10 %  3 ==  1   10 =  3 *  3 +  1
    //   10 / -3 == -3   10 % -3 ==  1   10 = -3 * -3 +  1
    //  -10 /  3 == -3  -10 %  3 == -1  -10 =  3 * -3 + -1
    //  -10 / -3 ==  3  -10 % -3 == -1  -10 = -3 *  3 + -1
    XEiJ.mpuCycleCount += 22;  //最大
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //除数。pcws。イミディエイトを分離
    int x = XEiJ.regRn[qqq];  //被除数
    if (y == 0) {  //ゼロ除算
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z | XEiJ.REG_CCR_V)  //XとNとZとVは変化しない
                     );  //Cは常にクリア
      m60Address = XEiJ.regPC0;  //アドレスは命令の先頭
      M68kException.m6eNumber = M68kException.M6E_DIVIDE_BY_ZERO;
      throw M68kException.m6eSignal;
    }
    int z = x / y;  //商
    if ((short) z != z) {  //オーバーフローあり
      //Dqは変化しない
      XEiJ.regCCR = (XEiJ.regCCR & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_N | XEiJ.REG_CCR_Z) |  //XとNとZは変化しない
                     XEiJ.REG_CCR_V  //Vは常にセット
                     );  //Cは常にクリア
    } else {  //オーバーフローなし
      XEiJ.regRn[qqq] = x - y * z << 16 | (char) z;  //Dqは余り<<16|商&$ffff
      XEiJ.regCCR = (XEiJ.regCCR & XEiJ.REG_CCR_X |  //Xは変化しない
                     (z < 0 ? XEiJ.REG_CCR_N : 0) |  //Nは商が負のときセット、さもなくばクリア
                     (z == 0 ? XEiJ.REG_CCR_Z : 0)  //Zは商が0のときセット、さもなくばクリア
                     //Vは常にクリア
                     );  //Cは常にクリア
    }
  }  //irpDivsWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUB.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1001_qqq_000_mmm_rrr
  public static void irpSubToRegByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離
    x = XEiJ.regRn[qqq];
    z = x - y;
    XEiJ.regRn[qqq] = ~255 & x | 255 & z;
    XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
           ((x ^ y) & (x ^ z)) >> 6 & XEiJ.REG_CCR_V |
           (byte) (x & (y ^ z) ^ (y | z)) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_byte
  }  //irpSubToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUB.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_001_mmm_rrr
  public static void irpSubToRegWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離
    x = XEiJ.regRn[qqq];
    z = x - y;
    XEiJ.regRn[qqq] = ~65535 & x | (char) z;
    XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
           ((x ^ y) & (x ^ z)) >> 14 & XEiJ.REG_CCR_V |
           (short) (x & (y ^ z) ^ (y | z)) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_word
  }  //irpSubToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUB.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1001_qqq_010_mmm_rrr
  public static void irpSubToRegLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    XEiJ.mpuCycleCount++;
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcls。イミディエイトを分離
    x = XEiJ.regRn[qqq];
    z = x - y;
    XEiJ.regRn[qqq] = z;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ y) & (x ^ z)) >> 30 & XEiJ.REG_CCR_V |
           (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
  }  //irpSubToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr
  //SUB.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_011_mmm_rrr [SUBA.W <ea>,Aq]
  //CLR.W Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_011_001_rrr [SUBA.W Ar,Ar]
  //
  //SUBA.W <ea>,Aq
  //  ソースを符号拡張してロングで減算する
  public static void irpSubaWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= z;  //r[op >> 9 & 15] -= ea < XEiJ.EA_MM ? (short) r[ea] : rws (efaAnyWord (ea));は不可
    //ccrは変化しない
  }  //irpSubaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_100_000_rrr
  //SUBX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_100_001_rrr
  //SUB.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_100_mmm_rrr
  public static void irpSubToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //SUBX.B Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount++;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~255 & x | 255 & z;
      } else {  //SUBX.B -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 2;
        m60Incremented -= 1L << (ea << 3);
        a = m60Address = --XEiJ.regRn[ea];
        y = mmuReadByteSignData (a, XEiJ.regSRS);  //このr[ea]はアドレスレジスタ
        int aqq = XEiJ.regOC >> 9 & 15;  //1qqq=aqq
        m60Incremented -= 1L << (aqq << 3);
        a = m60Address = --XEiJ.regRn[aqq];
        x = mmuModifyByteSignData (a, XEiJ.regSRS);
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        mmuWriteByteData (a, z, XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 4 & XEiJ.REG_CCR_N | (255 & z) - 1 >> 6 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //SUBXはZをクリアすることはあるがセットすることはない
             ((x ^ y) & (x ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) (x & (y ^ z) ^ (y | z)) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subx_byte
    } else {  //SUB.B Dq,<ea>
      XEiJ.mpuCycleCount++;
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      a = efaMltByte (ea);
      x = mmuModifyByteSignData (a, XEiJ.regSRS);
      z = x - y;
      mmuWriteByteData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
             ((x ^ y) & (x ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) (x & (y ^ z) ^ (y | z)) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_byte
    }
  }  //irpSubToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_101_000_rrr
  //SUBX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_101_001_rrr
  //SUB.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_101_mmm_rrr
  public static void irpSubToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //SUBX.W Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount++;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~65535 & x | (char) z;
      } else {  //SUBX.W -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 2;
        m60Incremented -= 2L << (ea << 3);
        a = m60Address = XEiJ.regRn[ea] -= 2;
        y = mmuReadWordSignData (a, XEiJ.regSRS);  //このr[ea]はアドレスレジスタ
        int aqq = XEiJ.regOC >> 9 & 15;
        m60Incremented -= 2L << (aqq << 3);
        a = m60Address = XEiJ.regRn[aqq] -= 2;
        x = mmuModifyWordSignData (a, XEiJ.regSRS);
        z = x - y - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        mmuWriteWordData (a, z, XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //ADDXはZをクリアすることはあるがセットすることはない
             ((x ^ y) & (x ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) (x & (y ^ z) ^ (y | z)) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subx_word
    } else {  //SUB.W Dq,<ea>
      XEiJ.mpuCycleCount++;
      y = (short) XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      a = efaMltWord (ea);
      x = mmuModifyWordSignData (a, XEiJ.regSRS);
      z = x - y;
      mmuWriteWordData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
             ((x ^ y) & (x ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) (x & (y ^ z) ^ (y | z)) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub_word
    }
  }  //irpSubToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1001_qqq_110_000_rrr
  //SUBX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1001_qqq_110_001_rrr
  //SUB.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1001_qqq_110_mmm_rrr
  public static void irpSubToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_MM) {
      int x;
      int y;
      int z;
      if (ea < XEiJ.EA_AR) {  //SUBX.L Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[qqq] = z = (x = XEiJ.regRn[qqq]) - (y = XEiJ.regRn[ea]) - (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
      } else {  //SUBX.L -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 2;
        m60Incremented -= 4L << (ea << 3);
        int a = m60Address = XEiJ.regRn[ea] -= 4;  //このr[ea]はアドレスレジスタ
        y = mmuReadLongData (a, XEiJ.regSRS);
        int aqq = XEiJ.regOC >> 9 & 15;
        m60Incremented -= 4L << (aqq << 3);
        a = m60Address = XEiJ.regRn[aqq] -= 4;
        mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) - y - (XEiJ.regCCR >> 4), XEiJ.regSRS);  //Xの左側はすべて0なのでCCR_X&を省略
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_subx
    } else {  //SUB.L Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      int x;
      int y;
      int z;
      mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) - (y = XEiJ.regRn[XEiJ.regOC >> 9 & 7]), XEiJ.regSRS);
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_sub
    }
  }  //irpSubToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SUBA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr
  //SUB.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1001_qqq_111_mmm_rrr [SUBA.L <ea>,Aq]
  //CLR.L Ar                                        |A|012346|-|-----|-----| A        |1001_rrr_111_001_rrr [SUBA.L Ar,Ar]
  public static void irpSubaLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    XEiJ.mpuCycleCount++;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcls。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] -= z;  //r[op >> 9 & 15] -= ea < XEiJ.EA_MM ? r[ea] : rls (efaAnyLong (ea));は不可
    //ccrは変化しない
  }  //irpSubaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //SXCALL <name>                                   |A|012346|-|UUUUU|*****|          |1010_0dd_ddd_ddd_ddd [ALINE #<data>]
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ALINE #<data>                                   |-|012346|-|UUUUU|*****|          |1010_ddd_ddd_ddd_ddd (line 1010 emulator)
  public static void irpAline () throws M68kException {
    irpExceptionFormat0 (M68kException.M6E_LINE_1010_EMULATOR << 2, XEiJ.regPC0);  //pcは命令の先頭
  }  //irpAline

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMP.B <ea>,Dq                                   |-|012346|-|-UUUU|-****|D M+-WXZPI|1011_qqq_000_mmm_rrr
  public static void irpCmpByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    int z = (byte) ((x = (byte) XEiJ.regRn[XEiJ.regOC >> 9 & 7]) - (y = ea < XEiJ.EA_AR ? (byte) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS)));  //pcbs。イミディエイトを分離
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMP.W <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_001_mmm_rrr
  public static void irpCmpWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    int z = (short) ((x = (short) XEiJ.regRn[XEiJ.regOC >> 9 & 7]) - (y = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS)));  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMP.L <ea>,Dq                                   |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_010_mmm_rrr
  public static void irpCmpLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int x;
    int y;
    int z = (x = XEiJ.regRn[XEiJ.regOC >> 9 & 7]) - (y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS));  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcls。イミディエイトを分離
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPA.W <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr
  //CMP.W <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_011_mmm_rrr [CMPA.W <ea>,Aq]
  //
  //CMPA.W <ea>,Aq
  //  ソースを符号拡張してロングで比較する
  public static void irpCmpaWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    //ソースを符号拡張してからロングで比較する
    int y = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int x;
    int z = (x = XEiJ.regRn[XEiJ.regOC >> 9 & 15]) - y;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EOR.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_100_mmm_rrr
  //CMPM.B (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_100_001_rrr
  public static void irpEorByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //CMPM.B (Ar)+,(Aq)+
      XEiJ.mpuCycleCount += 2;
      m60Incremented += 1L << (ea << 3);
      int a = m60Address = XEiJ.regRn[ea]++;  //このr[ea]はアドレスレジスタ
      int y = mmuReadByteSignData (a, XEiJ.regSRS);
      int x;
      int aqq = XEiJ.regOC >> 9 & 15;
      m60Incremented += 1L << (aqq << 3);
      a = m60Address = XEiJ.regRn[aqq]++;
      int z = (byte) ((x = mmuReadByteSignData (a, XEiJ.regSRS)) - y);
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    } else {
      int qqq = XEiJ.regOC >> 9 & 7;
      int z;
      if (ea < XEiJ.EA_AR) {  //EOR.B Dq,Dr
        XEiJ.mpuCycleCount++;
        z = XEiJ.regRn[ea] ^= 255 & XEiJ.regRn[qqq];  //0拡張してからEOR
      } else {  //EOR.B Dq,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltByte (ea);
        mmuWriteByteData (a, z = XEiJ.regRn[qqq] ^ mmuModifyByteSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      }
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    }
  }  //irpEorByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EOR.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_101_mmm_rrr
  //CMPM.W (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_101_001_rrr
  public static void irpEorWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int rrr = XEiJ.regOC & 7;
    int mmm = ea >> 3;
    if (mmm == XEiJ.MMM_AR) {  //CMPM.W (Ar)+,(Aq)+
      XEiJ.mpuCycleCount += 2;
      m60Incremented += 2L << (ea << 3);
      int a = m60Address = (XEiJ.regRn[ea] += 2) - 2;  //このr[ea]はアドレスレジスタ
      int y = mmuReadWordSignData (a, XEiJ.regSRS);
      int x;
      int aqq = XEiJ.regOC >> 9 & 15;
      m60Incremented += 2L << (aqq << 3);
      a = m60Address = (XEiJ.regRn[aqq] += 2) - 2;
      int z = (short) ((x = mmuReadWordSignData (a, XEiJ.regSRS)) - y);
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    } else {
      int qqq = XEiJ.regOC >> 9 & 7;
      int z;
      if (ea < XEiJ.EA_AR) {  //EOR.W Dq,Dr
        XEiJ.mpuCycleCount++;
        z = XEiJ.regRn[rrr] ^= (char) XEiJ.regRn[qqq];  //0拡張してからEOR
      } else {  //EOR.W Dq,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltWord (ea);
        mmuWriteWordData (a, z = XEiJ.regRn[qqq] ^ mmuModifyWordSignData (a, XEiJ.regSRS), XEiJ.regSRS);
      }
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpEorWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EOR.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|D M+-WXZ  |1011_qqq_110_mmm_rrr
  //CMPM.L (Ar)+,(Aq)+                              |-|012346|-|-UUUU|-****|          |1011_qqq_110_001_rrr
  public static void irpEorLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >> 3 == XEiJ.MMM_AR) {  //CMPM.L (Ar)+,(Aq)+
      XEiJ.mpuCycleCount += 2;
      m60Incremented += 4L << (ea << 3);
      int a = m60Address = (XEiJ.regRn[ea] += 4) - 4;  //このr[ea]はアドレスレジスタ
      int y = mmuReadLongData (a, XEiJ.regSRS);
      int x;
      int aqq = XEiJ.regOC >> 9 & 15;
      m60Incremented += 4L << (aqq << 3);
      a = m60Address = (XEiJ.regRn[aqq] += 4) - 4;
      int z = (x = mmuReadLongData (a, XEiJ.regSRS)) - y;
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
             ((x ^ y) & (x ^ z)) >>> 31 << 1 |
             (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
    } else {
      int qqq = XEiJ.regOC >> 9 & 7;
      int z;
      if (ea < XEiJ.EA_AR) {  //EOR.L Dq,Dr
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[ea] = z = XEiJ.regRn[ea] ^ XEiJ.regRn[qqq];
      } else {  //EOR.L Dq,<mem>
        XEiJ.mpuCycleCount++;
        int a = efaMltLong (ea);
        mmuWriteLongData (a, z = mmuModifyLongData (a, XEiJ.regSRS) ^ XEiJ.regRn[qqq], XEiJ.regSRS);
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    }
  }  //irpEorLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //CMPA.L <ea>,Aq                                  |-|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr
  //CMP.L <ea>,Aq                                   |A|012346|-|-UUUU|-****|DAM+-WXZPI|1011_qqq_111_mmm_rrr [CMPA.L <ea>,Aq]
  public static void irpCmpaLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcls。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    int x;
    int z = (x = XEiJ.regRn[XEiJ.regOC >> 9 & 15]) - y;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) |
           ((x ^ y) & (x ^ z)) >>> 31 << 1 |
           (x & (y ^ z) ^ (y | z)) >>> 31);  //ccr_cmp
  }  //irpCmpaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //AND.B <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_000_mmm_rrr
  public static void irpAndToRegByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & (XEiJ.regRn[XEiJ.regOC >> 9 & 7] &= ~255 | (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS)))];  //ccr_tst_byte。pcbs。イミディエイトを分離。1拡張してからAND
  }  //irpAndToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //AND.W <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_001_mmm_rrr
  public static void irpAndToRegWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] &= ~65535 | (ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS));  //pcws。イミディエイトを分離。1拡張してからAND
    XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
  }  //irpAndToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //AND.L <ea>,Dq                                   |-|012346|-|-UUUU|-**00|D M+-WXZPI|1100_qqq_010_mmm_rrr
  public static void irpAndToRegLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] &= ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //pcls。イミディエイトを分離
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpAndToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MULU.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_011_mmm_rrr
  public static void irpMuluWord () throws M68kException {
    XEiJ.mpuCycleCount += 2;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (char) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordZeroExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordZeroData (efaAnyWord (ea), XEiJ.regSRS);  //pcwz。イミディエイトを分離
    int z;
    XEiJ.regRn[qqq] = z = (char) XEiJ.regRn[qqq] * y;  //積の下位32ビット。オーバーフローは無視
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMuluWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ABCD.B Dr,Dq                                    |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_000_rrr
  //ABCD.B -(Ar),-(Aq)                              |-|012346|-|UUUUU|*U*U*|          |1100_qqq_100_001_rrr
  //AND.B Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_100_mmm_rrr
  public static void irpAndToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea >= XEiJ.EA_MM) {  //AND.B Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltByte (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & mmuModifyByteSignData (a, XEiJ.regSRS);
      mmuWriteByteData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.MPU_TSTB_TABLE[255 & z];  //ccr_tst_byte
    } else if (ea < XEiJ.EA_AR) {  //ABCD.B Dr,Dq
      int qqq = XEiJ.regOC >> 9 & 7;
      XEiJ.mpuCycleCount++;
      XEiJ.regRn[qqq] = ~0xff & XEiJ.regRn[qqq] | irpAbcd (XEiJ.regRn[qqq], XEiJ.regRn[ea]);
    } else {  //ABCD.B -(Ar),-(Aq)
      XEiJ.mpuCycleCount += 2;
      m60Incremented -= 1L << (ea << 3);
      int a = m60Address = --XEiJ.regRn[ea];  //このr[ea]はアドレスレジスタ
      int y = mmuReadByteZeroData (a, XEiJ.regSRS);
      int aqq = (XEiJ.regOC >> 9) - (96 - 8);
      m60Incremented -= 1L << (aqq << 3);
      a = m60Address = --XEiJ.regRn[aqq];
      mmuWriteByteData (a, irpAbcd (mmuModifyByteZeroData (a, XEiJ.regSRS), y), XEiJ.regSRS);
    }
  }  //irpAndToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXG.L Dq,Dr                                     |-|012346|-|-----|-----|          |1100_qqq_101_000_rrr
  //EXG.L Aq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_101_001_rrr
  //AND.W Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_101_mmm_rrr
  public static void irpAndToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_MM) {  //EXG
      XEiJ.mpuCycleCount++;
      if (ea < XEiJ.EA_AR) {  //EXG.L Dq,Dr
        int qqq = XEiJ.regOC >> 9 & 7;
        int t = XEiJ.regRn[qqq];
        XEiJ.regRn[qqq] = XEiJ.regRn[ea];
        XEiJ.regRn[ea] = t;
      } else {  //EXG.L Aq,Ar
        int aqq = (XEiJ.regOC >> 9) - (96 - 8);
        int t = XEiJ.regRn[aqq];
        XEiJ.regRn[aqq] = XEiJ.regRn[ea];  //このr[ea]アドレスレジスタ
        XEiJ.regRn[ea] = t;  //このr[ea]はアドレスレジスタ
      }
    } else {  //AND.W Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltWord (ea);
      int z = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & mmuModifyWordSignData (a, XEiJ.regSRS);
      mmuWriteWordData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (char) z - 1 >> 31 & XEiJ.REG_CCR_Z | ((short) z < 0 ? XEiJ.REG_CCR_N : 0);  //ccr_tst_word
    }
  }  //irpAndToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //EXG.L Dq,Ar                                     |-|012346|-|-----|-----|          |1100_qqq_110_001_rrr
  //AND.L Dq,<ea>                                   |-|012346|-|-UUUU|-**00|  M+-WXZ  |1100_qqq_110_mmm_rrr
  public static void irpAndToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    if (ea >> 3 == XEiJ.MMM_AR) {  //EXG.L Dq,Ar
      XEiJ.mpuCycleCount++;
      int t = XEiJ.regRn[qqq];
      XEiJ.regRn[qqq] = XEiJ.regRn[ea];  //このr[ea]はアドレスレジスタ
      XEiJ.regRn[ea] = t;  //このr[ea]はアドレスレジスタ
    } else {  //AND.L Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      int z;
      mmuWriteLongData (a, z = mmuModifyLongData (a, XEiJ.regSRS) & XEiJ.regRn[qqq], XEiJ.regSRS);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
    }
  }  //irpAndToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //MULS.W <ea>,Dq                                  |-|012346|-|-UUUU|-***0|D M+-WXZPI|1100_qqq_111_mmm_rrr
  public static void irpMulsWord () throws M68kException {
    XEiJ.mpuCycleCount += 2;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int y = ea < XEiJ.EA_AR ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //pcws。イミディエイトを分離
    int z;
    XEiJ.regRn[qqq] = z = (short) XEiJ.regRn[qqq] * y;  //積の下位32ビット。オーバーフローは無視
    XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X);  //ccr_tst
  }  //irpMulsWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADD.B <ea>,Dq                                   |-|012346|-|UUUUU|*****|D M+-WXZPI|1101_qqq_000_mmm_rrr
  public static void irpAddToRegByte () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_AR ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadByteSignExword ((XEiJ.regPC += 2) - 1, XEiJ.regSRS) : mmuReadByteSignData (efaAnyByte (ea), XEiJ.regSRS);  //pcbs。イミディエイトを分離
    x = XEiJ.regRn[qqq];
    z = x + y;
    XEiJ.regRn[qqq] = ~255 & x | 255 & z;
    XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
           ((x ^ z) & (y ^ z)) >> 6 & XEiJ.REG_CCR_V |
           (byte) ((x | y) ^ (x ^ y) & z) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_byte
  }  //irpAddToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADD.W <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_001_mmm_rrr
  public static void irpAddToRegWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離
    x = XEiJ.regRn[qqq];
    z = x + y;
    XEiJ.regRn[qqq] = ~65535 & x | (char) z;
    XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
           ((x ^ z) & (y ^ z)) >> 14 & XEiJ.REG_CCR_V |
           (short) ((x | y) ^ (x ^ y) & z) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_word
  }  //irpAddToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADD.L <ea>,Dq                                   |-|012346|-|UUUUU|*****|DAM+-WXZPI|1101_qqq_010_mmm_rrr
  public static void irpAddToRegLong () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int qqq = XEiJ.regOC >> 9 & 7;
    int x, y, z;
    y = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcls。イミディエイトを分離
    x = XEiJ.regRn[qqq];
    z = x + y;
    XEiJ.regRn[qqq] = z;
    XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
           ((x ^ z) & (y ^ z)) >> 30 & XEiJ.REG_CCR_V |
           ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
  }  //irpAddToRegLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDA.W <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr
  //ADD.W <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_011_mmm_rrr [ADDA.W <ea>,Aq]
  //
  //ADDA.W <ea>,Aq
  //  ソースを符号拡張してロングで加算する
  public static void irpAddaWord () throws M68kException {
    XEiJ.mpuCycleCount++;
    int ea = XEiJ.regOC & 63;
    int z = ea < XEiJ.EA_MM ? (short) XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadWordSignExword ((XEiJ.regPC += 2) - 2, XEiJ.regSRS) : mmuReadWordSignData (efaAnyWord (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcws。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] += z;  //r[op >> 9 & 15] += ea < XEiJ.EA_MM ? (short) r[ea] : rws (efaAnyWord (ea));は不可
    //ccrは変化しない
  }  //irpAddaWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDX.B Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_100_000_rrr
  //ADDX.B -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_100_001_rrr
  //ADD.B Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_100_mmm_rrr
  public static void irpAddToMemByte () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //ADDX.B Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount++;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~255 & x | 255 & z;
      } else {  //ADDX.B -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 2;
        m60Incremented -= 1L << (ea << 3);
        a = m60Address = --XEiJ.regRn[ea];  //このr[ea]はアドレスレジスタ
        y = mmuReadByteSignData (a, XEiJ.regSRS);
        int aqq = XEiJ.regOC >> 9 & 15;  //1qqq=aqq
        m60Incremented -= 1L << (aqq << 3);
        a = m60Address = --XEiJ.regRn[aqq];
        x = mmuModifyByteSignData (a, XEiJ.regSRS);
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        mmuWriteByteData (a, z, XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 4 & XEiJ.REG_CCR_N | (255 & z) - 1 >> 6 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //ADDXはZをクリアすることはあるがセットすることはない
             ((x ^ z) & (y ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) ((x | y) ^ (x ^ y) & z) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addx_byte
    } else {  //ADD.B Dq,<ea>
      XEiJ.mpuCycleCount++;
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      a = efaMltByte (ea);
      x = mmuModifyByteSignData (a, XEiJ.regSRS);
      z = x + y;
      mmuWriteByteData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = (XEiJ.MPU_TSTB_TABLE[255 & z] |
             ((x ^ z) & (y ^ z)) >> 6 & XEiJ.REG_CCR_V |
             (byte) ((x | y) ^ (x ^ y) & z) >> 7 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_byte
    }
  }  //irpAddToMemByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDX.W Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_101_000_rrr
  //ADDX.W -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_101_001_rrr
  //ADD.W Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_101_mmm_rrr
  public static void irpAddToMemWord () throws M68kException {
    int ea = XEiJ.regOC & 63;
    int a, x, y, z;
    if (ea < XEiJ.EA_MM) {
      if (ea < XEiJ.EA_AR) {  //ADDX.W Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount++;
        y = XEiJ.regRn[ea];
        x = XEiJ.regRn[qqq];
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        XEiJ.regRn[qqq] = ~65535 & x | (char) z;
      } else {  //ADDX.W -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 2;
        m60Incremented -= 2L << (ea << 3);
        a = m60Address = XEiJ.regRn[ea] -= 2;  //このr[ea]はアドレスレジスタ
        y = mmuReadWordSignData (a, XEiJ.regSRS);
        int aqq = XEiJ.regOC >> 9 & 15;
        m60Incremented -= 2L << (aqq << 3);
        a = m60Address = XEiJ.regRn[aqq] -= 2;
        x = mmuModifyWordSignData (a, XEiJ.regSRS);
        z = x + y + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
        mmuWriteWordData (a, z, XEiJ.regSRS);
      }
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.regCCR & XEiJ.REG_CCR_Z |  //ADDXはZをクリアすることはあるがセットすることはない
             ((x ^ z) & (y ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) ((x | y) ^ (x ^ y) & z) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addx_word
    } else {  //ADD.W Dq,<ea>
      XEiJ.mpuCycleCount++;
      a = efaMltWord (ea);
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7];
      x = mmuModifyWordSignData (a, XEiJ.regSRS);
      z = x + y;
      mmuWriteWordData (a, z, XEiJ.regSRS);
      XEiJ.regCCR = (z >> 12 & XEiJ.REG_CCR_N | (char) z - 1 >> 14 & XEiJ.REG_CCR_Z |
             ((x ^ z) & (y ^ z)) >> 14 & XEiJ.REG_CCR_V |
             (short) ((x | y) ^ (x ^ y) & z) >> 15 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add_word
    }
  }  //irpAddToMemWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDX.L Dr,Dq                                    |-|012346|-|*UUUU|*****|          |1101_qqq_110_000_rrr
  //ADDX.L -(Ar),-(Aq)                              |-|012346|-|*UUUU|*****|          |1101_qqq_110_001_rrr
  //ADD.L Dq,<ea>                                   |-|012346|-|UUUUU|*****|  M+-WXZ  |1101_qqq_110_mmm_rrr
  public static void irpAddToMemLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    if (ea < XEiJ.EA_MM) {
      int x;
      int y;
      int z;
      if (ea < XEiJ.EA_AR) {  //ADDX.L Dr,Dq
        int qqq = XEiJ.regOC >> 9 & 7;
        XEiJ.mpuCycleCount++;
        XEiJ.regRn[qqq] = z = (x = XEiJ.regRn[qqq]) + (y = XEiJ.regRn[ea]) + (XEiJ.regCCR >> 4);  //Xの左側はすべて0なのでCCR_X&を省略
      } else {  //ADDX.L -(Ar),-(Aq)
        XEiJ.mpuCycleCount += 2;
        m60Incremented -= 4L << (ea << 3);
        int a = m60Address = XEiJ.regRn[ea] -= 4;  //このr[ea]はアドレスレジスタ
        y = mmuReadLongData (a, XEiJ.regSRS);
        int aqq = XEiJ.regOC >> 9 & 15;
        m60Incremented -= 4L << (aqq << 3);
        a = m60Address = XEiJ.regRn[aqq] -= 4;
        mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) + y + (XEiJ.regCCR >> 4), XEiJ.regSRS);  //Xの左側はすべて0なのでCCR_X&を省略
      }
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_Z : 0) |
             ((x ^ z) & (y ^ z)) >>> 31 << 1 |
             ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_addx
    } else {  //ADD.L Dq,<ea>
      XEiJ.mpuCycleCount++;
      int a = efaMltLong (ea);
      int x;
      int y;
      int z;
      mmuWriteLongData (a, z = (x = mmuModifyLongData (a, XEiJ.regSRS)) + (y = XEiJ.regRn[XEiJ.regOC >> 9 & 7]), XEiJ.regSRS);
      XEiJ.regCCR = (z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) |
             ((x ^ z) & (y ^ z)) >>> 31 << 1 |
             ((x | y) ^ (x ^ y) & z) >> 31 & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C));  //ccr_add
    }
  }  //irpAddToMemLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ADDA.L <ea>,Aq                                  |-|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr
  //ADD.L <ea>,Aq                                   |A|012346|-|-----|-----|DAM+-WXZPI|1101_qqq_111_mmm_rrr [ADDA.L <ea>,Aq]
  public static void irpAddaLong () throws M68kException {
    int ea = XEiJ.regOC & 63;
    XEiJ.mpuCycleCount++;
    int z = ea < XEiJ.EA_MM ? XEiJ.regRn[ea] : ea == XEiJ.EA_IM ? mmuReadLongExword ((XEiJ.regPC += 4) - 4, XEiJ.regSRS) : mmuReadLongData (efaAnyLong (ea), XEiJ.regSRS);  //このr[ea]はデータレジスタまたはアドレスレジスタ。pcls。イミディエイトを分離。ここでAqが変化する可能性があることに注意
    XEiJ.regRn[XEiJ.regOC >> 9 & 15] += z;  //r[op >> 9 & 15] += ea < XEiJ.EA_MM ? r[ea] : rls (efaAnyLong (ea));は不可
    //ccrは変化しない
  }  //irpAddaLong

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_000_rrr
  //LSR.B #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_000_001_rrr
  //ROXR.B #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_000_010_rrr
  //ROR.B #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_011_rrr
  //ASR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_100_rrr
  //LSR.B Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_000_101_rrr
  //ROXR.B Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_000_110_rrr
  //ROR.B Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_000_111_rrr
  //ASR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_000_rrr [ASR.B #1,Dr]
  //LSR.B Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_000_001_rrr [LSR.B #1,Dr]
  //ROXR.B Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_000_010_rrr [ROXR.B #1,Dr]
  //ROR.B Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_000_011_rrr [ROR.B #1,Dr]
  //
  //ASR.B #<data>,Dr
  //ASR.B Dq,Dr
  //  算術右シフトバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｱｱｲｳｴｵｶｷ ｸｱ*0ｸ Z=ｱｲｳｴｵｶｷ==0
  //     2 ........................ｱｱｱｲｳｴｵｶ ｷｱ*0ｷ Z=ｱｲｳｴｵｶ==0
  //     3 ........................ｱｱｱｱｲｳｴｵ ｶｱ*0ｶ Z=ｱｲｳｴｵ==0
  //     4 ........................ｱｱｱｱｱｲｳｴ ｵｱ*0ｵ Z=ｱｲｳｴ==0
  //     5 ........................ｱｱｱｱｱｱｲｳ ｴｱ*0ｴ Z=ｱｲｳ==0
  //     6 ........................ｱｱｱｱｱｱｱｲ ｳｱ*0ｳ Z=ｱｲ==0
  //     7 ........................ｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //     8 ........................ｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSR.B #<data>,Dr
  //LSR.B Dq,Dr
  //  論理右シフトバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................0ｱｲｳｴｵｶｷ ｸ0*0ｸ Z=ｱｲｳｴｵｶｷ==0
  //     2 ........................00ｱｲｳｴｵｶ ｷ0*0ｷ Z=ｱｲｳｴｵｶ==0
  //     3 ........................000ｱｲｳｴｵ ｶ0*0ｶ Z=ｱｲｳｴｵ==0
  //     4 ........................0000ｱｲｳｴ ｵ0*0ｵ Z=ｱｲｳｴ==0
  //     5 ........................00000ｱｲｳ ｴ0*0ｴ Z=ｱｲｳ==0
  //     6 ........................000000ｱｲ ｳ0*0ｳ Z=ｱｲ==0
  //     7 ........................0000000ｱ ｲ0*0ｲ Z=ｱ==0
  //     8 ........................00000000 ｱ010ｱ
  //     9 ........................00000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROR.B #<data>,Dr
  //ROR.B Dq,Dr
  //  右ローテートバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*00 Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................ｸｱｲｳｴｵｶｷ Xｸ*0ｸ Z=ｱｲｳｴｵｶｷｸ==0
  //     :
  //     7 ........................ｲｳｴｵｶｷｸｱ Xｲ*0ｲ Z=ｱｲｳｴｵｶｷｸ==0
  //     8 ........................ｱｲｳｴｵｶｷｸ Xｱ*0ｱ Z=ｱｲｳｴｵｶｷｸ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最上位ビット
  //
  //ROXR.B #<data>,Dr
  //ROXR.B Dq,Dr
  //  拡張右ローテートバイト
  //       ........................ｱｲｳｴｵｶｷｸ XNZVC
  //     0 ........................ｱｲｳｴｵｶｷｸ Xｱ*0X Z=ｱｲｳｴｵｶｷｸ==0
  //     1 ........................Xｱｲｳｴｵｶｷ ｸX*0ｸ Z=ｱｲｳｴｵｶｷX==0
  //     2 ........................ｸXｱｲｳｴｵｶ ｷｸ*0ｷ Z=ｱｲｳｴｵｶｸX==0
  //     3 ........................ｷｸXｱｲｳｴｵ ｶｷ*0ｶ Z=ｱｲｳｴｵｷｸX==0
  //     4 ........................ｶｷｸXｱｲｳｴ ｵｶ*0ｵ Z=ｱｲｳｴｶｷｸX==0
  //     5 ........................ｵｶｷｸXｱｲｳ ｴｵ*0ｴ Z=ｱｲｳｵｶｷｸX==0
  //     6 ........................ｴｵｶｷｸXｱｲ ｳｴ*0ｳ Z=ｱｲｴｵｶｷｸX==0
  //     7 ........................ｳｴｵｶｷｸXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸX==0
  //     8 ........................ｲｳｴｵｶｷｸX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸX==0
  //     9 ........................ｱｲｳｴｵｶｷｸ Xｱ*0X Z=ｱｲｳｴｵｶｷｸ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxrToRegByte () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    XEiJ.mpuCycleCount++;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASR.B #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (t = (byte) x >> y) >> 1);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSR.B #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xff & x | (z = (t = (0xff & x) >>> y) >>> 1);
      XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXR.B #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 7 - 4 | (0xff & x) >>> 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = x << 9 - 1 - y | (t = z >>> y - (2 - 1)) >>> 1;
      }
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROR.B #<data>,Dr
      y = XEiJ.regOC >> 9 & 7;  //y=data&7
      XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) (x << 8 - y | (0xff & x) >>> y));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z >>> 7 & 1;  //Xは変化しない。Cは結果の最上位ビット
      break;
    case 0b100_000 >> 3:  //ASR.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      if (y == 0) {  //y=data=0
        z = (byte) x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (t = (byte) x >> (y <= 8 ? y - 1 : 7)) >> 1);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b101_000 >> 3:  //LSR.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      if (y == 0) {  //y=data=0
        z = (byte) x;
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (z < 0 ? XEiJ.REG_CCR_N : z == 0 ? XEiJ.REG_CCR_Z : 0);  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xff & x | (z = (t = y <= 8 ? (0xff & x) >>> y - 1 : 0) >>> 1);
        XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      break;
    case 0b110_000 >> 3:  //ROXR.B Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      //y %= 9;
      y = (y & 7) - (y >> 3);  //y=data=-7～7
      y += y >> 3 & 9;  //y=data=0～8
      if (y == 0) {  //y=data=0
        z = (byte) x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～8
        z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 7 - 4 | (0xff & x) >>> 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～8
          z = x << 9 - y | (t = z >>> y - 2) >>> 1;
        }
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) z);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROR.B Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      if (y == 0) {
        z = (byte) x;
        t = 0;  //Cはクリア
      } else {
        y &= 7;  //y=data=0～7
        XEiJ.regRn[rrr] = ~0xff & x | 0xff & (z = (byte) (x << 8 - y | (0xff & x) >>> y));
        t = z >>> 7 & 1;  //Cは結果の最上位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxrToRegByte

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_000_rrr
  //LSR.W #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_001_001_rrr
  //ROXR.W #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_001_010_rrr
  //ROR.W #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_011_rrr
  //ASR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_100_rrr
  //LSR.W Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_001_101_rrr
  //ROXR.W Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_001_110_rrr
  //ROR.W Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_001_111_rrr
  //ASR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_000_rrr [ASR.W #1,Dr]
  //LSR.W Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_001_001_rrr [LSR.W #1,Dr]
  //ROXR.W Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_001_010_rrr [ROXR.W #1,Dr]
  //ROR.W Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_001_011_rrr [ROR.W #1,Dr]
  //
  //ASR.W #<data>,Dr
  //ASR.W Dq,Dr
  //ASR.W <ea>
  //  算術右シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ｱｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀｱ*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ==0
  //     :
  //    15 ................ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //    16 ................ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSR.W #<data>,Dr
  //LSR.W Dq,Dr
  //LSR.W <ea>
  //  論理右シフトワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................0ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀ0*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ==0
  //     :
  //    15 ................000000000000000ｱ ｲ0*0ｲ Z=ｱ==0
  //    16 ................0000000000000000 ｱ010ｱ
  //    17 ................0000000000000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROR.W #<data>,Dr
  //ROR.W Dq,Dr
  //ROR.W <ea>
  //  右ローテートワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................ﾀｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ Xﾀ*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     :
  //    15 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀｱ Xｲ*0ｲ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //    16 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0ｱ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最上位ビット
  //
  //ROXR.W #<data>,Dr
  //ROXR.W Dq,Dr
  //ROXR.W <ea>
  //  拡張右ローテートワード
  //       ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ XNZVC
  //     0 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //     1 ................Xｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿ ﾀX*0ﾀ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿX==0
  //     2 ................ﾀXｱｲｳｴｵｶｷｸｹｺｻｼｽｾ ｿﾀ*0ｿ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾﾀX==0
  //     :
  //    15 ................ｳｴｵｶｷｸｹｺｻｼｽｾｿﾀXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX==0
  //    16 ................ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀX==0
  //    17 ................ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxrToRegWord () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    XEiJ.mpuCycleCount++;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASR.W #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (t = (short) x >> y) >> 1);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSR.W #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (z = (t = (char) x >>> y) >>> 1);
      XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b010_000 >> 3:  //ROXR.W #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 15 - 4 | (char) x >>> 1;
      if (y == 1 - 1) {  //y=data-1=1-1
        t = x;
      } else {  //y=data-1=2-1～8-1
        z = x << 17 - 1 - y | (t = z >>> y - (2 - 1)) >>> 1;
      }
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) z);
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b011_000 >> 3:  //ROR.W #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) (x << 16 - 1 - y | (char) x >>> y + 1));
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | z >>> 15 & 1;  //Xは変化しない。Cは結果の最上位ビット
      break;
    case 0b100_000 >> 3:  //ASR.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      if (y == 0) {  //y=data=0
        z = (short) x;
        t = XEiJ.regCCR & XEiJ.REG_CCR_X;  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (t = (short) x >> (y <= 16 ? y - 1 : 15)) >> 1);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b101_000 >> 3:  //LSR.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      if (y == 0) {  //y=data=0
        z = (short) x;
        XEiJ.regCCR = XEiJ.regCCR & XEiJ.REG_CCR_X | (z < 0 ? XEiJ.REG_CCR_N : z == 0 ? XEiJ.REG_CCR_Z : 0);  //Xは変化しない。Cはクリア
      } else {  //y=data=1～63
        XEiJ.regRn[rrr] = ~0xffff & x | (z = (t = y <= 16 ? (char) x >>> y - 1 : 0) >>> 1);
        XEiJ.regCCR = (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      break;
    case 0b110_000 >> 3:  //ROXR.W Dq,Dr
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      //y %= 17;
      y = (y & 15) - (y >> 4);  //y=data=-3～15
      y += y >> 4 & 17;  //y=data=0～16
      if (y == 0) {  //y=data=0
        z = (short) x;
        t = -(XEiJ.regCCR >> 4 & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //Xは変化しない。CはXのコピー
      } else {  //y=data=1～16
        z = (XEiJ.regCCR & XEiJ.REG_CCR_X) << 15 - 4 | (char) x >>> 1;
        if (y == 1) {  //y=data=1
          t = x;  //Cは最後に押し出されたビット
        } else {  //y=data=2～16
          z = x << 17 - y | (t = z >>> y - 2) >>> 1;
        }
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) z);
        t = -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | t;
      break;
    case 0b111_000 >> 3:  //ROR.W Dq,Dr
    default:
      y = XEiJ.regRn[XEiJ.regOC >> 9 & 7] & 63;  //y=0～63。Javaのシフト演算子は5ビットでマスクされることに注意
      if (y == 0) {
        z = (short) x;
        t = 0;  //Cはクリア
      } else {
        y &= 15;  //y=data=0～15
        XEiJ.regRn[rrr] = ~0xffff & x | (char) (z = (short) (x << 16 - y | (char) x >>> y));
        t = z >>> 15 & 1;  //Cは結果の最上位ビット
      }
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.regCCR & XEiJ.REG_CCR_X | XEiJ.REG_CCR_Z : XEiJ.regCCR & XEiJ.REG_CCR_X) | t;  //Xは変化しない
    }
  }  //irpXxrToRegWord

  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //                                                | |  MPU | |CCin |CCout|addressing|     1st opcode         2nd opcode
  //                           A:alias P:privileged |A|012346|P|XNZVC|XNZVC|DAM+-WXZPI|bbbb_bbb_bbb_bbb_bbb-bbbbbbbbbbbbbbbb
  //------------------------------------------------+-+------+-+-----+-----+----------+-------------------------------------
  //ASR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_000_rrr
  //LSR.L #<data>,Dr                                |-|012346|-|UUUUU|***0*|          |1110_qqq_010_001_rrr
  //ROXR.L #<data>,Dr                               |-|012346|-|*UUUU|***0*|          |1110_qqq_010_010_rrr
  //ROR.L #<data>,Dr                                |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_011_rrr
  //ASR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_100_rrr
  //LSR.L Dq,Dr                                     |-|012346|-|UUUUU|***0*|          |1110_qqq_010_101_rrr
  //ROXR.L Dq,Dr                                    |-|012346|-|*UUUU|***0*|          |1110_qqq_010_110_rrr
  //ROR.L Dq,Dr                                     |-|012346|-|-UUUU|-**0*|          |1110_qqq_010_111_rrr
  //ASR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_000_rrr [ASR.L #1,Dr]
  //LSR.L Dr                                        |A|012346|-|UUUUU|***0*|          |1110_001_010_001_rrr [LSR.L #1,Dr]
  //ROXR.L Dr                                       |A|012346|-|*UUUU|***0*|          |1110_001_010_010_rrr [ROXR.L #1,Dr]
  //ROR.L Dr                                        |A|012346|-|-UUUU|-**0*|          |1110_001_010_011_rrr [ROR.L #1,Dr]
  //
  //ASR.L #<data>,Dr
  //ASR.L Dq,Dr
  //  算術右シフトロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ｱｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ ﾐｱ*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ==0
  //     :
  //    31 ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｲｱ*0ｲ Z=ｱ==0
  //    32 ｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱｱ ｱｱ*0ｱ Z=ｱ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //LSR.L #<data>,Dr
  //LSR.L Dq,Dr
  //  論理右シフトロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 0ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ ﾐ0*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ==0
  //     :
  //    31 0000000000000000000000000000000ｱ ｲ0*0ｲ Z=ｱ==0
  //    32 00000000000000000000000000000000 ｱ010ｱ
  //    33 00000000000000000000000000000000 00100
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は最後に押し出されたビット
  //
  //ROR.L #<data>,Dr
  //ROR.L Dq,Dr
  //  右ローテートロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*00 Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 ﾐｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ Xﾐ*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     :
  //    31 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐｱ Xｲ*0ｲ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //    32 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*0ｱ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //  CCR
  //    X  常に変化しない
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときクリア。他は結果の最上位ビット
  //
  //ROXR.L #<data>,Dr
  //ROXR.L Dq,Dr
  //  拡張右ローテートロング
  //       ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ XNZVC
  //     0 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //     1 Xｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏ ﾐX*0ﾐ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏX==0
  //     2 ﾐXｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎ ﾏﾐ*0ﾏ Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾐX==0
  //     :
  //    31 ｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐXｱ ｲｳ*0ｲ Z=ｱｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐX==0
  //    32 ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐX ｱｲ*0ｱ Z=ｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐX==0
  //    33 ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ Xｱ*0X Z=ｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐ==0
  //  CCR
  //    X  countが0のとき変化しない。他は最後に押し出されたビット
  //    N  結果の最上位ビット
  //    Z  結果が0のときセット。他はクリア
  //    V  常にクリア
  //    C  countが0のときXのコピー。他は最後に押し出されたビット
  public static void irpXxrToRegLong () throws M68kException {
    int rrr;
    int x = XEiJ.regRn[rrr = XEiJ.regOC & 7];
    int y;
    int z;
    int t;
    XEiJ.mpuCycleCount++;
    switch (XEiJ.regOC >> 3 & 0b111_000 >> 3) {
    case 0b000_000 >> 3:  //ASR.L #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = z = (t = x >> y) >> 1;
      XEiJ.regCCR = z >> 28 & XEiJ.REG_CCR_N | (z == 0 ? XEiJ.REG_CCR_Z : 0) | -(t & 1) & (XEiJ.REG_CCR_X | XEiJ.REG_CCR_C);  //XとCは最後に押し出されたビット
      break;
    case 0b001_000 >> 3:  //LSR.L #<data>,Dr
      y = (XEiJ.regOC >> 9) - 1 & 7;  //y=data-1=1-1～8-1
      XEiJ.regRn[rrr] = z = (t = x >>> y) >>> 1;
      XEiJ.regCCR = (z == 0 ? X